/*
 * Copyright (c) 2008 Janne Kataja <janne.kataja@iki.fi>
 *
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-program.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>

// sudo command
#ifndef CMD_SUDO
#define CMD_SUDO "/usr/bin/sudo"
#endif

// helper script command
#ifndef CMD_HELPER
#define CMD_HELPER "/usr/sbin/usb-helper"
#endif

// Action to be passed to helper script 
typedef enum {
	HELPER_STORAGE,
	HELPER_NETWORK,
	HELPER_HOST,
	HELPER_TEST
} helper_action_t;

// Arguments to be passed to helper script relating to mode
static char* helper_action_msg[] = {
	"storage",
	"network",
	"host",
	"test"
};

// Helper returned error codes
typedef enum {
	CHILD_FAIL,
	CHILD_NOTROOT,
	CHILD_USBMODE,
	CHILD_GADGET_NA,
	CHILD_NETWORK,
	CHILD_PORT,
	CHILD_INSMOD
} child_error_t;

// Messages relating to helper returned error codes
static char* child_error_msg[] = {
	"Helper has returned with error",
	"Helper is missing from sudoers",
	"USB mode is not writable",
	"USB gadget driver is missing",
	"USB networking interface is not defined",
	"USB port already in use",
	"Module insertion has failed"
};

// USB port states. 
//
// CASE					MODE		CABLE
// cable is connected to a host		a_wait_bcon	Mini-B
// cable is plugged to a hub 		a_host 		Mini-A
// cable is unplugged			a_wait_bcon	Mini-A
// cable is connected to a host 	a_wait_bcon	Mini-A
//
// Also see:
// http://linux.omap.com/pipermail/linux-omap-open-source/2007-July/010569.html

// Keep pointers to UI elements
typedef struct {
	GtkWidget * setupbutton;
	GtkWidget * banner;
	GtkWidget * dialog;
	GtkWidget * reminder;
} user_data_t;

user_data_t * data;

////////////////////////////////////////////////////////////////////////
// Callbacks
////////////////////////////////////////////////////////////////////////

// Change radio buttons sensitivity
static void
set_sensitivity(gboolean yesno)
{
	gtk_widget_set_sensitive (GTK_WIDGET(data->setupbutton), yesno);
}

// Callback called when child returned
static void
child_exit(GPid pid, gint status, gpointer user_data)
{
	g_spawn_close_pid(pid);
	if (status != 0) {
		int child_error = status >> 8;
		if (child_error == CHILD_PORT) {
			gtk_widget_show (data->reminder);
		}
		gchar * error_msg = g_strdup(child_error_msg[child_error]);
		g_print("Helper %d returned with status %d (%s)\n", pid, status >> 8, error_msg);
		hildon_banner_show_information(GTK_WIDGET(data->dialog), NULL, error_msg);
		g_free(error_msg);
		return;
	}
	set_sensitivity(TRUE);
}

// Run helper program 
static gboolean
run_helper(helper_action_t usbstatus)
{
	gchar * act = helper_action_msg[usbstatus];
	GPid pid = 0;
	GError * err = 0;
	gchar *argv[] = { CMD_SUDO, CMD_HELPER, act, NULL };

	set_sensitivity(FALSE);
	
	g_debug("Running helper with arg '%s'\n", act);
	if (! g_spawn_async(NULL, argv, NULL, G_SPAWN_DO_NOT_REAP_CHILD, NULL, NULL, &pid, &err)) {
		hildon_banner_show_information(GTK_WIDGET(data->dialog), NULL, "Failed to run helper");
		g_message("Failed to run helper: %s\n", err->message);
		g_error_free(err);
		return FALSE;
	}
	g_child_watch_add(pid, child_exit, act);
	return TRUE;
}

// Setup button clicked, run helper
static void
setup_clicked(GtkToggleButton *togglebutton)
{
	run_helper(HELPER_NETWORK);
}

// Refresh button clicked, update UI based on port state
static void
refresh_clicked()
{
	gtk_widget_hide (data->reminder);
	set_sensitivity(FALSE);
	run_helper(HELPER_TEST);
}

////////////////////////////////////////////////////////////////////////
// Interface
////////////////////////////////////////////////////////////////////////

// Create new dialog
static GtkDialog *
usb_dialog_new()
{
	// TODO Move widgets to private data
	GtkWidget *dialog_vbox;
	GtkWidget *dialog_action_area;
	GtkWidget *closebutton;
	GtkWidget *refreshbutton;
	GtkWidget *setupbutton_alignment;
	GtkWidget *setupbutton_hbox;
	//GtkWidget *setupbutton_image;
	GtkWidget *setupbutton_label;
	//GtkWidget *remember;

	data = (user_data_t*)g_malloc(sizeof(user_data_t));
	g_assert(data != 0);

	// Dialog
	data->dialog = gtk_dialog_new ();
	gtk_widget_set_size_request(GTK_WIDGET(data->dialog), 400, 240);
	gtk_window_set_title (GTK_WINDOW (data->dialog), "USB networking");
	gtk_window_set_type_hint (GTK_WINDOW (data->dialog), GDK_WINDOW_TYPE_HINT_DIALOG);

	dialog_vbox = GTK_DIALOG (data->dialog)->vbox;

	// Options
	data->setupbutton = gtk_button_new ();
	gtk_box_pack_start (GTK_BOX (dialog_vbox), data->setupbutton, FALSE, FALSE, 0);

	setupbutton_alignment = gtk_alignment_new (0.5, 0.5, 0, 0);
	gtk_container_add (GTK_CONTAINER (data->setupbutton), setupbutton_alignment);

	setupbutton_hbox = gtk_hbox_new (FALSE, 2);
	gtk_container_add (GTK_CONTAINER (setupbutton_alignment), setupbutton_hbox);

	//setupbutton_image = gtk_image_new_from_stock ("gtk-connect", GTK_ICON_SIZE_BUTTON);
	//gtk_box_pack_start (GTK_BOX (setupbutton_hbox), setupbutton_image, FALSE, FALSE, 0);

	setupbutton_label = gtk_label_new_with_mnemonic ("Setup USB networking");
	gtk_box_pack_start (GTK_BOX (setupbutton_hbox), setupbutton_label, FALSE, FALSE, 0);

	data->reminder = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL(data->reminder), "USB connection in use.\n<small>Disconnect the cable, \nhit Refresh, and try again.</small>");
	gtk_box_pack_start (GTK_BOX (dialog_vbox), GTK_LABEL(data->reminder), TRUE, TRUE, 0);

	// Action area
	dialog_action_area = GTK_DIALOG (data->dialog)->action_area;
	gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area), GTK_BUTTONBOX_EDGE);

	refreshbutton = gtk_button_new_from_stock ("gtk-refresh");
	gtk_box_pack_start (GTK_BOX (dialog_action_area), refreshbutton, FALSE, FALSE, 0);

	closebutton = gtk_button_new_from_stock ("gtk-close");
	gtk_dialog_add_action_widget (GTK_DIALOG (data->dialog), closebutton, GTK_RESPONSE_CLOSE);
	GTK_WIDGET_SET_FLAGS (closebutton, GTK_CAN_DEFAULT);

	// Signals
	g_signal_connect(GTK_BUTTON(data->setupbutton), "clicked", G_CALLBACK(setup_clicked), NULL);
	g_signal_connect(GTK_BUTTON(refreshbutton), "clicked", G_CALLBACK(refresh_clicked), NULL);
	//g_signal_connect(GTK_CHECK_BUTTON(remember), "toggled", G_CALLBACK(remember_toggled), &data);

	gtk_widget_show_all (data->dialog);

	refresh_clicked();

	return GTK_DIALOG(data->dialog);
}


////////////////////////////////////////////////////////////////////////
// Entry
////////////////////////////////////////////////////////////////////////

// Called from control panel
osso_return_t 
execute(osso_context_t *osso, gpointer data, gboolean user_activated)
{
	HildonProgram *program;

	program = HILDON_PROGRAM(hildon_program_get_instance());

	GtkDialog *dialog = GTK_DIALOG(usb_dialog_new());
	gtk_dialog_run(dialog);
	gtk_widget_destroy(GTK_WIDGET(dialog));

	return OSSO_OK;
}
