/*
 * Copyright (c) 2008 Janne Kataja <janne.kataja@iki.fi>
 *
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-program.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>

////////////////////////////////////////////////////////////////////////
// invoke-rc.d and update-rc.d args
////////////////////////////////////////////////////////////////////////

#ifndef CMD_SUDO
#define CMD_SUDO "/usr/bin/sudo"
#endif

#ifndef CMD_INVOKE_RCD 
#define CMD_INVOKE_RCD "/usr/sbin/invoke-rc.d"
#endif

#ifndef CMD_UPDATE_RCD 
#define CMD_UPDATE_RCD "/usr/sbin/update-rc.d"
#endif

#define DIR_RC2 "/etc/rc2.d"
#define DIR_INITD "/etc/init.d"

/** 
 * invoke-rc.d actions:
 * start, stop, restart, reload, status
 */
typedef enum {
	invoke_start,
	invoke_stop,
	invoke_restart,
	invoke_reload,
	invoke_status
} invoke_action_t;

/**
 * invoke-rc.d actions as string.
 */
static char* invoke_action_msg[] = {
	"start",
	"stop",
	"restart",
	"reload",
	"status"
};

/**
 * Service state.
 */
typedef enum
{
	service_running, /// Service running
	service_stopped, /// Service stopped
	service_indeterminate /// Service state can't be detected
} service_state_t;

/**
 * update-rc.d actions.
 */

typedef enum {
	update_remove, /// Remove service from init
	update_defaults /// Add service to init with defaults
} update_action_t;

/**
 * update-rc.d actions as string.
 */
static char* update_action_msg[] = {
	"remove",
	"defaults"
};

/**
 * Helper command (invoke-rc.d and update-rc.d) returned error codes.
 */
typedef enum {
	child_fail,
	child_notroot
} child_error_t;

/**
 * Messages relating to helper returned error codes.
 */
static char* child_error_msg[] = {
	"Helper has returned with error",
	"Helper is missing from sudoers"
};

/**
 * Services to be always filtered from view.
 * TODO Load services from configuration file
 */
static char * filter_always[] = {
	"README",
	"rc",
	"rcS",
	// Package initscripts 
	"bootmisc.sh",
	"checkroot.sh",
	"halt",
	"hostname.sh",
	"mountvirtfs",
	"reboot",
	"sendsigs",
	"single",
	"umountfs",
	"urandom",
	// Other
	"fb-progress.sh",  
	"hwclockfirst.sh",
	"glibc.sh",
	"hwclock.sh",  
	"procps.sh",
	"ttyusb0", 
	"ifupdown", 
	"ifupdown-clean", 
	"minireboot", 
	"minishutdown", 
	"networking",
	"klogd",
	"sysklogd",
	"module-init-tools",
	"fb-progress", 
	"zzinitdone", 
	"pppd-dns", 
	"osso-applet-display", 
	"product-code", 
	NULL
};

/**
 * Methods for detecting running service.
 */ 
typedef enum {
	guess_file, /// Service detected by existing file
	guess_proc, /// Service detected by executing process
	guess_module, /// Service detected by loaded kernel module
	guess_indeterminate /// Service running is indeterminate
} guess_t;

/**
 * System services to hide when "Hide system services" is enabled
 */
typedef struct {
	const gchar * name; /// Service name as in /etc/init.d/
	const gchar * description; /// Service description (text)
	const gchar * icon; /// Icon to be shown next to service
	const gboolean system; // Service is system service
	const guess_t type; // Method of detecting running service.
	const gchar * guessstr; // Filename or program name to search for.
} service_t;

/**
 * List of services visible in the UI.
 * Also services not listed here will appear in the UI, but will not have description
 * and cannot be started/stopped.
 * TODO Load from configuration file
 */
static service_t services[] = {
	// name, description, system service t/f, guess type, guess file/process
	{ "hildon-desktop", "Hildon desktop", "qgn_list_shell_mydevice", TRUE, guess_proc, "/usr/bin/hildon-desktop" }, // TODO Icon UI
	{ "af-base-apps", "Application framework", "qgn_list_shell_mydevice", TRUE, guess_indeterminate, "" }, // TODO Icon UI
	{ "af-services", "Application framework", "qgn_list_shell_mydevice", TRUE, guess_indeterminate, ""  }, // TODO Icon UI
	{ "af-startup", "Application framework", "qgn_list_shell_mydevice", TRUE, guess_indeterminate, ""  }, // TODO Icon UI
	{ "alarmd", "Alarm dispatcher", "qgn_list_hclk_alarm_snoozed", TRUE, guess_proc, "/usr/bin/alarmd" },
	{ "bluez-utils", "Bluetooth tools", "qgn_list_cp_bluetooth", TRUE, guess_indeterminate, ""  },
	{ "bme-dbus-proxy", "Battery event proxy", "qgn_note_battery_low", TRUE, guess_proc, "/usr/bin/bme-dbus-proxy"  },
	{ "btcond", "Bluetooth connectivity", "qgn_list_cp_bluetooth", TRUE, guess_file , "/var/run/btcond.pid"  }, 
	{ "dbus", "D-Bus message bus", "qgn_list_cp_memory", TRUE, guess_file, "/var/run/dbus/pid"  }, 
	{ "dnsmasq", "DNS proxy and DHCP server", "qgn_list_connection_manager", TRUE, guess_file, "/var/run/dnsmasq.pid"  }, // TODO Icon connectivity
	{ "dsp-init", "DSP loader", "qgn_list_cp_memory", TRUE, guess_proc, "/usr/sbin/dsp_dld"  },
	{ "esd", "Enlightenment Sound Daemon", "qgn_list_cp_soundset", TRUE, guess_proc, "/usr/bin/esd"  },
	{ "gpsdriver", "GPS driver", "qgn_list_cp_location", TRUE, guess_proc, "/usr/sbin/gpsdriver"  }, // TODO Icon GPS
	{ "hildon-update-notifier", "Software upgrades", "qgn_list_gene_debian", TRUE, guess_indeterminate, ""  }, // TODO Icon
	{ "hulda", "Kernel events proxy", "qgn_list_cp_memory", TRUE, guess_proc, "/usr/sbin/hulda"  },
	{ "ke-recv", "Kernel events proxy", "qgn_list_cp_memory", TRUE, guess_proc, "/usr/sbin/ke-recv"  },
	{ "libgpsbt", "GPS device pairing", "qgn_list_cp_location", TRUE, guess_indeterminate, ""  }, // TODO Icon GPS
	{ "maemo-launcher", "Application launcher", "qgn_list_shell_mydevice", TRUE, guess_file, "/tmp/maemo-launcher.pid"  }, // TODO Icon UI
	{ "mce", "Machine Control Entity", "qgn_list_cp_memory", TRUE, guess_proc, "/sbin/mce"  },
	{ "mediaplayer-daemon", "Mediaplayer engine", "qgn_list_gene_notplayable", TRUE, guess_proc, "/usr/bin/mediaplayer-engine"  },
	{ "metalayer-crawler0", "Metalayer crawler", "qgn_list_gene_media_file", TRUE , guess_proc, "/usr/bin/metalayer-crawler" },
	{ "multimediad", "Multimedia daemon", "qgn_list_gene_music_file", TRUE, guess_proc, "/usr/sbin/multimediad"  },
	{ "obexsrv", "OBEX server", "qgn_list_gene_vcard", TRUE, guess_proc, "/usr/bin/obexsrv"  },
	{ "osso-ic", "Internet Connectivity", "qgn_list_connection_manager", TRUE, guess_file, "/var/run/icd.pid"  }, // TODO Icon connectivity
	{ "osso-systemui", "System UI", "qgn_list_cp_devicesetup", TRUE, guess_proc, "/usr/bin/systemui"  },
	{ "osso-systemui-early", "System UI", "qgn_list_cp_devicesetup", TRUE, guess_indeterminate, ""  },
	{ "wlancond", "WLAN connectivity", "qgn_list_connectivity_iaptype_wlan", TRUE, guess_file, "/var/run/wlancond.pid"  }, // TODO Icon connectivity
	{ "x-server", "X11 server", "qgn_indi_gene_images_attachment", TRUE, guess_proc, "/usr/bin/Xomap"  },
	// 3rd party
	{ "fuse", "Filesystem in USErspace", NULL, FALSE, guess_module, "fuse"  },
	{ "sbrshd", "Scratchbox Remote Shell", NULL, FALSE, guess_file, "/var/run/sbrshd.pid"  },
	{ "ssh", "Secure Shell server", NULL, FALSE, guess_file, "/var/run/sshd.pid"  },
	{ "portmap", "Portmapper", NULL, FALSE, guess_proc, "/sbin/portmap" },
	{ "nfs-common", "NFS common utilities", NULL, FALSE, guess_proc, "/sbin/rpc.statd" },
	{ NULL, NULL, NULL, FALSE, guess_indeterminate, ""  }
};

////////////////////////////////////////////////////////////////////////
// Callbacks
////////////////////////////////////////////////////////////////////////

static void populate_services_table();
static void create_services_table();
static void refresh_clicked();

/**
 * Application private data, keep pointers to UI elements.
 */
typedef struct {
	GtkWidget * dialog;
	GtkWidget * table;
	GtkWidget * filterbutton;
	GtkWidget * refreshbutton;
	GtkWidget * viewport;
	gboolean showsystem;
	osso_context_t * osso_context;
} user_data_t;

user_data_t data;

/**
 * Change services sensitivity.
 * @param yesno TRUE to make the services sensitive.
 */
static void
set_sensitivity(gboolean yesno)
{
	// TODO
}

/**
 * Called when helper command line child returns.
 */
static void
child_exit(GPid pid, gint status, gpointer user_data)
{
	g_spawn_close_pid(pid);
	if (status != 0) {
		gchar * error_msg = g_strdup(child_error_msg[status >> 8]);
		g_print("Helper %d returned with status %d (%s)\n", pid, status >> 8, error_msg);
		hildon_banner_show_information(GTK_WIDGET(data.dialog), NULL, error_msg);
		g_free(error_msg);
	}
	refresh_clicked();
}

/**
 * Call invoke-rc.d helper program.
 * @param act invoke-rc.d action
 * @param service Service to invoke.
 */
static gboolean
run_invoke(invoke_action_t act, gchar * service)
{
	GPid pid = 0;
	GError * err = 0;
	gchar *argv[] = { CMD_SUDO, CMD_INVOKE_RCD, service, invoke_action_msg[act], NULL };

	set_sensitivity(FALSE);

	g_debug("Running invoke-rc.d with arg '%s %s'\n", service, invoke_action_msg[act]);
	if (! g_spawn_async(NULL, argv, NULL, G_SPAWN_DO_NOT_REAP_CHILD, NULL, NULL, &pid, &err)) {
		hildon_banner_show_information(GTK_WIDGET(data.dialog), NULL, "Failed to run invoke-rc.d");
		g_message("Failed to run invoke-rc.d: %s\n", err->message);
		g_error_free(err);
		refresh_clicked();
		return FALSE;
	}
	g_child_watch_add(pid, child_exit, service);
	return TRUE;
}

/**
 * Call update-rc.d helper program.
 * @param act update-rc.d action
 * @param service Service to update.
 */
static gboolean
run_update(update_action_t act, gchar * service)
{
	GPid pid = 0;
	GError * err = 0;
	gchar *argv[] = { CMD_SUDO, CMD_UPDATE_RCD, "-f", service, update_action_msg[act], NULL };
	
	set_sensitivity(FALSE);

	g_debug("Running update-rc.d with arg '%s %s'\n", service, update_action_msg[act]);
	if (! g_spawn_async(NULL, argv, NULL, G_SPAWN_DO_NOT_REAP_CHILD, NULL, NULL, &pid, &err)) {
		hildon_banner_show_information(GTK_WIDGET(data.dialog), NULL, "Failed to run update-rc.d");
		g_message("Failed to run update-rc.d: %s\n", err->message);
		g_error_free(err);
		refresh_clicked();
		return FALSE;
	}
	g_child_watch_add(pid, child_exit, service);
	return TRUE;
}

/**
 * Service checkbox toggled, invoke init script.
 * @param togglebutton Pressed checkbox.
 */
static void
init_startstop(GtkToggleButton *togglebutton)
{
	gchar * init_name = g_object_get_qdata(G_OBJECT(togglebutton), g_quark_from_static_string ("init-name"));
	if (gtk_toggle_button_get_active(togglebutton))
		run_invoke(invoke_start, init_name);
	else
		run_invoke(invoke_stop, init_name);
}

/**
 * Service toggle button toggled, update init script.
 * @param togglebutton Pressed toggle button.
 */
static void
init_install(GtkToggleButton * togglebutton)
{
	gchar * init_name = g_object_get_qdata(G_OBJECT(togglebutton), g_quark_from_static_string ("init-name"));
	if (gtk_toggle_button_get_active(togglebutton))
		run_update(update_defaults, init_name);
	else
		run_update(update_remove, init_name);
}

/**
 * Refresh button clicked, reconstruct services table.
 */
static void
refresh_clicked()
{
	// FIXME Show animation
	//hildon_banner_show_information(GTK_WIDGET(data.dialog), NULL, "Updating services");
	set_sensitivity(FALSE);
	gtk_widget_destroy (data.table);
	create_services_table();
}

/**
 * Filter system services toggled , update services table.
 * @param togglebutton Filter toggle button.
 */
static void
filter_toggled(GtkToggleButton *togglebutton)
{
	data.showsystem = !(data.showsystem);
	refresh_clicked();
}

/**
 * Find matching service_t for init script.
 * @param key Service name as in /etc/init.d/
 * @return Pointer to existing service_t record, or NULL.
 */
static const service_t *
get_init(const char * key)
{
	for (int i = 0 ; services[i].name != NULL ; ++i)
		if (strcmp(services[i].name, key) == 0)
			return &services[i];
	return NULL;
}

/**
 * Detect if module is inserted in kernel.
 * @param module Kernel module name.
 * @return TRUE if kernel module is found in /proc/modules, FALSE otherwise
 */
static gboolean
find_module(const char * module)
{
	g_assert(module != NULL); // XXX
	gchar buf[1024];
	int len = strlen(module);
	FILE* f = fopen("/proc/modules", "r");
	while (fgets(buf, 1024, f) != NULL) {
		if (strncmp(buf, module, len) == 0) {
			fclose(f);
			return TRUE;
		}
	}
	fclose(f);
	return FALSE;
}

/**
 * Detect if service executable is found in /proc/.../exe .
 * @param key Service executable path.
 * @return TRUE if service is found in /proc, FALSE otherwise
 */
static gboolean
find_proc(const char * exe_path)
{
	g_assert(exe_path != NULL); // XXX

	gchar * command_line = 0;
	GError * err = 0;
	gint exit_status = 0;

	// FIXME Adding of line marker '$' caused not to match properly
	command_line = g_strdup_printf("/bin/sh -c \"/bin/grep -q '^%s' /proc/*/cmdline\"", exe_path);
	g_spawn_command_line_sync(command_line, 0, 0, &exit_status, &err);
	if (err != NULL) {
		g_message("g_spawn_command_line_sync(...) failed: %s\n", err->message);
		g_error_free(err);
	}
	g_free (command_line);
	return (exit_status == 0);
}

/**
 * Detect service state.
 * @param key Service name as in /etc/init.d/
 * @return Status running, stopped or indeterminte.
 */
static service_state_t
service_state(const char * key)
{
	g_assert(key != NULL); // XXX
	const service_t * init = get_init(key);

	if (init == NULL)
		return service_indeterminate;
	
	switch (init->type) {
		case guess_file:
			return (g_file_test(init->guessstr, G_FILE_TEST_EXISTS) ?  service_running : service_stopped );
			break;
		case guess_proc:
			return (find_proc(init->guessstr) ? service_running : service_stopped);
			break;
		case guess_module:
			return (find_module(init->guessstr) ? service_running : service_stopped);
		default:
			return service_indeterminate;
	}
}

/**
 * Detect if service is found in /etc/rc2.d/ and is started on boot.
 * @param key Service name as in /etc/init.d/
 * @return TRUE if service is found in /etc/rc2.d/, FALSE otherwise
 */
static gboolean
rc2_state(const char * key)
{
	g_assert(key != NULL); // XXX
	GError * err = 0;
	GDir * dir = g_dir_open(DIR_RC2, 0, NULL);
	if (dir == NULL) {
		g_printerr("No startup scripts found under " DIR_RC2);
		return FALSE;
	}
	const gchar * rc2_name;
	// Find matching symlink starting with '../init.d/'
	// TODO Cache entries instead of going thru directory for each init script
	while ((rc2_name = g_dir_read_name(dir)) != NULL) {
		gchar * rc2_file = 0;
		gchar * rc2_link = 0;
		rc2_file  = g_strdup_printf(DIR_RC2 "/%s", rc2_name);
		if (!g_file_test(rc2_file, G_FILE_TEST_IS_SYMLINK)) {
			g_free(rc2_file);
			continue;
		}
		rc2_link = g_file_read_link(rc2_file, &err);
		if (err != NULL) {
			g_message("g_file_read_link(...) failed: %s\n", err->message);
			g_error_free(err);
			g_free(rc2_file);
			g_free(rc2_link);
			continue;
		}
		if (rc2_link == NULL) {
			g_free(rc2_file);
			g_free(rc2_link);
			continue;
		}
		// Init script must start with '../init.d/'
		if (strncmp(rc2_link, "../init.d/", 10) != 0) {
			continue;
		}
		// Match with init key name (skip initial '../init.d/')
		if (strcmp(key, rc2_link + 10 * sizeof(char)) == 0) {
			g_free(rc2_file);
			g_free(rc2_link);
			g_dir_close(dir);
			return TRUE;
		}
		g_free(rc2_file);
		g_free(rc2_link);

	}
	g_dir_close(dir);
	return FALSE;
}

////////////////////////////////////////////////////////////////////////
// Interface
////////////////////////////////////////////////////////////////////////

/**
 * Compare function for service names. GCompareFunc 
 */

static gint
compare_init_name (gconstpointer a, gconstpointer b)
{
	const gchar * lhs = (const gchar *)a;
	const gchar * rhs = (const gchar *)b;
	return strcmp(lhs, rhs);
}

/**
 * Read services found in this computer (entries in /etc/init.d/) and
 * add matching entries in the UI table.
 */
static void
populate_services_table()
{
	GDir * dir = g_dir_open(DIR_INITD, 0, NULL);
	if (dir == NULL) {
		g_printerr("No startup scripts found under " DIR_INITD);
		return;
	}
	int row = 0;
	GList * list = NULL;
INIT_NEXT:
	for (const gchar * init_name = g_dir_read_name(dir) ; init_name != NULL ; init_name = g_dir_read_name(dir)) {
		// Next if listed in services to never show
		for (int i = 0 ; filter_always[i] != NULL ; ++i)
			if (strcmp(filter_always[i], init_name) == 0)
				goto INIT_NEXT;

		list = g_list_insert_sorted (list, init_name, compare_init_name);
	}

	for (GList * it = g_list_first(list) ; it != NULL ; it = g_list_next(it)) {
		const gchar * init_name = (const gchar *)it->data;

		// Skip if system services hide is on and marked as system service
		const service_t * init = get_init(init_name);
		if (init != NULL && init->system == TRUE && data.showsystem == FALSE)
			continue;
		
		// Add new table row n+1
		gtk_table_resize(GTK_TABLE (data.table), row+1, 2);

		// Service label
		gchar * service_text;
		if (init == NULL || init->description == NULL)
			service_text = g_strdup_printf("<i>%s</i>", init_name);
		else
			service_text = g_strdup_printf("%s (<i>%s</i>)", init->description, init_name);

		// Checkbutton
		GtkWidget * service_boot = gtk_check_button_new ();
		gtk_table_attach (GTK_TABLE (data.table), service_boot, 0, 1, 0+row, 1+row,
				(GtkAttachOptions) (GTK_FILL),
				(GtkAttachOptions) (0), 0, 0);

		GtkWidget * service_alignment = gtk_alignment_new (0.5, 0.5, 0, 0);
		gtk_container_add (GTK_CONTAINER (service_boot), service_alignment);

		GtkWidget * service_hbox = gtk_hbox_new (FALSE, 2);
		gtk_container_add (GTK_CONTAINER (service_alignment), service_hbox);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(service_boot), rc2_state(init_name));

		// Icon
		GtkWidget * service_icon;
		if (init && init->icon)
		       service_icon = gtk_image_new_from_icon_name (init->icon, GTK_ICON_SIZE_BUTTON);
	       else
		       service_icon = gtk_image_new_from_icon_name ( "qgn_list_gene_default_app", GTK_ICON_SIZE_BUTTON);
		gtk_box_pack_start (GTK_BOX (service_hbox), service_icon, FALSE, FALSE, 0);

		// Text
		GtkWidget * service_label = gtk_label_new_with_mnemonic (service_text);
		gtk_box_pack_start (GTK_BOX (service_hbox), service_label, FALSE, FALSE, 0);
		gtk_label_set_use_markup (GTK_LABEL (service_label), TRUE);
		g_free(service_text);

		// Service runtime state
		GtkWidget * service_run = gtk_toggle_button_new_with_mnemonic ("Running");
		gtk_table_attach (GTK_TABLE (data.table), service_run, 1, 2, 0+row, 1+row,
				(GtkAttachOptions) (GTK_FILL),
				(GtkAttachOptions) (0), 0, 0);

		// Start at runlevel 2 toggle button
		switch (service_state(init_name)) {
			case service_running:
				gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(service_run), TRUE);
				break;
			case service_stopped:
				gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(service_run), FALSE);
				break;
			default: // service_indeterminate
				gtk_toggle_button_set_inconsistent (GTK_TOGGLE_BUTTON(service_run), TRUE);
				gtk_widget_set_sensitive (GTK_WIDGET(service_run), FALSE);
				break;
		}

		// Add callbacks
		// TODO Check if duplicated init_name becomes managed by object
		g_object_set_qdata(G_OBJECT(service_boot), g_quark_from_static_string ("init-name"), g_strdup(init_name));
		g_object_set_qdata(G_OBJECT(service_run), g_quark_from_static_string ("init-name"), g_strdup(init_name));
		g_signal_connect(GTK_TOGGLE_BUTTON(service_run), "toggled", G_CALLBACK(init_startstop), NULL); 
		g_signal_connect(GTK_TOGGLE_BUTTON(service_boot), "toggled", G_CALLBACK(init_install), NULL); 

		++row;

	}
	g_list_free(list);
	g_dir_close(dir);

}

/**
 * Create UI table containing the list of services.
 */
static void
create_services_table()
{
	data.table = gtk_table_new (0, 2, FALSE);
	gtk_container_add (GTK_CONTAINER (data.viewport), data.table);
	populate_services_table();
	gtk_widget_show_all (data.table);
}

/**
 * Create application main dialog.
 */
static GtkWidget *
create_services_dialog (void)
{
	GtkWidget * dialog_vbox;
	GtkWidget * services_vbox;
	GtkWidget * titletext;
	GtkWidget * scrolledwindow;
	GtkWidget * dialog_action;
	GtkWidget * closebutton;

	data.dialog = NULL;
	data.table = NULL;
	data.filterbutton = NULL;
	data.showsystem = FALSE;

	// Dialog
	data.dialog = gtk_dialog_new ();
	gtk_widget_set_size_request(GTK_WIDGET(data.dialog), 620, 400);
	gtk_window_set_title (GTK_WINDOW (data.dialog), "Services");
	gtk_window_set_type_hint (GTK_WINDOW (data.dialog), GDK_WINDOW_TYPE_HINT_DIALOG);

	// Contents
	dialog_vbox = GTK_DIALOG (data.dialog)->vbox;

	services_vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (dialog_vbox), services_vbox, FALSE, FALSE, 0);

	titletext = gtk_label_new ("<b>Services started during system boot process:</b>");
	gtk_box_pack_start (GTK_BOX (services_vbox), titletext, FALSE, FALSE, 0);
	gtk_label_set_use_markup (GTK_LABEL (titletext), TRUE);

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (services_vbox), scrolledwindow, TRUE, TRUE, 0);
	gtk_widget_set_size_request(GTK_WIDGET(scrolledwindow), 440, 300); // FIXME Fixed layout
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	data.viewport = gtk_viewport_new (NULL, NULL);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), data.viewport);

	// List of services
	create_services_table();

	// Action area
	dialog_action = GTK_DIALOG (data.dialog)->action_area;
	gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action), GTK_BUTTONBOX_EDGE);

	data.filterbutton = gtk_toggle_button_new_with_label ("Hide system services");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(data.filterbutton), TRUE);
	gtk_box_pack_start (GTK_BOX (dialog_action), data.filterbutton, FALSE, FALSE, 0);

	data.refreshbutton = gtk_button_new_from_stock ("gtk-refresh");
	gtk_box_pack_start (GTK_BOX (dialog_action), data.refreshbutton, FALSE, FALSE, 0);

	closebutton = gtk_button_new_from_stock ("gtk-close");
	gtk_dialog_add_action_widget (GTK_DIALOG (data.dialog), closebutton, GTK_RESPONSE_CLOSE);
	GTK_WIDGET_SET_FLAGS (closebutton, GTK_CAN_DEFAULT);

	// Connect signals
	g_signal_connect(GTK_BUTTON(data.refreshbutton), "clicked", G_CALLBACK(refresh_clicked), NULL);
	g_signal_connect(GTK_BUTTON(data.filterbutton), "toggled", G_CALLBACK(filter_toggled), NULL);

	gtk_widget_show_all (data.dialog);
	return GTK_DIALOG(data.dialog);
}

////////////////////////////////////////////////////////////////////////
// Entry
////////////////////////////////////////////////////////////////////////

#ifdef CPA
/**
 * Application was started from control panel.
 */
osso_return_t 
execute(osso_context_t * osso, gpointer userdata, gboolean user_activated)
{
	HildonProgram *program;
	program = HILDON_PROGRAM(hildon_program_get_instance());
	
	if (osso == NULL)
		return OSSO_ERROR;
	data.osso_context = osso;

	// enable help system on dialog
	GtkDialog * dialog = GTK_DIALOG(create_services_dialog());

	gtk_dialog_run(dialog);
	gtk_widget_destroy(GTK_WIDGET(dialog));

	return OSSO_OK;
}
#else
/**
 * Application was started from command line.
 */
int main(int argc, char *argv[])
{
	HildonProgram *program = NULL;

	gtk_init(&argc, &argv);

	program = HILDON_PROGRAM(hildon_program_get_instance());
	g_set_application_name("Services");

	GtkDialog * dialog = GTK_DIALOG(create_services_dialog());
	gtk_dialog_run(dialog);
	gtk_widget_destroy(GTK_WIDGET(dialog));

	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(gtk_main_quit), NULL);

	gtk_main();

	return OSSO_OK;
}
#endif
