/*
 *  Miaouw - The Miaouw Library for Maemo Development
 *  Copyright (C) 2008 Henrik Hedberg <hhedberg@innologies.fi>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
 
#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <errno.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include <miaouw/miaouw.h>
#include "miaouwmarshalers.h"

/**
 * SECTION:miaouweventbox
 * @short_description: An event box with capturing event model
 * @see_also: #GtkEventBox
 *
 * Similar than the GtkEventBox but provides capturing event model.
 *
 * In GTK+, events are delivered first to child widgets, which may stop their propagation to parent widgets (propagate up, bubbling phase).
 * The MiaouwEventBox captures events before those are delivered to childs (propagate down, capturing phase). 
 **/
 
enum {
	SIGNAL_CAPTURED_EVENT,
	SIGNAL_CAPTURED_BUTTON_PRESS,
	SIGNAL_CAPTURED_SCROLL,
	SIGNAL_CAPTURED_BUTTON_RELEASE,
	SIGNAL_CAPTURED_MOTION_NOTIFY,
	SIGNAL_CAPTURED_DELETE,
	SIGNAL_CAPTURED_DESTROY,
	SIGNAL_CAPTURED_KEY_PRESS,
	SIGNAL_CAPTURED_KEY_RELEASE,
	SIGNAL_CAPTURED_ENTER_NOTIFY,
	SIGNAL_CAPTURED_LEAVE_NOTIFY,
	SIGNAL_CAPTURED_FOCUS_IN,
	SIGNAL_CAPTURED_FOCUS_OUT,
	SIGNAL_CAPTURED_CONFIGURE,
	SIGNAL_CAPTURED_MAP,
	SIGNAL_CAPTURED_UNMAP,
	SIGNAL_CAPTURED_WINDOW_STATE,
	SIGNAL_CAPTURED_PROPERTY_NOTIFY,
	SIGNAL_CAPTURED_SELECTION_CLEAR,
	SIGNAL_CAPTURED_SELECTION_REQUEST,
	SIGNAL_CAPTURED_SELECTION_NOTIFY,
	SIGNAL_CAPTURED_PROXIMITY_IN,
	SIGNAL_CAPTURED_PROXIMITY_OUT,
	SIGNAL_CAPTURED_NO_EXPOSE,
	SIGNAL_CAPTURED_CLIENT,
	SIGNAL_CAPTURED_EXPOSE,
	SIGNAL_CAPTURED_VISIBILITY_NOTIFY,
	SIGNAL_CAPTURED_GRAB_BROKEN,
	SIGNAL_CAPTURED_EVENT_AFTER,
	SIGNAL_COUNT
};


static guint signals[SIGNAL_COUNT];

struct _MiaouwEventBoxPrivate {
	gboolean has_window;
	GdkWindow* window;
};

gboolean boolean_accumulator (GSignalInvocationHint *ihint,
				  GValue                *return_accu,
				  const GValue          *handler_return,
				  gpointer               dummy)
{
  gboolean continue_emission;
  gboolean signal_handled;
  
  signal_handled = g_value_get_boolean (handler_return);
  g_value_set_boolean (return_accu, signal_handled);
  continue_emission = !signal_handled;
  
  return continue_emission;
}

static void size_request(GtkWidget* widget, GtkRequisition* requisition) {
	GtkRequisition child_requisition;

	requisition->width = 2 * GTK_CONTAINER(widget)->border_width;
	requisition->height = 2 * GTK_CONTAINER(widget)->border_width;
	if (GTK_BIN(widget)->child && GTK_WIDGET_VISIBLE(GTK_BIN(widget)->child)) {
		gtk_widget_size_request(GTK_BIN(widget)->child, &child_requisition);
		requisition->width += child_requisition.width;
		requisition->height += child_requisition.height;
	}
}

static void size_allocate(GtkWidget* widget, GtkAllocation* allocation) {
	GtkAllocation child_allocation;

	widget->allocation = *allocation;

	child_allocation.x = allocation->x + GTK_CONTAINER(widget)->border_width;
	child_allocation.y = allocation->y + GTK_CONTAINER(widget)->border_width;
	child_allocation.width = MAX(allocation->width - 2 * GTK_CONTAINER(widget)->border_width, 0);
	child_allocation.height = MAX(allocation->height - 2 * GTK_CONTAINER(widget)->border_width , 0);

	if (MIAOUW_EVENT_BOX(widget)->priv->window) {
		gdk_window_move_resize(MIAOUW_EVENT_BOX(widget)->priv->window, child_allocation.x, child_allocation.y,
		                       child_allocation.width, child_allocation.height);
	}

	if (GTK_BIN(widget)->child) {
		gtk_widget_size_allocate(GTK_BIN(widget)->child, &child_allocation);
	}
}

static void realize(GtkWidget* widget) {
	GtkWidgetClass* widget_class;
	GdkWindowAttr attributes;

	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_EVENT_BOX)));
	if (widget_class->realize) {
		widget_class->realize(widget);
	}

	widget->window = gtk_widget_get_parent_window (widget);
	g_object_ref (widget->window);

	if (MIAOUW_EVENT_BOX(widget)->priv->has_window) {
		attributes.x = widget->allocation.x + GTK_CONTAINER(widget)->border_width;
		attributes.y = widget->allocation.y + GTK_CONTAINER(widget)->border_width;
		attributes.width = widget->allocation.width - 2 * GTK_CONTAINER(widget)->border_width;
		attributes.height = widget->allocation.height - 2 * GTK_CONTAINER(widget)->border_width;
		attributes.window_type = GDK_WINDOW_CHILD;
		attributes.wclass = GDK_INPUT_ONLY;
		attributes.event_mask = gtk_widget_get_events(widget);
		MIAOUW_EVENT_BOX(widget)->priv->window = gdk_window_new(widget->window, &attributes, GDK_WA_X | GDK_WA_Y);
		gdk_window_set_user_data(MIAOUW_EVENT_BOX(widget)->priv->window, widget);
	}

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
}

static void unrealize(GtkWidget* widget) {
	MiaouwEventBox* event_box;
	GtkWidgetClass* widget_class;
	
	event_box = MIAOUW_EVENT_BOX(widget);

	if (event_box->priv->window) {
		gdk_window_set_user_data(event_box->priv->window, NULL);
		gdk_window_destroy(event_box->priv->window);
		event_box->priv->window = NULL;
	}

	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_EVENT_BOX)));
	if (widget_class->unrealize) {
		widget_class->unrealize(widget);
	}

	GTK_WIDGET_UNSET_FLAGS (widget, GTK_REALIZED);
}

static void map(GtkWidget* widget) {
	GtkWidgetClass* widget_class;

	if (MIAOUW_EVENT_BOX(widget)->priv->window) {
		gdk_window_show(MIAOUW_EVENT_BOX(widget)->priv->window);
	}

	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_EVENT_BOX)));
	if (widget_class->map) {
		widget_class->map(widget);
	}
}

static void unmap(GtkWidget* widget) {
	GtkWidgetClass* widget_class;
	
	widget_class = GTK_WIDGET_CLASS(g_type_class_peek_parent(g_type_class_peek(MIAOUW_TYPE_EVENT_BOX)));
	if (widget_class->unmap) {
		widget_class->unmap(widget);
	}

	if (MIAOUW_EVENT_BOX(widget)->priv->window) {
		gdk_window_hide(MIAOUW_EVENT_BOX(widget)->priv->window);
	}
}

static gboolean event_handler(GdkEvent* event, MiaouwEventHandlerState* state, gpointer data) {
	gint signal_id;
	GtkWidget* widget;
	GList* list;
	GList* item;
	gboolean stop_propagating;

	switch (event->type) {
		case GDK_NOTHING:
			signal_id = -1; break;
		case GDK_BUTTON_PRESS: case GDK_2BUTTON_PRESS: case GDK_3BUTTON_PRESS:
			signal_id = SIGNAL_CAPTURED_BUTTON_PRESS; break;
		case GDK_SCROLL:
			signal_id = SIGNAL_CAPTURED_SCROLL; break;
		case GDK_BUTTON_RELEASE:
			signal_id = SIGNAL_CAPTURED_BUTTON_RELEASE; break;
		case GDK_MOTION_NOTIFY:
			signal_id = SIGNAL_CAPTURED_MOTION_NOTIFY; break;
		case GDK_DELETE:
			signal_id = SIGNAL_CAPTURED_DELETE; break;
		case GDK_DESTROY:
			signal_id = SIGNAL_CAPTURED_DESTROY; break;
		case GDK_KEY_PRESS:
			signal_id = SIGNAL_CAPTURED_KEY_PRESS; break;
		case GDK_KEY_RELEASE:
			signal_id = SIGNAL_CAPTURED_KEY_RELEASE; break;
		case GDK_ENTER_NOTIFY:
			signal_id = SIGNAL_CAPTURED_ENTER_NOTIFY; break;
		case GDK_LEAVE_NOTIFY:
			signal_id = SIGNAL_CAPTURED_LEAVE_NOTIFY; break;
		case GDK_FOCUS_CHANGE:
			signal_id = event->focus_change.in ? SIGNAL_CAPTURED_FOCUS_IN : SIGNAL_CAPTURED_FOCUS_OUT; break;
		case GDK_CONFIGURE:
			signal_id = SIGNAL_CAPTURED_CONFIGURE; break;
		case GDK_MAP:
			signal_id = SIGNAL_CAPTURED_MAP; break;
		case GDK_UNMAP:
			signal_id = SIGNAL_CAPTURED_UNMAP; break;
		case GDK_WINDOW_STATE:
			signal_id = SIGNAL_CAPTURED_WINDOW_STATE; break;
		case GDK_PROPERTY_NOTIFY:
			signal_id = SIGNAL_CAPTURED_PROPERTY_NOTIFY; break;
		case GDK_SELECTION_CLEAR:
			signal_id = SIGNAL_CAPTURED_SELECTION_CLEAR; break;
		case GDK_SELECTION_REQUEST:
			signal_id = SIGNAL_CAPTURED_SELECTION_REQUEST; break;
		case GDK_SELECTION_NOTIFY:
			signal_id = SIGNAL_CAPTURED_SELECTION_NOTIFY; break;
		case GDK_PROXIMITY_IN:
			signal_id = SIGNAL_CAPTURED_PROXIMITY_IN; break;
		case GDK_PROXIMITY_OUT:
			signal_id = SIGNAL_CAPTURED_PROXIMITY_OUT; break;
		case GDK_NO_EXPOSE:
			signal_id = SIGNAL_CAPTURED_NO_EXPOSE; break;
		case GDK_CLIENT_EVENT:
			signal_id = SIGNAL_CAPTURED_CLIENT; break;
		case GDK_EXPOSE:
			signal_id = SIGNAL_CAPTURED_EXPOSE; break;
		case GDK_VISIBILITY_NOTIFY:
			signal_id = SIGNAL_CAPTURED_VISIBILITY_NOTIFY; break;
#if HILDON == 1
		case GDK_GRAB_BROKEN:
			signal_id = SIGNAL_CAPTURED_GRAB_BROKEN; break;
#endif
		default:
			g_warning ("MiaouwEventBox: uncaptured event type: %d", event->type);
			signal_id = -1;
			break;
	}

	list = NULL;
	if (signal_id != -1) {
		widget = gtk_get_event_widget(event);
		if (widget) {
			widget = gtk_widget_get_parent(widget);
			while (widget) {
				if (MIAOUW_IS_EVENT_BOX(widget)) {
					list = g_list_prepend(list, widget);
				}
				widget = gtk_widget_get_parent(widget);
			}
		}

		item = g_list_first(list);
		stop_propagating = FALSE;
		while (item && !stop_propagating) {
			g_signal_emit(item->data, signals[SIGNAL_CAPTURED_EVENT], 0, event, &stop_propagating);
			if (!stop_propagating) {
				g_signal_emit(item->data, signals[signal_id], 0, event, &stop_propagating);
			}
			item = g_list_next(item);
		}
	}

	if (!stop_propagating) {
		miaouw_event_handler_next(event, state);

		item = g_list_first(list);
		while (item && !stop_propagating) {
			g_signal_emit(item->data, signals[SIGNAL_CAPTURED_EVENT_AFTER], 0, event, &stop_propagating);
			item = g_list_next(item);
		}
	}

	g_list_free(list);
	
	return stop_propagating;
}

static void miaouw_event_box_class_init(gpointer klass, gpointer data) {
	GTK_WIDGET_CLASS(klass)->size_request = size_request;
	GTK_WIDGET_CLASS(klass)->size_allocate = size_allocate;
	GTK_WIDGET_CLASS(klass)->realize = realize;
	GTK_WIDGET_CLASS(klass)->unrealize = unrealize;
	GTK_WIDGET_CLASS(klass)->map = map;
	GTK_WIDGET_CLASS(klass)->unmap = unmap;

#define SIGNAL_NEW_EVENT(x) g_signal_new(x, MIAOUW_TYPE_EVENT_BOX, 0, 0, boolean_accumulator, NULL, \
                                         miaouw_marshal_BOOLEAN__BOXED, G_TYPE_BOOLEAN, 1, \
		                         GDK_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE)
				   
	signals[SIGNAL_CAPTURED_EVENT] = SIGNAL_NEW_EVENT("captured-event");
	signals[SIGNAL_CAPTURED_BUTTON_PRESS] = SIGNAL_NEW_EVENT("captured_button_press");
	signals[SIGNAL_CAPTURED_SCROLL] = SIGNAL_NEW_EVENT("captured_scroll");
	signals[SIGNAL_CAPTURED_BUTTON_RELEASE] = SIGNAL_NEW_EVENT("captured_button_release");
	signals[SIGNAL_CAPTURED_MOTION_NOTIFY] = SIGNAL_NEW_EVENT("captured_motion_notify");
	signals[SIGNAL_CAPTURED_DELETE] = SIGNAL_NEW_EVENT("captured_delete");
	signals[SIGNAL_CAPTURED_DESTROY] = SIGNAL_NEW_EVENT("captured_destroy");
	signals[SIGNAL_CAPTURED_KEY_PRESS] = SIGNAL_NEW_EVENT("captured_key_press");
	signals[SIGNAL_CAPTURED_KEY_RELEASE] = SIGNAL_NEW_EVENT("captured_key_release");
	signals[SIGNAL_CAPTURED_ENTER_NOTIFY] = SIGNAL_NEW_EVENT("captured_enter_notify");
	signals[SIGNAL_CAPTURED_LEAVE_NOTIFY] = SIGNAL_NEW_EVENT("captured_leave_notify");
	signals[SIGNAL_CAPTURED_FOCUS_IN] = SIGNAL_NEW_EVENT("captured_focus_in");
	signals[SIGNAL_CAPTURED_FOCUS_OUT] = SIGNAL_NEW_EVENT("captured_focus_out");
	signals[SIGNAL_CAPTURED_CONFIGURE] = SIGNAL_NEW_EVENT("captured_configure");
	signals[SIGNAL_CAPTURED_MAP] = SIGNAL_NEW_EVENT("captured_map");
	signals[SIGNAL_CAPTURED_UNMAP] = SIGNAL_NEW_EVENT("captured_unmap");
	signals[SIGNAL_CAPTURED_WINDOW_STATE] = SIGNAL_NEW_EVENT("captured_window_state");
	signals[SIGNAL_CAPTURED_PROPERTY_NOTIFY] = SIGNAL_NEW_EVENT("captured_property_notify");
	signals[SIGNAL_CAPTURED_SELECTION_CLEAR] = SIGNAL_NEW_EVENT("captured_selection_clear");
	signals[SIGNAL_CAPTURED_SELECTION_REQUEST] = SIGNAL_NEW_EVENT("captured_selection_request");
	signals[SIGNAL_CAPTURED_SELECTION_NOTIFY] = SIGNAL_NEW_EVENT("captured_selection_notify");
	signals[SIGNAL_CAPTURED_PROXIMITY_IN] = SIGNAL_NEW_EVENT("captured_proximity_in");
	signals[SIGNAL_CAPTURED_PROXIMITY_OUT] = SIGNAL_NEW_EVENT("captured_proximity_out");
	signals[SIGNAL_CAPTURED_NO_EXPOSE] = SIGNAL_NEW_EVENT("captured_no_expose");
	signals[SIGNAL_CAPTURED_CLIENT] = SIGNAL_NEW_EVENT("captured_client");
	signals[SIGNAL_CAPTURED_EXPOSE] = SIGNAL_NEW_EVENT("captured_expose");
	signals[SIGNAL_CAPTURED_VISIBILITY_NOTIFY] = SIGNAL_NEW_EVENT("captured_visibility_notify");
	signals[SIGNAL_CAPTURED_GRAB_BROKEN] = SIGNAL_NEW_EVENT("captured_grab_broken");
	signals[SIGNAL_CAPTURED_EVENT_AFTER] = SIGNAL_NEW_EVENT("captured-event-after");

	miaouw_event_handler_append(event_handler, NULL);
}

static void miaouw_event_box_init(GTypeInstance* instance, gpointer klass) {
	MiaouwEventBox* event_box = MIAOUW_EVENT_BOX(instance);

	event_box->priv = g_new0(MiaouwEventBoxPrivate, 1);
		GTK_WIDGET_SET_FLAGS(event_box, GTK_NO_WINDOW);
}

GType miaouw_event_box_get_type() {
	static GType type = 0;
	static const GTypeInfo info = {
		sizeof (MiaouwEventBoxClass),
		NULL,   /* base_init */
		NULL,   /* base_finalize */
		miaouw_event_box_class_init,
		NULL,   /* class_finalize */
		NULL,   /* class_data */
		sizeof (MiaouwEventBox),
		0,
		miaouw_event_box_init,
		NULL
	};

	if (!type) {
		type = g_type_register_static(GTK_TYPE_BIN, "MiaouwEventBox", &info, 0);
	}

	return type;
}

GtkWidget* miaouw_event_box_new() {
	return gtk_widget_new(MIAOUW_TYPE_EVENT_BOX, NULL);
}

GtkWidget* miaouw_event_box_new_with_window() {
	GtkWidget* event_box;
	
	event_box = gtk_widget_new(MIAOUW_TYPE_EVENT_BOX, NULL);
	MIAOUW_EVENT_BOX(event_box)->priv->has_window = TRUE;
	
	return event_box;
}
