/*
 *  Miaouw - The Miaouw Library for Maemo Development
 *  Copyright (C) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
 
#include <glib.h>
#include <gtk/gtk.h>

#include <miaouw/miaouwbox.h>

/**
 * SECTION:miaouwbox
 * @short_description: A simple but powerful container
 * @see_also: #GtkBox
 *
 * The #MiaouwBox is similar container than the GtkBox but simpler.
 *
 * The #MiaouwBox is more suitable when childs are added on to the start or to the middle of the container
 * (see #miaouw_box_add_before and #miaouw_box_add_after).
 **/

struct _MiaouwBoxPrivate {
	gboolean horizontal;
	GList* children;
};

static void miaouw_box_add(GtkContainer* container, GtkWidget* child);
static GType miaouw_box_child_type(GtkContainer* container);
static void miaouw_box_forall(GtkContainer* container, gboolean include_internals, GtkCallback callback, gpointer callback_data);
static void miaouw_box_remove(GtkContainer* container, GtkWidget* child);
static void miaouw_box_size_allocate(GtkWidget* widget, GtkAllocation* allocation);
static void miaouw_box_size_request(GtkWidget* widget, GtkRequisition* requisition);

static void miaouw_box_class_init(gpointer klass, gpointer data) {
	GtkContainerClass *container_class;
	GtkWidgetClass *widget_class;

	container_class = GTK_CONTAINER_CLASS(klass);
	container_class->add = miaouw_box_add;
	container_class->remove = miaouw_box_remove;
	container_class->forall = miaouw_box_forall;
	container_class->child_type = miaouw_box_child_type;

	widget_class = GTK_WIDGET_CLASS(klass);
	widget_class->size_request = miaouw_box_size_request;
	widget_class->size_allocate = miaouw_box_size_allocate;
}

static void miaouw_box_init(GTypeInstance* instance, gpointer klass) {
	MiaouwBox* box = MIAOUW_BOX(instance);

	box->priv = g_new0(MiaouwBoxPrivate, 1);
	GTK_WIDGET_SET_FLAGS(box, GTK_NO_WINDOW);
	gtk_widget_set_redraw_on_allocate (GTK_WIDGET(box), FALSE);
}

GType miaouw_box_get_type() {
	static GType type = 0;
	static const GTypeInfo info = {
		sizeof (MiaouwBoxClass),
		NULL,   /* base_init */
		NULL,   /* base_finalize */
		miaouw_box_class_init,
		NULL,   /* class_finalize */
		NULL,   /* class_data */
		sizeof (MiaouwBox),
		0,
		miaouw_box_init,
		NULL
	};

	if (!type) {
		type = g_type_register_static(GTK_TYPE_CONTAINER, "MiaouwBox", &info, 0);
	}

	return type;
}

GtkWidget* miaouw_box_new_horizontal() {
	GtkWidget* box;
	
	box = gtk_widget_new(MIAOUW_TYPE_BOX, NULL);
	MIAOUW_BOX(box)->priv->horizontal = TRUE;
	
	return box;
}

GtkWidget* miaouw_box_new_vertical() {
	return gtk_widget_new(MIAOUW_TYPE_BOX, NULL);
}

void miaouw_box_add_before(MiaouwBox* box, GtkWidget* child, GtkWidget* existing_child) {
	GList* existing_child_in_list;

	g_return_if_fail(MIAOUW_IS_BOX(box));
	g_return_if_fail(GTK_IS_WIDGET(child));
	g_return_if_fail(child->parent == NULL);
	g_return_if_fail(existing_child == NULL || GTK_IS_WIDGET(existing_child));
	g_return_if_fail(existing_child == NULL || existing_child->parent == (GtkWidget*)box);
	
	if (existing_child) {
		existing_child_in_list = g_list_find(box->priv->children, existing_child);
		g_return_if_fail(existing_child_in_list != NULL);
	
		box->priv->children = g_list_insert_before(box->priv->children, existing_child_in_list, child);
	} else {
		box->priv->children = g_list_prepend(box->priv->children, child);
	}
	gtk_widget_set_parent(child, GTK_WIDGET(box));
	
	if (GTK_WIDGET_VISIBLE(child) && GTK_WIDGET_VISIBLE(box)) {
		gtk_widget_queue_resize(child);
	}
}

void miaouw_box_add_after(MiaouwBox* box, GtkWidget* child, GtkWidget* existing_child) {
	GList* existing_child_in_list;

	g_return_if_fail(MIAOUW_IS_BOX(box));
	g_return_if_fail(GTK_IS_WIDGET(child));
	g_return_if_fail(child->parent == NULL);
	g_return_if_fail(existing_child == NULL || GTK_IS_WIDGET(existing_child));
	g_return_if_fail(existing_child == NULL || existing_child->parent == (GtkWidget*)box);

	if (existing_child) {
		existing_child_in_list = g_list_find(box->priv->children, existing_child);
		g_return_if_fail(existing_child_in_list != NULL);

		if ((existing_child_in_list = g_list_next(existing_child_in_list))) {
			box->priv->children = g_list_insert_before(box->priv->children, existing_child_in_list, child);
		} else {
			box->priv->children = g_list_append(box->priv->children, child);
		}
	} else {
		box->priv->children = g_list_append(box->priv->children, child);
	}
	gtk_widget_set_parent(child, GTK_WIDGET(box));
	
	if (GTK_WIDGET_VISIBLE(child) && GTK_WIDGET_VISIBLE(box)) {
		gtk_widget_queue_resize(child);
	}
}

static void miaouw_box_add(GtkContainer* container, GtkWidget* child) {
	miaouw_box_add_after(MIAOUW_BOX(container), child, NULL);
}

static GType miaouw_box_child_type(GtkContainer* container) {
	return GTK_TYPE_WIDGET;
}

static void miaouw_box_forall(GtkContainer* container, gboolean include_internals, GtkCallback callback, gpointer callback_data) {
	MiaouwBox* box;
	
	box = MIAOUW_BOX(container);
	g_return_if_fail(callback != NULL);
	
	g_list_foreach(box->priv->children, (GFunc)callback, callback_data);
}

static void miaouw_box_remove(GtkContainer* container, GtkWidget* child) {
	MiaouwBox* box;
	gboolean was_visible;
	
	box = MIAOUW_BOX(container);
	g_return_if_fail(GTK_IS_WIDGET(child));
	g_return_if_fail(child->parent == (GtkWidget*)box);
	
	was_visible = GTK_WIDGET_VISIBLE(child);

	box->priv->children = g_list_remove(box->priv->children, child);
	gtk_widget_unparent(child);

	if (was_visible) {
		gtk_widget_queue_resize(GTK_WIDGET(container));
	}
}

static void miaouw_box_size_allocate(GtkWidget* widget, GtkAllocation* allocation) {
	MiaouwBox* box;
	GList* children;
	GtkWidget* child;
	GtkAllocation child_allocation;
	GtkRequisition child_requisition;
	
	box = MIAOUW_BOX(widget);

	child_allocation.x = allocation->x + GTK_CONTAINER(widget)->border_width;
	child_allocation.y = allocation->y + GTK_CONTAINER(widget)->border_width;
	if (box->priv->horizontal) {
		child_allocation.height = MAX(1, (gint)allocation->height - (gint)GTK_CONTAINER(widget)->border_width * 2);
	} else {
		child_allocation.width = MAX(1, (gint)allocation->width - (gint)GTK_CONTAINER(widget)->border_width * 2);
	}

	for (children = box->priv->children; children; children = children->next) {
		child = (GtkWidget*)children->data;
		if (GTK_WIDGET_VISIBLE(child)) {
			gtk_widget_get_child_requisition(child, &child_requisition);
			if (box->priv->horizontal) {
				child_allocation.width = child_requisition.width;
				gtk_widget_size_allocate(child, &child_allocation);
				child_allocation.x += child_requisition.width;
			} else {
				child_allocation.height = child_requisition.height;
				gtk_widget_size_allocate(child, &child_allocation);
				child_allocation.y += child_requisition.height;		
			}
		}
	}
	
	widget->allocation = *allocation;
}

static void miaouw_box_size_request(GtkWidget* widget, GtkRequisition* requisition) {
	MiaouwBox* box;
	GList* children;
	GtkRequisition child_requisition;
	GtkWidget* child;
	
	box = MIAOUW_BOX(widget);
	
	requisition->width = requisition->height = 0;
	for (children = box->priv->children; children; children = children->next) {
		child = (GtkWidget*)children->data;
		if (GTK_WIDGET_VISIBLE(child)) {
			gtk_widget_size_request(child, &child_requisition);
			if (box->priv->horizontal) {
				requisition->width += child_requisition.width;
				requisition->height = MAX(requisition->height, child_requisition.height);
			} else {
				requisition->height += child_requisition.height;
				requisition->width = MAX(requisition->width, child_requisition.width);			
			}
		}
	}
	requisition->width += GTK_CONTAINER(widget)->border_width * 2;
	requisition->height += GTK_CONTAINER(widget)->border_width * 2;
}
