/*
 *  Miaouw - The Miaouw Library for Maemo Development
 *  Copyright (C) 2007-2008 Henrik Hedberg <hhedberg@innologies.fi>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libosso.h>
#include <signal.h>
#include <unistd.h>
#include <execinfo.h>
#include <time.h>

extern char **environ;

#include <miaouw/miaouw.h>
#include <miaouw/miaouwprivate.h>

#define __USE_GNU
#include <sys/ucontext.h>


typedef struct {
	MiaouwEventHandler event_handler;
	gpointer user_data;
} EventHandlerData;

struct _MiaouwEventHandlerState {
	GList* current_event_handler;
};

#define BACKTRACE_UPLOADER_DATA_SIZE 1024

gchar backtrace_uploader_data[BACKTRACE_UPLOADER_DATA_SIZE];
guint backtrace_uploader_max_log_lines = 0;
GList* backtrace_uploader_log_lines = NULL;
time_t backtrace_uploader_start_time = 0;
osso_context_t* miaouw_osso_context;
GList* event_handlers;

static void event_handler_func(GdkEvent* event, gpointer data);
static void log_handler_func(const gchar* log_domain, GLogLevelFlags log_level, const gchar* message, gpointer user_data);
static void signal_handler(int signum, siginfo_t* siginfo, void* ucontext);

/**
 * SECTION:miaouw
 * @short_description: The Miaouw Library
 *
 * This is the Miaouw Library.
 *
 **/
 

void miaouw_init(osso_context_t* osso_context) {
	miaouw_osso_context = osso_context;
	event_handlers = NULL;
}

void miaouw_backtrace_uploader_init(const gchar* application_path, const gchar* application_name,
                                    const gchar* application_version, const gchar* upload_url, const guint max_log_lines) {
	gsize length;
	gsize str_len;
	gchar* p;
	gchar checksum;
	stack_t stack;
	struct sigaction sa;

	length = 0;
	if ((str_len = strlen(application_path) + 1) < BACKTRACE_UPLOADER_DATA_SIZE - length) {
		memcpy(backtrace_uploader_data + length, application_path, str_len);
		length += str_len;
	} else {
		g_warning("MiaouwBacktraceUploader initializing failed: application path does not fit into a data area.");

		return;
	}
	if ((str_len = strlen(application_name) + 1) < BACKTRACE_UPLOADER_DATA_SIZE - length) {
		memcpy(backtrace_uploader_data + length, application_name, str_len);
		length += str_len;
	} else {
		g_warning("MiaouwBacktraceUploader initializing failed: application name does not fit into a data area.");

		return;
	}
	if ((str_len = strlen(application_version) + 1) < BACKTRACE_UPLOADER_DATA_SIZE - length) {
		memcpy(backtrace_uploader_data + length, application_version, str_len);
		length += str_len;
	} else {
		g_warning("MiaouwBacktraceUploader initializing failed: application version does not fit into a data area.");

		return;
	}
	if ((str_len = strlen(upload_url) + 1) < BACKTRACE_UPLOADER_DATA_SIZE - length) {
		memcpy(backtrace_uploader_data + length, upload_url, str_len);
		length += str_len;
	} else {
		g_warning("MiaouwBacktraceUploader initializing failed: upload url does not fit into a data area.");

		return;
	}
	if (length < BACKTRACE_UPLOADER_DATA_SIZE) {
		checksum = 0;
		for (p = backtrace_uploader_data; p < backtrace_uploader_data + length; p++) {
			checksum += *p;
		}
		backtrace_uploader_data[length] = checksum;
	} else {
		g_warning("MiaouwBacktraceUploader initializing failed: checksum does not fit into a data area.");

		return;
	}
	
	sa.sa_sigaction = signal_handler;
	sigemptyset(&sa.sa_mask);
	sa.sa_flags = SA_SIGINFO;
	if (sigaction(SIGILL, &sa, NULL) == -1 ||
	    sigaction(SIGABRT, &sa, NULL) == -1 ||
	    sigaction(SIGFPE, &sa, NULL) == -1 ||
	    sigaction(SIGSEGV, &sa, NULL) == -1 ||
	    sigaction(SIGPIPE, &sa, NULL) == -1) {
		g_warning("MiaouwBacktraceUploader initializing failed: could not set signal handler.");
	}

	backtrace_uploader_max_log_lines = max_log_lines;
	if (max_log_lines > 0) {
		g_log_set_default_handler(log_handler_func, NULL);	
	}
	
	time(&backtrace_uploader_start_time);
}

void miaouw_event_handler_prepend(MiaouwEventHandler event_handler, gpointer user_data) {
	EventHandlerData* data;
	
	data = g_new0(EventHandlerData, 1);
	data->event_handler = event_handler;
	data->user_data = user_data;
	event_handlers = g_list_prepend(event_handlers, data);

	gdk_event_handler_set((GdkEventFunc)event_handler_func, NULL, NULL);
}

void miaouw_event_handler_append(MiaouwEventHandler event_handler, gpointer user_data) {
	EventHandlerData* data;

	data = g_new0(EventHandlerData, 1);
	data->event_handler = event_handler;
	data->user_data = user_data;
	event_handlers = g_list_append(event_handlers, data);

	gdk_event_handler_set((GdkEventFunc)event_handler_func, NULL, NULL);
}

void miaouw_event_handler_remove(MiaouwEventHandler event_handler, gpointer user_data) {
	GList* item;
	
	for (item = g_list_first(event_handlers); item; item = g_list_next(item)) {
		if (((EventHandlerData*)item->data)->event_handler == event_handler) {
			item = g_list_remove_link(event_handlers, item);
			g_free(item->data);
			g_list_free_1(item);
		}
	}
}

void miaouw_event_handler_next(GdkEvent* event, MiaouwEventHandlerState* state) {
	EventHandlerData* data;
	gboolean stop_propagating;
	
	state->current_event_handler = g_list_next(state->current_event_handler);
	if (state->current_event_handler) {
		data = (EventHandlerData*)state->current_event_handler->data;
		stop_propagating = data->event_handler(event, state, data->user_data);
		if (!stop_propagating && state->current_event_handler) {
			g_critical("miaouw_event_handler: handler returned false without calling miaouw_event_handler_next first");		
		}
	} else {
		gtk_main_do_event(event);
	}
}

static void event_handler_func(GdkEvent* event, gpointer user_data) {
	MiaouwEventHandlerState* state;
	EventHandlerData* data;
	gboolean stop_propagating;

	state = g_new0(MiaouwEventHandlerState, 1);
	state->current_event_handler = g_list_first(event_handlers);
	if (state->current_event_handler) {
		data = (EventHandlerData*)state->current_event_handler->data;
		stop_propagating = data->event_handler(event, state, data->user_data);
		if (!stop_propagating && state->current_event_handler) {
			g_critical("miaouw_event_handler: handler returned false without calling miaouw_event_handler_next first");		
		}
	} else {
		gtk_main_do_event(event);
	}

	g_free(state);
}

static void log_handler_func(const gchar* log_domain, GLogLevelFlags log_level, const gchar* message, gpointer user_data) {
	gchar* line;
	gchar* level;
	gchar buffer[256];
	GList* list;
	time_t t;
	void* bt[500];
	size_t size;
	gchar** strings;
	int i;
	
	size = backtrace(bt, 500);
	if ((strings = backtrace_symbols(bt, size))) {
		for (i = size - 1; i > 2; i--) {
			backtrace_uploader_log_lines = g_list_prepend(backtrace_uploader_log_lines, g_strconcat(strings[i], "\n", NULL));
		}
		free(strings);
	}

	switch (log_level) {
		case G_LOG_LEVEL_ERROR: level = "Error"; break;
		case G_LOG_LEVEL_CRITICAL: level = "Critical"; break;
		case G_LOG_LEVEL_WARNING: level = "Warning"; break;
		case G_LOG_LEVEL_MESSAGE: level = "Message"; break;
		case G_LOG_LEVEL_INFO: level = "Info"; break;
		case G_LOG_LEVEL_DEBUG: level = "Debug"; break;
		default: level ="(unknown)"; break;
	}
	time(&t);
	if (strftime(buffer, 256, "%d%m%Y %H%M%S", localtime(&t)) == 0) {
		strcpy(buffer, "(unknown)");
	}
	line = g_strdup_printf("%s %s %s: %s\n", buffer,
	                       level, log_domain, message);
	backtrace_uploader_log_lines = g_list_prepend(backtrace_uploader_log_lines, line);

	while (g_list_length(backtrace_uploader_log_lines) > backtrace_uploader_max_log_lines) {
		list = g_list_last(backtrace_uploader_log_lines);
		g_free(list->data);
		backtrace_uploader_log_lines = g_list_remove_link(backtrace_uploader_log_lines, list);
		g_list_free_1(list);
	}
	
	g_log_default_handler(log_domain, log_level, message, user_data);
}

static void signal_handler(int signum, siginfo_t* siginfo, void* ucontext) {
	gchar* data[6];
	gint counter;
	gchar checksum;
	gchar* p;
	int pipe_fds[2];
	pid_t pid;
	void* bt[500];
	size_t size;
	time_t t;
	gchar buffer[256];
	GList* list;

	counter = 0;
	data[counter++] = "miaouw-backtrace-uploader";
	data[counter++] = backtrace_uploader_data;
	checksum = 0;
	for (p = backtrace_uploader_data; counter < 6 && p < backtrace_uploader_data + BACKTRACE_UPLOADER_DATA_SIZE; p++) {
		if (!*p) {
			data[counter++] = p + 1;
		}
		checksum += *p;
	}
	if (counter < 6 || *p != checksum) {
		exit(2);
	}
	data[5] = 0;
	
	if (pipe(pipe_fds) == -1 || (pid = fork()) == -1) {
		exit(3);
	}
	if (pid) {
		close(pipe_fds[0]);
		size = backtrace(bt, 500);

#if defined __i386__
		bt[1] = (void*)((ucontext_t*)ucontext)->uc_mcontext.gregs[REG_EIP];
#elif defined __arm__
		bt[1] = (void*)((struct sigcontext*)(&((ucontext_t*)ucontext)->uc_mcontext))->arm_pc;
#else
#error Unknown architecture. Only i386 and arm are supported (in GCC).
#endif

		time(&t);
		if (strftime(buffer, 256, "%d%m%Y %H%M%S", localtime(&t)) == 0) {
			strcpy(buffer, "(unknown)");
		}
		strcat(buffer, " crashed\n");
		write(pipe_fds[1], buffer, strlen(buffer));

		if (size <= 1) {
			size = 2;
		}
		backtrace_symbols_fd(bt + 1, size - 1, pipe_fds[1]);

		for (list = g_list_first(backtrace_uploader_log_lines); list; list = g_list_next(list)) {
			write(pipe_fds[1], list->data, strlen(list->data));
		}
		if (strftime(buffer, 256, "%d%m%Y %H%M%S", localtime(&backtrace_uploader_start_time)) == 0) {
			strcpy(buffer, "(unknown)");
		}
		strcat(buffer, " started\n");
		write(pipe_fds[1], buffer, strlen(buffer));
		close(pipe_fds[1]);
	} else if (dup2(pipe_fds[0], 0) != -1) {
		close(pipe_fds[1]);
		close(pipe_fds[0]);
		execve("/usr/bin/miaouw-backtrace-uploader", data, environ);
	}
	
	exit(1);
}

