/* 
 * Copyright (c) 2006, Jakub Pavelek <jpavelek@welho.com>
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *        documentation and/or other materials provided with the distribution.
 *     * Neither the name of Jakub Pavelek nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gerror.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <libintl.h>
#include <locale.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <assert.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <libosso.h>
#include <sched.h>
#include <linux/fb.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <jpeglib.h>
#include <setjmp.h>



#include <hildon/hildon-help.h>
#include <hildon/hildon-banner.h>
#include <libhildondesktop/hildon-status-bar-item.h>
#include <hildon/hildon-number-editor.h>
#include <gtk/gtkenums.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

#include "pidinfo.h"
#include "pid_ui.h"

#define _(String) dgettext (PACKAGE, String)

#define HILDON_STATUS_BAR_BORDER_WIDTH 20
#define HILDON_STATUS_BAR_MENU_Y_POS 60

#define HILDON_STATUS_BAR_LOAD_PRIORITY 2

#define ICON_DIR  "/usr/share/pixmaps/"
#define ICON_FRAME ICON_DIR"la_frame.png"
#define ICON_CPU ICON_DIR"la_addon_cpu.png"
#define ICON_MEM ICON_DIR"la_addon_mem.png"
#define ICON_SWAP ICON_DIR"la_addon_swap.png"
#define ICON_BAR2 ICON_DIR"la_in_bar_2.png"
#define ICON_BAR3 ICON_DIR"la_in_bar_3.png"
#define ICON_BAR4 ICON_DIR"la_in_bar_4.png"
#define ICON_TEST ICON_DIR"la_test.png"

#define CPUFILE "/proc/stat"
#define MEMFILE "/proc/meminfo"
#define SWAPFILE "/proc/swaps" 

#define MEMORY_LEVEL_LOW 0.8
#define MEMORY_LEVEL_MID 0.6
#define MEMORY_LEVEL_HIGH 0.5

#define CPU_LOW 30
#define CPU_MID 60
#define CPU_HIGH 90



typedef struct
{
    HildonStatusBarItem *item;
    /* Pixbufs for the loaded icons */
    GdkPixbuf *pixbuf_frame;
    GdkPixbuf *pixbuf_cpu;
    GdkPixbuf *pixbuf_mem;
    GdkPixbuf *pixbuf_swap;
    GdkPixbuf *pixbuf_bar2;
    GdkPixbuf *pixbuf_bar3;
    GdkPixbuf *pixbuf_bar4;
    GdkPixbuf *pixbuf_current; /* We compose into this one */
    /* GUI widgets */
    GtkWidget *icon;
    GtkWidget *button;
    GtkWidget *menu;
    GtkWidget *menu_separator;
    GtkWidget *menu_screenshot;
    GtkWidget *menu_delayed_screenshot;
    GtkWidget *menu_list_processes;
    GtkWidget *menu_settings;
    /* Flags */
    gboolean in_area;
    gboolean on_border;
    gboolean on_button;
    gboolean button_released;
	gboolean llscreenshots;
    /* Internal data */
    guchar last_cpu_level;
    guchar last_mem_level;
    gboolean is_swap_used;
    guint timeout_id;
    osso_context_t *osso_context;
} PluginInfo;


static gint lastU, lastN, lastIO, lastI;
gint deltaU, deltaN, deltaIO, deltaI;
static gint sshotn = 0;
static PluginInfo *info = NULL;
static gboolean im_inactive = FALSE;
#define MAX_READ_CHARS 128
static char read_buffer[MAX_READ_CHARS];
#define MAX_SSHOT_NAME_LEN 64
#define SSHOT_DEFAULT_NAME "screenshot"
static gchar *sshotfilename = NULL;
static char sshotname[MAX_SSHOT_NAME_LEN];
static gint screenshot_delay = 7;
static int swap_used;
        

/* Hildon Status Bar plugin API prototypes */
void *load_initialize(HildonStatusBarItem *item, GtkWidget **button);
void load_entry(HildonStatusBarPluginFn_st *fn);
void load_destroy(void *data);
gint load_get_priority(void *data);
void load_update(void *data, gint value1, gint value2, const gchar *str);
static void hildon_im_hw_cb(osso_hw_state_t*, gpointer);

/* Internal function prototypes */
static gboolean check_load_cpu (gpointer);
static GdkPixbuf* la_load_pixmap(const gchar*);
static void la_load_pixmaps(void);


static void close_window()
{
    gtk_menu_popdown(GTK_MENU(info->menu));
    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->button), FALSE);
}

static gboolean key_pressed(GtkWidget *widget, GdkEventKey *event,
                            gpointer data)
{
    switch(event->keyval) 
    {
    case GDK_F5:      /* Hardware home key */
        close_window();
        return FALSE;
    default:
        return FALSE;
    }
}

static void 
get_pointer_location(GtkWidget *widget)
{
    gint x, y;
    gint w = GTK_WIDGET(info->menu)->allocation.width;
    gint h = GTK_WIDGET(info->menu)->allocation.height;
    
    gtk_widget_get_pointer(GTK_WIDGET(info->menu), &x, &y);
    
    info->on_button = FALSE;
    
    /* Pointer on menu area */
    if ((x >= 0) && (x <= w) && (y >= 0) && (y <= h))
    {
        info->in_area = TRUE;
   
        if (x < HILDON_STATUS_BAR_BORDER_WIDTH
            || x > w-HILDON_STATUS_BAR_BORDER_WIDTH 
            || y < HILDON_STATUS_BAR_BORDER_WIDTH 
            || y > h-HILDON_STATUS_BAR_BORDER_WIDTH)
        {
            info->on_border = TRUE;  
        }
        else
        {
            info->on_border = FALSE;
        }
    }
    else if(!info->button_released)
    {
        w = GTK_WIDGET(info->button)->allocation.width;
        h =  GTK_WIDGET(info->button)->allocation.height;
        
        gtk_widget_get_pointer(GTK_WIDGET(info->button), &x, &y);
        
        /* Pointer on button area */
        if ((x >= 0) && (x <= w) && (y >= 0) && (y <= h))
        {   
            info->in_area = TRUE;
            info->on_button = TRUE;
        }
        else
            info->in_area = FALSE;
    }    
    
    /* Pointer out of menu or button area */
    else
        info->in_area = FALSE;
    
    info->button_released = TRUE;

}

static gboolean 
window_event(GtkWidget *widget,
             GdkEvent *event,
             gpointer data)
{
    g_return_val_if_fail(event, FALSE);
    g_return_val_if_fail(data, FALSE);
    
    get_pointer_location(widget);
    
    if (info->in_area)
    {
        if (info->on_border || info->on_button)
            return TRUE;
        else
            return FALSE;
    }
    
    close_window();

    return TRUE;
}

/* Funtion needed to stop others signal handlers */              
static gboolean 
button_press(GtkWidget *widget, 
             GdkEventButton *event,
             gpointer data)
{   
    return TRUE;
}

static void 
pos_menu(GtkMenu *menu, gint *x, gint *y, 
         gboolean *in, gpointer data)
{
    GtkAllocation *a, *ma;
    GtkWidget *toplevel;
    gint tx, ty;

    g_return_if_fail(x && y);
    g_return_if_fail(data);
    g_return_if_fail(GTK_IS_WIDGET(((PluginInfo*)(data))->item));

    a = &(GTK_WIDGET(((PluginInfo*)(data))->item)->allocation);

    gtk_widget_realize(GTK_WIDGET(menu));
    ma = &GTK_WIDGET(menu)->allocation;

    toplevel = 
        gtk_widget_get_toplevel(GTK_WIDGET(((PluginInfo*)(data))->item));

    if(GTK_WIDGET_TOPLEVEL(toplevel))
    {
	gdk_window_get_position(toplevel->window, &tx, &ty);
	*x = (tx + a->x + a->width) - ma->width;
    }
    *y = HILDON_STATUS_BAR_MENU_Y_POS;
}

static int
get_next_sshot (gint nr)
{
	FILE *f;
	gint n;

	n = nr;
	f = NULL;
	do {
		g_snprintf (sshotname, MAX_SSHOT_NAME_LEN-1, "%s/MyDocs/.images/%s%02d.png", getenv("HOME"), sshotfilename, n);
		f = g_fopen (sshotname, "r");
		if (f) {
			fclose (f); /* What is the Glib version of this? */
			n++;
		}
	}
	while (f != NULL);
		
	return n;
}

static void
do_little_dance (void)
{
	gtk_menu_popdown(GTK_MENU(info->menu));
	gtk_widget_hide(info->button);
    
	while (gtk_events_pending())
	{
		gtk_main_iteration ();
	}

	usleep(1000);

	sched_yield();

	/* Doing this twice ... cause for some reason once is not always enough :( */
	while (gtk_events_pending())
	{
		gtk_main_iteration ();
	}

	usleep(1000);
}

static void
setup_filename (gchar *suffix)
{
	const char *home;
	
	home = getenv("HOME");
	sshotn = get_next_sshot (sshotn);
	g_snprintf(sshotname, MAX_SSHOT_NAME_LEN-1, "%s/MyDocs/.images/%s%02d.%s", home, sshotfilename, sshotn, suffix);
}

static void
make_screenshot(void)
{
	int width, height;
	GdkDrawable *root_window;
	GdkPixbuf *pixbuf;
	gboolean ret;
	GError *error = NULL;

	do_little_dance ();
	setup_filename ("png");


	root_window = gdk_get_default_root_window ();
	gdk_drawable_get_size (root_window, &width, &height);
	pixbuf = gdk_pixbuf_get_from_drawable (NULL,
                                           root_window,
                                           gdk_drawable_get_colormap (root_window),
                                           0, 0,
                                           0, 0,
                                           width, height);
      

    ret = gdk_pixbuf_save (pixbuf, sshotname, "png", &error, NULL);
	g_object_unref (pixbuf);
	sshotn++;

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->button), FALSE);

	gtk_widget_show(info->button);
}


/*
 * Low-level screenshot
 */
void
make_ll_screenshot (void)
{
	// Framebuffer stuff
	int fbfd = 0;
	struct fb_var_screeninfo vinfo;
	struct fb_fix_screeninfo finfo;
	long int screensize = 0;
	char *fbp = NULL;
	// Converted buffer stuff
	char *rgb = NULL;
	unsigned int i, j, red, green, blue;
	// JPEG stuff
	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;
	FILE * outfile;
	JSAMPROW row_pointer[1];
	int row_stride;
	
	do_little_dance ();
	setup_filename ("jpeg");
	
	/*
	 * Framebuffer opening and mmaping code
	 */
	fbfd = open("/dev/fb0", O_RDWR);
	if (!fbfd) 
	{
		goto errorhandler;
	}

	if (ioctl(fbfd, FBIOGET_FSCREENINFO, &finfo)) 
	{
		close (fbfd);
		goto errorhandler;	
	}

	if (ioctl(fbfd, FBIOGET_VSCREENINFO, &vinfo)) 
	{
		close (fbfd);
		goto errorhandler;
	}

	screensize = vinfo.xres * vinfo.yres * vinfo.bits_per_pixel / 8;
        
	fbp = (char*) mmap (0, screensize, PROT_READ, MAP_SHARED, fbfd, 0);       
	if ( (int) fbp == -1) 
	{  
		close (fbfd);
		goto errorhandler;
	}
	
	/*
	 * Duplicate and transform from RGB565 to RGB24
	 */
	rgb = (char*) malloc (vinfo.xres * vinfo.yres * 3);
	for (i = j = 0; i < (vinfo.xres * vinfo.yres * vinfo.bits_per_pixel/8); i += 2, j += 3) 
	{
		/* 00000000 11111000 -> 11111000 */
		red = (fbp[i+1] & 0xf8);
		/* 11100000 00000111 -> 11111100 */
		green = ((fbp[i] & 0xe0) >> 3) | ((fbp[i+1] & 0x07) << 5);
		/* 00011111 00000000 -> 11111000 */
		blue = ((fbp[i] & 0x1f) << 3);
		rgb[j] = red;
		rgb[j+1] = green;
		rgb[j+2] = blue;
	}
	
	/*
	 * JPEG saving code
	 */
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_compress(&cinfo);
	if ((outfile = fopen(sshotname, "wb")) == NULL) 
	{;
		goto errorlate;
	}
	jpeg_stdio_dest(&cinfo, outfile);
	cinfo.image_width = vinfo.xres;
	cinfo.image_height = vinfo.yres;
	cinfo.input_components = 3;
	cinfo.in_color_space = JCS_RGB;
	jpeg_set_defaults(&cinfo);
	//full quality - this is a screenshot!  jpeg_set_quality(&cinfo, 50, TRUE);
	jpeg_start_compress(&cinfo, TRUE);
	row_stride = vinfo.xres * 3 ;
	while (cinfo.next_scanline < cinfo.image_height) 
	{
		row_pointer[0] = & rgb [cinfo.next_scanline * row_stride];
		(void) jpeg_write_scanlines(&cinfo, row_pointer, 1);
	}
	jpeg_finish_compress(&cinfo);
	fclose(outfile);
	jpeg_destroy_compress(&cinfo);

errorlate:
	free (rgb);
	munmap(fbp, screensize);
	close(fbfd);
	
errorhandler:	
	sshotn++;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->button), FALSE);
	gtk_widget_show(info->button);
	
	return;
}



static void
activate_screenshot_item(GtkMenuItem *item, gpointer data)
{
	if (info->llscreenshots)
		make_ll_screenshot ();
	else
		make_screenshot();
}

static gboolean
make_screenshot_tcb(gpointer data)
{
	if (info->llscreenshots)
		make_ll_screenshot ();
	else
		make_screenshot();
	
	return FALSE; /* Do not call this again */
}


static void
activate_delayed_screenshot_item(GtkMenuItem *item, gpointer data)
{
	gtk_timeout_add(screenshot_delay*1000, make_screenshot_tcb, NULL);
}

static void
activate_settings_item(GtkMenuItem *item, gpointer data)
{
	GtkWidget *dialog;
	GtkWidget *button_accept, *button_cancel;
	GtkWidget *vbox, *hboxtop, *hboxbottom, *hboxll;
	GtkWidget *labeldelay, *labelfilename;
	GtkWidget *sb_seconds;
	GtkWidget *entry_filename;
	GtkWidget *llcheck;
	gint result;

	dialog = gtk_dialog_new_with_buttons ("Screenshot settings", NULL, GTK_DIALOG_MODAL, NULL);
	button_accept = gtk_dialog_add_button(GTK_DIALOG(dialog), "Accept", LOAD_DIALOG_ACCEPT);
	button_cancel = gtk_dialog_add_button(GTK_DIALOG(dialog), "Cancel", LOAD_DIALOG_CANCEL);

	/* Now create the contents */
	vbox = gtk_vbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox, TRUE, TRUE, 0);
	/* Screenshot delay */
	hboxtop = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX(vbox), hboxtop, TRUE, TRUE, 2);
	labeldelay = gtk_label_new ("Screenshot delay (sec)");
	gtk_box_pack_start (GTK_BOX(hboxtop), labeldelay, TRUE, TRUE,5);
	sb_seconds = hildon_number_editor_new  (1, 60);	
	gtk_box_pack_start (GTK_BOX(hboxtop), sb_seconds, TRUE, TRUE, 0);
	hildon_number_editor_set_value (HILDON_NUMBER_EDITOR(sb_seconds), screenshot_delay);

	/* Screenshot name */
	hboxbottom = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX(vbox), hboxbottom, TRUE, TRUE, 2);
	labelfilename = gtk_label_new ("Screenshot filename");
	gtk_box_pack_start (GTK_BOX(hboxbottom), labelfilename, TRUE, TRUE, 5);
	entry_filename = gtk_entry_new_with_max_length(MAX_SSHOT_NAME_LEN - 5);
	gtk_entry_set_text(GTK_ENTRY(entry_filename), sshotfilename);
	gtk_box_pack_start (GTK_BOX(hboxbottom), entry_filename, TRUE, TRUE, 5);
	
	/* Low-level screenshot? */
	hboxll = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX(vbox), hboxll, TRUE, TRUE, 2);
	llcheck = gtk_check_button_new_with_label ("low-level screenshots?");
	gtk_box_pack_start (GTK_BOX(hboxll), llcheck, TRUE, TRUE, 5);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (llcheck), info->llscreenshots);

	
	hildon_help_dialog_help_enable (GTK_DIALOG(dialog), "Settings_LoadApplet_Main", info->osso_context);
	gtk_widget_show_all(dialog);
	result = gtk_dialog_run (GTK_DIALOG (dialog));
	switch (result)
	{
	case LOAD_DIALOG_CANCEL:
		gtk_widget_destroy(dialog);
		break;
	case LOAD_DIALOG_ACCEPT:
		/* Save new delay and screenshot name */
		if (sshotfilename) 
			g_free(sshotfilename);
		sshotfilename = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry_filename)));
		screenshot_delay = hildon_number_editor_get_value (HILDON_NUMBER_EDITOR(sb_seconds));
		info->llscreenshots = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (llcheck));
		gtk_widget_destroy(dialog);
		break;
	default:
		gtk_widget_destroy(dialog);
		break;
	}
}


static void 
deactivate_menu_cb(GtkMenuShell *menushell, gpointer data)
{
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->button), FALSE);
}

static void 
load_icon_pressed(GtkWidget *widget, gpointer data)
{
    if (!GTK_IS_MENU(info->menu)) 
    {
        info->menu = gtk_menu_new();

        /* The screenshot */
        info->menu_screenshot = gtk_menu_item_new_with_label("Take screenshot");
        gtk_menu_shell_append(GTK_MENU_SHELL(info->menu),
			info->menu_screenshot);
        g_signal_connect(G_OBJECT(info->menu_screenshot),
                         "activate",
                         G_CALLBACK(activate_screenshot_item),
                         data);
        /* Delayed screenshot */
        info->menu_delayed_screenshot = gtk_menu_item_new_with_label("Take delayed screenshot");
        gtk_menu_shell_append(GTK_MENU_SHELL(info->menu),
                        info->menu_delayed_screenshot);
        g_signal_connect(G_OBJECT(info->menu_delayed_screenshot),
                         "activate",
                         G_CALLBACK(activate_delayed_screenshot_item),
                         data);

        /* Separator */
        info->menu_separator = gtk_separator_menu_item_new();
        gtk_menu_shell_append(GTK_MENU_SHELL(info->menu),
                         info->menu_separator);
        /* List processes */
        info->menu_list_processes = gtk_menu_item_new_with_label("List processes");
        gtk_menu_shell_append(GTK_MENU_SHELL(info->menu),
                        info->menu_list_processes);
        g_signal_connect(G_OBJECT(info->menu_list_processes),
                        "activate",
                        G_CALLBACK(activate_list_processes_item),
                        data);
        /* Separator */
        info->menu_separator = gtk_separator_menu_item_new();
        gtk_menu_shell_append(GTK_MENU_SHELL(info->menu),
                         info->menu_separator);
        /* Settings */
        info->menu_settings = gtk_menu_item_new_with_label("Settings");
        gtk_menu_shell_append(GTK_MENU_SHELL(info->menu),
                        info->menu_settings);
        g_signal_connect(G_OBJECT(info->menu_settings),
                        "activate",
                        G_CALLBACK(activate_settings_item),
                        data);
        /* More signals to the menu */
        g_signal_connect(G_OBJECT(info->menu),
                         "key-press-event",
                         G_CALLBACK(key_pressed),
                         info);
  
        g_signal_connect(G_OBJECT(info->menu),
                         "button-release-event",
                         G_CALLBACK(window_event),
                         info);

        g_signal_connect(G_OBJECT(info->menu),
                         "button-press-event", 
                         G_CALLBACK(button_press),
                         info);

        g_signal_connect(G_OBJECT(info->menu),
                         "deactivate",
                         G_CALLBACK(deactivate_menu_cb),
                         info);

        gtk_widget_show_all(info->menu);
    }

    gtk_menu_shell_select_first(GTK_MENU_SHELL(info->menu), TRUE);
    
    gtk_menu_popup(GTK_MENU(info->menu), NULL, NULL, 
                   pos_menu, data, 
                   1/* left mouse button */,
                   gtk_get_current_event_time());

    g_signal_emit_by_name(G_OBJECT(info->button), "released");
    info->button_released = FALSE;
}

static void
blit_memory_bars (const guchar level)
{
    gdk_pixbuf_composite(info->pixbuf_frame, info->pixbuf_current, 18, 4, 16, 33, 18, 4, 1, 1, GDK_INTERP_NEAREST, 255);

    if (level >= 2)
        gdk_pixbuf_composite(info->pixbuf_bar2, info->pixbuf_current, 21, 21, 9, 5, 21, 21, 1, 1, GDK_INTERP_NEAREST, 255);
    if (level >= 3)
        gdk_pixbuf_composite(info->pixbuf_bar3, info->pixbuf_current, 21, 15, 9, 5, 21, 15, 1, 1, GDK_INTERP_NEAREST, 255);
    if (level == 4)
        gdk_pixbuf_composite(info->pixbuf_bar4, info->pixbuf_current, 21, 9, 9, 5, 21, 9, 1, 1, GDK_INTERP_NEAREST, 255);

    if (swap_used > 0)
    {
        gdk_pixbuf_composite(info->pixbuf_swap, info->pixbuf_current, 26, 22, 13, 17, 26, 22, 1, 1, GDK_INTERP_NEAREST, 255);
    } else {
        gdk_pixbuf_composite(info->pixbuf_mem, info->pixbuf_current, 26, 22, 13, 17, 26, 22, 1, 1, GDK_INTERP_NEAREST, 255);
    }
}

static void
blit_cpu_bars (const guchar level)
{
    gdk_pixbuf_composite(info->pixbuf_frame, info->pixbuf_current, 1, 4, 16, 33, 1, 4, 1, 1, GDK_INTERP_NEAREST, 255);

    if (level >= 2)
        gdk_pixbuf_composite(info->pixbuf_bar2, info->pixbuf_current, 4, 21, 9, 5, 4, 21, 1, 1, GDK_INTERP_NEAREST, 255);
    if (level >= 3)
        gdk_pixbuf_composite(info->pixbuf_bar3, info->pixbuf_current, 4, 15, 9, 5, 4, 15, 1, 1, GDK_INTERP_NEAREST, 255);
    if (level == 4)
        gdk_pixbuf_composite(info->pixbuf_bar4, info->pixbuf_current, 4, 9, 9, 5, 4, 9, 1, 1, GDK_INTERP_NEAREST, 255);
    
    gdk_pixbuf_composite(info->pixbuf_cpu, info->pixbuf_current, 7, 22, 13, 17, 7, 22, 1, 1,GDK_INTERP_NEAREST, 255);
}


static guchar
check_cpu (void)
{
	gint curU, curN, curIO, curI;
	int load, idle;
	GError *error = NULL;
	gchar *contents;
	gsize lenght;
	gchar **splits;

	if (!g_file_get_contents (CPUFILE, &contents, &lenght, &error)) {
		fprintf (stderr, "ERR: can't read file %s: %s\n", CPUFILE, error->message);
		g_error_free (error);
		return 0;
	}
	
	splits = g_strsplit_set (contents, " ",  -1);

	sscanf(splits[2] , "%d", &curU);
	sscanf(splits[3], "%d", &curN);
	sscanf(splits[4], "%d", &curIO);
	sscanf(splits[5], "%d", &curI);
	
	g_strfreev (splits);
	g_free (contents);
    
	idle = (curI - lastI);
	if (idle == 0) load = 100;
	else load = 100-idle;
	if (load>100) load = 0;
	deltaU = curU - lastU;
	deltaN = curN - lastN;
	deltaIO = curIO - lastIO;
	deltaI = curI - lastI;
	lastU = curU;
	lastN = curN;
	lastIO = curIO;
	lastI = curI;

	if (load > CPU_HIGH)
		return 4;
	else if (load > CPU_MID)
		return 3;
	else  if (load > CPU_LOW)
		return 2;
	else
		return 1;
}


static guchar
check_mem ()
{
    FILE *fin;
    int mem_used = 0;
    int mem_total = 0;
    int mem_cached = 0;
    int mem_buffers = 0;
    int mem_free = 0;
    int swap_total = 0;

    /* Open the memory info file and get current free memory */
    fin = fopen(MEMFILE, "r");
    if (fin == NULL)
    {
        g_warning("Can't open "MEMFILE"\n");
        return TRUE;
    }
    while (fgets(read_buffer, MAX_READ_CHARS, fin) != NULL)
    {
        if (strncmp(read_buffer, "MemTotal", 8) == 0)
        {
            sscanf(read_buffer + 10, "%d", &mem_total);
        }
        else if (strncmp(read_buffer, "MemFree", 6) == 0)
        {
            sscanf(read_buffer + 9, "%d", &mem_free);
        }
        else if (strncmp(read_buffer, "Buffers", 6) == 0)
        {
            sscanf(read_buffer + 9, "%d", &mem_buffers);
        }
        else if (strncmp(read_buffer, "Cached", 6) == 0)
        {
            sscanf(read_buffer + 8, "%d", &mem_cached);
            break;
        }
    }
    fclose(fin);
    mem_used = mem_total - mem_free - mem_buffers - mem_cached;

    /* If the SWAP is on and used,add its data to the calculation */
    fin = fopen(SWAPFILE, "r");
    if (fin != NULL)
    {
        while (fgets(read_buffer, MAX_READ_CHARS, fin) != NULL)
        {
            char* pointer = NULL;
            pointer = strstr(read_buffer, "/media/mmc");
            if (pointer) pointer = strstr(read_buffer, "file");
            if (pointer)
            {
                /* This is hard-coded for file based SWAP on Nokia 770, 
                 * not usual SWAP partitions */
                sscanf(pointer + strlen("file"), "%d %d", &swap_total, &swap_used);
                if (swap_used > 0)
                {
                    /* That swap used is misleading - can be non-zero even when SWAP is not used! */
                    mem_total = mem_total + swap_total;
                    mem_used = mem_used + swap_used;
                }
		break;
            }
        }
    fclose(fin);
    }
    if (mem_used > MEMORY_LEVEL_LOW*mem_total)
        return 4;
    else if (mem_used > MEMORY_LEVEL_MID*mem_total)
        return 3;
    else  if (mem_used > MEMORY_LEVEL_HIGH*mem_total)
        return 2;
    else
        return 1;
}


static gboolean
check_load_cpu (gpointer data)
{
    guchar current_cpu_level;
    guchar current_mem_level;
    gboolean currently_is_swap_used;
   
    current_cpu_level = check_cpu(); 
    current_mem_level = check_mem();
    currently_is_swap_used = swap_used > 0 ? FALSE : TRUE;

    if ((current_mem_level != info->last_mem_level) ||
        (current_cpu_level != info->last_cpu_level) ||
        (info->is_swap_used != currently_is_swap_used))
    {
        gdk_pixbuf_fill(info->pixbuf_current, 0x00000000);
        blit_memory_bars (current_mem_level);
        blit_cpu_bars (current_cpu_level);
        gtk_image_set_from_pixbuf(GTK_IMAGE(info->icon), info->pixbuf_current);
        info->last_mem_level = current_mem_level;
        info->last_cpu_level = current_cpu_level;
        info->is_swap_used = currently_is_swap_used; 
    }

    return TRUE;  
}




static void
hildon_im_hw_cb(osso_hw_state_t* state, gpointer data)
{
    /*
    (state->shutdown_ind)
    (state->save_unsaved_data_ind)
    (state->memory_low_ind)
    (state->system_inactivity_ind)
    */
    if ((im_inactive == TRUE) && (state->system_inactivity_ind == FALSE))
    {
        /* Time to wake up again */
        im_inactive = FALSE;
        info->timeout_id = gtk_timeout_add(1000, check_load_cpu, info);
    }
    if (state->system_inactivity_ind == TRUE)
    {
        /* Go sleep */
        im_inactive = TRUE;
        g_source_remove(info->timeout_id);
        info->timeout_id = 0;
    }
}


static GdkPixbuf*
la_load_pixmap(const gchar* location)
{
   GdkPixbuf *pixbuf;
   GError *error = NULL;

   pixbuf = gdk_pixbuf_new_from_file(location, &error); 
   if (error)
   {
       g_error("Loading %s failed with : %s", location, error->message);
       g_error_free (error);
   }
   return pixbuf;
}

static void
la_load_pixmaps()
{
    info->pixbuf_frame = la_load_pixmap(ICON_FRAME);
    info->pixbuf_cpu = la_load_pixmap(ICON_CPU);
    info->pixbuf_mem = la_load_pixmap(ICON_MEM);
    info->pixbuf_swap = la_load_pixmap(ICON_SWAP);
    info->pixbuf_bar2 = la_load_pixmap(ICON_BAR2);
    info->pixbuf_bar3 = la_load_pixmap(ICON_BAR3);
    info->pixbuf_bar4 = la_load_pixmap(ICON_BAR4);
    info->pixbuf_current = la_load_pixmap(ICON_TEST);
}


void*
load_initialize(HildonStatusBarItem *item,
                GtkWidget **button)
{
    osso_return_t status;

    info = g_new0(PluginInfo, 1);

    g_return_val_if_fail(info, NULL);
    g_return_val_if_fail(item, NULL);
    g_return_val_if_fail(button, NULL);

    bindtextdomain(PACKAGE, LOCALEDIR);

    info->item = item;
    info->icon = gtk_image_new_from_pixbuf(NULL);
    info->button = gtk_toggle_button_new();
                      
    gtk_container_add(GTK_CONTAINER(info->button),
                      GTK_WIDGET(info->icon));
    
    *button = info->button;
    
    g_signal_connect(G_OBJECT(info->button), "pressed",
                     G_CALLBACK(load_icon_pressed),
                     info);

    la_load_pixmaps();
    
    gtk_image_set_from_pixbuf(GTK_IMAGE(info->icon), info->pixbuf_current);
    gtk_widget_show_all(GTK_WIDGET(info->button));

    /* Libosso init */
    info->osso_context = osso_initialize("load-applet", VERSION, FALSE, NULL);
    if (!info->osso_context)
    {
        g_warning("Could not initialize osso from " PACKAGE);
    }
    status = osso_hw_set_event_cb(info->osso_context, NULL, hildon_im_hw_cb, NULL);
    if (status != OSSO_OK)
    {
        g_warning("Could not register the osso_hw_set_event_cb");
    }
    /* Install timeout handler for updating */
    info->timeout_id = gtk_timeout_add(1000, check_load_cpu, info);

    /* Screenshot stuff too */
    sshotfilename = g_strdup(SSHOT_DEFAULT_NAME);
    info->last_cpu_level = 0;
    info->last_mem_level = 0;
    info->is_swap_used = FALSE;
	info->llscreenshots = FALSE;
    
    return info;
}

void 
load_entry(HildonStatusBarPluginFn_st *fn)
{
    fn->initialize   = load_initialize;
    fn->update       = load_update;
    fn->destroy      = load_destroy;
    fn->get_priority = load_get_priority;
}

/* 
 * This function is a hack to get this working in Sardine :( 
 * Status Bar looks for load_entry in IT-2006 and for LoadApplet_entry in Sardine :(
 */
void
LoadApplet_entry(HildonStatusBarPluginFn_st *fn)
{
	load_entry(fn);
}

void 
load_destroy(void *data) 
{
    if (info->timeout_id)
        g_source_remove (info->timeout_id);

    g_object_unref (info->pixbuf_frame);
    g_object_unref (info->pixbuf_cpu);
    g_object_unref (info->pixbuf_mem);
    g_object_unref (info->pixbuf_swap);
    g_object_unref (info->pixbuf_bar2);
    g_object_unref (info->pixbuf_bar3);
    g_object_unref (info->pixbuf_bar4);
    g_object_unref (info->pixbuf_current);

   gtk_widget_destroy (info->menu);
    g_free (info);
    /* "data" is freed by Status Bar! */
}

/* if value1 == 1, the plugin is created, 
 * if value1 == 0 the widget is asked to be destroyed */
void 
load_update(void *data, gint value1, gint value2, const gchar *str)
{
    if (value1 == 0)
	gtk_widget_destroy(GTK_WIDGET(info->item));

}

gint 
load_get_priority(void *data)
{
    return HILDON_STATUS_BAR_LOAD_PRIORITY;
}
