/*
 * main.c
 */

#include <hildon/hildon-program.h>
#include <hildon/hildon-banner.h>
#include <gtk/gtk.h>

#include <glib.h>

#include <locale.h>
#include <libintl.h>
#define _(x)  gettext(x)

#include "selected.h"
#include "tileset.h"
#include "stack.h"
#include "appdata.h"
#include "menu.h"
#include "statusbar.h"

#include "gconf.h"

static gboolean
board_click_callback(GtkWidget *widget, GdkEventButton *event, 
		      gpointer data) {  
  appdata_t *appdata = (appdata_t*)data;
  GtkWidget *dialog;
  int x, y, state;

  if(appdata->game->state == STATE_IDLE) {
    hildon_banner_show_information(GTK_WIDGET(appdata->window), NULL, 
				   "No game running. Please select "
				   "\"New\" from the menu!");
    return TRUE;
  }

  if(appdata->game->state != STATE_HUMAN_PLACES) {
    hildon_banner_show_information(GTK_WIDGET(appdata->window), 
				   NULL, "Not your turn to place!");
    return TRUE;
  }

  tileset_get_slot(appdata->tileset, (int)event->x, (int)event->y, &x, &y);

  if(appdata->selected->tile != LOC_EMPTY) {

    /* only process if this is a valid click */
    if((x >= 0) && (y >= 0)) {
      if(game_get_tile_at(appdata->game, x, y) != LOC_EMPTY)
	hildon_banner_show_information(GTK_WIDGET(appdata->window), 
				       NULL, "Already occupied");
      else {
	/* inform game engine that the tile has been placed */
	game_place_tile(appdata->game, appdata->selected->tile, x, y);

	/* draw tile on board */
	tileset_draw_tile(appdata->tileset, appdata->selected->tile, x, y);

	/* remove tile from "selected" gui */
	selected_remove(appdata->selected);

	/* check for end of game */
	switch(state = game_check(appdata->game)) {
	case GAME_WON:
	case GAME_DRAW:
	  dialog = gtk_message_dialog_new(GTK_WINDOW(appdata->window),
		  GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_INFO,
		  GTK_BUTTONS_OK, 
	          (state == GAME_WON)?"Human wins!":"Draw game!");
  
	  gtk_dialog_run (GTK_DIALOG (dialog));
	  gtk_widget_destroy(dialog);
	  
	  appdata->game->state = STATE_IDLE;
	  statusbar_message_state(appdata->statusbar);
	  break;

	default:
	  /* now the human has to pick the next tile */
	  appdata->game->state = STATE_HUMAN_SELECTS;
	  statusbar_message_state(appdata->statusbar);
	}
      }
    }
  } else
    hildon_banner_show_information(GTK_WIDGET(appdata->window), 
				   NULL, "Nothing selected");


  return TRUE;
}

static GtkWidget *board_image_new(appdata_t *appdata) {
  GtkWidget *event_box;
  int x,y;
  
  /* create an event box and put image into it */
  event_box = gtk_event_box_new ();
  gtk_container_add(GTK_CONTAINER(event_box), 
		    GTK_WIDGET(appdata->tileset->board));
  
  g_signal_connect(G_OBJECT(event_box), "button_press_event",
		   G_CALLBACK (board_click_callback), appdata);

  /* draw initial table state */
  for(y=0;y<4;y++) 
    for(x=0;x<4;x++) 
      if(game_get_tile_at(appdata->game, x, y) != LOC_EMPTY)
	tileset_draw_tile(appdata->tileset, 
			  game_get_tile_at(appdata->game, x, y) , x, y);

  return event_box;
}

/* ------- signals emitted by the gui to process cpu player ------- */

/* cpu is to place a tile */
static gboolean
cb_cpu_places(GtkWidget *widget, gpointer data) {  
  appdata_t *appdata = (appdata_t*)data;

  g_assert(appdata->game->state == STATE_CPU_PLACES);
  g_assert(appdata->selected->tile != LOC_EMPTY);

  /* call cpu */
  game_place(appdata->game, appdata->selected->tile);
  return TRUE;
}

static gboolean
cb_cpu_has_placed(GtkWidget *widget, gpointer data) {  
  GtkWidget *dialog;
  appdata_t *appdata = (appdata_t*)data;
  int state;

  /* inform game engine that the tile has been placed */
  game_place_tile(appdata->selected->game, appdata->selected->tile, 
	  appdata->game->async_result%4, appdata->game->async_result/4);
    
  /* draw tile on board */
  tileset_draw_tile(appdata->tileset, appdata->selected->tile, 
	    appdata->game->async_result%4, appdata->game->async_result/4);
    
  /* remove tile from "selected" gui */
  selected_remove(appdata->selected);

  /* check for end of game */
  switch(state = game_check(appdata->game)) {
  case GAME_WON:
  case GAME_DRAW:
    dialog = gtk_message_dialog_new(GTK_WINDOW(appdata->window),
	  GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_INFO, GTK_BUTTONS_OK, 
          (state == GAME_WON)?"CPU wins!":"Draw game!");
  
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy(dialog);
	  
    appdata->game->state = STATE_IDLE;
    statusbar_message_state(appdata->statusbar);
    break;

  default:
    /* now the CPU has to pick the next tile */
    appdata->game->state = STATE_CPU_SELECTS;
    g_signal_emit_by_name(G_OBJECT(appdata->window), "cpu_selects");
  }

  return TRUE;
}

/* cpu is to place a tile */
static gboolean
cb_cpu_selects(GtkWidget *widget, gpointer data) {  
  appdata_t *appdata = (appdata_t*)data;

  g_assert(appdata->game->state == STATE_CPU_SELECTS);
  g_assert(appdata->selected->tile == LOC_EMPTY);

  game_give(appdata->game);

  return TRUE;
}

void game_quit(GtkWidget *widget, GdkEvent *event, gpointer data) {
  gconf_save((appdata_t*)data);
  gtk_main_quit();
}

/* cpu is to place a tile */
static gboolean
cb_cpu_has_selected(GtkWidget *widget, gpointer data) {  
  appdata_t *appdata = (appdata_t*)data;

  stack2selected(appdata->stack, appdata->game->async_result);

  appdata->game->state = STATE_HUMAN_PLACES;
  statusbar_message_state(appdata->statusbar);

  return TRUE;
}

int main(int argc, char *argv[]) {
  /* Create needed variables */
  GtkWidget *vbox, *hbox;

  appdata_t appdata;
  tileset_t tileset;
  stack_t stack;
  selected_t selected;
  statusbar_t statusbar;
  game_state_t game;

  printf("appdata = %p\n", &appdata);

  /* Initialize localization. */
  setlocale(LC_ALL, "");
  bindtextdomain(APPNAME, LOCALEDIR);
  bind_textdomain_codeset(APPNAME, "UTF-8");
  textdomain(APPNAME);

  /* setup the links between the objects */
  appdata.tileset = &tileset;   // the tileset the board is using
  appdata.game = &game;         // state of running game
  appdata.selected = &selected;
  appdata.stack = &stack;
  appdata.statusbar = &statusbar;

  /* prepare thread system */
  g_thread_init(NULL);

  /* Initialize the GTK. */
  gtk_init(&argc, &argv);

  gconf_client_init(&appdata);
  gconf_load(&appdata);

  /* init game state machine */
  game_init(&game);

  /* Create the hildon program and setup the title */
  appdata.program = HILDON_PROGRAM(hildon_program_get_instance());
  g_set_application_name("Kwarto");
  
  /* Create HildonWindow and set it to HildonProgram */
  game.window = appdata.window = HILDON_WINDOW(hildon_window_new());

  hildon_program_add_window(appdata.program, appdata.window);

  /************* main view **************/

  /* initially load a tileset */
  tileset_load(&tileset);

  gtk_container_add(GTK_CONTAINER(appdata.window),
		    GTK_WIDGET(hbox = gtk_hbox_new(FALSE, 0)));
  
  GtkWidget *align = gtk_alignment_new(0.5,0.5,0,0);
  gtk_container_add(GTK_CONTAINER(align), board_image_new(&appdata));
  gtk_box_pack_start(GTK_BOX(hbox), align, TRUE, TRUE, 0);

  gtk_container_add(GTK_CONTAINER(hbox),
		    GTK_WIDGET(vbox = gtk_vbox_new(FALSE, 0)));

  /* -------------- "selected" tile -------------------- */
  gtk_box_pack_start(GTK_BOX(vbox), selected_create_gui(&selected, 
		&tileset, &game), FALSE, FALSE, 0);

  /* ------------- create tile stack ----------------- */
  gtk_box_pack_start(GTK_BOX(vbox), 
		     stack_create_gui(&stack, &tileset, &game, 
			appdata.window, &selected), FALSE, FALSE, 0);
  
  /* ------------- create status bar ----------------- */


  gtk_box_pack_start(GTK_BOX(vbox), statusbar_create(&statusbar, &game, 
						     appdata.window),
		     FALSE, FALSE, 0);

  /* ------------- build a menu --------------- */

  menu_create(&appdata);

  /* Connect signal to X in the upper corner */
  g_signal_connect(G_OBJECT(appdata.window), "delete_event",
		   G_CALLBACK(game_quit), &appdata);


  /* internal signals used to handle the ai and control flow */
  g_signal_new("cpu_selects", GTK_TYPE_WINDOW, 
       (GSignalFlags)(G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION),
       0, NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
  g_signal_new("cpu_has_selected", GTK_TYPE_WINDOW, 
       (GSignalFlags)(G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION),
       0, NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

  g_signal_new("cpu_places", GTK_TYPE_WINDOW, 
       (GSignalFlags)(G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION),
       0, NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
  g_signal_new("cpu_has_placed", GTK_TYPE_WINDOW, 
       (GSignalFlags)(G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION),
       0, NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
  
  /* internal signals used to handle the ai */
  g_signal_connect(G_OBJECT(appdata.window), "cpu_selects",
		   G_CALLBACK(cb_cpu_selects), &appdata);
  g_signal_connect(G_OBJECT(appdata.window), "cpu_has_selected",
		   G_CALLBACK(cb_cpu_has_selected), &appdata);

  g_signal_connect(G_OBJECT(appdata.window), "cpu_places",
		   G_CALLBACK(cb_cpu_places), &appdata);
  g_signal_connect(G_OBJECT(appdata.window), "cpu_has_placed",
		   G_CALLBACK(cb_cpu_has_placed), &appdata);

  gtk_widget_show_all(GTK_WIDGET(appdata.window));

  /* if there's a tile in the "selected" from saved game state, update gui */
  if(selected.tile != LOC_EMPTY)
    stack2selected(&stack, selected.tile);

  /* if restored data contained a cpu run, then restart it */
  if(game.state == STATE_CPU_SELECTS)
    g_signal_emit_by_name(G_OBJECT(appdata.window), "cpu_selects");

  if(game.state == STATE_CPU_PLACES)
    g_signal_emit_by_name(G_OBJECT(appdata.window), "cpu_places");

  /* begin the main application */
  gtk_main();

  return 0;
}
