/*
  Knips - A simple Camera program.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Control.h"

#include <cstring>
#include <fstream>
#include <iostream>

#include <gst/interfaces/colorbalance.h>
#include <gst/interfaces/xoverlay.h>
#include <gst/interfaces/videoorientation.h>
#include <gst/video/video.h>

#include <Lum/Base/Util.h>

#include <Lum/Dialog.h>

#include <Lum/OS/Display.h>
#include <Lum/OS/EventLoopGMainLoop.h>

#include <Lum/OS/X11/Display.h>

#define CAMERA_ORIENTATION_FILE "/sys/devices/platform/gpio-switch/cam_turn/state"

static gboolean cb_have_data(GstElement* /*element*/,
                             GstBuffer *buffer,
                             GstPad* /*pad*/,
                             gpointer user_data)
{
  static_cast<VideoControl*>(user_data)->GotImage(static_cast<unsigned char*>(GST_BUFFER_DATA(buffer)));

  return true;
}

gboolean GstCallback(GstBus* bus, GstMessage* message, gpointer user_data)
{
  static_cast<VideoControl*>(user_data)->OnGstMessage(message);

  return true;
}

VideoControl::VideoControl()
: pipeline(NULL),videoSink(NULL),imageSink(NULL),
  source(NULL),
  videoWidth(0),videoHeight(0),
  storeImage(false),done(false),
  gstMsg(new Lum::Model::Action()),
  knipsed(new Lum::Model::Action()),
  timer(new Lum::Model::Action())
{
  SetDestroyOnCleanup(false);

  Observe(gstMsg);
  Observe(timer);
}

VideoControl::~VideoControl()
{
  if (pipeline!=NULL) {
    gst_element_set_state(pipeline,GST_STATE_NULL);

    while (true) {
      GstState current;

      gst_element_get_state(pipeline,&current,NULL,GST_CLOCK_TIME_NONE);
      if (current==GST_STATE_NULL) {
        break;
      }
      Lum::Base::SystemTime::Sleep(1);
    }

    gst_object_unref(pipeline);
    pipeline=NULL;
  }

  if (source!=NULL) {
    g_source_destroy(source);
    g_source_unref(source);
  }
}

void VideoControl::CheckCameraOrientation()
{
  std::ifstream file;
  std::string   value;

  file.open(CAMERA_ORIENTATION_FILE,std::ios::in);

  if (!file) {
    return;
  }

  std::getline(file,value);

  file.close();

  if (gst_element_implements_interface(videoSource,GST_TYPE_VIDEO_ORIENTATION)) {
    GstVideoOrientation *orientation=GST_VIDEO_ORIENTATION(videoSource);

    gst_video_orientation_set_vflip(orientation,value=="active");
    gst_video_orientation_set_hflip(orientation,value=="active");
  }
}

void VideoControl::GotImage(unsigned char* data)
{
  Lum::OS::MutexGuard guard(mutex);

  if (!storeImage || videoWidth==0 || videoHeight==0) {
    return;
  }

  Lum::Images::Pixel *pixel;

  image=Lum::Images::Factory::factory->CreateImage();

  pixel=new Lum::Images::Pixel[videoWidth*videoHeight];

  for (size_t i=0; i<videoWidth*videoHeight; i++) {
    pixel[i].r=data[i*3];
    pixel[i].g=data[i*3+1];
    pixel[i].b=data[i*3+2];
    pixel[i].a=255;
  }

  image->SetData(videoWidth,videoHeight,false,pixel);

  storeImage=false;

  Lum::OS::display->QueueActionForAsyncNotification(knipsed.Get());
}

void VideoControl::SetKnipsAction(Lum::Model::Action* action)
{
  knips=action;
}

Lum::Model::Action* VideoControl::GetKnipsedAction() const
{
  return knipsed.Get();
}

void VideoControl::GetImage(Lum::Images::ImageRef &image)
{
  Lum::OS::MutexGuard guard(mutex);

  image=this->image;
}

void VideoControl::CalcSize()
{
  width=160;
  height=120;

  Observe(dynamic_cast<Lum::Dialog*>(GetWindow()->GetMaster())->GetOpenedAction());

  minWidth=width;
  minHeight=height;

  Lum::OS::X11::SubWindow::CalcSize();
}

void VideoControl::Layout()
{
  if (!done) {
    GError *error=NULL;

    gst_init(NULL,NULL);

    // Pipeline elements
    /*
    pipeline=::gst_parse_launch("videotestsrc name=videosource ! "
                                "video/x-raw-yuv ! "
                                "ffmpegcolorspace ! "
                                "tee name=t ! "
                                "queue ! "
                                "{ xvimagesink name=videosink }"
                                "t. ! queue ! "
                                "{ ffmpegcolorspace ! "
                                "video/x-raw-rgb,bpp=24,depth=24,framerate:fraction=8/1 !"
                                "fakesink name=imagesink }",
                                &error);*/
    pipeline=::gst_parse_launch("v4l2src name=videosource ! "
                                "video/x-raw-yuv,width=640,height=480 ! "
                                "ffmpegcolorspace ! "
                                "tee name=t ! "
                                "queue ! "
                                "{ xvimagesink name=videosink }"
                                "t. ! "
                                "queue ! "
                                "{ ffmpegcolorspace ! "
                                "video/x-raw-rgb,bpp=24,depth=24,framerate=(fraction)8/1 ! "
                                "fakesink name=imagesink }",
                                &error);

    if (pipeline==NULL) {
      done=true;
      return;
    }

    videoSource=gst_bin_get_by_name(GST_BIN(pipeline),"videosource");
    videoSink=gst_bin_get_by_name(GST_BIN(pipeline),"videosink");
    imageSink=gst_bin_get_by_name(GST_BIN(pipeline),"imagesink");

    assert(videoSource!=NULL && videoSink!=NULL && imageSink!=NULL);

    g_object_set(G_OBJECT(imageSink),"signal-handoffs",TRUE,NULL);
    g_signal_connect(imageSink,"handoff",G_CALLBACK(cb_have_data),this);

    Lum::OS::GMainLoopEventLoop* eventLoop=dynamic_cast<Lum::OS::GMainLoopEventLoop*>(Lum::OS::display->GetEventLoop());
    GstBus*                      bus=gst_element_get_bus(pipeline);

    source=gst_bus_create_watch(bus);
    g_source_set_callback(source,(GSourceFunc)GstCallback,this,NULL);
    g_source_attach(source,eventLoop->context);
    gst_object_unref(GST_OBJECT(bus));

    gst_element_set_state(pipeline,GST_STATE_PLAYING);

    GstPad *videoSourcePad;

    videoSourcePad=gst_element_get_static_pad(videoSource,"src");

    if (videoSourcePad!=NULL) {
      gint w,h;

      if (gst_video_get_size(videoSourcePad,&w,&h)) {
        videoWidth=w;
        videoHeight=h;
      }

      gst_object_unref(videoSourcePad);
    }


    Lum::OS::display->AddTimer(1,0,timer);

    done=true;
  }

  Lum::OS::X11::SubWindow::Layout();
}

void VideoControl::RequestImage()
{
  Lum::OS::MutexGuard guard(mutex);

  image=NULL;
  storeImage=true;
}

void VideoControl::CancelRequestImage()
{
  Lum::OS::MutexGuard guard(mutex);

  storeImage=false;
  image=NULL;
}

void VideoControl::OnGstMessage(GstMessage* message)
{
  //std::cout << "Message: " << gst_message_type_get_name(GST_MESSAGE_TYPE(message)) << std::endl;

  if (GST_MESSAGE_TYPE(message)==GST_MESSAGE_ERROR) {
    GError *error;
    char   *text;

    gst_message_parse_error(message,&error,&text);

    std::cerr << "gstreamer ERROR: " << text << std::endl;

    g_free(error);
    g_free(text);
  }
  else if (GST_MESSAGE_TYPE(message)==GST_MESSAGE_WARNING) {
    GError *error;
    char   *text;

    gst_message_parse_warning(message,&error,&text);

    std::cerr << "gstreamer WARNING: " << text << std::endl;

    g_free(error);
    g_free(text);
  }
  else if (message->src==GST_OBJECT(pipeline) && GST_MESSAGE_TYPE(message)==GST_MESSAGE_STATE_CHANGED) {
    GstState newState;

    gst_message_parse_state_changed(message,NULL,&newState,NULL);

    if (newState==GST_STATE_PLAYING || newState==GST_STATE_PAUSED) {
      knips->Enable();
    }
    else {
      knips->Disable();
    }
  }
}

void VideoControl::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==timer && timer->IsFinished()) {
    CheckCameraOrientation();
    Lum::OS::display->AddTimer(1,0,timer);
  }
  else if (GetWindow()!=NULL &&
           dynamic_cast<Lum::Dialog*>(GetWindow()->GetMaster())!=NULL &&
           model==dynamic_cast<Lum::Dialog*>(GetWindow()->GetMaster())->GetOpenedAction() &&
           dynamic_cast<Lum::Dialog*>(GetWindow()->GetMaster())->GetOpenedAction()->IsFinished() &&
           videoSink!=NULL) {
    gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(videoSink),window);
  }

  Object::Resync(model,msg);
}

