/*
  Knips - A simple Camera program.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/AspectRatio.h>
#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/DirNameSelect.h>
#include <Lum/Label.h>
#include <Lum/LED.h>
#include <Lum/Panel.h>
#include <Lum/String.h>
#include <Lum/Text.h>

#include <Lum/Base/DateTime.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Boolean.h>
#include <Lum/Model/String.h>

#include <Lum/OS/Probe.h>

#include "config.h"
#include "Configuration.h"
#include "Control.h"
#include <iostream>

#if defined(Status)
#undef Status
#endif

static Lum::AppInfo info;

class Settings : public Lum::Dialog
{
public:
  Settings()
  : Dialog()
  {
    // no code
  }

  ~Settings()
  {
    // no code
  }

  void PreInit()
  {
    Lum::DirNameSelect *dirSel;
    Lum::Label         *label;
    Lum::Panel         *panel;

    panel=new Lum::VPanel();

    label=new Lum::Label();

    dirSel=new Lum::DirNameSelect();
    dirSel->SetFlex(true,false);
    dirSel->SetWidth(Lum::Base::Size::stdCharWidth,30);
    dirSel->SetDialogTitle(L"Select directory...");
    dirSel->SetModel(imageDirectory.Get());
    label->AddLabel(L"Image path:",dirSel);

    panel->Add(label);
    panel->AddSpace();
    panel->Add(Lum::ButtonRow::Create(true,false)
               ->Add(Lum::Button::Create(L"_Close",GetClosedAction(),Lum::Button::typeDefault,true,true)));

    SetMain(panel);

    Dialog::PreInit();
  }
};

class MainWindow : public Lum::Dialog
{
private:
  VideoControl           *control;

  Lum::Model::ActionRef  knips;
  Lum::Model::ActionRef  knipsTimer;
  Lum::Model::BooleanRef knipsing;

  Lum::Model::ActionRef  settings;
  Lum::Model::ActionRef  about;
  std::wstring           lastFilename; //! filename of last image
  size_t                 instance;     //! the nth image per second

public:
  MainWindow()
  : control(new VideoControl()),
    knips(new Lum::Model::Action()),
    knipsTimer(new Lum::Model::Action()),
    knipsing(new Lum::Model::Boolean(false)),
    settings(new Lum::Model::Action()),
    about(new Lum::Model::Action()),
    instance(1)
  {
    Observe(knipsTimer);

    knips->Disable();
    control->SetKnipsAction(knips);

    Observe(GetOpenedAction());
    Observe(knips);
    Observe(control->GetKnipsedAction());
    Observe(settings);
    Observe(about);
  }

  void StoreImage()
  {
    assert(imageDirectory.Valid() && imagePrefix.Valid());

    Lum::Images::ImageRef      image;
    Lum::Base::SystemTime      st;
    Lum::Base::DateTime        now;
    Lum::Base::Path            path;
    Lum::Base::Status          result;
    std::wstring               file;
    Lum::Images::WriterOptions options;

    control->GetImage(image);

    if (image.Invalid()) {
      std::cout << "Did not get an image from video control!" << std::endl;
      return;
    }

    options.SetQuality(80);

    st.GetLocalDateTime(now);

    file=imagePrefix->Get();
    if (!file.empty()) {
      file.append(L"_");
    }
    file.append(Lum::Base::NumberToWString(now.year));
    file.append(L"-");
    if (now.month<10) {
      file.append(L"0");
    }
    file.append(Lum::Base::NumberToWString(now.month));
    file.append(L"-");
    if (now.dayOfMonth<10) {
      file.append(L"0");
    }
    file.append(Lum::Base::NumberToWString(now.dayOfMonth));
    file.append(L"_");
    if (now.hour<10) {
      file.append(L"0");
    }
    file.append(Lum::Base::NumberToWString(now.hour));
    file.append(L"-");
    if (now.minute<10) {
      file.append(L"0");
    }
    file.append(Lum::Base::NumberToWString(now.minute));
    file.append(L"-");
    if (now.second<10) {
      file.append(L"0");
    }
    file.append(Lum::Base::NumberToWString(now.second));

    if (lastFilename==file) {
      instance++;
    }
    else {
      instance=1;
    }

    lastFilename=file;

    file.append(L"_");
    file.append(Lum::Base::NumberToWString(instance));

    file.append(L".jpg");

    path.SetNativeDirAndFile(imageDirectory->Get(),file);

    result=Lum::Images::WriteJPEG(path.GetPath(),image,options);

    if (!result) {
      Lum::Dlg::Msg::ShowOk(this,
                            L"Error while storing image!",
                            std::wstring(L"The was an error while storing the image with the filename\n")+
                            L"'"+path.GetPath()+L"'\n"+
                            L"\n"+
                            L"Error:\n"+
                            L"'"+result.GetDescription()+L"'");
    }
  }

  void PreInit()
  {
    Lum::LED           *led;
    Lum::Panel         *vpanel,*vpanel2,*hpanel;

    vpanel=Lum::VPanel::Create(true,true);

    hpanel=Lum::HPanel::Create(true,true);

    control->SetFlex(true,true);

    hpanel->Add(Lum::AspectRatio::Create(control,640,480,true,true));
    hpanel->AddSpace();

    vpanel2=Lum::VPanel::Create(false,true);

    led=new Lum::LED();
    led->SetModel(knipsing);
    vpanel2->Add(led);

    vpanel2->AddSpace();
    vpanel2->Add(Lum::Button::Create(L"_Klick!",knips,Lum::Button::typeDefault,false,true));

    hpanel->Add(vpanel2);

    vpanel->Add(hpanel);
    vpanel->AddSpace();
    vpanel->Add(Lum::HPanel::Create(true,false)
                ->Add(Lum::Button::Create(L"_Prefs",settings))
                ->AddSpace()
                ->Add(Lum::Button::Create(L"_About",about))
                ->AddSpace()
                ->Add(new Lum::Text(L"Prefix:"))
                ->AddSpace()
                ->Add(Lum::String::Create(imagePrefix,true,false)));

    SetMain(vpanel);

    Dialog::PreInit();
  }

  void ShowSettings()
  {
    Settings* settings;

    settings=new Settings();
    settings->SetParent(this);

    if (settings->Open()) {
      settings->SetExitAction(settings->GetClosedAction());
      settings->EventLoop();
      settings->Close();
    }

    delete settings;
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      if (!LoadConfig()) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"No configuration file found!",
                              L"The program was not able to find its configuration file.\n"
                              L"\n"
                              L"If this is the first start of your program, this is\n"
                              L"OK. The program will generate a configuration file if\n"
                              L"it quits.");
      }
    }
    else if (model==settings && settings->IsFinished()) {
      ShowSettings();
    }
    else if (model==about && about->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==knips && knips->IsFinished() && knips->IsEnabled()) {
      control->RequestImage();
      knips->Disable();
      knipsing->Set(true);
      Lum::OS::display->AddTimer(5,0,knipsTimer);
    }
    else if (model==control->GetKnipsedAction() && control->GetKnipsedAction()->IsFinished()) {
      Lum::OS::display->RemoveTimer(knipsTimer);
      StoreImage();
      knipsing->Set(false);
      knips->Enable();
    }
    else if (model==knipsTimer && knipsTimer->IsFinished()) {
      control->CancelRequestImage();
      Lum::Dlg::Msg::ShowOk(this,
                            L"Error while fetching image!",
                            L"The video control did not offer an image\n"
                            L"in time. Giving up!");
      knipsing->Set(false);
      knips->Enable();
    }
    Dialog::Resync(model,msg);
  }
};

int main(int argc, char* argv[])
{
  info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
  info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
  info.SetDescription(L"Cheeeseee...");
  info.SetAuthor(L"Tim Teulings");
  info.SetContact(L"Tim Teulings <tim@teulings.org>");
  info.SetCopyright(L"(c) 2007, Tim Teulings");
  info.SetLicense(L"GNU Public License");


  if (Lum::OS::prober->Open(L"Knips",argc,argv)) {

    Lum::Dialog *window;

    window=new MainWindow;

    if (window->Open()) {
      window->SetExitAction(window->GetClosedAction());
      window->EventLoop();
      SaveConfig();
      window->Close();
    }

    delete window;

    Lum::OS::display->Close();
  }
}
