/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Model/Table.h>

namespace Lum {
  namespace Model {

    Table::Table()
    {
      // no code
    }

    Table::~Table()
    {
      // no code
    }

    void Table::RedrawRow(size_t row)
    {
      RefreshRow msg;

      msg.row=row;

      Notify(msg);
    }

    void Table::RedrawCell(size_t column, size_t row)
    {
      RefreshCell msg;

      msg.row=row;
      msg.column=column;

      Notify(msg);
    }

    void Table::NotifyInsert(size_t row, size_t count)
    {
      InsertRow msg;

      msg.row=row;
      msg.count=count;

      Notify(msg);
    }

    void Table::NotifyDelete(size_t row, size_t count)
    {
      DeleteRow msg;

      msg.row=row;
      msg.count=count;

      Notify(msg);
    }

    bool Table::Sort(size_t column, bool down)
    {
      // base class does not implement this functionality!
      return false;
    }

    ListTable::Entry::Entry(ListTable* table)
    : table(table)
    {
    }

    ListTable::Entry::~Entry()
    {
      // no code
    }

    std::wstring ListTable::Entry::GetString(size_t /*column*/) const
    {
      return L"";
    }

    Object* ListTable::Entry::GetObject(size_t /*column*/) const
    {
      return NULL;
    }

    bool ListTable::Entry::IsGreater(const Entry* other,size_t column) const
    {
      return GetString(column)>other->GetString(column);
    }

    ListTable::StdEntry::Cell::Cell()
    : object(NULL)
    {
    }

    ListTable::StdEntry::StdEntry(ListTable* table)
    : Entry(table)
    {
      // no code
    }

    ListTable::StdEntry::~StdEntry()
    {
      for (size_t x=0; x<row.size(); x++) {
        delete row[x].object;
      }
    }

    std::wstring ListTable::StdEntry::GetString(size_t column) const
    {
      if (column-1<row.size()) {
        return row[column-1].string;
      }
      else {
        return L"";
      }
    }

    Object* ListTable::StdEntry::GetObject(size_t column) const
    {
      if (column-1<row.size()) {
        return row[column-1].object;
      }
      else {
        return NULL;
      }
    }

    void ListTable::StdEntry::SetString(size_t column, const std::wstring& string)
    {
      if (column>row.size()) {
        row.resize(column);
      }

      row[column-1].string=string;
      table->RedrawCell(column,table->GetRow(this));
    }

    void ListTable::StdEntry::SetObject(size_t column, Lum::Object* object)
    {
      if (column>row.size()) {
        row.resize(column);
      }

      delete row[column-1].object;
      row[column-1].object=object;
      table->RedrawCell(column,table->GetRow(this));
    }

    ListTable::ListTable()
    : current(list.end()),currentPos(0)
    {
      // no code
    }

    ListTable::~ListTable()
    {
      Clear();
    }

    ListTable::Entry* ListTable::GetEntry(size_t row) const
    {
      if (current==list.end()) {
        current=list.begin();
        currentPos=1;
      }

      if (currentPos<row) {
        while (currentPos<row) {
          ++current;
          ++currentPos;
        }
      }
      else if (currentPos>row) {
        while (currentPos>row) {
          --current;
          --currentPos;
        }
      }

      return *current;
    }

    size_t ListTable::GetRow(Entry* entry) const
    {
      if (current!=list.end() && entry==*current) {
        return currentPos;
      }
      else {
        std::list<Entry*>::const_iterator iter;
        size_t                            pos;

        pos=1;
        iter=list.begin();
        while (iter!=list.end()) {
          if (entry==*iter) {
            return pos;
          }

          ++iter;
          ++pos;
        }
      }

      assert(true);
      return 0;
    }

    size_t ListTable::GetRows() const
    {
      return list.size();
    }

    std::wstring ListTable::GetString(size_t column, size_t row) const
    {
      Entry* entry;

      entry=GetEntry(row);

      assert(entry!=NULL);

      return entry->GetString(column);
    }

    Object* ListTable::GetObject(size_t column, size_t row) const
    {
      Entry* entry;

      entry=GetEntry(row);

      assert(entry!=NULL);

      return entry->GetObject(column);
    }

    void ListTable::Prepend(Entry *entry)
    {
      list.push_front(entry);
      currentPos++;
      NotifyInsert(1,1);
    }

    void ListTable::Delete(size_t row)
    {
      Entry *entry;

      entry=GetEntry(row);

      assert(entry!=NULL);

      list.erase(current); // TODO Reuse current and recalculate currentPos

      delete entry;

      current=list.end();
      currentPos=0;

      NotifyDelete(row,1);
    }

    void ListTable::Append(Entry *entry)
    {
      list.push_back(entry);

      NotifyInsert(list.size(),1);
    }

    void ListTable::AppendString(const std::wstring& string)
    {
      StdEntry *entry;

      entry=new StdEntry(this);
      entry->SetString(1,string);
      Append(entry);
    }

    class ListCompare
    {
    private:
      size_t column;
      bool   down;

    public:
      ListCompare(size_t column, bool down)
      : column(column),down(down)
      {
        // no code
      }

      bool operator()(const ListTable::Entry* a, const ListTable::Entry* b)
      {
        if (down) {
          return !a->IsGreater(b,column);
        }
        else {
          return a->IsGreater(b,column);
        }
      }
    };

    bool ListTable::Sort(size_t column, bool down)
    {
      list.sort(ListCompare(column,down));

      current=list.end();
      currentPos=0;

      Notify();

      return true;
    }

    void ListTable::Clear()
    {
      std::list<Entry*>::const_iterator iter;

      iter=list.begin();
      while (iter!=list.end()) {
        delete (*iter);

        ++iter;
      }

      current=list.end();
      currentPos=0;

      list.clear();

      Notify();
    }
  }
}
