/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"

#define PP_WIDTH 150
#define PP_HEIGHT 150

#define MAX_POS 250   // positions saved per chunk 

typedef struct {
  double lat, lon;
} dpos_t;

typedef struct pos_list {
  int len;
  pos_t pos[MAX_POS];
  struct pos_list *next;
} pos_list_t;

typedef struct {
  appdata_t *appdata;
  guint handler_id;
  GtkWidget *area;
  GtkWidget *total_label, *range_label;
  GtkWidget *lat_label, *lon_label;
  GdkPixmap *pixmap;
  pos_list_t *pos_list;

  /* values calculated on update */
  pos_t min, max, mid;
  dpos_t avg;
  double scale;
  int total;

} pp_context_t;

GdkGC *clone_gc(GdkPixmap *pixmap, GtkWidget *widget, char *color) {
  GdkGC *gc = gdk_gc_new(pixmap);
  gdk_gc_copy(gc, widget->style->black_gc);
  GdkColor used_color;
  gdk_color_parse(color, &used_color);   // green
  gdk_gc_set_rgb_fg_color(gc,  &used_color);

  return gc;
}

static void context_update(pp_context_t *context) {
  /* count stored positions and get min/max values */
  context->total = 0;
  context->min.lat = context->min.lon =  180.0;
  context->max.lat = context->max.lon = -180.0;
  context->avg.lat = context->avg.lon = 0.0;

  pos_list_t *pos_list = context->pos_list;
  while(pos_list) {
    context->total += pos_list->len;

    int j;
    for(j=0;j<pos_list->len;j++) {
      if(pos_list->pos[j].lat > context->max.lat) 
	context->max.lat = pos_list->pos[j].lat;
      if(pos_list->pos[j].lat < context->min.lat)
	context->min.lat = pos_list->pos[j].lat;
      if(pos_list->pos[j].lon > context->max.lon)
	context->max.lon = pos_list->pos[j].lon;
      if(pos_list->pos[j].lon < context->min.lon)
	context->min.lon = pos_list->pos[j].lon;

      context->avg.lat += pos_list->pos[j].lat;
      context->avg.lon += pos_list->pos[j].lon;
    }

    pos_list = pos_list->next;
  }

  if(!context->total)
    return;

  context->avg.lat /= context->total;
  context->avg.lon /= context->total;

#define SCALE 0.9

  context->mid.lat = (context->max.lat+context->min.lat)/2.0;
  context->mid.lon = (context->max.lon+context->min.lon)/2.0;
  float lat_scale = SCALE/(context->max.lat-context->min.lat);
  float lon_scale = SCALE/(context->max.lon-context->min.lon);

  context->scale = (lat_scale < lon_scale)?lat_scale:lon_scale;
}

#define XOFF 1
static void gdk_draw_cross(GdkDrawable *drawable, GdkGC *gc, gint x, gint y) {
  gdk_draw_line(drawable, gc, x-XOFF, y-XOFF, x+XOFF, y+XOFF);
  gdk_draw_line(drawable, gc, x-XOFF, y+XOFF, x+XOFF, y-XOFF);
}

static void pp_draw(GtkWidget *widget, pp_context_t *context) {
  gint width = widget->allocation.width;
  gint height = widget->allocation.height;
  gint diameter = (height < width)?height:width;

  gint xcenter = width/2;
  gint ycenter = height/2;

  /* erase background */
  gdk_draw_rectangle(context->pixmap, 
		     widget->style->bg_gc[GTK_STATE_NORMAL], TRUE,
		     0, 0, width, height);

  GdkGC *circle_gc = widget->style->white_gc;
  if(widget->style->bg[GTK_STATE_NORMAL].red + 
     widget->style->bg[GTK_STATE_NORMAL].green +
     widget->style->bg[GTK_STATE_NORMAL].blue > 3*60000) {
    circle_gc = gdk_gc_new(context->pixmap);
    gdk_gc_copy(circle_gc, widget->style->black_gc);
    GdkColor lgrey_color;
    gdk_color_parse("#DDDDDD", &lgrey_color);
    gdk_gc_set_rgb_fg_color(circle_gc,  &lgrey_color);
  }

  gdk_draw_arc(context->pixmap, circle_gc, TRUE,
	       xcenter - (SCALE*diameter/2), 
	       ycenter - (SCALE*diameter/2), 
	       SCALE*diameter, SCALE*diameter,
	       0, 360*64);

  /* make sure data captured so far allows for useful rendering */
  if(context->total < 2) return;
  if(context->min.lat >= context->max.lat) return;
  if(context->min.lon >= context->max.lon) return;
  if(context->scale > 50000) return;

  /* setup required colors */
  GdkGC *green_gc = clone_gc(context->pixmap, widget, "#008000");
  GdkGC *red_gc = clone_gc(context->pixmap, widget, "#800000");
  GdkGC *blue_gc = clone_gc(context->pixmap, widget, "#000080");

#if 0
  printf("---------- %f\n", context->scale);
  printf("X: %f->%f->%f  Y: %f->%f->%f\n", 
	 context->min.lon, context->mid.lon, context->max.lon,
	 context->min.lat, context->mid.lat, context->max.lat);
#endif

  /* draw all dots */
  pos_list_t *pos_list = context->pos_list;
  double dscale = context->scale * diameter;
  while(pos_list) {
    int j;
    for(j=0;j<pos_list->len;j++) {
#if 0
      printf("%f %f -> y = %f, x = %f\n", 
	     pos_list->pos[j].lat, pos_list->pos[j].lon,
	     (pos_list->pos[j].lat-context->mid.lat)*context->scale,
	     (pos_list->pos[j].lon-context->mid.lon)*context->scale);
#endif
      
      if(pos_list->next || j != pos_list->len-1)
	gdk_draw_cross(context->pixmap, green_gc, 
	       xcenter + ((pos_list->pos[j].lon-context->mid.lon)*dscale),
	       ycenter + ((pos_list->pos[j].lat-context->mid.lat)*dscale));
      else
	gdk_draw_arc(context->pixmap, red_gc, TRUE,
	       xcenter + ((pos_list->pos[j].lon-context->mid.lon)*dscale)-3,
	       ycenter + ((pos_list->pos[j].lat-context->mid.lat)*dscale)-3,
	       7, 7, 0, 360*64);
    }
    
    pos_list = pos_list->next;
  }
#if 0
  printf("D y = %f, x = %f\n", 
	 (context->avg.lat-context->mid.lat)*context->scale,
	 (context->avg.lon-context->mid.lon)*context->scale);
#endif
  gdk_draw_arc(context->pixmap, blue_gc, TRUE,
	       xcenter + ((context->avg.lon-context->mid.lon)*dscale)-5,
	       ycenter + ((context->avg.lat-context->mid.lat)*dscale)-5,
	       11, 11, 0, 360*64);
}

/* Create a new backing pixmap of the appropriate size */
static gint pp_configure_event(GtkWidget *widget, GdkEventConfigure *event,
			    gpointer data) {
  pp_context_t *context = (pp_context_t*)data;

  if(context->pixmap)
    gdk_pixmap_unref(context->pixmap);
  
  context->pixmap = gdk_pixmap_new(widget->window,
				   widget->allocation.width,
				   widget->allocation.height,
				   -1);
  context_update(context);
  pp_draw(widget, context);
  
  return TRUE;
}

/* Redraw the screen from the backing pixmap */
static gint pp_expose_event(GtkWidget *widget, GdkEventExpose *event, 
			 gpointer data) {
  pp_context_t *context = (pp_context_t*)data;

  gdk_draw_pixmap(widget->window,
                  widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                  context->pixmap,
                  event->area.x, event->area.y,
                  event->area.x, event->area.y,
                  event->area.width, event->area.height);

  return FALSE;
}

gint pp_destroy_event(GtkWidget *widget, gpointer data ) {
  pp_context_t *context = (pp_context_t*)data;

  printf("destroying precise position view\n");

  /* stop timer */
  if(context->handler_id)
    gtk_timeout_remove(context->handler_id);

  pos_list_t *pos_list = context->pos_list;
  while(pos_list) {
    pos_list_t *next = pos_list->next;
    free(pos_list);
    pos_list = next;
  }

  /* destroy context itself */
  g_free(context);

  return FALSE;
}

/* called once a second */
static gboolean update(gpointer data) {
  pp_context_t *context = (pp_context_t*)data;
  
  pos_list_t **pos_list = &context->pos_list;
  while(*pos_list && ((*pos_list)->len == MAX_POS)) 
    pos_list = &(*pos_list)->next;

  if(!*pos_list) {
    printf("alloc new list\n");
    *pos_list = g_new0(pos_list_t, 1);
  }

  /* get one position */
  pos_t *p = gps_get_pos(context->appdata);
  if(p) (*pos_list)->pos[(*pos_list)->len++] = *p;
   
  context_update(context);
  
  if(context->pixmap) {
    /* draw sat view */
    pp_draw(context->area, context);
    gtk_widget_queue_draw_area(context->area, 0,0,
			       context->area->allocation.width, 
			       context->area->allocation.height);
  }
  
  /* and whatever else needs to be done ... */
  char str[32];
  snprintf(str, sizeof(str), _("Total: %d"), context->total);
  gtk_label_set_text(GTK_LABEL(context->total_label), str);

  /* calculate range */
  pos_t pos1 = { context->mid.lat, context->min.lon };
  pos_t pos2 = { context->mid.lat, context->max.lon };

  //   printf("Total = %d\n", context->total);

  if(p) {
    snprintf(str, sizeof(str), _("Diameter: "));
    if(context->total > 1) {
      float dist = gpx_pos_get_distance(pos1, pos2, 
					context->appdata->imperial);
      //      printf("dist = %f\n", dist);
      distance_str(str+strlen(str), sizeof(str)-strlen(str), 
		   dist, context->appdata->imperial);
    } else
      strcat(str+strlen(str), "---");
  } else
    strcpy(str, _("No fix"));

#ifndef USE_MAEMO
  gtk_label_set_text(GTK_LABEL(context->range_label), str);
#else
  char *mup = g_markup_printf_escaped("<span size='x-small'>%s</span>", str);
  gtk_label_set_markup(GTK_LABEL(context->range_label), mup);
  g_free(mup);
#endif
  
  pos_lat_str(str, sizeof(str), context->avg.lat);
  gtk_label_set_text(GTK_LABEL(context->lat_label), str);
  pos_lon_str(str, sizeof(str), context->avg.lon);
  gtk_label_set_text(GTK_LABEL(context->lon_label), str);

  return TRUE;   // fire again
}

#ifdef USE_MAEMO
static void on_mm_export_clicked(GtkButton *button, gpointer data) {
  pp_context_t *context = (pp_context_t*)data;

  if(!context->avg.lat || !context->avg.lon)
    return;

  pos_t pos = { context->avg.lat, context->avg.lon } ;
  dbus_mm_set_position(context->appdata, &pos);
}
#endif

static void on_copy_clicked(GtkButton *button, gpointer data) {
  pp_context_t *context = (pp_context_t*)data;
  char str[64];

  /* make a textual representation of the coordinate */
  pos_lat_str(str, sizeof(str), context->avg.lat);
  strcat(str, " ");
  pos_lon_str(str+strlen(str), sizeof(str)-strlen(str), context->avg.lon);

  printf("set clipboard to \"%s\"\n", str);
  gtk_clipboard_set_text(context->appdata->clipboard, str, -1);
}

void precise_position(appdata_t *appdata) {
  pp_context_t *context = g_new0(pp_context_t, 1);

  context->appdata = appdata;

  if(!appdata->use_gps) {
    errorf(_("GPS is disabled. Please enable it to use this feature."));
    g_free(context);
    return;
  }

  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Precise Position"),
	  GTK_WINDOW(appdata->window), GTK_DIALOG_MODAL,
          GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);

#ifdef USE_MAEMO
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog), 
		 HELP_ID_PRECPOS, appdata->osso_context);
#endif

  GtkWidget *hbox = gtk_hbox_new(FALSE,20);

  /* --------------- left part ------------------------ */
  GtkWidget *vbox = gtk_vbox_new(FALSE,2);
  context->area = gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(context->area), 
			PP_WIDTH, PP_HEIGHT);

  gtk_signal_connect(GTK_OBJECT(context->area), "expose_event",
		     G_CALLBACK(pp_expose_event), context);
  gtk_signal_connect(GTK_OBJECT(context->area),"configure_event",
		     G_CALLBACK(pp_configure_event), context);
  g_signal_connect(G_OBJECT(dialog), "destroy", 
		   G_CALLBACK(pp_destroy_event), context);
  
  gtk_box_pack_start_defaults(GTK_BOX(vbox), context->area);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
	      context->range_label = gtk_label_new(""));

  gtk_box_pack_start_defaults(GTK_BOX(hbox), vbox);

  /* --------------- right part ------------------------ */
  vbox = gtk_vbox_new(FALSE,2);

  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
		      context->total_label = gtk_label_new(""));

  gtk_box_pack_start_defaults(GTK_BOX(vbox), gtk_hseparator_new());

  gtk_box_pack_start_defaults(GTK_BOX(vbox), gtk_label_new(_("Latitude:")));
  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
			      context->lat_label = gtk_label_new(""));
  gtk_box_pack_start_defaults(GTK_BOX(vbox), gtk_label_new(_("Longitude:")));
  gtk_box_pack_start_defaults(GTK_BOX(vbox), 
			      context->lon_label = gtk_label_new(""));
  GtkWidget *copy_but = gtk_button_new_with_label(_("Copy"));
  gtk_signal_connect(GTK_OBJECT(copy_but), "clicked",
  		     (GtkSignalFunc)on_copy_clicked, context);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), copy_but);

  gtk_box_pack_start_defaults(GTK_BOX(hbox), vbox);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox);

  context->handler_id = gtk_timeout_add(1000, update, context);

#ifdef USE_MAEMO
  /* ------------- maemo mapper button ---------------- */  
  GtkWidget *button = gtk_button_new();
  gtk_button_set_image(GTK_BUTTON(button), icon_get_widget(ICON_MISC, 0));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
  		     (GtkSignalFunc)on_mm_export_clicked, context);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area), button);
#endif

  update(context);
  gtk_widget_show_all(dialog);
  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
}
