/*
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>

#include <time.h>
#include <sys/time.h>

#include "gpxview.h"
#include "custom_size_renderer.h"
#include "custom_rating_renderer.h"
#include "custom_type_renderer.h"

#include <locale.h>

extern char *strcasestr (__const char *__haystack, __const char *__needle);

#ifdef USE_MAEMO
static void crumb_add(appdata_t *appdata, char *name, int level, 
		      gpointer user_data);

enum {
  CRUMB_GPXLIST = 0,
  CRUMB_CACHELIST,
  CRUMB_SEARCH_GLOBAL,
  CRUMB_SEARCH_GPX,
  CRUMB_CACHE,
};

#endif
/* size of first buffer malloc; start small to exercise grow routines */
#define	FIRSTSIZE	1

void errorf(const char *fmt, ...) {
  va_list args;
  char *buf = NULL;
  size_t bufsize;
  char *newbuf;
  size_t nextsize = 0;
  int outsize;
  
  bufsize = 0;
  for (;;) {
    if (bufsize == 0) {
      if ((buf = (char *)malloc(FIRSTSIZE)) == NULL) 
	return;

      bufsize = 1;
    } else if ((newbuf = (char *)realloc(buf, nextsize)) != NULL) {
      buf = newbuf;
      bufsize = nextsize;
    } else {
      free(buf);
      return;
    }
    
    va_start(args, fmt);
    outsize = vsnprintf(buf, bufsize, fmt, args);
    va_end(args);
    
    if (outsize == -1) {
      nextsize = bufsize * 2;
    } else if (outsize == bufsize) {
      nextsize = bufsize * 2;
    } else if (outsize > bufsize) {
      nextsize = outsize + 2;  // Linux!
    } else if (outsize == bufsize - 1) {
      nextsize = bufsize * 2;
    } else {
      /* Output was not truncated */
      break;
    }
  }

  GtkWidget *dialog = gtk_message_dialog_new(
			   GTK_WINDOW(NULL),
			   GTK_DIALOG_DESTROY_WITH_PARENT,
			   GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
			   buf);

  gtk_window_set_title(GTK_WINDOW(dialog), _("ERROR"));

 gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);

  free(buf);
}

gpx_t *choose_file(appdata_t *appdata, gboolean whole_dir) {
  GtkWidget *dialog;
  gpx_t *gpx = NULL;
  
#ifdef USE_MAEMO
  dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(appdata->window), 
			whole_dir?GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER :
					GTK_FILE_CHOOSER_ACTION_OPEN);
  
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog), 
		 HELP_ID_IMPORT, appdata->osso_context);
#else
  dialog = gtk_file_chooser_dialog_new (whole_dir?_("Open Directory"):
					_("Open File"),
			GTK_WINDOW(appdata->window),
			whole_dir?GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER :
					GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			NULL);
#endif
  
  /* use path if one is present */
  if(appdata->path) 
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), 
					appdata->path);

  gtk_widget_show_all (GTK_WIDGET(dialog));
  if (gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_FM_OK) {
    char *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

    if(!whole_dir) 
      gpx = gpx_parse(filename);
    else {
      /* cur trailing '/' if present */
      if(strlastchr(filename) == '/')
	filename[strlen(filename)] = 0;

      gpx = gpx_parse_dir(filename);
    }

    /* save path if gpx was successfully loaded */
    if(gpx) {
      char *r = strrchr(filename, '/');

      /* there is a delimiter, use everything left of it as path */
      if(r && !whole_dir) {
	*r = 0;
	if(appdata->path) free(appdata->path);
	appdata->path = strdup(filename);
	/* restore path ... just in case ... */
	*r = '/';
      }

      if(whole_dir)
	appdata->path = strdup(filename);
    }

    g_free (filename);
  }
  
  gtk_widget_destroy (dialog);

  return gpx;
}

/******************** begin of cachelist ********************/

enum {
  CACHELIST_COL_TYPE = 0,
  CACHELIST_COL_ID,
  CACHELIST_COL_NAME,
  CACHELIST_COL_SIZE,
  CACHELIST_COL_RATING,
  CACHELIST_COL_BEARING,
  CACHELIST_COL_DISTANCE,
  CACHELIST_COL_DATA,
  CACHELIST_COL_AVAIL,
  CACHELIST_COL_ARCHIVE,
  CACHELIST_NUM_COLS
} ;

void cachelist_view_onRowActivated(GtkTreeView        *treeview,
				   GtkTreePath        *path,
				   GtkTreeViewColumn  *col,
				   gpointer            userdata) {
  appdata_t *appdata = (appdata_t*)userdata;
  GtkTreeIter   iter;
  GtkTreeModel *model = gtk_tree_view_get_model(treeview);

  if(gtk_tree_model_get_iter(model, &iter, path)) {
    cache_t *cache;
    gtk_tree_model_get(model, &iter, CACHELIST_COL_DATA, &cache, -1);
#ifndef USE_MAEMO
    cache_dialog(appdata, cache);
#else
    gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
    appdata->cur_view = cache_view(appdata, cache);
    gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
    gtk_widget_show_all(appdata->vbox);

    crumb_add(appdata, cache->name, CRUMB_CACHE, cache);
#endif
  }
}

typedef struct {
  appdata_t *appdata;
  GtkTreePath *path;  
  gboolean done;
} cachelist_expose_t;

static gboolean cachelist_expose(GtkWidget *widget, GdkEventExpose *event,
				 gpointer user_data) {
  cachelist_expose_t *ce = (cachelist_expose_t*)user_data;

  if(event->type == GDK_EXPOSE) {
    if(ce->path && !ce->done) {
      gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(widget), 
				   ce->path, NULL, TRUE, 0.5, 0.5);
      gtk_tree_path_free(ce->path);
      ce->done = TRUE;
    }
  }

  return FALSE;
}

static void cachelist_destroy(GtkWidget *widget, gpointer user_data) {
  cachelist_expose_t *ce = (cachelist_expose_t*)user_data;

  printf("cachelist timer removed\n");
  g_assert(ce->appdata->cachelist_handler_id);
  gtk_timeout_remove(ce->appdata->cachelist_handler_id);
  ce->appdata->cachelist_handler_id = 0;

  free(user_data);
}

#define CACHELIST_UPDATE_TIMEOUT (30000)

static GtkWidget *cachelist_create(appdata_t *appdata, gpx_t *gpx,
				   cache_t *sel_cache);

void cachelist_redraw(appdata_t *appdata) {
#ifndef USE_MAEMO
  //  gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
  //  appdata->cur_view = gpxlist_create_view_and_model(appdata, NULL);
  //  gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
  // gtk_widget_show_all(appdata->vbox);
#else

  g_assert(!appdata->cur_cache);
  int redraw = 0;
  if(appdata->search_results)
    redraw = 1;
  else {
    if(appdata->cur_gpx) 
      redraw = 2;  // redraw cachelist
  }

  if(redraw) {
    gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
    switch(redraw) {
    case 1:
      appdata->cur_view = cachelist_create(appdata, 
					   appdata->search_results, NULL);
      break;
    case 2:
      appdata->cur_view = cachelist_create(appdata, 
					   appdata->cur_gpx, NULL);
      break;
    }

    gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
    gtk_widget_show_all(appdata->vbox);
  }
#endif
}


static gboolean cachelist_update(gpointer data) {

  printf("cachelist timer fired!\n");

#ifdef USE_MAEMO
  appdata_t *appdata = (appdata_t*)data;

  if(appdata->cachelist_disable_screensaver)
    if (osso_display_blanking_pause(appdata->osso_context) != OSSO_OK) 
      fprintf(stderr, "error with display blank\n");

  if(appdata->cachelist_update)
    cachelist_redraw(appdata);
#endif

  return TRUE;
}

static void cachelist_timer_reset(appdata_t *appdata) {
  g_assert(appdata->cachelist_handler_id);
  gtk_timeout_remove(appdata->cachelist_handler_id);
  appdata->cachelist_handler_id = 
    gtk_timeout_add(CACHELIST_UPDATE_TIMEOUT, cachelist_update, appdata);  
}

static gboolean cachelist_update_reset0(GtkWidget *widget,
					GdkEventButton *event,
					gpointer user_data) {
  cachelist_timer_reset((appdata_t*)user_data);
  return FALSE;
}

static void cachelist_update_reset1(GtkAdjustment *adj,
				    gpointer user_data) {
  cachelist_timer_reset((appdata_t*)user_data);
}
  
static GtkWidget *cachelist_create(appdata_t *appdata, gpx_t *gpx,
				   cache_t *sel_cache) {
  GtkCellRenderer *renderer;
  GtkWidget       *view;
  GtkWidget       *scrolled_window;
  GtkListStore    *store;
  GtkTreeIter     iter;

  if(!gpx->notes_loaded) {
    notes_load_all(appdata, gpx);
    gpx->notes_loaded = TRUE;
  }

  appdata->cur_items = appdata->cachelist_items;

  /* first sort the caches */
  pos_t *refpos = get_pos(appdata);
  gpx_sort(gpx, GPX_SORT_BY_DISTANCE, refpos);

  view = gtk_tree_view_new();
  
  /* --- "Type" column --- */
  renderer = custom_cell_renderer_type_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Type", renderer, "type", CACHELIST_COL_TYPE, NULL);
  
  /* --- "Id" column --- */
  if(appdata->cachelist_items & CACHELIST_ITEM_ID) {
    renderer = gtk_cell_renderer_text_new();
    gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Id", renderer, "text", CACHELIST_COL_ID, NULL);
  }

  /* --- "Name" column --- */
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, NULL );

  GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(
		 "Name", renderer, "text", CACHELIST_COL_NAME, NULL);
  gtk_tree_view_column_set_expand(column, TRUE);
  gtk_tree_view_insert_column(GTK_TREE_VIEW(view), column, -1);

  g_object_set(renderer, "foreground", "#ff0000", NULL );
  gtk_tree_view_column_add_attribute(column, renderer, "strikethrough", 
				     CACHELIST_COL_AVAIL);
  gtk_tree_view_column_add_attribute(column, renderer, 
			     "foreground-set", CACHELIST_COL_ARCHIVE);

  /* --- "Size" column --- */
  if(appdata->cachelist_items & CACHELIST_ITEM_SIZE) {
    renderer = custom_cell_renderer_size_new();
    gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Size", renderer, "size", CACHELIST_COL_SIZE, NULL);
  }

  /* --- "Rating" column --- */
  if(appdata->cachelist_items & CACHELIST_ITEM_RATING) {
    renderer = custom_cell_renderer_rating_new();
    gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Rating", renderer, "rating", CACHELIST_COL_RATING, NULL);
  }

  /* --- "Bearing" column --- */
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Bearing", renderer, "pixbuf", CACHELIST_COL_BEARING, NULL);

  /* --- "Distance" column --- */
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view),
      -1, "Distance", renderer, "text", CACHELIST_COL_DISTANCE, NULL);

  store = gtk_list_store_new(CACHELIST_NUM_COLS, G_TYPE_INT, 
			     G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT, 
			     G_TYPE_INT, GDK_TYPE_PIXBUF, G_TYPE_STRING, 
			     G_TYPE_POINTER, G_TYPE_BOOLEAN, 
			     G_TYPE_BOOLEAN);

  GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));

  GtkTreeIter sel_iter;
  gboolean sel_iter_valid = FALSE;
  GtkTreePath *path = NULL; 
  cache_t *cache = gpx->cache;
  while(cache) {
    char str[32];
    gpx_pos_get_distance_str(str, sizeof(str), 
		     *refpos, gpx_cache_pos(cache), appdata->imperial);

    int dint = (int)(cache->difficulty*2-2);
    if(dint < 0) dint = 0;
    if(dint > 8) dint = 8;

    int tint = (int)(cache->terrain*2-2);
    if(tint < 0) tint = 0;
    if(tint > 8) tint = 8;

    /* cache type includes "solved" flag in lowest bit */
    int type = (cache->type<<8) +
      (cache->notes?4:0) +
      ((cache->notes && cache->notes->override)?1:0) +
      ((cache->notes && cache->notes->found)?2:0);

    if((!(type & 2)) || !appdata->cachelist_hide_found) {

      /* Append a row and fill in some data */
      gtk_list_store_append (store, &iter);
      gtk_list_store_set(store, &iter,
			 CACHELIST_COL_TYPE, type,
			 CACHELIST_COL_ID,   cache->id,
			 CACHELIST_COL_NAME, cache->name,
			 CACHELIST_COL_SIZE, cache->container,
			 CACHELIST_COL_RATING, 100 * dint + tint,
			 CACHELIST_COL_BEARING, 
			 icon_bearing(*refpos, gpx_cache_pos(cache)),
			 CACHELIST_COL_DISTANCE, str,
			 CACHELIST_COL_DATA, cache,
			 CACHELIST_COL_AVAIL, !cache->available || 
			 cache->archived,
			 CACHELIST_COL_ARCHIVE, cache->archived,
			 -1);

      if(cache == sel_cache) {
	sel_iter = iter;
	sel_iter_valid = TRUE;
      }
    }

    cache = cache->next;
  }

  gtk_tree_view_set_model(GTK_TREE_VIEW(view), GTK_TREE_MODEL(store));
  g_object_unref(store);

  if(sel_iter_valid) {
    gtk_tree_selection_select_iter(sel, &sel_iter);
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(store), &sel_iter);
  }

  /* make list react on clicks */
  g_signal_connect(view, "row-activated", 
		   (GCallback)cachelist_view_onRowActivated, appdata);

  cachelist_expose_t *ce = malloc(sizeof(cachelist_expose_t));
  ce->appdata = appdata;
  ce->path = path;
  ce->done = FALSE;
  g_signal_connect(view, "expose-event", 
		   (GCallback)cachelist_expose, ce);
  g_signal_connect(view, "destroy",
		   (GCallback)cachelist_destroy, ce);

  /* put this inside a scrolled view */
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
				 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scrolled_window), view);

  /* add a timer for automatic update */
  g_assert(!appdata->cachelist_handler_id);
  appdata->cachelist_handler_id = 
    gtk_timeout_add(CACHELIST_UPDATE_TIMEOUT, cachelist_update, appdata);

  /* update timer is being reset if the user scrolls or selects */
  g_signal_connect(view, "button-press-event", 
		   (GCallback)cachelist_update_reset0, appdata);

  g_signal_connect(gtk_scrolled_window_get_vadjustment(
		       GTK_SCROLLED_WINDOW(scrolled_window)), 
			   "value-changed", 
			   (GCallback)cachelist_update_reset1, appdata);

  return scrolled_window;
}

#ifndef USE_MAEMO
void cachelist_dialog(appdata_t *appdata, gpx_t *gpx) {
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(gpx->name, GTK_WINDOW(appdata->window),
		       GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL | 
				       GTK_DIALOG_DESTROY_WITH_PARENT,
		       GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                       NULL);

  gtk_window_set_default_size(GTK_WINDOW(dialog), DIALOG_WIDTH, DIALOG_HEIGHT);

  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 
		    cachelist_create(appdata, gpx, NULL));

  gtk_widget_show_all(dialog);

  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
}
#endif

/******************** end of cachelist ********************/

/******************** begin of gpxlist ********************/

enum {
  GPXLIST_COL_ICON = 0,
  GPXLIST_COL_FILENAME,
  GPXLIST_COL_NAME,
  GPXLIST_COL_DATE,
  GPXLIST_COL_CACHES,
  GPXLIST_COL_DATA,
  GPXLIST_NUM_COLS
} ;

void gpxlist_view_onRowActivated(GtkTreeView *treeview,
			  GtkTreePath        *path,
			  GtkTreeViewColumn  *col,
			  gpointer            userdata) {
  appdata_t *appdata = (appdata_t*)userdata;
  GtkTreeIter   iter;
  GtkTreeModel *model = gtk_tree_view_get_model(treeview);
  
  if (gtk_tree_model_get_iter(model, &iter, path)) {
    gpx_t *gpx;
    gtk_tree_model_get(model, &iter, GPXLIST_COL_DATA, &gpx, -1);
#ifndef USE_MAEMO
    cachelist_dialog(appdata, gpx);
#else
    gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
    appdata->cur_view = cachelist_create(appdata, gpx, NULL);
    gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
    gtk_widget_show_all(appdata->vbox);

    crumb_add(appdata, gpx->name, CRUMB_CACHELIST, gpx);
#endif
  }
}

static gboolean
view_selection_func(GtkTreeSelection *selection, GtkTreeModel *model,
		     GtkTreePath *path, gboolean path_currently_selected,
		     gpointer userdata) {
  appdata_t *appdata = (appdata_t*)userdata;
  GtkTreeIter iter;
    
  if(gtk_tree_model_get_iter(model, &iter, path)) 
    gtk_widget_set_sensitive(appdata->menu_remove, !path_currently_selected);
  
  return TRUE; /* allow selection state to change */
}

GdkPixbuf *gpx_icon_get(gpx_t *gpx) {
  if(gpx->filename && g_file_test(gpx->filename, G_FILE_TEST_IS_DIR)) 
    return icon_get(ICON_FILE, 1);

  if(gpx->filename&& !strcasecmp(gpx->filename+strlen(gpx->filename)-4,".zip")) 
    return icon_get(ICON_FILE, 2);

  return icon_get(ICON_FILE, 0);
}


static GtkWidget *gpxlist_create_view_and_model(appdata_t *appdata, 
						gpx_t *sel_gpx) {
  gpx_t               *gpx = appdata->gpx;
  GtkCellRenderer     *renderer;
  GtkWidget           *scrolled_window;

  /* saved displayed items */
  appdata->cur_items = appdata->gpxlist_items;

  /* nothing selected yet */
  gtk_widget_set_sensitive(appdata->menu_remove,  FALSE);

  appdata->gpxview = gtk_tree_view_new ();
  GtkTreeSelection *selection = 
    gtk_tree_view_get_selection(GTK_TREE_VIEW(appdata->gpxview));
  gtk_tree_selection_set_select_function(selection, view_selection_func, 
					 appdata, NULL);

  /* --- "Icon" column --- */
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(appdata->gpxview),
	      -1, "Icon", renderer, "pixbuf", GPXLIST_COL_ICON, NULL);

  /* --- "FileName" column --- */
  if(appdata->gpxlist_items & GPXLIST_ITEM_FILENAME) {
    renderer = gtk_cell_renderer_text_new();
    gtk_tree_view_insert_column_with_attributes(
		GTK_TREE_VIEW(appdata->gpxview),
		-1, "Filename", renderer, "text", GPXLIST_COL_FILENAME, NULL);
  }

  /* --- "Name" column --- */
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "ellipsize", PANGO_ELLIPSIZE_END, NULL );

  GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes(
		 "Name", renderer, "text", GPXLIST_COL_NAME, NULL);
  gtk_tree_view_column_set_expand(column, TRUE);
  gtk_tree_view_insert_column(GTK_TREE_VIEW(appdata->gpxview), column, -1);

  /* --- "Date" column --- */
  if(appdata->gpxlist_items & GPXLIST_ITEM_DATE) {
    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "xalign", 1.0, NULL );
    gtk_tree_view_insert_column_with_attributes(
		GTK_TREE_VIEW(appdata->gpxview),
		-1, "Date", renderer, "text", GPXLIST_COL_DATE, NULL);
  }

  /* --- "Number of caches" column --- */
  if(appdata->gpxlist_items & GPXLIST_ITEM_CNUM) {
    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "xalign", 1.0, NULL );
    gtk_tree_view_insert_column_with_attributes(
		GTK_TREE_VIEW(appdata->gpxview),
		-1, "#Caches", renderer, "text", GPXLIST_COL_CACHES, NULL);
  }

  /* build and fill the store */
  appdata->gpxstore = gtk_list_store_new(GPXLIST_NUM_COLS, GDK_TYPE_PIXBUF,
			 G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, 
			 G_TYPE_INT, G_TYPE_POINTER);

  GtkTreePath *path = NULL; 
  GtkTreeIter sel_iter;
  gboolean sel_iter_valid = FALSE;
  while(gpx) {
    GtkTreeIter         iter;
    char date_str[32];

    if(gpx->year && gpx->month && gpx->day) {
      GDate *date = g_date_new_dmy(gpx->day, gpx->month, gpx->year);
      g_date_strftime(date_str, sizeof(date_str), "%x", date);
      g_date_free(date);
    } else
      strcpy(date_str, "---");

    char *fname = strrchr(gpx->filename, '/');
    if(!fname) fname = gpx->filename;
    else       fname++;  /* skip '/' */

    /* Append a row and fill in some data */
    gtk_list_store_append(appdata->gpxstore, &iter);
    gtk_list_store_set(appdata->gpxstore, &iter,
			GPXLIST_COL_ICON, gpx_icon_get(gpx),
			GPXLIST_COL_FILENAME, fname,
			GPXLIST_COL_NAME, gpx->name,
			GPXLIST_COL_DATE, date_str,
			GPXLIST_COL_CACHES, gpx_total_caches(gpx),
			GPXLIST_COL_DATA, gpx,
			-1);

    if(gpx == sel_gpx) {
      sel_iter = iter;
      sel_iter_valid = TRUE;
    }

    gpx = gpx->next;
  }
  
  gtk_tree_view_set_model(GTK_TREE_VIEW(appdata->gpxview), 
			  GTK_TREE_MODEL(appdata->gpxstore));

  g_object_unref(appdata->gpxstore);

  if(sel_iter_valid) {
    gtk_tree_selection_select_iter(selection, &sel_iter);
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(appdata->gpxstore), 
				   &sel_iter);
    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(appdata->gpxview), 
				 path, NULL, TRUE, 0.0, 0.0);
    gtk_tree_path_free(path);
  }

  /* make list react on clicks */
  g_signal_connect(appdata->gpxview, "row-activated", 
		   (GCallback)gpxlist_view_onRowActivated, appdata);

  /* put it into a scrolled window */
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrolled_window), 
				  GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scrolled_window), appdata->gpxview);

  return scrolled_window;
}

/* add last entry in gpx list to visual representation */
static void gpxlist_add(appdata_t *appdata, gpx_t *new) {
  GtkTreeIter         iter;
  char date_str[32];

  if(new->year && new->month && new->day) {
    GDate *date = g_date_new_dmy(new->day, new->month, new->year);
    g_date_strftime(date_str, sizeof(date_str), "%x", date);
  } else
    strcpy(date_str, "---");

  char *fname = strrchr(new->filename, '/');
  if(!fname) fname = new->filename;
  else       fname++;  /* skip '/' */
  
  /* Append a row and fill in some data */
  gtk_list_store_append(appdata->gpxstore, &iter);
  gtk_list_store_set(appdata->gpxstore, &iter,
		     GPXLIST_COL_ICON, gpx_icon_get(new),
		     GPXLIST_COL_FILENAME, fname,
		     GPXLIST_COL_NAME, new->name,
		     GPXLIST_COL_DATE, date_str,
		     GPXLIST_COL_CACHES, gpx_total_caches(new),
		     GPXLIST_COL_DATA, new,
		     -1);

  /* and attach entry to end of list */
  gpx_t **gpx = &appdata->gpx;
  while(*gpx) gpx = &((*gpx)->next);
  *gpx = new;
}

/******************** end of gpxlist ********************/

/******************** begin of menu *********************/

static void 
cb_menu_about(GtkWidget *window, gpointer data) {
  GtkAboutDialog *about = GTK_ABOUT_DIALOG(gtk_about_dialog_new());

  gtk_about_dialog_set_name(about, "GPXView");
  gtk_about_dialog_set_version(about, VERSION);
  gtk_about_dialog_set_copyright(about, 
	 _("GPXView (c) 2008 by\n" 
	 "Till Harbaum <till@harbaum.org>\n"
	 "Geocaching.com: Tantil"));

  gtk_about_dialog_set_website(about,
       _("http://www.harbaum.org/till/maemo"));
  
  gtk_about_dialog_set_comments(about, 
       _("Geocaching.com GPX file viewer"));

  gtk_widget_show_all(GTK_WIDGET(about));
  gtk_dialog_run(GTK_DIALOG(about));
  gtk_widget_destroy(GTK_WIDGET(about));
}

#ifdef USE_MAEMO
static void 
cb_menu_help(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  hildon_help_show(appdata->osso_context, HELP_ID_INTRO, 0);
}
#endif

static void 
cb_menu_add(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;

  gpx_t *new = choose_file(appdata, FALSE);
  if(new) gpxlist_add(appdata, new);
}

static void 
cb_menu_adddir(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;

  gpx_t *new = choose_file(appdata, TRUE);
  if(new) gpxlist_add(appdata, new);
}

static void 
cb_menu_geomath(GtkWidget *window, gpointer data) {
  geomath_dialog((appdata_t *)data);
}

static void 
cb_menu_geotext(GtkWidget *window, gpointer data) {
  geotext_dialog((appdata_t *)data);
}

static void 
cb_menu_precpos(GtkWidget *window, gpointer data) {
  precise_position((appdata_t *)data);
}

static void 
cb_menu_remove(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;

  /* disable menu item */
  gtk_widget_set_sensitive(appdata->menu_remove,  FALSE);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(appdata->gpxview));
  if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gpx_t *gpx;
    gtk_tree_model_get(model, &iter, GPXLIST_COL_DATA, &gpx, -1);

    /* de-chain */
    gpx_t **prev = &appdata->gpx;
    while(*prev != gpx) prev = &((*prev)->next);
    *prev = gpx->next;

    /* free gpx itself */
    gpx_free(gpx);

    /* and remove from store */
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
  } else {
    g_print ("no row selected.\n");
  }
}

static void search_result_free(gpx_t *result) {
  printf("freeing search results\n");    

  /* free found chain */
  cache_t *cache = result->cache;
  while(cache) {
    cache_t *next = cache->next;
    free(cache);
    cache = next;
  }
  free(result->name);
  free(result);
}

#define MAX_HITS 50

static time_t localize_time(time_t in) {
  time_t ret;
  char *tz;
  struct tm *tm = localtime(&in);
  
  tz = getenv("TZ");
  setenv("TZ", "", 1);
  tzset();
  ret = mktime(tm);
  if (tz)
    setenv("TZ", tz, 1);
  else
    unsetenv("TZ");
  tzset();
  return ret;
}

static int days_ago(time_t in) {
  int day_in  = localize_time(in) / (60*60*24);
  int day_now = localize_time(time(NULL)) / (60*60*24);

  return day_now - day_in;
}

gpx_t *search_do(appdata_t *appdata, gpx_t *gpx, char *phrase, 
		 int what, gboolean local) {
  /* walk through all caches */
      
  int hits = 0;
  gpx_t *found = malloc(sizeof(gpx_t));
  memset(found, 0, sizeof(gpx_t));
  cache_t **cacheP = &(found->cache);

  if(what & SEARCH_FINDS) {
    time_t loc_now = localize_time(time(NULL));
    printf("now: %ld days since 1/1/1970, days hour is %ld\n", 
	   loc_now/(60*60*24), loc_now%(60*60*24)/(60*60));
  }

  while(gpx && hits < MAX_HITS) {

    /* we need all notes ... */
    if(what & SEARCH_FINDS) {
      notes_load_all(appdata, gpx);
      gpx->notes_loaded = TRUE;      
    }

    cache_t *cache = gpx->cache;

    while(cache && hits < MAX_HITS) {
      gboolean hit = FALSE;

      if(what & SEARCH_FINDS) {
	if(cache->notes && cache->notes->found ) {
	  int days = days_ago(cache->notes->ftime);

	  if(cache->id)
	    printf("find of %s is %d days ago\n", cache->id, days);

	  if(days <= appdata->search_days)
	    hit = 1;
	}
      } else if(cache->id && (what & SEARCH_ID) && 
	 strcasestr(cache->id, phrase)) 
	hit = 1;
      else if(cache->name && (what & SEARCH_NAME) && 
	      strcasestr(cache->name, phrase))
	hit = 1;
      else if(cache->short_description && (what & SEARCH_DESC) && 
	      strcasestr(cache->short_description, phrase))
	hit = 1;
      else if(cache->long_description && (what & SEARCH_DESC) && 
	      strcasestr(cache->long_description, phrase))
	hit = 1;
      else if(cache->owner && (what & SEARCH_OWNER) && 
	      strcasestr(cache->owner, phrase))
	hit = 1;
      
      if(hit) {
	/* chain a copy of this cache structure into the found list */
	*cacheP = malloc(sizeof(cache_t));
	memcpy(*cacheP, cache, sizeof(cache_t));
	(*cacheP)->next = NULL;
	cacheP = &((*cacheP)->next);
	hits++;
      }
      cache = cache->next;
    }

    if(!local) gpx = gpx->next;
    else       gpx = NULL;       /* local search within one gpx only */
  }
  
  found->name = strdup(_("Search results"));

  return found;
}

typedef struct {
  appdata_t *appdata;
  GtkWidget *entry, *spinner;
  GtkWidget *in_id, *in_name, *in_desc, *in_owner, *in_finds;
} search_context_t;

static void callback_finds_toggled(GtkWidget *widget, gpointer data ) {
  search_context_t *context = (search_context_t*)data;

  gboolean in_finds = gtk_toggle_button_get_active(
	  GTK_TOGGLE_BUTTON(context->in_finds));

  gtk_widget_set_sensitive(context->entry,    !in_finds);
  gtk_widget_set_sensitive(context->in_id,    !in_finds);
  gtk_widget_set_sensitive(context->in_name,  !in_finds);
  gtk_widget_set_sensitive(context->in_desc,  !in_finds);
  gtk_widget_set_sensitive(context->in_owner, !in_finds);
  gtk_widget_set_sensitive(context->spinner,   in_finds);
}

static void 
cb_menu_search(GtkWidget *window, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;

  search_context_t context;
  memset(&context, 0, sizeof(search_context_t));
  context.appdata = appdata;

  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Enter search phrase"),
  	    GTK_WINDOW(appdata->window), GTK_DIALOG_MODAL,
            GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
            GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
            NULL);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
			      gtk_label_new(_("Search in:")));

  GtkWidget *table = gtk_table_new(2, 2, TRUE);
  gtk_table_set_col_spacing(GTK_TABLE(table), 0, 8);

  context.in_id = gtk_check_button_new_with_label(_("Waypoint IDs"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.in_id), 
			       appdata->search & SEARCH_ID);
  gtk_table_attach_defaults(GTK_TABLE(table), context.in_id, 0, 1, 0, 1);

  context.in_name = gtk_check_button_new_with_label(_("Names"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.in_name), 
			       appdata->search & SEARCH_NAME);
  gtk_table_attach_defaults(GTK_TABLE(table), context.in_name, 1, 2, 0, 1);

  context.in_desc = gtk_check_button_new_with_label(_("Descriptions"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.in_desc), 
			       appdata->search & SEARCH_DESC);
  gtk_table_attach_defaults(GTK_TABLE(table), context.in_desc, 0, 1, 1, 2);

  context.in_owner = gtk_check_button_new_with_label(_("Owner"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.in_owner), 
			       appdata->search & SEARCH_OWNER);
  gtk_table_attach_defaults(GTK_TABLE(table), context.in_owner, 1, 2, 1, 2);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), table);

  /* -------------------------------------------------------------- */

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
			      gtk_label_new(_("Search for:")));
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox),
			      context.entry = gtk_entry_new());
  if(appdata->search_str)
    gtk_entry_set_text(GTK_ENTRY(context.entry), appdata->search_str);

  /* -------------------------------------------------------------- */

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
			      gtk_hseparator_new());

  GtkWidget *hbox = gtk_hbox_new(FALSE, 5);

  context.in_finds = gtk_check_button_new_with_label(_("Search finds for"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(context.in_finds), 
			       appdata->search & SEARCH_FINDS);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context.in_finds);
  g_signal_connect(G_OBJECT(context.in_finds), "toggled",
		   G_CALLBACK(callback_finds_toggled), &context);

#ifndef USE_MAEMO
  GtkObject *adj = gtk_adjustment_new(appdata->search_days, 0, 99, 1, 10, 10);
  context.spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 0);
#else
  context.spinner = hildon_number_editor_new(0, 99);
  hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(context.spinner), 
				 appdata->search_days);
#endif
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context.spinner);

  gtk_box_pack_start_defaults(GTK_BOX(hbox), gtk_label_new(_("days")));

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox);

  /* -------------------------------------------------------------- */

  gtk_widget_show_all(dialog);
  callback_finds_toggled(NULL, &context);

  if(GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
    char *p = strdup(gtk_entry_get_text(GTK_ENTRY(context.entry)));

    /* update saved search string */
    if(appdata->search_str) free(appdata->search_str);
    if(strlen(p) > 0)
      appdata->search_str = strdup(p);

#ifndef USE_MAEMO
    appdata->search_days = gtk_spin_button_get_value_as_int(
		    GTK_SPIN_BUTTON(context.spinner));
#else
    appdata->search_days = hildon_number_editor_get_value(
		    HILDON_NUMBER_EDITOR(context.spinner));
#endif

    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.in_finds)))
      appdata->search |=  SEARCH_FINDS;
    else
      appdata->search &= ~SEARCH_FINDS;

    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.in_id)))
      appdata->search |=  SEARCH_ID;
    else
      appdata->search &= ~SEARCH_ID;

    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.in_name)))
      appdata->search |=  SEARCH_NAME;
    else
      appdata->search &= ~SEARCH_NAME;

    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.in_desc)))
      appdata->search |=  SEARCH_DESC;
    else
      appdata->search &= ~SEARCH_DESC;
 
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(context.in_owner)))
      appdata->search |=  SEARCH_OWNER;
    else
      appdata->search &= ~SEARCH_OWNER;
 
    gtk_widget_destroy(dialog);

    /* don't search if we are asked to search for nothing */
    if(((appdata->search & (SEARCH_ID|SEARCH_NAME|SEARCH_DESC|SEARCH_OWNER)) &&
	strlen(p) > 0) || (appdata->search & SEARCH_FINDS)) {
      gpx_t *found = NULL;

      printf("Search for %s (flags = %x)...\n", p, appdata->search);

#ifndef USE_MAEMO
      found = search_do(appdata, appdata->gpx, p, appdata->search, FALSE);

      /* do search result dialog here ... */
      cachelist_dialog(appdata, found);
      search_result_free(found);
#else
      if(appdata->cur_gpx)
	found = search_do(appdata, appdata->cur_gpx, p, appdata->search, TRUE);
      else
	found = search_do(appdata, appdata->gpx, p, appdata->search, FALSE);

      gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
      appdata->cur_view = cachelist_create(appdata, found, NULL);
      gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
      gtk_widget_show_all(appdata->vbox);
      crumb_add(appdata, found->name, 
		appdata->cur_gpx?CRUMB_SEARCH_GPX:CRUMB_SEARCH_GLOBAL, found);
#endif
    } else
      printf("No valid search: \"%s\" with flags %x!\n", p, appdata->search);

    free(p);
  } else 
    gtk_widget_destroy(dialog);
}

void on_window_destroy (GtkWidget *widget, gpointer data);

static void 
cb_menu_quit(GtkWidget *window, gpointer data) {
  on_window_destroy(window, data);
}

#ifdef USE_MAEMO
static void 
cb_menu_fullscreen(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t *)data;

  if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget)))
    gtk_window_fullscreen(GTK_WINDOW(appdata->window));
  else
    gtk_window_unfullscreen(GTK_WINDOW(appdata->window));
}

static void 
cb_menu_zoomin(GtkWidget *widget, gpointer data) {
  html_zoom((appdata_t *)data, TRUE);
}

static void 
cb_menu_zoomout(GtkWidget *widget, gpointer data) {
  html_zoom((appdata_t *)data, FALSE);
}
#endif

static void 
cb_menu_cut(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(appdata->active_buffer) {
    if(GTK_WIDGET_TYPE(appdata->active_buffer) == GTK_TYPE_TEXT_BUFFER) {
      gtk_text_buffer_cut_clipboard(GTK_TEXT_BUFFER(appdata->active_buffer), 
				    appdata->clipboard, TRUE);
    } else
      printf("cut: ERROR, not a text buffer\n");
  } else
    printf("cut: ERROR, no active buffer\n");
}

static void 
cb_menu_copy(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(appdata->active_buffer) {
    if(GTK_WIDGET_TYPE(appdata->active_buffer) == GTK_TYPE_TEXT_BUFFER) {
      gtk_text_buffer_copy_clipboard(GTK_TEXT_BUFFER(appdata->active_buffer), 
				     appdata->clipboard);
    } else if(GTK_WIDGET_TYPE(appdata->active_buffer) == gtk_html_get_type()) {
      printf("copy from html buffer\n");
      html_copy_to_clipboard(appdata);
    } else
      printf("copy: ERROR, not a text nor a html buffer\n");
  } else 
    printf("copy: ERROR, no active buffer\n");
}

static void 
cb_menu_paste(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  if(appdata->active_buffer) {
    if(GTK_WIDGET_TYPE(appdata->active_buffer) == GTK_TYPE_TEXT_BUFFER) {
      gtk_text_buffer_paste_clipboard(GTK_TEXT_BUFFER(appdata->active_buffer), 
				      appdata->clipboard, NULL, TRUE);
    } else
      printf("paste: ERROR, not a text buffer\n");
  } else
    printf("paste: ERROR, no active buffer\n");
}

static void 
cb_menu_export_log(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  notes_log_export(appdata);
}

static void 
cb_menu_export_mmpoi(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  mmpoi_export(appdata);
}

static void 
cb_menu_export_garmin(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  garmin_export(appdata);
}

void menu_create(appdata_t *appdata) { 
  GtkWidget *menu, *item;  
  menu = gtk_menu_new();

#ifdef USE_MAEMO
  appdata->menu_import = 
#endif
  item = gtk_menu_item_new_with_label(_("Import"));
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  GtkWidget *submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

  item = gtk_menu_item_new_with_label( _("Add file...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_add), appdata);

  item = gtk_menu_item_new_with_label( _("Add dir...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_adddir), appdata);

  appdata->menu_remove = 
    item = gtk_menu_item_new_with_label( _("Remove") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_remove), appdata);

#ifdef USE_MAEMO
  appdata->menu_export = 
#endif
    item = gtk_menu_item_new_with_label(_("Export"));
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

  item = gtk_menu_item_new_with_label( _("Maemo Mapper POI...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_export_mmpoi), appdata);

  item = gtk_menu_item_new_with_label( _("Garmin Field Notes...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_export_log), appdata);

  item = gtk_menu_item_new_with_label( _("Garmin GPX...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_export_garmin), appdata);

#ifdef USE_MAEMO
  appdata->menu_search = 
#endif
  item = gtk_menu_item_new_with_label( _("Search...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_search), appdata);

  gtk_menu_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());

  /* ----------- copy'n paste submenu ----------------- */
  appdata->clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
  gtk_clipboard_set_can_store(appdata->clipboard, NULL, 0);

  item = gtk_menu_item_new_with_label(_("Edit"));
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

  appdata->menu_cut = item = gtk_menu_item_new_with_label( _("Cut") );
  gtk_widget_set_sensitive(item, FALSE);
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_cut), appdata);
  appdata->menu_copy = item = gtk_menu_item_new_with_label( _("Copy") );
  gtk_widget_set_sensitive(item, FALSE);
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_copy), appdata);
  appdata->menu_paste = item = gtk_menu_item_new_with_label( _("Paste") );
  gtk_widget_set_sensitive(item, FALSE);
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_paste), appdata);

  /* ----------- view submenu ----------------- */

#ifdef USE_MAEMO
  item = gtk_menu_item_new_with_label(_("View"));
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

  appdata->fullscreen_menu_item = 
    item = gtk_check_menu_item_new_with_label( _("Fullscreen") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_fullscreen), 
		   appdata);

  appdata->zoomout_menu_item = item = gtk_menu_item_new_with_label( _("Zoom +" ));
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_zoomin), 
		   appdata);
  gtk_widget_set_sensitive(item,  FALSE);

  appdata->zoomin_menu_item = item = gtk_menu_item_new_with_label( _("Zoom -") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_zoomout), 
		   appdata);
  gtk_widget_set_sensitive(item,  FALSE);
#endif

  item = gtk_menu_item_new_with_label( _("Settings...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_settings), 
		   appdata);

  gtk_menu_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());

  item = gtk_menu_item_new_with_label(_("Tools"));
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  submenu = gtk_menu_new();
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

  item = gtk_menu_item_new_with_label( _("Geomath...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_geomath), appdata);

  item = gtk_menu_item_new_with_label( _("Geotext...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_geotext), appdata);

  item = gtk_menu_item_new_with_label( _("Precise Position...") );
  gtk_menu_append(GTK_MENU_SHELL(submenu), item);
  g_signal_connect(item, "activate", 
		   GTK_SIGNAL_FUNC(cb_menu_precpos), appdata);

  gtk_menu_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());

#ifdef USE_MAEMO
  item = gtk_menu_item_new_with_label( _("Help...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_help), appdata);
#endif

  item = gtk_menu_item_new_with_label( _("About...") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_about), appdata);


  item = gtk_menu_item_new_with_label( _("Quit") );
  gtk_menu_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_menu_quit), appdata);

#ifdef USE_MAEMO
  hildon_window_set_menu(appdata->window, GTK_MENU(menu));
#else
  /* attach ordinary gtk menu */
  GtkWidget *menu_bar = gtk_menu_bar_new();

  GtkWidget *root_menu = gtk_menu_item_new_with_label (_("Menu"));
  gtk_widget_show(root_menu);

  gtk_menu_bar_append(menu_bar, root_menu); 
  gtk_menu_item_set_submenu(GTK_MENU_ITEM (root_menu), menu);

  gtk_widget_show(menu_bar);
  gtk_box_pack_start(GTK_BOX(appdata->vbox), menu_bar, 0, 0, 0);
#endif
}

/********************* end of menu **********************/

void cleanup(appdata_t *appdata) {
  gpx_free_all(appdata->gpx);
  if(appdata->path) free(appdata->path);
  if(appdata->image_path) free(appdata->image_path);
  if(appdata->search_str) free(appdata->search_str);

  gnome_vfs_shutdown();
  icons_free();
  gps_release(appdata);

#ifdef USE_MAEMO
  if(appdata->search_results) {
    printf("freeing pending search\n");
    search_result_free(appdata->search_results);
  }

  if(appdata->osso_context)
    osso_deinitialize(appdata->osso_context);

  appdata->program = NULL;
#endif

  /* free chain of locations */
  location_t *loc = appdata->location;
  while(loc) {
    location_t *next = loc->next;
    if(loc->name) free(loc->name);
    free(loc);
    loc = next;
  }

  puts("everything is gone");
}

void on_window_destroy (GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  gconf_save_state(appdata);
  gtk_main_quit();
  appdata->window = NULL;
}

gboolean on_window_key_press(GtkWidget *widget, 
			 GdkEventKey *event, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  int handled = FALSE;

  //  printf("key event %d\n", event->keyval);

  switch(event->keyval) {
#ifdef USE_MAEMO

#if 0  /* don't do this as it interferes with gtkentry fields */
  case HILDON_HARDKEY_SELECT:
    /* remove as long as we haven't reached the main gpx list */
    if(appdata->cur_gpx)
      hildon_bread_crumb_trail_pop(HILDON_BREAD_CRUMB_TRAIL(appdata->bct));
    handled = TRUE;
    break;
#endif
    
  case HILDON_HARDKEY_INCREASE:
    html_zoom(appdata, TRUE);
    handled = TRUE;
    break;
    
  case HILDON_HARDKEY_DECREASE:
    html_zoom(appdata, FALSE);
    handled = TRUE;
    break;

  case HILDON_HARDKEY_FULLSCREEN:
    {
      gboolean fullscreen = !gtk_check_menu_item_get_active(
	       GTK_CHECK_MENU_ITEM(appdata->fullscreen_menu_item));
      gtk_check_menu_item_set_active(
	       GTK_CHECK_MENU_ITEM(appdata->fullscreen_menu_item), fullscreen);

      if(fullscreen)
	gtk_window_fullscreen(GTK_WINDOW(appdata->window));
      else
	gtk_window_unfullscreen(GTK_WINDOW(appdata->window));

      handled = TRUE;
    }
    break;
#else
  case '+':
    printf("zoom+\n");
    html_zoom(appdata, TRUE);
    handled = TRUE;
    break;
  case '-':
    printf("zoom-\n");
    html_zoom(appdata, FALSE);
    handled = TRUE;
    break;
#endif
  }
  
  return handled;
}

#ifdef USE_MAEMO
typedef struct {
  int level;
  appdata_t *appdata;
  gpointer data;
} crumb_t;

static void
crumb_back(gpointer data) {
  crumb_t *crumb = (crumb_t*)data;
  printf("crumb_back called with %d\n", crumb->level);

  /* don't do anything if main window has already been destroyed */
  if(!crumb->appdata->window) {
    printf("Main window gone ...\n");
    return;
  }

  /* whatever is being displayed: we don't need it anymore */
  gtk_container_remove(GTK_CONTAINER(crumb->appdata->vbox), 
		       crumb->appdata->cur_view);

  /* back from cache to cachelist */
  if(crumb->level == CRUMB_CACHE) {
    gpx_t *gpx = crumb->appdata->search_results;

    if(!gpx) {
      gtk_widget_set_sensitive(crumb->appdata->menu_search,  TRUE);
      gtk_widget_set_sensitive(crumb->appdata->menu_export,  TRUE);
      printf("no search data found, return to gpx\n");
      gpx = crumb->appdata->cur_gpx;
    } else
      printf("returning to search result\n");
      
    g_assert(gpx != NULL);

    crumb->appdata->cur_view = cachelist_create(crumb->appdata, gpx,
						crumb->appdata->cur_cache);

    /* returning from cache view: invalidate cache reference */
    crumb->appdata->cur_cache = NULL;

    gtk_box_pack_start_defaults(GTK_BOX(crumb->appdata->vbox), 
				crumb->appdata->cur_view);
  }

  if(crumb->level == CRUMB_SEARCH_GPX) {
    printf("returning from a local search!\n");

    g_assert((gpx_t*)crumb->data == crumb->appdata->search_results);

    search_result_free((gpx_t*)crumb->data);
    crumb->appdata->search_results = NULL;

    gtk_widget_set_sensitive(crumb->appdata->menu_search,  TRUE);

    crumb->appdata->cur_view = cachelist_create(crumb->appdata, 
					crumb->appdata->cur_gpx, NULL);
    gtk_box_pack_start_defaults(GTK_BOX(crumb->appdata->vbox), 
				crumb->appdata->cur_view);
  }

  /* back from cachelist to gpxlist */
  if((crumb->level == CRUMB_CACHELIST) || 
     (crumb->level == CRUMB_SEARCH_GLOBAL)) {

    crumb->appdata->cur_view = gpxlist_create_view_and_model(
			      crumb->appdata, crumb->appdata->cur_gpx);

    /* returning from cachelist/global search view: */
    /* invalidate gpx reference */
    crumb->appdata->cur_gpx = NULL;

    gtk_box_pack_start_defaults(GTK_BOX(crumb->appdata->vbox), 
				crumb->appdata->cur_view);

    if((crumb->level == CRUMB_SEARCH_GLOBAL) ||
       (crumb->level == CRUMB_SEARCH_GPX)) {
      g_assert((gpx_t*)crumb->data == crumb->appdata->search_results);

      search_result_free((gpx_t*)crumb->data);
      crumb->appdata->search_results = NULL;
    }

    /* enable gpxlist related menu entries */
    gtk_widget_set_sensitive(crumb->appdata->menu_import,  TRUE);
    gtk_widget_set_sensitive(crumb->appdata->menu_search,  TRUE);
    gtk_widget_set_sensitive(crumb->appdata->menu_export,  TRUE);
  }

  gtk_widget_show_all(crumb->appdata->vbox);    
  g_free(data);
}

static void crumb_add(appdata_t *appdata, char *name, int level, 
		      gpointer user_data) {
  crumb_t *crumb = malloc(sizeof(crumb_t));
  crumb->level = level;
  crumb->appdata = appdata;
  crumb->data = user_data;

  printf("crumb_add with level %d\n", level);

  /* save that we are working on search results */
  if((level == CRUMB_SEARCH_GLOBAL) ||
     (level == CRUMB_SEARCH_GPX)) {
    appdata->search_results = (gpx_t*)user_data;

    /* searches cannot be nested */
    gtk_widget_set_sensitive(appdata->menu_search,  FALSE);
  }

  /* save "path" pointers in appdata */
  if(crumb->level == CRUMB_CACHELIST) 
    appdata->cur_gpx = (gpx_t*)user_data;

  if(crumb->level == CRUMB_CACHE) {
    appdata->cur_cache = (cache_t*)user_data;
    /* the cache view cannot be searched */
    gtk_widget_set_sensitive(appdata->menu_search,  FALSE);
    gtk_widget_set_sensitive(appdata->menu_export,  FALSE);
  }

  if(level != CRUMB_GPXLIST) {
    /* disable gpxlist related menu entries */
    gtk_widget_set_sensitive(appdata->menu_import,  FALSE);
    gtk_widget_set_sensitive(appdata->menu_remove,  FALSE);
  }

  hildon_bread_crumb_trail_push_text(HILDON_BREAD_CRUMB_TRAIL(appdata->bct),
				     name, crumb, (GDestroyNotify)crumb_back);
}
#endif // USE_MAEMO

void main_after_settings_redraw(appdata_t *appdata, int flags) {
#ifndef USE_MAEMO
  // in non-maemo setup this can only affect the main screen as
  // the menu is blocked while a dialog is open. also the main
  // screen is always present
  if(appdata->gpxlist_items != appdata->cur_items) {
    /* re-do the main screen */
    gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
    appdata->cur_view = gpxlist_create_view_and_model(appdata, NULL);
    gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
    gtk_widget_show_all(appdata->vbox);
  }
#else
  /* a cache screen cannot be changed from the settings and thus doesn't */
  /* need to be redrawn */
  if(appdata->cur_cache) {
    printf("No redraw in cache view required\n");
    return;
  }

  int redraw = 0;    // nothing to redraw

  if(appdata->search_results) {
    if((appdata->cur_items != appdata->cachelist_items) || flags)
      redraw = 1;
  } else {
    if(!appdata->cur_gpx) {
      if(appdata->cur_items != appdata->gpxlist_items) 
	redraw = 2;  // redraw gpxlist
    } else {
      if((appdata->cur_items != appdata->cachelist_items) || flags)
	redraw = 3;  // redraw cachelist
    }
  }

  if(redraw) {
    gtk_container_remove(GTK_CONTAINER(appdata->vbox), appdata->cur_view);
    switch(redraw) {
    case 1:
      appdata->cur_view = cachelist_create(appdata, 
					   appdata->search_results, NULL);
      break;
    case 2:
      appdata->cur_view = gpxlist_create_view_and_model(appdata, NULL);
      break;
    case 3:
      appdata->cur_view = cachelist_create(appdata, 
					   appdata->cur_gpx, NULL);
      break;
    }

    gtk_box_pack_start_defaults(GTK_BOX(appdata->vbox), appdata->cur_view);
    gtk_widget_show_all(appdata->vbox);
  }
#endif
}

int main(int argc, char *argv[]) {
  appdata_t appdata;

  /* init appdata */
  memset(&appdata, 0, sizeof(appdata));

  printf("Using locale for %s in %s\n", PACKAGE, LOCALEDIR);

  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(PACKAGE, "UTF-8");
  textdomain(PACKAGE);

  /* prepare thread system */
  g_thread_init(NULL);

  gtk_init (&argc, &argv);

#ifdef USE_MAEMO
  printf("Installing osso context for \"org.harbaum." APP "\"\n");
  appdata.osso_context = osso_initialize("org.harbaum."APP, 
					 VERSION, TRUE, NULL);
  if(appdata.osso_context == NULL) {
    fprintf(stderr, "error initiating osso context\n");
  }

  dbus_register(&appdata);
#endif

  icons_init();

  if(!gnome_vfs_init()) {
    g_error("Gnome VFS init failed\n");
  }

  appdata.gconf_client = gconf_client_get_default();
  gconf_load_state(&appdata);

  gps_init(&appdata);

  /* attach all gpx files given on command line */
  if(argc >= 2) {
    int i;
    gpx_t **cur = &appdata.gpx;
    while(*cur) cur = &((*cur)->next);

    for(i=1;i<argc;i++) {
      if((*cur = gpx_parse(argv[i])))
	cur = &(*cur)->next;
    }
  }

  //  gpx_display_all(gpx);

#ifdef USE_MAEMO
  /* Create the hildon program and setup the title */
  appdata.program = HILDON_PROGRAM(hildon_program_get_instance());
  g_set_application_name("GPXView");
  
  /* Create HildonWindow and set it to HildonProgram */
  appdata.window = HILDON_WINDOW(hildon_window_new());
  hildon_program_add_window(appdata.program, appdata.window);
#else
  /* Create a Window. */
  appdata.window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(appdata.window), "GPXView");
  /* Set a decent default size for the window. */
  gtk_window_set_default_size(GTK_WINDOW(appdata.window), 500, 300);
#endif

  g_signal_connect(G_OBJECT(appdata.window), "destroy", 
		   G_CALLBACK(on_window_destroy), &appdata);

  g_signal_connect(G_OBJECT(appdata.window), "key_press_event",
		   G_CALLBACK(on_window_key_press), &appdata);

  appdata.vbox = gtk_vbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(appdata.window), appdata.vbox);

  menu_create(&appdata);

#ifdef USE_MAEMO
  appdata.bct = hildon_bread_crumb_trail_new();

  gtk_box_pack_start(GTK_BOX(appdata.vbox), appdata.bct, FALSE,FALSE,0);

  hildon_bread_crumb_trail_clear(HILDON_BREAD_CRUMB_TRAIL(appdata.bct));
  crumb_add(&appdata, "GPX", CRUMB_GPXLIST, NULL);
#endif

  appdata.cur_view = gpxlist_create_view_and_model(&appdata, NULL);
  gtk_box_pack_start_defaults(GTK_BOX(appdata.vbox), appdata.cur_view);

  gtk_widget_show_all(GTK_WIDGET(appdata.window));
  gtk_main();

  cleanup(&appdata);

  return 0;
}
