/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include <math.h>

typedef struct {
  appdata_t *appdata;
  GtkTextBuffer *buffer;
  GtkWidget *entry, *label_sum, *menu;
} geotext_context_t;

/* buffer edited */
static void callback_modified(GtkAction * action, gpointer data ) {
  geotext_context_t *context = (geotext_context_t*)data;

  if(!context->label_sum) return;

  GtkTextIter start, end;
  gtk_text_buffer_get_start_iter(context->buffer, &start);
  gtk_text_buffer_get_end_iter(context->buffer, &end);
  char *text = gtk_text_buffer_get_text(context->buffer, &start, &end, FALSE);

  int i=0, sum=0;
  while(text[i]) {
    if(text[i] >= 'A' && text[i] <= 'Z') sum += text[i]-'A'+1;
    if(text[i] >= 'a' && text[i] <= 'z') sum += text[i]-'a'+1;
    i++;
  }

  char str[32];
  snprintf(str, sizeof(str), "%d", sum);
  gtk_label_set_text(GTK_LABEL(context->label_sum), str);
}

static unsigned char rot_chr(unsigned char chr, int dir) {
  /* only characters can be shifted */
  if((chr >= 'A') && (chr <= 'Z')) chr = (chr - 'A' + dir)%26 + 'A';
  if((chr >= 'a') && (chr <= 'z')) chr = (chr - 'a' + dir)%26 + 'a';

  return chr;
}

static void text_shift(geotext_context_t *context, int dir) {
  int step = atoi((char*)gtk_entry_get_text(GTK_ENTRY(context->entry)));

  if((step < 0) || (step > 26)) {
    step %= 26;
    if(step < 0) step += 26;

    char str[32];
    snprintf(str, sizeof(str), "%d", step);
    gtk_entry_set_text(GTK_ENTRY(context->entry), str);
  }

  dir *= step;
  if(dir < 0) dir += 26;

  GtkTextIter start, end;
  gtk_text_buffer_get_start_iter(context->buffer, &start);
  gtk_text_buffer_get_end_iter(context->buffer, &end);
  char *text = gtk_text_buffer_get_text(context->buffer, &start, &end, FALSE);

  int i;
  for(i=0;i<strlen(text);i++) 
    text[i] = rot_chr(text[i], dir);

  gtk_text_buffer_set_text(context->buffer, text, -1);
  free(text);

  callback_modified(NULL, context);
}

static void on_left_shift_clicked(GtkButton *button, gpointer data) {
  text_shift((geotext_context_t*)data, -1);
}

static void on_right_shift_clicked(GtkButton *button, gpointer data) {
  text_shift((geotext_context_t*)data, +1);
}

static void 
cb_cut(GtkWidget *widget, gpointer data) {
  geotext_context_t *context = (geotext_context_t*)data;

  gtk_text_buffer_cut_clipboard(GTK_TEXT_BUFFER(context->buffer), 
				context->appdata->clipboard, TRUE);
}

static void 
cb_copy(GtkWidget *widget, gpointer data) {
  geotext_context_t *context = (geotext_context_t*)data;

  gtk_text_buffer_copy_clipboard(GTK_TEXT_BUFFER(context->buffer), 
				 context->appdata->clipboard);
}

static void 
cb_paste(GtkWidget *widget, gpointer data) {
  geotext_context_t *context = (geotext_context_t*)data;

  gtk_text_buffer_paste_clipboard(GTK_TEXT_BUFFER(context->buffer), 
				  context->appdata->clipboard, NULL, TRUE);
}

static gint button_press(GtkWidget *widget, GdkEventButton *event, 
			 gpointer data) {
  geotext_context_t *context = (geotext_context_t*)data;

  if (event->type == GDK_BUTTON_PRESS) {
    printf("button press %d %d\n", event->button, event->time);

    if(!context->menu) {
      context->menu = gtk_menu_new();
      GtkWidget *item = gtk_menu_item_new_with_label(_("Cut"));
      g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_cut), context);
      gtk_menu_shell_append(GTK_MENU_SHELL(context->menu),item);
      item = gtk_menu_item_new_with_label(_("Copy"));
      g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_copy), context);
      gtk_menu_shell_append(GTK_MENU_SHELL(context->menu),item);
      item = gtk_menu_item_new_with_label(_("Paste"));
      g_signal_connect(item, "activate", GTK_SIGNAL_FUNC(cb_paste), context);
      gtk_menu_shell_append(GTK_MENU_SHELL(context->menu),item);
      gtk_widget_show_all( GTK_WIDGET(context->menu) );
    }

    gtk_menu_popup(GTK_MENU(context->menu), NULL, NULL, NULL, NULL,
		   event->button, event->time);
    /* Tell calling code that we have handled this event; the buck
     * stops here. */
    return TRUE;
  }
  
  /* Tell calling code that we have not handled this event; pass it on. */
  return FALSE;
}

void geotext_dialog(appdata_t *appdata) {
  geotext_context_t context;

  memset(&context, 0, sizeof(geotext_context_t));
  context.appdata = appdata;

  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Geotext"),
                          GTK_WINDOW(appdata->window),
			  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                          GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                          NULL);

#ifdef USE_MAEMO
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog), HELP_ID_GEOTEXT, 
				 appdata->osso_context);
#endif

  gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 250);

  GtkWidget *hbox = gtk_hbox_new(FALSE, 2);

  GtkWidget *label = gtk_label_new(_("Text:"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.f, 0.5f);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);

  GtkWidget *but = gtk_button_new_with_label(_("Edit"));
  gtk_widget_set_events(but, GDK_EXPOSURE_MASK);
  gtk_widget_add_events(but, GDK_BUTTON_PRESS_MASK);
  gtk_signal_connect(GTK_OBJECT(but), "button-press-event",
		     (GtkSignalFunc)button_press, &context);
  gtk_box_pack_start(GTK_BOX(hbox), but, FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
		     hbox, FALSE, FALSE, 0);

  GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), 
  				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  
  context.buffer = gtk_text_buffer_new(NULL);
  g_signal_connect(G_OBJECT(context.buffer), "changed",
		   G_CALLBACK(callback_modified), &context);
 
  /* set saved text */
  if(appdata->geotext_text)
    gtk_text_buffer_set_text(context.buffer, appdata->geotext_text, -1);

  GtkWidget *view = gtk_text_view_new_with_buffer(context.buffer);
  gtk_text_view_set_editable(GTK_TEXT_VIEW(view), TRUE);
  //  gtk_text_view_set_left_margin(GTK_TEXT_VIEW(view), 2 );
  //  gtk_text_view_set_right_margin(GTK_TEXT_VIEW(view), 2 );

  gtk_container_add(GTK_CONTAINER(scrolled_window), view);

  gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW(scrolled_window),
				       GTK_SHADOW_IN);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
		     scrolled_window, TRUE, TRUE, 0);
  

  /* ----------------- cesar/shift -------------------  */
  hbox = gtk_hbox_new(FALSE,2);

  gtk_box_pack_start(GTK_BOX(hbox), 
     gtk_label_new(_("Cesar/Shift:")), FALSE, FALSE, 0);

  GtkWidget *button = gtk_button_new_with_label(_("<<"));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     (GtkSignalFunc)on_left_shift_clicked, &context);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

  context.entry = gtk_entry_new();
  char str[32];
  snprintf(str, sizeof(str), "%d", appdata->geotext_shift);
  gtk_entry_set_text(GTK_ENTRY(context.entry), str);
  gtk_box_pack_start(GTK_BOX(hbox), context.entry, FALSE, FALSE, 0);

  button = gtk_button_new_with_label(_(">>"));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     (GtkSignalFunc)on_right_shift_clicked, &context);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
		     hbox, FALSE, FALSE, 0);
  
  /* ----------------- character sum -------------------  */

  hbox = gtk_hbox_new(FALSE,2);

  gtk_box_pack_start(GTK_BOX(hbox), 
     gtk_label_new(_("Character sum (A=1,B=2,...):")), FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(hbox), 
	     context.label_sum = gtk_label_new(""), FALSE, FALSE, 0);
  callback_modified(NULL, &context);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), 
		     hbox, FALSE, FALSE, 0);

  /* ------------------------------------------------  */
  
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_CLOSE);

  gtk_widget_show_all(dialog);
  gtk_dialog_run(GTK_DIALOG(dialog));

  /* ----------------- read state ----------------------  */
  GtkTextIter start, end;
  gtk_text_buffer_get_start_iter(context.buffer, &start);
  gtk_text_buffer_get_end_iter(context.buffer, &end);
  appdata->geotext_text = 
    gtk_text_buffer_get_text(context.buffer, &start, &end, FALSE);

  int step = atoi((char*)gtk_entry_get_text(GTK_ENTRY(context.entry)));
  if((step < 0) || (step > 26)) { step %= 26; if(step < 0) step += 26; }
  appdata->geotext_shift = step;

  gtk_widget_destroy(dialog);
}
