/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <libxml/parser.h>
#include <libxml/tree.h>

#include <glib.h>
#include <glib/gstdio.h>

#include <zlib.h>

#include "gpxview.h"
#include "unzip.h"

#ifndef LIBXML_TREE_ENABLED
#error "Tree not enabled in libxml"
#endif

void gpx_free_wpt(wpt_t *wpt) {
  if(wpt->id)   xmlFree(wpt->id);
  if(wpt->cmt)  xmlFree(wpt->cmt);
  if(wpt->desc) xmlFree(wpt->desc);
  free(wpt);
}

void gpx_free_log(log_t *log) {
  if(log->finder)  xmlFree(log->finder);
  if(log->text)    xmlFree(log->text);
  free(log);
}

void gpx_free_tb(tb_t *tb) {
  if(tb->name) xmlFree(tb->name);
  if(tb->ref)  xmlFree(tb->ref);
  free(tb);
}

void gpx_free_cache(cache_t *cache) {
  log_t *log = cache->log;
  wpt_t *wpt = cache->wpt;
  tb_t  *tb  = cache->tb;

  if(cache->id)                xmlFree(cache->id);
  if(cache->name)              xmlFree(cache->name);
  if(cache->owner)             xmlFree(cache->owner);
  if(cache->short_description) xmlFree(cache->short_description);
  if(cache->long_description)  xmlFree(cache->long_description);
  if(cache->hint)              xmlFree(cache->hint);
  if(cache->url)               xmlFree(cache->url);

  /* free all logs */
  while(log) { log_t *next = log->next; gpx_free_log(log); log = next; }

  /* free all waypoints */
  while(wpt) { wpt_t *next = wpt->next; gpx_free_wpt(wpt); wpt = next; }

  /* free all tbs */
  while(tb) { tb_t *next = tb->next; gpx_free_tb(tb); tb = next; }

  if(cache->notes)             notes_free(cache->notes);

  free(cache);
}

void gpx_free(gpx_t *gpx) {
  cache_t *cache = gpx->cache;

  if(gpx->name)     xmlFree(gpx->name);
  if(gpx->desc)     xmlFree(gpx->desc);
  if(gpx->filename) free(gpx->filename);
  
  /* free all caches */
  while(cache) {
    cache_t *next = cache->next;
    gpx_free_cache(cache);
    cache = next;
  }

  free(gpx);
}

void gpx_free_all(gpx_t *gpx) {
  while(gpx) {
    gpx_t *next = gpx->next;
    gpx_free(gpx);
    gpx = next;
  }
}

static const char *cache_type_str[] = { "<Unknown>",
  "Traditional Cache|Traditional|Geocache", "Multi-cache|Multi", 
  "Unknown Cache|Other",
  "Virtual Cache|Virtual", "Webcam Cache|Webcam", "Event Cache|Event",
  "Letterbox Hybrid", "Earthcache", "Wherigo Cache",
  "Mega-Event Cache", "Cache In Trash Out Event",
  ""};

static const char *cache_container_str[] = { "<Unknown>",
  "Regular", "Small", "Micro", "Not chosen|Unknown", 
  "Other", "Large", "Virtual"
  ""};

static const char *log_type_str[] = { "<Unknown>",
  "Found it|Found", "Didn't find it|Not Found", "Owner Maintenance", 
  "Write Note|Note|Other", 
  "Post Reviewer Note", "Enable Listing", "Publish Listing", "Will Attend", 
  "Attended", "Webcam Photo taken", "Temporarily Disable Listing", 
  "Needs Maintenance", "Update Coordinates", "Unarchive",
  "Needs Archived", "Archive",
  ""};

static const char *wpt_sym_str[] = { "<Unknown>",
  "Stages of a Multicache", "Parking Area", "Final Location", 
  "Question to Answer", "Trailhead", "Reference Point",
  ""};

static int str_search(const char *pstr[], char *str, char *type) {
  int i=0, ret = -1;

  while(pstr[i+1][0]) {
    char *p = (char*)pstr[i+1];

    /* multiple substrings in pattern? */
    while(strchr(p, '|')) {
      if(!strncasecmp(p, str, strchr(p, '|')-p))
	return i;

      p = strchr(p, '|')+1;
    }

    if(!strcasecmp(p, str))
      return i;

    i++;
  }

  fprintf(stderr, "ERROR parsing \"%s\": Unknown \"%s\"\n", type, str);
  return -1;
}

static tb_t *gpx_parse_gpx_wpt_tbs(xmlDocPtr doc, xmlNode *a_node) {
  tb_t *tb_chain = NULL, *tb;
  xmlNode *cur_node = NULL;

  for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      if(strcasecmp(cur_node->name, "travelbug") == 0) {
	xmlNode *sub_node = cur_node->children;
	
	/* create a new log entry */
	tb = malloc(sizeof(tb_t));
	memset(tb, 0, sizeof(tb_t));
	tb->ref = xmlGetProp(cur_node, "ref");
	if(!tb->ref) tb->ref = strdup("<NONE>");
	  
	while (sub_node != NULL) {
	  if (sub_node->type == XML_ELEMENT_NODE) {
	    if(strcasecmp(sub_node->name, "name") == 0) 
	      tb->name = xmlNodeListGetString(doc, sub_node->children, 1);
	    else
	      printf("  tb unused %s\n", sub_node->name);
	  }
	  sub_node = sub_node->next;
	}

	/* add tb to chain */
	tb_t **cur = &tb_chain;
	while(*cur && (strcmp(tb->ref, (*cur)->ref)<0))
	  cur = &((*cur)->next);

	tb->next = *cur;
	*cur = tb;
      }
    }
  }
  return tb_chain;
}

static int log_is_older(log_t *a, log_t *b) {
  if(a->year < b->year)  return TRUE;
  else if(a->year == b->year) {
    if(a->month < b->month) return TRUE;
    else if(a->month == b->month) {
      if(a->day   < b->day)   return TRUE;
    }
  }

  return FALSE;
}

/* parse log entry */
static log_t *gpx_parse_gpx_wpt_logs(xmlDocPtr doc, xmlNode *a_node) {
  log_t *log_chain = NULL, *log;
  xmlNode *cur_node = NULL;

  for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {

      if(strcasecmp(cur_node->name, "log") == 0) {
	xmlNode *sub_node = cur_node->children;
	
	/* create a new log entry */
	log = malloc(sizeof(log_t));
	memset(log, 0, sizeof(log_t));

	while (sub_node != NULL) {
	  if (sub_node->type == XML_ELEMENT_NODE) {
	    char *str;

	    if((strcasecmp(sub_node->name, "date") == 0) ||
	       (strcasecmp(sub_node->name, "time") == 0)) {
	      if(str = xmlNodeListGetString(doc, sub_node->children, 1)) {
		sscanf(str, "%d-%d-%d", &log->year, &log->month, &log->day);
		xmlFree(str);
	      }
	    } else if(strcasecmp(sub_node->name, "type") == 0) {
	      if(str = xmlNodeListGetString(doc, sub_node->children, 1)) {
		log->type = str_search(log_type_str, str, "log");
		xmlFree(str);
	      }
	    } else if((strcasecmp(sub_node->name, "finder") == 0) ||
		      (strcasecmp(sub_node->name, "geocacher") == 0)) {
	      log->finder = xmlNodeListGetString(doc, sub_node->children, 1);
	    } else if(strcasecmp(sub_node->name, "text") == 0) {
	      log->text = xmlNodeListGetString(doc, sub_node->children, 1);
	    } else {
	      //	      printf("  log unused = %s\n", sub_node->name);
	    }
	  }
	  sub_node = sub_node->next;
	}

	/* add log to chain */
	log_t **cur = &log_chain;
	while(*cur && log_is_older(log, *cur))
	  cur = &((*cur)->next);

	log->next = *cur;
	*cur = log;
      }      
    }
  }
  return log_chain;
}

int is_white(char c) {
  return((c==' ')||(c=='\r')||(c=='\n'));
}

static int all_is_white(char *str) {
  while(*str) {
    if(!is_white(*str))
      return FALSE;

    str++;
  }
  return TRUE;
}

/* parse waypoint entry */
static void gpx_parse_gpx_wpt(gpx_t *gpx, xmlDocPtr doc, xmlNode *a_node,
			      pos_t *pos) {
  xmlNode *cur_node = NULL;
  cache_t **cache = &gpx->cache;

  /* skip to end of list and create a new entry */
  while(*cache) 
    cache = &(*cache)->next;

  *cache = malloc(sizeof(cache_t));
  memset(*cache, 0, sizeof(cache_t));

  (*cache)->pos.lat = pos->lat;
  (*cache)->pos.lon = pos->lon;

  /* set some defaults */
  (*cache)->type = CACHE_TYPE_UNKNOWN;
  (*cache)->container = CACHE_CONT_UNKNOWN;
  (*cache)->available = TRUE;
  
  for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      char *prop, *str;

      if(strcasecmp(cur_node->name, "name") == 0) {
	(*cache)->id = xmlNodeListGetString(doc, cur_node->children, 1);
      } else if(strcasecmp(cur_node->name, "url") == 0) {
	(*cache)->url = xmlNodeListGetString(doc, cur_node->children, 1);

	/* GC names them cache, OC names them geocache */
      } else if((strcasecmp(cur_node->name, "cache") == 0) ||  
		(strcasecmp(cur_node->name, "geocache") == 0)) {

	/* try to read "available" and "archived" marker */
	if((prop = xmlGetProp(cur_node, "available"))) {
	  (*cache)->available = (strcasecmp(prop, "true") == 0);
	  xmlFree(prop);
	}

	if((prop = xmlGetProp(cur_node, "archived"))) {
	  (*cache)->archived = (strcasecmp(prop, "true") == 0);
	  xmlFree(prop);
	}

	/* think about the children only */
	xmlNode *sub_node = cur_node->children;
	while (sub_node != NULL) {
	  if (sub_node->type == XML_ELEMENT_NODE) {

	    if(strcasecmp(sub_node->name, "name") == 0) {
	      (*cache)->name = 
		xmlNodeListGetString(doc, sub_node->children, 1);
	    } else if(strcasecmp(sub_node->name, "owner") == 0) {
	      (*cache)->owner = 
		xmlNodeListGetString(doc, sub_node->children, 1);
	    } else if(strcasecmp(sub_node->name, "type") == 0) {
	      if(str = xmlNodeListGetString(doc, sub_node->children, 1)) {
		(*cache)->type = str_search(cache_type_str, str, "cache type");
		xmlFree(str);
	      }
	    } else if(strcasecmp(sub_node->name, "container") == 0) {
	      if(str = xmlNodeListGetString(doc, sub_node->children, 1)) {
		(*cache)->container = str_search(cache_container_str, str, 
						 "container");
		xmlFree(str);
	      }
	    } else if((strcasecmp(sub_node->name, "short_description") == 0) ||
		      (strcasecmp(sub_node->name, "summary") == 0)) {
	      (*cache)->short_description = 
		xmlNodeListGetString(doc, sub_node->children, 1);
	      if((prop = xmlGetProp(sub_node, "html"))) {
		(*cache)->short_is_html = (strcasecmp(prop, "true") == 0);
		xmlFree(prop);
	      }
	    } else if((strcasecmp(sub_node->name, "long_description") == 0) ||
		      (strcasecmp(sub_node->name, "description") == 0)) {
	      (*cache)->long_description = 
		xmlNodeListGetString(doc, sub_node->children, 1);
	      if((prop = xmlGetProp(sub_node, "html"))) {
		(*cache)->long_is_html = (strcasecmp(prop, "true") == 0);
		xmlFree(prop);
	      }
	    } else if((strcasecmp(sub_node->name, "encoded_hints") == 0) ||
		      (strcasecmp(sub_node->name, "hints") == 0)) {
	      if(str = xmlNodeListGetString(doc, sub_node->children, 1)) {
		/* often hints aren't more than just a bunch of blanks ... */
		if(!all_is_white(str)) {
		  (*cache)->hint = str;
		  if((prop = xmlGetProp(sub_node, "html"))) {
		    (*cache)->hint_is_html = (strcasecmp(prop, "true") == 0);
		    xmlFree(prop);
		  }
		} else
		  xmlFree(str);		
	      }
	    } else if(strcasecmp(sub_node->name, "difficulty") == 0) {
	      if(str = xmlNodeListGetString(doc, sub_node->children, 1)) {
		(*cache)->difficulty = g_ascii_strtod(str, NULL);
		xmlFree(str);
	      }
	    } else if(strcasecmp(sub_node->name, "terrain") == 0) {
	      if(str = xmlNodeListGetString(doc, sub_node->children, 1)) {
		(*cache)->terrain = g_ascii_strtod(str, NULL);
		xmlFree(str);
	      }
	    } else if(strcasecmp(sub_node->name, "logs") == 0) {
	      (*cache)->log = gpx_parse_gpx_wpt_logs(doc, sub_node->children);
	    } else if(strcasecmp(sub_node->name, "travelbugs") == 0) {
	      (*cache)->tb = gpx_parse_gpx_wpt_tbs(doc, sub_node->children);
	    } else {
	      if((strcasecmp(sub_node->name, "country") != 0) &&
		 (strcasecmp(sub_node->name, "licence" /* OC */) != 0) &&
		 (strcasecmp(sub_node->name, "state") != 0) &&
		 (strcasecmp(sub_node->name, "locale" /* OC */) != 0) &&
		 (strcasecmp(sub_node->name, "placed_by" /* OC */)  != 0)) {
		if(str = xmlNodeListGetString(doc, sub_node->children, 1)) {
		  printf("  cache unused %s: %s\n", sub_node->name, str);
		  xmlFree(str);		
		}
	      }
	    }
	  }
	  sub_node = sub_node->next;
	}
      } else {
	/* ignore some by purpose (some are parsed seperately below) */
	if((strcasecmp(cur_node->name, "time") != 0) &&
	   (strcasecmp(cur_node->name, "desc") != 0) &&
	   (strcasecmp(cur_node->name, "urlname") != 0) &&
	   (strcasecmp(cur_node->name, "sym") != 0) &&
	   (strcasecmp(cur_node->name, "cmt") != 0) &&
	   (strcasecmp(cur_node->name, "src") != 0) &&
	   (strcasecmp(cur_node->name, "wptExtension") != 0) &&
	   (strcasecmp(cur_node->name, "type") != 0)) {

	  if(str = xmlNodeListGetString(doc, cur_node->children, 1)) {
	    printf("  wpt unused %s: %s\n", cur_node->name, str);
	    xmlFree(str);		
	  }
	}
      }
    }
 }

  /* special handling for opencaching.de caches */
  if(strncasecmp((*cache)->id, "OC", 2) == 0) {
    /* the html attributes are either missing or wrong on OC ... */
    (*cache)->long_is_html = TRUE;
    (*cache)->hint_is_html = TRUE;
    //    (*cache)->logs_are_html = TRUE;
  }

  /* neither geocaching.com GC* nor opencaching.com OC* nor */
  /* geocaching australia GA* waypoint */
  if((strncasecmp((*cache)->id, "GC", 2) != 0) &&
     (strncasecmp((*cache)->id, "OC", 2) != 0) &&
     (strncasecmp((*cache)->id, "GA", 2) != 0)) {
    cache_t *parent = gpx->cache;

    /* check if the gpx file contains a cache with matching name */
    while(parent && strcasecmp(parent->id+2, (*cache)->id+2)) 
      parent = parent->next;

    if(parent && parent != *cache) {
      wpt_t **wpt = &parent->wpt;
      char *str;

      /* search end of list */
      while(*wpt && (strcmp((*wpt)->id, (*cache)->id)<0))
	wpt = &(*wpt)->next;

      *wpt = malloc(sizeof(wpt_t));
      memset(*wpt, 0, sizeof(wpt_t));
      
      /* transfer name to waypoint entry */
      (*wpt)->id = (*cache)->id;
      (*cache)->id = NULL;

      (*wpt)->pos.lat = pos->lat;
      (*wpt)->pos.lon = pos->lon;

      /* re-parse entry */
      for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
	if (cur_node->type == XML_ELEMENT_NODE) {
	  /*  pos_t pos; */
	  if(strcasecmp(cur_node->name, "cmt") == 0) {
	    (*wpt)->cmt = xmlNodeListGetString(doc, cur_node->children, 1);
	  } else if(strcasecmp(cur_node->name, "desc") == 0) {
	    (*wpt)->desc = xmlNodeListGetString(doc, cur_node->children, 1);
	  } else if(strcasecmp(cur_node->name, "sym") == 0) {
	    if(str = xmlNodeListGetString(doc, cur_node->children, 1)) {
	      (*wpt)->sym = str_search(wpt_sym_str, str, "wpt sym");
	      xmlFree(str);
	    }
	  }
	}
      }

      /* and just free the current cache entry as we now have used */
      /* the data for a caches waypoint */
      gpx_free_cache(*cache);
      *cache = NULL;
    } else {
      /* if it doesn't have a name etc, it's probably not a real */
      /* cache, so drop it */
      if(!(*cache)->name || !(*cache)->id) {
	printf("Orphaned waypoint: %s\n", (*cache)->id);
	gpx_free_cache(*cache);
	*cache = NULL;
      }
    }
  } else {
    /* this is known to be a geocache due to its waypoint name */
    /* (gc*, oc*, ga*) and is thus forces to be an entry */
    if(!(*cache)->name) {
      (*cache)->name = malloc(strlen((*cache)->id) + sizeof("Unnamed ()")+1);
      snprintf((*cache)->name, strlen((*cache)->id) + sizeof("Unnamed ()")+1,
	       "Unnamed(%s)", (*cache)->id);
    }
  }
}

void gpx_display_log(log_t *log) {
  printf("  Log:\n");
  printf("    date:     %d.%d.%d\n", log->day, log->month, log->year);
  printf("    type:     %s\n", log_type_str[log->type+1]);
  printf("    finder:   %s\n", log->finder);
  //  printf("    text:     %s\n", log->text);
}

void gpx_display_cache(cache_t *cache) {
  log_t *log = cache->log;

  printf("\nCache:\n");
  printf("  id:         %s\n", cache->id);
  printf("  name:       %s\n", cache->name);
  printf("  latitude:   %f\n", cache->pos.lat);
  printf("  longitude:  %f\n", cache->pos.lon);
  printf("  owner:      %s\n", cache->owner);
  printf("  type:       %s\n", cache_type_str[cache->type+1]);
  printf("  container:  %s\n", cache_container_str[cache->container+1]);
  printf("  difficulty: %.1f\n", cache->difficulty);
  printf("  terrain:    %.1f\n", cache->terrain);
  //  printf("  short:      %s\n", cache->short_description);
  //  printf("  long:       %s\n", cache->long_description);
  //  printf("  hint:       %s\n", cache->hint);  

  while(log) {
    gpx_display_log(log);
    log = log->next;
  }
}

void gpx_display_all(gpx_t *gpx) {
  while(gpx) {
    cache_t *cache = gpx->cache;

    printf("GPX name: %s\n", gpx->name);
    printf("GPX desc: %s\n", gpx->desc);
    printf("GPX date: %d.%d.%d\n", gpx->day, gpx->month, gpx->year);
    while(cache) {
      gpx_display_cache(cache);
      cache = cache->next;
    }
    gpx = gpx->next;
  }
}

/* parse gpx entry */
static void gpx_parse_gpx(gpx_t *gpx, xmlDocPtr doc, xmlNode * a_node) {
  xmlNode *cur_node = NULL;

  for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      /* some global information */
      if(!gpx->name && strcasecmp(cur_node->name, "name") == 0) {
	gpx->name = xmlNodeListGetString(doc, cur_node->children, 1);
      } else if(!gpx->desc && strcasecmp(cur_node->name, "desc") == 0) {
	gpx->desc = xmlNodeListGetString(doc, cur_node->children, 1);
      } else if((strcasecmp(cur_node->name, "date") == 0) ||
		(strcasecmp(cur_node->name, "time") == 0)) {
	char *str;
	if(str = xmlNodeListGetString(doc, cur_node->children, 1)) {
	  sscanf(str, "%d-%d-%d", &gpx->year, &gpx->month, &gpx->day);
	  xmlFree(str);
	}
      } else if(strcasecmp(cur_node->name, "wpt") == 0) {
	char *str;
	pos_t pos = { 0.0, 0.0 };

	if((str = xmlGetProp(cur_node, "lat"))) {
	  pos.lat = g_ascii_strtod(str, NULL);
	  xmlFree(str);
	}

	if((str = xmlGetProp(cur_node, "lon"))) {
	  pos.lon = g_ascii_strtod(str, NULL);
	  xmlFree(str);
	}

	gpx_parse_gpx_wpt(gpx, doc, cur_node->children, &pos);
      }
    }
  }
}

/* parse loc waypoint entry */
static void gpx_parse_loc_wpt(gpx_t *gpx, xmlDocPtr doc, xmlNode *a_node) {
  xmlNode *cur_node = NULL;
  cache_t **cache = &gpx->cache;
  
  /* skip to end of list and create a new entry */
  while(*cache) 
    cache = &(*cache)->next;

  *cache = malloc(sizeof(cache_t));
  memset(*cache, 0, sizeof(cache_t));

  /* set some defaults */
  (*cache)->type = CACHE_TYPE_UNKNOWN;
  (*cache)->container = CACHE_CONT_UNKNOWN;
  (*cache)->available = TRUE;
  
  for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      char *str;

      if(strcasecmp(cur_node->name, "name") == 0) {
	(*cache)->name = xmlNodeListGetString(doc, cur_node->children, 1);
	(*cache)->id = xmlGetProp(cur_node, "id");
      } else if(strcasecmp(cur_node->name, "link") == 0) {
	(*cache)->url =  
	  xmlNodeListGetString(doc, cur_node->children, 1);
      } else if(strcasecmp(cur_node->name, "type") == 0) {
	if(str = xmlNodeListGetString(doc, cur_node->children, 1)) {
	  (*cache)->type = str_search(cache_type_str, str, "cache type");
	  xmlFree(str);
	}
      } else if(strcasecmp(cur_node->name, "coord") == 0) {
	if((str = xmlGetProp(cur_node, "lat"))) {
	  (*cache)->pos.lat = g_ascii_strtod(str, NULL);
	  xmlFree(str);
	}
	if((str = xmlGetProp(cur_node, "lon"))) {
	  (*cache)->pos.lon = g_ascii_strtod(str, NULL);
	  xmlFree(str);
	}
      }
    }
  }
}

/* parse loc entry */
static void gpx_parse_loc(gpx_t *gpx, xmlDocPtr doc, xmlNode * a_node) {
  xmlNode *cur_node = NULL;

  for (cur_node = a_node; cur_node; cur_node = cur_node->next) 
    if (cur_node->type == XML_ELEMENT_NODE) 
      if(strcasecmp(cur_node->name, "waypoint") == 0) 
	gpx_parse_loc_wpt(gpx, doc, cur_node->children);
}

/* parse root element and search for "gpx" */
static gpx_t *gpx_parse_root(xmlDocPtr doc, xmlNode * a_node, 
			     char *filename, gpx_t *in) {
  gpx_t *gpx;
  xmlNode *cur_node = NULL;

  /* no gpx entry given, create a new one */
  if(!in) {
    /* allocate memory to hold gpx file description */
    gpx = malloc(sizeof(gpx_t));
    memset(gpx, 0, sizeof(gpx_t));
    gpx->filename = strdup(filename);
  } else
    gpx = in;

  for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      /* parse gpx file ... */
      if(strcasecmp(cur_node->name, "gpx") == 0) 
	gpx_parse_gpx(gpx, doc, cur_node->children);
      /* ... or loc file */
      else if(strcasecmp(cur_node->name, "loc") == 0) 
	gpx_parse_loc(gpx, doc, cur_node->children);
    }
  }

  /* check if a name has been set and use filename if not */
  if(!in && !gpx->name) {
    if(!gpx->desc) { 
      char *str = strrchr(filename, '/');
      if(str) gpx->name = strdup(str+1);
      else    gpx->name = strdup(filename);
    } else
      gpx->name = strdup(gpx->desc);
  }

  return gpx;
}

static gpx_t *gpx_parse_doc(xmlDocPtr doc, char *filename, gpx_t *gpx_in) {
  gpx_t *gpx;

  /* Get the root element node */
  xmlNode *root_element = xmlDocGetRootElement(doc);

  gpx = gpx_parse_root(doc, root_element, filename, gpx_in);  

  /*free the document */
  xmlFreeDoc(doc);

  /*
   * Free the global variables that may
   * have been allocated by the parser.
   */
  xmlCleanupParser();

  if(!gpx) return gpx_in;

  return gpx;
}

static gpx_t *gpx_parse_file(char *filename) {
  gpx_t *gpx = NULL;
  xmlDoc *doc = NULL;

  LIBXML_TEST_VERSION;

  /* parse the file and get the DOM */
  if ((doc = xmlReadFile(filename, NULL, 0)) == NULL) {
    xmlErrorPtr	errP = xmlGetLastError();
    errorf("While parsing \"%s\":\n\n%s", filename, errP->message);
    return NULL;
  }
  
  gpx = gpx_parse_doc(doc, filename, NULL);

  /* check of there's a waypoints file (*-wpts.gpx) for this */
  if(strrchr(filename, '.')) {
    char *dot = strrchr(filename, '.');
    char wpts_name[128];
    *dot = 0;
    snprintf(wpts_name, sizeof(wpts_name), "%s-wpts.gpx", filename);
    *dot = '.';

    if(g_file_test(wpts_name,  G_FILE_TEST_EXISTS)) {

      /* and do the same for this file again */
      if((doc = xmlReadFile(wpts_name, NULL, 0)) == NULL) {
	xmlErrorPtr errP = xmlGetLastError();
	errorf("While parsing \"%s\":\n\n%s", wpts_name, errP->message);
      } else 
	gpx = gpx_parse_doc(doc, filename, gpx);
    }
  }

  return gpx;
}

static gpx_t *decompress_file(unzFile file, char *name, char *filename,
			      gpx_t *gpx_in) {
  unz_file_info info;
  gpx_t *gpx;

  if((unzLocateFile(file, name, FALSE) != Z_OK) ||
     (unzGetCurrentFileInfo(file, &info, NULL,0, NULL,0, NULL,0) != Z_OK) ||
     (unzOpenCurrentFile(file) != UNZ_OK)) {

    /* do not complain if we are processing a waypoints file */
    if(!gpx_in)
      errorf("Unable to locate/get info/open\n%s\ninside\n%s",
	     name, filename);
    else
      printf("Unable to locate/get info/open %s inside %s\n",
	     name, filename);

    return gpx_in;
  }

  printf("file size is %d\n", info.uncompressed_size);

  char *buffer = malloc(info.uncompressed_size);
  if(!buffer) {
    errorf("Out of memory while uncompressing file");
    unzCloseCurrentFile(file);
    return gpx_in;
  }

  if(unzReadCurrentFile(file, buffer, info.uncompressed_size) < 0) {
    errorf("Read error on compressed file");
    free(buffer);
    unzCloseCurrentFile(file);
    return gpx_in;
  }

  /* fire up libxml */
  LIBXML_TEST_VERSION;
  xmlDoc *doc = NULL;

  /* parse the file and get the DOM */
  if ((doc = xmlReadMemory(buffer, info.uncompressed_size, 
			   NULL, NULL, 0)) == NULL) {
    xmlErrorPtr	errP = xmlGetLastError();
    errorf("While parsing \"%s\":\n\n%s", filename, errP->message);
    free(buffer);
    unzCloseCurrentFile(file);
    return gpx_in;
  }

  printf("loaded successfully, parse ...\n");

  gpx = gpx_parse_doc(doc, filename, gpx_in);
  
  free(buffer);
  unzCloseCurrentFile(file);
  return gpx;
}

static gpx_t *decompress_zip(char *filename) {
  char *gpx_name, *fbase;
  gpx_t *gpx = NULL;

  /* extract base name and allocate space for file names */
  fbase = strrchr(filename, '/');
  if(!fbase) fbase = filename;
  else       fbase++;  /* skip '/' */
  gpx_name = malloc(strlen(fbase)+strlen("-wpts")+1);

  unzFile file = unzOpen(filename);
  if(!file) {
    errorf("Error opening file %s for unzip", filename);
    free(gpx_name);
    return NULL;
  }

  printf("ZIP file successfully opened\n");

  /* try to open gpx file inside */
  strcpy(gpx_name, fbase);
  strcpy(gpx_name+strlen(gpx_name)-4, ".gpx");
  printf("gpx file name is %s\n", gpx_name);

  gpx = decompress_file(file, gpx_name, filename, NULL);

  /* try to open -wpts.gpx file inside */
  strcpy(gpx_name, fbase);
  strcpy(gpx_name+strlen(gpx_name)-4, "-wpts.gpx");
  printf("gpx wpts file name is %s\n", gpx_name);

  gpx = decompress_file(file, gpx_name, filename, gpx);

  unzClose(file);
  free(gpx_name);
  return gpx;
}

gpx_t *gpx_parse(char *filename) {
  if((strlen(filename) > 4) &&
     !strcasecmp(filename+strlen(filename)-4, ".zip")) {
    printf("trying to load a zip file!\n");

    return decompress_zip(filename);
  }

  return gpx_parse_file(filename);
}

/* scan entire directory */
gpx_t *gpx_parse_dir(char *dirname) {
  GnomeVFSResult result;
  GnomeVFSDirectoryHandle *handle;
  GnomeVFSFileInfo *finfo = gnome_vfs_file_info_new();;

  gpx_t *gpx = NULL;
  xmlDoc *doc = NULL;
  xmlNode *root_element = NULL;
  
  LIBXML_TEST_VERSION;

  result = gnome_vfs_directory_open(&handle, dirname, 
				    GNOME_VFS_FILE_INFO_DEFAULT);

  if(result != GNOME_VFS_OK) {
    errorf("Unable to open directory \"%s\":\n%s",
	   dirname,  gnome_vfs_result_to_string(result));
    
    return NULL;
  }

  while(GNOME_VFS_OK == gnome_vfs_directory_read_next(handle, finfo)) {
    if(finfo->type == GNOME_VFS_FILE_TYPE_REGULAR) {
      char *ext = finfo->name+strlen(finfo->name)-4;

      /* check if file ends with .gpx or .loc */
      if((strcasecmp(ext, ".gpx") == 0) || (strcasecmp(ext, ".loc") == 0)) {
	char *filename = malloc(strlen(dirname)+strlen(finfo->name)+2);

	strcpy(filename, dirname);
	if(strlastchr(filename) != '/')
	  strcat(filename, "/");
	strcat(filename, finfo->name);

	/* parse the file and get the DOM */
	doc = xmlReadFile(filename, NULL, 0);

	if (doc == NULL) {
	  xmlErrorPtr errP = xmlGetLastError();
	  errorf("While parsing \"%s\":\n\n%s", filename, errP->message);
	} else {
	  /*Get the root element node */
	  root_element = xmlDocGetRootElement(doc);

	  gpx = gpx_parse_root(doc, root_element, filename, gpx);
	  
	  xmlFreeDoc(doc);
	  xmlCleanupParser();
	}

	free(filename);
      }
    }
  }

  if(gpx) {
    /* replace file name with directory name */
    free(gpx->filename);
    gpx->filename = strdup(dirname);

    /* replace gpx name with directory name */
    free(gpx->name);

    /* retrieve pure dirname if possible */
    char *n = strrchr(dirname, '/');
    if(!n) n = dirname;
    else   n++;

    //    gpx->name = malloc(strlen("<DIR> ")+strlen(n)+1);
    //    strcpy(gpx->name, "<DIR> ");
    //    strcat(gpx->name, n);
    gpx->name = strdup(n);
  }

  gnome_vfs_file_info_unref(finfo);
  gnome_vfs_directory_close(handle);

  return gpx;
}

/* return number of caches in given gpx file */
int gpx_total_caches(gpx_t *gpx) {
  cache_t *cache = gpx->cache;
  int num = 0;
  
  while(cache) {
    num++;
    cache = cache->next;
  }

  return num;
}

int gpx_number_of_waypoints(wpt_t *wpt) {
  int num = 0;
  
  while(wpt) {
    num++;
    wpt = wpt->next;
  }

  return num;
}

int gpx_number_of_logs(log_t *log) {
  int num = 0;
  
  while(log) {
    num++;
    log = log->next;
  }

  return num;
}

int gpx_number_of_tbs(tb_t *tb) {
  int num = 0;
  
  while(tb) {
    num++;
    tb = tb->next;
  }

  return num;
}

/* http://mathforum.org/library/drmath/view/55417.html */
float gpx_pos_get_bearing(pos_t p1, pos_t p2) {
  /* convert to radians */
  p1.lat *= (M_PI/180.0); p1.lon *= (M_PI/180.0);
  p2.lat *= (M_PI/180.0); p2.lon *= (M_PI/180.0);

  return fmodf(360.0 + (180.0/M_PI) * 
	       (atan2( sin(p2.lon - p1.lon) * cos(p2.lat),
		       cos(p1.lat) * sin(p2.lat) - 
		       sin(p1.lat) * cos(p2.lat) * cos(p2.lon - p1.lon))),
	       360.0);
}

/* http://mathforum.org/library/drmath/view/51722.html */
float gpx_pos_get_distance(pos_t p1, pos_t p2, int miles) {
  /* convert to radians */
  p1.lat *= (M_PI/180.0); p1.lon *= (M_PI/180.0);
  p2.lat *= (M_PI/180.0); p2.lon *= (M_PI/180.0);

  float aob = acos(cos(p1.lat) * cos(p2.lat) * cos(p2.lon - p1.lon) +
		   sin(p1.lat) * sin(p2.lat));

  if(miles) 
    return(aob * 3959.0);   /* great circle radius in miles */

  return(aob * 6371.0);     /* great circle radius in kilometers */
}

void gpx_pos_get_distance_str(char *str, int len, 
			      pos_t p1, pos_t p2, int mil) {

  float dist = gpx_pos_get_distance(p1, p2, mil);
  distance_str(str, len, dist, mil);
}

/* http://library.gnome.org/devel/gtk/unstable/GtkRadioButton.html */
void gpx_sort(gpx_t *gpx, int by, pos_t *refpos) {
  cache_t **new;
  cache_t *cur = gpx->cache;
  int total = gpx_total_caches(gpx);
  float *dist_cache = malloc(total * sizeof(float));

  gpx->cache = NULL;  /* detach old chain */
  while(cur) {
    float cur_dist = gpx_pos_get_distance(*refpos, gpx_cache_pos(cur), 0);
    int cur_cnt = 0;

    new = &(gpx->cache);

    /* search for currect insertion point */
    while(*new && (dist_cache[cur_cnt] < cur_dist)) {
      new = &((*new)->next);
      cur_cnt++;
    }

    /* save distance for further comparisons */
    memmove(dist_cache+cur_cnt+1, dist_cache+cur_cnt, 
	    sizeof(float)*(total-cur_cnt-1));
    dist_cache[cur_cnt++] = cur_dist;
    
    cache_t *next = cur->next;

    /* insert into "new" chain */
    cur->next = *new;
    *new = cur;

    cur = next;
  }

  free(dist_cache);
}

gpx_t *gpx_cache2gpx(gpx_t *gpx, cache_t *search_cache) {
  while(gpx) {
    cache_t *cache = gpx->cache;
    while(cache) {
      if(cache == search_cache)
	return gpx;

      cache = cache->next;
    }
    gpx = gpx->next;
  }

  return NULL;
}

/* since the actual cache position may be overridden, we */
/* always access the position through this function */
pos_t gpx_cache_pos(cache_t *cache) {
  if(cache->notes && cache->notes->override) 
    return cache->notes->pos;

  return cache->pos;
}
