/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include "custom_size_renderer.h"

/* http://scentric.net/tutorial/sec-custom-cell-renderers.html */
/* https://stage.maemo.org/svn/maemo/projects/haf/trunk/gtk+/gtk/gtkcellrendererpixbuf.c */
/* https://stage.maemo.org/svn/maemo/projects/haf/trunk/gtk+/gtk/gtkcellrenderertext.c */

/* This is based mainly on GtkCellRendererProgress
 *  in GAIM, written and (c) 2002 by Sean Egan
 *  (Licensed under the GPL), which in turn is
 *  based on Gtk's GtkCellRenderer[Text|Toggle|Pixbuf]
 *  implementation by Jonathan Blandford */

/* Some boring function declarations: GObject type system stuff */

static void     custom_cell_renderer_size_init       (CustomCellRendererSize      *cellsize);
static void     custom_cell_renderer_size_class_init (CustomCellRendererSizeClass *klass);
static void     custom_cell_renderer_size_get_property  (GObject                    *object,
                                                             guint                       param_id,
                                                             GValue                     *value,
                                                             GParamSpec                 *pspec);
static void     custom_cell_renderer_size_set_property  (GObject                    *object,
                                                             guint                       param_id,
                                                             const GValue               *value,
                                                             GParamSpec                 *pspec);
static void     custom_cell_renderer_size_finalize (GObject *gobject);


/* These functions are the heart of our custom cell renderer: */

static void     custom_cell_renderer_size_get_size   (GtkCellRenderer            *cell,
                                                          GtkWidget                  *widget,
                                                          GdkRectangle               *cell_area,
                                                          gint                       *x_offset,
                                                          gint                       *y_offset,
                                                          gint                       *width,
                                                          gint                       *height);

static void     custom_cell_renderer_size_render     (GtkCellRenderer            *cell,
                                                          GdkWindow                  *window,
                                                          GtkWidget                  *widget,
                                                          GdkRectangle               *background_area,
                                                          GdkRectangle               *cell_area,
                                                          GdkRectangle               *expose_area,
                                                          guint                       flags);


enum { PROP_SIZE = 1, };

static const char *size_str[] = { 
  "Regular", "Small", "Micro", "Other", "Not chosen", "Large", "Virtual" 
};

static gpointer parent_class;

GType custom_cell_renderer_size_get_type (void) {
  static GType cell_size_type = 0;

  if (cell_size_type == 0)
  {
    static const GTypeInfo cell_size_info =
    {
      sizeof (CustomCellRendererSizeClass),
      NULL,                                                     /* base_init */
      NULL,                                                     /* base_finalize */
      (GClassInitFunc) custom_cell_renderer_size_class_init,
      NULL,                                                     /* class_finalize */
      NULL,                                                     /* class_data */
      sizeof (CustomCellRendererSize),
      0,                                                        /* n_preallocs */
      (GInstanceInitFunc) custom_cell_renderer_size_init,
    };

    /* Derive from GtkCellRenderer */
    cell_size_type = g_type_register_static (GTK_TYPE_CELL_RENDERER,
                                                 "CustomCellRendererSize",
                                                  &cell_size_info,
                                                  0);
  }
  return cell_size_type;
}


/***************************************************************************
 *
 *  custom_cell_renderer_size_init: set some default properties of the
 *                                      parent (GtkCellRenderer).
 *
 ***************************************************************************/

static void
custom_cell_renderer_size_init (CustomCellRendererSize *cellrenderersize)
{
  GTK_CELL_RENDERER(cellrenderersize)->mode = GTK_CELL_RENDERER_MODE_INERT;
  GTK_CELL_RENDERER(cellrenderersize)->xpad = 0;
  GTK_CELL_RENDERER(cellrenderersize)->ypad = 0;
}


/***************************************************************************
 *
 *  custom_cell_renderer_size_class_init:
 *
 *  set up our own get_property and set_property functions, and
 *  override the parent's functions that we need to implement.
 *  And make our new "size" property known to the type system.
 *  If you want cells that can be activated on their own (ie. not
 *  just the whole row selected) or cells that are editable, you
 *  will need to override 'activate' and 'start_editing' as well.
 *
 ***************************************************************************/

static void
custom_cell_renderer_size_class_init (CustomCellRendererSizeClass *klass) {
  GtkCellRendererClass *cell_class   = GTK_CELL_RENDERER_CLASS(klass);
  GObjectClass         *object_class = G_OBJECT_CLASS(klass);

  parent_class           = g_type_class_peek_parent (klass);
  object_class->finalize = custom_cell_renderer_size_finalize;

  /* Hook up functions to set and get our
   *   custom cell renderer properties */
  object_class->get_property = custom_cell_renderer_size_get_property;
  object_class->set_property = custom_cell_renderer_size_set_property;

  /* Override the two crucial functions that are the heart
   *   of a cell renderer in the parent class */
  cell_class->get_size = custom_cell_renderer_size_get_size;
  cell_class->render   = custom_cell_renderer_size_render;

  /* Install our very own properties */
  g_object_class_install_property (object_class, PROP_SIZE,
    g_param_spec_int ("size", "Size", "Container size", -1, CACHE_CONT_MAX, 0,
		      G_PARAM_READWRITE));
}


/***************************************************************************
 *
 *  custom_cell_renderer_size_finalize: free any resources here
 *
 ***************************************************************************/

static void custom_cell_renderer_size_finalize (GObject *object) {
  (* G_OBJECT_CLASS (parent_class)->finalize) (object);
}


/***************************************************************************
 *
 *  custom_cell_renderer_size_get_property: as it says
 *
 ***************************************************************************/

static void
custom_cell_renderer_size_get_property (GObject    *object,
                                            guint       param_id,
                                            GValue     *value,
                                            GParamSpec *psec) {
  CustomCellRendererSize  *cellsize = CUSTOM_CELL_RENDERER_SIZE(object);

  switch (param_id) {
    case PROP_SIZE:
      g_value_set_int(value, cellsize->size);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, psec);
      break;
  }
}


/***************************************************************************
 *
 *  custom_cell_renderer_size_set_property: as it says
 *
 ***************************************************************************/

static void
custom_cell_renderer_size_set_property (GObject      *object,
                                            guint         param_id,
                                            const GValue *value,
                                            GParamSpec   *pspec) {
  CustomCellRendererSize *cellsize = CUSTOM_CELL_RENDERER_SIZE (object);

  switch (param_id) {
    case PROP_SIZE:
      cellsize->size = g_value_get_int(value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, param_id, pspec);
      break;
  }
}

/***************************************************************************
 *
 *  custom_cell_renderer_size_new: return a new cell renderer instance
 *
 ***************************************************************************/

GtkCellRenderer *custom_cell_renderer_size_new (void) {
  return g_object_new(CUSTOM_TYPE_CELL_RENDERER_SIZE, NULL);
}

static PangoLayout *new_layout(GtkWidget *widget, const char *str) {
#ifndef USE_MAEMO
  return gtk_widget_create_pango_layout(widget, str);
#else
  char tmp[48];
  snprintf(tmp, sizeof(tmp), "<span size=\"xx-small\">%s</span>", str);
  PangoLayout *layout = gtk_widget_create_pango_layout(widget, NULL);
  pango_layout_set_markup(layout, tmp, strlen(tmp));
  return layout;
#endif
}

static void
custom_cell_renderer_size_get_size (GtkCellRenderer *cell,
	    GtkWidget *widget, GdkRectangle *cell_area, gint *x_offset, 
	    gint *y_offset, gint *width, gint *height) {
  GdkPixbuf *pixbuf = NULL;
  CustomCellRendererSize *cellsize = CUSTOM_CELL_RENDERER_SIZE (cell);

  if(cellsize->size < 0) return;
 
  pixbuf = icon_get(ICON_CACHE_SIZE, cellsize->size);
  if (!pixbuf) return;

  pixbuf = icon_get(ICON_CACHE_SIZE, cellsize->size);
  gint calc_width  = (gint) gdk_pixbuf_get_width(pixbuf);
  gint calc_height  = (gint) gdk_pixbuf_get_height(pixbuf);

  /* do the text */
  PangoLayout *layout = new_layout(widget, size_str[cellsize->size]);

  PangoRectangle rect;
  pango_layout_get_pixel_extents(layout, NULL, &rect);
  g_object_unref (layout);

  calc_height += rect.height;
  if(rect.width > calc_width) calc_width = rect.width;

  //  printf("req size = %d*%d\n", calc_width, calc_height);

  if (x_offset) *x_offset = 0;
  if (y_offset) *y_offset = 0;

  if (cell_area) {
    //    printf("cell size = %d*%d\n", cell_area->width, cell_area->height);

    if (x_offset) 
      *x_offset = (cell_area->width - calc_width)/2;

    if (y_offset)
      *y_offset = (cell_area->height - calc_height)/2;
  }

  if (width)  *width = calc_width;
  if (height) *height = calc_height;
}


/***************************************************************************
 *
 *  custom_cell_renderer_size_render: crucial - do the rendering.
 *
 ***************************************************************************/

static void
custom_cell_renderer_size_render (GtkCellRenderer *cell,
                                      GdkWindow       *window,
                                      GtkWidget       *widget,
                                      GdkRectangle    *background_area,
                                      GdkRectangle    *cell_area,
                                      GdkRectangle    *expose_area,
                                      guint            flags)
{
  CustomCellRendererSize *cellsize = CUSTOM_CELL_RENDERER_SIZE (cell);
  GdkPixbuf *pixbuf = NULL;
  GdkRectangle all_rect;
  GdkRectangle draw_rect;

  if(cellsize->size < 0) return;
 
  pixbuf = icon_get(ICON_CACHE_SIZE, cellsize->size);
  if (!pixbuf) return;

  gint pix_width  = gdk_pixbuf_get_width(pixbuf);
  gint pix_height  = gdk_pixbuf_get_height(pixbuf);

  
  custom_cell_renderer_size_get_size (cell, widget, cell_area,
				     &all_rect.x, &all_rect.y,
				     &all_rect.width, &all_rect.height);

  all_rect.x += cell_area->x;
  all_rect.y += cell_area->y;
  
  if (!gdk_rectangle_intersect (cell_area, &all_rect, &draw_rect) ||
      !gdk_rectangle_intersect (expose_area, &draw_rect, &draw_rect))
    return;

  /* do the text */
  PangoLayout *layout = new_layout(widget, size_str[cellsize->size]);
  PangoRectangle rect;
  pango_layout_get_pixel_extents(layout, NULL, &rect);
  //  printf("cell width = %d, text width = %d\n", all_rect.width, rect.width);

  int yoff = (all_rect.height - (rect.height + pix_height))/2;

  gtk_paint_layout(widget->style, window, GTK_STATE_NORMAL,
		    TRUE, expose_area, widget, "cellrenderersize",
                    all_rect.x + (all_rect.width - rect.width)/2,
                    all_rect.y + yoff,
                    layout);

  g_object_unref (layout);

  /* draw the bitmap */
  cairo_t *cr = gdk_cairo_create(window);
  gdk_cairo_set_source_pixbuf(cr, pixbuf, 
		      all_rect.x + (all_rect.width - pix_width)/2, 
		      all_rect.y + rect.height + yoff);
  gdk_cairo_rectangle(cr, &draw_rect);
  cairo_fill(cr);
  cairo_destroy(cr);
}
