/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007, 2008 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "ui.h"
#include "uistat.h"
#include "uireport.h"

/* Statistics window related*/
#define X1NOFULL_FP .15
#define Y1NOFULL_FP .1
#define X2NOFULL_FP .85
#define Y2NOFULL_FP .75
#define XNOFULL_NEXT .95
#define YNOFULL_NEXT .95
#define XNOFULL_PREV .05
#define YNOFULL_PREV .95

enum analysis {CURRENT, CONSUM, FILL, TRIP, PRICEPERLITRE};

int currentyear;

extern unit_t curunit;
extern enum papertype reportpaper;

/* Local function prototypes */
void plot_set_titles( AppUIData *pui, enum analysis method);
void cb_stat( GtkAction * action, AppUIData *pui );
void cb_trip( GtkAction * action, AppUIData *pui );
void cb_priceperlitre( GtkAction * action, AppUIData *pui );

/**
 * \fn gboolean delete_statwin( GtkWidget *widget, GdkEvent  *event, gpointer   data )
 * \brief Statistics window delete_event callback
 * \return FALSE
 */
static gboolean delete_statwin( GtkWidget *widget,
                                GdkEvent  *event,
                                gpointer   data )
{
  extern AppUIData ui;

  gtk_widget_destroy(GTK_WIDGET(ui.app->statwindow));
  return FALSE;
}

void destroy_statwin( GtkWidget *widget,
		      gpointer   data )
{
  AppUIData *a;

  a=(AppUIData *)data;

  gtk_widget_destroy(GTK_WIDGET(a->app->statwindow));
  a->app->statwindow=NULL;
}

/**
 * \fn void plot_monthly_data(AppUIData *pui, enum analysis method)
 * \brief Plots/replots the selected monthly data
 * \param *pui Pointer to UI data
 * \param method which data is to be plotted
 *
 * Plots the data selected by the method argument for year currentyear
 * (a global variable).
 *
 */
void
plot_monthly_data(AppUIData *pui, enum analysis method)
{
  static GtkPlotData *dataset;

  char yearbuf[YEARBUFLEN];
  GdkColor color;
  static double px2[12] = {0.0};
  static double fill[12] = {0.0};
  static double trip[12] = {0.0};
  static double consum[12] = {0.0};
  static double priceperlitre[12] = {0.0};
  double *yval;
  double ymax = 0.0;
  int nummonths;
  int i;

  static enum analysis currentmethod=0;

  if (method != CURRENT) {
    currentmethod = method;
    plot_set_titles(pui, method);
  }

  gdk_color_parse("blue", &color);
  gdk_color_alloc(gdk_colormap_get_system(), &color); 

  if (dataset!=NULL)
    gtk_plot_remove_data(GTK_PLOT(pui->active_plot), dataset);
  dataset = GTK_PLOT_DATA(gtk_plot_bar_new(GTK_ORIENTATION_VERTICAL));
  gtk_plot_add_data(GTK_PLOT(pui->active_plot), dataset);
  gtk_widget_show(GTK_WIDGET(dataset));

  for (i=0;i<12;i++) {
    fill[i]=0.0;
    trip[i]=0.0;
    consum[i]=0.0;
    priceperlitre[i]=0.0;
    px2[i]=i+1;
  }

  if (currentyear==0)
    currentyear=getyear();
  nummonths=db_get_monthly_data(currentyear,px2,fill,trip,consum,priceperlitre);
  for (i=0;i<12;i++) {
    switch (currentmethod) {
    case CONSUM:
      yval=consum;
      if (curunit.consumeunit != SI) {
	if (consum[i] > 0.0001)
	  consum[i]=SIconsumption2user(consum[i]);
      }
      break;
    case FILL:
      yval=fill;
      if (curunit.volumeunit != SI) {
	if (fill[i] > 0.0001)
	  fill[i]=SIvolume2user(fill[i]);
      }
      break;
    case TRIP:
      yval=trip;
      if (curunit.lengthunit != SI) {
	if (trip[i] > 0.0001)
	  trip[i]=SIlength2user(trip[i]);
      }
      break;
    case PRICEPERLITRE:
      yval=priceperlitre;
      if (curunit.consumeunit != SI) {
	if (priceperlitre[i] > 0.0001)
	  priceperlitre[i]=SIppl2user(priceperlitre[i]);
      }
      break;
    default:
      yval=consum;
    }
    if (yval[i] > ymax) ymax=yval[i];
  }
  gtk_plot_data_set_points(dataset, px2, yval, NULL, NULL, nummonths);
  if (ymax>0.0001)
    gtk_plot_set_yrange(GTK_PLOT(pui->active_plot), 0., 1.1*ymax);
  else {
    switch (currentmethod) {
    case CONSUM:
      if (curunit.consumeunit==SI)
	ymax=13.0;
      else
	ymax=30.0;
      break;
    case FILL:
      if (curunit.volumeunit == SI)
	ymax=100.0;
      else
	ymax=30.0;
      break;
    case TRIP:
      ymax=2000;
      break;
    case PRICEPERLITRE:
      if (curunit.consumeunit == SI)
	ymax=1.5;
      else
	ymax=5.0;
      break;
    default:
      ymax=1000;
    }
    gtk_plot_set_yrange(GTK_PLOT(pui->active_plot), 0., ymax);

  }

  gtk_plot_data_set_symbol(dataset,
			   GTK_PLOT_SYMBOL_NONE,
			   GTK_PLOT_SYMBOL_OPAQUE,
			   10, 2, &color, &color);

  gtk_plot_data_set_line_attributes(dataset,
				    GTK_PLOT_LINE_NONE,
				    0, 0, 1, &color);
  g_snprintf(yearbuf,YEARBUFLEN,"%s %d",_("Year"),currentyear);
  gtk_plot_data_set_legend(dataset, yearbuf);

  gtk_plot_bar_set_width(GTK_PLOT_BAR(dataset), 0.25);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(pui->canvas));
  gtk_widget_queue_draw(GTK_WIDGET(pui->canvas));

  gtk_plot_axis_set_tick_labels(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_BOTTOM), pui->months);
  gtk_plot_axis_use_custom_tick_labels(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_BOTTOM), TRUE);
}

void browse_prev_year(AppUIData *pui)
{
  currentyear-=1;
  plot_monthly_data(pui, CURRENT);
}

void browse_next_year(AppUIData *pui)
{
  currentyear+=1;
  plot_monthly_data(pui, CURRENT);
}

void browse_this_year(AppUIData *pui)
{
  currentyear=getyear();
  plot_monthly_data(pui, CURRENT);
}

void callback_next_year( GtkAction * action, gpointer data )
{
  browse_next_year(data);
}

void callback_prev_year( GtkAction * action, gpointer data )
{
  browse_prev_year(data);
}

void callback_this_year( GtkAction * action, gpointer data )
{
  browse_this_year(data);
}

void stat_window_fullscreen(gpointer data)
{
  AppUIData *a;

  a=(AppUIData *)data;
  a->statfullscreen = !a->statfullscreen;
  if (a->statfullscreen) {
/*     if (reportpaper == A4PAPER) */
      gtk_plot_canvas_set_magnification(GTK_PLOT_CANVAS(a->canvas), 800.0/GTK_PLOT_A4_W);
/*     else */
/*       gtk_plot_canvas_set_magnification(GTK_PLOT_CANVAS(a->canvas), 800.0/GTK_PLOT_LETTER_W); */
    gtk_plot_canvas_paint(GTK_PLOT_CANVAS(a->canvas));
    gtk_window_fullscreen(GTK_WINDOW(a->app->statwindow));
  } else {
/*     if (reportpaper == A4PAPER) */
      gtk_plot_canvas_set_magnification(GTK_PLOT_CANVAS(a->canvas), 720.0/GTK_PLOT_A4_W);
/*     else */
/*       gtk_plot_canvas_set_magnification(GTK_PLOT_CANVAS(a->canvas), 720.0/GTK_PLOT_LETTER_W); */
    gtk_plot_canvas_paint(GTK_PLOT_CANVAS(a->canvas));
    gtk_window_unfullscreen(GTK_WINDOW(a->app->statwindow));
  }
}

void cb_stat_fullscreen( GtkAction * action, gpointer data )
{
  stat_window_fullscreen(data);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(((AppUIData *)(data))->stat_menu_item_fullscreen),
				 ((AppUIData *)(data))->statfullscreen);
}

#if PLAINGTK == 0
gboolean stat_key_press_cb(GtkWidget * widget, GdkEventKey * event,
			   AppUIData *pui)
{
  switch (event->keyval) {

    case GDK_F6:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->stat_menu_item_fullscreen),
				     !pui->statfullscreen);
      return TRUE;
    case GDK_Left:
      browse_prev_year(pui);
      return TRUE;
      break;
    case GDK_Right:
      browse_next_year(pui);
      return TRUE;
      break;
    case GDK_Escape:
      browse_this_year(pui);
      return TRUE;
      break;
    }

    return FALSE;
}
#endif

GtkWidget *
new_layer(AppUIData *pui)
{
  static gint nlayers = 0;

  nlayers++;

  pui->plots = (GtkWidget **)g_realloc(pui->plots, nlayers * sizeof(GtkWidget *));

  pui->plots[nlayers-1] = gtk_plot_new_with_size(NULL, .5, YSCALE(.25));
  gtk_widget_show(pui->plots[nlayers-1]);

  return pui->plots[nlayers-1];
}

void callback_save_stat( GtkAction * action, AppUIData *pui )
{
/*   callback_save_report(action,data); */

  gint pagesize;
  gchar *filename = NULL;
  GdkColor color;
  GdkColor oldcolor;
  GtkWidget *canvas;

  canvas = pui->canvas;
  filename = interface_file_chooser ("Save postscript file",
				     GTK_FILE_CHOOSER_ACTION_SAVE,
				     "ps",
				     HELP_ID_STATISTICS);

  if (filename == NULL) {
    filename = "NULL";
  }
  else {
/*     report_convert_to_iso8859_15(GTK_WIDGET(canvas)); */
/*     if (GTK_PLOT_CANVAS(canvas)->width == GTK_PLOT_A4_W) */
    if (reportpaper == A4PAPER)
      pagesize=GTK_PLOT_A4;
    else
      pagesize=GTK_PLOT_LETTER;
    oldcolor=GTK_PLOT_CANVAS(canvas)->background;
    gdk_color_parse("white", &color);
    gdk_colormap_alloc_color(gtk_widget_get_colormap(canvas), &color, FALSE, TRUE);
    gtk_plot_canvas_set_background(GTK_PLOT_CANVAS(canvas), &color);
    gtk_plot_canvas_export_ps(GTK_PLOT_CANVAS(canvas), filename, GTK_PLOT_PORTRAIT, FALSE, pagesize);
    gtk_plot_canvas_set_background(GTK_PLOT_CANVAS(canvas), &oldcolor);
    ui_show_banner(_("Figure saved."));
  }
}

#if PLAINGTK == 1
static void create_stat_toolbar(GtkWidget *window, GtkWidget *vbox, AppUIData *pui)
#else
static void create_stat_toolbar(HildonWindow *window, AppUIData *pui)
#endif
{
  GtkWidget *toolbar;
  GtkToolItem *tb_next;
  GtkToolItem *tb_prev;
  GtkToolItem *tb_save;
  GtkToolItem *tb_close;

  /* Create toolbar */
  toolbar = gtk_toolbar_new();

  /* Create toolbar button items */
  tb_prev = gtk_tool_button_new_from_stock(GTK_STOCK_GO_BACK);
  tb_next = gtk_tool_button_new_from_stock(GTK_STOCK_GO_FORWARD);
  tb_save = gtk_tool_button_new_from_stock(GTK_STOCK_SAVE);
  tb_close = gtk_tool_button_new_from_stock(GTK_STOCK_CLOSE);

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_prev),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_next),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_save),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_close),TRUE);

  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_prev, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_next, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_save, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_close, -1);

  g_signal_connect(G_OBJECT(tb_save), "clicked",
		   G_CALLBACK(callback_save_stat), pui);
  g_signal_connect(G_OBJECT(tb_next), "clicked",
		   G_CALLBACK(callback_next_year), pui);
  g_signal_connect(G_OBJECT(tb_prev), "clicked",
		   G_CALLBACK(callback_prev_year), pui);
  g_signal_connect_swapped(G_OBJECT(tb_close), "clicked",
			   G_CALLBACK(delete_statwin), G_OBJECT(window));

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(window, GTK_TOOLBAR(toolbar));
#endif

}
#if PLAINGTK == 1
void create_statwin_menu(GtkWidget *window, GtkWidget *root, AppUIData *pui)
#else
  void create_statwin_menu(HildonWindow *window, AppUIData *pui)
#endif
{
  GtkWidget *main_menu;
  GtkWidget *menu_view;
  GtkWidget *menu_analysis;
  GtkWidget *item_analysis;
  GtkWidget *item_this;
  GtkWidget *item_prev;
  GtkWidget *item_next;
  GtkWidget *item_save;
  GSList *radio_menu_group = NULL;
  GtkWidget *item_view;
  GtkWidget *item_help;
  GtkWidget *item_exit;
#if PLAINGTK == 1
  GtkWidget *menubar;
  GtkWidget *item_file;
#endif
 
  main_menu = gtk_menu_new();
  menu_view = gtk_menu_new();
  menu_analysis = gtk_menu_new();

  /* Create main menu items */
  item_analysis = gtk_menu_item_new_with_label(_("Statistics"));
  item_save = gtk_menu_item_new_with_label(_("Save figure..."));
  item_view = gtk_menu_item_new_with_label(_("View"));
#if PLAINGTK == 0
  item_help = gtk_menu_item_new_with_label(_("Help"));
#endif
  item_exit = gtk_menu_item_new_with_label(_("Close window"));

  /* Create statistics menu items */
  pui->sm_item_monthly = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Monthly consumption"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->sm_item_monthly));
  pui->sm_item_monthlytrip = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Monthly driving distance"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->sm_item_monthlytrip));
  pui->sm_item_monthlypriceperlitre = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Monthly average fuel price"));

  /* Create view menu items */
  pui->stat_menu_item_fullscreen = gtk_check_menu_item_new_with_label(_("Full screen"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->stat_menu_item_fullscreen),
				 pui->statfullscreen);
  item_this = gtk_menu_item_new_with_label(_("Current year"));
  item_prev = gtk_menu_item_new_with_label(_("Previous year"));
  item_next = gtk_menu_item_new_with_label(_("Next year"));

  /* Add menu items to right menus */
  /* Main menu */
  gtk_menu_append(main_menu, item_analysis);
  gtk_menu_append(main_menu, item_save);
  gtk_menu_append(main_menu, item_view);
  gtk_menu_append(main_menu, gtk_separator_menu_item_new());
#if PLAINGTK == 0
  gtk_menu_append(main_menu, item_help);
#endif
  gtk_menu_append(main_menu, item_exit);

  /* View menu */
  gtk_menu_append(menu_view, pui->stat_menu_item_fullscreen);
  gtk_menu_append(menu_view, item_this);
  gtk_menu_append(menu_view, item_prev);
  gtk_menu_append(menu_view, item_next);

  /* Statistics menu */
  gtk_menu_append(menu_analysis, pui->sm_item_monthly);
  gtk_menu_append(menu_analysis, pui->sm_item_monthlytrip);
  gtk_menu_append(menu_analysis, pui->sm_item_monthlypriceperlitre);

#if PLAINGTK == 1
  menubar = gtk_menu_bar_new();
  gtk_box_pack_start (GTK_BOX (root), menubar, FALSE, FALSE, 2);
  gtk_widget_show (menubar);
  item_file = gtk_menu_item_new_with_label ("fuelpad");
  gtk_widget_show (item_file);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (item_file), main_menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menubar), item_file);
#else
  hildon_window_set_menu(HILDON_WINDOW(window), GTK_MENU(main_menu));
#endif

  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item_view), menu_view);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item_analysis), menu_analysis);

  /* Attach the callback functions to the activate signal */
  g_signal_connect(G_OBJECT(pui->sm_item_monthly), "toggled",
			   G_CALLBACK(cb_stat), pui);
  g_signal_connect(G_OBJECT(pui->sm_item_monthlytrip), "toggled",
			   G_CALLBACK(cb_trip), pui);
  g_signal_connect(G_OBJECT(pui->sm_item_monthlypriceperlitre), "toggled",
			   G_CALLBACK(cb_priceperlitre), pui);
  g_signal_connect(G_OBJECT(item_save), "activate",
		   G_CALLBACK(callback_save_stat), pui);
#if PLAINGTK == 0
  g_signal_connect(G_OBJECT(item_help), "activate",
		   G_CALLBACK(help_activated), HELP_ID_STATISTICS);
#endif
  g_signal_connect_swapped(G_OBJECT(item_exit), "activate",
			   G_CALLBACK(delete_statwin), G_OBJECT(window));

  g_signal_connect(G_OBJECT(pui->stat_menu_item_fullscreen), "toggled",
		   G_CALLBACK(cb_stat_fullscreen), pui);
  g_signal_connect(G_OBJECT(item_this), "activate",
		   G_CALLBACK(callback_this_year), pui);
  g_signal_connect(G_OBJECT(item_prev), "activate",
		   G_CALLBACK(callback_prev_year), pui);
  g_signal_connect(G_OBJECT(item_next), "activate",
		   G_CALLBACK(callback_next_year), pui);

  /* Make all menu widgets visible */
  gtk_widget_show_all(GTK_WIDGET(main_menu));
}

void plot_set_titles( AppUIData *pui, enum analysis method)
{
  gtk_plot_canvas_remove_child(GTK_PLOT_CANVAS(pui->canvas), pui->maintitle);

  switch (method) {
  case CONSUM:
    if (curunit.consumeunit==SI) {
      gtk_plot_axis_set_major_ticks(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), 1.0);
      gtk_plot_axis_set_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), _("Fuel consumption [l/100 km]"));
    }
    else {
      gtk_plot_axis_set_major_ticks(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), 2.0);
      gtk_plot_axis_set_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), _("Miles per gallon"));
    }
    pui->maintitle = gtk_plot_canvas_text_new("Helvetica-Bold", 16, 0, NULL, NULL, TRUE,
				     GTK_JUSTIFY_CENTER,
				     _("Average monthly fuel consumption"));
    break;
  case FILL:
    if (curunit.volumeunit == SI) {
      gtk_plot_axis_set_major_ticks(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), 10);
      gtk_plot_axis_set_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), _("[litres]"));
    }
    else {
      gtk_plot_axis_set_major_ticks(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), 2);
      gtk_plot_axis_set_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), _("[gallons]"));
    }
    pui->maintitle = gtk_plot_canvas_text_new("Helvetica-Bold", 16, 0, NULL, NULL, TRUE,
				     GTK_JUSTIFY_CENTER,
				     _("Monthly total fuel consumed"));
    break;
  case TRIP:
    if (curunit.lengthunit == SI) {
      gtk_plot_axis_set_major_ticks(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), 200);
      gtk_plot_axis_set_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), _("[km]"));
    }
    else {
      gtk_plot_axis_set_major_ticks(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), 200);
      gtk_plot_axis_set_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), _("[miles]"));
    }
    pui->maintitle = gtk_plot_canvas_text_new("Helvetica-Bold", 16, 0, NULL, NULL, TRUE,
				     GTK_JUSTIFY_CENTER,
				       _("Monthly driving distance"));
    break;
  case PRICEPERLITRE:
    if (curunit.consumeunit == SI) {
      gtk_plot_axis_set_major_ticks(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), 0.1);
      gtk_plot_axis_set_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), curunit.currency->str);
    }
    else {
      gtk_plot_axis_set_major_ticks(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), 0.1);
      gtk_plot_axis_set_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), curunit.currency->str);
    }
    pui->maintitle = gtk_plot_canvas_text_new("Helvetica-Bold", 16, 0, NULL, NULL, TRUE,
				     GTK_JUSTIFY_CENTER,
				       _("Average fuel price"));
    break;
  }
  gtk_plot_canvas_put_child(GTK_PLOT_CANVAS(pui->canvas), pui->maintitle, .50, YSCALE(.050), .0, .0);

}

void create_statwin( AppUIData *pui, enum analysis method)
{
  GtkPlotCanvasChild *fuelplot;
  GtkWidget *vbox1;
  GdkColor color;
  gint page_width, page_height;
  gfloat scale;

  static gchar *month_labels[] = {"\0","Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

  if (pui->app->statwindow != NULL) {
    gtk_window_present(GTK_WINDOW(pui->app->statwindow));
    return;
  }

/*   if (reportpaper == A4PAPER) { */
    scale = 720.0/GTK_PLOT_A4_W;
    page_width = GTK_PLOT_A4_W;
    page_height = GTK_PLOT_A4_H;
/*   } */
/*   else { */
/*     scale = 720.0/GTK_PLOT_LETTER_W; */
/*     page_width = GTK_PLOT_LETTER_W; */
/*     page_height = GTK_PLOT_LETTER_H; */
/*   } */

#if PLAINGTK == 1
  pui->app->statwindow=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(pui->app->statwindow), "Plot");
  gtk_widget_set_usize(pui->app->statwindow,800,400);
  gtk_container_border_width(GTK_CONTAINER(pui->app->statwindow),0);
#else
  pui->app->statwindow = HILDON_WINDOW(hildon_window_new());
  hildon_program_add_window(pui->app->program, pui->app->statwindow);

  /* Callback for hardware keys */
  g_signal_connect(G_OBJECT(pui->app->statwindow), "key_press_event", 
		   G_CALLBACK(stat_key_press_cb), pui);
#endif

  g_signal_connect(G_OBJECT(pui->app->statwindow), "destroy",
		   G_CALLBACK(destroy_statwin), pui);

  vbox1=gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(pui->app->statwindow),vbox1);
  gtk_widget_show(vbox1);

  pui->statscrollwin=gtk_scrolled_window_new(NULL, NULL);
  gtk_container_border_width(GTK_CONTAINER(pui->statscrollwin),0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(pui->statscrollwin),
				  GTK_POLICY_AUTOMATIC,GTK_POLICY_NEVER);
   gtk_box_pack_start(GTK_BOX(vbox1),pui->statscrollwin, TRUE, TRUE,0);
   gtk_widget_show(pui->statscrollwin);

  pui->canvas = gtk_plot_canvas_new(page_width, page_height, scale);
  gtk_widget_set_usize(pui->canvas,page_width,page_height);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pui->statscrollwin), pui->canvas);

  gdk_color_parse("light blue", &color);
/*   gdk_color_alloc(gtk_widget_get_colormap(pui->canvas), &color); */
    gdk_colormap_alloc_color(gtk_widget_get_colormap(pui->canvas), &color, FALSE, TRUE);
  gtk_plot_canvas_set_background(GTK_PLOT_CANVAS(pui->canvas), &color);

  gtk_widget_show(pui->canvas);

#if PLAINGTK == 1
  create_statwin_menu(pui->app->statwindow, vbox1, pui);
#else
  create_statwin_menu(pui->app->statwindow, pui);
#endif

#if PLAINGTK == 1
  create_stat_toolbar(pui->app->statwindow, vbox1, pui);
#else
  create_stat_toolbar(pui->app->statwindow, pui);
#endif

  pui->active_plot = new_layer(pui);
  gdk_color_parse("light yellow", &color);
  gdk_color_alloc(gtk_widget_get_colormap(pui->active_plot), &color);
  gtk_plot_set_background(GTK_PLOT(pui->active_plot), &color);

  gdk_color_parse("light blue", &color);
  gdk_color_alloc(gtk_widget_get_colormap(pui->canvas), &color);
  gtk_plot_legends_set_attributes(GTK_PLOT(pui->active_plot),
				  NULL, 0,
				  NULL,
				  &color);

  gtk_plot_set_xrange(GTK_PLOT(pui->active_plot), 0. , 13.0);

  gtk_plot_axis_set_labels_numbers(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_LEFT), GTK_PLOT_LABEL_FLOAT, 1);

  /* (*plot, vmajor, vminor, hmajor, hminor) */
  if (curunit.consumeunit==SI)
    gtk_plot_grids_set_visible(GTK_PLOT(pui->active_plot), TRUE, FALSE, TRUE, TRUE);
  else
    gtk_plot_grids_set_visible(GTK_PLOT(pui->active_plot), TRUE, FALSE, TRUE, FALSE);

  gtk_plot_axis_show_labels(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_TOP), FALSE);
  gtk_plot_axis_set_visible(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_TOP), FALSE);
  gtk_plot_axis_hide_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_TOP));

  gtk_plot_axis_show_labels(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_RIGHT), FALSE);
  gtk_plot_axis_set_visible(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_RIGHT), FALSE);
  gtk_plot_axis_hide_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_RIGHT));

  /* (GtkPlot *plot, GtkPlotBorderStyle border, gint shadow_width);*/
  gtk_plot_set_legends_border(GTK_PLOT(pui->active_plot), 2, 0);
  gtk_plot_legends_move(GTK_PLOT(pui->active_plot), .9, YSCALE(.05));
  /*  gtk_plot_hide_legends(GTK_PLOT(pui->active_plot)); */

  gtk_plot_axis_set_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_BOTTOM), _("Month"));
  /* It seems that the coordinates of bottom title do not work very
     well with scaling */
  gtk_plot_axis_move_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_BOTTOM), 0, .4, YSCALE(0.6));
  gtk_plot_axis_justify_title(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_BOTTOM), GTK_JUSTIFY_CENTER);

  gtk_widget_show(pui->active_plot);

  fuelplot = gtk_plot_canvas_plot_new(GTK_PLOT(pui->active_plot));
  gtk_plot_canvas_put_child(GTK_PLOT_CANVAS(pui->canvas), fuelplot, X1NOFULL_FP, YSCALE(Y1NOFULL_FP), X2NOFULL_FP, YSCALE(Y2NOFULL_FP));
  gtk_widget_show(pui->active_plot);

  plot_set_titles(pui, method);
  gtk_plot_axis_set_major_ticks(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_BOTTOM), 1.0);
  pui->months = GTK_PLOT_ARRAY(gtk_plot_array_new(NULL, month_labels, 13, GTK_TYPE_STRING, FALSE));
  gtk_plot_axis_set_tick_labels(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_BOTTOM), pui->months);
  gtk_plot_axis_use_custom_tick_labels(gtk_plot_get_axis(GTK_PLOT(pui->active_plot), GTK_PLOT_AXIS_BOTTOM), TRUE);

  plot_monthly_data(pui, method);

  gtk_widget_show_all(GTK_WIDGET(pui->app->statwindow));

  if (pui->mainfullscreen) 
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->stat_menu_item_fullscreen),
				   !pui->statfullscreen);
}

void callback_stat( GtkAction * action, AppUIData *pui )
{
  create_statwin(pui, CONSUM);
}

void callback_stat_trip( GtkAction * action, AppUIData *pui )
{
  create_statwin(pui, TRIP);
}

void callback_stat_fill( GtkAction * action, AppUIData *pui )
{
  create_statwin(pui, FILL);
}

void callback_stat_priceperlitre( GtkAction * action, AppUIData *pui )
{
  create_statwin(pui, PRICEPERLITRE);
}

void cb_stat( GtkAction * action, AppUIData *pui )
{
  plot_monthly_data(pui, CONSUM);
}

void cb_trip( GtkAction * action, AppUIData *pui )
{
  plot_monthly_data(pui, TRIP);
}

void cb_priceperlitre( GtkAction * action, AppUIData *pui )
{
  plot_monthly_data(pui, PRICEPERLITRE);
}
