/*
 *  Copyright (c) 2002-2007 Jiri Benc <jbenc@upir.cz>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * readdata.c
 *
 * Funkce na cteni dat (z baliku nebo z disku).
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "common.h"
#include "params.h"
#include "level.h"
#include "readdata.h"
#include "pack.h"

#ifdef HAVE_ZLIB
#include <zlib.h>
#endif


#ifdef DATA_DIR
#define PACK_PREFIX DATA_DIR
#else
#define PACK_PREFIX ""
#endif

static char fpackname[] = PACK_PREFIX "fred.dat";
static FILE *fpack;

/* cache hlavicek */
static int cnthead;
static pack_head *headers;

static char langprefix[4];

void reinit_pack(void)
{
	switch (param_lang) {
	case LANG_EN:
		strcpy(langprefix, "en_");
		break;
	case LANG_CZ:
		strcpy(langprefix, "cz_");
		break;
	}
}

/* Inicializace baliku. */
void init_pack()
{
  char *name;
  pack_head h;

#ifdef DATA_DIR
  name = fpackname;
#else
  name = emalloc(strlen(exedir) + sizeof(fpackname) + 1);
  strcpy(name, exedir);
  strcat(name, fpackname);
#endif
  fpack = fopen(name, "rb");
  if (!fpack) error(ls(STR_READDATA_ERR_PACK_OPEN), name);
#ifndef win32
  printf(ls(STR_READDATA_USING_PACK), name);
#endif
#ifndef DATA_DIR
  free(name);
#endif
  if (fread(&h, sizeof(h), 1, fpack) != 1)
    error(ls(STR_READDATA_ERR_PACK_READ_HDR));
  if (strcmp(h.name, PACK_ID))
    error(ls(STR_READDATA_ERR_PACK_INVAL));
  cnthead = h.len;
  headers = emalloc(sizeof(pack_head) * cnthead);
  if (fread(headers, sizeof(h), cnthead, fpack) != cnthead)
    error(ls(STR_READDATA_ERR_PACK_READ_HDRS));

  reinit_pack();
}


/* "Otevreni" souboru v baliku (tj. zjisteni informaci o nem). */
void pack_fopen(pack_file *f, char *name)
{
  int i1;
  pack_head *h;
  char intname[16];

  strcpy(intname, langprefix);
  strncat(intname, name, 12);
  for (i1 = 0, h = headers; i1 < cnthead; i1++, h++) {
    if (!strcmp(name, h->name) || !strcmp(intname, h->name)) {
      f->start = h->start;
      f->len = h->len;
      f->compressed = h->compressed;
      return;
    }
  }
  error(ls(STR_READDATA_ERR_CONT_MISSING), name);
}


/* Nacteni souboru v baliku (soubor musel byt predtim "otevren" funkci
 * pack_fopen()). */
void pack_fread(pack_file *f, void *buf)
{
#ifdef HAVE_ZLIB
  z_stream z;
  char *inbuf;
#endif

  if (f->compressed) {
    /* soubor je zkomprimovany */
#ifdef HAVE_ZLIB
    /* alokace a nacteni bufferu */
    inbuf = emalloc(f->compressed);
    fseek(fpack, f->start, SEEK_SET);
    if (fread(inbuf, 1, f->compressed, fpack) != f->compressed)
      error(ls(STR_READDATA_ERR_CONT_READ));
    /* inicializace dekomprese */
    z.zalloc = Z_NULL;
    z.zfree = Z_NULL;
    z.opaque = Z_NULL;
    z.next_in = inbuf;
    z.avail_in = f->compressed;
    if (inflateInit(&z) != Z_OK)
      error(ls(STR_READDATA_ERR_CONT_UNPACK), z.msg);
    /* dekomprese */
    z.next_out = buf;
    z.avail_out = f->len;
    if (inflate(&z, Z_FINISH) != Z_STREAM_END)
      error(ls(STR_READDATA_ERR_CONT_UNPACK), z.msg);
    /* uklid po dekompresi */
    inflateEnd(&z);
    free(inbuf);
#else
    /* pokud ho neumime rozbalit, je to zavazna chyba */
    error(ls(STR_READDATA_ERR_CONT_PACKED));
#endif
  }
  else {
    /* soubor je nezkomprimovany */
    fseek(fpack, f->start, SEEK_SET);
    if (fread(buf, 1, f->len, fpack) != f->len)
      error(ls(STR_READDATA_ERR_CONT_READ));
  }
}


/* Nacteni celeho souboru z baliku (vcetne alokace pameti). Pokud neni len
 * rovno NULL, je do nej ulozena delka dat. */
void *pack_fget(char *name, int *len)
{
  pack_file f;
  void *buf;
  
  pack_fopen(&f, name);
  if (len) *len = f.len;
  buf = emalloc(f.len);
  pack_fread(&f, buf);
  return buf;
}


/* Vrati pocet kol v jednotlivych sadach (1 = jeden hrac, jednoducha kola;
 * 2 = jeden hrac, obtizna kola; 3 = dva hraci; 0 = dema). */
int cnt_level(int levelset)
{
  char name[16];
  int res;
  pack_file f;

  /* informaci o poctu kol zjistujeme ze souboru 1cntlvl, 2cntlvl, atd. */
  strcpy(name, "?cntlvl");
  name[0] = levelset + '0';
  pack_fopen(&f, name);
  if (f.len > 8) f.len = 8;
  pack_fread(&f, name);
  name[f.len] = '\0';
  if (sscanf(name, "%d", &res) != 1 || res <= 0)
    error(ls(STR_READDATA_BAD_LEVELS_NO), levelset);

  return res;
}


/* Rozpakuje definici kola. Vrati 1, pokud je vse v poradku, nebo 0, pokud
 * dodanych dat je malo. */
int unpack_level(uchar *data, int size, uchar *lev_def, uchar *lev_bagger)
{
  int i1, cnt;
  uchar b;

  for (i1 = 0; i1 < XSIZE_LEVELDEF * YSIZE_LEVELDEF; ) {
    if (--size < 0) return 0;
    b = *data++;
    if (b & 128 || !b) 
      /* je-li nacteny byte roven 0, nebo ma-li nastaven nejvyssi bit, jde
       * primo o hodnotu */
      cnt = 1;
    else {
      /* jinak je to RLE a nasledujici byte udava hodnotu */
      cnt = b;
      if (--size < 0) return 0;
      b = *data++;
    }
    for (; cnt > 0 && i1 < XSIZE_LEVELDEF * YSIZE_LEVELDEF; cnt--, i1++)
      *lev_def++ = b;
  }
  /* definice baggeru je vzdy nezkomprimovana na konci */
  if (size < CNT_BAGGER * SIZEDEF_BAGGER) return 0;
  if (lev_bagger)
    memcpy(lev_bagger, data, CNT_BAGGER * SIZEDEF_BAGGER);
  return 1;
}


/* Nacte kolo (z baliku). lev_bagger smi byt NULL (pak se ignoruje). */
void load_level(int levelset_no, int level_no, uchar *lev_def, uchar *lev_bagger)
{
  char name[16];
  uchar *buf;
  pack_file f;

  sprintf(name, "%dlvl%03d", levelset_no, level_no);
  pack_fopen(&f, name);
  buf = emalloc(f.len);
  pack_fread(&f, buf);
  if (!unpack_level(buf, f.len, lev_def, lev_bagger))
    error(ls(STR_READDATA_DAMAGED_LEVEL), levelset_no, level_no);
  free(buf);
}


/* Nacte externi kolo (tj. ze souboru). */
void load_extern_level(const char *name, uchar *lev_def, uchar *lev_bagger)
{
  FILE *f;
  uchar *buf;
  int size;

  f = fopen(name, "rb");
  if (!f) error(ls(STR_READDATA_ERR_FILE_OPEN), name);
  fseek(f, 0, SEEK_END);
  size = ftell(f);
  if (size < 0) error(ls(STR_READDATA_ERR_FILE_READ), name);
  rewind(f);
  buf = emalloc(size);
  if (fread(buf, 1, size, f) != size)
    error(ls(STR_READDATA_ERR_FILE_READ), name);
  fclose(f);
  if (!unpack_level(buf, size, lev_def, lev_bagger))
    error(ls(STR_READDATA_ERR_FILE_BAD), name);
  free(buf);
}


/* ulozi heslo prislusneho kola do pwd (pwd musi byt alokovano) */
void get_level_password(int levelset_no, int level_no, char *pwd)
{
  char name[16];
  char *buf;
  int len;

  strcpy(name, "?passwd.pwd");
  name[0] = levelset_no + '0';
  buf = pack_fget(name, &len);
  /* do len pocet hesel */
  len = (len + 1) / (PWD_LEN + 1);
  if (level_no > len)
    error(ls(STR_READDATA_MISSING_PASSWD), level_no);
  memcpy(pwd, buf + (level_no - 1) * (PWD_LEN + 1), PWD_LEN);
  pwd[PWD_LEN] = '\0';
  free(buf);
}


/* najde kolo prislusne zadanemu heslu - vrati cislo kola nebo -1 v pripade
 * neuspechu */
int find_level_password(int levelset_no, char *pwd)
{
  char name[16];
  char *buf, *pos;
  int len, i1;

  strcpy(name, "?passwd.pwd");
  name[0] = levelset_no + '0';
  buf = pack_fget(name, &len);
  /* do len pocet hesel */
  len = (len + 1) / (PWD_LEN + 1);
  pos = buf;
  for (i1 = 0; i1 < len; i1++) {
    if (!memcmp(pos, pwd, PWD_LEN)) {
      free(buf);
      return i1 + 1;
    }
    pos += PWD_LEN + 1;
  }
  free(buf);
  return -1;
}
