
/*

Flip Clock GStreamer based Media Playerback Module
Part of FlipClock C SDL for Maemo.

This is the media playback module for FlipClockC. It's pretty basic at the moment
but could grow in complexity if required in the future. It's loosely based
on my previous GStreamer concepts from the old Python version of Flip, mashed
together with the examples from the Maemo 5 SDK:
https://garage.maemo.org/svn/maemoexamples/trunk/maemo-examples/example_wavlaunch.c

-Rob Williams, Sept 27, 2009.


*/




#include <gtk/gtk.h>

#include <stdlib.h>
#include <gst/gst.h>

//****************** Structure defs *******************************************//

typedef struct _alarmSoundObj alarmSoundObj;


struct _alarmSoundObj {
	 GstElement *pipeline;		//Pipeline for the sound
	 char *fileType;				//Type of file, for now only MP3s
	 int looping;					//should file loop or not?
	 int stepTime;					//used for fading sound
	 int maxVol;					//used for fading sound

};

//****************** Done structure defs **************************************//

//****************** Global variables ******************************************//

char sinkModule[30];	//Name of sink module to use (dspmp3, pulseaudio, alsa); detected automatically

int playerReady = 0;

int canDecodeMP3 = 0;

int bahbah = 0;

//****************** Done global vars *****************************************//



int playPipe(GstElement *playerPipe);

int stopPipe(GstElement *playerPipe);

int stepVolumeUp( alarmSoundObj *alarmSound);

void cleanupPlayer(alarmSoundObj *alarmSound);

int playPipeTO(GstElement *playerPipe);

//****************** Function Definitions **************************************//

/*********************************************************
* void  eos_message_received(GstBus * bus, GstMessage * message,GstElement *pipeline) ;
*
* Global function that gets called by a steam whenever it reaches end of file (end of stream)
* 
*
**********************************************************/
void eos_message_received (GstBus * bus, GstMessage * message, alarmSoundObj *alarmObj) {
	GstState currentState = 0;

	if (message->type == GST_MESSAGE_EOS) {
		if (alarmObj->looping == 1) {
			gst_element_get_state(alarmObj->pipeline, &currentState, NULL, 0);
			
			//printf("state: %i\n", currentState);
			//printf("looping\n");
			
			bahbah++;
			
			if (bahbah < 5) {
				//g_timeout_add(5, (GSourceFunc) playPipeTO, alarmObj->pipeline);
			
				playPipe(alarmObj->pipeline);
				gst_element_get_state(alarmObj->pipeline, &currentState, NULL, 0);
				
				//printf("state: %i\n", currentState);
			}
		
		}	else {
			//Nope we're done so clean up
			//cleanupPlayer(alarmObj);
			//Stop the alarm since we're done
			stopAlarm();
			
		}
	}
	
}


/*********************************************************
* gstHasModule(char *modName)
*
* Tries to detect if gstreamer plugin of name modName is installed and available on the system
*
**********************************************************/

int gstFindBestSink(char *modName) {
	GstRegistry *thisReg;
	GList *pluginList, *current;
	GstPlugin *thisPlug;
	
	const gchar *plugName;
	
	int hasDSP = 0;
	int hasPulse = 0;
	int hasAlsa = 0;
	
	
	thisReg = gst_registry_get_default(); 
	pluginList = gst_registry_get_plugin_list(thisReg);
	
	//Loop over the plugins
	current = g_list_first(pluginList);
	
	while (current != NULL) {
		plugName = gst_plugin_get_name(current->data);
		
		if (!strcmp(plugName,"dspmp3")) {
			hasDSP = 1;
		}
		if (!strcmp(plugName,"pulseaudio")) {
			hasPulse = 1;
		}
		if (!strcmp(plugName,"alsa")) {
			hasAlsa = 1;
		}
		
		if (!strcmp(plugName,"mad")) {
			canDecodeMP3 = 1;
		}
		printf("gst plugin is %s\n", gst_plugin_get_name(current->data));
		current = g_list_next(current);
	}
	
	//Free up
	gst_plugin_list_free(pluginList);
	
	bzero(sinkModule, sizeof(sinkModule));
	if (hasDSP) {
		//best choice is DSP
		sprintf(sinkModule, "dspmp3sink");
	} else if (hasPulse) {
		sprintf(sinkModule, "pulsesink");
	} else if (hasAlsa) {
		sprintf(sinkModule, "alsasink");
	}
	
	printf("best sink mode is %s\n", sinkModule);
	return 1;

}


/*********************************************************
* GstElement *createGstPipe(char *outputType);
*
* Creates a new GStreamer pipeline for the given playback type
*
**********************************************************/

alarmSoundObj *createPlayer(char *outputType) {
	
    GstElement *pipeline = NULL;
    GError *error = NULL;
	GstBus *bus;
	
	alarmSoundObj *alarmSound;


	alarmSound = g_new0(alarmSoundObj, 1);
	
	alarmSound->looping = 0;	//No loop by default

	//Right now only mp3s, but ogg/etc could be added later theoretically...
	if (strcmp(outputType, "mp3") == 0) {
    	if (!strcmp(sinkModule, "dspmp3sink")) {
        	//use the DSP on the tablet to make things mroe CPU efficient?...
			//Remember because DSP does the decoding we need to set the fvolume on the sink...
           	alarmSound->pipeline = gst_parse_launch ("gnomevfssrc name=source ! dspmp3sink name=sink", &error);
         } else if (!strcmp(sinkModule, "pulsesink")) {
		 	if (canDecodeMP3) {
				//Use either alsa or pulse accordingly
				alarmSound->pipeline = gst_parse_launch ("gnomevfssrc name=source ! mad ! volume name=vol volume=1.0 ! pulsesink name=sink", &error);
			}
		} else if(!strcmp(sinkModule, "alsasink")) {
        	if (canDecodeMP3) {
				alarmSound->pipeline = gst_parse_launch ("gnomevfssrc name=source ! mad ! volume name=vol volume=1.0 ! alsasink name=sink", &error);
    		}
	    }
    }
	
	if (!alarmSound->pipeline) {
		printf("Error couldn't create pipeline for whatever reason\n");
		return alarmSound;
	}
	
	
	//Setup end of file handler
	  /* setup message handling */
	bus = gst_pipeline_get_bus (GST_PIPELINE (alarmSound->pipeline));
	gst_bus_add_signal_watch_full (bus, G_PRIORITY_HIGH);
	g_signal_connect (bus, "message::eos", (GCallback) eos_message_received, alarmSound);
	gst_object_unref (GST_OBJECT(bus));

	playerReady = 1;
	return alarmSound;
}


/*********************************************************
* void cleanupPlayer(alarmSoundObj *alarmSound);
*
* Cleans up and frees a previously created alarmSound object
*
**********************************************************/

void cleanupPlayer(alarmSoundObj *alarmSound) {
	if (playerReady != 0) {
		if (alarmSound != NULL) {
			//Clean up the player pipe
			if (alarmSound->pipeline != NULL) {
				stopPipe(alarmSound->pipeline);
				
				gst_object_unref(GST_OBJECT(alarmSound->pipeline));
				alarmSound->pipeline = NULL;
			}
	
			free(alarmSound);
	
			alarmSound = NULL;
		}
		playerReady = 0;
	}
}


/*********************************************************
* int loadFile createGstPipe(char *file, GstElement *playerPipe);
*
* Tries to load a file into the given playerPipe
*
**********************************************************/

int loadFile (char *file, GstElement *playerPipe) {

	int hasPrefix = 0;	//does thefile have the right vfs prefix?
	char *theFileName;
	GstElement *audioSrc = NULL;

	if (playerPipe->current_state == GST_STATE_PLAYING) {
		//We need to stop the pipe from playing
		stopPipe(playerPipe);
		
	}
	
	//Check to see if the file has a vfs prefix or not
	if (strstr(file, "file://") != NULL) {
		hasPrefix = 1;
	}
	
	theFileName = calloc(strlen(file) + 15, sizeof(char));
	if (!hasPrefix) {
		strcat(theFileName, "file://");
	} 
	strcat(theFileName, file);
	
	audioSrc = gst_bin_get_by_name (GST_BIN (playerPipe), "source");
	if (!audioSrc) {
	    printf ("Parse error: no sink\n");
		return 0;
	}

	g_object_set(G_OBJECT(audioSrc), "location", theFileName, NULL);	

	//printf("set source as %s\n", theFileName);
	free(theFileName);


	return 1;
}

/*********************************************************
* int setLoopMode(struct alarmSoundObj, int loopMode);
*
* Sets looping on or off for the given alarmSoundObj
*
**********************************************************/

int setLoopMode(int loopMode, alarmSoundObj *alarmSound) {
	alarmSound->looping = loopMode;
	printf("looping %i\n", loopMode);
	return 1;
}


/*********************************************************
* int playPipe(GstElement *playerPipe);
*
* Tries to play the given pipe 
*
**********************************************************/

int playPipe(GstElement *playerPipe) {
	GstStateChangeReturn result;

	gst_element_set_state (playerPipe, GST_STATE_READY);
	result = gst_element_set_state (playerPipe, GST_STATE_PLAYING);
	printf("play attempt result %i\n", result);
	
	return 1;
}

/*********************************************************
* int playPipe(GstElement *playerPipe);
*
* Tries to play the given pipe (from a timeout)
*
**********************************************************/

int playPipeTO(GstElement *playerPipe) {
	playPipe(playerPipe);
	return FALSE;
}


/*********************************************************
* int stopPipe(GstElement *playerPipe);
*
* Tries to stop the given pipe 
*
**********************************************************/

int stopPipe(GstElement *playerPipe) {
	GstState currentState = 0;
	if (playerReady == 1) {
		gst_element_get_state(playerPipe, &currentState, NULL, 0);
		if (currentState != GST_STATE_PAUSED) {
			gst_element_set_state (playerPipe, GST_STATE_PAUSED);
		}
		gst_element_get_state(playerPipe, &currentState, NULL, 0);
		if (currentState != GST_STATE_NULL) {
			gst_element_set_state (playerPipe, GST_STATE_NULL);
		}
	}
	return 1;
}

/*********************************************************
* int setPipeVolume(GstElement *playerPipe);
*
* Tries to set the volume of the given pipe
**********************************************************/

int setPipeVolume(GstElement *playerPipe, int volLevel) {
	gdouble realVol = 0;
	int useSinkVol = 0;			//Are we controlling volume element vol, or sink element volume? (i.e. tablet)
	GstElement *audioSink = NULL;
	
	audioSink = gst_bin_get_by_name (GST_BIN (playerPipe), "vol");
	if (!audioSink) {
		//No volume plugin, so grab the sink instead
		audioSink = gst_bin_get_by_name (GST_BIN (playerPipe), "sink");
	
		if (!audioSink) {
	    	printf ("Parse error: no sink\n");
			return 0;
		} else {
			useSinkVol = 1;
		}
	}
	
	if (0 <= volLevel  && volLevel <= 100) {
		realVol = volLevel / 100.0;
		if (useSinkVol == 0) {
			g_object_set(G_OBJECT (audioSink), "volume", realVol, NULL);
		} else {
			g_object_set(G_OBJECT (audioSink), "fvolume", realVol, NULL);
		}
		return 1;
	}
	
	return 0;
}

/*********************************************************
* int getPipeVolume(GstElement *playerPipe);
*
* Tries to get the volume of the given pipe
**********************************************************/

int getPipeVolume(GstElement *playerPipe) {
	
	int realVol = 0;
	gdouble rawVol	=0;
	int useSinkVol = 0;			//Are we controlling volume element vol, or sink element volume? (i.e. tablet)

	GstElement *audioSink = NULL;
	
	audioSink = gst_bin_get_by_name (GST_BIN (playerPipe), "vol");
	if (!audioSink) {
		//No volume plugin, so grab the sink instead
		audioSink = gst_bin_get_by_name (GST_BIN (playerPipe), "sink");
	
		if (!audioSink) {
	    	printf ("Parse error: no sink\n");
			return 0;
		} else {
			useSinkVol = 1;
		}
	}

	if (useSinkVol == 0) {
		g_object_get(G_OBJECT (audioSink), "volume", &rawVol, NULL);	
			
	} else {
		g_object_get(G_OBJECT (audioSink), "fvolume", &rawVol, NULL);	
		//realVol = (rawVol / 65535.0) * 100;
	}
	realVol = rawVol * 100;
	
	return realVol;

}


/*********************************************************
* int fadeSoundIn( alarmSoundObj *alarmSound, int millisecondsIn, int maxVol);
*
* Begins fading a sound in over millisecondsIn to maxVol
**********************************************************/

int fadeSoundIn( alarmSoundObj *alarmSound, int millisecondsIn, int maxVol) {
	
	//Start by setting vol low
	setPipeVolume(alarmSound->pipeline, 5);
	
	//Start playing
	playPipe(alarmSound->pipeline);
	
	alarmSound->stepTime = millisecondsIn / 20;		//5 steps by default
	alarmSound->maxVol = maxVol;
	
	g_timeout_add(alarmSound->stepTime, (GSourceFunc) stepVolumeUp, alarmSound);

	return 1;
}

/*********************************************************
* int stepVolumeUp(alarmSoundObj *alarmSound);
*
*Fades an alarm sound up
**********************************************************/

int stepVolumeUp( alarmSoundObj *alarmSound) {
	int currVol = 0;
	
	
	if (alarmSound != NULL) {
		if (playerReady == 1) {
			//Get the current volume
			currVol = getPipeVolume(alarmSound->pipeline);
			
			if (currVol < alarmSound->maxVol) {
				//Start by setting vol low
				setPipeVolume(alarmSound->pipeline, currVol + 5);
				
						
				g_timeout_add(alarmSound->stepTime, (GSourceFunc) stepVolumeUp, alarmSound);
			}
		}
	}
	return FALSE;

}