/*
  This source is part of the FindMine program.
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <iostream>

#include <Lum/OS/Probe.h>

#include <Lum/Base/Object.h>
#include <Lum/Base/String.h>
#include <Lum/Base/Util.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Boolean.h>
#include <Lum/Model/Integer.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/Array.h>
#include <Lum/Boolean.h>
#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/Grid.h>
#include <Lum/Label.h>
#include <Lum/Menu.h>
#include <Lum/Slider.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/StatusLine.h>
#include <Lum/View.h>
#include <Lum/WindowGroup.h>

#include "config.h"
#include "Configuration.h"
#include "Control.h"

static Lum::AppInfo info;

class ScoreView : public Lum::Dialog
{
private:
  Lum::Model::ListTableRef list;

public:
  ScoreView(Sweeper* sweeper)
  : Dialog(),
    list(new Lum::Model::ListTable(2))
  {
    Hiscore *score;

    score=GetConfig(sweeper->GetAreaWidth(),
                    sweeper->GetAreaHeight(),
                    sweeper->GetMines());

    for (size_t x=0; x<score->list.size(); x++) {
      Lum::Model::ListTable::StdEntry* entry;

      entry=new Lum::Model::ListTable::StdEntry(list);
      entry->SetString(1,score->list[x].name);
      entry->SetString(2,Lum::Base::NumberToWString((unsigned long)score->list[x].time));
      list->Append(entry);
    }
  }

  void PreInit()
  {
    Lum::Button           *button;
    Lum::ButtonRow        *bRow;
    Lum::Panel            *panel;
    Lum::Table            *table;
    Lum::WindowGroup      *wGroup;
    Lum::Model::HeaderRef headerModel;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);

    panel=new Lum::VPanel();
    panel->SetFlex(true,true);

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Name",Lum::Base::Size::stdCharWidth,18,true);
    headerModel->AddColumn(L"Time",Lum::Base::Size::stdCharWidth,8);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetMinWidth(Lum::Base::Size::stdCharWidth,30);
    table->SetShowHeader(true);
    table->GetTableView()->SetAutoFitColumns(true);
    table->GetTableView()->SetAutoVSize(true);
    table->SetModel(list);
    table->SetHeaderModel(headerModel);
    panel->Add(table);

    panel->Add(new Lum::VSpace());

    bRow=new Lum::ButtonRow();
    bRow->SetFlex(true,false);
    panel->Add(bRow);

    button=new Lum::Button();
    button->SetFlex(true,true);
    button->SetModel(GetClosedAction());
    button->SetType(Lum::Button::typeDefault);
    button->SetText(L"_Close");

    bRow->Add(button);

    wGroup->SetMain(panel);

    SetTop(wGroup);

    Dialog::PreInit();
  }
};

class ScoreEdit : public Lum::Dialog
{
private:
  Lum::Model::ActionRef closeAction;
  Lum::Model::StringRef name;
  size_t                pos;
  Hiscore               *score;

public:
  ScoreEdit(Sweeper* sweeper, size_t pos)
  : Dialog(),
    closeAction(new Lum::Model::Action()),
    name(new Lum::Model::String()),
    pos(pos)
  {
    score=GetConfig(sweeper->GetAreaWidth(),
                    sweeper->GetAreaHeight(),
                    sweeper->GetMines());

    AttachModel(closeAction);

    name->Set(score->list[pos].name);
    AttachModel(name);
  }

  ~ScoreEdit()
  {
    UnattachModel(name);
    UnattachModel(closeAction);
  }

  void PreInit()
  {
    Lum::Button      *button;
    Lum::ButtonRow   *bRow;
    Lum::Grid        *grid;
    Lum::Panel       *panel;
    Lum::String      *string;
    Lum::WindowGroup *wGroup;

    wGroup=new Lum::WindowGroup();

    panel=new Lum::VPanel();
    panel->SetFlex(true,true);

    grid=new Lum::Grid();
    grid->SetSpace(true,true);

    for (size_t x=0; x<10; x++) {
      Lum::Text *text;

      text=new Lum::Text();
      text->SetText(Lum::Base::NumberToWString((unsigned long)x+1)+L":");
      grid->SetObject(0,x,text);

      if (pos==x) {
        string=new Lum::String();
        string->SetFlex(true,false);
        string->SetWidth(Lum::Base::Size::stdCharWidth,10);
        string->SetModel(name);
        grid->SetObject(1,x,string);
      }
      else {
        text=new Lum::Text();
        text->SetFlex(true,false);
        text->SetAlignment(Lum::Text::left);
        if (x<score->list.size()) {
          text->SetText(score->list[x].name);
        }
        else {
          text->SetText(L"");
        }
        grid->SetObject(1,x,text);
      }

      text=new Lum::Text();
      text->SetAlignment(Lum::Text::left);
      text->SetFlex(true,false);
      if (x<score->list.size()) {
        text->SetText(Lum::Base::NumberToWString((unsigned long)score->list[x].time)+L" second(s)");
      }
      else {
        text->SetText(L"");
      }
      grid->SetObject(2,x,text);
    }

    panel->Add(grid);

    panel->Add(new Lum::VSpace());

    bRow=new Lum::ButtonRow();
    bRow->SetFlex(true,false);
    panel->Add(bRow);

    button=new Lum::Button();
    button->SetFlex(true,true);
    button->SetModel(closeAction);
    button->SetType(Lum::Button::typeDefault);
    button->SetText(L"_Close");

    bRow->Add(button);

    wGroup->SetMain(panel);

    SetTop(wGroup);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==closeAction &&  closeAction->IsFinished()) {
      Exit();
    }
    else if (model==name) {
      score->list[pos].name=name->Get();
      SetUserName(name->Get());
    }
    else {
      Dialog::Resync(model,msg);
    }
  }
};

class SelectTheme : public Lum::Dialog
{
private:
  Lum::Model::ActionRef              okAction;
  Lum::Model::ListTableRef           list;
  Lum::Model::SingleLineSelectionRef selection;
  std::wstring                       result;

public:
  SelectTheme()
  : okAction(new Lum::Model::Action),
    list(new Lum::Model::ListTable(2)),
    selection(new Lum::Model::SingleLineSelection),
    result(L"")
  {
    okAction->Disable();

    AttachModel(okAction);
    AttachModel(selection);

    for (size_t i=0; i<themes.size(); ++i) {
      Lum::Model::ListTable::StdEntry *entry;

      entry=new Lum::Model::ListTable::StdEntry(list);
      entry->SetString(1,themes[i].name);
      entry->SetString(2,Lum::Base::NumberToWString(themes[i].boxSize));

      list->Append(entry);
    }
  }

  ~SelectTheme()
  {
    UnattachModel(selection);
    UnattachModel(okAction);
  }

  void PreInit()
  {
    Lum::Button           *button;
    Lum::ButtonRow        *bRow;
    Lum::Panel            *panel;
    Lum::Table            *table;
    Lum::WindowGroup      *wGroup;
    Lum::Model::HeaderRef headerModel;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);
    wGroup->SetWidth(Lum::Base::Size::modePixel,(GetWindow()->GetParent()->GetWidth()*80)/100);
    wGroup->SetHeight(Lum::Base::Size::modePixel,(GetWindow()->GetParent()->GetHeight()*80)/100);

    panel=new Lum::VPanel();
    panel->SetFlex(true,true);

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Theme name",Lum::Base::Size::stdCharWidth,25);
    headerModel->AddColumn(L"box size",Lum::Base::Size::stdCharWidth,13);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetShowHeader(true);
    table->SetModel(list);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetDoubleClickAction(okAction);
    panel->Add(table);

    panel->Add(new Lum::VSpace());

    bRow=new Lum::ButtonRow();
    bRow->SetFlex(true,false);
    panel->Add(bRow);

    button=new Lum::Button();
    button->SetFlex(true,true);
    button->SetModel(okAction);
    button->SetType(Lum::Button::typeCommit);
    button->SetText(L"_Ok");
    bRow->Add(button);

    button=new Lum::Button();
    button->SetFlex(true,true);
    button->SetModel(GetClosedAction());
    button->SetType(Lum::Button::typeCancel);
    button->SetText(L"_Cancel");
    bRow->Add(button);

    wGroup->SetMain(panel);

    SetTop(wGroup);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      if (selection->HasSelection()) {
        result=themes[selection->GetLine()-1].name;
      }

      Exit();
    }
    else if (model==selection) {
      if (selection->HasSelection()) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }
    }

    Dialog::Resync(model,msg);
  }

  std::wstring GetResult() const
  {
    return result;
  }
};

class Settings : public Lum::Dialog
{
private:
  Lum::Model::ActionRef  smallAction;
  Lum::Model::ActionRef  mediumAction;
  Lum::Model::ActionRef  bigAction;
  Lum::Model::ActionRef  themeAction;

public:
  Lum::Model::IntRef     horiz;
  Lum::Model::IntRef     vert;
  Lum::Model::IntRef     mines;
  Lum::Model::BooleanRef oneButtonMode;
  Lum::Model::StringRef  themeName;

public:
  Settings(Sweeper* sweeper)
  : Dialog(),
    smallAction(new Lum::Model::Action()),
    mediumAction(new Lum::Model::Action()),
    bigAction(new Lum::Model::Action()),
    themeAction(new Lum::Model::Action()),
    horiz(new Lum::Model::Int(sweeper->GetAreaWidth())),
    vert(new Lum::Model::Int(sweeper->GetAreaHeight())),
    mines(new Lum::Model::Int(sweeper->GetMinesPercent())),
    oneButtonMode(new Lum::Model::Boolean(GetOneButtonMode())),
    themeName(new Lum::Model::String())
  {
    AttachModel(smallAction);
    AttachModel(mediumAction);
    AttachModel(bigAction);
    AttachModel(themeAction);

    horiz->SetRange(1,30);
    vert->SetRange(1,30);
    mines->SetRange(0,100);

    themeName->Set(::themeName);
  }

  ~Settings()
  {
    UnattachModel(themeAction);
    UnattachModel(smallAction);
    UnattachModel(mediumAction);
    UnattachModel(bigAction);
  }

  void PreInit()
  {
    Lum::Boolean     *boolean;
    Lum::Button      *button;
    Lum::ButtonRow   *bRow;
    Lum::Label       *label;
    Lum::Panel       *panel,*hPanel;
    Lum::Array       *array;
    Lum::Slider      *slider;
    Lum::TextValue   *textValue;
    Lum::WindowGroup *wGroup;

    wGroup=new Lum::WindowGroup();

    panel=new Lum::VPanel();

    label=new Lum::Label();

    slider=new Lum::HSlider();
    slider->SetFlex(true,false);
    slider->SetScale(true);
    slider->SetModel(horiz);
    label->AddLabel(L"Horizontal:",slider);

    slider=new Lum::HSlider();
    slider->SetFlex(true,false);
    slider->SetScale(true);
    slider->SetModel(vert);
    label->AddLabel(L"Vertical:",slider);

    slider=new Lum::HSlider();
    slider->SetFlex(true,false);
    slider->SetScale(true);
    slider->SetModel(mines);
    label->AddLabel(L"Mines (in %):",slider);

    array=new Lum::Array();
    array->SetFlex(true,false);
    array->SetSpace(true,false);
    array->SetVerticalCount(1);

    button=new Lum::Button();
    button->SetFlex(true,true);
    button->SetModel(smallAction);
    button->SetText(L"_Small");
    array->Add(button);

    button=new Lum::Button();
    button->SetFlex(true,true);
    button->SetModel(mediumAction);
    button->SetText(L"_Medium");
    array->Add(button);

    button=new Lum::Button();
    button->SetFlex(true,true);
    button->SetModel(bigAction);
    button->SetText(L"_Big");
    array->Add(button);

    label->AddLabel(L"Presets:",array);

    boolean=new Lum::Boolean();
    boolean->SetModel(oneButtonMode);

    label->AddLabel(L"One mouse button mode:",boolean);

    hPanel=new Lum::HPanel();
    hPanel->SetFlex(true,false);

    textValue=new Lum::TextValue();
    textValue->SetFlex(true,false);
    textValue->SetModel(themeName);
    hPanel->Add(textValue);

    button=new Lum::Button();
    button->SetFlex(false,true);
    button->SetModel(themeAction);
    button->SetText(L"...");
    hPanel->Add(button);

    label->AddLabel(L"Theme:",hPanel);

    panel->Add(label);

    panel->Add(new Lum::VSpace());

    bRow=new Lum::ButtonRow();
    bRow->SetFlex(true,false);
    panel->Add(bRow);

    button=new Lum::Button();
    button->SetFlex(true,true);
    button->SetModel(GetClosedAction());
    button->SetType(Lum::Button::typeDefault);
    button->SetText(L"_Close");

    bRow->Add(button);

    wGroup->SetMain(panel);

    SetTop(wGroup);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==smallAction && smallAction->IsFinished()) {
      horiz->Set(8);
      vert->Set(8);
      mines->Set(Lum::Base::RoundDiv(10*100,8*8));
    }
    else if (model==mediumAction && mediumAction->IsFinished()) {
      horiz->Set(16);
      vert->Set(16);
      mines->Set(Lum::Base::RoundDiv(40*100,16*16));
    }
    else if (model==bigAction && bigAction->IsFinished()) {
      horiz->Set(30);
      vert->Set(16);
      mines->Set(Lum::Base::RoundDiv(99*100,30*16));
    }
    else if (model==themeAction && themeAction->IsFinished()) {
      SelectTheme *dialog=new SelectTheme();

      dialog->SetParent(this);
      if (dialog->Open()) {
        dialog->SetExitAction(dialog->GetClosedAction());
        dialog->EventLoop();
        dialog->Close();
      }

      if (!dialog->GetResult().empty()) {
        ::themeName=dialog->GetResult();
        themeName->Set(dialog->GetResult());
      }

      delete dialog;
    }
    else {
      Dialog::Resync(model,msg);
    }
  }
};

class MyWindow : public Lum::Dialog
{
private:
  Sweeper               *sweeper;
  Lum::Model::ActionRef newGameAction;
  Lum::Model::ActionRef hiscoreAction;
  Lum::Model::ActionRef settingsAction;
  Lum::Model::ActionRef helpAction;
  Lum::Model::ActionRef aboutAction;
  Lum::Model::ActionRef timerAction;

  Lum::Text             *marks;
  Lum::Text             *time;

public:

  MyWindow()
  : sweeper(NULL),
    newGameAction(new Lum::Model::Action),
    hiscoreAction(new Lum::Model::Action),
    settingsAction(new Lum::Model::Action),
    helpAction(new Lum::Model::Action),
    aboutAction(new Lum::Model::Action),
    timerAction(new Lum::Model::Action)
  {
    AttachModel(newGameAction);
    AttachModel(hiscoreAction);
    AttachModel(settingsAction);
    AttachModel(helpAction);
    AttachModel(aboutAction);
    AttachModel(timerAction);

    AttachModel(GetOpenedAction());
  }

  ~MyWindow()
  {
    UnattachModel(GetOpenedAction());

    UnattachModel(timerAction);
    UnattachModel(aboutAction);
    UnattachModel(helpAction);
    UnattachModel(settingsAction);
    UnattachModel(hiscoreAction);
    UnattachModel(newGameAction);
  }

  void PreInit()
  {
    Lum::Button      *button;
    Lum::MenuStrip   *strip;
    Lum::Menu        *menu;
    Lum::Panel       *vPanel,*hPanel;
    Lum::View        *view;
    Lum::WindowGroup *wGroup;
    size_t           digitWidth,widestDigit;
    std::wstring     timeFormat,marksFormat;
    Lum::OS::FontRef font;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);

    vPanel=new Lum::VPanel;
    vPanel->SetFlex(true,true);

    hPanel=new Lum::HPanel();
    hPanel->SetFlex(true,false);

    button=new Lum::Button();
    button->SetText(L"_New game");
    button->SetModel(newGameAction);
    hPanel->Add(button);

    hPanel->Add(new Lum::HSpace(Lum::Space::sizeNormal,true));

    font=Lum::OS::display->GetFont(Lum::OS::Display::fontTypeFixed,210);

    digitWidth=0;
    widestDigit=0;
    for (size_t d=0; d<=9; d++) {
      size_t width;

      width=font->StringWidth(Lum::Base::NumberToWString(d));

      if (width>digitWidth) {
        digitWidth=width;
        widestDigit=d;
      }
    }

    marksFormat=L"-";
    for (size_t x=1; x<=3; x++) {
      marksFormat.append(Lum::Base::NumberToWString(widestDigit));
    }

    marks=new Lum::Text();
    marks->SetFont(font);
    marks->SetStyle(Lum::OS::Font::bold);
    marks->SetAlignment(Lum::Text::right);
    marks->SetMinWidth(Lum::Base::Size::modePixel,font->StringWidth(marksFormat));
    marks->SetText(L"0");

    hPanel->Add(marks);

    hPanel->Add(new Lum::HSpace(Lum::Space::sizeNormal,true));

    for (size_t x=1; x<=4; x++) {
      timeFormat.append(Lum::Base::NumberToWString(widestDigit));
    }

    time=new Lum::Text();
    time->SetFont(font);
    time->SetStyle(Lum::OS::Font::bold);
    time->SetAlignment(Lum::Text::centered);
    time->SetMinWidth(Lum::Base::Size::modePixel,font->StringWidth(timeFormat));
    time->SetText(L"0000");

    hPanel->Add(time);

    vPanel->Add(hPanel);

    vPanel->Add(new Lum::VSpace());

    view=new Lum::View();
    view->SetFlex(true,true);

    sweeper=new Sweeper();
    sweeper->SetFlex(true,true);
    AttachModel(sweeper->GetStatusModel());
    AttachModel(sweeper->GetMarksModel());

    view->SetObject(sweeper);
    vPanel->Add(view);

    strip=new Lum::MenuStrip();

    menu=new Lum::Menu();
    menu->AddActionItem(L"_New Game",newGameAction,Lum::OS::qualifierControl,L"n");
    menu->AddSeparator();
    menu->AddActionItem(L"_Hiscore...",hiscoreAction);
    menu->AddSeparator();
    menu->AddActionItem(L"_Quit",GetClosedAction(),Lum::OS::qualifierControl,L"q");
    strip->AddPullDownMenu(L"_Game",menu);

    menu=new Lum::Menu();
    menu->AddActionItem(L"_Settings...",settingsAction);

    strip->AddPullDownMenu(L"_Settings",menu);

    menu=new Lum::Menu();
    menu->AddActionItem(L"_Help...",helpAction);
    menu->AddSeparator();
    menu->AddActionItem(L"_About...",aboutAction);

    strip->AddPullDownMenu(L"_Help",menu);

    wGroup->SetMenuStrip(strip);
    wGroup->SetMain(vPanel);

    SetTop(wGroup);

    Dialog::PreInit();
  }

  size_t GetHiscorePos() const
  {
    Hiscore *score;
    size_t  pos;

    score=GetConfig(sweeper->GetAreaWidth(),
                    sweeper->GetAreaHeight(),
                    sweeper->GetMines());

    pos=0;
    while (pos<score->list.size() && sweeper->GetElapsedTime()>=score->list[pos].time) {
      pos++;
    }

    return pos;
  }

  void ShowSettings()
  {
    Settings* settings;

    settings=new Settings(sweeper);
    settings->SetParent(this);
    if (settings->Open()) {
      settings->SetExitAction(settings->GetClosedAction());
      settings->EventLoop();
      settings->Close();

      if (sweeper->SetSize(settings->horiz->Get(),
                           settings->vert->Get(),
                           Lum::Base::RoundDiv(settings->horiz->Get()*settings->vert->Get()*settings->mines->Get(),100))) {
        SetCurrentGame(sweeper->GetAreaWidth(),
                       sweeper->GetAreaHeight(),
                       sweeper->GetMines());
        SetOneButtonMode(settings->oneButtonMode->Get());

        sweeper->SetOneButtonMode(settings->oneButtonMode->Get());
        sweeper->EvaluateThemeName();
        sweeper->Run();
      }

    }

    delete settings;
  }

  void ShowHiscore()
  {
    ScoreView* score;

    score=new ScoreView(sweeper);
    score->SetTitle(L"Hiscore");
    score->SetParent(this);
    if (score->Open()) {
      score->SetExitAction(score->GetClosedAction());
      score->EventLoop();
      score->Close();
    }

    delete score;
  }

  void ShowHiscoreEdit(size_t pos)
  {
    Hiscore::Entry                        entry;
    ScoreEdit                             *scoreDlg;
    Hiscore*                              score;
    std::vector<Hiscore::Entry>::iterator iter;

    score=GetConfig(sweeper->GetAreaWidth(),
                    sweeper->GetAreaHeight(),
                    sweeper->GetMines());

    entry.time=sweeper->GetElapsedTime();
    entry.name=GetUserName();

    iter=score->list.begin();
    iter+=pos;

    score->list.insert(iter,entry);

    scoreDlg=new ScoreEdit(sweeper,pos);
    scoreDlg->SetTitle(L"Hiscore");
    scoreDlg->SetParent(this);

    if (scoreDlg->Open()) {
      scoreDlg->SetExitAction(scoreDlg->GetClosedAction());
      scoreDlg->EventLoop();
      scoreDlg->Close();
    }

    delete scoreDlg;
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      LoadThemes();

      if (!LoadConfig()) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"No configuration file found!",
                              L"The program was not able to find its configuration file.\n"
                              L"\n"
                              L"If this is the first start of your program, this is\n"
                              L"OK. The program will generate a configuration file if\n"
                              L"it quits.\n"
                              L"If this is not your first use of this program, you may not\n"
                              L"have solved any levels up to now, have not correctly quit\n"
                              L"the program, or have deleted the history of solved levels!");
      }

      size_t width,height,mines;

      GetCurrentGame(width,height,mines);
      sweeper->EvaluateThemeName();
      sweeper->SetSize(width,height,mines);
      sweeper->SetOneButtonMode(GetOneButtonMode());
    }
    if (model==newGameAction && newGameAction->IsFinished()) {
      sweeper->Run();
    }
    else if (model==hiscoreAction && hiscoreAction->IsFinished()) {
      ShowHiscore();
    }
    else if (model==settingsAction && settingsAction->IsFinished()) {
      ShowSettings();
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==helpAction && helpAction->IsFinished()) {
      Lum::Dlg::Msg::ShowOk(this,
                            L"Game playing hints...",
                            L"In normal mode, you just open a square by pressing\n"
                            L"the left button on it.\n"
                            L"You mark a square as containing a bomb by pressing\n"
                            L"the right mouse button on it.\n"
                            L"\n"
                            L"If you don't have a right mouse button you can\n"
                            L"activate the one button mouse mode in the settings\n"
                            L"dialog.\n"
                            L"In this case you mark squares as containing a bomb with\n"
                            L"the left mouse button, and open a square by doing a\n"
                            L"diagonal stroke over the square.");
    }
    else if (sweeper!=NULL && model==sweeper->GetStatusModel()) {
      if (sweeper->GetStatus()==Sweeper::won) {
        sweeper->SetStatusToWaiting();
        size_t pos;

        pos=GetHiscorePos();
        if (pos<10) {
          ShowHiscoreEdit(pos);
        }
        else {
          if (Lum::Dlg::Msg::Ask(this,
                                 L"You have won!",
                                 L"You have won.\n"
                                 L"\n"
                                 L"...but sadly you did not make it into the hiscore.",
                                 L"Play one more!*|No, enough!^")==0) {
            sweeper->Run();
          }
        }
      }
      else if (sweeper->GetStatus()==Sweeper::lost) {
        sweeper->SetStatusToWaiting();
        if (Lum::Dlg::Msg::Ask(this,
                               L"Sorry, you have lost.",
                               L"Sorry, you have lost. Please try again.",
                               L"Play one more!*|No, enough!^")==0) {
          sweeper->Run();
        }
      }
      else if (sweeper->GetStatus()==Sweeper::playing) {
        Lum::OS::display->AddTimer(1,0,timerAction);
      }
    }
    else if (model==timerAction && timerAction->IsFinished()) {
      if (sweeper->GetStatus()==Sweeper::playing) {
        time_t       time=sweeper->GetElapsedTime();
        std::wstring value;

        if (time>9999) {
          time=9999;
        }

        value=Lum::Base::NumberToWString(time);

        while (value.length()<4) {
          value.insert(0,L"0");
        }

        this->time->SetText(value);
      }
      Lum::OS::display->AddTimer(1,0,timerAction);
    }
    else if (sweeper!=NULL && model==sweeper->GetMarksModel()) {
      this->marks->SetText(Lum::Base::NumberToWString(sweeper->GetMarksModel()->Get()));
    }
    else if (model==GetWindow()->GetMapedAction() && GetWindow()->GetMapedAction()->IsFinished()) {
      Lum::OS::display->AddTimer(1,0,timerAction);
    }
    else if (model==GetWindow()->GetUnmapedAction() && GetWindow()->GetMapedAction()->IsFinished()) {
      Lum::OS::display->RemoveTimer(timerAction);
    }

    Dialog::Resync(model,msg);
  }
};

int main(int argc, char* argv[])
{
  info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
  info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
  info.SetDescription(L"Mark all the hidden mines...");
  info.SetAuthor(L"Tim Teulings");
  info.SetContact(L"Tim Teulings <tim@teulings.org>");
  info.SetCopyright(L"(c) 2004, Tim Teulings");
  info.SetLicense(L"GNU Public License");

  if (Lum::OS::prober->Open(L"FindMine",argc,argv)) {
    MyWindow* window;

    window=new MyWindow;

    if (window->Open()) {
      window->SetExitAction(window->GetClosedAction());
      window->EventLoop();
      window->Close();
    }

    delete window;

    Lum::OS::display->Close();
  }

  SaveConfig();

  FreeConfig();
}
