# coding: utf-8

import gobject
import gtk
import pango
from threading import Thread
from threading import Lock
import hildon
import uuid
import os
import ConfigParser
from datetime import date
from datetime import timedelta
from datetime import datetime
import feedcircuit
import urllib
import webbrowser
import glob
import shutil
import sys
from BeautifulSoup import BeautifulStoneSoup

gtk.gdk.threads_init()

#This is to be able to start Feed Circuit on PC
try:
    import osso
    osso_context = osso.Context("org.maemo.feedcircuit", "0.6.9", False)
except ImportError:
    pass

#This is also to be able to start Feed Circuit on PC
try:
    import conic


    func_to_exec = None
    args_to_use = None


    def connection_cb(connection, event, magic):
        global func_to_exec, args_to_use
        if event.get_status() == conic.STATUS_CONNECTED:
            if func_to_exec:
                func_to_exec(*args_to_use)
                func_to_exec = None


    connection = conic.Connection()
    connection.connect("connection-event", connection_cb, None)


    def connect_and_exec(func, *args):
        global func_to_exec, args_to_use
        func_to_exec = func
        args_to_use = args
        connection.request_connection(conic.CONNECT_FLAG_NONE)


except ImportError:
    
    def connect_and_exec(func, *args):
        func(*args)


def create_config_dir():
    path = os.path.expanduser("~/.config/feedcircuit")
    if not os.path.exists(path):
        os.makedirs(path)
    return path


def save_icon(filename, icon_data):
    f = open(filename, "w")
    try:
        f.write(icon_data)
    finally:
        f.close()


class BannerThread(Thread):


    def __init__(self, parent, text):
        Thread.__init__(self)
        self.__banner = hildon.hildon_banner_show_animation(parent, "qgn_indi_pball_a", text)


    def run(self):
        try:
            result = self.__callback(*self.__args)
            gobject.idle_add(self.done, result, None)
        except Exception, ex:
            gobject.idle_add(self.done, None, ex)


    def done(self, result, ex):
        self.__result = result
        self.__exception = ex
        gtk.main_quit()

    def execute(self, callback, *args):
        self.__callback = callback
        self.__args = args
        self.__result = None
        self.__exception = None
        self.start()
        gtk.main()
        self.__banner.destroy()
        if self.__exception:
            raise self.__exception
        return self.__result


class ConfigParserEx(ConfigParser.SafeConfigParser):


    def get_default(self, section, name, default):
        if self.has_option(section, name):
            return self.get(section, name)
        return default


    def getint_default(self, section, name, default):
        if self.has_option(section, name):
            return self.getint(section, name)
        return default


    def getboolean_default(self, section, name, default):
        if self.has_option(section, name):
            return self.getboolean(section, name)
        return default


    def set(self, section, option, value):
        val = value
        if val:
            val = val.replace("%", "%%")
        ConfigParser.SafeConfigParser.set(self, section, option, val)

class FeedConfig:


    mode_ommit = 0
    mode_cache = 1
    mode_inline = 2


    def __init__(self, cache_path, url, name, mode, days_to_keep,
        include, exclude, ignore_print_version, icon_filename,
        message_count, as_is, content_tag, print_version_link, reformatter_url,
        next_page_link, id = None):
        self.id = id
        if not self.id:
            self.id = uuid.uuid1().hex
        self.cache_path = cache_path
        self.url = url
        self.name = name
        self.mode = mode
        self.days_to_keep = days_to_keep
        self.include = include
        self.exclude = exclude
        self.ignore_print_version = ignore_print_version
        self.icon_filename = icon_filename
        self.message_count = message_count
        self.as_is = as_is
        self.content_tag = content_tag
        self.print_version_link = print_version_link
        self.reformatter_url = reformatter_url
        self.next_page_link = next_page_link
        self.icon = None
        self.iter = None
        self.ord_num = 0
        self.checked = False


    def get_icon(self):
        if not self.icon:
            if self.icon_filename:
                icon_filepath = os.path.join(self.cache_path, self.icon_filename)
                if os.path.exists(icon_filepath):
                    self.icon = gtk.gdk.pixbuf_new_from_file(icon_filepath)
                    if self.icon.get_height() > 32:
                        self.icon = self.icon.scale_simple(
                            self.icon.get_width() * 32 / self.icon.get_height(), 32, gtk.gdk.INTERP_BILINEAR)
        return self.icon


    def get_time(self):
        filename = os.path.join(self.cache_path, self.id + ".html")
        if os.path.exists(filename):
            return datetime.fromtimestamp(os.path.getmtime(filename)).strftime("%c")
        return ""


    def get_markup(self):
        if self.message_count:
            return "<b>" + self.name + " (" + str(self.message_count) + ")</b>"
        return self.name


    def get_print_version_keywords(self):
        if self.print_version_link:
            return [self.print_version_link, ]


    def get_content_kwargs(self):
        result = {}
        if self.content_tag:
            parts = self.content_tag.split(" ")
            if len(parts):
                result["name"] = parts[0]
                result["attrs"] = {}
                for part in parts[1:]:
                    name_val = part.split("=")
                    if len(name_val) == 2:
                        result["attrs"][name_val[0]] = \
                            name_val[1].replace('"', "").replace("&quot;", '"')
        return result


    def get_urls(self):
        return filter(len, [str.strip() for str in self.url.split(" ")])


class FeedListModel(gtk.GenericTreeModel):


    def __init__(self, config_path):
        gtk.GenericTreeModel.__init__(self)
        self.column_types = (gtk.gdk.Pixbuf, str, str, gobject.TYPE_BOOLEAN)
        self.list = []
        self.map = {}
        self.config_path = config_path
        self.load_config()


    def on_get_flags(self):
        return gtk.TREE_MODEL_LIST_ONLY|gtk.TREE_MODEL_ITERS_PERSIST


    def on_get_n_columns(self):
        return len(self.column_types)


    def on_get_column_type(self, n):
        return self.column_types[n]


    def on_get_iter(self, path):
        if path[0] < len(self.list):
            return self.list[path[0]].id
        return None


    def on_get_path(self, rowref):
        return self.map[rowref]


    def on_get_value(self, rowref, column):
        if self.map.has_key(rowref):
            if column is 0:
                return self.list[self.map[rowref]].get_icon()
            elif column is 1:
                return self.list[self.map[rowref]].get_markup()
            elif column is 2:
                return self.list[self.map[rowref]].get_time()
            elif column is 3:
                return self.list[self.map[rowref]].checked


    def on_iter_next(self, rowref):
        index = self.map[rowref] + 1
        if index < len(self.list):
            return self.list[index].id
        return None


    def on_iter_children(self, rowref):
        if rowref:
            return None
        return self.list[0].id


    def on_iter_has_child(self, rowref):
        return False


    def on_iter_parent(self, child):
        return None


    def on_iter_n_children(self, rowref):
        if rowref:
            return 0
        return len(self.list)


    def on_iter_nth_child(self, rowref, n):
        if not rowref and n < len(self.list):
            return self.list[n].id
        return None


    def get_config_file_name(self):
        return os.path.join(self.config_path, "config")


    def load_config(self):
        parser = ConfigParserEx()
        parser.read([self.get_config_file_name()])
        self.cache_path = parser.get_default("global", "cache path", self.config_path)
        self.use_checkboxes = parser.getboolean_default("global", "use checkboxes", False)
        for section in parser.sections():
            if section != "global":
                feed_config = FeedConfig(
                    self.cache_path,
                    parser.get(section, "url"),
                    parser.get(section, "name"),
                    parser.getint_default(section, "mode", FeedConfig.mode_ommit),
                    parser.getint_default(section, "days", 7),
                    parser.get_default(section, "include", ""),
                    parser.get_default(section, "exclude", ""),
                    parser.getboolean_default(section, "ignore print version", False),
                    parser.get_default(section, "icon", ""),
                    parser.getint_default(section, "message count", 0),
                    parser.getboolean_default(section, "as is", False),
                    parser.get_default(section, "content tag", ""),
                    parser.get_default(section, "print version link", ""),
                    parser.get_default(section, "reformatter url", ""),
                    parser.get_default(section, "next page link", ""),
                    section)
                feed_config.ord_num = parser.getint_default(section, "order number", 0)
                self.list.append(feed_config)
        self.list.sort(lambda f1, f2: f1.ord_num.__cmp__(f2.ord_num))
        self.rebuild_map()


    def save_config(self):
        f = open(os.path.join(self.config_path, "config"), "w")
        try:
            parser = ConfigParserEx()
            parser.add_section("global")
            parser.set("global", "cache path", self.cache_path)
            parser.set("global", "use checkboxes", str(self.use_checkboxes))
            i = 0
            for feed_config in self.list:
                parser.add_section(feed_config.id)
                parser.set(feed_config.id, "url", feed_config.url)
                parser.set(feed_config.id, "name", feed_config.name)
                parser.set(feed_config.id, "mode", str(feed_config.mode))
                parser.set(feed_config.id, "days", str(feed_config.days_to_keep))
                parser.set(feed_config.id, "include", feed_config.include)
                parser.set(feed_config.id, "exclude", feed_config.exclude)
                parser.set(feed_config.id, "ignore print version", str(feed_config.ignore_print_version))
                parser.set(feed_config.id, "icon", feed_config.icon_filename)
                parser.set(feed_config.id, "message count", str(feed_config.message_count))
                parser.set(feed_config.id, "as is", str(feed_config.as_is))
                parser.set(feed_config.id, "content tag", feed_config.content_tag)
                parser.set(feed_config.id, "print version link", feed_config.print_version_link)
                parser.set(feed_config.id, "reformatter url", feed_config.reformatter_url)
                parser.set(feed_config.id, "next page link", feed_config.next_page_link)
                parser.set(feed_config.id, "order number", str(i))
                i += 1
            parser.write(f)
        finally:
            f.close()


    def rebuild_map(self):
        self.map.clear()
        for i in range(len(self.list)):
            self.map[self.list[i].id] = i

    def update(self, rowref):
        index = self.map[rowref]
        self.row_changed((index,), self.get_iter((index,)))
        self.save_config()


    def delete(self, rowref):
        index = self.map[rowref]
        del self.list[index]
        self.rebuild_map()
        self.row_deleted((index,))
        self.save_config()


    def append(self, feed_config):
        index = len(self.list)
        self.map[feed_config.id] = index
        self.list.append(feed_config)
        self.row_inserted((index,), self.get_iter((index,)))
        self.save_config()


    def update_cache_path(self, cache_path):
        self.cache_path = cache_path
        for feed_config in self.list:
            feed_config.cache_path = cache_path


    def move(self, id, pos):
        before_id = None
        if pos >= 0 and pos < len(self.list):
            before_id = self.list[pos].id
        if before_id == id:
            return
        old_order = [f.id for f in self.list]
        index = self.map[id]
        feed_config = self.list[index]
        del self.list[index]
        self.rebuild_map()
        if before_id:
            self.list.insert(self.map[before_id], feed_config)
        else:
            self.list.append(feed_config)
        self.rebuild_map()
        self.rows_reordered(None, None, [self.map[id] for id in old_order])
        self.save_config()


    def toggle(self, path):
        feed_config = self.list[path[0]]
        feed_config.checked = not feed_config.checked
        self.row_changed(path, self.get_iter(path))


    def set_checkboxes(self, value):
        self.use_checkboxes = value
        if not self.use_checkboxes:
            for feed_config in self.list:
                feed_config.checked = False


class UrlsWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
             gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.resize(*parent.get_size())
        self.set_title(_("Feed urls"))
        self.vbox.pack_start(gtk.Label(_("One url per line")), expand = False)
        self.text_buffer = gtk.TextBuffer()
        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.set_shadow_type(gtk.SHADOW_IN)
        text_view = gtk.TextView(self.text_buffer)
        sw.add(text_view)
        self.vbox.pack_start(sw, expand = True)
        self.show_all()


class PreferencesWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
             gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.set_title(_("Preferences"))
        hbox = gtk.HBox()
        self.path_entry = gtk.Entry()
        self.path_entry.set_width_chars(24)
        hbox.pack_start(self.path_entry, expand = True)
        self.choose_button = gtk.ToolButton(stock_id = gtk.STOCK_DIRECTORY)
        self.choose_button.connect("clicked", self.choose_button_clicked)
        hbox.pack_start(self.choose_button, expand = False)
        self.checkboxes_check = gtk.CheckButton()
        table = gtk.Table(rows = 2, columns = 2)
        texts = [_("Cache path"), _("Show checkboxes")]
        for i in range(0, len(texts)):
            label = gtk.Label(texts[i])
            label.set_alignment(1, 0.5)
            table.attach(label, 0, 1, i, i + 1, xpadding = 3, ypadding = 3)
        table.attach(hbox, 1, 2, 0, 1, xpadding = 3, ypadding = 3)
        table.attach(self.checkboxes_check, 1, 2, 1, 2, xpadding = 3, ypadding = 3)
        self.vbox.add(table)
        self.show_all()


    def choose_button_clicked(self, source):
        dlg = gtk.FileChooserDialog(title = _("Select folder"), parent = self, action = gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER,
            buttons = (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT, gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        dlg.set_filename(self.path_entry.get_text())
        if dlg.run() == gtk.RESPONSE_ACCEPT:
            self.path_entry.set_text(dlg.get_filename())
        dlg.destroy()


class ImportWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
             gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.set_title(_("Import"))
        hbox = gtk.HBox()
        self.path_entry = gtk.Entry()
        self.path_entry.set_width_chars(24)
        hbox.pack_start(self.path_entry, expand = True)
        self.choose_button = gtk.ToolButton(stock_id = gtk.STOCK_DIRECTORY)
        self.choose_button.connect("clicked", self.choose_button_clicked)
        hbox.pack_start(self.choose_button, expand = False)
        self.stock_check = gtk.CheckButton()
        self.stock_check.connect("toggled", self.stock_check_changed)
        table = gtk.Table(rows = 2, columns = 2)
        texts = [_("From default reader"), _("OPML file")]
        for i in range(0, len(texts)):
            label = gtk.Label(texts[i])
            label.set_alignment(1, 0.5)
            table.attach(label, 0, 1, i, i + 1, xpadding = 3, ypadding = 3)
        table.attach(self.stock_check, 1, 2, 0, 1, xpadding = 3, ypadding = 3)
        table.attach(hbox, 1, 2, 1, 2, xpadding = 3, ypadding = 3)
        self.vbox.add(table)
        self.show_all()


    def choose_button_clicked(self, source):
        dlg = gtk.FileChooserDialog(title = _("Select opml file"), parent = self, action = gtk.FILE_CHOOSER_ACTION_OPEN,
            buttons = (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT, gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        if self.path_entry.get_text():
            dlg.set_filename(os.path.abspath(self.path_entry.get_text()))
        if dlg.run() == gtk.RESPONSE_ACCEPT:
            self.path_entry.set_text(dlg.get_filename())
        dlg.destroy()


    def stock_check_changed(self, source):
        self.path_entry.set_sensitive(self.stock_check.get_active())
        self.choose_button.set_sensitive(self.stock_check.get_active())


class AddWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
             gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.set_title(_("New feed"))
        hbox = gtk.HBox()
        hbox.set_spacing(5)
        hbox.pack_start(gtk.Label(_("Feed url")), expand = False)
        self.url_entry = gtk.Entry()
        self.url_entry.set_width_chars(32)
        hbox.pack_start(self.url_entry, expand = False)
        self.vbox.add(hbox)
        self.show_all()


class ExtraWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
             gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.set_title(_("Extra feed settings"))
        self.days_editor = hildon.NumberEditor(1, 365)
        self.include_entry = gtk.Entry()
        self.include_entry.set_width_chars(24)
        self.exclude_entry = gtk.Entry()
        table = gtk.Table(rows = 4, columns = 2)
        texts = [_("Days to keep"), _("Include urls"), _("Exclude urls") ]
        for i in range(0, len(texts)):
            label = gtk.Label(texts[i])
            label.set_alignment(1, 0.5)
            table.attach(label, 0, 1, i, i + 1, xpadding = 3, ypadding = 3)
        table.attach(self.days_editor, 1, 2, 0, 1, xpadding = 3, ypadding = 3)
        table.attach(self.include_entry, 1, 2, 1, 2, xpadding = 3, ypadding = 3)
        table.attach(self.exclude_entry, 1, 2, 2, 3, xpadding = 3, ypadding = 3)
        self.vbox.add(table)
        self.show_all()


class ContentWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
             gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.set_title(_("Content settings"))
        self.as_is_button = gtk.CheckButton()
        self.as_is_button.connect("toggled", self.something_changed)
        self.content_tag_entry = gtk.Entry()
        self.content_tag_entry.set_width_chars(24)
        self.content_tag_entry.connect("changed", self.something_changed)
        self.reformatter_url_entry = gtk.Entry()
        self.reformatter_url_entry.connect("changed", self.something_changed)
        self.ignore_print_version_button = gtk.CheckButton()
        self.ignore_print_version_button.connect("toggled", self.something_changed)
        self.print_version_link_entry = gtk.Entry()
        self.print_version_link_entry.connect("changed", self.something_changed)
        self.next_page_link_entry = gtk.Entry()
        self.next_page_link_entry.connect("changed", self.something_changed)
        table = gtk.Table(rows = 6, columns = 2)
        texts = [_("Use as is"), _("Content tag"), _("Reformatting service url"),
            _("Ignore printable version"), _("Printable version link"), _("Next page link") ]
        for i in range(0, len(texts)):
            label = gtk.Label(texts[i])
            label.set_alignment(1, 0.5)
            table.attach(label, 0, 1, i, i + 1, xpadding = 3, ypadding = 3)
        table.attach(self.as_is_button, 1, 2, 0, 1, xpadding = 3, ypadding = 3)
        table.attach(self.content_tag_entry, 1, 2, 1, 2, xpadding = 3, ypadding = 3)
        table.attach(self.reformatter_url_entry, 1, 2, 2, 3, xpadding = 3, ypadding = 3)
        table.attach(self.ignore_print_version_button, 1, 2, 3, 4, xpadding = 3, ypadding = 3)
        table.attach(self.print_version_link_entry, 1, 2, 4, 5, xpadding = 3, ypadding = 3)
        table.attach(self.next_page_link_entry, 1, 2, 5, 6, xpadding = 3, ypadding = 3)
        self.vbox.add(table)
        self.show_all()


    def something_changed(self, source):
        self.ignore_print_version_button.set_sensitive(not self.as_is_button.get_active())
        self.print_version_link_entry.set_sensitive(
            not self.as_is_button.get_active() and
            not self.ignore_print_version_button.get_active() and
            len(self.next_page_link_entry.get_text()) == 0)
        self.content_tag_entry.set_sensitive(
            not self.as_is_button.get_active() and len(self.reformatter_url_entry.get_text()) == 0)
        self.reformatter_url_entry.set_sensitive(
            not self.as_is_button.get_active() and len(self.content_tag_entry.get_text()) == 0)
        self.next_page_link_entry.set_sensitive(
            not self.as_is_button.get_active() and
            len(self.print_version_link_entry.get_text()) == 0)


class EditWindow(gtk.Dialog):


    def __init__(self, parent, feed_config):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
             gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.set_title(_("Edit feed"))
        self.feed_config = feed_config
        self.days_to_keep = feed_config.days_to_keep 
        self.include = feed_config.include
        self.exclude = feed_config.exclude
        self.ignore_print_version = feed_config.ignore_print_version
        self.as_is = feed_config.as_is
        self.content_tag = feed_config.content_tag
        self.print_version_link = feed_config.print_version_link
        self.reformatter_url = feed_config.reformatter_url
        self.next_page_link = feed_config.next_page_link
        hbox = gtk.HBox()
        self.url_entry = gtk.Entry()
        self.url_entry.set_width_chars(32)
        self.url_entry.set_text(feed_config.url)
        self.edit_button = gtk.ToolButton(stock_id = gtk.STOCK_EDIT)
        self.edit_button.connect("clicked", self.edit_button_clicked)
        hbox.pack_start(self.url_entry, expand = True)
        hbox.pack_start(self.edit_button, expand = False)
        self.name_entry = gtk.Entry()
        self.name_entry.set_text(feed_config.name)
        hbox2 = gtk.HBox(spacing = 3)
        self.content_button = gtk.ToolButton(gtk.STOCK_PROPERTIES)
        self.content_button.connect("clicked", self.content_clicked)
        self.mode_combo = gtk.combo_box_new_text()
        self.mode_combo.connect("changed", self.mode_changed)
        self.mode_combo.append_text(_("Do nothing"))
        self.mode_combo.append_text(_("Cache"))
        self.mode_combo.append_text(_("Inline"))
        self.mode_combo.set_active(feed_config.mode)
        hbox2.pack_start(self.mode_combo)
        hbox2.pack_start(self.content_button, expand = False)
        self.extra_button = gtk.Button(_("More"))
        self.extra_button.connect("clicked", self.extra_clicked)
        table = gtk.Table(rows = 4, columns = 2)
        texts = [_("Url"), _("Name"), _("Content")]
        for i in range(0, len(texts)):
            label = gtk.Label(texts[i])
            label.set_alignment(1, 0.5)
            table.attach(label, 0, 1, i, i + 1, xpadding = 3, ypadding = 3)
        table.attach(hbox, 1, 2, 0, 1, xpadding = 3, ypadding = 3)
        table.attach(self.name_entry, 1, 2, 1, 2, xpadding = 3, ypadding = 3)
        table.attach(hbox2, 1, 2, 2, 3, xpadding = 3, ypadding = 3)
        hbox3 = gtk.HBox()
        hbox3.pack_end(self.extra_button, expand = False)
        table.attach(hbox3, 1, 2, 3, 4, xpadding = 3, ypadding = 8)
        self.vbox.add(table)
        self.show_all()


    def mode_changed(self, source):
        self.content_button.set_sensitive(self.mode_combo.get_active() != FeedConfig.mode_ommit)


    def update_feed(self):
        self.feed_config.url = self.url_entry.get_text()
        self.feed_config.name = self.name_entry.get_text()
        self.feed_config.mode = self.mode_combo.get_active()
        self.feed_config.days_to_keep = self.days_to_keep
        self.feed_config.include = self.include
        self.feed_config.exclude = self.exclude
        self.feed_config.as_is = self.as_is
        self.feed_config.ignore_print_version = self.ignore_print_version
        self.feed_config.content_tag = self.content_tag
        self.feed_config.print_version_link = self.print_version_link
        self.feed_config.reformatter_url = self.reformatter_url
        self.feed_config.next_page_link = self.next_page_link


    def extra_clicked(self, source):
        ext_window = ExtraWindow(self)
        ext_window.days_editor.set_value(self.days_to_keep)
        ext_window.include_entry.set_text(self.include)
        ext_window.exclude_entry.set_text(self.exclude)
        if ext_window.run() == gtk.RESPONSE_ACCEPT:
            self.days_to_keep = ext_window.days_editor.get_value()
            self.include = ext_window.include_entry.get_text()
            self.exclude = ext_window.exclude_entry.get_text()
        ext_window.destroy()


    def content_clicked(self, source):
        cnt_window = ContentWindow(self)
        cnt_window.as_is_button.set_active(self.as_is)
        cnt_window.ignore_print_version_button.set_active(self.ignore_print_version)
        cnt_window.content_tag_entry.set_text(self.content_tag)
        cnt_window.reformatter_url_entry.set_text(self.reformatter_url)
        cnt_window.print_version_link_entry.set_text(self.print_version_link)
        cnt_window.next_page_link_entry.set_text(self.next_page_link)
        if cnt_window.run() == gtk.RESPONSE_ACCEPT:
            self.as_is = cnt_window.as_is_button.get_active()
            self.ignore_print_version = cnt_window.ignore_print_version_button.get_active()
            self.content_tag = cnt_window.content_tag_entry.get_text()
            self.reformatter_url = cnt_window.reformatter_url_entry.get_text()
            self.print_version_link = cnt_window.print_version_link_entry.get_text()
            self.next_page_link = cnt_window.next_page_link_entry.get_text()
        cnt_window.destroy()


    def edit_button_clicked(self, source):
        urls_window = UrlsWindow(self)
        tb = urls_window.text_buffer
        tb.set_text(self.url_entry.get_text().replace(" ", "\n"))
        if urls_window.run() == gtk.RESPONSE_ACCEPT:
            text = tb.get_text(tb.get_start_iter(), tb.get_end_iter(), include_hidden_chars = False)
            lines = filter(len, [line.strip().replace(" ", "%20") for line in text.split("\n")])
            self.url_entry.set_text(reduce(lambda s1, s2: s1 + " " + s2, lines))
        urls_window.destroy()


class Program(hildon.Program):


    def __init__(self):
        hildon.Program.__init__(self)

        self.__window = hildon.Window()
        self.__window.set_title("Feed circuit")
        self.__window.connect("destroy", self.stop)
        self.__window.connect("key-press-event", self.on_key_press)
        self.__window.connect("window-state-event", self.on_window_state_change)
        self.__window_in_fullscreen = False
        self.add_window(self.__window)
       
        self.__abort_lock = Lock()
        self.__update_lock = Lock()
        
        self.__model = FeedListModel(create_config_dir())
        if not os.path.exists(self.__model.get_config_file_name()):
            self.import_opml()

        menu = gtk.Menu()
        self.__toolbar = gtk.Toolbar()
        self.__toolbar.set_style(gtk.TOOLBAR_ICONS)
        self.append_action(menu, self.__toolbar, None, gtk.STOCK_REFRESH, self.refresh)
        self.append_action(menu, self.__toolbar, None, gtk.STOCK_ADD, self.add)
        menu.append(gtk.SeparatorMenuItem())
        self.refresh_selected_item = \
            self.append_action(menu, None, _("Refresh selected"), None, self.refresh_current)
        self.append_action(menu, self.__toolbar, None, gtk.STOCK_OPEN, self.open)
        self.append_action(menu, self.__toolbar, None, gtk.STOCK_EDIT, self.edit)
        self.append_action(menu, self.__toolbar, None, gtk.STOCK_DELETE, self.remove)
        menu.append(gtk.SeparatorMenuItem())
        self.append_action(menu, None, _("Import"), None, self.import_feeds)
        self.append_action(menu, self.__toolbar, None, gtk.STOCK_PREFERENCES, self.preferences)
        menu.append(gtk.SeparatorMenuItem())
        self.append_action(menu, None, None, gtk.STOCK_ABOUT, self.about)
        self.append_action(menu, None, None, gtk.STOCK_QUIT, self.stop)
        #self.__progress_label = gtk.Label()
        #self.__progress_label.set_alignment(0, 0.5)
        #self.__progress_label.set_ellipsize(pango.ELLIPSIZE_MIDDLE)
        #status_item = gtk.ToolItem()
        #status_item.add(self.__progress_label)
        #status_item.set_expand(True)
        #self.__toolbar.insert(status_item, -1)
        self.__abort_button = gtk.ToolButton(gtk.STOCK_STOP)
        self.__abort_button.connect("clicked", self.abort)
        self.__abort_button.set_no_show_all(True)
        self.__toolbar.insert(self.__abort_button, -1)

        self.__window.set_menu(menu)
        
        self.refresh_selected_item.set_property("visible",
            not self.__model.use_checkboxes)

        self.__progress_banner = None

        self.__view = gtk.TreeView(self.__model)
        self.__view.set_rubber_banding(True)
        self.__view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)

        self.check_col = gtk.TreeViewColumn()
        render_toggle = gtk.CellRendererToggle()
        render_toggle.set_property("activatable", True)
        render_toggle.connect("toggled", self.feed_toggled)
        self.check_col.pack_start(render_toggle, expand = False)
        self.check_col.add_attribute(render_toggle, "active", 3)
        self.__view.append_column(self.check_col)
        self.check_col.set_visible(self.__model.use_checkboxes)

        col = gtk.TreeViewColumn()
        render_pixbuf = gtk.CellRendererPixbuf()
        col.pack_start(render_pixbuf, expand=False)
        col.add_attribute(render_pixbuf, 'pixbuf', 0)
        render_text = gtk.CellRendererText()
        render_text.set_property("ellipsize", pango.ELLIPSIZE_MIDDLE)
        col.pack_start(render_text, expand=True)
        col.add_attribute(render_text, 'markup', 1)
        col.set_expand(True)
        self.__view.append_column(col)
        
        col = gtk.TreeViewColumn()
        render_text = gtk.CellRendererText()
        col.pack_start(render_text, expand=False)
        col.add_attribute(render_text, 'text', 2)
        self.__view.append_column(col)

        self.__view.connect("row_activated", self.row_activated)

        dnd_targets = [("FeedCircuitFeed", gtk.TARGET_SAME_WIDGET, 0)]
        self.__view.enable_model_drag_source(gtk.gdk.BUTTON1_MASK, dnd_targets, gtk.gdk.ACTION_MOVE)
        self.__view.drag_source_set(gtk.gdk.BUTTON1_MASK, dnd_targets, gtk.gdk.ACTION_MOVE)
        self.__view.enable_model_drag_dest(dnd_targets, gtk.gdk.ACTION_MOVE)
        self.__view.connect("drag-data-get", self.drag_data_get)
        self.__view.connect("drag-data-received", self.drag_data_received)
        self.__view.connect("drag-drop", self.drag_drop)


        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.add(self.__view)
        self.__window.add(sw)
        self.__window.add_toolbar(self.__toolbar)


    def append_action(self, menu, toolbar, title, stock, callback):
        if stock:
            item = gtk.ImageMenuItem(stock)
        else:
            item = gtk.MenuItem(title)
        item.connect("activate", callback)
        menu.append(item)
        if toolbar:
            button = gtk.ToolButton(stock)
            button.connect("clicked", callback)
            toolbar.insert(button, -1)
        return item


    def on_key_press(self, widget, event, *args):
        if event.keyval == gtk.keysyms.F6:
            # The "Full screen" hardware key has been pressed
            if self.__window_in_fullscreen:
                self.__window.unfullscreen ()
            else:
                self.__window.fullscreen ()


    def on_window_state_change(self, widget, event, *args):
        if event.new_window_state & gtk.gdk.WINDOW_STATE_FULLSCREEN:
            self.__window_in_fullscreen = True
        else:
            self.__window_in_fullscreen = False


    def drag_drop(self, widget, drag_context, x, y, timestamp):
        self.__view.emit_stop_by_name('drag-drop')
        self.__view.drag_get_data(drag_context, drag_context.targets[0], timestamp)
        return 1


    def drag_data_get(self, widget, drag_context, selection_data, info, timestamp):
        feed_config = self.get_current_feed()
        if feed_config:
            selection_data.set(selection_data.target, len(feed_config.id), feed_config.id)


    def drag_data_received(self, widget, drag_context, x, y, selection_data, info, timestamp):
        self.__view.emit_stop_by_name('drag-data-received')
        pos = None
        row_data = self.__view.get_dest_row_at_pos(x, y)
        if row_data:
            pos = row_data[0][0]
            if row_data[1] == gtk.TREE_VIEW_DROP_AFTER:
                pos += 1
        self.__model.move(selection_data.data, pos)


    def feed_toggled(self, cell, path):
        self.__model.toggle((int(path),))


    def run(self):
        self.__window.show_all()
        gtk.main()


    def stop(self, source):
        gtk.main_quit()


    def open_feed(self, feed_config):
        path = os.path.join(self.__model.cache_path, feed_config.id + ".html")
        if os.path.exists(path):
            webbrowser.open(path)
            feed_config.message_count = 0
            self.__model.update(feed_config.id)
        else:
            msg_dlg = gtk.MessageDialog(parent = self.__window, flags = gtk.DIALOG_MODAL,
                type=gtk.MESSAGE_WARNING, buttons = gtk.BUTTONS_OK, 
                message_format = _("The feed is empty"))
            msg_dlg.run()
            msg_dlg.destroy()


    def get_selected_feeds(self):
        if self.__model.use_checkboxes:
            return filter(lambda fc: fc.checked, self.__model.list)
        else:
            pathes = self.__view.get_selection().get_selected_rows()[1]
            return map(lambda path: self.__model.list[path[0]], pathes)


    def get_current_feed(self):
        pathes = self.__view.get_selection().get_selected_rows()[1]
        if len(pathes):
            return self.__model.list[pathes[0][0]]


    def open(self, source):
        feed_config = self.get_current_feed()
        if feed_config:
            self.open_feed(feed_config)


    def row_activated(self, source, path, view_column):
        self.open_feed(self.__model.list[path[0]])


    def feed_updated(self, feed_config, new_item_count):
        feed_config.message_count += new_item_count
        self.__model.update(feed_config.id)


    def abort(self, source):
        self.__abort_lock.acquire()
        self.__aborted = True
        self.__abort_lock.release()
        if self.__thread:
            self.__thread.join()


    def update_started(self):
        self.__abort_button.show()


    def update_complete(self):
        #self.update_status("")
        if self.__progress_banner:
            self.__progress_banner.destroy()
            self.__progress_banner = None
        self.__abort_button.hide()


    def update_status(self, text):
        if self.__progress_banner:
            self.__progress_banner.set_text(text)
        else:
            self.__progress_banner = \
                hildon.hildon_banner_show_animation(self.__window, "qgn_indi_pball_a", text)
        #self.__progress_label.set_text(text)


    def callback(self, item):
        self.__abort_lock.acquire()
        abort = self.__aborted
        self.__aborted = False
        self.__abort_lock.release()
        if abort:
            sys.exit()
        text = "(" + str(self.__progress_info[0]) + "/" + str(self.__progress_info[1]) + ") " + \
            _(str(item[0])) + " " + str(item[1])
        gobject.idle_add(self.update_status, text)


    def update_feeds(self, feed_configs):
        self.__update_lock.acquire()
        self.__abort_lock.acquire()
        self.__aborted = False
        self.__abort_lock.release()
        gobject.idle_add(self.update_started)
        try:
            i = 0
            for feed_config in feed_configs:
                i += 1
                self.__progress_info = (i, len(feed_configs))
                urls = feed_config.get_urls()
                new_item_count = 0
                title = None
                if len(urls) == 1:
                    title = feed_config.name
                for url in urls:
                    feed = feedcircuit.Feed(url = url, path = self.__model.cache_path, cache_path = ".cache",
                        title = title, filename = feed_config.id + ".html",
                        callback = self.callback, try_use_print_version = not feed_config.ignore_print_version,
                        delete_before_date = date.today() - timedelta(feed_config.days_to_keep),
                        inline_items = feed_config.mode == FeedConfig.mode_inline,
                        cache_items = feed_config.mode == FeedConfig.mode_cache,
                        include = feed_config.include, exclude = feed_config.exclude,
                        as_is = feed_config.as_is, content_kwargs = feed_config.get_content_kwargs(),
                        print_version_keywords = feed_config.get_print_version_keywords(),
                        reformatter_url = feed_config.reformatter_url, next_page_keyword = feed_config.next_page_link)
                    new_item_count += feed.update()
                gobject.idle_add(self.feed_updated, feed_config, new_item_count)
        finally:
            gobject.idle_add(self.update_complete)
            self.__update_lock.release()


    def update_feeds_async(self, feeds_configs):
        self.__thread = Thread(target = self.update_feeds, args = (feeds_configs,))
        self.__thread.start()


    def refresh_current(self, source):
        connect_and_exec(self.update_feeds_async, self.get_selected_feeds())


    def refresh(self, source):
        feedlist = None
        if self.__model.use_checkboxes:
            feedlist = self.get_selected_feeds()
        if not feedlist:
            feedlist = list(self.__model.list)
        connect_and_exec(self.update_feeds_async, feedlist)


    def edit(self, source):
        feed_config = self.get_current_feed()
        if feed_config:
            edit_window = EditWindow(self.__window, feed_config)
            if edit_window.run() == gtk.RESPONSE_ACCEPT:
                edit_window.update_feed()
                self.__model.update(feed_config.id)
            edit_window.destroy()


    def add(self, source):
        add_window = AddWindow(self.__window)
        response = add_window.run()
        add_window.hide()
        if response == gtk.RESPONSE_ACCEPT:
            url = add_window.url_entry.get_text()
            title = ""
            icon_filename = ""
            try:
                discovery_thread = BannerThread(self.__window, _("Discovering feed"))
                (url, title, icon_filename, icon_data) = \
                    discovery_thread.execute(feedcircuit.discover_feed, url)
                if icon_filename:
                    icon_filename = uuid.uuid1().hex + icon_filename
                    save_icon(os.path.join(self.__model.cache_path, icon_filename), icon_data)
            except Exception, ex:
                msg_dlg = gtk.MessageDialog(parent = self.__window, flags = gtk.DIALOG_MODAL,
                    type=gtk.MESSAGE_WARNING, buttons = gtk.BUTTONS_OK, 
                    message_format = _("Cannot discover feed at given url") + "\n" + str(ex))
                msg_dlg.run()
                msg_dlg.destroy()
            feed_config = FeedConfig(self.__model.cache_path, url, title, FeedConfig.mode_ommit, 7, "", "", False, icon_filename, 0,
                False, "", "", "", "")
            edit_window = EditWindow(self.__window, feed_config)
            if edit_window.run() == gtk.RESPONSE_ACCEPT:
                edit_window.update_feed()
                self.__model.append(feed_config)
            elif icon_filename:
                os.remove(os.path.join(self.__model.cache_path, icon_filename))
            edit_window.destroy()
        add_window.destroy()



    def remove(self, source):
        selected = list(self.get_selected_feeds())
        if len(selected):
            dialog = gtk.MessageDialog(self.__window, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                gtk.MESSAGE_QUESTION, gtk.BUTTONS_YES_NO, _("Are you sure you want to delete selected feeds?"))
            hbox = gtk.HBox()
            button = gtk.CheckButton()
            hbox.pack_start(button, expand = False)
            hbox.pack_start(gtk.Label(_("Clear cache only")), expand = False)
            hbox2 = gtk.HBox()
            hbox2.pack_start(hbox, expand = True, fill = False)
            hbox2.show_all()
            dialog.vbox.pack_start(hbox2)
            response = dialog.run()
            only_cache = button.get_active()
            dialog.destroy()
            if response == gtk.RESPONSE_YES:
                for feed_config in selected:
                    feed_config.message_count = 0
                    feed = feedcircuit.Feed(None, path = self.__model.cache_path,
                        cache_path = ".cache", filename = feed_config.id + ".html")
                    feed.delete()
                    if not only_cache:
                        if feed_config.icon_filename:
                            icon_filepath = os.path.join(self.__model.cache_path, feed_config.icon_filename)
                            if os.path.exists(icon_filepath):
                                os.remove(icon_filepath)
                        self.__model.delete(feed_config.id)


    def move_cache(self, from_path, to_path):
        if not os.path.exists(to_path):
            os.makedirs(to_path)
        for file in glob.glob(os.path.join(from_path, "*")):
            filename = os.path.split(file)[1]
            if filename != "config":
                shutil.move(file, os.path.join(to_path, filename))
        shutil.move(os.path.join(from_path, ".cache"), os.path.join(to_path, ".cache"))


    def import_opml(self, filename = None):
        fav_dir = None
        if not filename:
            filename = os.path.expanduser("~/.osso_rss_feed_reader/feedlist.opml")
            fav_dir = os.path.expanduser("~/.osso_rss_feed_reader/cache/favicons")
        f = open(filename, "r")
        try:
            opml = BeautifulStoneSoup(f)
            outlines = filter(
                lambda o: o.has_key("title") and o.has_key("xmlurl"),
                opml.findAll("outline"))
            for outline in outlines:
                icon_file_name = ""
                if fav_dir:
                    if outline.has_key("id"):
                        icon_file = outline["id"] + ".png"
                        icon_file_src = os.path.join(fav_dir, icon_file)
                        if os.path.exists(icon_file_src):
                            icon_file_name = uuid.uuid1().hex + icon_file
                            shutil.copy(icon_file_src,
                                os.path.join(self.__model.cache_path, icon_file_name))
                else:
                    try:
                        discovery_thread = BannerThread(self.__window, _("Importing feeds"))
                        (url, title, icon_file_name, icon_data) = \
                            discovery_thread.execute(feedcircuit.get_feed_details, outline["xmlurl"])
                        if icon_file_name:
                            icon_file_name = uuid.uuid1().hex + icon_file_name
                            save_icon(os.path.join(self.__model.cache_path, icon_file_name), icon_data)
                    except Exception, ex:
                        msg_dlg = gtk.MessageDialog(parent = self.__window, flags = gtk.DIALOG_MODAL,
                            type=gtk.MESSAGE_WARNING, buttons = gtk.BUTTONS_OK, 
                            message_format = _("Cannot discover feed at given url") + "\n" + str(ex))
                        msg_dlg.run()
                        msg_dlg.destroy()
                feed_config = FeedConfig(self.__model.cache_path, outline["xmlurl"], outline["title"],
                    FeedConfig.mode_ommit, 7, "", "", False, icon_file_name, 0,
                    False, "", "", "", "")
                self.__model.append(feed_config)
        finally:
            f.close()


    def import_feeds(self, source):
        imp_window = ImportWindow(self.__window)
        try:
            result = imp_window.run()
            imp_window.hide()
            if result == gtk.RESPONSE_ACCEPT:
                filename = imp_window.path_entry.get_text()
                if imp_window.stock_check.get_active():
                    self.import_opml()
                elif filename:
                    self.import_opml(filename)
        except Exception, ex:
            msg_dlg = gtk.MessageDialog(parent = self.__window, flags = gtk.DIALOG_MODAL,
                type=gtk.MESSAGE_ERROR, buttons = gtk.BUTTONS_OK, 
                message_format = str(ex))
            msg_dlg.run()
            msg_dlg.destroy()
        finally:
            imp_window.destroy()


    def preferences(self, source):
        pref_window = PreferencesWindow(self.__window)
        try:
            pref_window.path_entry.set_text(self.__model.cache_path)
            pref_window.checkboxes_check.set_active(self.__model.use_checkboxes)
            if pref_window.run() == gtk.RESPONSE_ACCEPT:
                new_path = os.path.abspath(pref_window.path_entry.get_text())
                if new_path != self.__model.cache_path:
                    self.move_cache(self.__model.cache_path, new_path)
                    self.__model.update_cache_path(new_path)
                self.__model.set_checkboxes(pref_window.checkboxes_check.get_active())
                self.check_col.set_visible(self.__model.use_checkboxes)
                self.refresh_selected_item.set_property("visible", 
                    not self.__model.use_checkboxes)
                self.__model.save_config()
        except Exception, ex:
            msg_dlg = gtk.MessageDialog(parent = self.__window, flags = gtk.DIALOG_MODAL,
                type=gtk.MESSAGE_ERROR, buttons = gtk.BUTTONS_OK, 
                message_format = str(ex))
            msg_dlg.run()
            msg_dlg.destroy()
        finally:
            pref_window.destroy()


    def about(self, source):
        dialog = gtk.AboutDialog()
        dialog.set_name("FeedCircuit")
        dialog.set_version("0.6.9")
        dialog.set_website("http://feedcircuit.garage.maemo.org")
        dialog.set_authors([_("George Kibardin")])
        dialog.run()
        dialog.destroy()
