#!/usr/bin/python

import ConfigParser
import os.path
import os
import string
from utils import *

CONFIG_FILE = os.path.expanduser("~/.erminig.cfg")

class ErminigCfg:

    def __init__(self):
        self.config = ConfigParser.ConfigParser()
        self.config.read(CONFIG_FILE)

        self.http_proxy = ""
        self.https_proxy = ""
        self.user_proxy = ""
        self.password_proxy = ""

        self.google_login = ""
        self.google_password = ""

        self.profiles = []

        # runtime options (not saved in config file)
        self.proxy_status = False
        self.confirm_gdeletions = False
        self.debug = False

        self.load_configuration()

    def get_debug_status(self):
        return self.debug

    def set_debug_status(self, state):
        self.debug = state

    def flip_debug_status(self):
        self.debug = not self.debug

    def get_proxy_status(self):
        return self.proxy_status

    def set_proxy_status(self, state):
        self.proxy_status = state

    def flip_proxy_status(self):
        self.proxy_status = not self.proxy_status

    def get_confirm_gdeletions(self):
        return self.confirm_gdeletions

    def set_confirm_gdeletions(self, state):
        self.confirm_gdeletions = state

    def read_cfg_entry(self, section, option, default=""):
        value = default
        try:
            value = self.config.get(section, option)
        except:
            value = default

        return value

    def read_cfg_entry_int(self, section, option, default=0):
        value = default
        try:
            value = self.config.getint(section, option)
        except:
            value = default

        return value

    def read_cfg_entry_bool(self, section, option, default=True):
        value = default
        try:
            value = self.config.getboolean(section, option)
        except:
            value = default

        return value

    def add_cfg_section(self, section):
        if not self.config.has_section(section):
            self.config.add_section(section)

    def set_cfg_value(self, section, key, value):
        self.config.set(section, key, value)

    def remove_cfg_section(self, section):
        self.config.remove_section(section)

    def write_config(self):
        try:
            self.config.write(open(CONFIG_FILE, 'w'))
        except:
            return -1

    def load_configuration(self):
        self.load_google_configuration()
        self.load_proxy_configuration()
        self.load_profiles()

    def load_google_configuration(self):
        printd("Loading Google parameters")
        self.google_login = self.read_cfg_entry('Google', 'GoogleLogin')
        self.google_password = self.read_cfg_entry('Google', 'GooglePassword')
        
    def load_proxy_configuration(self):
        printd("Loading proxy parameters")
        self.http_proxy = self.read_cfg_entry('Proxy', 'http')
        self.https_proxy = self.read_cfg_entry('Proxy', 'http')
        self.user_proxy = self.read_cfg_entry('Proxy', 'user')
        self.password_proxy = self.read_cfg_entry('Proxy', 'password')

    def load_profiles(self):
        self.profiles = []
        printd("Loading profiles from config file")
        profilesList = self.read_cfg_entry('Profiles', 'List')
        printd("Profiles list : " + str(profilesList))
        if profilesList == "":
            self.profiles = []
        else:
            self.profiles = string.split(profilesList,';')

    def delete_profile(self, profile_name):

        profiles = self.read_cfg_entry('Profiles', 'List', [])
        profiles_list = string.split(profiles,';')
        profiles_list.remove(profile_name)
        self.profiles = profiles_list

        self.set_cfg_value('Profiles', 'List', ";".join(profiles_list))
        self.remove_cfg_section(profile_name)

        self.write_config()

    def get_proxy_params(self):
        proxy_params = {}
        proxy_params['http_proxy'] = self.http_proxy
        proxy_params['https_proxy'] = self.https_proxy
        proxy_params['user_proxy'] = self.user_proxy
        proxy_params['password_proxy'] = self.password_proxy

        return proxy_params

    def get_google_login(self):
        return self.google_login

    def get_google_password(self):
        return self.google_password

    def get_profile(self, profile_name):
        profile = {}

        profile['name'] = profile_name
        profile['gpecalendarid'] = self.read_cfg_entry(profile_name, 
                'gpecalendarid')
        profile['gpecalendartitle'] = self.read_cfg_entry(profile_name, 
                'gpecalendartitle')
        profile['timezone_name'] = self.read_cfg_entry(profile_name, 
                'Timezone_Name')
        profile['googlecalendarid'] = self.read_cfg_entry(profile_name, 
                'googlecalendarid')
        profile['googlecalendartitle'] = self.read_cfg_entry(profile_name, 
                'googlecalendartitle')
        profile['gpeupdate'] = str(self.read_cfg_entry_int(profile_name, 
            'gpeupdate'))
        profile['googleupdate'] = self.read_cfg_entry(profile_name, 
                'googleupdate', "1970-01-01T01:01:01.000Z")
        profile['enabled'] = self.read_cfg_entry_bool(profile_name, 
                'enabled', False)
        profile['startdate'] = self.read_cfg_entry(profile_name, 
                'startdate', "1970-01-01")
        profile['accesslevel'] = self.read_cfg_entry(profile_name, 'accesslevel')

        return profile

    def get_profile_info(self, profile_name):
        profile = {}
        profile['name'] = profile_name
        profile['enabled'] = self.read_cfg_entry_bool(profile_name, 'enabled',
                False)
        profile['gpeupdate'] = self.read_cfg_entry(profile_name, 'gpeupdate')

        return profile


    def write_profile(self, profile):
        #find an existing profile
        profiles_list = self.profiles
        profilename = profile['name']
        
        if profilename not in profiles_list:
            #Profile not found, we increment the profiles
            #counter and add the profile name to the list

            profiles_list.append(profilename)

            self.add_cfg_section('Profiles')

            self.set_cfg_value('Profiles', 'List', ";".join(profiles_list))

            #Now we write the profile
            #The profile does not exist, so we create a new section
            self.add_cfg_section(profilename)
        
        self.set_cfg_value(profilename, 'GoogleCalendarTitle',
                profile['googlecalendartitle'])
        self.set_cfg_value(profilename, 'GoogleCalendarId', 
                profile['googlecalendarid'])
        self.set_cfg_value(profilename, 'GpeCalendarTitle', 
                profile['gpecalendartitle'])
        self.set_cfg_value(profilename, 'GpeCalendarId', 
                profile['gpecalendarid'])
        self.set_cfg_value(profilename, 'Timezone_Name', 
                profile['timezone_name'])
        self.set_cfg_value(profilename, 'gpeupdate', profile['gpeupdate'])
        self.set_cfg_value(profilename, 'googleupdate', profile['googleupdate'])
        self.set_cfg_value(profilename, 'enabled', repr(profile['enabled']))
        self.set_cfg_value(profilename, 'startdate', profile['startdate'])
        self.set_cfg_value(profilename, 'accesslevel', profile['accesslevel'])

        # flush config to file:
        self.write_config()

    def update_lastupdate(self, profilename, gpe_update, google_update):
        self.set_cfg_value(profilename, 'gpeupdate', gpe_update)
        self.set_cfg_value(profilename, 'googleupdate', google_update)

        # flush config to file:
        self.write_config()

    def get_google_params(self):
        google_params = {}
        google_params['user'] = self.google_login
        google_params['password'] = self.google_password

        return google_params

    def write_google_parameters(self, google_params):
        self.google_login = google_params['user']
        self.google_password = google_params['password']

        self.add_cfg_section('Google')
        self.set_cfg_value('Google', 'GoogleLogin', self.google_login)
        self.set_cfg_value('Google', 'GooglePassword', self.google_password)

        # flush config to file:
        self.write_config()
        
    def write_proxy_parameters(self, proxy_params):
        self.http_proxy = proxy_params['http_proxy']
        self.https_proxy = proxy_params['https_proxy']
        self.user_proxy = proxy_params['user_proxy']
        self.password_proxy = proxy_params['password_proxy']

        self.add_cfg_section('Proxy')
        self.set_cfg_value('Proxy', 'http', self.http_proxy)
        self.set_cfg_value('Proxy', 'https', self.https_proxy)
        self.set_cfg_value('Proxy', 'user', self.user_proxy)
        self.set_cfg_value('Proxy', 'password', self.password_proxy)

        # flush config to file:
        self.write_config()
