/*
  DiceJinni - A dice simulator
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "DiceEdit.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/Value.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/String.h>
#include <Lum/Table.h>

DiceEdit::DiceEdit(Dice& dice)
 : dice(dice),
   valuesModel(new ValuesModel(dice.values)),
   selection(new Lum::Model::SingleLineSelection()),
   addAction(new Lum::Model::Action()),
   editAction(new Lum::Model::Action()),
   removeAction(new Lum::Model::Action())
{
  editAction->Disable();
  removeAction->Disable();

  Observe(selection);

  Observe(addAction);
  Observe(editAction);
  Observe(removeAction);

  Observe(GetClosedAction());
}

void DiceEdit::PreInit()
{
  Lum::Panel            *horiz;
  Lum::Table            *table;
  Lum::Model::HeaderRef headerModel;

  horiz=Lum::HPanel::Create(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(L"Value",Lum::Base::Size::stdCharWidth,30,true);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetHeight(Lum::Base::Size::stdCharHeight,8);
  table->SetShowHeader(true);
  table->GetTableView()->SetAutoHSize(true);
  table->GetTableView()->SetAutoFitColumns(true);
  table->SetModel(valuesModel);
  table->SetHeaderModel(headerModel);
  table->SetSelection(selection);
  table->SetDoubleClickAction(editAction);
  horiz->Add(table);

  horiz->AddSpace();

  horiz->Add(Lum::VPanel::Create(false,true)
             ->Add(Lum::Button::Create(_(L"BTN_ADD",L"_Add"),addAction,true,false))
             ->AddSpace()
             ->Add(Lum::Button::Create(_(L"BTN_UPDATE",L"_Edit"),editAction,true,false))
             ->AddSpace()
             ->Add(Lum::Button::Create(_(L"BTN_REMOVE",L"_Remove"),removeAction,true,false)));

  SetMain(Lum::VPanel::Create(true,true)
          ->Add(horiz)
          ->AddSpace()
          ->Add(Lum::ButtonRow::CreateOk(GetClosedAction(),true,false)));

  Dialog::PreInit();
}

void DiceEdit::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    Exit();
  }
  else if (model==selection && dynamic_cast<const Lum::Model::Selection::Selected*>(&msg)!=NULL) {
    if (selection->HasSelection()) {
      editAction->Enable();
      removeAction->Enable();
    }
    else {
      editAction->Disable();
      removeAction->Disable();
    }
  }
  else if (model==addAction && addAction->IsFinished()) {
    bool                  result;
    Lum::Model::StringRef value(new Lum::Model::String(L""));

    result=Lum::Dlg::TextInput::GetText(GetWindow(),
                                        L"Enter dice value",
                                        L"Please enter the value for this\n"
                                        L"side of the current dice:",
                                        value);

    if (result) {
      Value v;

      v.value=*value;
      valuesModel->Append(v);
    }

    configurationChanged=true;
  }
  else if (model==editAction && editAction->IsFinished()) {
    assert(selection->HasSelection());

    bool                  result;
    Lum::Model::StringRef value(new Lum::Model::String(valuesModel->GetEntry(selection->GetLine()).value));

    result=Lum::Dlg::TextInput::GetText(GetWindow(),
                                        L"Enter dice value",
                                        L"Please enter the value for this\n"
                                        L"side of the current dice:",
                                        value);

    if (result) {
      valuesModel->GetEntry(selection->GetLine()).value=*value;
      valuesModel->RedrawRow(selection->GetLine());
    }

    configurationChanged=true;
  }
  else if (model==removeAction && removeAction->IsFinished()) {
    assert(selection->HasSelection());

    valuesModel->Delete(selection->GetLine());
    // Could be the currently displayed one
    setsModel->Notify();

    configurationChanged=true;
  }

  Dialog::Resync(model,msg);
}

