/*
  DiceJinni - A dice simulator
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "DiceSetEdit.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/String.h>
#include <Lum/Table.h>

#include "DiceEdit.h"

DiceSetEdit::DiceSetEdit(DiceSet& diceSet)
 : diceSet(diceSet),
   name(new Lum::Model::String(diceSet.name)),
   dicesModel(new DicesModel(diceSet.dices)),
   selection(new Lum::Model::SingleLineSelection()),
   addAction(new Lum::Model::Action()),
   editAction(new Lum::Model::Action()),
   removeAction(new Lum::Model::Action()),
   okAction(new Lum::Model::Action())
{
  editAction->Disable();
  removeAction->Disable();

  Observe(selection);

  Observe(addAction);
  Observe(editAction);
  Observe(removeAction);

  Observe(okAction);
  Observe(GetClosedAction());
}

void DiceSetEdit::PreInit()
{
  Lum::Label            *label;
  Lum::Panel            *panel,*horiz;
  Lum::Table            *table;
  Lum::Model::HeaderRef headerModel;

  panel=Lum::VPanel::Create();

  label=Lum::Label::Create(true,true);
  label->AddLabel(L"Name:",Lum::String::Create(name,true,false));

  horiz=Lum::HPanel::Create(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(L"Values",Lum::Base::Size::stdCharWidth,30,true);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetShowHeader(true);
  table->GetTableView()->SetAutoHSize(true);
  table->GetTableView()->SetAutoFitColumns(true);
  table->SetModel(dicesModel);
  table->SetHeaderModel(headerModel);
  table->SetSelection(selection);
  table->SetDoubleClickAction(editAction);
  horiz->Add(table);

  horiz->AddSpace();
  horiz->Add(Lum::VPanel::Create(false,true)
             ->Add(Lum::Button::Create(_(L"BTN_ADD",L"_Add"),addAction,true,false))
             ->Add(Lum::Button::Create(_(L"BTN_UPDATE",L"_Edit"),editAction,true,false))
             ->Add(Lum::Button::Create(_(L"BTN_REMOVE",L"_Remove"),removeAction,true,false)));

  label->AddLabel(L"Dices:",horiz);

  panel->Add(label);
  panel->AddSpace();
  panel->Add(Lum::ButtonRow::CreateOkCancel(okAction,GetClosedAction(),true,false));

  SetMain(panel);

  Dialog::PreInit();
}

void DiceSetEdit::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==okAction && okAction->IsFinished()) {
    // Copy back values
    diceSet.name=*name;

    result=true;
    Exit();
  }
  else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    result=false;
    Exit();
  }
  else if (model==selection && dynamic_cast<const Lum::Model::Selection::Selected*>(&msg)!=NULL) {
    if (selection->HasSelection()) {
      editAction->Enable();
      removeAction->Enable();
    }
    else {
      editAction->Disable();
      removeAction->Disable();
    }
  }
  else if (model==addAction && addAction->IsFinished()) {
    DiceEdit *dialog;

    Dice     dice;

    dialog=new DiceEdit(dice);
    dialog->SetParent(this);
    if (dialog->Open()) {
      dialog->EventLoop();
      dialog->Close();

      // Redraw entry because it name could have changed...
      dicesModel->Append(dice);
    }

    delete dialog;

    configurationChanged=true;
  }
  else if (model==editAction && editAction->IsFinished()) {
    assert(selection->HasSelection());

    DiceEdit *dialog;

    dialog=new DiceEdit(dicesModel->GetEntry(selection->GetLine()));
    dialog->SetParent(this);
    if (dialog->Open()) {
      dialog->EventLoop();
      dialog->Close();

      // Redraw entry because it name could have changed...
      dicesModel->RedrawRow(selection->GetLine());
    }

    delete dialog;

    configurationChanged=true;
  }
  else if (model==removeAction && removeAction->IsFinished()) {
    assert(selection->HasSelection());

    dicesModel->Delete(selection->GetLine());
    // Could be the currently displayed one
    setsModel->Notify();

    configurationChanged=true;
  }

  Dialog::Resync(model,msg);
}

bool DiceSetEdit::GetResult() const
{
  return result;
}

