#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys, time, os
from PyQt4 import QtCore
from PyQt4 import QtGui
from PyQt4.QtCore import Qt
from PyQt4.QtGui import QApplication
try:
    from PyQt4.QtMaemo5 import QMaemo5InformationBox
except:
    pass


greekSymbols = ["alpha", "beta", "chi", "delta", "Delta", "epsilon", "eta", "gamma", \
"Gamma", "iota", "kappa", "nu", "omega", "Omega", "omicron", "phi", "Phi", "pi", "Pi", \
"psi", "Psi", "rho", "sigma", "Sigma", "tau", "theta", "upsilon", "xi", "Xi"]

sympy_symbols_functions = ["acos()", "acosh()", "acot()", "acoth()", "asin()", \
"asinh()", "atan()", "atanh()", "cos()", "cosh()", "cot()", "coth()", "DiracDelta()", \
"E", "EulerGamma", "exp()", "factorial()", "fibonacci()", "gamma()", "GoldenRatio", \
"harmonic()", "hermite(n,x)", "I", "im()", "legendre(n,x)", "ln()", "ln(n,b)", "log()", \
"log(n,b)", "oo", "re()", "sin()", "sinh()", "sqrt()", "tan()", "tanh()", "zeta()"]

historyExpressionSamples = ["1/x**n", "atan(x)", "a*x**2+b*x+c", "cos(2*x**2)**4", \
"exp(-(x-x_0)**2/(2*sigma**2))", "f(x)/g(x)", "log(x)", "sin(k*x-omega*t+phi)", \
"sin(x*y*z)", "sqrt(2*x**3-4*x+5)", "sqrt(x/(x**3+1))", "G*M/r", "G*M/r**2", \
"q/(4*pi*epsilon*r)", "q/(4*pi*epsilon*r**2)", "f(x,y,z)", "f(rho,theta,z)", \
"f(r,theta,phi)"]

derivativeType = {'derivative':1, 'gradient':2, 'divergence':3, 'curl':4, 'laplacian':5}

coordSysNumber = {'cartesian3D':0, 'cylindrical':1, 'spherical':2}
coordSysType = {0:'cartesian3D', 1:'cylindrical', 2:'spherical'}
varCoordSys = {'cartesian3D':['x','y','z'], 'cylindrical':['rho','theta','z'], \
               'spherical':['r','theta','phi']}
scaleFactorCoordSys = {'cartesian3D':['1','1','1'], 'cylindrical':['1','rho','1'], \
                       'spherical':['1','r','r*sin(theta)']}

simplifyType = {'none':0, 'expandterms':1, 'simplifyterms':2, 'expandall':3, 'simplifyall':4}
outputType = {'simple':0, 'bidimensional':1, 'typesetting':2, 'latex':3, 'mathml':4, \
              'c':5, 'fortran':6}
derivativeErrorMessage = u'Error: derivative not calculated'
gradientErrorMessage = u'Error: gradient not calculated'
divergenceErrorMessage = u'Error: divergence not calculated'
curlErrorMessage = u'Error: curl not calculated'
laplacianErrorMessage = u'Error: laplacian not calculated'

zoomFontSize = (10, 12, 14, 16, 20, 24, 28, 32)
zoomNumColumnsMaemo4 = (84, 68, 62, 52, 43, 36, 32, 27)
zoomNumColumnsMaemo5 = (97, 77, 70, 59, 49, 41, 36, 30)

maemo5InformationBoxTime = 3000


class DerivativeWindow(QtGui.QMainWindow):
    def __init__(self, parent=None):
        QtGui.QMainWindow.__init__(self, parent)
        self.setWindowTitle("Derivative")

        self.window = QtGui.QWidget()
        self.setCentralWidget(self.window)

        if os.path.exists("/proc/component_version"):
            self.isMaemo = True
            try:
                self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
                self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
                self.zoomNumColumns = zoomNumColumnsMaemo5
                self.isMaemo5 = True
            except:
                self.isMaemo5 = False
                self.zoomNumColumns = zoomNumColumnsMaemo4
        else:
            self.zoomNumColumns = zoomNumColumnsMaemo5
            self.isMaemo = False
            self.isMaemo5 = False

        if not self.isMaemo:
            self.resize(800, 600)

        self.settings = QtCore.QSettings()
        self.loadSettings()
        # Setting the pretty printing of SymPy to use unicode characters and correct number of columns.
        init_printing(use_unicode=True, num_columns=self.zoomNumColumns[self.zoomLevel])

        self.historyInitExpression = QtCore.QStringList()
        for i in greekSymbols+sympy_symbols_functions+historyExpressionSamples:
            self.historyInitExpression.append(QtCore.QString(i))
        self.historyExpression=self.historyInitExpression+self.historyUserExpression

        self.historyVariable = QtCore.QStringList()
        for i in greekSymbols:
            self.historyVariable.append(QtCore.QString(i))

        self.nonCalculatedDerivative = u""
        self.resultDerivative = u""
        self.resultDerivativeSimp = u""
        self.timeDerivative = 0.0
        self.configUpdateResult = False
        self.createMenuAndActions()
        self.createItems()
        self.showHideItems()
        self.createLayouts()

    def createItems(self):
        self.expressionEditCompleter = QtGui.QCompleter()
        self.expressionEditCompleter.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.expressionEditModel = QtGui.QStringListModel()
        self.expressionEditCompleter.setModel(self.expressionEditModel)
        self.expressionEditModel.setStringList(self.historyExpression)

        self.expressionLabel = QtGui.QLabel(self.tr("Expression"))
        self.expressionEdit = QtGui.QLineEdit("")
        self.expressionEdit.setCompleter(self.expressionEditCompleter)

        self.variableEditCompleter = QtGui.QCompleter()
        self.variableEditCompleter.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.variableEditModel = QtGui.QStringListModel()
        self.variableEditCompleter.setModel(self.variableEditModel)
        self.variableEditModel.setStringList(self.historyVariable)

        self.variable1Label = QtGui.QLabel("Var.1")
        self.variable1Edit = QtGui.QLineEdit("x")
        self.variable1Edit.setCompleter(self.variableEditCompleter)

        self.numVar1Label = QtGui.QLabel(self.tr("#"))
        self.numVar1Edit = QtGui.QLineEdit("1")
        self.numVar1Edit.setValidator(QtGui.QIntValidator(0,9999,self.numVar1Edit))
        self.numVar1Edit.setMaximumSize(QtCore.QSize(60,100))

        self.variable2Label = QtGui.QLabel("Var.2")
        self.variable2Edit = QtGui.QLineEdit("y")
        self.variable2Edit.setCompleter(self.variableEditCompleter)

        self.numVar2Label = QtGui.QLabel(self.tr("#"))
        self.numVar2Edit = QtGui.QLineEdit("0")
        self.numVar2Edit.setValidator(QtGui.QIntValidator(0,9999,self.numVar2Edit))
        self.numVar2Edit.setMaximumSize(QtCore.QSize(60,100))

        self.variable3Label = QtGui.QLabel("Var.3")
        self.variable3Edit = QtGui.QLineEdit("z")
        self.variable3Edit.setCompleter(self.variableEditCompleter)

        self.numVar3Label = QtGui.QLabel(self.tr("#"))
        self.numVar3Edit = QtGui.QLineEdit("0")
        self.numVar3Edit.setValidator(QtGui.QIntValidator(0,9999,self.numVar3Edit))
        self.numVar3Edit.setMaximumSize(QtCore.QSize(60,100))

        self.gradientLabel = QtGui.QLabel(self.tr("Gradient"))
        self.divergenceLabel = QtGui.QLabel(self.tr("Divergence"))
        self.curlLabel = QtGui.QLabel(self.tr("Curl"))
        self.laplacianLabel = QtGui.QLabel(self.tr("Laplacian"))

        self.coordSystemLabel = QtGui.QLabel(self.tr("Orthogonal coordinate system"))
        self.coordSystemComboBox = QtGui.QComboBox()
        self.coordSystemComboBox.addItem(self.tr("Cartesian (x,y,z)"))
        self.coordSystemComboBox.addItem(self.tr("Cylindrical (rho,theta,z)"))
        self.coordSystemComboBox.addItem(self.tr("Spherical (r,theta,phi)"))
        self.coordSystemComboBox.setCurrentIndex(self.coordSystem)

        self.scalarFLabel = QtGui.QLabel(self.tr("Scalar F = "))
        self.scalarFEdit = QtGui.QLineEdit("")
        self.scalarFEdit.setCompleter(self.expressionEditCompleter)

        self.Fe1Label = QtGui.QLabel(self.tr("Vector F = <"))
        self.Fe1Edit = QtGui.QLineEdit("")
        self.Fe1Edit.setCompleter(self.expressionEditCompleter)

        self.Fe1sepLabel = QtGui.QLabel(self.tr(","))
        self.Fe2Edit = QtGui.QLineEdit("")
        self.Fe2Edit.setCompleter(self.expressionEditCompleter)

        self.Fe2sepLabel = QtGui.QLabel(self.tr(","))
        self.Fe3Edit = QtGui.QLineEdit("")
        self.Fe3Edit.setCompleter(self.expressionEditCompleter)
        self.Fe3sepLabel = QtGui.QLabel(self.tr(">"))

        self.font = QtGui.QFont()
        self.font.setFamily("DejaVu Sans Mono")
        if QtGui.QFontInfo(self.font).family() == "DejaVu Sans Mono":
            self.dejavuSansMono = True
        else:
            self.font.setFamily("Monospace")
            self.dejavuSansMono = False
        self.font.setPointSize(zoomFontSize[self.zoomLevel])
        self.resultEdit = QtGui.QTextBrowser()
        self.resultEdit.setReadOnly(False)
        self.resultEdit.setFont(self.font)

        # Connect buttons
        self.connect(self.expressionEdit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.variable1Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.numVar1Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.variable2Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.numVar2Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.variable3Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.numVar3Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.coordSystemComboBox,QtCore.SIGNAL("activated(int)"),self.changeCoordSystem)
        self.connect(self.scalarFEdit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.Fe1Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.Fe2Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.Fe3Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)

    def createLayouts(self):    
        self.h_box_expression = QtGui.QHBoxLayout()
        self.h_box_expression.addWidget(self.expressionLabel)
        self.h_box_expression.addWidget(self.expressionEdit)
        self.h_box_variables = QtGui.QHBoxLayout()
        self.h_box_variables.addWidget(self.variable1Label)
        self.h_box_variables.addWidget(self.variable1Edit)
        self.h_box_variables.addWidget(self.numVar1Label)
        self.h_box_variables.addWidget(self.numVar1Edit)
        self.h_box_variables.addWidget(self.variable2Label)
        self.h_box_variables.addWidget(self.variable2Edit)
        self.h_box_variables.addWidget(self.numVar2Label)
        self.h_box_variables.addWidget(self.numVar2Edit)
        self.h_box_variables.addWidget(self.variable3Label)
        self.h_box_variables.addWidget(self.variable3Edit)
        self.h_box_variables.addWidget(self.numVar3Label)
        self.h_box_variables.addWidget(self.numVar3Edit)
        self.h_box_coordSystem = QtGui.QHBoxLayout()
        self.h_box_coordSystem.addWidget(self.gradientLabel)
        self.h_box_coordSystem.addWidget(self.divergenceLabel)
        self.h_box_coordSystem.addWidget(self.curlLabel)
        self.h_box_coordSystem.addWidget(self.laplacianLabel)
        self.coordSystemSpacer = QtGui.QSpacerItem(0, 0, QtGui.QSizePolicy.Expanding,QtGui.QSizePolicy.Minimum)
        self.h_box_coordSystem.addItem(self.coordSystemSpacer)
        self.h_box_coordSystem.addWidget(self.coordSystemLabel)
        self.h_box_coordSystem.addWidget(self.coordSystemComboBox)
        self.h_box_scalarF = QtGui.QHBoxLayout()
        self.h_box_scalarF.addWidget(self.scalarFLabel)
        self.h_box_scalarF.addWidget(self.scalarFEdit)
        self.h_box_vectorF = QtGui.QHBoxLayout()
        self.h_box_vectorF.addWidget(self.Fe1Label)
        self.h_box_vectorF.addWidget(self.Fe1Edit)
        self.h_box_vectorF.addWidget(self.Fe1sepLabel)
        self.h_box_vectorF.addWidget(self.Fe2Edit)
        self.h_box_vectorF.addWidget(self.Fe2sepLabel)
        self.h_box_vectorF.addWidget(self.Fe3Edit)
        self.h_box_vectorF.addWidget(self.Fe3sepLabel)
        self.v_box = QtGui.QVBoxLayout()
        self.v_box.addLayout(self.h_box_coordSystem)
        self.v_box.addLayout(self.h_box_scalarF)
        self.v_box.addLayout(self.h_box_vectorF)
        self.v_box.addLayout(self.h_box_expression)
        self.v_box.addLayout(self.h_box_variables)
        self.v_box.addWidget(self.resultEdit)
        self.window.setLayout(self.v_box)     

    def showHideItems(self):
        if self.typeDerivative == derivativeType['derivative']:
            self.gradientLabel.hide()
            self.divergenceLabel.hide()
            self.curlLabel.hide()
            self.laplacianLabel.hide()
            self.coordSystemLabel.hide()
            self.coordSystemComboBox.hide()
            self.scalarFLabel.hide()
            self.scalarFEdit.hide()
            self.Fe1Label.hide()
            self.Fe1Edit.hide()
            self.Fe1sepLabel.hide()
            self.Fe2Edit.hide()
            self.Fe2sepLabel.hide()
            self.Fe3Edit.hide()
            self.Fe3sepLabel.hide()
            self.expressionLabel.show()
            self.expressionEdit.show()
            self.variable1Label.show()
            self.variable1Edit.show()
            self.numVar1Label.show()
            self.numVar1Edit.show()
            self.variable2Label.show()
            self.variable2Edit.show()
            self.numVar2Label.show()
            self.numVar2Edit.show()
            self.variable3Label.show()
            self.variable3Edit.show()
            self.numVar3Label.show()
            self.numVar3Edit.show()
        else:
            self.expressionLabel.hide()
            self.expressionEdit.hide()
            self.variable1Label.hide()
            self.variable1Edit.hide()
            self.numVar1Label.hide()
            self.numVar1Edit.hide()
            self.variable2Label.hide()
            self.variable2Edit.hide()
            self.numVar2Label.hide()
            self.numVar2Edit.hide()
            self.variable3Label.hide()
            self.variable3Edit.hide()
            self.numVar3Label.hide()
            self.numVar3Edit.hide()
            self.coordSystemLabel.show()
            self.coordSystemComboBox.show()
            if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['laplacian']):
                self.divergenceLabel.hide()
                self.curlLabel.hide()
                if self.typeDerivative == derivativeType['gradient']:
                    self.gradientLabel.show()
                    self.laplacianLabel.hide()
                elif self.typeDerivative == derivativeType['laplacian']:
                    self.gradientLabel.hide()
                    self.laplacianLabel.show()
                self.Fe1Label.hide()
                self.Fe1Edit.hide()
                self.Fe1sepLabel.hide()
                self.Fe2Edit.hide()
                self.Fe2sepLabel.hide()
                self.Fe3Edit.hide()
                self.Fe3sepLabel.hide()
                self.scalarFLabel.show()
                self.scalarFEdit.show()
            else:
                self.gradientLabel.hide()
                self.laplacianLabel.hide()
                if self.typeDerivative == derivativeType['divergence']:
                    self.divergenceLabel.show()
                    self.curlLabel.hide()
                elif self.typeDerivative == derivativeType['curl']:
                    self.divergenceLabel.hide()
                    self.curlLabel.show()
                self.scalarFLabel.hide()
                self.scalarFEdit.hide()
                self.Fe1Label.show()
                self.Fe1Edit.show()
                self.Fe1sepLabel.show()
                self.Fe2Edit.show()
                self.Fe2sepLabel.show()
                self.Fe3Edit.show()
                self.Fe3sepLabel.show()

    def createMenuAndActions(self):
        if not self.isMaemo:
            fileMenu = self.menuBar().addMenu(self.tr("&File"))
#           saveAction = fileMenu.addAction(self.tr("&Save"))
#           shareAction = fileMenu.addAction(self.tr("&Share"))
            fileMenu.addSeparator()
            exitAction = QtGui.QAction(self.tr("E&xit"), self, shortcut="Ctrl+Q")
            fileMenu.addAction(exitAction)
      
        viewMenu = self.menuBar().addMenu(self.tr("&View"))
        zoomInAction = viewMenu.addAction(self.tr("Zoom in"))
        zoomOutAction = viewMenu.addAction(self.tr("Zoom out"))
     
        derivativesMenu = self.menuBar().addMenu(self.tr("&Derivatives"))
        derivativeAction = QtGui.QAction(self.tr("Derivative"), self, checkable=True)
        gradientAction = QtGui.QAction(self.tr("Grad"), self, checkable=True)
        divergenceAction = QtGui.QAction(self.tr("Div"), self, checkable=True)
        curlAction = QtGui.QAction(self.tr("Curl"), self, checkable=True)
        laplacianAction = QtGui.QAction(self.tr("Laplacian"), self, checkable=True)
        typeDerivativesGroup = QtGui.QActionGroup(self)
        typeDerivativesGroup.addAction(derivativeAction)
        typeDerivativesGroup.addAction(gradientAction)
        typeDerivativesGroup.addAction(divergenceAction)
        typeDerivativesGroup.addAction(curlAction)
        typeDerivativesGroup.addAction(laplacianAction)
        if self.typeDerivative == derivativeType['derivative']:
            derivativeAction.setChecked(True)
        elif self.typeDerivative == derivativeType['gradient']:
            gradientAction.setChecked(True)
        elif self.typeDerivative == derivativeType['divergence']:
            divergenceAction.setChecked(True)
        elif self.typeDerivative == derivativeType['curl']:
            curlAction.setChecked(True)
        else:
            laplacianAction.setChecked(True)     
        derivativesMenu.addAction(derivativeAction)
        derivativesMenu.addAction(gradientAction)
        derivativesMenu.addAction(divergenceAction)
        derivativesMenu.addAction(curlAction)
        derivativesMenu.addAction(laplacianAction)
        derivativesMenu.addSeparator()
        configAction =  derivativesMenu.addAction(self.tr("Settings"))

        helpMenu = self.menuBar().addMenu(self.tr("&Help"))
        helpAction = helpMenu.addAction(self.tr("&Help"))
        helpMenu.addSeparator()
        aboutAction = helpMenu.addAction(self.tr("&About"))
        aboutQtAction = helpMenu.addAction(self.tr("About &Qt"))

        if self.isMaemo and (not self.isMaemo5):
            exitAction = QtGui.QAction(self.tr("E&xit"), self, shortcut="Ctrl+Q")
            self.menuBar().addAction(exitAction)

        self.connect(zoomInAction, QtCore.SIGNAL("triggered()"), self.zoomInMenu)
        self.connect(zoomOutAction, QtCore.SIGNAL("triggered()"), self.zoomOutMenu)
        self.connect(derivativeAction, QtCore.SIGNAL("triggered()"), self.derivativeMenu)
        self.connect(gradientAction, QtCore.SIGNAL("triggered()"), self.gradientMenu)
        self.connect(divergenceAction, QtCore.SIGNAL("triggered()"), self.divergenceMenu)
        self.connect(curlAction, QtCore.SIGNAL("triggered()"), self.curlMenu)
        self.connect(laplacianAction, QtCore.SIGNAL("triggered()"), self.laplacianMenu)
        if not self.isMaemo5:
            self.connect(exitAction, QtCore.SIGNAL("triggered()"), app, QtCore.SLOT("quit()"))
        self.connect(configAction, QtCore.SIGNAL("triggered()"), self.showConfigWindow)
        self.connect(helpAction, QtCore.SIGNAL("triggered()"), self.showHelpBox)
        self.connect(aboutAction, QtCore.SIGNAL("triggered()"), self.showAboutBox)
        self.connect(aboutQtAction, QtCore.SIGNAL("triggered()"), QtGui.qApp.aboutQt)

    def showHelpBox(self):
        QtGui.QMessageBox.about(self, self.tr("Help on Derivative"),
            u'Mathematical operators : + - * / ** (power)\nExample of functions : sqrt, exp, log, sin, acos\n' +\
            u'Examples of expressions : 1/x**n, sqrt(x/(x**3+1)), q/(4*pi*epsilon0*r**2), exp(-(x-x_0)**2/(2*sigma**2))\n'+\
            u'Examples of variables : x, y, z, t, rho, theta, phi, Omega\n'+\
            u'Look at the SymPy site: http://sympy.org/')

    def showAboutBox(self):
        versionDerivative = '1.0.2'
        siteDerivative = 'http://www.RobertoColistete.net/Derivative\n'
        infoDerivative = self.tr("Calculation of derivatives using ")
        infoPython = "Python v"+python_version()
        infoSymPy = "SymPy v"+__version__+" (http://sympy.org/)"
        QtGui.QMessageBox.about(self, self.tr("About Derivative"),
            'Derivative v'+versionDerivative+' - LGPLv3 (C) 2011 Roberto Colistete Jr.\n'+\
            siteDerivative+infoDerivative+infoPython+" and "+infoSymPy+u'\n\n* In loving memory of my wife Lorena *')

    def showConfigWindow(self):
        if self.isMaemo5:
            self.configWin = QtGui.QMainWindow(self)
            self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
            self.configWin.setAttribute(Qt.WA_Maemo5StackedWindow)
            self.configWin.window = QtGui.QWidget()
            self.configWin.setCentralWidget(self.configWin.window)
        else:
            self.configWin = QtGui.QDialog(self)

        self.configWin.setWindowTitle(self.tr("Derivative (Settings)"))
        self.configWin.v_box = QtGui.QVBoxLayout()

        self.configUpdateResult = False

        self.configWin.showDerivativeCheckBox = QtGui.QCheckBox(self.tr("Show not calculated derivative before result"))
        if self.showDerivative:
            self.configWin.showDerivativeCheckBox.toggle()

        self.configWin.showTimeCheckBox = QtGui.QCheckBox(self.tr("Show calculation time before derivative result"))
        if self.showTime:
            self.configWin.showTimeCheckBox.toggle()

        self.configWin.numerApproxCheckBox = QtGui.QCheckBox(self.tr("Numerical approximation of the derivative result"))
        if self.numerApprox:
            self.configWin.numerApproxCheckBox.toggle()

        self.configWin.h_box_numDig = QtGui.QHBoxLayout()
        self.configWin.numDigLabel = QtGui.QLabel(self.tr("Number of digits for numerical approximation"))
        self.configWin.numDigSpacer = QtGui.QSpacerItem(40, 20, QtGui.QSizePolicy.Expanding, QtGui.QSizePolicy.Minimum)
        self.configWin.numDigEdit = QtGui.QLineEdit(str(self.numDig))
        self.configWin.numDigEdit.setValidator(QtGui.QIntValidator(1,1000000,self.configWin.numDigEdit))
        self.configWin.numDigEdit.setMaximumSize(QtCore.QSize(100,100))
        self.configWin.h_box_numDig.addWidget(self.configWin.numDigLabel)
        self.configWin.h_box_numDig.addWidget(self.configWin.numDigEdit)
        self.configWin.h_box_numDig.addItem(self.configWin.numDigSpacer)

        self.configWin.h_box_simplify = QtGui.QHBoxLayout()
        self.configWin.simplifyLabel = QtGui.QLabel(self.tr("Simplification for non-numerical derivative"))
        self.configWin.simplifySpacer = QtGui.QSpacerItem(40, 20, QtGui.QSizePolicy.Expanding,\
                                                          QtGui.QSizePolicy.Minimum)
        self.configWin.simplifyComboBox = QtGui.QComboBox()
        self.configWin.simplifyComboBox.addItem(self.tr("None"))
        self.configWin.simplifyComboBox.addItem(self.tr("Expand terms"))
        self.configWin.simplifyComboBox.addItem(self.tr("Simplify terms"))
        self.configWin.simplifyComboBox.addItem(self.tr("Expand all"))
        self.configWin.simplifyComboBox.addItem(self.tr("Simplify all"))
        self.configWin.simplifyComboBox.setCurrentIndex(self.simplifyResult)
        self.configWin.h_box_simplify.addWidget(self.configWin.simplifyLabel)
        self.configWin.h_box_simplify.addWidget(self.configWin.simplifyComboBox)
        self.configWin.h_box_simplify.addItem(self.configWin.simplifySpacer)

        self.configWin.h_box_output = QtGui.QHBoxLayout()
        self.configWin.outputLabel = QtGui.QLabel(self.tr("Output type for derivative result"))
        self.configWin.outputSpacer = QtGui.QSpacerItem(40, 20, QtGui.QSizePolicy.Expanding, QtGui.QSizePolicy.Minimum)
        self.configWin.outputComboBox = QtGui.QComboBox()
        self.configWin.outputComboBox.addItem(self.tr("Simple"))
        self.configWin.outputComboBox.addItem(self.tr("Bidimensional"))
        self.configWin.outputComboBox.addItem(self.tr("Typesetting"))
        self.configWin.outputComboBox.addItem("LaTex")
        self.configWin.outputComboBox.addItem("MathML")
        self.configWin.outputComboBox.addItem("C")
        self.configWin.outputComboBox.addItem("Fortran")
        self.configWin.outputComboBox.setCurrentIndex(self.outputResult)
        self.configWin.h_box_output.addWidget(self.configWin.outputLabel)
        self.configWin.h_box_output.addWidget(self.configWin.outputComboBox)
        self.configWin.h_box_output.addItem(self.configWin.outputSpacer)

        if not self.isMaemo5:
            self.configWin.h_box_close = QtGui.QHBoxLayout()
            self.configWin.closeButton = QtGui.QPushButton("Close")
            self.configWin.h_box_close.addStretch(1)
            self.configWin.h_box_close.addWidget(self.configWin.closeButton)

        self.configWin.v_box.addWidget(self.configWin.showDerivativeCheckBox)
        self.configWin.v_box.addWidget(self.configWin.showTimeCheckBox)
        self.configWin.v_box.addWidget(self.configWin.numerApproxCheckBox)
        self.configWin.v_box.addLayout(self.configWin.h_box_numDig)
        self.configWin.v_box.addLayout(self.configWin.h_box_simplify)
        self.configWin.v_box.addLayout(self.configWin.h_box_output)
        if not self.isMaemo5:
            self.configWin.v_box.addStretch(1)
            self.configWin.v_box.addSpacing(12)
            self.configWin.v_box.addLayout(self.configWin.h_box_close)
            self.configWin.setLayout(self.configWin.v_box)
        else:
            self.configWin.window.setLayout(self.configWin.v_box)

        self.connect(self.configWin.showDerivativeCheckBox, QtCore.SIGNAL("stateChanged(int)"), self.toggleShowDerivative)
        self.connect(self.configWin.showTimeCheckBox, QtCore.SIGNAL("stateChanged(int)"), self.toggleShowTime)
        self.connect(self.configWin.numerApproxCheckBox, QtCore.SIGNAL("stateChanged(int)"), self.toggleNumerApprox)
        self.connect(self.configWin.numDigEdit, QtCore.SIGNAL("textEdited (const QString&)"),self.changeNumDig)
        self.connect(self.configWin.simplifyComboBox, QtCore.SIGNAL("activated(int)"), self.changeSimplifyResult)
        self.connect(self.configWin.outputComboBox, QtCore.SIGNAL("activated(int)"), self.changeOutputResult)
        if not self.isMaemo5:
            self.configWin.closeButton.clicked.connect(self.configWin.close)
        self.configWin.show()

    def loadSettings(self):
        self.zoomLevel = self.settings.value('zoomLevel',3).toInt()[0]
        self.typeDerivative = self.settings.value('TypeDerivative',derivativeType['derivative']).toInt()[0]
        self.coordSystem = self.settings.value('CoordSystem',coordSysNumber['cartesian3D']).toInt()[0]
        self.showDerivative = self.settings.value('ShowDerivative',True).toBool()
        self.showTime = self.settings.value('ShowTime',True).toBool()
        self.numerApprox = self.settings.value('NumerApprox',False).toBool()
        self.numDig = self.settings.value('NumDig',15).toInt()[0]
        self.simplifyResult = self.settings.value('SimplifyResult',simplifyType['simplifyterms']).toInt()[0]
        self.outputResult = self.settings.value('OutputResult',outputType['bidimensional']).toInt()[0]
        self.historyUserExpression = self.settings.value('historyUserExpression','').toStringList()

    def saveSettings(self):
        self.settings.setValue('zoomLevel',self.zoomLevel)
        self.settings.setValue('TypeDerivative',self.typeDerivative) 	
        self.settings.setValue('CoordSystem',self.coordSystem) 	
        self.settings.setValue('ShowDerivative',self.showDerivative) 	
        self.settings.setValue('ShowTime',self.showTime)
        self.settings.setValue('numerApprox',self.numerApprox) 	
        self.settings.setValue('NumDig',self.numDig)
        self.settings.setValue('SimplifyResult',self.simplifyResult)
        self.settings.setValue('OutputResult',self.outputResult)
        self.settings.setValue('historyUserExpression',self.historyUserExpression)

    def zoomInMenu(self):
        if self.zoomLevel < (len(zoomFontSize)-1):
            self.zoomLevel += 1
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nZoom in to font size "+str(zoomFontSize[self.zoomLevel])+"\n"), \
                                                  maemo5InformationBoxTime)
                QApplication.processEvents()
            self.font.setPointSize(zoomFontSize[self.zoomLevel])
            self.resultEdit.setFont(self.font)
            # Setting the pretty printing of SymPy to use unicode characters and correct number of columns.
            init_printing(use_unicode=True, num_columns=self.zoomNumColumns[self.zoomLevel])
            self.updateResult(flagCalculate=False)
            self.configUpdateResult = True
            self.saveSettings()          

    def zoomOutMenu(self):
        if self.zoomLevel > 0:
            self.zoomLevel -= 1
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nZoom out to font size "+str(zoomFontSize[self.zoomLevel])+"\n"), \
                                                  maemo5InformationBoxTime)
                QApplication.processEvents()
            self.font.setPointSize(zoomFontSize[self.zoomLevel])
            self.resultEdit.setFont(self.font)
            # Setting the pretty printing of SymPy to use unicode characters and correct number of columns.
            init_printing(use_unicode=True, num_columns=self.zoomNumColumns[self.zoomLevel])
            self.updateResult(flagCalculate=False)
            self.configUpdateResult = True
            self.saveSettings()          

    def toggleShowDerivative(self, status):
        if status == Qt.Checked:
            self.showDerivative = True
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nNot calculated derivative is visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        else:
            self.showDerivative = False
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nNot calculated derivative is not visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        self.updateResult(flagCalculate=False)
        self.configUpdateResult = True
        self.saveSettings()

    def toggleShowTime(self, status):
        if status == Qt.Checked:
            self.showTime = True
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nCalculation time is visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        else:
            self.showTime = False
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nCalculation time is not visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        self.updateResult(flagCalculate=False)
        self.configUpdateResult = True
        self.saveSettings()

    def toggleNumerApprox(self, status):
        if status == Qt.Checked:
            self.numerApprox = True
        else:
            self.numerApprox= False
        self.saveSettings()

    def changeNumDig(self):
        if self.configWin.numDigEdit.text():
            self.numDig = int(self.configWin.numDigEdit.text())
        else:
            self.numDig = 0
        self.saveSettings()

    def changeSimplifyResult(self, option):
        self.simplifyResult = option
        self.configUpdateResult = True
        self.saveSettings()

    def changeOutputResult(self, option):
        self.outputResult = option
        if self.isMaemo5:
            if self.outputResult == outputType['simple']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nDerivative result converted to simple output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['bidimensional']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nDerivative result converted to bidimensional output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['typesetting']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nDerivative result converted to typesetting output\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
            elif self.outputResult == outputType['latex']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nDerivative result converted to LaTeX output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['mathml']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nDerivative result converted to MathML output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['c']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nDerivative result converted to C output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['fortran']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nDerivative result converted to Fortran output\n"), maemo5InformationBoxTime)
            QApplication.processEvents()
        self.updateResult(flagCalculate=False)
        self.configUpdateResult = True
        self.saveSettings()

    def changeCoordSystem(self, option):
        self.coordSystem = option
        self.saveSettings()

    def derivativeMenu(self):
        self.typeDerivative = derivativeType['derivative']
        self.showHideItems()
        self.saveSettings()

    def gradientMenu(self):
        self.typeDerivative = derivativeType['gradient']
        self.showHideItems()
        self.saveSettings()

    def divergenceMenu(self):
        self.typeDerivative = derivativeType['divergence']
        self.showHideItems()
        self.saveSettings()

    def curlMenu(self):
        self.typeDerivative = derivativeType['curl']
        self.showHideItems()
        self.saveSettings()

    def laplacianMenu(self):
        self.typeDerivative = derivativeType['laplacian']
        self.showHideItems()
        self.saveSettings()

    def mapexpr(self,expr,func):
        if isinstance(expr,Add):
            return apply(Add,map(func,expr.args))
        else:
            return func(expr)

    def updateHistoriesAndResult(self):
        text = unicode(self.expressionEdit.text())
        if not text in self.historyExpression:
            self.historyUserExpression.prepend(text)
            self.historyUserExpression=self.historyUserExpression[:100]
            self.historyExpression=self.historyInitExpression+self.historyUserExpression
            self.expressionEditModel.setStringList(self.historyExpression)
        text = unicode(self.scalarFEdit.text())
        if not text in self.historyExpression:
            self.historyUserExpression.prepend(text)
            self.historyUserExpression=self.historyUserExpression[:100]
            self.historyExpression=self.historyInitExpression+self.historyUserExpression
            self.expressionEditModel.setStringList(self.historyExpression)
        text = unicode(self.Fe1Edit.text())
        if not text in self.historyExpression:
            self.historyUserExpression.prepend(text)
            self.historyUserExpression=self.historyUserExpression[:100]
            self.historyExpression=self.historyInitExpression+self.historyUserExpression
            self.expressionEditModel.setStringList(self.historyExpression)
        text = unicode(self.Fe2Edit.text())
        if not text in self.historyExpression:
            self.historyUserExpression.prepend(text)
            self.historyUserExpression=self.historyUserExpression[:100]
            self.historyExpression=self.historyInitExpression+self.historyUserExpression
            self.expressionEditModel.setStringList(self.historyExpression)
        text = unicode(self.Fe3Edit.text())
        if not text in self.historyExpression:
            self.historyUserExpression.prepend(text)
            self.historyUserExpression=self.historyUserExpression[:100]
            self.historyExpression=self.historyInitExpression+self.historyUserExpression
            self.expressionEditModel.setStringList(self.historyExpression)
        self.saveSettings()
        self.updateResult()

    def updateResult(self, flagCalculate=True):
        if flagCalculate:
            self.resultEdit.clear()
            self.resultEdit.setTextColor(QtGui.QColor("darkGreen"))
            if self.typeDerivative == derivativeType['derivative']:
                self.resultEdit.append(u"Calculating derivative...")
            elif self.typeDerivative == derivativeType['divergence']:
                self.resultEdit.append(u"Calculating divergence...")
            elif self.typeDerivative == derivativeType['curl']:
                self.resultEdit.append(u"Calculating curl...")
            elif self.typeDerivative == derivativeType['laplacian']:
                self.resultEdit.append(u"Calculating laplacian...")
            self.resultEdit.setTextColor(QtGui.QColor("black"))
            QApplication.processEvents()
            if self.isMaemo5:
                self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator,True)
                QApplication.processEvents()
    	    timet1=time.time()
            if self.typeDerivative == derivativeType['derivative']:
                expressionDerivative = unicode(self.expressionEdit.text())
                variable1Derivative = unicode(str(self.variable1Edit.text()).strip())
                numVar1Derivative = unicode(str(self.numVar1Edit.text()).strip())
                if not numVar1Derivative:
                    numVar1Derivative = u'0'
                variable2Derivative = unicode(str(self.variable2Edit.text()).strip())
                numVar2Derivative = unicode(str(self.numVar2Edit.text()).strip())
                if not numVar2Derivative:
                    numVar2Derivative = u'0'
                variable3Derivative = unicode(str(self.variable3Edit.text()).strip())
                numVar3Derivative = unicode(str(self.numVar3Edit.text()).strip())
                if not numVar3Derivative:
                    numVar3Derivative = u'0'
                derivativeExpr = u'('+expressionDerivative
                if variable1Derivative and (eval(numVar1Derivative) > 0):
                    derivativeExpr += u','+variable1Derivative+u','+numVar1Derivative
                if variable2Derivative and (eval(numVar2Derivative) > 0):
                    derivativeExpr += u','+variable2Derivative+u','+numVar2Derivative
                if variable3Derivative and (eval(numVar3Derivative) > 0):
                    derivativeExpr += u','+variable3Derivative+u','+numVar3Derivative
                derivativeExpr += u')'
                try:
                    self.nonCalculatedDerivative = sympify(u'Derivative'+derivativeExpr)
                except:
                    self.nonCalculatedDerivative = u'Derivative'+derivativeExpr
                try:
                    if self.numerApprox:
                        self.resultDerivative = sympify(u'N(diff'+derivativeExpr+u','+unicode(str(self.numDig))+u')')
                    else:
                        self.resultDerivative = sympify(u'diff'+derivativeExpr)
                except:
                    self.resultDerivative = derivativeErrorMessage
            elif (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['laplacian']):
                self.nonCalculatedDerivative = u''
                scalarF = unicode(self.scalarFEdit.text())
                if self.typeDerivative == derivativeType['gradient']:
                    try:
                        self.nonCalculatedDerivative = sympify(u'grad('+scalarF+u')')
                    except:
                        self.nonCalculatedDerivative = u'grad('+scalarF+u')'
                    self.resultDerivative = gradient(scalarF, self.coordSystem)
                elif self.typeDerivative == derivativeType['laplacian']:
                    try:
                        self.nonCalculatedDerivative = sympify(u'laplacian('+scalarF+u')')
                    except:
                        self.nonCalculatedDerivative = u'laplacian('+scalarF+u')'
                    self.resultDerivative = laplacian(scalarF, self.coordSystem)
            else:
                Fe1 = unicode(self.Fe1Edit.text())
                Fe2 = unicode(self.Fe2Edit.text())
                Fe3 = unicode(self.Fe3Edit.text())
                if self.typeDerivative == derivativeType['divergence']:
                    try:
                        self.nonCalculatedDerivative = sympify(u'['+Fe1+u','+Fe2+u','+Fe3+u']')
                    except:
                        self.nonCalculatedDerivative = u'['+Fe1+u','+Fe2+u','+Fe3+u']'
                    self.resultDerivative = divergence([Fe1,Fe2,Fe3], self.coordSystem)
                elif self.typeDerivative == derivativeType['curl']:
                    try:
                        self.nonCalculatedDerivative = sympify(u'['+Fe1+u','+Fe2+u','+Fe3+u']')
                    except:
                        self.nonCalculatedDerivative = u'['+Fe1+u','+Fe2+u','+Fe3+u']'
                    self.resultDerivative = curl([Fe1,Fe2,Fe3], self.coordSystem)
            if (self.resultDerivative) and (type(self.resultDerivative) != unicode):
                if (self.simplifyResult == simplifyType['none']) or (self.numerApprox):
                    self.resultDerivativeSimp = sympify(self.resultDerivative)
                elif self.simplifyResult == simplifyType['expandterms']:
                    if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                        self.resultDerivativeSimp = [sympify(self.mapexpr(self.resultDerivative[0],expand)),\
                                                     sympify(self.mapexpr(self.resultDerivative[1],expand)),\
                                                     sympify(self.mapexpr(self.resultDerivative[2],expand))]
                    else:
                        self.resultDerivativeSimp = sympify(self.mapexpr(self.resultDerivative,expand))
                elif self.simplifyResult == simplifyType['simplifyterms']:
                    if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                        self.resultDerivativeSimp = [sympify(self.mapexpr(self.resultDerivative[0],simplify)),\
                                                     sympify(self.mapexpr(self.resultDerivative[1],simplify)),\
                                                     sympify(self.mapexpr(self.resultDerivative[2],simplify))]
                    else:
                        self.resultDerivativeSimp = sympify(self.mapexpr(self.resultDerivative,simplify))
                elif self.simplifyResult == simplifyType['expandall']:
                    if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                        self.resultDerivativeSimp = [sympify(expand(self.resultDerivative[0])),\
                                                     sympify(expand(self.resultDerivative[1])),\
                                                     sympify(expand(self.resultDerivative[2]))]
                    else:
                        self.resultDerivativeSimp = sympify(expand(self.resultDerivative))
                elif self.simplifyResult == simplifyType['simplifyall']:
                    if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                        self.resultDerivativeSimp = [sympify(simplify(self.resultDerivative[0])),\
                                                     sympify(simplify(self.resultDerivative[1])),\
                                                     sympify(simplify(self.resultDerivative[2]))]
                    else:
                        self.resultDerivativeSimp = sympify(simplify(self.resultDerivative))
            else:
                self.resultDerivativeSimp = self.resultDerivative
    	    timet2=time.time()
            self.timeDerivative=timet2-timet1
            if self.isMaemo5:
                self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator,False)
                QApplication.processEvents()
        self.nonCalculatedDerivativeOutput = self.nonCalculatedDerivative
        self.resultOutput = self.resultDerivativeSimp
        if self.outputResult == outputType['bidimensional']:
            if (type(self.nonCalculatedDerivative) != unicode):
                self.nonCalculatedDerivativeOutput = self.fixUnicodeText(printing.pretty(self.nonCalculatedDerivative))
            if (type(self.resultDerivativeSimp) != unicode):
                if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                    self.resultOutput = [self.fixUnicodeText(printing.pretty(self.resultDerivativeSimp[0])),\
                                         self.fixUnicodeText(printing.pretty(self.resultDerivativeSimp[1])),\
                                         self.fixUnicodeText(printing.pretty(self.resultDerivativeSimp[2]))]
                else:
                    self.resultOutput = self.fixUnicodeText(printing.pretty(self.resultDerivativeSimp))
        if self.outputResult == outputType['typesetting']:
            if (type(self.nonCalculatedDerivative) != unicode):
                self.nonCalculatedDerivativeOutput = self.fixUnicodeText(printing.pretty(self.nonCalculatedDerivative))
            if (type(self.resultDerivativeSimp) != unicode):
                if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                    self.resultOutput = [self.fixUnicodeText(printing.pretty(self.resultDerivativeSimp[0])),\
                                         self.fixUnicodeText(printing.pretty(self.resultDerivativeSimp[1])),\
                                         self.fixUnicodeText(printing.pretty(self.resultDerivativeSimp[2]))]
                else:
                    self.resultOutput = self.fixUnicodeText(printing.pretty(self.resultDerivativeSimp))
        elif self.outputResult == outputType['latex']:
            if (type(self.nonCalculatedDerivative) != unicode):
                self.nonCalculatedDerivativeOutput = latex(self.nonCalculatedDerivative)
            if (type(self.resultDerivativeSimp) != unicode):
                if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                    self.resultOutput = [latex(self.resultDerivativeSimp[0]),latex(self.resultDerivativeSimp[1]),latex(self.resultDerivativeSimp[2])]
                else:
                    self.resultOutput = latex(self.resultDerivativeSimp)
        elif self.outputResult == outputType['mathml']:
            if (type(self.nonCalculatedDerivative) != unicode):
                self.nonCalculatedDerivativeOutput = mathml(self.nonCalculatedDerivative)
            if (type(self.resultDerivativeSimp) != unicode):
                if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                    self.resultOutput = [mathml(self.resultDerivativeSimp[0]),mathml(self.resultDerivativeSimp[1]),mathml(self.resultDerivativeSimp[2])]
                else:
                    self.resultOutput = mathml(self.resultDerivativeSimp)
        elif self.outputResult == outputType['c']:
            if (type(self.nonCalculatedDerivative) != unicode):
                self.nonCalculatedDerivativeOutput = self.nonCalculatedDerivative
            if (type(self.resultDerivativeSimp) != unicode):
                if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                    self.resultOutput = [ccode(self.resultDerivativeSimp[0]),ccode(self.resultDerivativeSimp[1]),ccode(self.resultDerivativeSimp[2])]
                else:
                    self.resultOutput = ccode(self.resultDerivativeSimp)
        elif self.outputResult == outputType['fortran']:
            if (type(self.nonCalculatedDerivative) != unicode):
                self.nonCalculatedDerivativeOutput = self.nonCalculatedDerivative
            if (type(self.resultDerivativeSimp) != unicode):
                if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                    self.resultOutput = [fcode(self.resultDerivativeSimp[0]),fcode(self.resultDerivativeSimp[1]),fcode(self.resultDerivativeSimp[2])]
                else:
                    self.resultOutput = fcode(self.resultDerivativeSimp)
        self.resultEdit.clear()
        if self.showTime:
            if self.timeDerivative > 0.0:
                self.resultEdit.setTextColor(QtGui.QColor("darkGreen"))
                self.resultEdit.append(u"Calculated after %f seconds :" % self.timeDerivative)
                self.resultEdit.setTextColor(QtGui.QColor("black"))
        if self.showDerivative and self.nonCalculatedDerivativeOutput:
            self.resultEdit.setTextColor(QtGui.QColor("Blue"))
            if self.typeDerivative == derivativeType['divergence']:
                self.resultEdit.append(u"div of \n%s\n=" % self.nonCalculatedDerivativeOutput)
            elif self.typeDerivative == derivativeType['curl']:
                self.resultEdit.append(u"curl of \n%s\n=" % self.nonCalculatedDerivativeOutput)
            else:
                self.resultEdit.append(u"%s\n=" % self.nonCalculatedDerivativeOutput)
            self.resultEdit.setTextColor(QtGui.QColor("black"))
        if (type(self.resultDerivativeSimp) != unicode):
            self.resultEdit.setTextColor(QtGui.QColor("black"))
            if (self.typeDerivative == derivativeType['gradient']) or (self.typeDerivative == derivativeType['curl']):
                self.resultEdit.append(u"Component %s :" % varCoordSys[coordSysType[self.coordSystem]][0])
                self.resultEdit.append(u"%s" % unicode(self.resultOutput[0]))
                self.resultEdit.append(u"Component %s :" % varCoordSys[coordSysType[self.coordSystem]][1])
                self.resultEdit.append(u"%s" % unicode(self.resultOutput[1]))
                self.resultEdit.append(u"Component %s :" % varCoordSys[coordSysType[self.coordSystem]][2])
                self.resultEdit.append(u"%s" % unicode(self.resultOutput[2]))
            else:
                self.resultEdit.append(u"%s" % unicode(self.resultOutput))
        else:
            self.resultEdit.setTextColor(QtGui.QColor("Red"))
            self.resultEdit.append(u"%s" % unicode(self.resultOutput))

    def fixUnicodeText(self, str):
        if self.isMaemo:
            str = str.replace(u"⎽","_")
        if self.isMaemo5 and (not self.dejavuSansMono):
            str = str.replace(u"⎮",u'\u2223')
            str = str.replace(u"⎜",u'\u2223')
            str = str.replace(u"⎟",u'\u2223')
            str = str.replace(u"⎢",u'\u2223')
            str = str.replace(u"⎥",u'\u2223')
            str = str.replace(u"⎛",u"⌈")
            str = str.replace(u"⎡",u"⌈")
            str = str.replace(u"⎝",u"⌊")
            str = str.replace(u"⎣",u"⌊")
            str = str.replace(u"⎞",u"⌉")
            str = str.replace(u"⎤",u"⌉")
            str = str.replace(u"⎠",u"⌋")
            str = str.replace(u"⎦",u"⌋")
        str = str.replace(u"ℯ","e")
        str = str.replace(u"ⅈ","i")
        return str

def gradient(f,coordSys):
    gradResult=[0,0,0]
    j = coordSysType[coordSys]
    h = scaleFactorCoordSys[j]
    u = varCoordSys[j]
    try:
        for i in range(len(u)):
            gradResult[i] = sympify('diff('+f+','+u[i]+')/'+h[i])
    except:
        gradResult = gradientErrorMessage
    return gradResult
    
def divergence(vf,coordSys):
    divResult=0
    j = coordSysType[coordSys]
    h = scaleFactorCoordSys[j]
    u = varCoordSys[j]
    prodh = h[0]+'*'+h[1]+'*'+h[2]
    try:
        for i in range(len(u)):
            divResult += sympify('diff(('+prodh+'/'+h[i]+')*'+vf[i]+','+u[i]+')/('+prodh+')')
        if coordSysType[coordSys] == 'spherical':
            if sympify('diff('+vf[0]+'*r**2,'+u[0]+')') == 0:
                divResult += sympify('4*pi*DeltaDirac(x)*DeltaDirac(y)*DeltaDirac(z)*'+vf[0]+'*r**2')
    except:
        divResult = divergenceErrorMessage
    return divResult

def curl(vf,coordSys):
    curlResult=[0,0,0]
    j = coordSysType[coordSys]
    h = scaleFactorCoordSys[j]
    u = varCoordSys[j]
    prodh = h[0]+'*'+h[1]+'*'+h[2]
    numDim = len(u)
    try:
        for i in range(numDim):
            if i > 0:
                iprev = i-1
            else:
                iprev = numDim-1
            if i < (numDim-1):
                inext = i+1
            else:
                inext = 0
            curlResult[i] = sympify('(diff('+h[iprev]+'*'+vf[iprev]+','+u[inext]+')-diff('\
                            +h[inext]+'*'+vf[inext]+','+u[iprev]+'))*('+h[i]+'/('+prodh+'))')
    except:
        curlResult = curlErrorMessage
    return curlResult

def laplacian(f,coordSys):
    laplacianResult=0
    j = coordSysType[coordSys]
    h = scaleFactorCoordSys[j]
    u = varCoordSys[j]
    prodh = h[0]+'*'+h[1]+'*'+h[2]
    try:
        for i in range(len(u)):
            laplacianResult += sympify('diff(('+prodh+'/('+h[i]+'**2))*diff('+f+','+u[i]+'),'+u[i]+')/('+prodh+')')
    except:
        laplacianResult = laplacianErrorMessage
    return laplacianResult


if __name__ == "__main__":
    #This function means this was run directly, not called from another python file.
    app = QtGui.QApplication(sys.argv)
    app.setOrganizationDomain("RobertoColistete.net")
    app.setApplicationName("Derivative")

    from platform import python_version
    from sympy import *
    from sympy import __version__
    from sympy.interactive.printing import init_printing
    from sympy.printing.mathml import mathml

    myapp = DerivativeWindow()
    myapp.show()
    sys.exit(app.exec_())
