/*
 * Biloba
 * Copyright (C) 2004-2005 Guillaume Demougeot, Colin Leroy
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

/**
 * Biloba - Q1 2005
	* Game by Guillaume Demougeot <dmgt@wanadoo.fr>
	* Code by Colin Leroy <colin@colino.net>
	*
	* This file contains all functions useful to any other
	* file
	*/

#include <SDL.h>
#include <SDL_image.h>
#ifdef MAEMO
#include <SDL_syswm.h>
#include <X11/Xutil.h>
#endif
#include <stdlib.h>
#include <string.h>
#include "utils.h"
#include "board.h"
#include "logic.h"
#include "keyboard.h"
#ifdef __MINGW32__
#include <windows.h>
#endif

char *progpath = NULL;
char *langpath = NULL;

int get_x(int x)
{
	return ((50 * x) + X_OFFSET);	
}

int get_y(int y)
{
	return ((50 * y) + Y_OFFSET);
}

void put_image(SDL_Surface *surface, int x, int y)
{
	static SDL_Rect *tmp_rect = NULL;
	
	assert(surface != NULL);

	if (tmp_rect == NULL)
		tmp_rect = malloc(sizeof(SDL_Rect));
	tmp_rect->x = x;
	tmp_rect->y = y;
	if (!board_frozen())
		SDL_BlitSurface(surface, NULL, screen, tmp_rect);	
}

int wait_key(void)
{
	SDL_Event event;
	int code = 0;
	do {
		SDL_WaitEvent(&event);
		if (event.type == SDL_KEYUP) {
			SDL_KeyboardEvent *kevent = &event.key;
			code = kevent->keysym.sym;
		}
	} while (code == 0);
	
	return code;
}

static int num_players = 2;
int game_num_players(void)
{
	return num_players;
}

void set_num_players(int num)
{
	assert(num > 1 && num < 5);
	num_players = num;
}

#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif

int min (int a, int b)
{
	if (a < b)
		return a;
	else 
		return b;
}

int max (int a, int b)
{
	if (a > b)
		return a;
	else 
		return b;
}

int is_in_box(int x, int y, int a, int b, int c, int d)
{
#ifdef DEBUG
	printf("is_in_box(%d,%d,%d,%d,%d,%d): (%d,%d,%d,%d)\n",
			x,y,a,b,c,d, min(x,a) == a, min(y,b) == b,
			max(x,c) == c, max(y,d) == d);
#endif
	return (min(x,a) == a && min(y,b) == b &&
		max(x,c) == c && max(y,d) == d);
}

static int playing = FALSE;
void set_playing(int play) 
{
	playing = play;
}

int is_playing(void)
{
	return playing;
}

static SDL_mutex *input_mutex = NULL;

void init_mutexes(void)
{
	if (input_mutex == NULL)
		input_mutex = SDL_CreateMutex();	
}

SDL_Event get_sdl_event(int event_type)
{
	SDL_Event event;
	
	SDL_LockMutex(input_mutex);
	
	do {
		SDL_WaitEvent(&event);
		if (event.type != event_type) {
			if (event.type == SDL_KEYUP) {
				SDL_KeyboardEvent *kevent = &event.key;
				keyboard_push_event(kevent->keysym.sym, kevent->keysym.mod);
			}
			if (event.type == SDL_USEREVENT)
				break; /* push it anyway */
			if (event.type == SDL_QUIT) {
				stop_game();
				break;
			}
		}
		if (game_ended())
			break;
	} while (event.type != event_type);

	SDL_UnlockMutex(input_mutex);

	return event;
}

static int squit = FALSE;

void notify_quit(void)
{
	squit = TRUE;
}

int should_quit(void)
{
	return squit;
}

static int is_inited = FALSE;

void game_init(int bool)
{
	is_inited = bool;
}

int game_inited(void)
{
	return is_inited;
}

static int suspend = FALSE;

void game_suspend(int i)
{
	suspend = i;
}
int game_suspended(void)
{
	return suspend;
}

#ifndef __MINGW32__
void set_language(void)
{
	if (getenv("LANG") == NULL)
		langpath = "en";
	else if (!strncmp(getenv("LANG"), "fr", 2))
		langpath = "fr";
	else if (!strncmp(getenv("LANG"), "en", 2))
		langpath = "en";
	else if (!strncmp(getenv("LANG"), "es", 2))
		langpath = "es";
	else 
		langpath = "en";
}
#else
void set_language(void)
{
	LCID lcid;
	int primary;
	lcid = GetThreadLocale();
	primary = PRIMARYLANGID (LANGIDFROMLCID (lcid));

	if (primary == LANG_FRENCH)
		langpath = "fr";
	else if (primary == LANG_ENGLISH)
		langpath = "en";
	else
		langpath = "en";

}
#endif

#undef DEBUG_IMG_LOAD
static SDL_Surface *load_image_prefix(const char *prefix, const char *name)
{
	SDL_Surface *surface = NULL;
	char *path = malloc(strlen(prefix)+ strlen (DIR_SEP) + strlen(name) 
			+ strlen("en") + strlen(DIR_SEP) + 1);
#ifdef DEBUG_IMG_LOAD
	FILE *dbgfp = fopen("debug.txt", "a");
#endif

	if (langpath == NULL) {
		set_language();
		assert(langpath != NULL);
	}
	
	strcpy(path, prefix);
	strcat(path, DIR_SEP);
	strcat(path, langpath);
	strcat(path, DIR_SEP);
	strcat(path, name);
#ifdef DEBUG_IMG_LOAD
	fprintf(dbgfp, "loading from %s", path);
#endif
	surface = IMG_Load(path);
	
#ifdef DEBUG_IMG_LOAD
	if (surface)
		fprintf(dbgfp, ": OK\n");
	else
		fprintf(dbgfp, ": not OK\n");
	fclose(dbgfp);
#endif
	free(path);

	return surface;
}

SDL_Surface *biloba_load_image(const char *name)
{
	SDL_Surface *result = NULL;
	
	result = load_image_prefix(PREFIX DIR_SEP "res", name);
	
	if (!result)
		result = load_image_prefix(PREFIX, name);
	
	if (!result && progpath != NULL)
		result = load_image_prefix(progpath, name);
	
	return result;
}

#ifdef MAEMO
void SetWMName(const char *name, int is_fullscreen)
{
        SDL_SysWMinfo info;
        SDL_VERSION(&info.version);
        if ( SDL_GetWMInfo(&info) ) {
                Display *dpy = info.info.x11.display;
                Window win;
		
		if (is_fullscreen)
			win = info.info.x11.fswindow;
		else
			win = info.info.x11.wmwindow;
		XStoreName(dpy, win, name);
        }
}
#endif
