//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006,2007,2008,2009 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//

#define _GNU_SOURCE
#include <stdio.h>

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <unistd.h>

#include "atsend.h"
#include "btio.h"
#include "cmds.h"
#include "conf.h"
#include "common.h"
#include "parse.h"
#include "utils.h"
#include "xemulate.h"

extern void closePort(int final);
extern int  isConnected;

extern int  	    serverMode;
extern char 	    callerId[MAXLEN];
extern int  	    repeatNow;
extern type_key*    repeatCmdPtr;
extern char         tmp[MAXMAXLEN];

int macroCmd(char *descr, char *exec, cmdParams* params);
 
char tmptext[MAXCMDLEN];
int  remoteOn       = 1;

// This is not clean-up function. It just finished the connection.
static void finishAll()
{
        printf("Finish events forwarding\n");
        logger("INFO","Finish events forwarding");

        if (serverMode == CLIENT_AT ) {
		sendCMER(CMER_OFF);
        }
	
        closePort(1);
        return;
}

static int execInFork(const char *cmd)
{
        int cpid = fork();
        if( cpid < 0) {
        	logger("ERROR", "Can not fork!");
                return -1;
        }

        if(cpid) {      // father
        	int rf = 0;
        	waitpid(cpid,&rf,0);
                if(rf!=-1 && WIFEXITED(rf)) {
                	rf = WEXITSTATUS(rf);
                }
                return rf;
        } 
        
        // else - child
        int rc = 0;
        //logger("INFO", "Close port in child");
        closePort(0);
         
        rc = system(cmd);
        if(rc!=-1 && WIFEXITED(rc)) {
        	rc = WEXITSTATUS(rc);
        }
        exit(rc);
}

static char resBuffer[1024];

static char *readResultsFile(int * count) 
{
	FILE *fp;
	struct stat buf;
	
	char* resfile = getResFile();
        if(resfile == NULL) {
		return NULL;
	}
	
        if(stat(resfile, &buf) == -1) {
		return NULL;
	}
	
	long fLen = buf.st_size;
	DEBUG2("readResultsFile >%ld<", fLen);
	
        fp=fopen(resfile,"r");
	if (fp == NULL) {
		return NULL;
	}
    
    	char * fBuffer = NULL;
        if (fLen < 1024) {	// try to avoid memory fragmentation
        	fBuffer = resBuffer;
                memset(fBuffer, 0, 1024);
        } else {
    		fBuffer = calloc(fLen+1,1);
        	if (fBuffer == NULL) {
        		logger("ERROR","no more memory!");
			return NULL;
        	}
        }
	(*count) = fread(fBuffer, sizeof(char), fLen, fp);
    	fclose(fp);
    
	//logger("DEBUG","readResultsFile: exiting");
	return fBuffer;
}

static void freeResults(char * ptr) 
{
	//logger("DEBUG","freeResults");
	if (ptr != resBuffer) {
        	free(ptr);
        }
}
static char* substParams(const char* in, cmdParams *params) {


        if (in == NULL) {
                logger("ERROR","substParams(): null input");
                return NULL;
        }
        DEBUG2("substParams >%s<", in);

        if (*in == '\0') { // Empty command. Skip it.
        	return NULL;
        }
        
        char *command =  strdup(in);
        char *ptr     = command;
        char *papameterized = NULL;

        int pc = strlen(PARAM_CALLID);
        int pm = strlen(PARAM_MODE);
        int pp = strlen(PARAM_PARAM);
        int pi = strlen(PARAM_INDEX);
        //int pd = strlen(PARAM_CFGDIR);
        //int ph = strlen(PARAM_HOME);
        int pb = strlen(PARAM_BTADDR);
        
        // calculate size to allocate
        int needSize = strlen(ptr)+1; 
        int nParams  = 0;
        while(ptr != NULL) {
                papameterized = strstr(ptr, PARAM_START);   // $(
                
                if (papameterized == NULL) { // Just copy
                        break;
                } else if(strncmp(papameterized+2,PARAM_CALLID,pc) == 0) {
                	needSize += strlen(callerId) - pc - 2;
                        ptr = papameterized + pc + 2;
                        nParams++;
		} else if (strncmp(papameterized + 2,PARAM_MODE,pm) == 0) {
                	char *m = getMode();
                        needSize += strlen(m) - pm - 2;	
                        ptr = papameterized + pm + 2;
                        nParams++;
                } else if (strncmp(papameterized + 2,PARAM_PARAM,pp) == 0) {
                        if (params != NULL && params->value != NULL) {   // value can be empty
                        	needSize += strlen(params->value) - pp - 2;
                	}
                        ptr = papameterized + pp + 2;
                        nParams++;
                } else if (strncmp(papameterized + 2,PARAM_INDEX,pi) == 0) {
                        if (params != NULL && params->index != NULL && params->index[0] != '\0') {
                        	needSize += strlen(params->index) - pi - 2;
                        }
                        ptr = papameterized + pi + 2;
                        nParams++;
                /*} else if (strncmp(papameterized + 2,PARAM_CFGDIR,pd) == 0) {
			int c = 0;
                        char *d;
                        if ((d = getCfgDir()) != NULL) {
				c = strlen(d);
                                needSize += c - pd - 2;
                        }
                        ptr = papameterized + pd + 2;
                        nParams++;
                } else if (strncmp(papameterized + 2,PARAM_HOME,ph) == 0) {
			int c = 0;
                        char *d;
                        if ((d = getenv("HOME")) != NULL) {
				c = strlen(d);
                                needSize += c - ph - 2;
                        }
                        ptr = papameterized + ph + 2;
                        nParams++;*/
                } else if (strncmp(papameterized + 2,PARAM_BTADDR,pb) == 0) {
			int c = 0;
                        char *d;
                        if ((d = getBtAddress()) != NULL) {
				c = strlen(d);
                                needSize += c - pb - 2;
                        }
                        ptr = papameterized + pb + 2;
                        nParams++;
                } else {
                	int pLen = (papameterized == NULL ? 0 : 1);
                        if (pLen) {
                        	char * br = index(papameterized+2,')');
                        	pLen = (br == NULL ? 0 : br - papameterized - 2);
                        }
			
                	varData* var = NULL;

                        if ((var = searchVar(papameterized + 2, pLen)) != NULL) {
                
                        	logger("DEBUG", "searchVar() OK");
                        
                		needSize += (var == NULL ? 0 : var->size - strlen(var->name) - 3); 
        
				sprintf(tmp, "Add bytes: %d", (int) (var->size - strlen(var->name) - 3));
        			logger("DEBUG",tmp);
                                
                                ptr = papameterized + strlen(var->name) + 3;
                                nParams++;
               	 	} else {
                		break;
                        }
        	}
        }
        
        if (nParams == 0) {		// non-parametrized command
        	return command; 
        }

        papameterized = NULL;
        ptr = command;
        
        DEBUG2("Allocate >%d< bytes for parametrized command", needSize);

	char* cmd  = (char*) calloc(needSize+1,1);
	int  count = 0;
       
        while(ptr != NULL) {
        	DEBUG2("cur cmd >%s<", ptr);

                papameterized = strstr(ptr, PARAM_START);   // $(
        
                if (papameterized != NULL) {
                	DEBUG2("papameterized >%s<", papameterized);
                }
        
                if (papameterized == NULL) { // Just copy
                	logger("DEBUG", "no more parameters");
                        strcpy(cmd+count, ptr);
                        break;
                } else if(strncmp(papameterized+2,PARAM_CALLID,pc) == 0) {
                	
                        *papameterized = '\0';
			
                        strcpy(cmd+count, ptr);		// First part
                        strcat(cmd+count, callerId);    // Add params
                
                        count += strlen(ptr) + strlen(callerId);
                        ptr = papameterized + pc + 2;
		     
                } else if (strncmp(papameterized + 2,PARAM_MODE,pm) == 0) {
                	
                        *papameterized = '\0';
                        char *m = getMode();
                       
                        strcpy(cmd+count, ptr); 	// First part
                        strcat(cmd+count, m);		// Add params
                                
                        count += strlen(ptr) + strlen(m);
                        ptr = papameterized + pm + 2;
                        
                
                } else if (strncmp(papameterized + 2,PARAM_PARAM,pp) == 0) {
                	// This is a trick to make Bemused commands like VOLM<volume> works
                        // also it is used in Java Client list handling
                
                        *papameterized = '\0';

                        if (params == NULL) {   // value can be empty
                        	logger("WARNING", "No parameters supplied!");

                        	strcpy(cmd+count, PARAM_START);
                        	strcat(cmd+count, PARAM_PARAM);
                                
                                count += strlen(PARAM_START) + strlen(PARAM_PARAM);
                        } else {
                        	strcpy(cmd+count, ptr);			// First part
                        	strcat(cmd+count, params->value);	// Add params
                                
                                count += strlen(ptr) + strlen(params->value);
                	}
                        ptr = papameterized + pp + 2;

                } else if (strncmp(papameterized + 2,PARAM_INDEX,pi) == 0) {
                	// Used in Java Client list handling
                
                        *papameterized = '\0';

                        if (params == NULL || params->index == NULL || params->index[0] == '\0') {
                        	logger("WARNING", "No parameters supplied!");

                        	strcpy(cmd+count, PARAM_START);
                        	strcat(cmd+count, PARAM_INDEX);
                                
                                count += strlen(PARAM_START) + strlen(PARAM_INDEX);
                        } else {
                        	strcpy(cmd+count, ptr);			// First part
                        	strcat(cmd+count, params->index);	// Add params
                               
                                count += strlen(ptr) + strlen(params->index);
                        }
                        ptr = papameterized + pi + 2;

                /*} else if (strncmp(papameterized + 2,PARAM_CFGDIR,pd) == 0) {
                
                        *papameterized = '\0';
                
                        // First part
                        strcpy(cmd+count, ptr);
                	count += strlen(ptr);

                        // Add cfg. dir.
                        char *d;
                        if ((d = getCfgDir()) != NULL) {
				strcpy(cmd+count, d);
                        	count += strlen(d);
                        }
                
                        ptr = papameterized + pd + 2;
                
                } else if (strncmp(papameterized + 2,PARAM_HOME,ph) == 0) {
                
                        *papameterized = '\0';
                
                        // First part
                        strcpy(cmd+count, ptr);
                	count += strlen(ptr);

                        // Add $HOME
                        char *d;
                        if ((d = getenv("HOME")) != NULL) {
				strcpy(cmd+count, d);
                        	count += strlen(d);
                        }
                
                        ptr = papameterized + ph + 2;
		*/
                } else if (strncmp(papameterized + 2,PARAM_BTADDR,pb) == 0) {
                
                        *papameterized = '\0';
                
                        // First part
                        strcpy(cmd+count, ptr);
                	count += strlen(ptr);

                        // Add XX:XX:XX:XX:XX:XX
                        char *d;
                        if ((d = getBtAddress()) != NULL) {
				strcpy(cmd+count, d);
                        	count += strlen(d);
                        }
                
                        ptr = papameterized + pb + 2;
                
                } else { 

                	int pLen = (papameterized == NULL ? 0 : 1);
                        if (pLen) {
                        	char * br = index(papameterized+2,')');
                        	pLen = (br == NULL ? 0 : br - papameterized - 2);
                        }
			
			varData* var = NULL;
			
                	if ((var = searchVar(papameterized + 2, pLen)) != NULL) {
                        	logger("DEBUG", "substitute: searchVar() OK");
                           
                        	*papameterized = '\0'; 
                                strcpy(cmd+count, ptr);		// First part
				count += strlen(ptr);
				
                                if (var->size > 0) {
                        		memcpy(cmd+count, var->value, var->size);	// Add params
                                        count += var->size;
                        	}
                                
        			DEBUG2("cmd var value >%s<", cmd);
                                
                        	ptr = papameterized + strlen(var->name) + 3;
                
                	} else {    // Just copy
        			//sprintf(tmp, "Just copy >%s<", ptr);
        			//logger("DEBUG",tmp);
                		strcpy(cmd+count, ptr);
                                count += strlen(ptr);
                        	break;
                        }
                }
        }
        DEBUG2("final cmd >%s< %d", cmd,count);
        
        free(command);
        
        return cmd;
}

static int execSimpleCmd(const char *cmd, int toFile)
{
	if (cmd == NULL || remoteOn != 1) {
        	logger("DEBUG", "execSimpleCmd null input or remote if OFF");
        	return EXIT_OK;
        }
        DEBUG2("execSimpleCmd >%s<", cmd);

        int add = 0;
        if (toFile) { 
                add = strlen(getResFile()) + 4;
        } else {
        	add = 2;
        }
	char* p = calloc(strlen(cmd) + add,1);
        if (p == NULL) {
        	return EXIT_NOK;
        }
        strcpy(p,cmd);
         
        if (toFile) {       // We needs to wait results, so do not run in background
        	strcat(p, " > ");		
                strcat(p, getResFile());
        } else {
        	if (! index(p, '&')) {
                	strcat(p, "&");
                }
        }
        int ret = execInFork(p);
        free(p);
        
        return (ret == -1 ? EXIT_NOK : EXIT_OK);
}


static int processSendCmd(char* sDescr)
{
	int ret = EXIT_OK;

        char all[MAXMAXLEN];
        strncpy(all, sDescr, MAXMAXLEN-1);

        char *tag  = strtok(all, ",");
        if (tag == NULL) {
        	logger("DEBUG", "processSendCmd(): incorrect input (1)");                
        	return EXIT_NOK;
        }
        if (strlen(sDescr) <= strlen(tag) + 1) {   // Set(tag,) or Set(tag) without any value
        	logger("DEBUG", "processSendCmd(): incorrect input (2)");                
                return EXIT_NOK;
        }
        char *ptr   = tag + strlen(tag) - 1;
        // strip spaces from tail
        while( *ptr == '\0' || *ptr == '\n' || *ptr == '\t' || *ptr == ' ') {
        	*ptr = '\0';
        	ptr--;
        }
        ptr = tag + strlen(tag) + 1;
        // strip spaces from head of value
        while( *ptr != '\0' && (*ptr == '\t' || *ptr == ' ')) {
        	ptr++;
        }

        DEBUG2("processSendCmd() is >%s< >%s<",tag,ptr);

        if (strcmp(tag, SEND_STRING) == 0) {
        	ret = writeSocketConn(ptr, strlen(ptr));
                if (ret != EXIT_OK) {
                	logger("DEBUG","Fails in writeSocketConn()");
                }
        } else if (strcmp(tag, SEND_BYTES) == 0) {
        	ret = writeBytesSocketConn(ptr);
                if (ret != EXIT_OK) {
                	logger("DEBUG","Fails in writeBytesSocketConn()");
                }    
        } else {
        	logger("DEBUG", "processSendCmd(): incorrect input (3)");                
                return EXIT_NOK;
        }
        return ret;
}

static void sendData(int type, int flag, char *data, int count)
{
        DEBUG2("sendData() >%d<",count);
        
	int ret = EXIT_OK;
	if (type == ID_SEND && flag) {
	        #ifdef USE_L2CAP
		if (serverMode == SERVER_L2CAP) {
			ret = writeBytesL2capConn(data);
		} else {
		#endif
			ret = writeBytesSocketConn(data);
		#ifdef USE_L2CAP
		}
		#endif
	} else if (count > 0) { // just check count to be sure we goes here from right place
                #ifdef USE_L2CAP
		if (serverMode == SERVER_L2CAP) {
			ret = writeL2capConn(data, count);
		} else {
		#endif
        		ret = writeSocketConn(data, count);
		#ifdef USE_L2CAP
		}
		#endif
	}
	if (ret != EXIT_OK) {
		logger("DEBUG","Fails in write(Bytes)SocketConn()");
	}
}

int execDynamically(char *command) 
{
	logger("DEBUG","execDynamically");
	cmdItem* tmpCmd = NULL;
	storeCmds(&tmpCmd, command); // parse command on-the-fly 
                
	int ret = processCommands(tmpCmd,NULL);
	freeCmds(tmpCmd);
	
 	if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
		return ret;
	}
                
	return EXIT_OK;
}

static int loadCmds(char *file) 
{
	logger("DEBUG","loadCmds");
        if (file == NULL) {
        	return EXIT_NOK;
        }

	FILE *fp=fopen(file,"r");
	if (fp == NULL) {
		return EXIT_NOK;
	}

        struct stat file_status;
	stat(file, &file_status);
	
	if (file_status.st_size <= 0) {
		logger("DEBUG","loadCmds - file is empty");
		return EXIT_NOK;
	}
	
	char *command = malloc(file_status.st_size+1);
	*command = '\0';
	
	int ret = EXIT_OK;
	while (fgets(command, file_status.st_size+1, fp) != NULL) {
        	DEBUG2("Retrieved line %s of length %d ", command, strlen(command));
		ret = execDynamically(command); 	
        }
        free(command);
        fclose(fp);
	return ret;
}

int handleHook(int hook)
{
	DEBUG2("handleHook() >%d<",hook);
	char hookFile[MAXLEN];
	strcpy(hookFile, getenv("HOME"));
	strcat(hookFile, "/.anyRemote/");

	if (hook == ID_EVT_INIT) {
		strcat(hookFile, "hook_init");
	} else if (hook == ID_EVT_CONNECT) {
		strcat(hookFile, "hook_connect");
	} else if (hook == ID_EVT_DISCONNECT) {
		strcat(hookFile, "hook_disconnect");
	} else if (hook == ID_EVT_EXIT) {
		strcat(hookFile, "hook_exit");
	} else {
		return EXIT_NOK;	
	} 
	return loadCmds(hookFile);	
}

int processTimerCmd(cmdItem* ci)
{
	if (ci->exec == NULL && ci->tdata == NULL) {
        	logger("DEBUG", "processTimerCmd(): nothing to exec or timer data absent");
                return EXIT_NOK;
        }
        
        if (ci->tdata == NULL) {	// Timer(key,cancel|pause|continue|reset|restart)
        	timerCmd *tm = findTimer(ci->descr, NULL);
        	if ( tm == NULL) {
        		logger("DEBUG", "processTimerCmd(): timer did not exists");                
                	return EXIT_NOK;
                } else {
                	sprintf(tmp, "processTimerCmd() change status of %s timer",ci->descr);
                	logger("DEBUG", tmp); 
                        
                        if (strcmp(ci->exec,"pause") == 0) {
                        	logger("DEBUG", "processTimerCmd(): new status is paused");
                                tm->status = TIMER_PAUSE;
                        } else if (strcmp(ci->exec,"cancel") == 0) {
                        	logger("DEBUG", "processTimerCmd(): cancel this timer");
                        	cancelTimer(ci->descr); 
                        } else if (strcmp(ci->exec,"continue") == 0) {
                        	logger("DEBUG", "processTimerCmd(): new status is running");
                                tm->status = TIMER_RUN;
                        } else if (strcmp(ci->exec,"reset") == 0) {
                        	logger("DEBUG", "processTimerCmd(): reset timer");
        			tm->ticks  = 0;
                        } else if (strcmp(ci->exec,"restart") == 0) {
                        	logger("DEBUG", "processTimerCmd(): restart timer");
        			tm->ticks  = 0;
				tm->times  = 1;
                        } else {
                        	logger("DEBUG", "processTimerCmd(): incorrect status");
                        } 
        	} 
        } else {
        	timerCmd* tc = NULL;
		if ((tc = createTimer(ci)) == NULL) {
        		logger("DEBUG", "processTimerCmd(): timer already exists or input was incorrect");                
                	return EXIT_NOK;
        	}

		/* execute it only after timeout
		int ret = macroCmd(ci->descr,NULL,NULL); 
        	if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
        		return ret;
        	}*/
	}
        return EXIT_OK;
}

extern mode       *currentMode;

static int includeCmd(char *file) 
{
        if (file == NULL) {
		return EXIT_NOK;
        }
        //sprintf(tmp, "includeCmd %s", file);
	//logger("DEBUG",tmp);

	// Will automatically run (Connect) command sequence
        // if it will be created in load_cfg() 
        int i = 0;    
        int isNew = (currentMode == NULL ? 1 : 0);
	
        if (load_cfg(file, 0) == -1) {
		logger("ERROR","Can not include file !");
	}
        
        printConf();
        
        DEBUG2("isNew %d isConnected %d", isNew, isConnected);
        
        if (isNew == 1 && isConnected == 1) {
        	i = 0;    
		type_key *k = findItem(EVT_CONNECT, &i, NULL);
		if (k && i == FLAG_EXACT) {
			logger("INFO", "Exec cmd on connect");
			int ret = handleCmdByKey(k,NULL);
			if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
				return EXIT_NOK;
			}
		}
        }
        
	//sprintf(tmp,"Mode is %s", getMode());
	//logger("DEBUG",tmp);
	return EXIT_OK;
}

static int setModeCmd(const char *sDescr) 
{
	int i = 0;
        int ret;    
        type_key* k = findItem(EVT_EXITMODE, &i, NULL);
        if (k && i == FLAG_EXACT) {
                 logger("DEBUG", "Exec cmd on exit from mode");
                 ret = handleCmdByKey(k,NULL);
                 if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
                 	return ret;
                 }
        } else {
         	logger("DEBUG", "Leave mode");
        }
        setMode(sDescr);

        i = 0; 
        k = findItem(EVT_ENTRMODE, &i, NULL);
        if (k && i == FLAG_EXACT) {
                 logger("DEBUG", "Exec cmd on enter to mode");
                 ret = handleCmdByKey(k,NULL);
                 if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
                 	return ret;
                 }
        } else {
         	logger("DEBUG", "Entered to mode");
        }
        return EXIT_OK;
}

static int getCmd(const char *sDescr) 
{
	if (sDescr == NULL) return EXIT_NOK;
	
        strcpy(tmptext, CMD_GET);
        strcat(tmptext, "(");
        strcat(tmptext, sDescr);
        strcat(tmptext, ");");
	
        if (serverMode == SERVER_BT  ||
            serverMode == SERVER_TCP || 
            serverMode == SERVER_UX  || 
            serverMode == CLIENT_NOAT) {
             

        	int ret = writeSocketConn(tmptext, strlen(tmptext));
        	if (ret != EXIT_OK) {
         		logger("DEBUG","Fails in writeSocketConn() (Get)");
         	}
                return ret;
        }  
	#ifdef USE_L2CAP  
        if (serverMode == SERVER_L2CAP) {
             
        	int ret = writeL2capConn(tmptext, strlen(tmptext));
        	if (ret != EXIT_OK) {
         		logger("DEBUG","Fails in writeL2capConn() (Get)");
         	}
                return ret;
        } 
	#endif   
	return  EXIT_OK;
}

static int makeCmd(cmdItem* ci, cmdParams* p) 
{
	int ret = EXIT_OK;
        
        if (ci->descr == NULL) {
        	logger("DEBUG", "Empty command description");
                return EXIT_NOK;
        }
        
	if (strstr(ci->descr,MAKE_VAR) == ci->descr) {	
               
		logger("INFO", "Command: Make(var,...)");
                
                char *eValue = ci->exec;
                if (ci->exec && strstr(ci->exec,"by_value") == ci->exec) {
                	eValue = (ci->exec + 9);
                }
        
		char* sExec = substParams(eValue, p);
                //printf("AAA: %s\n", sExec ? sExec : "NULL");
                
                int resLen = -1;
                char *res  = NULL;
                
		if (eValue == ci->exec) {
                	if(sExec == NULL) {
		        	logger("DEBUG", "Empty Make(var,...) command after substParams()");
		        	return EXIT_NOK;
			}
                
        		ret = execSimpleCmd(sExec,1);
        		if (ret != EXIT_OK) {
                		DEBUG2("Return code from execSimpleCmd() is %d",ret);
                		free(sExec);
                		return ret;
        		}
        
        		res  = readResultsFile(&resLen);
        		if (res == NULL) {
        			logger("ERROR", "readResultsFile() returns NULL");
                		free(sExec);
                		return EXIT_NOK;
        		}

			if (*(res+resLen-1) == '\n') {
				resLen--;
                        }
		} else {	// set "by_value"
                	if (sExec) {
                		resLen = strlen(sExec);
                        	res    = strdup(sExec);
                        } else {
                        	resLen = 0;
                        }
                }
		
		ret = setVar(ci->descr+4,res,resLen);
		if (ret != EXIT_OK) {
		        logger("DEBUG","Fails in setVar() (Make)");
		}
                
                freeResults(res); res = NULL;
		free(sExec); sExec = NULL;
               
        } else if (strstr(ci->descr,MAKE_MODE) == ci->descr) {
        
		logger("INFO", "Command: Make(mode,...)");
                
		char* sExec = substParams(ci->exec, p);
                
                ret = setModeCmd(sExec); 
		
		free(sExec); sExec = NULL;               
        
        } else if (strstr(ci->descr,MAKE_REMOTE) == ci->descr) {
        
        	if (ci->exec && strcmp(ci->exec,"on")==0) {
			logger("INFO", "Command: Make(remote,on)");
                	remoteOn = 1;
                } else if (ci->exec && strcmp(ci->exec,"off")==0) {
			logger("INFO", "Command: Make(remote,off)");
                	remoteOn = 0;
        	}
                
        } else if (strstr(ci->descr,MAKE_DISCONN) == ci->descr) {
        	
                logger("INFO", "Temporary disconnect command processed");
         	finishAll();
         	return EXIT_DISCON; // But what with the rest of the command line ? (now just skip)
        
        } else if (strstr(ci->descr,MAKE_FLUSH) == ci->descr) {
        	
                ret = flushData();
        
	} else if (strstr(ci->descr,MAKE_STOP) == ci->descr) {
        	
                return EXIT_STOP;
	
        } else if (strstr(ci->descr,MAKE_EXIT) == ci->descr) {
        	
        	logger("INFO", "Exit command processed");
                finishAll();                        
	        return EXIT_ABORT;
	
        //} else if (strstr(ci->descr,MAKE_NONE) == ci->descr) {
        //        ;
        }
                
	return ret;
}

static int ckpdCmd(const char *sDescr) 
{
	if (serverMode == CLIENT_AT) {
        	int ret = sendSeq(sDescr);
        	if (ret != EXIT_OK) {
         		logger("DEBUG","Fails in sendSeq()");
                        return ret;
         	}
        }    
	return  EXIT_OK;
}

static void uploadPix(char *cmdLine)
{
        DEBUG2("uploadPix >%s<", cmdLine);
	int prefixSz = 0;          		// image,icon,_name_,_file_ or image,window,_file_ or cover, _file_
        
        if (strstr(cmdLine,UPLOAD_PIX)   != NULL) {
        	prefixSz = strlen(UPLOAD_PIX) + 1;
        } else if (strstr(cmdLine,UPLOAD_COVER) != NULL) {
        	prefixSz = strlen(UPLOAD_COVER) + 1;
        } else {
        	int nCh = strlen(UPLOAD_ICON) + 1;
       		prefixSz = nCh + strstr(cmdLine+nCh,",") - (cmdLine+nCh) + 1;
	}
        
	char* fname = cmdLine + prefixSz;
        char* p = fname;
        while (*p != '\0') {	// remove trailing \n if exists
        	if (*p == '\n') {
                	*p = '\0';
                        break;
                }
                p++;
        }
        //sprintf(tmp, "uploadPix file >%s<", fname); 
        //logger("INFO",tmp);

	if (strstr(cmdLine,UPLOAD_COVER) != NULL && 
            isDataNew(ID_COVER, fname, strlen(fname)) == EXIT_NOK) {
            
        	logger("INFO","Skip upload same cover");
                return;   
	}        

	FILE *fp;
	struct stat buf;
	
	long fLen = 0;
        if(stat(fname, &buf) == -1) {	
        	logger("ERROR","can't get file size!");
		if (strstr(cmdLine,UPLOAD_COVER) == NULL) {	// in case UPLOAD_COVER we will send empty image to clear previous one
			return;
		} else {
                	logger("DEBUG","uploadPix(): send empty cover");
			int ret = writeSocketConn("Set(cover);",11);
			if (ret != EXIT_OK) {
				logger("DEBUG","uploadPix(): Fails in writeSocketConn(empty cover)");
			}
			return;
		
		}
	} else {
		fLen = buf.st_size;
	}
	
	if (!S_ISREG (buf.st_mode)) {
		logger("ERROR","not regular file");
		return;
	}
	
	//sprintf(tmp, "uploadPix >%ld<", fLen);
	//logger("DEBUG",tmp);
	
        fp=fopen(fname,"r");
	if (fp == NULL) {
        	logger("ERROR","can't open file!");
		return;
	}
        
    	uint32_t szh32 = (uint32_t) fLen;
    	uint32_t szi32 = htonl(szh32);
    
    	char * fBuffer = NULL;
    	fBuffer = calloc(fLen+prefixSz+12,1);	// 11 =  "Set("     ,<size of bin data=4bytes>    ");\0"
        if (fBuffer == NULL) {
        	logger("ERROR","no more memory!");
		return;
        }
        
    	strcpy(fBuffer, "Set(");
    	strncat(fBuffer, cmdLine, prefixSz+1);
        memcpy((void*)fBuffer+prefixSz+4, (const void *) &szi32, 4);	// length on binary data
        
	if (fp) {
		fread(fBuffer+prefixSz+8, sizeof(char), fLen, fp);
    		fclose(fp);
	}
        strcpy(fBuffer+prefixSz+fLen+8,");");
        
	int ret = writeSocketConn(fBuffer,prefixSz+fLen+10);
	if (ret != EXIT_OK) {
		logger("DEBUG","uploadPix(): Fails in writeSocketConn()");
	}

        free(fBuffer);
	
        return;
}


static char* executeCommand(const char* exec, int* resLen, cmdParams *params) 
{
	//logger("DEBUG","executeCommand");
        
        char* sExec = substParams(exec, params);
        if (sExec == NULL) {
        	logger("DEBUG", "Empty exec. part after substParams()");
                return NULL;
        }

        int ret = execSimpleCmd(sExec,1);
        if (ret != EXIT_OK) {
                DEBUG2("Return code from execSimpleCmd() is %d",ret);
                free(sExec);
                return NULL;
        }
        
        *resLen = -1;
        char *res  = readResultsFile(resLen);
        
	free(sExec);
        
        return res;
}

int macroCmd(char *descr, char *exec, cmdParams* params) 
{
	logger("DEBUG","macroCmd/timerCmd");
        
        if (descr == NULL) {
        	return EXIT_NOK;
        }
        
        int runIt = 1;
        if (exec != NULL) {
        	int resLen = -1;
                
                if (exec && strstr(exec,"if_mode_at") == exec) {
			if (serverMode != CLIENT_AT) {
                        	runIt = 0; 
                        }                   	
                } else if (exec && strstr(exec,"if_mode_server") == exec) {
			if (serverMode != SERVER_BT  &&
			    #ifdef USE_L2CAP
                            serverMode != SERVER_L2CAP &&
			    #endif
                            serverMode != SERVER_TCP &&
                            serverMode != SERVER_UX  &&
                            serverMode != CLIENT_NOAT) {
                        	runIt = 0; 
                        }                   	
                } else if (exec && strstr(exec,"if_mode_stdin") == exec) {
			if (serverMode != SERVER_STDIN) {
                        	runIt = 0; 
                        }                   	
                } else if (exec && strstr(exec,"if_mode_ilirc") == exec) {
			if (serverMode != CLIENT_ILIRC) {
                        	runIt = 0; 
                        }                   	
		} else {
                
        		char* res = executeCommand(exec, &resLen, params);
		
        		if (res == NULL) {
                		runIt = 0;
                	} else {
        			if (strcmp(res,"0") != 0 && strcmp(res,"0\n") != 0) {	// test return code $?
        				runIt = 0;
                        	}
                        	freeResults(res);
                	}
                }	
        }
        
        if (runIt == 0) {
        	logger("DEBUG","macroCmd condition not met");
                return EXIT_OK;
        }
        
 	int ret;
	type_key *k = findItem(descr, &ret, NULL);

	if (k && ret == FLAG_EXACT) {
		ret = handleCmdByKey(k, params);
		if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
			return ret;
		}
	}
	return EXIT_OK;
}

static int execCmdAndGetResults(int type, cmdItem* ci, timerCmd* timer, cmdParams *params) 
{
	logger("DEBUG","execCmdAndGetResults");
        
        if (serverMode == CLIENT_AT    || 
	    serverMode == CLIENT_ILIRC ||
	    remoteOn != 1 || ci->exec == NULL) {
        	return EXIT_OK;
        }
        
	//printf("TRACE %s->%s\n",ci->descr,(ci->exec ? ci->exec : "NULL"));

        char* sDescr = substParams(ci->descr, params);
        if (sDescr == NULL) {
        	logger("DEBUG", "Empty command after substParams()");
                return EXIT_NOK;
        }
        
        int resLen = -1;
        char *res = executeCommand(ci->exec, &resLen, params);
        if (res == NULL) {
        	logger("ERROR", "executeCommand() returns NULL");
                free(sDescr);
                return EXIT_NOK;
        }

        if (type == ID_SEND) {  // Just send string
        	
                int bytesFlag = 0;
                if (strstr(sDescr, SEND_BYTES)) {
                	bytesFlag = 1;
                }
                sendData(type, bytesFlag, res, resLen);
                
        } else {                // ID_SET
		//logger("DEBUG", "ID_SET");

                if (strstr(sDescr,UPLOAD_ICON)   == sDescr || 
                    strstr(sDescr,UPLOAD_PIX)    == sDescr || 
                    strstr(sDescr,UPLOAD_COVER)  == sDescr) {
                                    
                    	char *chunk = calloc(strlen(sDescr)+resLen+2,1);
                        
                        strcpy(chunk,sDescr);
                        strcat(chunk,",");
                        memcpy(chunk+strlen(sDescr)+1,res,resLen);
                       
                	uploadPix(chunk);
                        
                        free (chunk);

                } else {
                
                	if (isDataNew(cmdSet2id(sDescr), res, resLen)) {
                                logger("INFO", "Skip to send the same data");
                        } else {
				//logger("INFO", "Data is new");
                                
                		// Replace all ");" inside buffer to "@@" since Java Client will separate commands by ");"
                		char *p;
                		while ((p = strstr(res,");")) != NULL) {	// binary data ???
                        		*p = '@';
                        		p++;
                        		*p = '@';
                		}
                
                		int memSize = resLen + strlen(sDescr) + 7;
                		char *chunk = calloc(1,memSize+1);
                		strcpy(chunk,CMD_SET);

                		strcat(chunk,"(");  // For example, Set(title
                		strcat(chunk,sDescr);
                
                		strcat(chunk,",");
                		memcpy(chunk+5+strlen(sDescr),res,resLen);	// could be binary data here
                		strcpy(chunk+5+strlen(sDescr)+resLen,");");
                	
                		//printf("CHUNK: %s\n", chunk);
                
                		sendData(type, 0, chunk, memSize);
                 
               			free (chunk);
                        }
        	}
  	}
        
	freeResults(res);
	free(sDescr);

        //logger("INFO","execCmdAndGetResults finished");
        return EXIT_OK;
}

static int processOneCommand(cmdItem* ci, cmdParams* p)
{
	 int ret = 0;

         if (ci->exec == NULL || ci->descr == NULL) {
         	if (ci->descr == NULL) {
	         	sprintf(tmp, "processOneCommand >no description< >%d<", remoteOn);
                }        
         	if (ci->exec == NULL) {
	         	sprintf(tmp, "processOneCommand >no exec params< >%d<", remoteOn);
                }        
         } else {
         	sprintf(tmp, "processOneCommand >%s< >%s< >%d<", ci->descr, ci->exec, remoteOn);
         }
         logger("INFO",tmp);
         
         if (ci->type == ID_MAKE && 
             strstr(ci->descr,MAKE_REMOTE) == ci->descr &&
             strcmp(ci->exec,"on") == 0) {
             
                 logger("INFO", "Switch remote on");
                 remoteOn = 1;        
                 return EXIT_OK;
         }
         
         if (remoteOn != 1) {
         	logger("INFO", "Remote is off. Skip");
                return EXIT_OK;
         }
         
         switch (ci->type) {	// Non parametrized commands
         
         	case ID_EXIT:               
             		logger("INFO", "Command: Exit");
                    	finishAll();                        
			return EXIT_ABORT;
                      
            	case ID_TIMER:   
                	logger("INFO", "Command: Timer");
                	return processTimerCmd(ci);
                        
                default:
                	break;        
        }
        
	switch (ci->type) {		// Parametrized commands, will parametrize in handler
                 
             	case ID_MAKE:          
			logger("INFO", "Command: Make");
                        return makeCmd(ci, p);

            	case ID_EXECSET: 
               		logger("INFO", "Command: ExecAndSet");
                	if (serverMode == SERVER_BT  ||
			    #ifdef USE_L2CAP 
                            serverMode == SERVER_L2CAP || 
			    #endif
                            serverMode == SERVER_TCP || 
                            serverMode == SERVER_UX  || 
                            serverMode == CLIENT_NOAT) { 
                		execCmdAndGetResults(ID_SET, ci, NULL, p);
               			logger("DEBUG", "Command: ExecAndSet FINISHED");
                        }
                	return EXIT_OK;
                 
            	case ID_EXECSEND:  
                	logger("INFO", "Command: ExecAndSend");
                	if (serverMode == SERVER_BT  || 
 			    #ifdef USE_L2CAP 
                            serverMode == SERVER_L2CAP || 
			    #endif
                            serverMode == SERVER_TCP || 
                            serverMode == SERVER_UX  || 
                            serverMode == CLIENT_NOAT) {
               			execCmdAndGetResults(ID_SEND, ci, NULL, p);
                        }
                	return EXIT_OK;

            	case ID_MACRO:   
                	logger("INFO", "Command: Macro");
                 	ret = macroCmd(ci->descr,ci->exec,p);
               		if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
                        	return ret;
                        }
               		return EXIT_OK;
                 
                default:
                	break;        
        }

        if (ci->type == ID_EXEC) {	// Parametrized command, handler will use already parametrized data 
        	logger("INFO", "Command: Exec");
                
         	char* sExec = substParams(ci->exec, p);
        	if (sExec == NULL) {
        		logger("DEBUG", "Empty command after substParams()");
        		return EXIT_NOK;
        	}

                execSimpleCmd(sExec, 0);
                free(sExec);
                
                return EXIT_OK;
        }
        
        char* sDescr = substParams(ci->descr, p);
        if (sDescr == NULL) {
        	logger("DEBUG", "Empty command after substParams()");
                return EXIT_NOK;
        }
                       
	switch (ci->type) {		// Parametrized commands, handler will use already parametrized data 
                 
             	case ID_SENDCKPD:
                	logger("INFO", "Command: SendCKPD");
                        ret = ckpdCmd(sDescr);
                 	break;
                 
            	case ID_SEND:
                	logger("INFO", "Command: Send");
                	if (serverMode == SERVER_BT  || 
			    #ifdef USE_L2CAP 
                            serverMode == SERVER_L2CAP || 
			    #endif
                            serverMode == SERVER_TCP || 
                            serverMode == SERVER_UX  ||  
                            serverMode == CLIENT_NOAT) {
                        	ret = processSendCmd(sDescr);
                	}    
                	break;
                        
            	case ID_SET:          
                	logger("INFO", "Command: Set");
                	if (sDescr != NULL &&
			    (serverMode == SERVER_BT   || 
			     #ifdef USE_L2CAP 
                             serverMode == SERVER_L2CAP || 
			     #endif
			     serverMode == SERVER_TCP  || 
			     serverMode == SERVER_UX   || 
			     serverMode == CLIENT_NOAT
                            )) {
                            
                        	strcpy(tmptext, CMD_SET);
                        	strcat(tmptext, "(");
                                
                 		//sprintf(tmp, "ID_SET >%s<", sDescr);
                 		//logger("DEBUG", tmp);
                                                 
                 		if (strstr(sDescr,UPLOAD_ICON)  == sDescr || 	// handle Set(cover|image,icon|window,[...]_image_file_name_); separately.
                    		    strstr(sDescr,UPLOAD_PIX)   == sDescr ||
                                    strstr(sDescr,UPLOAD_COVER) == sDescr) {
                                    
                                	uploadPix(sDescr); //ci->descr);
                                    
                                } else {
				
					if (isDataNew(cmdSet2id(sDescr), sDescr, strlen(sDescr))) {
                                		logger("INFO", "Skip to send the same data");
					} else {
						logger("INFO", "Data is new");
						
						// Change \n in input to real '\n'
                        			char *p1 = sDescr;
                        			char *p2 = sDescr;
                        			while (p2 != NULL) {
                                			p2 = strstr(p1, "\\n");
                         
                                			if (p2 == NULL) {
                                 				strncat(tmptext, p1, p2-p1);
                                 			} else {
                                        			strncat(tmptext, p1, p2-p1);
                                        			strcat(tmptext, "\n");
                               				}
                                			p1 = p2 + 2;
                         			}
                        			strcat(tmptext, ");");
                 
                        			ret = writeSocketConn(tmptext, strlen(tmptext));
                        			if (ret != EXIT_OK) {
                         				logger("DEBUG","Fails in writeSocketConn() (Set)");
                         			}
					}
                                }
            		}    
             		break;

             	case ID_GET:          
			logger("INFO", "Command: Get");
                        ret = getCmd(sDescr);
             		break;
                
           	case ID_LOAD:   
               		logger("INFO", "Command: Load");
                 	ret = loadCmds(sDescr);
                 	break;

           	case ID_INCLUDE:   
               		logger("INFO", "Command: Include");
                 	ret = includeCmd(sDescr);
                 	break;
                
		case ID_EMU:   
               		logger("INFO", "Command: Emulate");
                 	ret = emulateCommands(sDescr);
                 	break; 
			
		default:  
                	logger("INFO", "Unknown command");                 
                	break;
        }
	free(sDescr);
        
        if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
        	return ret;
        }
        return EXIT_OK;
}

int processCommands(cmdItem* cmdList,cmdParams* p) 
{
	while (cmdList) {
        	/*if (cmdList->descr == NULL) {
                        sprintf(tmp, "Process next command NULL parameters");
                } else {
                        sprintf(tmp, "Process next command %s",cmdList->descr);
                }
                logger("DEBUG", tmp);
                */    
                int ret = processOneCommand(cmdList, p);
                if (ret != EXIT_OK) {
                	//sprintf(tmp, "Return code from processOneCommand() is %d",ret);
                        //logger("DEBUG", tmp);
                         
                        if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
                        	logger("DEBUG", "Abort or disconnect after processOneCommand()");
                                return ret;
                        } else if (ret == EXIT_STOP) {
				return EXIT_OK; 	// just stop execution of command sequence
			}
                }
                
                cmdList = cmdList->next;
	}
        
        return EXIT_OK;
}

//
// Top-level command processing function
//
int handleCmdByKey(type_key* k, cmdParams* p)
{
	if (k == NULL) {
        	logger("DEBUG", "handleCmdByKey() NULL input");
                return EXIT_OK;
        }
        
        sprintf(tmp, "handleCmdByKey() >%s<",k->key);
        logger("DEBUG", tmp);
        
        /*if (p != NULL) {
	        sprintf(tmp,"handleCmdByKey params >%s,%s< ", p->index,p->value);
               	logger("DEBUG",tmp);
	}*/

        cmdItem* cmdList = getCommand(k);
        if (cmdList == NULL) {
        	logger("DEBUG", "No appropriate command was found by getCommand()");
        } else {
        	//sprintf(tmp, "ready to execute command");
                //logger("DEBUG", tmp);
        
                int ret = processCommands(cmdList, p);
                if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
                	repeatNow    = 0;
                        repeatCmdPtr = NULL;
			logger("DEBUG", "handleCmdByKey() EXIT_ABORT || EXIT_DISCON");
                        return ret;
                }
        
                // In AT mode: User had pressed some key ... 
                // Send ToMainMenu sequence of CKPD's to show main menu again
		// Do we send it even if remote is off ?
                if (remoteOn == 1 && serverMode == CLIENT_AT) {
                	if (sendSeq(getToMainMenu()) != EXIT_OK) {
                        	logger("DEBUG","Fails in sendSeq()");
                        }
                }
        }   

        return EXIT_OK;     
}

