/* vim:set ts=2 sw=4 sts=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the MICROB EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Oleg Romashin <oleg.romashin@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#include "gmozillaconnectivity.h"
#include "common.h"
#include "gtkmozembed_common.h"
#include "gmozillaengine.h"
#include "gmozillaweb.h"
#include "gmozillatransferitem.h"
#include <string.h>

#ifdef USE_GCONF
#include <gconf/gconf-client.h>
#endif

#ifdef USE_DBUS
#ifdef USE_CONIC
#include <osso-ic-dbus.h>
#endif
#include <dbus/dbus.h>
#endif

#ifdef USE_CONIC
#include <conic/conic.h>
#define USER_DATA_MAGIC 0xacdcacdc
static ConIcConnection *connection;
#endif

static GMainLoop *gsloop = NULL;
static GObject *webglobal = NULL;
static GObject *transferitem = NULL;
static gboolean can_connect = TRUE;
static gboolean is_connecting = FALSE;

#ifdef USE_CONIC
#define OSSO_CON_IC_CONNECTING             0x05

static void connection_cb(ConIcConnection *connection,
                          ConIcConnectionEvent *event,
                          gpointer user_data)
{
    const gchar *iap_id, *bearer;
    ConIcConnectionStatus status;
    ConIcConnectionError error;

    g_assert(GPOINTER_TO_INT(user_data) == USER_DATA_MAGIC);
    g_assert(CON_IC_IS_CONNECTION_EVENT(event));

    status = con_ic_connection_event_get_status(event);
    error = con_ic_connection_event_get_error(event);
    iap_id = con_ic_event_get_iap_id(CON_IC_EVENT(event));
    bearer = con_ic_event_get_bearer_type(CON_IC_EVENT(event));

    switch (status) {

        case CON_IC_STATUS_CONNECTED:
            connected = TRUE;
            is_connecting = FALSE;
            g_signal_emit_by_name(G_OBJECT(webglobal), "connectivity-status", status);
            return ;

        case CON_IC_STATUS_DISCONNECTED:
#ifdef USE_DBUS
        // return;
#else
        break;
#endif
        case CON_IC_STATUS_DISCONNECTING:
            break;

        default:
            break;
    }

    connected = FALSE;
    g_signal_emit_by_name(G_OBJECT(webglobal), "connectivity-status", status);
    if (gsloop && g_main_loop_is_running(gsloop))
        g_main_loop_quit (gsloop);
    gsloop = NULL;
    is_connecting = FALSE;
}
#endif

void g_mozilla_connectivity_handle (GMozillaEngine *self, gint event)
{
    TRACE_LOG();
    switch (event) {
#ifdef USE_CONIC
    case CON_IC_STATUS_CONNECTED:
        ULOG_INFO_F("CON_IC_STATUS_CONNECTED: self:%p, global:%p\n", self, webglobal);

        // fix for bug 55796
        if (transferitem) {
            if (G_MOZILLA_TRANSFER_ITEM (transferitem)->gtk_moz_embed_download)
                gtk_moz_embed_download_do_command (G_MOZILLA_TRANSFER_ITEM (transferitem)->gtk_moz_embed_download, GTK_MOZ_EMBED_DOWNLOAD_RESUME);

            // init progress timeout.
            init_progress_timeout (G_MOZILLA_TRANSFER_ITEM (transferitem));

            // setting it to NULL
            transferitem = NULL;

        } else if (self) {
            if (self->waiting_connection && self->will_open_location) {
                g_mozilla_connectivity_set_proxy();
                ULOG_INFO_F("Engine: %p: LoadUrl: '%s'\n", self->engine, self->will_open_location);
                self->waiting_connection = FALSE;
                if (!g_str_has_prefix(self->will_open_location, "file://"))
                    gtk_moz_embed_load_url(GTK_MOZ_EMBED(self->engine), self->will_open_location);
            } else if (self->progress_bar_state == 0 && !self->is_loading) {
                g_signal_emit_by_name(G_OBJECT(self), "finished_loading", 0, self->last_net_status, 15000, TRUE);
                self->progress_bar_state = -1;
            }
              /*else if (self->is_download)
                g_signal_emit_by_name(G_OBJECT(self), "finished_loading", 0, self->last_net_status, 15000, TRUE);*/
        } 
        if (gsloop && g_main_loop_is_running(gsloop))
            g_main_loop_quit (gsloop);
        gsloop = NULL;
        return;
    case CON_IC_STATUS_DISCONNECTED:
        ULOG_INFO_F("CON_IC_STATUS_DISCONNECTED, arg:%p, global:%p\n", self, webglobal);
        g_mozilla_web_pause_all_transfer_item (G_MOZILLA_WEB(webglobal));

        // from g_mozilla_engine_stop_load body
        if (self->is_loading) {
            g_mozilla_connectivity_check_buggy();
            gtk_moz_embed_stop_load(GTK_MOZ_EMBED(self->engine));
            self->is_loading = FALSE;
            //load_finished_cb(GTK_MOZ_EMBED(self->engine), self);
        }

        connected = FALSE;
        break;
    // FIXME: Handle ERRORS here, please ...
    // case OSSO_IAP_ERROR:
    //    connected = FALSE;
    //    break;
    case OSSO_CON_IC_CONNECTING:
        ULOG_INFO_F("OSSO_CON_IC_CONNECTING, arg:%p, global:%p\n", self, webglobal);
        connected = FALSE;
        if (self->progress_bar_state != 0 && self->user_iteraction_happens && self->waiting_connection)
          g_signal_emit_by_name(G_OBJECT(self), "start_loading" , 0, self->will_open_location);
        self->progress_bar_state = 0;
        return;
#endif
    }
    can_connect = FALSE;
    if (self && self->waiting_connection) {
        g_signal_emit_by_name(G_OBJECT(self), "finished_loading", 0, self->last_net_status, 15000, TRUE);
        self->progress_bar_state = -1;
        self->user_iteraction_happens = FALSE;
    }

    self->waiting_connection = FALSE;
    can_connect = TRUE;
}

#ifdef USE_DBUS
/* Called whenever ICD status changes */
static DBusHandlerResult
connectivity_cb (DBusConnection *connection,
                 DBusMessage    *message,
                 gpointer        data)
{
    TRACE_LOG ("connectivity_cb:, %p", data);
    const char *iap_name = NULL;
    const char *iap_type = NULL;
    const char *iap_state = NULL;

    DBusHandlerResult result = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    //GMozillaWeb *priv = (GMozillaWeb *)data;
#ifdef USE_CONIC
    if (!message ||
        !dbus_message_is_signal (message,
                                 ICD_DBUS_INTERFACE,
                                 ICD_STATUS_CHANGED_SIG)) {
        return result;
    }
#endif
    if (!dbus_message_get_args (message,
                                NULL,
                                DBUS_TYPE_STRING, &iap_name,
                                DBUS_TYPE_STRING, &iap_type,
                                DBUS_TYPE_STRING, &iap_state,
                                DBUS_TYPE_INVALID) || !iap_name || !iap_state) {
        return result;
    }
    TRACE_LOG ("iap_name: %s", iap_name);
    TRACE_LOG ("iap_type: %s", iap_type);
    TRACE_LOG ("iap_state: %s", iap_state);

    //       Disconnected
    if (!strcmp (iap_state, "IDLE")) {
        connected = FALSE;
#ifdef USE_CONIC
        g_signal_emit_by_name(G_OBJECT(webglobal), "connectivity-status", CON_IC_STATUS_DISCONNECTED);
#endif
        if (gsloop && g_main_loop_is_running(gsloop))
            g_main_loop_quit (gsloop);
        gsloop = NULL;
        is_connecting = FALSE;
    } else if (!strcmp (iap_state, "CONNECTED")) {
        g_mozilla_connectivity_set_proxy();
        connected = TRUE;
        is_connecting = FALSE;
        can_connect = FALSE;
#ifdef USE_CONIC
        g_signal_emit_by_name(G_OBJECT(webglobal), "connectivity-status", CON_IC_STATUS_CONNECTED);
#endif

    } else if (!strcmp (iap_state, "CONNECTING")) {
#ifdef USE_CONIC
        g_signal_emit_by_name(G_OBJECT(webglobal), "connectivity-status", OSSO_CON_IC_CONNECTING);
#endif
    }

    return result;
}
#endif

void
g_mozilla_connectivity_set_global (GMozillaWeb *self)
{
    TRACE_LOG();
    webglobal = G_OBJECT(self);
#ifdef USE_CONIC
    connection = con_ic_connection_new();
    g_assert(connection != NULL);

    g_signal_connect(G_OBJECT(connection), "connection-event",
                     G_CALLBACK(connection_cb),
                     GINT_TO_POINTER(USER_DATA_MAGIC));
#endif
#ifdef USE_DBUS
    DBusConnection *sys_conn;
    char filter[256] = {0};
    sys_conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
    dbus_connection_add_filter (sys_conn, connectivity_cb, self, NULL);
#ifdef USE_CONIC
    g_snprintf(filter, sizeof (filter),
               "type='signal',interface='%s'",
               ICD_DBUS_INTERFACE);
#endif
    TRACE_LOG("filter: %s", filter);
    dbus_bus_add_match (sys_conn, filter, NULL);
#endif /* USE_DBUS */
}

void
g_mozilla_connectivity_unset_global (GMozillaWeb *self)
{
    TRACE_LOG();
#ifdef USE_DBUS
    DBusConnection *sys_conn;
    char filter[256] = {0};
    sys_conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
#ifdef USE_CONIC
    g_snprintf(filter, sizeof (filter),
               "type='signal',interface='%s'",
               ICD_DBUS_INTERFACE);
#endif /* USE_DBUS */
    dbus_bus_remove_match(sys_conn, filter, NULL);
    dbus_connection_remove_filter(sys_conn, connectivity_cb, self);
#endif /* USE_DBUS */
}

gboolean
g_mozilla_connectivity_connect (gpointer data, gboolean reconnect)
{
    TRACE_LOG();
    //Fixme
#ifdef __arm__
    if (connected && !reconnect)
#endif
        return FALSE;

    if (gsloop)
        return TRUE;

    if (!can_connect || is_connecting)
        return FALSE;

    if (data) {
        if (G_IS_MOZILLA_ENGINE (data)) {
            webglobal = G_MOZILLA_ENGINE(data)->global;
            if (!G_MOZILLA_ENGINE(data)->user_iteraction_happens)
                return TRUE;
            G_MOZILLA_ENGINE(data)->waiting_connection = TRUE;
            if (G_MOZILLA_ENGINE(data)->progress_bar_state != 0)
                g_signal_emit_by_name(G_OBJECT(data), "start_loading" , 0, G_MOZILLA_ENGINE(data)->will_open_location);
            G_MOZILLA_ENGINE(data)->progress_bar_state = 0;

        // fix for bug 55796
        } else if (G_IS_MOZILLA_TRANSFER_ITEM (data)) {
            transferitem = data;
        }
    }
#ifdef USE_CONIC
    connected = FALSE;
    con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
    is_connecting = TRUE;
#endif

    if (gsloop) {
        g_main_loop_run(gsloop);
        g_main_loop_unref(gsloop);
    }

    return TRUE;
}

gboolean
g_mozilla_connectivity_check_buggy()
{
    if (gsloop && g_main_loop_is_running(gsloop) && is_connecting)
    {
        g_main_loop_quit (gsloop);
        gsloop = NULL;
        connected = FALSE;
        is_connecting = FALSE;
        ULOG_INFO_F("\n*******************************\n\n"
               "\tBROWSER FILL SELF BAD :(\n"
               "\tOLD CONNECTIVITY API SUXXX, need dbus implementation :(\n"
               "\n*******************************\n");
        return TRUE;
    }
    return FALSE;
}

#define SAFE_ENV_READ 100

static gboolean
g_mozilla_parse_env_proxy (const gchar *env_name, const gchar *moz_proto)
{
    g_return_val_if_fail(env_name && moz_proto, FALSE);
    TRACE_LOG(">>>>>>>>>>.Env: %s > %s", env_name, moz_proto);
    const gchar * env = getenv(env_name);
    if (!env || !strlen(env))
        return FALSE;
    gchar server[20];
    gchar port[10];
    gint i = 0, k = 0, env_len = strlen(env);
    gboolean port_parse = FALSE;
    gchar c = 0;
    for (i = 0; i < env_len || i > SAFE_ENV_READ; i++) {
        c = env[i];
        if (c == ':') {
            if (env[i+1] != '/') {
                port_parse = TRUE; server[k] = 0; k = 0; continue;
            } else {
                k = 0; continue;
            }                
        };
        if (!port_parse && k < 20 && (c == '.' || g_ascii_isalnum(c)))
            server[k++] = c;
        else if (port_parse && k < 10 && g_ascii_isdigit(c))
            port[k++] = c;
        else break;
    }
    port[k] = 0;
    gint port_int = atoi((const char*)&port);
    gchar *moz_port = g_strdup_printf("network.proxy.%s_port", moz_proto);
    gchar *moz_server = g_strdup_printf("network.proxy.%s", moz_proto);
    gtk_moz_embed_common_set_pref (G_TYPE_STRING, moz_server, &server);
    gtk_moz_embed_common_set_pref (G_TYPE_INT, moz_port, &port_int);
    g_free(moz_port);
    g_free(moz_server);
    return TRUE;
}

void g_mozilla_connectivity_set_proxy ()
{
    TRACE_LOG();
    gchar *proxytype = NULL, *autoconf = NULL;
    gchar *proxy_http = NULL, *proxy_https = NULL, *proxy_ftp = NULL;
    gint proxy_http_port, proxy_https_port, proxy_ftp_port;
    gchar *omit_proxy = NULL;
    gint gval = 0;
#ifdef USE_GCONF
    GConfClient *gconf_client;
    gconf_client = gconf_client_get_default();
    proxytype = gconf_client_get_string(gconf_client,
                                        "/system/proxy/mode",
                                        NULL);
#endif
    gval = 1;
    gtk_moz_embed_common_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
    if (getenv("http_proxy")) {
        if (g_mozilla_parse_env_proxy ("http_proxy", "http")) {
            if (!g_mozilla_parse_env_proxy ("ftp_proxy", "ftp"))
                g_mozilla_parse_env_proxy ("http_proxy", "ftp");
            if (!g_mozilla_parse_env_proxy ("https_proxy", "ssl"))
                g_mozilla_parse_env_proxy ("http_proxy", "ssl");
            gval = 1;
            gtk_moz_embed_common_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
            gtk_moz_embed_common_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_OMITTED_HOSTS_PROXY, "127.0.0.1");
        }
    } else if (g_ascii_strcasecmp(proxytype, G_MOZILLA_PROXY_NONE) == 0) {
        gval = 0;
        gtk_moz_embed_common_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
    } else if (g_ascii_strcasecmp(proxytype, G_MOZILLA_PROXY_AUTO) == 0) {
        gval = 2;
        gtk_moz_embed_common_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
#ifdef USE_GCONF
        autoconf = gconf_client_get_string(gconf_client,
                                           "/system/proxy/autoconfig_url",
                                           NULL);
#endif
        //g_print ("%s\n", autoconf);
        if (autoconf)
            gtk_moz_embed_common_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_AUTOCONFIG_URL_PROXY, autoconf);
    } else if (g_ascii_strcasecmp(proxytype, G_MOZILLA_PROXY_MANUAL) == 0) {
        gval = 1;
        gtk_moz_embed_common_set_pref (G_TYPE_INT, G_MOZILLA_PREF_PROXY_TYPE, &gval);
#ifdef USE_GCONF
        proxy_http = gconf_client_get_string(gconf_client,
                                             "/system/http_proxy/host",
                                             NULL);
#endif
        //printf("HTTP PROXY: %s\n", proxy_http);
        if (proxy_http) {
            if (g_str_has_prefix(proxy_http, "http://"))
              proxy_http += 7;
            gtk_moz_embed_common_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_HTTP_PROXY, proxy_http);
#ifdef USE_GCONF
            proxy_http_port = gconf_client_get_int(gconf_client,
                                                   "/system/http_proxy/port",
                                                   NULL);
            //printf("HTTP PORT: %d\n", proxy_http_port);
            gtk_moz_embed_common_set_pref (G_TYPE_INT, G_MOZILLA_PREF_HTTP_PROXY_PORT, &proxy_http_port);
#endif
        }
        else
            g_mozilla_parse_env_proxy ("http_proxy", "http");

#ifdef USE_GCONF
        proxy_https = gconf_client_get_string(gconf_client,
                                              "/system/proxy/secure_host",
                                              NULL);
#endif
        //printf("HTTPS PROXY: %s\n", proxy_http);
        if (proxy_https) {
            if (g_str_has_prefix(proxy_https, "https://"))
              proxy_https += 8;
            gtk_moz_embed_common_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_HTTPS_PROXY, proxy_https);
#ifdef USE_GCONF
            proxy_https_port = gconf_client_get_int(gconf_client,
                                                "/system/proxy/secure_port",
                                                NULL);
            //printf("HTTPS PORT: %d\n", proxy_https_port);
            gtk_moz_embed_common_set_pref (G_TYPE_INT, G_MOZILLA_PREF_HTTPS_PROXY_PORT, &proxy_https_port);
#endif
        }
        else
            g_mozilla_parse_env_proxy ("https_proxy", "ssl");

#ifdef USE_GCONF
        proxy_ftp = gconf_client_get_string(gconf_client,
                                            "/system/proxy/ftp_host",
                                            NULL);
        //printf("FTP PROXY: %s\n", proxy_ftp);
#endif
        if (proxy_ftp) {
            if (g_str_has_prefix(proxy_ftp, "ftp://"))
              proxy_ftp += 6;
            gtk_moz_embed_common_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_FTP_PROXY, proxy_ftp);
#ifdef USE_GCONF
            proxy_ftp_port = gconf_client_get_int(gconf_client,
                                                  "/system/proxy/ftp_port",
                                                  NULL);
            //printf("FTP PORT: %d\n", proxy_ftp_port);
#endif
            gtk_moz_embed_common_set_pref (G_TYPE_INT, G_MOZILLA_PREF_FTP_PROXY_PORT, &proxy_ftp_port);
        }
        else
            g_mozilla_parse_env_proxy ("ftp_proxy", "ftp");


#ifdef USE_GCONF

        omit_proxy = gconf_client_get_string(gconf_client,
                                             "/system/proxy/ignore_hosts",
                                             NULL);
#endif
        //printf("OMITED_HOSTS: %s\n", omit_proxy);
        if (omit_proxy)
            gtk_moz_embed_common_set_pref (G_TYPE_STRING, G_MOZILLA_PREF_OMITTED_HOSTS_PROXY, omit_proxy);
    }
    gtk_moz_embed_common_save_prefs ();
    TRACE_LOG();
}
