from utils import maemo

import gobject
import dbus, dbus.glib, dbus.service
import subprocess
import os
import fcntl
import time


_SERVICE = "de.pycage.mediabox"
_OBJECT = "/de/pycage/mediabox/MPlayer"
_IFACE = "de.pycage.mediabox.MPlayer"

_MPLAYER = "/usr/bin/mplayer"

_LOGGING = False



class MPlayerService(dbus.service.Object):
    """
    Standalone DBus service for controlling mplayer.
    """

    def __init__(self):

        self.__stdin = None
        self.__stdout = None
        self.__mplayer_pid = 0
        
        self.__xid = -1
        self.__opts = ""
        self.__needs_restart = True
        
        self.__playing = False
        self.__position = 0
        self.__total_length = 0
        self.__width = 0
        self.__height = 0

        self.__heartbeat = False
        

        bus = maemo.get_session_bus()
        name = dbus.service.BusName(_SERVICE, bus = bus)
        dbus.service.Object.__init__(self, name, _OBJECT)



    def __run_heartbeat(self):
    
        if (not self.__heartbeat):
            self.__heartbeat = gobject.idle_add(self.__heartbeat_func)
            
            
    def __heartbeat_func(self):
    
        self.__collect_values()
        
        # check for end of file
        if (self.__playing and self.__total_length > 0 and self.__position >= 1.0):
            self.stop()
            self.end_of_stream()
            #if (self.__player_values[_FILENAME]):
            #    self.__next_time_check = now + 1
            #elif (now > self.__next_time_check):            
            #    self.__on_eof()
            #    self.__next_time_check = now + 1

            #self.__player_values[_FILENAME] = None
        #end if

        self.__heartbeat = gobject.timeout_add(500, self.__heartbeat_func)





    def __read(self):
                
        data = self.__stdout.readline()
        if (_LOGGING): print "    " + data[:-1]
        return data        



    def __collect_values(self):        
        
        if (not self.__stdout): return
        
        # get all new data
        for i in range(100):
            try:
                data = self.__read()
            except IOError, err:
                # no data left
                break
                
            self.__parse_value(data)
        #end for
        
        
    def __read_ans(self, data):

        idx = data.find("=")
        value = data[idx + 1:].strip()
        return value
        
        
    def __read_info(self, data):
    
        idx = data.find(":")
        value = data[idx + 1:].strip()
        return value
        
        
    def __parse_value(self, data):
    
        if (data.startswith("ANS_PERCENT_POSITION")):
            self.__position = float(self.__read_ans(data)) / 100
        #if (data.startswith("ANS_PERCENT_POSITION")):
        #    self.__player_values[_PERCENT_POSITION] = self.__read_ans(data)
        elif (data.startswith("ANS_LENGTH")):
            self.__total_length = float(self.__read_ans(data))
        #elif (data.startswith("ANS_VIDEO_RESOLUTION")):
        #    self.__player_values[_VIDEO_RESOLUTION] = self.__read_ans(data)
        #elif (data.startswith("ANS_filename")):
        #    self.__player_values[_FILENAME] = self.__read_ans(data)
        #elif (data.startswith("ANS_volume")):
        #    print "VOLUME", self.__read_ans(data)

        if (data.startswith("AUDIO: ")):
            self.detail_received("has_audio", "1")
        elif (data.startswith("VIDEO: ")):
            self.detail_received("has_video", "1")

        elif (data.startswith("ID_VIDEO_WIDTH")):
            self.__width = float(self.__read_ans(data))
        elif (data.startswith("ID_VIDEO_HEIGHT")):
            self.__height = float(self.__read_ans(data))
            try:
                aspect = self.__width / self.__height
                self.__send_cmd("switch_ratio %f" % aspect)
                self.detail_received("aspect", `aspect`)
            except:
                import traceback; traceback.print_exc()

        #elif (data.startswith("ID_LENGTH")):
        #    self.__media_length = float(self.__read_ans(data))

        #elif (data.startswith("Name   : ")):
        #    self.__player_values[_NAME] = self.__read_info(data)
        #elif (data.startswith("Genre  : ")):
        #    self.__player_values[_GENRE] = self.__read_info(data)
        #elif (data.startswith("Website: ")):
        #    self.__player_values[_WEBSITE] = self.__read_info(data)
       
        #elif (data.startswith("ICY Info: ")):
        #    self.__player_values[_ICY_INFO] = \
        #      self.__parse_icy_info(self.__read_info(data))
            
        #elif (data.startswith("Demuxer info Name changed to ")):
        #    idx = data.find("changed to ")
        #    name = data[idx + 11:].strip()
        #    self.__player_values[_NAME] = name
        #    self.__player_values[_ICY_INFO] = name

        #elif (data.startswith("Demuxer info Artist changed to ")):
        #    idx = data.find("changed to ")
        #    artist = data[idx + 11:].strip()
        #    self.__player_values[_ARTIST] = artist
        #    self.__player_values[_ICY_INFO] = self.__player_values[_NAME] + \
        #                                      " - " + artist
        
        elif (data.startswith("Starting playback...")):
            self.__playing = True
            self.state_changed("playing")
            
        #elif (data.startswith("File not found: ")):
        #    self.__broken = True
        #    self.update_observer(self.OBS_ERROR, self.__context_id,
        #                         self.ERR_NOT_FOUND)
        #elif (data.startswith("[file] No filename")):
        #    self.__broken = True
        #    self.update_observer(self.OBS_ERROR, self.__context_id,
        #                         self.ERR_NOT_FOUND)
        elif (data.startswith("Cache size set to ")):
            self.state_changed("buffering")
        #elif (data.endswith("No stream found.\n")):
        #    self.__broken = True
        #    self.update_observer(self.OBS_ERROR, self.__context_id,
        #                         self.ERR_NOT_FOUND)
        #elif (" failed to load: " in data):
        #    self.__broken = True
        #    self.update_observer(self.OBS_ERROR, self.__context_id,
        #                         self.ERR_INVALID)



        
    def __parse_icy_info(self, data):
        
        key = "StreamTitle='"
        idx = data.find(key)
        if (idx != -1):
            idx2 = data.find("';", idx + len(key))
            return data[idx + len(key):idx2]
        return ""





    def __start_mplayer(self):
    
        cmd = "LANG=C %s -quiet -slave " \
              "-noconsolecontrols -nojoystick -nolirc -nomouseinput " \
              "-idle -osdlevel 0 -idx " \
              "-identify -wid %d %s 2>&1 3>/dev/null" \
              % (_MPLAYER, self.__xid, self.__opts)

        p = subprocess.Popen([cmd],
                             shell=True, cwd="/tmp",
                             stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                             close_fds=True)
        time.sleep(0.25)
        if (p.poll()):
            print "Startup failed"
        else:
            print "running"
        
        self.__stdin = p.stdin
        self.__stdout = p.stdout
        self.__mplayer_pid = p.pid
        print "PID", self.__mplayer_pid

        # set non-blocking for better performance in an unthreaded environment
        fcntl.fcntl(self.__stdout, fcntl.F_SETFL, os.O_NONBLOCK)


    def __stop_mplayer(self):
        
        self.__playing = False
        self.state_changed("stopped")
        #self.__has_video = False
        #self.__has_audio = False
        #self.update_observer(self.OBS_KILLED)
        
        #if (self.__stdin): self.__stdin.write("quit\n")

        self.__stdin = None
        self.__stdout = None

        if (not self.__mplayer_pid): return

        #try:
        #    os.kill(self.__mplayer_pid, 15)
        #except:
        #    import traceback; traceback.print_exc()
        #    self.__mplayer_pid = 0
        #    return

        os.system("killall mplayer 2>/dev/null >/dev/null")
        time.sleep(0.25)
        if (os.path.isdir("/proc/%d" % self.__mplayer_pid)):
            #try:
            #    os.kill(self.__mplayer_pid, 9)
            #except:
            #    import traceback; traceback.print_exc()
            #    self.__mplayer_pid = 0            
            #    return
            os.system("killall -9 mplayer 2>/dev/null >/dev/null")
        self.__mplayer_pid = 0


    def __restart_mplayer(self):
    
        self.__stop_mplayer()
        self.__start_mplayer()


    def __ensure_mplayer(self):

        if (self.__needs_restart or not self.__stdin):
            self.__restart_mplayer()
            self.__needs_restart = False


    def __send_cmd(self, data):
    
        if (_LOGGING): print "--> " + data
        try:
            self.__stdin.write(data + "\n")
        except IOError:
            # broken pipe
            self.__needs_restart = True


    @dbus.service.signal(dbus_interface = _IFACE, signature = 's')
    def state_changed(self, state): pass


    @dbus.service.signal(dbus_interface = _IFACE, signature = 'ss')
    def detail_received(self, key, value): pass
    

    @dbus.service.signal(dbus_interface = _IFACE, signature = '')
    def end_of_stream(self): pass



    @dbus.service.method(dbus_interface = _IFACE, in_signature = '', out_signature = '')
    def close(self):
    
        self.__stop_mplayer()
        self.__needs_restart = True


    @dbus.service.method(dbus_interface = _IFACE, in_signature = 'i', out_signature = '')
    def set_window(self, xid):
    
        if (self.__xid != xid):
            self.__xid = xid
            self.__needs_restart = True
        
        
    @dbus.service.method(dbus_interface = _IFACE, in_signature = 's', out_signature = '')
    def set_options(self, opts):
    
        if (self.__opts != opts):
            self.__opts = opts
            self.__needs_restart = True
        
        
    @dbus.service.method(dbus_interface = _IFACE, in_signature = 's', out_signature = '')
    def load(self, uri):
    
        self.__ensure_mplayer()
        self.__send_cmd("loadfile \"%s\"" % uri.encode("utf-8"))
        self.__run_heartbeat()
        #self.state_changed("loaded")
        

    @dbus.service.method(dbus_interface = _IFACE, in_signature = '', out_signature = 'dd')
    def get_position(self):

        self.__send_cmd("get_percent_pos")
        self.__send_cmd("get_time_length")
        cnt = 0
        self.__position = 0
        self.__total_length = 0
        while ((self.__position < 0.001 or self.__total_length < 0.001) and
               cnt < 50):
            self.__collect_values()
            time.sleep(0.001)
            cnt += 1
            #print cnt
        #end while
        
        print (self.__total_length * self.__position, self.__total_length)
        
        return (self.__total_length * self.__position, self.__total_length)

        

    @dbus.service.method(dbus_interface = _IFACE, in_signature = 'i', out_signature = '')
    def seek(self, pos):
    
        self.__send_cmd("seek %d 2" % pos)

    @dbus.service.method(dbus_interface = _IFACE, in_signature = 'i', out_signature = '')
    def set_volume(self, vol):

        self.__send_cmd("volume %f 1" % vol)    
        

    @dbus.service.method(dbus_interface = _IFACE, in_signature = '', out_signature = '')
    def play(self):

        if (not self.__playing):
            self.__send_cmd("play")
            self.__playing = True
            self.state_changed("playing")
        

    @dbus.service.method(dbus_interface = _IFACE, in_signature = '', out_signature = '')
    def pause(self):

        self.__playing = not self.__playing
        self.__send_cmd("pause")
        if (self.__playing):
            self.state_changed("playing")
        else:
            self.state_changed("paused")
        

    @dbus.service.method(dbus_interface = _IFACE, in_signature = '', out_signature = '')
    def stop(self):
    
        self.__send_cmd("stop")
        self.__playing = False
        self.__position = 0
        self.state_changed("stopped")



mps = MPlayerService()
if (__name__ == "__main__"):
    import gobject
    gobject.MainLoop().run()

