/*
* Copyright (C) 2005-2006  Koos Vriezen <koos.vriezen@gmail.com>
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _KMPLAYER_TYPES_H_
#define _KMPLAYER_TYPES_H_

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gconf/gconf-client.h>

#undef KDE_NO_CDTOR_EXPORT
#undef KDE_NO_EXPORT
#if __GNUC__ - 0 > 3 || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 > 3)
  #define KMPLAYER_NO_CDTOR_EXPORT __attribute__ ((visibility("hidden")))
  #define KMPLAYER_NO_EXPORT __attribute__ ((visibility("hidden")))
  #define KMPLAYER_EXPORT __attribute__ ((visibility("default")))
#elif __GNUC__ - 0 > 3 || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 > 2)
  #define KMPLAYER_NO_CDTOR_EXPORT
  #define KMPLAYER_NO_EXPORT __attribute__ ((visibility("hidden")))
  #define KMPLAYER_EXPORT
#else
  #define KMPLAYER_NO_CDTOR_EXPORT
  #define KMPLAYER_NO_EXPORT
  #define KMPLAYER_EXPORT
#endif
#ifndef KDE_EXPORT
  #define KDE_EXPORT KMPLAYER_EXPORT
  #define KDE_NO_EXPORT KMPLAYER_NO_EXPORT
  #define KDE_NO_CDTOR_EXPORT KMPLAYER_NO_CDTOR_EXPORT
#endif


#ifndef ASSERT
#define ASSERT g_assert
//#define ASSERT Q_ASSERT
#endif

struct GnomeVFSURI;

namespace KMPlayer {

/**
 * Type meant for screen coordinates
 */
class KMPLAYER_NO_EXPORT Single {
    int value;
    friend Single operator + (const Single s1, const Single s2);
    friend Single operator - (const Single s1, const Single s2);
    friend Single operator * (const Single s1, const Single s2);
    friend Single operator / (const Single s1, const Single s2);
    friend Single operator + (const Single s1, const int i);
    friend Single operator - (const Single s1, const int i);
    friend float operator * (const Single s, const float f);
    friend double operator * (const Single s, const double f);
    friend Single operator * (const int i, const Single s);
    friend float operator * (const float f, const Single s);
    friend double operator * (const double d, const Single s);
    friend Single operator / (const Single s, const int i);
    friend double operator / (const Single s, const double d);
    friend double operator / (const double d, const Single s);
    friend bool operator > (const Single s1, const Single s2);
    friend bool operator > (const Single s, const int i);
    friend bool operator > (const int i, const Single s);
    friend bool operator >= (const Single s1, const Single s2);
    friend bool operator == (const Single s1, const Single s2);
    friend bool operator != (const Single s1, const Single s2);
    friend bool operator < (const Single s1, const Single s2);
    friend bool operator < (const Single s, const int i);
    friend bool operator < (const int i, const Single s);
    friend bool operator <= (const Single s1, const Single s2);
    friend bool operator <= (const Single s, const int i);
    friend Single operator - (const Single s);
public:
    Single () : value (0) {}
    Single (const int v) : value (v << 8) {}
    Single (const float v) : value (int (256 * v)) {}
    Single (const double v) : value (int (256 * v)) {}
    Single & operator = (const Single & s) { value = s.value; return *this; }
    Single & operator = (const int v) { value = v << 8; return *this; }
    Single & operator = (const float v) { value = int (256 * v); return *this; }
    Single & operator = (const double v) { value = int(256 * v); return *this; }
    Single & operator += (const Single & s) { value += s.value; return *this; }
    Single & operator += (const int i) { value += (i << 8); return *this; }
    Single & operator -= (const Single & s) { value -= s.value; return *this; }
    Single & operator -= (const int i) { value -= (i << 8); return *this; }
    Single & operator *= (const Single & s);
    Single & operator *= (const float f) { value = int(value*f); return *this; }
    Single & operator /= (const int i) { value /= i; return *this; }
    operator int () const { return value >> 8; }
    operator double () const { return 1.0 * value / 256; }
};

/**                                   a  b  0
 * Matrix for coordinate transforms   c  d  0
 *                                    tx ty 1     */
class KMPLAYER_NO_EXPORT Matrix {
    friend class SizeEvent;
    float a, b, c, d;
    Single tx, ty; 
public:
    Matrix ();
    Matrix (const Matrix & matrix);
    Matrix (Single xoff, Single yoff, float xscale, float yscale);
    void getXY (Single & x, Single & y) const;
    void getXYWH (Single & x, Single & y, Single & w, Single & h) const;
    void transform (const Matrix & matrix);
    void scale (float sx, float sy);
    void translate (Single x, Single y);
    // void rotate (float phi); // add this when needed
};

class KMPLAYER_NO_EXPORT SRect {
    Single _x, _y, _w, _h;
public:
    SRect () {}
    SRect (Single a, Single b, Single w, Single h)
        : _x (a), _y (b), _w (w), _h (h) {}
    Single x () const { return _x; }
    Single y () const { return _y; }
    Single width () const { return _w; }
    Single height () const { return _h; }
    SRect unite (const SRect & r) const;
    SRect intersect (const SRect & r) const;
    bool isValid () const { return _w >= Single (0) && _h >= Single (0); }
    bool operator == (const SRect & r) const;
    bool operator != (const SRect & r) const;
};

template <class T>
class KMPLAYER_EXPORT CharTmpl {
    T m_unichar;
public:
    CharTmpl ();
    CharTmpl (const T & char_impl);
    CharTmpl (const char ch);
    CharTmpl (const int unichar);
    CharTmpl (const CharTmpl <T> & other);
    ~CharTmpl ();
    bool operator == (const CharTmpl <T> & other) const;
    bool operator == (const T & char_impl) const;
    bool operator == (const char ch) const;
    bool operator != (const CharTmpl <T> & other) const;
    bool operator != (const T & char_impl) const;
    bool operator != (const char ch) const;
    CharTmpl <T> & operator = (const CharTmpl <T> & other);
    CharTmpl <T> & operator = (const T & char_impl);
    CharTmpl <T> & operator = (const char ch);
    operator T ();
    operator T () const;
    operator char () const;
    int unicode () const;
    // these isXX should be specialized
    bool isSpace () const;
    bool isLetterOrNumber () const;
};

typedef CharTmpl <gunichar> Char;


template <class T>
class ByteArrayTmpl {
    T m_bytearray;
public:
    ByteArrayTmpl ();
    ByteArrayTmpl (unsigned sz);
    ByteArrayTmpl (const ByteArrayTmpl <T> & other);
    ~ByteArrayTmpl ();
    unsigned size () const;
    void resize (unsigned ns);
    char * data () const;
    bool isEmpty () const;
    ByteArrayTmpl <T> & operator = (const ByteArrayTmpl <T> & other);
    ByteArrayTmpl <T> & operator = (const char * buf);
};

class ByteArrayImpl {
    char * m_data;
    unsigned m_size;
    unsigned m_capacity;
public:
    ByteArrayImpl ();
    ByteArrayImpl (unsigned sz);
    ByteArrayImpl (const ByteArrayImpl & other);
    ~ByteArrayImpl ();
    unsigned size () const;
    void resize (unsigned ns);
    char * data () const;
    bool isEmpty () const;
    ByteArrayImpl & operator = (const ByteArrayImpl & other);
};

typedef ByteArrayTmpl <ByteArrayImpl> ByteArray;

template <class T>
class KMPLAYER_EXPORT StringTmpl {
    T string;
public:
    StringTmpl ();
    StringTmpl (const T & str_impl);
    StringTmpl (const char * str, int maxlen = -1);
    StringTmpl (const Char & ch);
    StringTmpl (const StringTmpl <T> & other);
    ~StringTmpl ();
    bool operator == (const StringTmpl <T> & other) const;
    bool operator == (const T & str_impl) const;
    bool operator == (const char * str) const;
    bool operator != (const StringTmpl <T> & other) const;
    bool operator != (const T & str_impl) const;
    bool operator != (const char * str) const;
    StringTmpl <T> & operator = (const StringTmpl <T> & other);
    StringTmpl <T> & operator = (const T & str_impl);
    StringTmpl <T> & operator = (const char * s);
    StringTmpl <T> & operator = (const Char & ch);
    StringTmpl <T> & operator += (const StringTmpl <T> & other);
    StringTmpl <T> & operator += (const T & str_impl);
    StringTmpl <T> & operator += (const char * str);
    StringTmpl <T> & operator += (const Char & ch);
    StringTmpl <T> & operator << (const StringTmpl <T> & other);
    StringTmpl <T> & operator << (const T & str_impl);
    StringTmpl <T> & operator << (const char * str);
    StringTmpl <T> operator + (const char * str);
    StringTmpl <T> operator + (const T & str_impl);
    StringTmpl <T> operator + (const StringTmpl <T> & other);
    StringTmpl <T> operator + (const Char & ch);
    StringTmpl <T> mid (const int index, unsigned length=(unsigned)-1) const;
    StringTmpl <T> left (int index) const;
    int toInt (bool *valid=0L, int base=10) const;
    double toDouble (bool *valid=0L) const;
    Char operator [] (int index) const;
    operator T & ();
    operator T () const;
    operator const char * () const;
    bool operator < (const StringTmpl <T> & other) const;
    static StringTmpl <T> fill (Char ch, int length);
    StringTmpl <T> lower () const;
    StringTmpl <T> upper () const;
    StringTmpl <T> stripWhiteSpace () const;
    static StringTmpl <T> number (int nr);
    bool hasPrefix (const StringTmpl <T> & prefix) const;
    bool hasSuffix (const StringTmpl <T> & suffix) const;
    int find (const StringTmpl <T> & needle) const;
    int findRev (const StringTmpl <T> & needle) const;
    bool isEmpty () const;
    bool isNull () const;
    int length () const;
    int compare (const StringTmpl <T> & other) const;
    void truncate (int pos);
    void clear ();
};

template <class T, class U>
StringTmpl <T> operator + (const CharTmpl <U> & c, const StringTmpl <T> & s) {
    StringTmpl <T> str (c);
    return str += s;
}

template <class T, class U>
StringTmpl <T> operator + (const StringTmpl <T> & s, const CharTmpl <U> & c) {
    StringTmpl <T> str (s);
    return str += c;
}

struct SharedChars {
    gchar * ptr;
    int ref_count;
    SharedChars (gchar * p) : ptr (p), ref_count (1) {}
    ~SharedChars () { g_free (ptr); }
    void ref () { ref_count++; }
    void unref () { if (--ref_count <= 0) delete this; }
};

class Shared {
public:
    ~Shared ();
    void attach (char * p);  // p must be valid utf8
    void clear ();
    bool isNull () const;
protected:
    SharedChars * data;
    Shared ();
    void attach (const Shared & other);
    void detach ();
};

class KMPLAYER_EXPORT GLibString : public Shared {
public:
    GLibString ();
    GLibString (const GLibString & str);
    GLibString (const char * str, int maxlen = -1);
    GLibString (const Char & str);
    ~GLibString ();
    int compare (const GLibString & str) const;
    GLibString & operator = (const GLibString & str);
    GLibString & operator = (const char * s);
    GLibString & operator = (const Char & ch);
    GLibString & operator += (const GLibString & str);
    GLibString & operator += (const char * str);
    GLibString & operator += (const Char & ch);
    Char operator [] (int index) const;
    operator const char * () const;
    void take (char * s);
    GLibString lower () const;
    GLibString upper () const;
    GLibString stripWhiteSpace () const;
    static GLibString fill (Char ch, int length);
    static GLibString number (int nr);
    bool hasPrefix (const GLibString & prefix) const;
    bool hasSuffix (const GLibString & suffix) const;
    int find (const GLibString & needle) const;
    int findRev (const GLibString & needle) const;
    bool isEmpty () const;
    GLibString mid (const int index, unsigned length=(unsigned)-1) const;
    GLibString left (int index) const;
    int toInt (bool *valid=0L, int base=10) const;
    double toDouble (bool *valid=0L) const;
    int length () const;
    void truncate (int pos);
};

typedef StringTmpl <GLibString> String;


enum FileMode { IO_ReadOnly=0x01, IO_WriteOnly=0x02, IO_ReadWrite=0x03 };

template <class T>
class FileTmpl {
    T m_file;
public:
    FileTmpl (const String & path);
    ~FileTmpl ();
    bool open (FileMode mode);
    void close ();
    ByteArray readAll ();
    off_t readBlock (char *, off_t max);
    off_t writeBlock (const char * data, off_t length);
    bool exists ();
    off_t size ();
};

class GnomeVFSFilePrivate;

class GnomeVFSFile {
    GnomeVFSFilePrivate * d;
public:
    GnomeVFSFile (const String & path);
    ~GnomeVFSFile ();
    bool open (FileMode mode);
    void close ();
    ByteArray readAll ();
    off_t readBlock (char *, off_t max);
    off_t writeBlock (const char * data, off_t length);
    bool exists ();
    off_t size ();
};

typedef FileTmpl <GnomeVFSFile> File;


template <class T>
class TextStreamTmpl {
    T m_textstream;
public:
    TextStreamTmpl ();
    TextStreamTmpl (const String & init);
    TextStreamTmpl (ByteArray & init);
    ~TextStreamTmpl ();
    TextStreamTmpl <T> & operator << (const String & str);
    TextStreamTmpl <T> & operator << (const char * str);
    TextStreamTmpl <T> & operator << (const Char & ch);
    TextStreamTmpl <T> & operator >> (Char & ch);
    TextStreamTmpl <T> & readRawBytes (char * b, unsigned length);
    TextStreamTmpl <T> & operator = (const TextStreamTmpl <T> & other);
    String release ();
    String readLine ();
    bool atEnd () const;
    bool isEmpty () const;
private:
    TextStreamTmpl (const TextStreamTmpl <T> & other);  // hide this for now
};

class GLibTextStream {
    GString * m_gstring;
    gchar * m_utf8_str;
    gchar * m_iter;
public:
    GLibTextStream ();
    GLibTextStream (const String & init);
    GLibTextStream (ByteArray & init);
    ~GLibTextStream ();
    GLibTextStream & operator << (const String & str);
    GLibTextStream & operator << (const char * str);
    GLibTextStream & operator << (const Char & ch);
    GLibTextStream & operator >> (Char & ch);
    GLibTextStream & readRawBytes (char * b, unsigned length);
    GLibTextStream & operator = (const GLibTextStream & other);
    bool atEnd () const;
    bool isEmpty () const;
    String release ();
    String readLine ();
};


typedef TextStreamTmpl <GLibTextStream> TextStream;

template <class T>
class URLTmpl {
    T m_url;
public:
    URLTmpl ();
    URLTmpl (const String & url);
    URLTmpl (const T & url_impl);
    URLTmpl (const URLTmpl <T> & base, const String & rel);
    URLTmpl (const URLTmpl <T> & url);
    ~URLTmpl ();
    URLTmpl <T> & operator = (const String & url);
    URLTmpl <T> & operator = (const URLTmpl <T> & url);
    URLTmpl <T> & operator = (const T & url_impl);
    bool operator == (const URLTmpl <T> & url) const;
    String url () const;
    String path () const;
    String protocol () const;
    static String decode_string (const String & str);
    bool isLocalFile () const;
};

class GnomeVFSUrl {
    GnomeVFSURI * m_vfsurl;
public:
    GnomeVFSUrl ();
    GnomeVFSUrl (const String & ur);
    GnomeVFSUrl (const GnomeVFSUrl & base, const String & rel);
    GnomeVFSUrl (const GnomeVFSUrl & url);
    ~GnomeVFSUrl ();
    GnomeVFSUrl & operator = (const String & url);
    GnomeVFSUrl & operator = (const GnomeVFSUrl & url);
    String url () const;
    String path () const;
    String protocol () const;
    static String decode_string (const String & str);
    bool isLocalFile () const;
};

typedef URLTmpl <GnomeVFSUrl> URL;


template <class T>
class MimeTypeTmpl {
public:
    static String findByContent (const char * data, int size);
    //static String findByURL (const URL & url);
};

class GMimeType {
public:
    static String findByContent (const char * data, int size);
    //static String findByURL (const URL & url);
};

typedef MimeTypeTmpl <GMimeType> MimeType;


class GnomeVFSJobPrivate;
class JobInterface;
class AsyncJobInterface;
typedef JobInterface Job;

Job * asyncGet (AsyncJobInterface * receiver, const String & url);

class JobInterface {
public:
    virtual void kill (bool quiet = true) = 0;
    virtual ~JobInterface () {};
protected:
    JobInterface () {}
};

class AsyncJobInterface {
public:
    virtual ~AsyncJobInterface () {}
    virtual void jobData (Job * job, ByteArray & data) = 0;
    virtual void jobResult (Job * job) = 0;
};

template <class T>
class ConfigTmpl {
    T m_config;
public:
    String readEntry (const String & key, const String & def);
    int readNumEntry (const String & key, int def);
    double readNumEntry (const String & key, double def);
    void writeEntry (const String & key, const String & val);
    void writeEntry (const String & key, int val);
    void writeEntry (const String & key, double val);
};

class GConf {
    GConfClient * client;
public:
    GConf ();
    ~GConf ();
    String readEntry (const String & key, const String & def);
    int readNumEntry (const String & key, int def);
    double readNumEntry (const String & key, double def);
    void writeEntry (const String & key, const String & val);
    void writeEntry (const String & key, int val);
    void writeEntry (const String & key, double val);
};

typedef ConfigTmpl <GConf> Config;


class Log;

typedef Log & (*LOGFUNC) (Log &);

class Log {
    Log ();
    friend Log debugLog ();
    friend Log warningLog ();
    friend Log errorLog ();
public:
    ~Log ();
    Log & operator << (const String & s);
    Log & operator << (const char * s);
    //Log & operator << (const Char & s);
    Log & operator << (const URL & u);
    Log & operator << (int i);
    Log & operator << (unsigned int i);
    Log & operator << (double d);
    Log & operator << (void *);
    Log & operator << (LOGFUNC manip);
};

Log debugLog ();
Log warningLog ();
Log errorLog ();


template <class T>
class RectTmpl {
    T m_rect;
public:
    RectTmpl ();
    RectTmpl (const RectTmpl <T> & other);
    RectTmpl (const T & rect_impl);
    RectTmpl (int x, int y, int w, int h);
    ~RectTmpl ();
    int x () const;
    int y () const;
    int width () const;
    int height () const;
    bool isValid () const;
    RectTmpl <T> unite (const RectTmpl <T> & other) const;
    RectTmpl <T> intersect (const RectTmpl <T> & other) const;
    operator T & ();
    operator const T & () const;
    operator T () const;
    RectTmpl <T> & operator = (const RectTmpl <T> & other);
    bool operator == (const RectTmpl <T> & other) const;
    bool operator != (const RectTmpl <T> & other) const;
};

class GRect {
    GdkRectangle * m_rect;
public:
    GRect ();
    GRect (const GRect & other);
    GRect (int x, int y, int w, int h);
    ~GRect ();
    int x () const;
    int y () const;
    int width () const;
    int height () const;
    bool isValid () const;
    GRect unite (const GRect & other) const;
    GRect intersect (const GRect & other) const;
    GRect & operator = (const GRect & other);
    operator GdkRectangle * () const;
    bool operator == (const GRect & other) const;
};

typedef RectTmpl <GRect> Rect;


template <class T>
class ColorTmpl {
    T m_color;
public:
    ColorTmpl ();
    ColorTmpl (const String & desc);
    ColorTmpl (int rgb);
    ColorTmpl (const ColorTmpl <T> & other);
    ColorTmpl (const T & color_impl);
    ~ColorTmpl ();
    operator T & ();
    operator const T & () const;
    operator T () const;
    ColorTmpl <T> & operator = (const ColorTmpl <T> & other);
    ColorTmpl <T> & operator = (const String & desc);
    ColorTmpl <T> & operator = (int rgb);
    unsigned rgb () const;
};

class GColor {
    GdkColor * m_color;
public:
    GColor ();
    GColor (const String & desc);
    GColor (int rgb);
    GColor (const GColor & other);
    ~GColor ();
    void clear ();
    GColor & operator = (const GColor & other);
    operator GdkColor * () const;
    unsigned rgb () const;
};

typedef ColorTmpl <GColor> Color;
typedef unsigned Rgb;


template <class T> class ImageTmpl;
class GPixbuf;
typedef ImageTmpl <GPixbuf> Image;

template <class T>
class PixmapTmpl {
    T m_pixmap;
public:
    PixmapTmpl ();
    PixmapTmpl (int w, int h);
    PixmapTmpl (const ByteArray & data);
    PixmapTmpl (T & data);
    PixmapTmpl (const Image & image);
    ~PixmapTmpl ();
    bool isNull () const;
    int width () const;
    int height () const;
    void fill (const Color & c = Color (0));
    operator const T & () const;
private:
    PixmapTmpl (const PixmapTmpl <T> & other); // forbidden for now
};

class GPixmap {
    GdkPixmap * m_pixmap;
public:
    GPixmap ();
    GPixmap (int w, int h);
    GPixmap (const ByteArray & data);
    GPixmap (GdkPixbuf * pix);
    GPixmap (const GPixmap & other); // only a g_object_ref on other.m_pixmap
    GPixmap (const Image & image);
    ~GPixmap ();
    bool isNull () const;
    int width () const;
    int height () const;
    void fill (const Color & c = Color (0));
    operator GdkDrawable * () const;
    GPixmap & operator = (GdkPixbuf * pixbuf);
    GdkPixmap * pixmap () const;
};

typedef PixmapTmpl <GPixmap> Pixmap;


template <class T>
class ImageTmpl {
    T m_image;
public:
    ImageTmpl ();
    ImageTmpl (int w, int h);
    ImageTmpl (const ByteArray & data);
    ImageTmpl (const T & data);
    ~ImageTmpl ();
    bool isNull () const;
    int width () const;
    int height () const;
    void fill (const Color & c = Color (0));
    void reset ();
    ImageTmpl <T> scale (int w, int h);
    ImageTmpl <T> & operator = (const ImageTmpl <T> & other);
    ImageTmpl <T> & operator = (const Pixmap & pixmap);
    operator const T & () const;
};

class GPixbuf {
    GdkPixbuf * m_pixbuf;
public:
    GPixbuf ();
    GPixbuf (int w, int h);
    GPixbuf (const ByteArray & data);
    GPixbuf (GdkPixbuf * pix, bool skip_addref=false);
    GPixbuf (const GPixbuf & other); // only a g_object_ref on other.m_pixmap
    ~GPixbuf ();
    bool isNull () const;
    int width () const;
    int height () const;
    void fill (const Color & c = Color (0));
    void reset ();
    GPixbuf scale (int w, int h);
    operator GdkPixbuf * () const;
    GPixbuf & operator = (const GPixbuf & other);
    GPixbuf & operator = (const Pixmap & pixmap);
    GPixbuf & operator = (GdkPixbuf * pixbuf);
    GdkPixbuf * pixbuf () const;
};

typedef ImageTmpl <GPixbuf> Image;

/*
template <class T>
class PainterTmpl {
    T m_painter;
public:
    PainterTmpl ();
    ~PainterTmpl ();
    void setClipRect (const Rect & rect);
    void setClipping (bool enable);
    void fillRect (int x, int y, int w, int h, const Color & c);
    bool begin (const Pixmap *);
    bool end ();
    operator T & ();
    operator const T & () const;
    void drawPixmap (int dx, int dy, const Pixmap & pm, int sx=0, int sy=0, int sw=-1, int sh=-1);
    void drawImage (int dx, int dy, const Image & im, int sx=0, int sy=0, int sw=-1, int sh=-1);
private:
    PainterTmpl (const PainterTmpl <T> & other); // no copies
};

class GPainterPrivate;

class GPainter {
    Rect m_cliprect;
    Rect m_drawrect;
    GPainterPrivate * d;
public:
    GPainter ();
    ~GPainter ();
    bool begin (GdkDrawable * d, const Rect & r);
    bool begin (const Pixmap *);
    bool end ();
    void setClipRect (const Rect & rect);
    void setClipping (bool enable);
    void fillRect (int x, int y, int w, int h, const Color & c);
    void drawPixbuf (GdkPixbuf *p, int sx, int sy, int x, int y, int w, int h);
    void drawText (const String & txt, int x, int y, int w, int h);
    void setPen (const Color & c);
    void setBackgroundColor (const Color & c);
    void setFont (const String & str);
    //operator GdkGC * () const;
    operator GdkDrawable * () const;
    void drawPixmap (int dx, int dy, const Pixmap & pm, int sx=0, int sy=0, int sw=-1, int sh=-1);
    void drawImage (int dx, int dy, const Image & im, int sx=0, int sy=0, int sw=-1, int sh=-1);
};

typedef PainterTmpl <GPainter> Painter;
*/
//-----------------------------------------------------------------------------


inline Single & Single::operator *= (const Single & s) {
    value = (((int64_t)value) * s.value) >> 8;
    return *this;
}

inline Single operator + (const Single s1, const Single s2) {
    Single s;
    s.value = s1.value + s2.value;
    return s;
}

inline Single operator - (const Single s1, const Single s2) {
    Single s;
    s.value = s1.value - s2.value;
    return s;
}

inline Single operator * (const Single s1, const Single s2) {
    Single s;
    s.value = (((int64_t)s1.value) * s2.value) >> 8;
    return s;
}

inline Single operator / (const Single s1, const Single s2) {
    Single s;
    s.value = ((int64_t)s1.value << 8) / s2.value;
    return s;
}

inline Single operator + (const Single s, const int i) {
    return s + Single (i);
}

inline Single operator - (const Single s, const int i) {
    return s - Single (i);
}

inline Single operator * (const int i, const Single s) {
    Single s1;
    s1.value = s.value * i;
    return s1;
}

inline Single operator * (const Single s, const int i) {
    return i * s;
}
inline float operator * (const Single s, const float f) {
    return s.value * f / 256;
}

inline double operator * (const Single s, const double d) {
    return s.value * d / 256;
}

inline float operator * (const float f, const Single s) {
    return s.value * f / 256;
}

inline double operator * (const double d, const Single s) {
    return s.value * d / 256;
}

inline Single operator / (const Single s, const int i) {
    Single s1;
    s1.value = s.value / i;
    return s1;
}

inline double operator / (const Single s, const double d) {
    return (s.value / d ) / 256;
}

inline double operator / (const double d, const Single s) {
    return (d * 256 / s.value);
}

inline bool
operator > (const Single s1, const Single s2) { return s1.value > s2.value; }

inline bool
operator > (const Single s, const int i) { return s > Single (i); }

inline bool
operator > (const int i, const Single s) { return Single (i) > s; }

inline bool
operator >= (const Single s1, const Single s2) { return s1.value >= s2.value; }

inline bool
operator == (const Single s1, const Single s2) { return s1.value == s2.value; }

inline bool
operator != (const Single s1, const Single s2) { return s1.value != s2.value; }

inline bool
operator < (const Single s1, const Single s2) { return s1.value < s2.value; }

inline bool
operator < (const Single s, const int i) { return s < Single (i); }

inline bool
operator < (const int i, const Single s) { return Single (i) < s; }

inline bool
operator <= (const Single s1, const Single s2) { return s1.value <= s2.value; }

inline bool
operator <= (const Single s, const int i) { return s <= Single (i); }

inline Single operator - (const Single s) {
    Single s1;
    s1.value = -s.value;
    return s1;
}

//-----------------------------------------------------------------------------

inline bool SRect::operator == (const SRect & r) const {
    return _x == r._x && _y == r._y && _w == r._w && _h == r._h;
}

inline bool SRect::operator != (const SRect & r) const { return !(*this == r); }

//-----------------------------------------------------------------------------

template <class T> inline CharTmpl<T>::CharTmpl () : m_unichar (0) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const T & char_impl) : m_unichar (char_impl) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const char ch) : m_unichar (ch) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const int unichar) : m_unichar (unichar) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const CharTmpl <T> & other) 
    :  m_unichar (other.m_unichar) {}

template <class T> inline CharTmpl<T>::~CharTmpl () {}

template <class T> inline bool CharTmpl<T>::operator == (const CharTmpl <T> & other) const {
    return m_unichar == other.m_unichar;
}

template <class T> inline bool CharTmpl<T>::operator == (const T & char_impl) const {
    return m_unichar == char_impl;
}

template <class T> inline bool CharTmpl<T>::operator == (const char ch) const {
    return m_unichar == ch;
}

template <class T> inline bool CharTmpl<T>::operator != (const CharTmpl <T> & other) const {
    return m_unichar != other.m_unichar;
}

template <class T> inline bool CharTmpl<T>::operator != (const T & char_impl) const {
    return m_unichar != char_impl;
}

template <class T> inline  bool CharTmpl<T>::operator != (const char ch) const {
    return m_unichar != ch;
}

template <class T>
inline CharTmpl <T> & CharTmpl<T>::operator = (const CharTmpl <T> & other) {
    m_unichar = other.m_unichar;
    return *this;
}

template <class T>
inline CharTmpl <T> & CharTmpl<T>::operator = (const T & char_impl) {
    m_unichar = char_impl;
    return *this;
}

template <class T>
inline CharTmpl <T> & CharTmpl<T>::operator = (const char ch) {
    m_unichar = ch;
    return *this;
}

template <class T> inline CharTmpl<T>::operator T () { return m_unichar; }

template <class T> inline CharTmpl<T>::operator T () const { return m_unichar; }

template <class T> inline CharTmpl<T>::operator char () const {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (m_unichar, buf);
    return (char) buf [nr-1];
}

template <class T> inline int CharTmpl<T>::unicode () const {
    return (int) m_unichar;
}

template <class T> inline bool CharTmpl<T>::isSpace () const {
    return false;
}

template <class T> inline bool CharTmpl<T>::isLetterOrNumber () const {
    return false;
}

//-----------------------------------------------------------------------------

template <>
inline bool CharTmpl<gunichar>::isSpace () const {
    return g_unichar_isspace (m_unichar);
}

template <>
inline bool CharTmpl<gunichar>::isLetterOrNumber () const {
    return g_unichar_isalnum (m_unichar);
}

template <>
inline CharTmpl <gunichar> & CharTmpl<gunichar>::operator = (const char ch) {
    char buf [] = { ch, 0 };
    const gchar *end;
    if (!g_utf8_validate (buf, -1, &end)) {
        gsize bytes_read, bytes_written;
        gchar *utf8 =g_locale_to_utf8(buf, -1, &bytes_read, &bytes_written, 0L);
        m_unichar = g_utf8_get_char (utf8);
        g_free (utf8);
    } else
        m_unichar = g_utf8_get_char (buf);
    return *this;
}

template <>
inline CharTmpl<gunichar>::CharTmpl (const char ch) {
    operator = (ch);
}

template <>
inline  bool CharTmpl<gunichar>::operator == (const char ch) const {
    return m_unichar == Char (ch). m_unichar;
}

template <>
inline  bool CharTmpl<gunichar>::operator != (const char ch) const {
    return m_unichar != Char (ch).m_unichar;
}

//-----------------------------------------------------------------------------

template <class T> inline ByteArrayTmpl <T>::ByteArrayTmpl () {}

template <class T>
inline ByteArrayTmpl <T>::ByteArrayTmpl (unsigned sz) : m_bytearray (sz) {}

template <class T>
inline ByteArrayTmpl <T>::ByteArrayTmpl (const ByteArrayTmpl <T> & b)
    : m_bytearray (b.m_bytearray) {}

template <class T> inline ByteArrayTmpl <T>::~ByteArrayTmpl () {}

template <class T> inline unsigned ByteArrayTmpl <T>::size () const {
    return m_bytearray.size ();
}

template <class T> inline void ByteArrayTmpl <T>::resize (unsigned ns) {
    m_bytearray.resize (ns);
}

template <class T>
inline char * ByteArrayTmpl <T>::data () const {
    return m_bytearray.data ();
}

template <class T> inline
bool ByteArrayTmpl <T>::isEmpty () const { return m_bytearray.isEmpty (); }

template <class T> inline
ByteArrayTmpl<T> & ByteArrayTmpl <T>::operator = (const ByteArrayTmpl<T> & b) {
    m_bytearray = b.m_bytearray;
    return *this;
}

template <class T> inline
ByteArrayTmpl<T> & ByteArrayTmpl <T>::operator = (const char * b) {
    int len = strlen (b);
    m_bytearray.resize (len);
    memcpy (m_bytearray.data (), b, len);
    return *this;
}

//-----------------------------------------------------------------------------

template <class T> inline StringTmpl<T>::StringTmpl () {}

template <class T>
inline StringTmpl<T>::StringTmpl (const T & str_impl) : string (str_impl) {}

template <class T>
inline StringTmpl<T>::StringTmpl (const char * str, int maxlen)
    : string (str, maxlen) {}

template <class T>
inline StringTmpl<T>::StringTmpl (const Char & ch) : string (ch) {}

template <class T>
inline StringTmpl<T>::StringTmpl (const StringTmpl <T> & other)
    : string (other.string) {}

template <class T> inline StringTmpl<T>::~StringTmpl () {}

template <class T>
inline int StringTmpl<T>::compare (const StringTmpl <T> & other) const {
    return string.compare (other.string);
}

template <class T>
inline bool StringTmpl<T>::operator == (const StringTmpl <T> & other) const {
    return !string.compare (other.string);
}

template <class T>
inline bool StringTmpl<T>::operator == (const T & str_impl) const {
    return !string.compare (str_impl);
}

template <class T>
inline bool StringTmpl<T>::operator == (const char * str) const {
    return *this == String (str);
}

template <class T>
inline bool StringTmpl<T>::operator != (const StringTmpl <T> & other) const {
    return string.compare (other.string);
}

template <class T>
inline bool StringTmpl<T>::operator != (const T & str_impl) const {
    return string.compare (str_impl);
}

template <class T>
inline bool StringTmpl<T>::operator != (const char * str) const {
    return *this != String (str);
}

template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator = (const StringTmpl <T> & other) {
    string = other.string;
    return *this;
}

template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator = (const T & str_impl) {
    string = str_impl;
    return *this;
}

template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator = (const char * s) {
    string = s;
    return *this;
}

template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator = (const Char & ch) {
    string = ch;
    return *this;
}

template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator += (const StringTmpl <T> & other) {
    string += other.string;
    return *this;
}

template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator += (const T & str_impl) {
    string += str_impl;
    return *this;
}

template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator += (const char * str) {
    string += str;
    return *this;
}

template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator += (const Char & ch) {
    string += ch;
    return *this;
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::operator + (const StringTmpl <T> & other) {
    StringTmpl<T> str (*this);
    return str += other;
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::operator + (const T & str_impl) {
    StringTmpl<T> str (*this);
    return str += str_impl;
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::operator + (const char * str) {
    StringTmpl<T> str1 (*this);
    return str1 += str;
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::operator + (const Char & ch) {
    StringTmpl<T> str (*this);
    return str += ch;
}


template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator << (const StringTmpl <T> & other) {
    string += other.string;
    return *this;
}

template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator << (const T & str_impl) {
    string += str_impl;
    return *this;
}

template <class T>
inline StringTmpl <T> & StringTmpl<T>::operator << (const char * str) {
    string += str;
    return *this;
}

template <class T>
inline Char StringTmpl<T>::operator [] (int index) const {
    return string [index];
}

template <class T> inline StringTmpl<T>::operator T & () { return string; }

template <class T> inline StringTmpl<T>::operator T () const { return string; }

template <class T> inline StringTmpl<T>::operator const char * () const {
    return (const char *) string;
}

template <class T>
inline bool StringTmpl<T>::operator < (const StringTmpl <T> & other) const {
    return string.compare (other.string) < 0;
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::fill (Char ch, int length) {
    return T::fill (ch, length);
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::lower () const {
    return string.lower ();
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::upper () const {
    return string.upper ();
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::stripWhiteSpace () const {
    return string.stripWhiteSpace ();
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::number (int nr) {
    return T::number (nr);
}

template <class T>
inline bool StringTmpl<T>::hasPrefix (const StringTmpl <T> & prefix) const {
    return string.hasPrefix (prefix.string);
}

template <class T>
inline bool StringTmpl<T>::hasSuffix (const StringTmpl <T> & suffix) const {
    return string.hasSuffix (suffix.string);
}

template <class T>
inline int StringTmpl<T>::find (const StringTmpl <T> & needle) const {
    return string.find (needle.string);
}

template <class T>
inline int StringTmpl<T>::findRev (const StringTmpl <T> & needle) const {
    return string.findRev (needle.string);
}

template <class T>
inline bool StringTmpl<T>::isEmpty () const {
    return string.isEmpty ();
}

template <class T>
inline bool StringTmpl<T>::isNull () const {
    return string.isNull ();
}

template <class T>
inline int StringTmpl<T>::length () const {
    return string.length ();
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::mid (const int index, unsigned len) const {
    return string.mid (index, len);
}

template <class T>
inline StringTmpl <T> StringTmpl<T>::left (int index) const {
    return string.left (index);
}

template <class T>
inline int StringTmpl<T>::toInt (bool * valid, int base) const {
    return string.toInt (valid, base);
}

template <class T>
inline double StringTmpl<T>::toDouble (bool * valid) const {
    return string.toDouble (valid);
}

template <class T>
inline void StringTmpl<T>::truncate (int pos) {
    string.truncate (pos);
}

template <class T>
inline void StringTmpl<T>::clear () {
    string.clear ();
}

//-----------------------------------------------------------------------------

template <class T>
inline FileTmpl <T>::FileTmpl (const String & path) : m_file (path) {}

template <class T>
inline FileTmpl <T>::~FileTmpl () {}

template <class T>
inline bool FileTmpl <T>::open (FileMode mode) {
    return m_file.open (mode);
}

template <class T>
inline void FileTmpl <T>::close () {
    m_file.close ();
}

template <class T>
inline ByteArray FileTmpl <T>::readAll () {
    return m_file.readAll ();
}

template <class T>
inline off_t FileTmpl <T>::readBlock (char * data, off_t max) {
    return m_file.readBlock (data, max);
}

template <class T>
inline off_t FileTmpl <T>::writeBlock (const char * data, off_t length) {
    return m_file.writeBlock (data, length);
}

template <class T>
inline bool FileTmpl <T>::exists () {
    return m_file.exists ();
}

template <class T>
inline off_t FileTmpl <T>::size () {
    return m_file.size ();
}

//-----------------------------------------------------------------------------

template <class T> inline TextStreamTmpl <T>::TextStreamTmpl () {}

template <class T>
inline TextStreamTmpl <T>::TextStreamTmpl (const String & init)
    : m_textstream (init) {}

template <class T>
inline TextStreamTmpl <T>::TextStreamTmpl (ByteArray & init)
    : m_textstream (init) {}

template <class T> inline TextStreamTmpl <T>::~TextStreamTmpl () {}

template <class T>
inline TextStreamTmpl<T>& TextStreamTmpl <T>::operator << (const String & str) {
    m_textstream << str;
    return *this;
}

template <class T>
inline TextStreamTmpl <T> & TextStreamTmpl <T>::operator << (const char * str) {
    m_textstream << str;
    return *this;
}

template <class T>
inline TextStreamTmpl <T> & TextStreamTmpl <T>::operator << (const Char & ch) {
    m_textstream << ch;
    return *this;
}

template <class T>
inline TextStreamTmpl <T> & TextStreamTmpl <T>::operator >> (Char & ch) {
    m_textstream >> ch;
    return *this;
}

template <class T> inline
TextStreamTmpl <T> & TextStreamTmpl <T>::readRawBytes (char * b, unsigned len) {
    m_textstream.readRawBytes (b, len);
    return *this;
}

template <class T> inline
TextStreamTmpl <T> & TextStreamTmpl <T>::operator = (const TextStreamTmpl <T> & s) {
    m_textstream = s.m_textstream;
    return *this;
}

template <class T> inline bool TextStreamTmpl <T>::atEnd () const {
    return m_textstream.atEnd ();
}

template <class T> inline bool TextStreamTmpl <T>::isEmpty () const {
    return m_textstream.isEmpty ();
}

template <class T>
inline String TextStreamTmpl <T>::release () {
    return m_textstream.release ();
}

template <class T>
inline String TextStreamTmpl <T>::readLine () {
    return m_textstream.readLine ();
}

//-----------------------------------------------------------------------------

template <class T>
inline URLTmpl <T>::URLTmpl () {}

template <class T>
inline URLTmpl <T>::URLTmpl (const String & url) : m_url (url) {}

template <class T>
inline URLTmpl <T>::URLTmpl (const T & url_impl) : m_url (url_impl) {}

template <class T>
inline URLTmpl <T>::URLTmpl (const URLTmpl <T> & base, const String & rel)
    : m_url (base.m_url, rel) {}

template <class T>
inline URLTmpl <T>::URLTmpl (const URLTmpl <T> & url) : m_url (url.m_url) {}

template <class T>
inline URLTmpl <T>::~URLTmpl () {}

template <class T>
inline URLTmpl <T> & URLTmpl <T>::operator = (const String & url) {
    m_url = url;
    return *this;
}

template <class T>
inline URLTmpl <T> & URLTmpl <T>::operator = (const URLTmpl <T> & url) {
    m_url = url.m_url;
    return *this;
}

template <class T>
inline URLTmpl <T> & URLTmpl <T>::operator = (const T & url_impl) {
    m_url = url_impl;
    return *this;
}

template <class T>
inline bool URLTmpl <T>::operator == (const URLTmpl <T> & url) const {
    return m_url.url () == url.url ();
}

template <class T>
inline String URLTmpl <T>::url () const {
    return m_url.url ();
}

template <class T>
inline String URLTmpl <T>::path () const {
    return m_url.path ();
}

template <class T>
inline String URLTmpl <T>::protocol () const {
    return m_url.protocol ();
}

template <class T>
inline String URLTmpl <T>::decode_string (const String & str) {
    return T::decode_string (str);
}

template <class T>
inline bool URLTmpl <T>::isLocalFile () const {
    return m_url.isLocalFile ();
}

//-----------------------------------------------------------------------------

template <class T>
inline String MimeTypeTmpl <T>::findByContent (const char * data, int size) {
    return T::findByContent (data, size);
}

/*template <class T>
inline String MimeTypeTmpl <T>::findByURL (const URL & url) {
    return T::findByURL (url);
}*/

//-----------------------------------------------------------------------------

template <class T>
String ConfigTmpl <T>::readEntry (const String & key, const String & def) {
    return m_config.readEntry (key, def);
}

template <class T>
int ConfigTmpl <T>::readNumEntry (const String & key, int def) {
    return m_config.readEntry (key, def);
}

template <class T>
double ConfigTmpl <T>::readNumEntry (const String & key, double def) {
    return m_config.readEntry (key, def);
}

template <class T>
void ConfigTmpl <T>::writeEntry (const String & key, const String & val) {
    m_config.writeEntry (key, val);
}

template <class T>
void ConfigTmpl <T>::writeEntry (const String & key, int val) {
    m_config.writeEntry (key, val);
}

template <class T>
void ConfigTmpl <T>::writeEntry (const String & key, double val) {
    m_config.writeEntry (key, val);
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline Log::Log () {}

KMPLAYER_NO_CDTOR_EXPORT inline Log::~Log () {}

KMPLAYER_NO_EXPORT inline Log & Log::operator << (const String & s) {
    return *this << (const char *) s;
}

KMPLAYER_NO_EXPORT inline Log & Log::operator << (const URL & u) {
    return *this << u.url ();
}

KMPLAYER_NO_EXPORT inline Log & endl (Log & log) {
    log << "\n";
    return log;
}

KMPLAYER_NO_EXPORT inline Log & Log::operator << (LOGFUNC f) {
    return (*f) (*this);
}

KMPLAYER_NO_EXPORT inline Log debugLog () {
    return Log ();
}
    
KMPLAYER_NO_EXPORT inline Log warningLog () {
    Log log;
    return log << "[WARNING] ";
}

KMPLAYER_NO_EXPORT inline Log errorLog () {
    Log log;
    return log << "[ERROR] ";
}

//-----------------------------------------------------------------------------

template <class T> inline RectTmpl <T>::RectTmpl () {}

template <class T>
inline RectTmpl <T>::RectTmpl (const RectTmpl <T> & r) : m_rect (r.m_rect) {}

template <class T>
inline RectTmpl <T>::RectTmpl (const T & rect_impl) : m_rect (rect_impl) {}

template <class T> inline
RectTmpl <T>::RectTmpl (int x, int y, int w, int h) : m_rect (x, y, w, h) {}

template <class T> inline RectTmpl <T>::~RectTmpl () {}

template <class T> inline int RectTmpl <T>::x () const {
    return m_rect.x ();
}

template <class T> inline int RectTmpl <T>::y () const {
    return m_rect.y ();
}

template <class T> inline int RectTmpl <T>::width () const {
    return m_rect.width ();
}

template <class T> inline int RectTmpl <T>::height () const {
    return m_rect.height ();
}

template <class T> inline bool RectTmpl <T>::isValid () const {
    return m_rect.isValid ();
}

template <class T>
inline RectTmpl <T> RectTmpl <T>::unite (const RectTmpl <T> & r) const {
    return m_rect.unite (r);
}

template <class T>
inline RectTmpl <T> RectTmpl <T>::intersect (const RectTmpl <T> & r) const {
    return m_rect.intersect (r);
}

template <class T> inline RectTmpl <T>::operator T & () {
    return m_rect;
}

template <class T> inline RectTmpl <T>::operator const T & () const {
    return m_rect;
}

template <class T> inline RectTmpl <T>::operator T () const {
    return m_rect;
}

template <class T>
inline RectTmpl <T> & RectTmpl <T>::operator = (const RectTmpl <T> & r) {
    m_rect = r.m_rect;
    return *this;
}

template <class T>
inline bool RectTmpl <T>::operator == (const RectTmpl <T> & r) const {
    return m_rect == r.m_rect;
}

template <class T>
inline bool RectTmpl <T>::operator != (const RectTmpl <T> & r) const {
    return !(m_rect == r.m_rect);
}

//-----------------------------------------------------------------------------

template <class T> inline ColorTmpl <T>::ColorTmpl () {}

template <class T>
inline ColorTmpl <T>::ColorTmpl (const String & desc) : m_color (desc) {}

template <class T> inline ColorTmpl <T>::ColorTmpl (int rgb) : m_color (rgb) {}

template <class T> inline
ColorTmpl <T>::ColorTmpl (const ColorTmpl <T> & c) : m_color (c.m_color) {}

template <class T>
inline ColorTmpl <T>::ColorTmpl (const T & color_impl) : m_color (color_impl) {}

template <class T> inline ColorTmpl <T>::~ColorTmpl () {}

template <class T> inline ColorTmpl <T>::operator T & () { return m_color; }

template <class T>
inline ColorTmpl <T>::operator const T & () const { return m_color; }

template <class T> inline ColorTmpl <T>::operator T () const { return m_color; }

template <class T>
inline ColorTmpl <T> & ColorTmpl <T>::operator = (const ColorTmpl <T> & c) {
    m_color = c.m_color;
    return *this;
}

template <class T>
inline ColorTmpl <T> & ColorTmpl <T>::operator = (const String & desc) {
    m_color = T (desc);
    return *this;
}

template <class T> inline ColorTmpl <T> & ColorTmpl <T>::operator = (int rgb) {
    m_color = T (rgb);
    return *this;
}

template <class T> inline unsigned ColorTmpl <T>::rgb () const {
    return m_color.rgb ();
}

//-----------------------------------------------------------------------------

template <class T> inline PixmapTmpl <T>::PixmapTmpl () {}

template <class T>
inline PixmapTmpl <T>::PixmapTmpl (int w, int h) : m_pixmap (w, h) {}

template <class T>
inline PixmapTmpl <T>::PixmapTmpl (const ByteArray & data) : m_pixmap (data) {}

template <class T>
inline PixmapTmpl <T>::PixmapTmpl (T & impl) : m_pixmap (impl) {}

template <class T>
inline PixmapTmpl <T>::PixmapTmpl (const Image & image) : m_pixmap (image) {}

template <class T> inline PixmapTmpl <T>::~PixmapTmpl () {}

template <class T> inline bool PixmapTmpl <T>::isNull () const {
    return m_pixmap.isNull ();
}

template <class T> inline int PixmapTmpl <T>::width () const {
    return m_pixmap.width ();
}

template <class T> inline int PixmapTmpl <T>::height () const {
    return m_pixmap.height ();
}

template <class T> inline void PixmapTmpl <T>::fill (const Color & c) {
    m_pixmap.fill (c);
}

template <class T> inline PixmapTmpl <T>::operator const T & () const {
    return m_pixmap;
}

//-----------------------------------------------------------------------------

template <class T> inline ImageTmpl <T>::ImageTmpl () {}

template <class T>
inline ImageTmpl <T>::ImageTmpl (int w, int h) : m_image (w, h) {}

template <class T>
inline ImageTmpl <T>::ImageTmpl (const ByteArray & data) : m_image (data) {}

template <class T>
inline ImageTmpl <T>::ImageTmpl (const T & impl) : m_image (impl) {}

template <class T> inline ImageTmpl <T>::~ImageTmpl () {}

template <class T> inline void ImageTmpl <T>::reset () { m_image.reset (); }

template <class T> inline bool ImageTmpl <T>::isNull () const {
    return m_image.isNull ();
}

template <class T> inline int ImageTmpl <T>::width () const {
    return m_image.width ();
}

template <class T> inline int ImageTmpl <T>::height () const {
    return m_image.height ();
}

template <class T> inline void ImageTmpl <T>::fill (const Color & c) {
    m_image.fill (c);
}

template <class T> inline ImageTmpl <T> ImageTmpl <T>::scale (int w, int h) {
    return ImageTmpl <T> (m_image.scale (w, h));
}

template <class T>
inline ImageTmpl <T> & ImageTmpl <T>::operator = (const ImageTmpl <T> & i) {
    m_image = i.m_image;
    return *this;
}

template <class T>
inline ImageTmpl <T> & ImageTmpl <T>::operator = (const Pixmap & pm) {
    m_image = pm;
    return *this;
}

template <class T> inline ImageTmpl <T>::operator const T & () const {
    return m_image;
}

//-----------------------------------------------------------------------------
/*
template <class T> inline PainterTmpl <T>::PainterTmpl () {}

template <class T> inline PainterTmpl <T>::~PainterTmpl () {}

template <class T> inline void PainterTmpl <T>::setClipRect (const Rect & r) {
    m_painter.setClipRect (r);
}

template <class T> inline void PainterTmpl <T>::setClipping (bool enable) {
    m_painter.setClipping (enable);
}

template <class T> inline
void PainterTmpl <T>::fillRect (int x, int y, int w, int h, const Color & c) {
    m_painter.fillRect (x, y, w, h, c);
}

template <class T> inline
bool PainterTmpl <T>::begin (const Pixmap * pm) { return m_painter.begin (pm); }

template <class T> inline
bool PainterTmpl <T>::end () { return m_painter.end (); }

template <class T> inline PainterTmpl <T>::operator T & () {
    return m_painter;
}

template <class T> inline PainterTmpl <T>::operator const T & () const {
    return m_painter;
}

template <class T>
inline void PainterTmpl <T>::drawPixmap (int dx, int dy, const Pixmap & pm, int sx, int sy, int sw, int sh) {
    m_painter.drawPixmap (dx, dy, pm, sx, sy, sw, sh);
}

template <class T>
inline void PainterTmpl <T>::drawImage (int dx, int dy, const Image & im, int sx, int sy, int sw, int sh) {
    m_painter.drawImage (dx, dy, im, sx, sy, sw, sh);
}
*/
//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline ByteArrayImpl::ByteArrayImpl ()
 : m_data (0L), m_size (0), m_capacity (0) {}

KMPLAYER_NO_CDTOR_EXPORT inline ByteArrayImpl::ByteArrayImpl (unsigned sz)
 : m_data ((char *) malloc (sz)), m_size (sz), m_capacity (sz) {}

KMPLAYER_NO_CDTOR_EXPORT inline ByteArrayImpl::ByteArrayImpl (const ByteArrayImpl & b)
 : m_data (b.m_size ? (char *) malloc (b.m_size) : 0L),
   m_size (b.m_size),
   m_capacity (b.m_size) {
    if (m_size)
        memcpy (m_data, b.m_data, m_size);
}

KMPLAYER_NO_CDTOR_EXPORT inline ByteArrayImpl::~ByteArrayImpl () {
    if (m_data)
        free (m_data);
}

KMPLAYER_NO_EXPORT inline unsigned ByteArrayImpl::size () const {
    return m_size;
}

KMPLAYER_NO_EXPORT inline char * ByteArrayImpl::data () const {
    return m_data;
}

KMPLAYER_NO_EXPORT inline bool ByteArrayImpl::isEmpty () const {
    return (!m_data || m_size ==0);
}

KMPLAYER_NO_EXPORT
inline ByteArrayImpl & ByteArrayImpl::operator = (const ByteArrayImpl & b) {
    if (m_data)
        free (m_data);
    m_data = (b.m_size ? (char*) malloc (b.m_size) : 0L);
    m_size = b.m_size;
    m_capacity = b.m_size;
    if (m_size)
        memcpy (m_data, b.m_data, m_size);
}

//-----------------------------------------------------------------------------

inline Shared::Shared () : data (0) {}

inline Shared::~Shared () {
    clear ();
}

inline bool Shared::isNull () const {
    return !data;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GLibString::GLibString () {}

KMPLAYER_NO_CDTOR_EXPORT inline GLibString::GLibString (const GLibString & s) {
    attach (s);
}

KMPLAYER_NO_CDTOR_EXPORT
inline GLibString::GLibString (const char * str, int maxlen) {
    if (str)
        take (maxlen > -1 ? g_strndup (str, maxlen) : g_strdup (str));
}

KMPLAYER_NO_EXPORT inline GLibString::~GLibString () {}

KMPLAYER_NO_EXPORT
inline GLibString & GLibString::operator = (const GLibString & str) {
    attach (str);
    return *this;
}

KMPLAYER_NO_EXPORT
inline GLibString & GLibString::operator += (const char * str) {
    return *this += GLibString (str);
}

KMPLAYER_NO_EXPORT
inline GLibString & GLibString::operator += (const Char & ch) {
    return *this += GLibString (ch);
}

KMPLAYER_NO_EXPORT inline GLibString::operator const char * () const {
    return data ? data->ptr : "";
}

KMPLAYER_NO_EXPORT inline Char GLibString::operator [] (int index) const {
    // unchecked pointer, TODO add a modifying operator []
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    return Char (g_utf8_get_char (off));
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::number (int nr) {
    GLibString str;
    str.attach (g_strdup_printf ("%d", nr));
    return str;
}

KMPLAYER_NO_EXPORT inline bool GLibString::isEmpty () const {
    return !data || !data->ptr [0];
}

KMPLAYER_NO_EXPORT inline int GLibString::length () const {
    return data ? g_utf8_strlen (data->ptr, -1) : 0;
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::lower () const {
    GLibString str;
    if (data)
        str.take (g_utf8_strdown (data->ptr, -1));
    return str;
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::upper () const {
    GLibString str;
    if (data)
        str.take (g_utf8_strup (data->ptr, -1));
    return str;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GLibTextStream::GLibTextStream ()
    //: m_gstring (g_string_new ("")), m_utf8_str (0L), m_iter (0L) {}
    : m_gstring (g_string_sized_new (128)), m_utf8_str (0L), m_iter (0L) {}

KMPLAYER_NO_CDTOR_EXPORT
inline GLibTextStream::GLibTextStream (const String & init)
    : m_gstring (g_string_new ((const char *) init)),
      m_utf8_str (0L),
      m_iter (0L) {}

KMPLAYER_NO_CDTOR_EXPORT
inline GLibTextStream::GLibTextStream (ByteArray & init)
    : m_gstring (g_string_new_len (init.data (), init.size ())),
      m_utf8_str (0L),
      m_iter (0L) {}

KMPLAYER_NO_CDTOR_EXPORT inline GLibTextStream::~GLibTextStream () {
    if (m_gstring)
        g_string_free (m_gstring, true);
    else if (m_utf8_str)
        g_free (m_utf8_str);
}

KMPLAYER_NO_EXPORT inline String GLibTextStream::release () {
    GLibString str;
    if (m_gstring) {
        str .attach (g_string_free (m_gstring, false));
        m_gstring = 0L;
    }
    return str;
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (const char * str) {
    if (m_gstring)
        g_string_append (m_gstring, (const char *) str);
    return *this;
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (const String & str) {
    return *this << (const char *) str;
}

KMPLAYER_NO_EXPORT inline GLibTextStream & GLibTextStream::operator << (const Char & ch) {
    if (m_gstring)
        g_string_append_unichar (m_gstring, (gunichar) ch.unicode ());
    return *this;
}

KMPLAYER_NO_EXPORT inline
GLibTextStream & GLibTextStream::readRawBytes (char * b, unsigned len) {
    if (m_gstring)
        g_string_append_len (m_gstring, b, len);
    return *this;
}

KMPLAYER_NO_EXPORT inline bool GLibTextStream::atEnd () const {
    return (m_iter && !m_iter[0]) || (!m_gstring && !m_iter);
}

KMPLAYER_NO_EXPORT inline bool GLibTextStream::isEmpty () const {
    return (!m_gstring && (!m_utf8_str || !m_utf8_str [0]) ||
            (m_gstring && !m_gstring->len));
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GnomeVFSUrl::GnomeVFSUrl () : m_vfsurl (0L) {}

//-----------------------------------------------------------------------------

/*inline String GMimeType::findByURL (const URL & url) {
    return String (gnome_vfs_get_mime_type ((const char *) url.url ()));
}*/

//-----------------------------------------------------------------------------

KMPLAYER_NO_EXPORT
inline String GConf::readEntry (const String & key, const String & def) {
    gchar * val = gconf_client_get_string (client, (const char *) key, 0L);
    return val ? String (val) : def;
}

KMPLAYER_NO_EXPORT
inline void GConf::writeEntry (const String & key, const String & val) {
    gconf_client_set_string (client, (const char *) key, (const char *)val, 0L);
}

KMPLAYER_NO_EXPORT
inline void GConf::writeEntry (const String & key, int val) {
    gconf_client_set_int (client, (const char *) key, val, 0L);
}

KMPLAYER_NO_EXPORT
inline void GConf::writeEntry (const String & key, double val) {
    gconf_client_set_float (client, (const char *) key, val, 0L);
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT
inline GRect::GRect () : m_rect (g_new (GdkRectangle, 1)) {
    m_rect->x = 0; m_rect->y = 0; m_rect->width = 0; m_rect->height = 0;
}

KMPLAYER_NO_CDTOR_EXPORT
inline GRect::GRect (const GRect & r) : m_rect (g_new (GdkRectangle, 1)) {
    *m_rect = *r.m_rect;
}

KMPLAYER_NO_CDTOR_EXPORT inline GRect::GRect (int x, int y, int w, int h)
 : m_rect (g_new (GdkRectangle, 1)) {
    m_rect->x = x; m_rect->y = y; m_rect->width = w; m_rect->height = h;
}

KMPLAYER_NO_CDTOR_EXPORT inline GRect::~GRect () {
    g_free (m_rect);
}

KMPLAYER_NO_EXPORT inline int GRect::x () const { return m_rect->x; }

KMPLAYER_NO_EXPORT inline int GRect::y () const { return m_rect->y; }

KMPLAYER_NO_EXPORT inline int GRect::width () const { return m_rect->width; }

KMPLAYER_NO_EXPORT inline int GRect::height () const { return m_rect->height; }

KMPLAYER_NO_EXPORT inline bool GRect::isValid () const {
    return m_rect->width > 0 && m_rect->height > 0;
}

KMPLAYER_NO_EXPORT inline GRect & GRect::operator = (const GRect & r) {
    *m_rect = *r.m_rect;
    return *this;
}

KMPLAYER_NO_EXPORT inline GRect::operator GdkRectangle * () const {
    return m_rect;
}

KMPLAYER_NO_EXPORT inline bool GRect::operator == (const GRect & r) const {
    return m_rect->x == r.m_rect->x && m_rect->y == r.m_rect->y &&
        m_rect->width == r.m_rect->width && m_rect->height == r.m_rect->height;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GColor::GColor () : m_color (0L) {}

KMPLAYER_NO_CDTOR_EXPORT inline GColor::GColor (const GColor & c)
    : m_color (c.m_color ? gdk_color_copy (c.m_color) : 0L) {}

KMPLAYER_NO_EXPORT inline void GColor::clear () {
    if (m_color) {
        gdk_colormap_free_colors (gdk_colormap_get_system (), m_color, 1);
        //g_free (m_color); // ??
        m_color = 0L;
    }
}

KMPLAYER_NO_CDTOR_EXPORT inline GColor::~GColor () {
    clear ();
}

KMPLAYER_NO_EXPORT inline GColor & GColor::operator = (const GColor & c) {
    clear ();
    m_color = c.m_color ? gdk_color_copy (c.m_color) : 0L;
    return *this;
}

KMPLAYER_NO_EXPORT inline GColor::operator GdkColor * () const {
    return m_color;
}

KMPLAYER_NO_EXPORT inline unsigned GColor::rgb () const {
    return m_color ? ((m_color->red >> 8) << 16) | ((m_color->green >> 8) << 8) | (m_color->blue >> 8) : 0L;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_EXPORT inline GPixmap::GPixmap () : m_pixmap (0L) {}

KMPLAYER_NO_EXPORT inline bool GPixmap::isNull () const { return !m_pixmap; }

KMPLAYER_NO_EXPORT inline
GPixmap::operator GdkDrawable * () const { return GDK_DRAWABLE (m_pixmap); }

KMPLAYER_NO_EXPORT
inline GdkPixmap * GPixmap::pixmap () const { return m_pixmap; }

//-----------------------------------------------------------------------------

KMPLAYER_NO_EXPORT inline GPixbuf::GPixbuf () : m_pixbuf (0L) {}

KMPLAYER_NO_EXPORT inline bool GPixbuf::isNull () const { return !m_pixbuf; }

KMPLAYER_NO_EXPORT inline
GPixbuf::operator GdkPixbuf * () const { return m_pixbuf; }

KMPLAYER_NO_EXPORT
inline GdkPixbuf * GPixbuf::pixbuf () const { return m_pixbuf; }

KMPLAYER_NO_EXPORT inline int GPixbuf::width () const {
    return m_pixbuf ? gdk_pixbuf_get_width (m_pixbuf) : 0;
}

KMPLAYER_NO_EXPORT inline int GPixbuf::height () const {
    return m_pixbuf ? gdk_pixbuf_get_height (m_pixbuf) : 0;
}

} // namespace KMPlayer

#include "triestring.h"

#endif  // _KMPLAYER_TYPES_H_

