// -*- c++ -*-
//------------------------------------------------------------------------------
// $Id: DeckPlayer.cpp,v 1.122 2007/01/14 02:59:57 vlg Exp $
//------------------------------------------------------------------------------
//                            DeckPlayer.cpp
//------------------------------------------------------------------------------
//  Copyright (c) 2004-2006 by Vladislav Grinchenko 
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Fri Feb  6 23:37:11 EST 2004
//
//------------------------------------------------------------------------------

#include <gtkmm/stock.h>
#include <gtkmm/separator.h>
#include <gtkmm/spinbutton.h>
#include <gtkmm/box.h>
#include <gtkmm/handlebox.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/radiobutton.h>
#include <gtkmm/frame.h>
#include <gtkmm/toolbar.h>
#include <gtkmm/textattributes.h>
#include <gtkmm/main.h>
#include <gtkmm/progressbar.h>
#include <gtkmm/eventbox.h>

using sigc::bind;
using sigc::mem_fun;

#ifdef IS_HILDON
#   include <hildon-libsmm.h>
#endif

#include "ButtonWithImageLabel.h"

#include "Granule-main.h"
#include "Granule.h"
#include "MainWindow.h"
#include "Card.h"
#include "Deck.h"
#include "CardBox.h"
#include "DeckPlayer.h"
#include "DeckView.h"
#include "CardView.h"
#include "Intern.h"

static const Glib::ustring clabels [] = { 
	N_("Check"), N_("Next "), N_("Correct  "), N_("Incorrect"), 
	"               "
};

enum { 
	CHECK_ANSWER, 
	NEXT, 
	CORRECT, 
	INCORRECT, 
	EMPTY 
};

static const gchar* states [] = { "VC_START", "VC_ARMED", "VC_VALIDATED" };

static const int QUESTION_PADDING_X   =4; // Distance between text and an edge
static const int QUESTION_PADDING_Y   =0; // Distance between text and an edge

static const int EXAMPLE_PADDING_X    =2; // Distance between text and an edge
static const int EXAMPLE_PADDING_Y    =0; // Distance between text and an edge

#ifdef IS_PDA
static const int ANSWER_BOX_SPACING   =0; // How far box is placed from the edge
static const int ANSWER_CHILD_SPACING =0; // Spacing between child widgets
#else
static const int ANSWER_BOX_SPACING   =4; 
static const int ANSWER_CHILD_SPACING =4; 
#endif

#if defined(IS_PDA) && (GTKMM_MAJOR_VERSION == 2 && GTKMM_MINOR_VERSION >= 6)
#  define DP_SHRINK_ICONS ,10,10,false
#else
#  define DP_SHRINK_ICONS
#endif

static int ustring_to_int (const ustring& s_)
{
	return (::atoi (s_.c_str ()));
}

/*******************************************************************************
 *                       TestLineEntry Member Functions                        *
 *******************************************************************************
 */
TestLineEntry::
TestLineEntry (VerifyControl* vc_)
	:
	m_adjustment (1, 1, 10),
	m_label (_("Test line: "))
{
#ifdef IS_PDA
	m_entry.set_flags (Gtk::CAN_FOCUS);
	m_entry.set_visibility (true);
	m_entry.set_editable (true);
	m_entry.set_has_frame (false);
	m_entry.set_activates_default (false); // Don't close dialog on [Enter]
	m_entry.set_max_length (1);
	m_entry.set_width_chars (1);
	m_entry.set_text ("1");

	pack_start (m_entry, Gtk::PACK_SHRINK, 0);

	m_entry.signal_changed ().connect (
			mem_fun (*vc_, &VerifyControl::on_test_line_changed));
#else
	m_test_line_button.set_adjustment (m_adjustment);
	pack_start (m_label, Gtk::PACK_SHRINK, 0);
	pack_start (m_test_line_button, Gtk::PACK_SHRINK, 0);

	m_test_line_button.signal_value_changed ().connect (
		mem_fun (*vc_, &VerifyControl::on_test_line_changed));
#endif

	vc_->attach (this);
	vc_->set_focus_on_entry ();
}

int
TestLineEntry::
get_value_as_int () const
{
	int v;

#ifdef IS_PDA
	Glib::ustring s = m_entry.get_text ();
	DL ((GRAPP,"TestLineEntry::get_value_as_int: s=\"%s\"\n", s.c_str ()));
	if (s == "" || s == "0") {
		v = 1;
		TestLineEntry* tle_ptr = const_cast<TestLineEntry*> (this);
		tle_ptr->m_entry.set_text ("1");
	}
	else {
		v = ::atoi (s.c_str ());
	}
#else
	v = m_test_line_button.get_value_as_int ();
#endif

	return (v);
}


/*******************************************************************************
 *                      SideFlipControl Member Functions                       *
 *******************************************************************************
 */
SideFlipControl::
SideFlipControl (DeckPlayer& parent_)
	: m_parent (parent_)
{
#ifdef IS_PDA
	m_flip_button.set_flags  (Gtk::CAN_FOCUS);
	m_flip_button.set_relief (Gtk::RELIEF_NORMAL);

	m_flip_button.set_label  ("F");
    m_side = FRONT;

	m_flip_button.signal_clicked ().connect (
		mem_fun (*this, &SideFlipControl::on_flip_button_clicked));

	add (m_flip_button);

#else  // Desktop, Hildon

    Gtk::RadioButton::Group group;

	m_radio_fb.set_group (group);
	m_radio_bb.set_group (group);
	
    m_radio_fb.set_label (_("Ask front"));
    m_radio_bb.set_label (_("Ask back"));

    m_button_box.pack_end (m_radio_fb, false, false, 1); 
    m_button_box.pack_end (m_radio_bb, false, false, 1); 

    m_radio_fb.signal_clicked ().connect (bind<SideSelection>(
			mem_fun (parent_, &DeckPlayer::on_radio_selected), FRONT));
    m_radio_bb.signal_clicked ().connect (bind<SideSelection>(
			mem_fun (parent_, &DeckPlayer::on_radio_selected), BACK));

#ifndef IS_HILDON
    m_button_box.set_homogeneous ();
    m_button_box.set_spacing (1);
#endif	

	add (m_button_box);
#endif
}

void
SideFlipControl::
set_active (SideSelection s_)
{
#ifdef IS_PDA
	if (s_ != m_side) {
		on_flip_button_clicked ();
	}
#else
	if (s_ == FRONT) {
		m_radio_fb.set_active ();
	}
	else {
		m_radio_bb.set_active ();
	}
#endif
}

void
SideFlipControl::
on_flip_button_clicked ()
{
#ifdef IS_PDA
	if (m_side == FRONT) {
		m_flip_button.set_label ("B");
		m_parent.on_radio_selected (BACK);
		m_side = BACK;
	}
	else {
		m_flip_button.set_label ("F");
		m_parent.on_radio_selected (FRONT);
		m_side = FRONT;
	}
#endif
}

/*******************************************************************************
 *                       VerifyControl Member Functions                        *
 *                                                                             *
 * Widget composition:                                                         *
 * ==================                                                          *
 *   Table                                                                     *
 *     +--hbox1                                                                *
 *     |    +--Entry      (m_entry)       <-- modify_base/text                 *
 *     |    +--Button     (m_button)                                           *
 *     |    +--Label      (blank1)                                             *
 *     |                                                                       *
 *     +--hbox2                                                                *
 *          +--EventBox   (m_report_box)                                       *
 *          |    +--Label (m_report)                                           *
 *          +--Label      (blank2)                                             *
 *                                                                             *
 *******************************************************************************
 */
VerifyControl::
VerifyControl (DeckPlayer& parent_) 
	: 
	Gtk::Table  (2, 2, false),
	m_parent    (parent_),
	m_state     (VC_START),
	m_red       ("red"),
	m_green     ("MediumSeaGreen"),
	m_white     ("white"),
	m_black     ("black"),
	m_test_line (NULL),
	m_line_num  (1)
	
{
    trace_with_mask("VerifyControl::VerifyControl", GUITRACE);

	Gtk::Table* table = this;

	m_report = Gtk::manage (new Gtk::Label);

	m_report->set_alignment  (0, 0.5);
	m_report->set_padding    (0,0);
	m_report->set_justify    (Gtk::JUSTIFY_FILL);
	m_report->set_line_wrap  (false);
	m_report->set_use_markup (false);
	m_report->set_selectable (false);
#if !defined (IS_PDA)
	m_report->set_size_request (80, -1); // width, height
#endif
	m_report->set_text (clabels [EMPTY]);
	m_report->set_sensitive (false);

	m_report_box = Gtk::manage (new Gtk::EventBox);
	
    m_entry = Gtk::manage (new Gtk::Entry);
    m_entry->set_flags       (Gtk::CAN_FOCUS);
    m_entry->set_visibility  (true);
    m_entry->set_editable    (true);
    m_entry->set_max_length  (0);
    m_entry->set_text        ("");
    m_entry->set_has_frame   (true);
    m_entry->set_width_chars (30);
    m_entry->set_activates_default (false);
	m_entry->modify_font     (CONFIG->get_input_font ());

	Gdk::Color text_bgcolor (CONFIG->text_bgcolor ());
	Gdk::Color text_fgcolor (CONFIG->text_fgcolor ());

	m_entry->modify_base   (Gtk::STATE_NORMAL, text_bgcolor);
	m_entry->modify_text   (Gtk::STATE_NORMAL, text_fgcolor);

	for (int i = 0; i < LABELS_SZ; i++) {
		m_label [i] = Gtk::manage (new Gtk::Label (clabels [i]));
		m_label [i]->set_alignment  (0, 0.5);
		m_label [i]->set_padding    (0,0);
		m_label [i]->set_justify    (Gtk::JUSTIFY_FILL);
		m_label [i]->set_line_wrap  (false);
		m_label [i]->set_use_markup (false);
		m_label [i]->set_selectable (false);
	}
	m_button = Gtk::manage  (new Gtk::Button);
    m_button->set_flags     (Gtk::CAN_FOCUS);
    m_button->set_relief    (Gtk::RELIEF_NORMAL);
	m_button->set_label     (clabels [CHECK_ANSWER]);
	m_button->set_sensitive (false);

	Gtk::HBox* hbox1  = manage (new Gtk::HBox (false, 0));
	Gtk::HBox* hbox2  = manage (new Gtk::HBox (false, 0));

	Gtk::Label* blank1 = manage (new Gtk::Label);
	Gtk::Label* blank2 = manage (new Gtk::Label);

	blank1->set_text ("            ");
	blank2->set_text ("            ");

	m_report_box->add  (*m_report);

	hbox1->pack_start (*m_entry,  Gtk::PACK_EXPAND_WIDGET, 3);
	hbox1->pack_start (*m_button, Gtk::PACK_SHRINK, 0);
#if !defined(IS_PDA)
	hbox1->pack_start (*blank1,   Gtk::PACK_SHRINK, 0);
#endif

	hbox2->pack_start (*m_report_box, Gtk::PACK_SHRINK, 0);

#if !defined(IS_PDA)
	hbox2->pack_start (*blank2,       Gtk::PACK_SHRINK, 0);
#endif

	table->set_row_spacings (0);
	table->set_col_spacings (2);

	table->attach (*hbox1, 0, 1, 0, 1, 
				   Gtk::EXPAND|Gtk::SHRINK|Gtk::FILL, 
				   Gtk::FILL, 
				   0, 0);

	table->attach (*hbox2, 1, 2, 0, 1, 
				   Gtk::EXPAND|Gtk::FILL, 
				   Gtk::FILL, 
				   0, 0);

	m_entry->signal_changed ().connect (
		mem_fun (*this, &VerifyControl::on_entry_changed));

    m_button->signal_clicked ().connect (
        mem_fun (m_parent, &DeckPlayer::on_verify_clicked));
}

void
VerifyControl::
set_focus_on_entry () 
{ 
    trace_with_mask("VerifyControl::set_focus_on_entry", KEYIN);
	m_entry->grab_focus (); 
}

void
VerifyControl::
on_test_line_changed ()
{
    trace_with_mask("VerifyControl::on_test_line_changed", GUITRACE);

	if (m_test_line != NULL) {
		m_line_num = m_test_line->get_value_as_int ();
		DL ((GRAPP,"Testing line #%d\n", m_line_num));
	}
}

/**
 * Here is the tricky part. The answer_ can be marked up with
 * Pango markup - so strip it before comparing. Strip all well-known
 * prefixes as well and reduce multiple white spaces to a single one.
 */
void
VerifyControl::
compare (const ustring& answer_)
{
    trace_with_mask("VerifyControl::compare", GUITRACE);

	m_button->set_label (clabels [NEXT]);

	gchar* pc1;
	gchar* text1;
	gchar* pc2;
	gchar* text2;
	pc1 = text1 = pc2 = text2 = NULL;

	Glib::ustring tmp (answer_);

    /** If we are comparing the back of the card and
		the m_line_num > 1, then we fetch that line instead.
		If we are short of lines, that's a failed match.
	*/
	if (m_line_num > 1) 
	{
		Glib::ustring::value_type end_of_line ('\n');
		std::vector<Glib::ustring> vec;
		Glib::ustring token;
		ustring::const_iterator iter = answer_.begin ();
		while (iter != answer_.end ()) {
			if (*iter == end_of_line) {
				vec.push_back (token);
				token = "";
			}
			else {
				token += *iter;
			}
			iter++;
		}
		if (token.size ()) {
			vec.push_back (token);
		}

		if (vec.size () >= m_line_num) {
			tmp = vec [m_line_num-1];
		}
		else {
			tmp = "";			// Selected answer line is missing
		}
	}

	DeckPlayer::shrink_to_single_line (tmp);

	DL ((GRAPP,"tmp = \"%s\"\n", tmp.c_str ()));

	text1 = pc1 = Granule::strip_pango_markup (tmp.c_str ());
	Granule::remove_common_prefixes (text1);

	text2 = pc2 = Granule::strip_pango_markup (m_entry->get_text ().c_str ());
	Granule::remove_common_prefixes (text2);

	DL ((GRAPP,"m_entry = \"%s\"\n", text2));
	DL ((GRAPP,"answer  = \"%s\"\n", text1));
	m_was_equal = true;

	if (!strcmp (text1, text2)) {
		m_report_box->modify_fg (Gtk::STATE_INSENSITIVE, m_green);
		m_report->modify_fg     (Gtk::STATE_INSENSITIVE, m_green);
		m_report->set_label     (clabels [CORRECT]);
	}
	else {
		m_report_box->modify_fg (Gtk::STATE_INSENSITIVE, m_red);
		m_report->modify_fg     (Gtk::STATE_INSENSITIVE, m_red);
		m_report->set_label     (clabels [INCORRECT]);

		m_was_equal= false;

		/** Remember, you are comparing UTF-8 string and simple
			gchar* arithmetic won't work.
		*/
		Glib::ustring lhs (text1);
		Glib::ustring rhs (text2);

		for (int idx = 0; idx < lhs.size () && idx < rhs.size (); idx++) 
		{
			if (lhs [idx] != rhs [idx]) 
			{
				m_entry->select_region (idx, -1);
				break;
			}
		}
	}

	m_entry->set_editable (false);
	m_parent.flip_card ();
	set_state (VC_VALIDATED);
	g_free (pc1);
	g_free (pc2);
}

void
VerifyControl::
on_entry_changed ()
{
    trace_with_mask("VerifyControl::on_entry_changed", KEYIN);

	m_button->set_sensitive (true);

	if (get_state () != VC_ARMED     &&
		m_entry->get_text () != " "  && 
		m_entry->get_text ().size ()) 
	{
		m_parent.disable_answer_controls ();
		set_state (VC_ARMED);
		Glib::ustring str ("<span size=\"medium\">");
		str += m_entry->get_text ();
		str += "</span>";
		m_entry->get_layout ()->set_markup (str);
		DL ((KEYIN,"Gtk::Entry markup enabled\n"));
	}
}

void
VerifyControl::
clear ()
{
	set_state (VC_START);

	m_report->set_text (clabels [EMPTY]);
	m_report->set_sensitive (false);
	m_report_box->modify_fg   (Gtk::STATE_INSENSITIVE, m_black);

	m_entry->set_text ("");
	m_entry->set_editable (true);

	m_button->set_label (clabels [CHECK_ANSWER]);
	m_button->set_sensitive (false);

	set_focus_on_entry ();
	m_parent.enable_answer_controls ();
}

VCState 
VerifyControl::
get_state () const
{
//	DL ((GRAPP,"current state = %s\n", states [m_state]));
	return (m_state);
}
void 
VerifyControl::
set_state (VCState state_)
{
	if (m_state != state_) {
		DL ((GRAPP,"change of state: %s -> %s\n", 
			 states [m_state], states [state_]));
		m_state = state_;
	}
}

/*******************************************************************************
 *                         DeckPlayer Member Functions                         *
 *******************************************************************************
 */
Gtk::StockID DeckPlayer::SAY_IT       ("say-it");
Gtk::StockID DeckPlayer::THUMB_UP     ("thumb-up");
Gtk::StockID DeckPlayer::THUMB_DOWN   ("thumb-down");
//Gtk::StockID DeckPlayer::SHOW_CONTROL ("show-control");
//Gtk::StockID DeckPlayer::HIDE_CONTROL ("hide-control");

DeckPlayer::
DeckPlayer (VDeck& vdeck_) 
		:
    m_close_button      (manage (new ButtonWithImageLabel (
									 Gtk::Stock::CLOSE, _("Close")))),
#ifdef IS_PDA
    m_shuffle_button    (manage (new ButtonWithImageLabel (
									 Gtk::Stock::REFRESH, ""))),
    m_edit_button       (manage (new ButtonWithImageLabel (
									 Gtk::Stock::DND_MULTIPLE, ""))),
    m_vcard_edit_button (manage (new ButtonWithImageLabel (
									 Gtk::Stock::DND, ""))),
#else /* DESKTOP */
    m_shuffle_button    (manage (new ButtonWithImageLabel (
									 Gtk::Stock::REFRESH, _("Shuffle")))),
    m_edit_button       (manage (new ButtonWithImageLabel (
									 Gtk::Stock::DND_MULTIPLE,
									 _("Edit Deck")))),
    m_vcard_edit_button (manage (new ButtonWithImageLabel (
									 Gtk::Stock::DND, _("Edit Card")))),
#endif
    m_selected_side     (FRONT),
    m_notebook          (manage (new Gtk::Notebook)),
    m_play_progress     (manage (new Gtk::ProgressBar)),
    m_question_box_width(0),
    m_answer_box_width  (0),
	m_verify_control    (manage (new VerifyControl (*this))),
    m_deck              (vdeck_),
    m_deck_iter         (m_deck.end ()),         /* empty deck */
    m_is_lesson_learned (false),
    m_is_real_deck      (true),
	m_tab_activated     (false),
	m_initialized       (false)
{
    trace_with_mask("DeckPlayer::DeckPlayer", GUITRACE);

	Gtk::Viewport *viewport;

    set_title ("DeckPlayer");

	m_text_fgcolor = CONFIG->text_fgcolor ();
	m_text_bgcolor = CONFIG->text_bgcolor ();
	m_sep_color    = CONFIG->sep_color    ();

	/**
	 * Not a dialog any more - it is a Bin (AppView) instead.
	 * We handle packaging ourselves and mute all attributes of a 
	 * Dialog everywhere down the line.
	 */
    m_vbox = manage (new Gtk::VBox (false, 0));
	add (*m_vbox);

#ifdef IS_HILDON
	/** 
	 * hildon-specific
	 */
#elif IS_PDA
    /** 
	 *  Make DeckPlayer expand to fill the whole screen. The screen size
	 *  is 240x320, but oddly enough, the WM still leave a few pixels
	 *  around. BTW, set_size_request() doesn't work - it makes the dialog
	 *  shorter then the MainWindow by the order of the titlebar height.
	 *  And, stay away from setting transient on parent.
	 */
	Gdk::Geometry dp_geometry =	{ 240, 320,	240, 320, 240, 320, -1, -1, -1, -1};
	set_geometry_hints (*this, dp_geometry, 
						Gdk::HINT_MIN_SIZE | 
						Gdk::HINT_MAX_SIZE | 
						Gdk::HINT_BASE_SIZE);

#else  // Desktop
	/** For desktop version we can resize and 
	 *  move the window around freely.
	 */	
	set_modal (true);
	set_transient_for (*MAINWIN);
    set_resizable (true);
	Gdk::Rectangle geom = CONFIG->get_deck_player_geometry ();
	DL ((GRAPP,"Setting size to width=%d, height=%d\n",
		 geom.get_width (), geom.get_height ()));
	resize (geom.get_width (), geom.get_height ());

    /** 
	 *  For desktop only, we optionally reinforce the card's geometry
	 *  ratio as setup in Preferences->Geometry:
	 *
	 *  "The other useful fields are the min_aspect and max_aspect fields; 
	 *   these contain a width/height ratio as a floating point number. 
	 *   If a geometry widget is set, the aspect applies to the geometry 
	 *   widget rather than the entire window. The most common use of these 
	 *   hints is probably to set min_aspect and max_aspect to the same value, 
	 *   thus forcing the window to keep a constant aspect ratio. 
	 */
	if (CONFIG->keep_deck_player_aspect ()) 
	{
		Gdk::Rectangle& rec = CONFIG->get_deck_player_aspect ();
		gdouble aspect = rec.get_width () * 1.0 / rec.get_height ();
		Gdk::Geometry answer_box_geometry =  { 
			geom.get_width(),/*min_width*/ geom.get_height (), /* min_height  */
			10000,	         /* max_width; */	10000,	       /* max_height  */
			-1,	             /* base_width */	-1,	           /* base_height */
			-1,	             /* width_inc  */	-1,	           /* height_inc  */
			aspect,	         /* min_aspect (width/height) */
			aspect	         /* max_aspect (width/height) */
		};
		set_geometry_hints (*this, answer_box_geometry, 
							Gdk::HINT_ASPECT | Gdk::HINT_MIN_SIZE);
	}
#endif	// !(IS_PDA)

    /** Are we dealing with CardDeck?
     */
    if (dynamic_cast<Deck*>(&m_deck) == NULL) {
		m_is_real_deck = false;
		DL((GRAPP,"Deck isn't REAL!\n"));
    }
    else {
		DL((GRAPP,"Yep, this IS a real deck\n"));
    }

    /** HBox holds both the notebook and controls/info side panel
     */
    Gtk::HBox* nb_box = manage (new Gtk::HBox (false, 2));
    nb_box->pack_start (*m_notebook, Gtk::PACK_EXPAND_WIDGET, 2);

    /********************
	 * Add the notebook *
	 ********************
     */
    m_notebook->set_tab_pos     (Gtk::POS_BOTTOM);
    m_notebook->set_show_border (true);

    /***************************************************************************
	 *                    Fill up notebook with elements                       *
	 *                                                                         *
	 * COLORS: We defeat the theme settings and set our own colors             *
	 *         according to the configuration file. The foreground color       *
	 *         can be set on the Gtk::Label itself via modify_fg(Gdk::Color).  *
	 *         The background color, however, must be set on the enclosing     *
	 *         container via modify_bg(Gdk::Color). But strangely enough,      *
	 *         if you put Gtk::Label into Gtk::ScrolledWindow which is a       *
	 *         Gtk::Container, it won't work! You need a Gtk::EventBox         *
	 *         sandwitched inbetween (???). Gtk::ScrolledWindow takes its      *
	 *         background color hint from the theme style, period. BTW, same   *
	 *         goes for Gtk::Box as well.                                      *
	 *                                                                         *
	 * For more, read                                                          *
	 *         <http://ometer.com/gtk-colors.html> - Gtk Colors Mini HOWTO     *
	 *                                                                         *
	 ***************************************************************************
     */

	/***************************************************************************
	 * Front of the card composition                                           *
	 *                                                                         *
	 * Question Label:                                                         *
	 * ==============                                                          *
	 *         Notebook                                                        *
	 *           +--Page                                                       *
	 *                +--ScrolledWindow                                        *
	 *                     +--VBox                                             *
	 *                          +--EventBox          <- modify_bg              *
	 *                               +--Label        <- modify_fg              *
	 *                                                                         *
     ***************************************************************************
	 */
    m_question_box = manage (new Gtk::VBox (false, ANSWER_BOX_SPACING));
	m_question_box->set_border_width (ANSWER_CHILD_SPACING); 

	m_front.set_alignment  (x_alignment_enum (FRONT),
						    y_alignment_enum (FRONT));
	m_front.set_padding    (x_padding_val (FRONT),
						    y_padding_val (FRONT));
	m_front.set_justify    (justification_enum (FRONT));
    m_front.set_use_markup ();
	m_front.set_line_wrap  ();
    m_front.modify_font    (question_font ());

	m_front.modify_fg   (Gtk::STATE_NORMAL, m_text_fgcolor);

#if !defined(IS_HILDON)
    m_front.set_selectable ();
#endif

 	Gtk::EventBox* question_evbox = Gtk::manage (new Gtk::EventBox);
	question_evbox->modify_bg   (Gtk::STATE_NORMAL, m_text_bgcolor);
	question_evbox->add (m_front);
	m_question_box->pack_start (*question_evbox);

	Gtk::ScrolledWindow* scrollw;

	scrollw = Gtk::manage (new Gtk::ScrolledWindow);
	scrollw->set_flags (Gtk::CAN_FOCUS);
	scrollw->set_shadow_type (Gtk::SHADOW_NONE);

#ifdef GLIBMM_PROPERTIES_ENABLED
	scrollw->property_window_placement ().set_value (Gtk::CORNER_TOP_LEFT);
#else
	scrollw->set_property ("window_placement", Gtk::CORNER_TOP_LEFT);
#endif

	scrollw->set_policy (Gtk::POLICY_NEVER , Gtk::POLICY_AUTOMATIC);
	scrollw->add (*m_question_box);

    m_notebook->pages ().push_back (
		Gtk::Notebook_Helpers::TabElem (*scrollw, _("Front")));

	/***************************************************************************
	 * Back of the card composition:                                           *
	 * ============================                                            *
	 *  Notebook                                                               *
	 *    +--Page                                                              *
	 *         +--ScrolledWindow                                               *
	 *              +--Viewport                               <- modify_bg     *
	 *                     +--VBox (m_answer_box)                              *
	 *                          +--Label (m_back)             <- modify_fg     *
	 *                          +--EventBox                   <- modify_bg     *
	 *                               +--HSeparator                             *
	 *                          +--Label (m_example)          <- modify_fg     *
	 *                                                                         *
     ***************************************************************************
	 */
    m_answer_box = manage (new Gtk::VBox (false, ANSWER_BOX_SPACING));
	m_answer_box->set_border_width (ANSWER_CHILD_SPACING); 

    /* -Back (the answer)- 
	 */
	m_back.set_alignment  (x_alignment_enum (BACK), y_alignment_enum (BACK));
	m_back.set_padding    (x_padding_val (BACK), y_padding_val (BACK));
	m_back.set_justify    (justification_enum (BACK));

    m_back.modify_font    (CONFIG->get_answer_font ());
	m_back.modify_fg      (Gtk::STATE_NORMAL, m_text_fgcolor);
    m_back.set_use_markup ();
    m_back.set_line_wrap  ();

#if !defined(IS_HILDON)
    m_back.set_selectable ();
#endif

    /* -Separator- 
	 */
    Gtk::HSeparator* separator = manage (new Gtk::HSeparator);

 	Gtk::EventBox* sep_evbox = Gtk::manage (new Gtk::EventBox);
	sep_evbox->modify_bg   (Gtk::STATE_NORMAL, m_sep_color);
	sep_evbox->add (*separator);

    /* -Example- 
	 */
	m_example.set_alignment  (x_alignment_enum (EXAMPLE),
							  y_alignment_enum (EXAMPLE));
	m_example.set_padding    (x_padding_val (EXAMPLE),
							  y_padding_val (EXAMPLE));
	m_example.set_justify    (justification_enum (EXAMPLE));
    m_example.set_use_markup ();
    m_example.set_line_wrap  ();
    m_example.modify_font    (CONFIG->get_example_font ());
	m_example.modify_fg      (Gtk::STATE_NORMAL, m_text_fgcolor);

#if !defined(IS_HILDON)
    m_example.set_selectable ();
#endif

	m_answer_box->pack_start (m_back,     Gtk::PACK_SHRINK, 1);
    m_answer_box->pack_start (*sep_evbox, Gtk::PACK_SHRINK, 1);
    m_answer_box->pack_start (m_example,  Gtk::PACK_SHRINK, 1); 

	/** You need viewport to get background color set right. 
	 *  Without it the whole Notebook page takes its background style
	 *  from the theme, and then Answer/Example labels would have 
	 *  user-defined colors - yuk!
	 */
	viewport = Gtk::manage (new Gtk::Viewport (
								*manage (new Gtk::Adjustment(0,0,1)),
								*manage (new Gtk::Adjustment(0,0,1))));
	viewport->set_shadow_type(Gtk::SHADOW_IN);
	viewport->modify_bg (Gtk::STATE_NORMAL, m_text_bgcolor);
	viewport->add (*m_answer_box);

	/** Without horizontal AUTOMATIC policy, dialog's resizing
	 *  by the user doesn't function properly. Instead, the window
	 *  keeps growing in size every time there is more text to fit then
	 *  size allows.
	 */
	scrollw = Gtk::manage (new Gtk::ScrolledWindow);
	scrollw->set_flags (Gtk::CAN_FOCUS);
	scrollw->set_shadow_type (Gtk::SHADOW_NONE);
	scrollw->set_policy (Gtk::POLICY_NEVER , Gtk::POLICY_AUTOMATIC);

#ifdef GLIBMM_PROPERTIES_ENABLED
	scrollw->property_window_placement ().set_value (Gtk::CORNER_TOP_LEFT);
#else
	scrollw->set_property ("window_placement", Gtk::CORNER_TOP_LEFT);
#endif
	scrollw->add (*viewport);

    m_notebook->pages ().push_back (
		Gtk::Notebook_Helpers::TabElem (*scrollw, _("Back")));

    /********************************************
	 *             Side-panel controls          *
	 ********************************************
	 * For PDAs, we spread side_box horizontally 
	 * and place it to the left of [x Close] button.
     */
#ifdef IS_PDA
    Gtk::HBox* side_box = manage (new Gtk::HBox (false, 3));
#else
    Gtk::VBox* side_box = manage (new Gtk::VBox (false, 3));
#endif

	/**
	 * Radio buttons
	 *   o Ask front
	 *   o Ask back
	 */
	SideFlipControl* sf_control = manage (new SideFlipControl (*this));

	/** Add Test Line selection spin button
	 *    Test line: [ 1 ]
	 */
	m_test_line_entry = manage (new TestLineEntry (m_verify_control));

	/** Fetch out the side history and adjust accordingly
	 */
	m_selected_side = vdeck_.get_side_selection ();
	sf_control->set_active (m_selected_side);
	m_test_line_entry->set_sensitive (true);

#ifdef IS_PDA
	/** Setup hide/show buttons to manipulate verify_control
	 ** and navigation buttons toolbar.
	 */
	Gtk::VBox* hs_vbox = manage (new Gtk::VBox (false, 0));
	m_hs_button_vc  = manage (new Gtk::Button ());
	m_hs_button_ntb = manage (new Gtk::Button ());

	m_hide_control = 
		Granule::create_from_file (GRAPPDATDIR "/pixmaps/hide_control.png"
									   DP_SHRINK_ICONS);
	m_show_control = 
		Granule::create_from_file (GRAPPDATDIR "/pixmaps/show_control.png"
									   DP_SHRINK_ICONS);

#if (GTKMM_MAJOR_VERSION == 2 && GTKMM_MINOR_VERSION >= 6)
	m_hs_button_vc->set_image  (*manage (new Gtk::Image (m_show_control)));
	m_hs_button_ntb->set_image (*manage (new Gtk::Image (m_show_control)));
#else
	m_hs_button_vc->set_label  (".");
	m_hs_button_ntb->set_label (".");
#endif

	hs_vbox->pack_start (*m_hs_button_vc,  Gtk::PACK_SHRINK, 0);
	hs_vbox->pack_start (*m_hs_button_ntb, Gtk::PACK_SHRINK, 0);

	m_hs_button_vc->signal_clicked ().connect (
		mem_fun (*this, &DeckPlayer::on_verify_control_hide_clicked));

	m_hs_button_ntb->signal_clicked ().connect (
		mem_fun (*this, &DeckPlayer::on_navigation_toolbar_hide_clicked));

	side_box->pack_start (*hs_vbox, Gtk::PACK_SHRINK, 0);
#endif // (IS_PDA)

    /************************************************************************** 
	 * Side-panel buttons:                                                    *
	 * ==================                                                     *
	 *    [<> Front ]                                                         *
	 *    [<> Back  ]                                                         *
	 *    [Test line: <1>]                                                    *
	 *    [Shuffle  ]                                                         *
	 *    [Edit Deck]                                                         *
	 *    [Edit Card]                                                         *
	 *                                                                        *
	 * or for PDAs:                                                           *
	 *                                                                        *
	 *    [%][<F>]<1> [Shuffle] [Edit Deck] [Edit Card]  [x Close]            *
	 *                                                                        *
     **************************************************************************/
    side_box->pack_start (*sf_control,          Gtk::PACK_SHRINK, 0);
    side_box->pack_start (*m_test_line_entry,   Gtk::PACK_SHRINK, 0);

    side_box->pack_start (*m_shuffle_button,    Gtk::PACK_SHRINK, 0);
    side_box->pack_start (*m_edit_button,       Gtk::PACK_SHRINK, 0);
    side_box->pack_start (*m_vcard_edit_button, Gtk::PACK_SHRINK, 0);

    m_shuffle_button->signal_clicked ().connect (
		mem_fun (*this, &DeckPlayer::on_shuffle_clicked));

    m_edit_button->signal_clicked ().connect (
		mem_fun (*this, &DeckPlayer::on_edit_clicked));

    m_vcard_edit_button->signal_clicked ().connect (
		bind<bool>(mem_fun (*this, &DeckPlayer::on_vcard_edit_clicked), false));

	/** Count label and progress bar
	 *
	 *       1/138
	 *  [XXXXX________]
	 */
    m_count_label.set_alignment (0.5, 0.5);
    m_count_label.set_padding   (0, 0);
    m_count_label.set_justify   (Gtk::JUSTIFY_FILL);

#ifdef IS_PDA	
	get_vbox ()->set_homogeneous (false);
	get_vbox ()->set_spacing (0);
	get_vbox ()->pack_start (*nb_box, Gtk::PACK_EXPAND_WIDGET, 0);
#else
    side_box->pack_end (*m_play_progress, Gtk::PACK_SHRINK, 2);
    side_box->pack_end (m_count_label,    Gtk::PACK_SHRINK, 2);
    nb_box->pack_start (*side_box, Gtk::PACK_SHRINK, 2);

    get_vbox ()->pack_start (*nb_box);
#endif

    nb_box->show_all ();

    /** Bottom playback controls toolbar panel
	 * --------------------------------------------------
	 *  [spkr] [|<] [<] [>] [>|]  [V]  [A]  [N/M]
	 * --------------------------------------------------
     */
    Glib::RefPtr<Gtk::IconFactory> icon_factory = Gtk::IconFactory::create ();

	//------------------------------------------------------------------------
    m_playit_xpm = 
		Granule::create_from_file (GRAPPDATDIR "/pixmaps/speaker_24.png");
    Gtk::IconSet spkr_set;
    Gtk::IconSource spkr_source;
    spkr_source.set_pixbuf (m_playit_xpm);
    spkr_source.set_size (Gtk::ICON_SIZE_BUTTON);
    spkr_set.add_source (spkr_source);
    icon_factory->add (DeckPlayer::SAY_IT, spkr_set);
    Gtk::Stock::add (Gtk::StockItem (DeckPlayer::SAY_IT, _("Say It!")));

	Glib::RefPtr<Gdk::Pixbuf> thumbup_xpm = 
		Granule::create_from_file (GRAPPDATDIR "/pixmaps/thumbup.png");
    Gtk::IconSet thumbup_set;
    Gtk::IconSource thumbup_source;
    thumbup_source.set_pixbuf (thumbup_xpm);
    thumbup_source.set_size (Gtk::ICON_SIZE_BUTTON);
    thumbup_set.add_source (thumbup_source);
    icon_factory->add (DeckPlayer::THUMB_UP, thumbup_set);
    Gtk::Stock::add (Gtk::StockItem (DeckPlayer::THUMB_UP, _("I know it!")));

	Glib::RefPtr<Gdk::Pixbuf> thumbdown_xpm = 
		Granule::create_from_file (GRAPPDATDIR "/pixmaps/thumbdown.png");
    Gtk::IconSet thumbdown_set;
    Gtk::IconSource thumbdown_source;
    thumbdown_source.set_pixbuf (thumbdown_xpm);
    thumbdown_source.set_size (Gtk::ICON_SIZE_BUTTON);
    thumbdown_set.add_source (thumbdown_source);
    icon_factory->add (DeckPlayer::THUMB_DOWN, thumbdown_set);
    Gtk::Stock::add (Gtk::StockItem (DeckPlayer::THUMB_DOWN, _("No clue")));

    icon_factory->add_default ();

    m_actgroup = Gtk::ActionGroup::create ();

    m_actgroup->add (Gtk::Action::create ("Say_It", DeckPlayer::SAY_IT),
					 mem_fun (*this, &DeckPlayer::on_sayit_clicked));

    m_actgroup->add (Gtk::Action::create ("First", Gtk::Stock::GOTO_FIRST),
					 mem_fun (*this, &DeckPlayer::go_first_cb));

    m_actgroup->add (Gtk::Action::create ("Prev", Gtk::Stock::GO_BACK),
					 mem_fun (*this, &DeckPlayer::go_prev_cb));

    m_actgroup->add (Gtk::Action::create ("Next", Gtk::Stock::GO_FORWARD),
					 mem_fun (*this, &DeckPlayer::go_next_cb));

    m_actgroup->add (Gtk::Action::create ("Last", Gtk::Stock::GOTO_LAST),
					 mem_fun (*this, &DeckPlayer::go_last_cb));

    m_actgroup->add (Gtk::Action::create ("IKnowIt", DeckPlayer::THUMB_UP),
					 mem_fun (*this, &DeckPlayer::know_answer_cb));

    m_actgroup->add (Gtk::Action::create ("NoClue", DeckPlayer::THUMB_DOWN),
					 mem_fun (*this, &DeckPlayer::no_clue_cb));

    m_uimanager = Gtk::UIManager::create ();
    m_uimanager->insert_action_group (m_actgroup);

#if !defined(IS_HILDON)
    add_accel_group (m_uimanager->get_accel_group ());
#endif

    try {
#if !defined(IS_PDA)
		Glib::ustring ui_info =
			"<ui>"
			"  <toolbar name='Controls'>"
			"     <toolitem action='Say_It'/>"
			"     <separator name='1'/>"
			"     <toolitem action='First'/>"
			"     <toolitem action='Prev'/>"
			"     <toolitem action='Next'/>"
			"     <toolitem action='Last'/>"
			"     <separator name='2'/>"
			"     <toolitem action='IKnowIt'/>"
			"     <toolitem action='NoClue'/>"
			"  </toolbar>"
			"</ui>";
#else
		Glib::ustring ui_info =
			"<ui>"
			"  <toolbar name='Controls'>"
			"     <toolitem action='Say_It'/>"
			"     <separator name='1'/>"
			"     <toolitem action='Prev'/>"
			"     <toolitem action='Next'/>"
			"     <separator name='2'/>"
			"     <toolitem action='IKnowIt'/>"
			"     <toolitem action='NoClue'/>"
			"  </toolbar>"
			"</ui>";
#endif
#ifdef GLIBMM_EXCEPTOINS_ENABLED
		m_uimanager->add_ui_from_string (ui_info);
#else
		std::auto_ptr<Glib::Error> err;
		m_uimanager->add_ui_from_string (ui_info, err);
#endif

    }
    catch (const Glib::Error& ex_) {
		DL((GRAPP,"Building menues failed (%s)\n", ex_.what ().c_str ()));
    }

    m_toolbar_go_first = m_uimanager->get_widget ("/Controls/First");
    m_toolbar_go_last  = m_uimanager->get_widget ("/Controls/Last");
    m_toolbar_go_prev  = m_uimanager->get_widget ("/Controls/Prev");
    m_toolbar_go_next  = m_uimanager->get_widget ("/Controls/Next");
    m_toolbar_know_it  = m_uimanager->get_widget ("/Controls/IKnowIt");
    m_toolbar_no_clue  = m_uimanager->get_widget ("/Controls/NoClue");

	/**
	 *  Add input verification control
	 */
	Gtk::HSeparator* hsep = manage (new Gtk::HSeparator);
#ifdef IS_PDA
	get_vbox ()->pack_start (*hsep, Gtk::PACK_SHRINK, 0);
#else
	get_vbox ()->pack_start (*hsep, Gtk::PACK_SHRINK, 4);
#endif
	get_vbox ()->pack_start (*m_verify_control, Gtk::PACK_SHRINK, 0);

	m_verify_control->show_all ();

    /** 
	 * Add play controls' bar and <Close> button
     */
    Gtk::Widget* controls = m_uimanager->get_widget ("/Controls");
    Gtk::Toolbar* ctrl_toolbar = dynamic_cast<Gtk::Toolbar*> (controls);
    ctrl_toolbar->set_orientation (Gtk::ORIENTATION_HORIZONTAL);
    ctrl_toolbar->set_toolbar_style (Gtk::TOOLBAR_ICONS);
    ctrl_toolbar->set_tooltips (true);

    m_navigation_controls_hbox = manage (new Gtk::HBox);
    m_navigation_controls_hbox->pack_start (*ctrl_toolbar, 
											Gtk::PACK_EXPAND_WIDGET, 2);
#ifdef IS_PDA
	m_navigation_controls_hbox->pack_start (m_count_label, Gtk::PACK_SHRINK, 0);
	side_box->pack_end (*m_close_button, Gtk::PACK_SHRINK, 0);
#else
    m_navigation_controls_hbox->pack_start (*m_close_button,Gtk::PACK_SHRINK,0);
#endif

    get_vbox ()->pack_start (*m_navigation_controls_hbox, Gtk::PACK_SHRINK, 0);
	
#ifdef IS_PDA
	get_vbox ()->pack_start (*side_box, Gtk::PACK_SHRINK, 0);
#endif

    m_navigation_controls_hbox->show_all ();

	/************
	 * Callbacks
	 ************
	 */
    m_close_button->signal_clicked ().connect (
		mem_fun (*this, &DeckPlayer::on_close_clicked));

	/**
	 * Setting resize callbacks. The answer box holds 
	 * Back/Separator/Example triplet trapped in the scrollwindow.
	 */
    m_question_box->signal_size_allocate ().connect (
		mem_fun (*this, &DeckPlayer::question_box_size_allocate_cb));

    m_answer_box->signal_size_allocate ().connect (
		mem_fun (*this, &DeckPlayer::answer_box_size_allocate_cb));
	
    signal_size_allocate ().connect (
		mem_fun (*this, &DeckPlayer::size_allocate_cb));

	add_events (Gdk::BUTTON_PRESS_MASK|Gdk::EXPOSURE_MASK);

	/** 'false' argument tells Gtk+ to call on_key_pressed() BEFORE
		keypress event handling. Returning 'true' from on_key_pressed()
		will stop event processing.
	*/
	if (!CONFIG->disable_key_shortcuts ()) {
		signal_key_press_event ().connect(
			mem_fun (*this, &DeckPlayer::on_key_pressed), false);
	}
	
	/* For maemo, reduce overall font size
	 */
	Pango::FontDescription fd ("Bitstream Vera Sans 8");
	m_shuffle_button->modify_font (fd);
	m_edit_button->modify_font (fd);
	m_vcard_edit_button->modify_font (fd);

	/** Grand finale
	 */
#if !defined(IS_HILDON)
	set_icon_from_file (GRAPPDATDIR "/pixmaps/deckplayer_32x32.png");
#endif
    set_sensitivity ();
	sensitize_controls ();
    m_deck.reset_progress ();
    set_win_name ();
    show_deck ();
    switch_side ();

    show_all_children ();	
	m_initialized = true;
}

//==============================================================================
//                              Utilities
//==============================================================================
//
gint
DeckPlayer::
run ()
{
    trace_with_mask("DeckPlayer::run",GUITRACE);

#ifdef IS_HILDON
	m_verify_control->set_focus_on_entry ();
#endif

	show ();
	DL ((APP,"Enter Main::run()\n"));
	Gtk::Main::run ();

	DL ((APP,"Exit Main::run()\n"));
	return (Gtk::RESPONSE_NONE);
}

void
DeckPlayer::
on_close_clicked ()
{
    trace_with_mask("DeckPlayer::on_close_clicked",GUITRACE);

	hide ();

	DL ((GRAPP,"Call Main::quit()\n"));
	Gtk::Main::quit ();
}

void
DeckPlayer::
show_deck ()
{
    trace_with_mask("DeckPlayer::show_deck",GUITRACE);

    if (m_deck.empty ()) {
		DL((GRAPP,"The deck is empty!\n"));
		return;
    }
    reset ();
}

void
DeckPlayer::
reset ()
{
    trace_with_mask("DeckPlayer::reset",GUITRACE);

    m_deck_iter = m_deck.begin ();
	DL ((APP,"m_deck_iter = begin()\n"));
    show_card ();
}

/**
 * Always clear all fields first. Then, try to fill them up.
 * If we were given an invalid markup text, then set_markup()
 * would fail with 
 * "Gtk-WARNING **: Failed to set label from markup due to error parsing markup"
 *
 * I'd like ultimately to catch the error and show it in a warning dialog 
 * to the user, but for now I don't see any way of doing it with GTK+.
 */
void
DeckPlayer::
show_card (bool with_adjust_count_)
{
    trace_with_mask("DeckPlayer::show_card",GUITRACE);

	clear_text ();

    if (m_deck_iter != m_deck.end () || m_deck.size () > 0) 
	{
		while (!markup_is_valid ((*m_deck_iter))) {
			Gtk::MessageDialog em (
				_("The card you are about to see has invalid markup.\n"
				  "To proceed further, first, fix the broken markup.\n"
				  "If unsure about the correct syntax, consult Help."), 
				Gtk::MESSAGE_ERROR);
			em.run ();
			em.hide ();
			on_vcard_edit_clicked (true);
		}
		try {
			DL ((GRAPP,"Seting markup Question field.\n"));
			m_front.set_markup   ((*m_deck_iter)->get_question ());
			DL ((GRAPP,"Seting markup Answer field.\n"));
			m_back.set_markup    ((*m_deck_iter)->get_answer ());
			DL ((GRAPP,"Seting markup Example field.\n"));
			m_example.set_markup (markup_tildas((*m_deck_iter)->get_example()));
			DL ((GRAPP,"All fields are set.\n"));
		}
		catch (const Glib::ConvertError& e) {
			DL ((GRAPP,"Caught conversion error: \"%s\"\n",
				 e.what ().c_str ()));
			switch (e.code ()) {
			case Glib::ConvertError::NO_CONVERSION: 
				DL ((GRAPP,"code: no conversion\n")); 
				break;
			case Glib::ConvertError::ILLEGAL_SEQUENCE:
				DL ((GRAPP,"code: illegal sequence\n")); 
				break;
			case Glib::ConvertError::FAILED:
				DL ((GRAPP,"code: failed\n")); 
				break;
			case Glib::ConvertError::PARTIAL_INPUT:
				DL ((GRAPP,"code: partial input\n")); 
				break;
			case Glib::ConvertError::BAD_URI:
				DL ((GRAPP,"code: bad uri\n"));
				break;
			case Glib::ConvertError::NOT_ABSOLUTE_PATH:
				 DL ((GRAPP,"code: not absolute path\n")); 
				 break;
			}
		}

		if ((*m_deck_iter)->get_reversed ()) {
			m_notebook->set_current_page (BACK);
		}
    }

	if (with_adjust_count_) {
		adjust_count ();
	}

#if !defined(IS_HILDON)
	/** Calling set_focus_on_entry() causes crash with Maemo 2.0 when
	 *  you open CardFile twice in the row. 
	 *  No clue - investigate later.
     */
	m_verify_control->set_focus_on_entry ();
#endif

}

void
DeckPlayer::
adjust_count ()
{
    trace_with_mask("DeckPlayer::adjust_count",GUITRACE);

    if (m_deck.empty ()) {
		DL((GRAPP,"The deck is empty!\n"));
		m_count_label.set_text ("0/0");
		return;
    }
    string pmsg;
    float pcnt_done;
    m_deck.get_progress (pcnt_done, pmsg, m_deck_iter - m_deck.begin ());
    m_count_label.set_text (pmsg);
    m_play_progress->set_fraction (pcnt_done);
}

void
DeckPlayer::
set_sensitivity ()
{
    trace_with_mask("DeckPlayer::set_sensitivity", GUITRACE);

    if (m_is_real_deck) {
		m_toolbar_know_it ->set_sensitive (false);
		m_toolbar_no_clue ->set_sensitive (false);
    }
    else {
#if !defined(IS_PDA)
		m_toolbar_go_first->set_sensitive (false);
		m_toolbar_go_last ->set_sensitive (false);
#endif
		m_toolbar_go_prev ->set_sensitive (false);
		m_toolbar_go_next ->set_sensitive (false);
    }
}

void
DeckPlayer::
sensitize_controls ()
{
	trace_with_mask("DeckPlayer::set_sensitize", GUITRACE);

    if (!m_is_real_deck) {
		return;
	}
	DL ((GRAPP,"Deck is empty\n"));

	if (m_deck.empty ()) 
	{
		m_shuffle_button    ->set_sensitive (false);
		m_vcard_edit_button ->set_sensitive (false);
#if !defined(IS_PDA)
		m_toolbar_go_first  ->set_sensitive (false);
		m_toolbar_go_last   ->set_sensitive (false);
#endif
		m_toolbar_go_prev   ->set_sensitive (false);
		m_toolbar_go_next   ->set_sensitive (false);
    }
	else {
		m_shuffle_button    ->set_sensitive (true);
		m_vcard_edit_button ->set_sensitive (true);
#if !defined(IS_PDA)
		m_toolbar_go_first  ->set_sensitive (true);
		m_toolbar_go_last   ->set_sensitive (true);
#endif
		m_toolbar_go_prev   ->set_sensitive (true);
		m_toolbar_go_next   ->set_sensitive (true);
	}
}

void 
DeckPlayer::
enable_answer_controls ()
{
    if (!m_is_real_deck) {
		m_toolbar_know_it->set_sensitive (true);
		m_toolbar_no_clue->set_sensitive (true);
	}
}

void 
DeckPlayer::
disable_answer_controls ()
{
	m_toolbar_know_it->set_sensitive (false);
	m_toolbar_no_clue->set_sensitive (false);
}

//==============================================================================
//                          Action Buttons callbacks
//==============================================================================
void
DeckPlayer::
go_last_cb ()
{
    if ((m_deck_iter + 1) != m_deck.end ()) {
		m_deck_iter = m_deck.end ();
		m_deck_iter--;
		DL ((APP,"m_deck_iter--\n"));
		show_card ();
    }
    switch_side ();
	m_verify_control->clear ();
}

void
DeckPlayer::
go_first_cb ()
{
    if (m_deck_iter != m_deck.begin ()) {
		m_deck_iter = m_deck.begin ();
		DL ((APP,"m_deck_iter = begin()\n"));
		show_card ();
    }
    switch_side ();
	m_verify_control->clear ();
}

void
DeckPlayer::
go_prev_cb ()
{
    if (m_deck_iter != m_deck.begin ()) {
		m_deck_iter--;
		DL ((APP,"m_deck_iter--\n"));
		show_card ();
    }
    switch_side ();
	m_verify_control->clear ();
}

void
DeckPlayer::
go_next_cb ()
{
    trace_with_mask("DeckPlayer::go_next_cb",GUITRACE);

    if (m_deck_iter != m_deck.end () &&	         // empty deck
		(m_deck_iter + 1) != m_deck.end ())      // at the last element
	{
	    DL((GRAPP,"Go next card=>\n"));
	    m_deck_iter++;
		DL ((APP,"m_deck_iter++\n"));
	    show_card ();
	}
    /** We flip the side only if user has gone (hopefully)
     *  through an entire Deck (for *real* Decks only).
     */
    if (m_is_real_deck) {
		if (m_deck_iter + 1 == m_deck.end () && !m_is_lesson_learned) {	
			DL((GRAPP,"Reached the last card!\n"));
			CONFIG->set_flipside_history (m_deck.get_name (), 
										  m_selected_side==FRONT ? BACK:FRONT);
			m_is_lesson_learned = true;
		}
    }
    switch_side ();
	m_verify_control->clear ();
}

/** This callback is activated only when learning flashcards in the
    Card Box. We are always at the beginning of the list, moving
    elements to the next box or in case of the last box, moving 
    them to the back of the same list. We proceed in this manner until 
    all cards in the box are exhausted.
*/
void
DeckPlayer::
know_answer_cb ()
{
    trace_with_mask("DeckPlayer::know_answer_cb",GUITRACE);

    if (m_deck_iter == m_deck.end ()) {	// empty deck
		DL((GRAPP,"The deck is empty!\n"));
		return;
    }
    int old_idx = m_deck.get_index ();
    DL((GRAPP,"Moving card \"%s\" : deck %d => %d\n",
		(*m_deck_iter)->get_question ().c_str (), old_idx+1, old_idx+2));

    CARDBOX->move_card_to_box (m_deck_iter, old_idx, old_idx + 1, 
					   (dynamic_cast<CardRef*> (*m_deck_iter))->is_expired ());
    switch_side ();
	m_verify_control->clear ();
}

/** Move the card to the first box or if already in the first box,
    move it to the back of the list.
*/
void
DeckPlayer::
no_clue_cb ()
{
    trace_with_mask("DeckPlayer::no_clue_cb",GUITRACE);

    if (m_deck_iter == m_deck.end ()) {	// empty deck
		DL((GRAPP,"The deck is empty!\n"));
		return;
    }
    int old_idx = m_deck.get_index ();
    DL((GRAPP,"Moving card \"%s\" : deck %d => 1\n",
		(*m_deck_iter)->get_question ().c_str (), old_idx+1));

    CARDBOX->move_card_to_box (m_deck_iter, old_idx, 0,
					   (dynamic_cast<CardRef*> (*m_deck_iter))->is_expired ());
    switch_side ();
	m_verify_control->clear ();
}


void
DeckPlayer::
on_radio_selected (SideSelection selected_)
{
    trace_with_mask("DeckPlayer::on_radio_selected",GUITRACE);

    m_selected_side = selected_;
	DL ((GRAPP,"selected = %s\n",(m_selected_side == BACK ? "BACK" : "FRONT")));

	m_deck.set_side_selection (m_selected_side);
    switch_side ();
}

void
DeckPlayer::
on_sayit_clicked ()
{
    trace_with_mask("DeckPlayer::on_sayit_clicked",GUITRACE);

    if (m_deck_iter != m_deck.end ()) {
		m_deck.play_card (SideSelection (m_notebook->get_current_page ()), 
						  m_deck_iter);
    }
}

void
DeckPlayer::
on_shuffle_clicked ()
{
    trace_with_mask("DeckPlayer::on_shuffle_clicked",GUITRACE);

    if (m_is_real_deck) 
	{
		random_shuffle (m_deck.begin (), m_deck.end ());
		Gtk::MessageDialog qm (
				   _("Would you like to make cards\norder changes permanent?"),
							   false,
                               Gtk::MESSAGE_QUESTION, 
							   Gtk::BUTTONS_NONE,
                               true);
        qm.add_button (Gtk::Stock::NO,  Gtk::RESPONSE_NO);
        qm.add_button (Gtk::Stock::YES, Gtk::RESPONSE_YES);

		if (qm.run () == Gtk::RESPONSE_YES) 
		{
			Deck& real_deck = dynamic_cast<Deck&>(m_deck);
			real_deck.mark_as_modified ();
		}
    }
	/** For CardDeck, shuffle all subsets.
	 */
	else {
		CardDeck& card_deck = dynamic_cast<CardDeck&>(m_deck);
		card_deck.shuffle_subsets ();
	}

	/** Refresh display, but don't adjust progress bar
	 */
    m_deck_iter = m_deck.begin ();
	DL ((APP,"m_deck_iter = begin()\n"));
    show_card (false);
}

void
DeckPlayer::
on_edit_clicked ()
{
    trace_with_mask("DeckPlayer::on_edit_clicked",GUITRACE);
    DL((GRAPP,"Bring up deck viewer dialog\n"));

	hide (); // hide DeckPlayer while DeckView is visible.

	int idx = m_deck_iter == m_deck.end() ? -1 : (m_deck_iter - m_deck.begin());
	bool dv_modified = false;

	{
		DeckView dv (m_deck, idx);
		dv.run ();
		dv_modified = dv.is_modified ();
		if (dv.appearance_changed ()) {
			repaint ();
		}
	}

    DL((GRAPP,"Deck size after edit = %d cards (modified = %s)\n", 
		m_deck.size (), (dv_modified ? "TRUE" : "FALSE")));

	sensitize_controls ();

    /** 
     * Update viewer - iterators might have become invalid.
     */
    if (m_deck.size () == 0) 	// Last card removed - clean up
	{
		m_deck_iter = m_deck.end ();
		CARDBOX->mark_as_modified ();
		show_card ();
		
//#if !defined(IS_HILDON)
		/** Under Hildon, this messes up return chain.
		 */
		on_close_clicked ();
//#endif
    }
    else 
	{
		if (dv_modified) {
			DL ((APP,"Deck's been modified - go to the beginning\n"));
			CARDBOX->mark_as_modified ();
			show_deck ();
		}
		else {
			DL ((APP,"Deck's intact - stay where we were\n"));
			show_card (false);
		}
    }

	show ();					// Restore DeckPlayer
}

void
DeckPlayer::
on_vcard_edit_clicked (bool disable_cancel_)
{
    trace_with_mask("DeckPlayer::on_vcard_edit_clicked",GUITRACE);

	if (m_deck_iter != m_deck.end () || m_deck.size () > 0) {
		hide ();
		VCard* vc = *m_deck_iter;
		CardView cview (vc);
		int ret = cview.run (disable_cancel_);

		if (ret == Gtk::RESPONSE_OK) {
			show_card (false);
		}
		show ();
	}
	else {
        Gtk::MessageDialog em (_("The deck is empty!"), Gtk::MESSAGE_ERROR);
        em.run ();
        return;
	}
}

void
DeckPlayer::
on_verify_clicked ()
{
    trace_with_mask("DeckPlayer::on_verify_clicked",GUITRACE);

	if (m_verify_control->get_state () == VC_ARMED) 
	{
		m_verify_control->compare (get_answer ());
		auto_pronounce_card (BACK);
	}
	else if (m_verify_control->get_state () == VC_VALIDATED) 
	{
		if (m_is_real_deck) {
			go_next_cb ();
		}
		else { /* CardBox Player*/ 
			if (m_verify_control->matched ()) {
				know_answer_cb ();
			}
			else {
				no_clue_cb ();
			}
		}
		m_verify_control->clear ();
	}
}

/** 
  Key values can be found in /usr/include/gtk-2.0/gdk/gdkkeysyms.h

  Key bindings:

  Deck mode:
    /------------------------+------------------------.
    |     Action             |        Key(s)          |
    +------------------------+------------------------+
    | Play the sound         | UpArrow                |
    | Previous card          | LeftArrow              |
    | Next card              | RigthArrow, Enter      |
    | Flip the card          | DownArrow, Spacebar    |
    | Iconize windows        | 'u'                    |
    `------------------------+------------------------/

  CardBox mode:
    /------------------------+------------------------.
    |     Action             |        Key(s)          |
    +------------------------+------------------------+
    | Play the sound         | UpArrow                |
    | 'NO' answer            | LeftArrow, 'n'         |
    | 'YES' answer           | RigthArrow, Enter, 'y' |
    | Flip the card          | DownArrow, Spacebar    |
    | Iconize windows        | 'u'                    |
    `------------------------+------------------------/

  Engaging Verification Control (any mode):

    /-------------------------+------------------------.
	| Activate [Check] button | Type in the answer     |
    +-------------------------+------------------------+
	| Check the answer,       | [Check], Enter         |
	| flip the card,          |                        |
	| activate [Next] button  |                        |
    `-------------------------+------------------------/
*/
/*
  GDK_space     0x0020

  GDK_Left      0xFF51
  GDK_Up        0xFF52
  GDK_Right     0xFF53
  GDK_Down      0xFF54

  GDK_KP_Left   0xFF96
  GDK_KP_Up     0xFF97
  GDK_KP_Right  0xFF98
  GDK_KP_Down   0xFF99

  GDK_Tab       0xFF09
  GDK_Return    0xFF0D
  GDK_KP_Enter  0xFF8D
  GDK_F1        0xFFBE
  GDK_F6        0xFFC3
  GDK_F7        0xFFC4
  GDK_F8        0xFFC5
*/

bool 
DeckPlayer::
on_key_pressed (GdkEventKey* evt_)
{
    trace_with_mask("DeckPlayer::on_key_pressed",KEYIN);

#if !defined(IS_HILDON)
	if (evt_->keyval == GDK_F1) {
		MAINWIN->iconify ();
	}
#endif

	if (evt_->keyval == GDK_space || evt_->keyval > 0xFF51) {
		DL ((KEYIN,"Key pressed: 0x%04X, real_deck: %s, state: %s\n", 
			 evt_->keyval, (m_is_real_deck ? "Yes" : "No"),
			 (m_verify_control->get_state() == VC_START ? "START" :
			  m_verify_control->get_state() == VC_ARMED ? "ARMED" : "VALIDATED")
				));
	}
	
	if (evt_->keyval == GDK_space && m_tab_activated) {
		m_tab_activated = false;
		return false;
	}
	
	if (evt_->keyval == GDK_Tab) {
		m_tab_activated = true;
		return false;
	}

	if (check_F6_key (evt_->keyval)) {
		return true;
	}

	/*********************************************
	 * Real Deck                                 *
	 *********************************************/

	if (m_is_real_deck) {                         
		switch (m_verify_control->get_state ()) {

		case VC_START:
		case VC_VALIDATED:
			if (evt_->keyval == GDK_Right    || 
				evt_->keyval == GDK_KP_Right ||
				evt_->keyval == GDK_Return   || 
				evt_->keyval == GDK_KP_Enter ||
				check_F7_key (evt_->keyval)) 
			{
				go_next_cb ();
			}
			else if (evt_->keyval == GDK_Left    || 
					 evt_->keyval == GDK_KP_Left ||
					 check_F8_key (evt_->keyval))
			{
				go_prev_cb ();
			}
			break;

		case VC_ARMED:
			if (evt_->keyval == GDK_Return || 
				evt_->keyval == GDK_KP_Enter) 
			{
				m_verify_control->compare (get_answer ());
				auto_pronounce_card (BACK);
			}
			else if (evt_->keyval == GDK_space) {
				return false;
			}
			break;
		}
	}
	/*********************************************
	 * CardBox Player                            *
	 *********************************************/

	else {
		switch (m_verify_control->get_state ()) 
		{
		case VC_START:
			if (evt_->keyval == GDK_Right    || 
				evt_->keyval == GDK_KP_Right ||
				check_F7_key (evt_->keyval))
			{
				know_answer_cb ();
			}
			else if (evt_->keyval == GDK_Left    || 
					 evt_->keyval == GDK_KP_Left ||
					 check_F8_key (evt_->keyval))
			{
				no_clue_cb ();
			}
			break;

		case VC_ARMED:
			if (evt_->keyval == GDK_Return || 
				evt_->keyval == GDK_KP_Enter) 
			{
				m_verify_control->compare (get_answer ());
				auto_pronounce_card (BACK);
			}
			else if (evt_->keyval == GDK_space) {
				return false;
			}
			break;

		case VC_VALIDATED:
			if (evt_->keyval == GDK_Return || 
				evt_->keyval == GDK_KP_Enter) 
			{
				if (m_verify_control->matched ()) {
					know_answer_cb ();
				}
				else {
					no_clue_cb ();
				}
				m_verify_control->clear ();
			}
			break;
		}

	} // end
	
	switch (evt_->keyval)
		{
		case GDK_Up:          
		case GDK_KP_Up:
			on_sayit_clicked ();
			break;
			
		case GDK_space: 
		{ 
			m_tab_activated = false; 
		}
		case GDK_Down:        
		case GDK_KP_Down:
			flip_card ();
			break;

		default:
			return false;
		}
	
	return true;
}	

void
DeckPlayer::
set_win_name ()
{
    string title = m_deck.get_name ();
    string::size_type idx = title.rfind (G_DIR_SEPARATOR);
    if (idx != string::npos) {
        title.replace (0, idx+1, "");
    }
    set_title ("Deck Player : " + title);
}

/**
   Remember the size of the window if user resized it.
*/
void
DeckPlayer::
size_allocate_cb (Gtk::Allocation& allocation_)
{
    trace_with_mask("DeckPlayer::size_allocate_cb",GEOM);

	DL ((GEOM,"New size change requested: w=%d, h=%d\n", 
		 allocation_.get_width (), allocation_.get_height ()));

	CONFIG->set_deck_player_geometry (allocation_.get_width (),
									  allocation_.get_height ());
}

/** 
	Adjust the size of the 'question' label when size of 
	the enclosing VBox is changed by the user via resize.
*/
void
DeckPlayer::
question_box_size_allocate_cb (Gtk::Allocation& allocation_)
{
    trace_with_mask("DeckPlayer::question_box_size_allocate_cb",GEOM);

    DL((GEOM,"new question_box size (w=%d; h=%d)\n", 
		allocation_.get_width (), allocation_.get_height ()));

    if (m_question_box_width != allocation_.get_width ()) {
		m_front.set_size_request (allocation_.get_width () - 20,  -1);
		m_question_box_width = allocation_.get_width ();
	}
}

/** 
	Adjust the size of the 'example' label when size of 
	the enclosing VBox is changed by the user via resize.
*/
void
DeckPlayer::
answer_box_size_allocate_cb (Gtk::Allocation& allocation_)
{
    trace_with_mask("DeckPlayer::answer_box_size_allocate_cb",GEOM);

    DL((GEOM,"new answer_box size (w=%d; h=%d)\n", 
		allocation_.get_width (), allocation_.get_height ()));

    if (m_answer_box_width != allocation_.get_width ()) {
		m_example.set_size_request (allocation_.get_width () - 20,  -1);
		m_answer_box_width = allocation_.get_width ();
	}
}

void
DeckPlayer::
repaint ()
{
   trace_with_mask("DeckPlayer::repaint",GUITRACE);

	Deck* deckp = dynamic_cast<Deck*>(&m_deck);
	if (deckp != NULL && deckp->with_custom_appearance ()) {
		Pango::FontDescription fdesk (deckp->question_font ());
		m_front.modify_font (fdesk);
	}
	else {
		m_front.modify_font (CONFIG->get_question_font ());
	}
	m_back   .modify_font (CONFIG->get_answer_font ());
	m_example.modify_font (CONFIG->get_example_font ());

	DL ((GRAPP,"Answer padding (x, y) = %d, %d\n",
		 x_padding_val (BACK), y_padding_val (BACK)));

   m_verify_control->repaint ();
   update_geometry ();

   /* Update text alignment
	*/
   	m_front.set_alignment (x_alignment_enum (FRONT),
						   y_alignment_enum (FRONT));
	m_front.set_justify   (justification_enum (FRONT));

	m_back.set_alignment (x_alignment_enum (BACK),
						  y_alignment_enum (BACK));
	m_back.set_justify   (justification_enum (BACK));
}

/** Resize only if geometry stored in Configurator differs
 *  from the actual size;
 */
void
DeckPlayer::
update_geometry ()
{
#if !defined(IS_HILDON)
	int actual_height = 0;
	int actual_width  = 0;

	Gdk::Rectangle geom = CONFIG->get_deck_player_geometry ();
	get_size (actual_width, actual_height);

	if (actual_width != geom.get_width () || 
		actual_height != geom.get_height ())
	{
		resize (geom.get_width (), geom.get_height ());
	}
#endif
}

/**
 * Replace all newline characters first.
 * Then remove all leading and following extra spaces.
 * Then shrink all  whitespaces that are left to a single space.
 */
void
DeckPlayer::
shrink_to_single_line (Glib::ustring& txt_)
{
   trace_with_mask("DeckPlayer::shrink_to_single_line",GUITRACE);

   Glib::ustring::size_type idx;

   idx = txt_.find_first_of ("\n");
   if (idx != Glib::ustring::npos) {
	   txt_.replace (idx, txt_.size (), "");
   }

   while ((idx = txt_.find ("  ")) != Glib::ustring::npos) {
	   txt_.replace (idx, 2, " ");
   }

   idx = txt_.find_first_not_of (" \t");
   if (idx != Glib::ustring::npos) {
	   txt_.replace (0, idx, "");
   }
   
   idx = txt_.find_last_not_of (" \t");
   if (idx != Glib::ustring::npos) {
	   txt_.replace (idx + 1, txt_.size (), "");
   }
}


Glib::ustring
DeckPlayer::
markup_tildas (const Glib::ustring& src_)
{
//    trace_with_mask("DeckPlayer::markup_tildas", GUITRACE);

	static Glib::ustring tilda ("<span foreground=\"red\">~</span>");
	Glib::ustring ret;

	if (src_.size () == 0) {
		return ret;
	}

	u_int i;
	u_int b;

	for (i = 0, b = 0; i < src_.size (); i++) {
		if (src_[i] == '~') {
			if (i == 0) {
				ret = tilda;
				b++;
				continue;
			}
			ret += src_.substr (b, i-b) + tilda;
			b = i+1;
		}
	}
	ret += src_.substr (b);
	return ret;
}

bool
DeckPlayer::
markup_is_valid (VCard* card_)
{
	return (Granule::check_markup (card_->get_question ()) &&
			Granule::check_markup (card_->get_answer   ()) &&
			Granule::check_markup (card_->get_example  ()));
}

/** If card we view is reversed, disregard Deck's orientation. 
 *  In this case, each card carries its own *sticky* orientation history
 *  which makes the whole experience of coding and testing maddening!!!
 */
void
DeckPlayer::
switch_side ()
{
    trace_with_mask("DeckPlayer::switch_side", GUITRACE);

	if (m_deck_iter == m_deck.end ()) {
		m_notebook->set_current_page (m_selected_side);
		return;
	}

	if (!m_is_real_deck && (*m_deck_iter)->get_reversed ())
	{
		m_notebook->set_current_page (m_selected_side == FRONT ? BACK : FRONT);
	}
	else {
		m_notebook->set_current_page (m_selected_side);
	}

	auto_pronounce_card (!(*m_deck_iter)->get_reversed () ? FRONT : BACK);
}

/** We want to have card already painted in the window.
 */
void
DeckPlayer::
auto_pronounce_card (SideSelection side_)
{
    trace_with_mask("DeckPlayer::auto_pronounce_card", GUITRACE);

	if (!m_initialized || m_deck_iter == m_deck.end ()) {
		return;
	}

	if (m_selected_side == side_ && CONFIG->auto_pronounce ())
    {
		m_deck.play_card (m_selected_side, m_deck_iter);
	}
}

Glib::ustring
DeckPlayer::
get_answer ()
{
    trace_with_mask("DeckPlayer::get_answer", GUITRACE);

	if ((*m_deck_iter)->get_reversed () || m_selected_side == BACK) {
		return (m_front.get_text ());
	}
	else {
		return (m_back.get_text ());
	}
}

/*******************************************************************************
	Convert string alignment to Gtk::AlignmentEnum
*/

Gtk::AlignmentEnum
DeckPlayer::
x_alignment_enum (SideSelection side_) 
{
	Gtk::AlignmentEnum result = Gtk::ALIGN_CENTER;
	std::string val;

	Deck* deckp = dynamic_cast<Deck*>(&m_deck);
	if (deckp != NULL && deckp->with_custom_appearance ()) {
		val = deckp->x_alignment (side_);
	}
	else {
		val = CONFIG->x_alignment (side_);
	}

	if (val == "left") {
		result = Gtk::ALIGN_LEFT;
	}
	else if (val == "right") {
		result = Gtk::ALIGN_RIGHT;
	}

	return (result);
}

Gtk::AlignmentEnum
DeckPlayer::
y_alignment_enum (SideSelection side_) 
{
	Gtk::AlignmentEnum result = Gtk::ALIGN_CENTER;
	std::string val;

	Deck* deckp = dynamic_cast<Deck*>(&m_deck);
	if (deckp != NULL && deckp->with_custom_appearance ()) {
		val = deckp->y_alignment (side_);
	}
	else {
		val = CONFIG->y_alignment (side_);
	}

	if (val == "top") {
		result = Gtk::ALIGN_TOP;
	}
	else if (val == "bottom") {
		result = Gtk::ALIGN_BOTTOM;
	}
	return (result);
}


/*******************************************************************************
	Convert string justification to Gtk::Justification
*/
Gtk::Justification
DeckPlayer::
justification_enum (SideSelection side_)
{
	Gtk::Justification result = Gtk::JUSTIFY_CENTER;
	std::string val;

	Deck* deckp = dynamic_cast<Deck*>(&m_deck);
	if (deckp != NULL && deckp->with_custom_appearance ()) {
		val = deckp->justification (side_);
	}
	else {
		val = CONFIG->justification (side_);
	}

	if (val == "left") {
		result = Gtk::JUSTIFY_LEFT;
	}
	else if (val == "right") {
		result = Gtk::JUSTIFY_RIGHT;
	}
	else if (val == "fill") {
		result = Gtk::JUSTIFY_FILL;
	}
	return (result);
}

int
DeckPlayer::
x_padding_val (SideSelection side_)
{
	int result;
	Deck* deckp = dynamic_cast<Deck*>(&m_deck);
	if (deckp != NULL && deckp->with_custom_appearance ()) {
		result = ustring_to_int (deckp->x_padding (side_));
	}
	else {
		result = ustring_to_int (CONFIG->x_padding (side_));
	}

	return (result);
}

int
DeckPlayer::
y_padding_val (SideSelection side_)
{
	int result;
	Deck* deckp = dynamic_cast<Deck*>(&m_deck);
	if (deckp != NULL && deckp->with_custom_appearance ()) {
		result = ustring_to_int (deckp->y_padding (side_));
	}
	else {
		result = ustring_to_int (CONFIG->y_padding (side_));
	}

	return (result);
}

Pango::FontDescription
DeckPlayer::
question_font ()
{
	Deck* deckp = dynamic_cast<Deck*>(&m_deck);

	if (deckp != NULL && deckp->with_custom_appearance ()) {
		return Pango::FontDescription (deckp->question_font ());
	}

	return (CONFIG->get_question_font ());
}

void
DeckPlayer::
on_verify_control_hide_clicked ()
{
	if (m_verify_control->is_visible ()) {
		m_verify_control->hide ();
#if (GTKMM_MAJOR_VERSION == 2 && GTKMM_MINOR_VERSION >= 6)
		m_hs_button_vc->set_image (*manage (new Gtk::Image (m_hide_control)));
#endif
	}
	else {
		m_verify_control->show ();
#if (GTKMM_MAJOR_VERSION == 2 && GTKMM_MINOR_VERSION >= 6)
		m_hs_button_vc->set_image (*manage (new Gtk::Image (m_show_control)));
#endif
	}
}

void
DeckPlayer::
on_navigation_toolbar_hide_clicked ()
{
	if (m_navigation_controls_hbox->is_visible ()) {
		m_navigation_controls_hbox->hide ();
#if (GTKMM_MAJOR_VERSION == 2 && GTKMM_MINOR_VERSION >= 6)
		m_hs_button_ntb->set_image (*manage (new Gtk::Image (m_hide_control)));
#endif
	}
	else {
		m_navigation_controls_hbox->show ();
#if (GTKMM_MAJOR_VERSION == 2 && GTKMM_MINOR_VERSION >= 6)
		m_hs_button_ntb->set_image (*manage (new Gtk::Image (m_show_control)));
#endif
	}
}
