// -*- c++ -*-
//------------------------------------------------------------------------------
//                              CardView.cpp
//------------------------------------------------------------------------------
// $Id: CardView.cpp,v 1.50 2007/01/14 02:00:13 vlg Exp $
//
//  Copyright (c) 2004-2006 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//
// Date: Feb 15, 2004
//------------------------------------------------------------------------------

#include <gtkmmconfig.h>
#include <gtkmm/image.h>
#include <gtkmm/label.h>
#include <gtkmm/box.h>
#include <gtkmm/alignment.h>
#include <gtkmm/button.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/table.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/paned.h>

#include "Granule.h"
#include "MainWindow.h"
#include "GrappConf.h"
#include "CardView.h"
#include "DeckView.h"
#include "Card.h"
#include "CardRef.h"
#include "EditControls.h"
#include "Intern.h"				/* always last */

#define SET_CV_LABEL(l) \
	l->set_alignment  (0.5,0.5);  \
	l->set_padding    (0,0);      \
	l->set_justify    (Gtk::JUSTIFY_LEFT); \
	l->set_line_wrap  (false);    \
	l->set_use_markup (false);    \
	l->set_selectable (false);   

#define SET_CV_TEXT(t) \
	t->set_flags (Gtk::CAN_FOCUS);   \
	t->set_border_width (3);         \
	t->set_editable (true);          \
	t->set_cursor_visible (true);    \
	t->set_pixels_above_lines (3);   \
	t->set_pixels_below_lines (0);   \
	t->set_pixels_inside_wrap (0);   \
	t->set_left_margin  (4);         \
	t->set_right_margin (4);         \
	t->set_indent (0);               \
	t->set_wrap_mode (Gtk::WRAP_WORD);         \
	t->set_justification (Gtk::JUSTIFY_LEFT);  \
	t->set_accepts_tab (false);

static void 
set_cv_scrollwin (Gtk::ScrolledWindow& win_, Gtk::Widget& widget_)
{
	win_.set_flags       (Gtk::CAN_FOCUS);
	win_.set_shadow_type (Gtk::SHADOW_NONE);
	win_.set_policy      (Gtk::POLICY_ALWAYS, Gtk::POLICY_ALWAYS); 
#ifdef GLIBMM_PROPERTIES_ENABLED 
	win_.property_window_placement ().set_value (Gtk::CORNER_TOP_LEFT);
#else
	win_.set_property ("window_placement", Gtk::CORNER_TOP_LEFT);
#endif 

/** Defeat Hildons' default theme and provide
 *  our own visual border around Text Entry
 */
#ifdef IS_HILDON

	Gtk::Frame* f = manage (new Gtk::Frame ());
	f->set_border_width (1);
    f->set_shadow_type  (Gtk::SHADOW_ETCHED_OUT);
	f->set_label_align  (0, 0);
	f->add (widget_);
	win_.add (*f);
#else
	win_.add (widget_);
#endif
}

/**
 * Enclose Gtk::TextView in a visible box. Some default themes (hildon)
 * blend the surrounding, so it is hard to see where the editable
 * region lies.
 */
static Gtk::Widget*
set_text_border (Gtk::Widget* widget_)
{
#ifdef IS_HILDON
	Gtk::Frame* frame = manage (new Gtk::Frame ());
	frame->set_border_width (1);
    frame->set_shadow_type  (Gtk::SHADOW_ETCHED_OUT);
	frame->set_label_align  (0, 0);
	frame->add (*widget_);
	return (frame);
#else
	return (widget_);
#endif
}


/*------------------------------------------------------------------------------
** CardView class implementation
**------------------------------------------------------------------------------
*/
CardView::
CardView (VCard* card_) 
	: 
	m_card          (card_),
	m_syntax_error  (false),
	m_keep_open     (false),

	m_cancel_button (NULL),
	m_count_button  (NULL),
	m_add_n_close_button (NULL),
	m_add_button    (NULL),
	m_help_button   (NULL),

	m_front_text    (NULL),
	m_back_text     (NULL),
	m_example_text  (NULL),
	m_asf_text      (NULL),
	m_asb_text      (NULL),

	m_edit_controls (NULL),
	m_help_dialog   (NULL)
{
	trace_with_mask("CardView::CardView(VCard&)",GUITRACE);

	Gtk::Image* image1;
	Gtk::Image* image2;
	Gtk::Image* image3;
	Gtk::Image* image4;
	Gtk::Image* image5;

	Gtk::HBox* hbox1;
	Gtk::HBox* hbox2;
	Gtk::HBox* hbox3;
	Gtk::HBox* hbox4;
	Gtk::HBox* hbox5;

	Gtk::Label* label1;
	Gtk::Label* label2;
	Gtk::Label* label3;
	Gtk::Label* label4;
	Gtk::Label* label5;

	Gtk::Alignment* alignment1;
	Gtk::Alignment* alignment2;
	Gtk::Alignment* alignment3;
	Gtk::Alignment* alignment4;

	Gtk::Alignment* bt_alignment;
	Gtk::Alignment* ft_alignment;
	Gtk::Alignment* et_alignment;

	Gtk::Alignment* asf_alignment;
	Gtk::Alignment* asb_alignment;

	Gtk::ScrolledWindow* front_scrollwin;
	Gtk::ScrolledWindow* backtext_scrollwin;
	Gtk::ScrolledWindow* example_scrollwin;
	Gtk::ScrolledWindow* asf_scrollwin;
	Gtk::ScrolledWindow* asb_scrollwin;

	Gtk::Label* back_label;
	Gtk::Label* example_label;
	Gtk::Label* front_label;
	Gtk::Label* asf_label;
	Gtk::Label* asb_label;

	Gtk::VBox* vbox_frontlabel;
	Gtk::VBox* vbox_backlabel;
	Gtk::VBox* vbox_examplelabel;
	Gtk::HBox* hbox_asf;
	Gtk::HBox* hbox_asb;

	Gtk::Table* elems_table_1;
	Gtk::Table* elems_table_2;
	Gtk::Table* elems_table_3;

	Gtk::VPaned* paned_1;		/* Holds elems_table_1 and paned_2       */
	Gtk::VPaned* paned_2;		/* Holds elems_table_2 and elems_table_3 */

	Gtk::VBox*  vbox_all;

	/** The dialog setup
	 */
	set_title ("Card View");

#ifdef IS_HILDON
//	set_size_request (672, 396);
#else
	set_modal (true);  // very important - we're not protected from multi-clicks
//	property_window_position ().set_value (Gtk::WIN_POS_CENTER);

#ifdef IS_PDA
	set_resizable (false);
	Gdk::Geometry dp_geometry =	{ 240, 320,	240, 320, 240, 320,-1, -1, -1, -1};
	set_geometry_hints (*this, dp_geometry, 
				Gdk::HINT_MIN_SIZE | Gdk::HINT_MAX_SIZE | Gdk::HINT_BASE_SIZE);
#else // DESKTOP
	set_resizable (true);
	set_icon_from_file (GRAPPDATDIR "/pixmaps/cardview_32x32.png");
	Gdk::Rectangle geom = CONFIG->get_card_view_geometry ();
	set_size_request (geom.get_width (), geom.get_height ());
	set_transient_for (*MAINWIN);
#endif // (!IS_PDA)

#endif	// (!IS_HILDON)

	/** <Clear> button - OBSOLETE -
	 */
	image1 = Gtk::manage (new Gtk::Image (Gtk::StockID ("gtk-help"), 
										  Gtk::IconSize (4)));
	label1        = Gtk::manage (new Gtk::Label (_("Help")));
	hbox1         = Gtk::manage (new Gtk::HBox (false, 2));
	alignment1    = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	m_help_button = Gtk::manage (new Gtk::Button ());

	/** <Cancel> Button
	 */
	image2     = Gtk::manage (new Gtk::Image (Gtk::StockID ("gtk-cancel"), 
											  Gtk::IconSize (4)));
	label2          = Gtk::manage (new Gtk::Label (_("Cancel")));
	hbox2           = Gtk::manage (new Gtk::HBox (false, 2));
	alignment2      = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	m_cancel_button = Gtk::manage (new Gtk::Button ());

	/** <Count> *Button*
	 */
	m_count_button  = Gtk::manage (new Gtk::Button ());

	/** <Add> Button
	 */
	image3       = Gtk::manage (new Gtk::Image (Gtk::StockID ("gtk-add"), 
												Gtk::IconSize (4)));
	label3       = Gtk::manage (new Gtk::Label (_("Add")));
	hbox3        = Gtk::manage (new Gtk::HBox (false, 2));
	alignment3   = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	m_add_button = Gtk::manage (new Gtk::Button ());

	/** <Add'n'Close> Button
	 */
	image4      = Gtk::manage (new Gtk::Image (Gtk::StockID ("gtk-apply"), 
												   Gtk::IconSize (4)));
	label4      = Gtk::manage (new Gtk::Label (_("Ok")));
	hbox4       = Gtk::manage (new Gtk::HBox (false, 2));
	alignment4  = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	m_add_n_close_button = Gtk::manage (new Gtk::Button ());

	/** Text
	 */
	m_front_text       = Gtk::manage (new Gtk::TextView);
	front_scrollwin    = Gtk::manage (new Gtk::ScrolledWindow);
	front_label        = Gtk::manage (new Gtk::Label (_("Front")));
	ft_alignment       = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	vbox_frontlabel    = Gtk::manage (new Gtk::VBox (false, 0));

	m_back_text        = Gtk::manage (new Gtk::TextView);
	backtext_scrollwin = Gtk::manage (new Gtk::ScrolledWindow);
	back_label         = Gtk::manage (new Gtk::Label (_("Back")));
	bt_alignment       = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	vbox_backlabel     = Gtk::manage (new Gtk::VBox (false, 0));

	m_example_text     = Gtk::manage (new Gtk::TextView);
	example_scrollwin  = Gtk::manage (new Gtk::ScrolledWindow);
	example_label      = Gtk::manage (new Gtk::Label (_("Example")));
	et_alignment       = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	vbox_examplelabel  = Gtk::manage (new Gtk::VBox (false, 0));

	m_asf_text     = Gtk::manage (new Gtk::TextView);
	asf_scrollwin  = Gtk::manage (new Gtk::ScrolledWindow);
	asf_label      = Gtk::manage (new Gtk::Label (_("ASF")));
	asf_alignment  = Gtk::manage (new Gtk::Alignment (0, 0.5, 0, 0));
	hbox_asf       = Gtk::manage (new Gtk::HBox (false, 0));

	m_asb_text     = Gtk::manage (new Gtk::TextView);
	asb_scrollwin  = Gtk::manage (new Gtk::ScrolledWindow);
	asb_label      = Gtk::manage (new Gtk::Label (_("ASB")));
	asb_alignment  = Gtk::manage (new Gtk::Alignment (0, 0.5, 0, 0));
	hbox_asb       = Gtk::manage (new Gtk::HBox (false, 0));

	elems_table_1  = Gtk::manage (new Gtk::Table (2, 2, false));
	elems_table_2  = Gtk::manage (new Gtk::Table (2, 1, false));
	elems_table_3  = Gtk::manage (new Gtk::Table (2, 2, false));

	paned_1        = Gtk::manage (new Gtk::VPaned);
	paned_2        = Gtk::manage (new Gtk::VPaned);

	vbox_all       = Gtk::manage (new Gtk::VBox (false, 0));

	m_front_text   ->modify_font (CONFIG->get_example_font ());
	m_back_text    ->modify_font (CONFIG->get_example_font ());
	m_example_text ->modify_font (CONFIG->get_example_font ());
	m_asf_text     ->modify_font (CONFIG->get_example_font ());
	m_asb_text     ->modify_font (CONFIG->get_example_font ());

#ifdef IS_HILDON
	m_textbuf_ref [TV_FRONT] = Gtk::TextBuffer::create();
	m_front_text->set_buffer (m_textbuf_ref [TV_FRONT]);

	m_textbuf_ref [TV_BACK] = Gtk::TextBuffer::create();
	m_back_text->set_buffer (m_textbuf_ref [TV_BACK]);

	m_textbuf_ref [TV_EXAMPLE] = Gtk::TextBuffer::create();
	m_example_text->set_buffer (m_textbuf_ref [TV_EXAMPLE]);

	m_textbuf_ref [TV_ASF] = Gtk::TextBuffer::create();
	m_asf_text->set_buffer (m_textbuf_ref [TV_ASF]);

	m_textbuf_ref [TV_ASB] = Gtk::TextBuffer::create();
	m_asb_text->set_buffer (m_textbuf_ref [TV_ASB]);
#endif

	/**----------------**
	 **  Edit Controls **
	 **----------------**
	 */
	m_edit_controls = Gtk::manage(new EditControls(m_front_text->get_buffer()));
	m_edit_controls->unset_flags (Gtk::CAN_FOCUS);
	vbox_all->pack_start (*m_edit_controls, Gtk::PACK_SHRINK, 0);

	/**---------------**
	 ** Setup buttons **
	 **---------------**
	 */
	
	/** [Help] Button
	 */
	image1->set_alignment  (0.5,0.5);
	image1->set_padding    (0,0);

#if !defined(IS_PDA)
	SET_CV_LABEL(label1);
	hbox1->pack_start (*image1, Gtk::PACK_SHRINK, 0);
	hbox1->pack_start (*label1, Gtk::PACK_SHRINK, 0);
	alignment1->add (*hbox1);
	m_help_button->set_relief  (Gtk::RELIEF_NORMAL);
	m_help_button->add (*alignment1);
#else
	m_help_button->add (*image1);
#endif
	m_help_button->unset_flags (Gtk::CAN_FOCUS); // Take help off the chain

	/** [Cancel] Button
	 */
	image2->set_alignment  (0.5,0.5);
	image2->set_padding    (0,0);

	SET_CV_LABEL(label2);
	hbox2->pack_start (*image2, Gtk::PACK_SHRINK, 0);
	hbox2->pack_start (*label2, Gtk::PACK_SHRINK, 0);
	alignment2->add (*hbox2);
	m_cancel_button->unset_flags (Gtk::CAN_FOCUS);
	m_cancel_button->set_relief (Gtk::RELIEF_NORMAL);
	m_cancel_button->add (*alignment2);

	/** [Count] Label should be invisible
	 */
	m_count_button->set_focus_on_click (false);
	m_count_button->unset_flags (Gtk::CAN_FOCUS);
	m_count_button->set_relief (Gtk::RELIEF_NONE);
	m_count_button->set_use_stock (false);
	m_count_button->set_use_underline (false);

	/** [+ Add] Button
	 */
	image3->set_alignment  (0.5,0.5);
	image3->set_padding    (0,0);
	SET_CV_LABEL(label3);
	hbox3->pack_start (*image3, Gtk::PACK_SHRINK, 0);
	hbox3->pack_start (*label3, Gtk::PACK_SHRINK, 0);
	alignment3->add (*hbox3);
	m_add_button->set_flags (Gtk::CAN_FOCUS);
	m_add_button->set_relief (Gtk::RELIEF_NORMAL);
	m_add_button->add (*alignment3);

	/** [Add-n-Close] 
	 */
	image4->set_alignment  (0.5,0.5);
	image4->set_padding    (0,0);
	SET_CV_LABEL(label4);
	hbox4->pack_start (*image4, Gtk::PACK_SHRINK, 0);
	hbox4->pack_start (*label4, Gtk::PACK_SHRINK, 0);
	alignment4->add (*hbox4);

	m_add_n_close_button->set_flags (Gtk::CAN_FOCUS);
	m_add_n_close_button->set_relief (Gtk::RELIEF_NORMAL);
	m_add_n_close_button->add (*alignment4);

	/**--------------**
	 **  Front Text  **
	 **--------------**
	 */
	SET_CV_TEXT(m_front_text);
	set_cv_scrollwin(*front_scrollwin, *m_front_text);

	SET_CV_LABEL(front_label);
	Granule::rotate_label (front_label, 90.0, "Q:");
	ft_alignment->add (*front_label);
	vbox_frontlabel->pack_start (*ft_alignment, Gtk::PACK_SHRINK, 0);
	
	/**--------------**
	 **   Back Text  **
	 **--------------**
	 */
	SET_CV_TEXT(m_back_text);
	set_cv_scrollwin (*backtext_scrollwin, *m_back_text);
	SET_CV_LABEL(back_label);
	Granule::rotate_label (back_label, 90.0, "A:");
	bt_alignment->add (*back_label);
	vbox_backlabel->pack_start (*bt_alignment, Gtk::PACK_SHRINK, 0);

	/**----------------**
	 **  Example Text  **
	 **----------------**
	 */
	SET_CV_TEXT(m_example_text);
	set_cv_scrollwin (*example_scrollwin, *m_example_text);
	SET_CV_LABEL(example_label);
	Granule::rotate_label (example_label, 90.0, "E:");
	vbox_examplelabel->pack_start (*example_label, Gtk::PACK_SHRINK, 0);

	/**------------------------------------**
	 **  AltSpell Text for Front and Back  **
	 **------------------------------------**
	 */
	SET_CV_TEXT(m_asf_text);
	SET_CV_LABEL(asf_label);
	asf_alignment->add (*asf_label);

	hbox_asf->pack_start (*asf_alignment, Gtk::PACK_SHRINK, 0);
	hbox_asf->pack_start (*(set_text_border (m_asf_text)), 
						  Gtk::PACK_EXPAND_WIDGET, 0);
	hbox_asf->unset_flags(Gtk::CAN_FOCUS);

	SET_CV_TEXT(m_asb_text);
	SET_CV_LABEL(asb_label);
	asb_alignment->add (*asb_label);

	hbox_asb->pack_start (*asb_alignment, Gtk::PACK_SHRINK, 0);
	hbox_asb->pack_start (*(set_text_border (m_asb_text)), 
						  Gtk::PACK_EXPAND_WIDGET, 0);
	hbox_asb->unset_flags(Gtk::CAN_FOCUS);

	/** Hook up callbacks
	 */
	m_front_text->signal_grab_focus ().connect (
		bind<Gtk::TextView*> (mem_fun (*m_edit_controls, 
									   &EditControls::on_focus_changed),
							  m_front_text));

	m_back_text->signal_grab_focus ().connect (
		bind<Gtk::TextView*> (mem_fun (*m_edit_controls, 
									   &EditControls::on_focus_changed),
							  m_back_text));

	m_example_text->signal_grab_focus ().connect (
		bind<Gtk::TextView*> (mem_fun (*m_edit_controls, 
									   &EditControls::on_focus_changed),
							  m_example_text));

	m_asf_text->signal_grab_focus ().connect (
		bind<Gtk::TextView*> (mem_fun (*m_edit_controls, 
									   &EditControls::on_focus_changed),
							  m_asf_text));
	m_asb_text->signal_grab_focus ().connect (
		bind<Gtk::TextView*> (mem_fun (*m_edit_controls, 
									   &EditControls::on_focus_changed),
							  m_asb_text));
#ifdef IS_HILDON
	/** Trap 'on virtual keyboard close' event to restore
	 *  original size of the dialog
	 */
#ifdef HILDON_DEPRICATED
	HILDONAPPWIN->signal_im_close ().connect (
		mem_fun (*this, &CardView::hildon_im_close_cb));
#endif
#endif

	/*************************************************
	 *  Table holds elements together:
	 *
	 *    0                1                  2
	 *  0 .----------------+------------------.
	 *    | vbox(label)    |  scrollwin(text) | Front
	 *  1 `----------------+------------------'
	 *    --------------------------[=]-------- <-- pane
     *    0                1                  2
	 *  0 .----------------+------------------.
	 *    |                |  scrollwin(text) | Front Alt Spell
	 *  1 `----------------+------------------'
	 *    | vbox(label)    |  scrollwin(text) | Back 
	 *  2 `----------------+------------------'
	 *    --------------------------[=]-------- <-- paen
	 *    0                1                  2
	 *  0 .----------------+------------------.
	 *    | vbox(label)    |  scrollwin(text) | Example
	 *  1 `----------------+------------------'
	 *    |                |  scrollwin(text) | Back Alt Spell
	 *  2 `----------------+------------------'
	 *
	 *************************************************
	 */

	elems_table_1->set_border_width (4);
	elems_table_1->set_row_spacings (2);
	elems_table_1->set_col_spacings (2);
	elems_table_1->set_name ("CVElemsTableFront");
	
	elems_table_2->set_border_width (4);
	elems_table_2->set_row_spacings (2);
	elems_table_2->set_col_spacings (2);
	elems_table_2->set_name ("CVElemsTableBack");
	
	elems_table_3->set_border_width (4);
	elems_table_3->set_row_spacings (2);
	elems_table_3->set_col_spacings (2);
	elems_table_3->set_name ("CVElemsTableExample");
	
	/*-----------------------------------------------------------*/
	/*                               y-beg, y-end, x-beg, x-end  */
	/*-----------------------------------------------------------*/
	elems_table_1->attach (*vbox_frontlabel, 0, 1, 0, 1, 
						   Gtk::FILL, Gtk::FILL, 0, 0);

	elems_table_1->attach (*front_scrollwin, 1, 2, 0, 1, 
						   Gtk::FILL|Gtk::EXPAND, Gtk::FILL|Gtk::EXPAND, 0, 0);

	/*-----------------------------------------------------------*/
	elems_table_2->attach (*hbox_asf, 1, 2, 0, 1, 
						   Gtk::FILL|Gtk::EXPAND, Gtk::SHRINK, 0, 0);

	/*-----------------------------------------------------------*/
	elems_table_2->attach (*vbox_backlabel, 0, 1, 1, 2,
						   Gtk::FILL, Gtk::FILL, 0, 0);

	elems_table_2->attach (*backtext_scrollwin, 1, 2, 1, 2, 
						   Gtk::FILL|Gtk::EXPAND, Gtk::FILL|Gtk::EXPAND, 0, 0);

	/*-----------------------------------------------------------*/
	elems_table_3->attach (*vbox_examplelabel, 0, 1, 0, 1, 
						   Gtk::FILL, Gtk::EXPAND|Gtk::FILL, 0, 0);

	elems_table_3->attach (*example_scrollwin, 1, 2, 0, 1, 
						   Gtk::FILL|Gtk::EXPAND, Gtk::FILL|Gtk::EXPAND, 0, 0);

	/*-----------------------------------------------------------*/
	elems_table_3->attach (*hbox_asb, 1, 2, 1, 2, 
						   Gtk::FILL|Gtk::EXPAND, Gtk::SHRINK, 0, 0);
	/*-----------------------------------------------------------*/

	/** Pane 1 holds top elems_table and Pane 2.
	 *	Pane 2 holds both middle and the bottom elems_table.
	 */
	paned_2->pack1 (*elems_table_2, false, false);
	paned_2->pack2 (*elems_table_3, true,  false);

	paned_1->pack1 (*elems_table_1, true,  false);
	paned_1->pack2 (*paned_2,       true,  false);

    /** Both PDAs and Maemo have a narrow screen and thus requires
	 *  smart packing.
	 */
#if defined(IS_HILDON) || defined (IS_PDA)
	Gtk::ScrolledWindow* scrollw = Gtk::manage (new Gtk::ScrolledWindow);
	set_cv_scrollwin (*scrollw, *paned_1);
	vbox_all->pack_start (*scrollw);
#else                                      // Regular desktop 
	vbox_all->pack_start (*paned_1);
#endif

	/**--------------------------------------------------**
	 ** Pack [Help] [Cancel][+ Add][Add-n-Close] Buttons **
	 **--------------------------------------------------**
	 */
	Gtk::HBox* hbbox = Gtk::manage(new Gtk::HBox(false, 0));
	Gtk::HSeparator* hseparator = manage (new Gtk::HSeparator);

#if !defined(IS_PDA)
	Gtk::HButtonBox*  hbuttonbox = manage (new Gtk::HButtonBox);

    hbuttonbox->set_homogeneous (false);
    hbuttonbox->set_spacing (0);
	hbuttonbox->set_layout (Gtk::BUTTONBOX_END);
	hbuttonbox->set_name ("CardView_CtrlButtonBox");

	hbuttonbox->pack_end (*m_count_button,       Gtk::PACK_SHRINK, 0);
	hbuttonbox->pack_end (*m_cancel_button,      Gtk::PACK_SHRINK, 0);
	hbuttonbox->pack_end (*m_add_button,         Gtk::PACK_SHRINK, 0);
	hbuttonbox->pack_end (*m_add_n_close_button, Gtk::PACK_SHRINK, 0);

	hbbox->pack_start (*m_help_button, Gtk::PACK_SHRINK, 0);
	hbbox->pack_start (*hbuttonbox);
#else
	hbbox->pack_start (*m_help_button,        Gtk::PACK_SHRINK, 0);
	hbbox->pack_end   (*m_add_button,         Gtk::PACK_SHRINK, 0);
	hbbox->pack_end   (*m_add_n_close_button, Gtk::PACK_SHRINK, 0);
	hbbox->pack_end   (*m_cancel_button,      Gtk::PACK_SHRINK, 0);
	hbbox->pack_end   (*m_count_button,       Gtk::PACK_SHRINK, 0);
#endif	

	vbox_all->pack_start (*hseparator, Gtk::PACK_SHRINK, 0);
	vbox_all->pack_start (*hbbox,      Gtk::PACK_SHRINK, 0);

	add (*vbox_all);

	/**---------------**
	 ** Setup signals **
	 **---------------**
	 */
	m_help_button->signal_clicked ().connect (
		sigc::mem_fun (*this, &CardView::on_help_clicked));

	m_cancel_button->signal_clicked ().connect (
		sigc::mem_fun (*this, &CardView::on_cancel_clicked));

	m_add_button->signal_clicked ().connect (
		sigc::mem_fun (*this, &CardView::on_add_clicked));

	m_add_n_close_button->signal_clicked ().connect (
		sigc::mem_fun (*this, &CardView::on_addnclose_clicked));

    signal_size_allocate ().connect (
		sigc::mem_fun (*this, &CardView::size_allocate_cb));

	/**------------------**
	 ** Fill up the form **
	 **------------------**
	 */
	if (m_card != NULL) 
	{
		m_front_text   ->get_buffer ()->set_text (m_card->get_question ());
		m_back_text    ->get_buffer ()->set_text (m_card->get_answer   ());
		m_example_text ->get_buffer ()->set_text (m_card->get_example  ());
		m_asf_text->get_buffer ()->set_text (m_card->get_alt_spelling (FRONT));
		m_asb_text->get_buffer ()->set_text (m_card->get_alt_spelling (BACK));
	}

	/** Set Widget names - very helpful with debugging!
	 */
	m_front_text        ->set_name ("FrontText");
	m_back_text         ->set_name ("BackText");
	m_example_text      ->set_name ("ExampleText");
	m_asf_text          ->set_name ("AltSpellFrontText");
	m_asb_text          ->set_name ("AltSpellBackText");
	m_help_button       ->set_name ("HelpButton");
	m_cancel_button     ->set_name ("CancelButton");
	m_add_button        ->set_name ("AddButton");
	m_add_n_close_button->set_name ("AddNCloseButton");

	vbox_all->set_name ("CardView_VBoxAll");

	show_all_children ();
}

//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//                              Callbacks
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

/**
 * Called by the following methods to edit an existing card:
 *
 *    - DeckPlayer::on_vcard_edit_clicked ()
 *    - DeckView::on_edit_clicked ()
 */
int
CardView::
run (bool disable_cancel_)
{
	trace_with_mask("CardView::run(bool)",GUITRACE);

	m_syntax_error = false;
	m_add_button->set_sensitive (false);
	if (disable_cancel_) {
		m_cancel_button->set_sensitive (false);
	}

	take_snapshot ();
	Gtk::Main::run (*this);

	return (m_syntax_error ? Gtk::RESPONSE_REJECT : m_response);
}

/**
 * Handling the very first card is different from the rest.
 * The first card is created by the DeckView.
 * The rest of the cards are created here.
 *
 * Called by:
 *    
 *   - DeckView::on_add_clicked ()
 *
 * Possible values returned by run():
 *
 *    <Cancel> - Gtk::RESPONSE_CANCEL
 *    <Add>    - Gtk::RESPONSE_APPLY
 *    <Ok>     - Gtk::RESPONSE_OK
 */
int
CardView::
run (DeckView& deckview_)
{
	trace_with_mask("CardView::run(DeckView&)",GUITRACE);

	m_syntax_error = false;
	take_snapshot ();
	set_cards_count (deckview_);

	Gtk::Main::run (*this);

	DL ((GRAPP,"Processing syntax_error and response...\n"));

	if (m_syntax_error) 
	{
		DL ((GRAPP,"Caught syntax error (RESPONSE_REJECT)\n"));
		delete m_card;
		m_response = Gtk::RESPONSE_REJECT;
		goto done;
	}

	if (m_response == Gtk::RESPONSE_CANCEL) 
	{
		DL ((GRAPP,"Dialog cancelled (RESPONSE_CANCEL)\n"));
		delete m_card;
		goto done;
	}

	deckview_.add_new_card (dynamic_cast<Card*>(m_card));

	while (m_keep_open) 
	{
		DL ((GRAPP,"Looping ...\n"));
		m_card = new Card;

		Gtk::Main::run (*this);

		if (m_syntax_error) {
			delete m_card;
			m_response = Gtk::RESPONSE_REJECT;
			break;
		}
		if (m_response == Gtk::RESPONSE_CANCEL) {
			delete m_card;
			break;
		}
		deckview_.add_new_card (dynamic_cast<Card*>(m_card));
		bump_cards_count ();
	}

  done:
	return (m_response);
}

void
CardView::
on_cancel_clicked  ()
{
	trace_with_mask("CardView::on_cancel_clicked",GUITRACE);
	
	m_response = Gtk::RESPONSE_CANCEL;

	DL ((GRAPP,"Call Main::quit()\n"));
	Gtk::Main::quit ();
}

/** Fend off clicking Esc key on the keyboard.
	This often happens when I want to enter ~ (tilda) key, and instead
	press Esc. The effect is to close the editing window which leads to
	the loss of all data already entered. The only way to close dialog
	is to click on Cancel button.

	@param event_ Event data structure.
*/
bool
CardView::
on_key_press_event (GdkEventKey* event_)
{
	trace_with_mask("CardView::on_key_press_event",GEOM);
	DL((GEOM,"key_pressed = %d\n", event_->keyval));

	/** Trap and disable ESC key event.
	 *	The values for all keysyms are found 
	 *	in $prefix/gtk-2.0/gdk/gdkkeysyms.h
	 */
	if (event_->keyval == GDK_Escape) { 
		return false;
	}

    /** Keep processing keystroke
	 */
#ifdef IS_HILDON
#ifdef GLIBMM_VFUNCS_ENABLED
	return Bin::on_key_press_event (event_);
#else
	return false;   // don't know how to handle for IT2007: HILDON_DEPRICATED
#endif
#else
	return Window::on_key_press_event (event_);
#endif
}


void
CardView::
on_addnclose_clicked  ()
{
	trace_with_mask("CardView::on_addnclose_clicked",GUITRACE);

	m_response = Gtk::RESPONSE_OK;

	Glib::ustring text = m_front_text->get_buffer ()->get_text ();

	if (text.length () == 0) 
	{
		Gtk::MessageDialog err0 (
	_("<b>Front:</b> text field <span foreground=\"red\">IS EMPTY</span>!\n"
	  "This is an illegal syntax.\n"
	  "You must fill it with (unique) text.\n"),
	true, Gtk::MESSAGE_ERROR);
		err0.run ();
		m_syntax_error = true;
		return;
	}

	if (!Granule::check_markup (text)) 
	{
		Gtk::MessageDialog err1 (
			_("FRONT field contains illegal markup tag(s)."),
			false, Gtk::MESSAGE_ERROR);
		err1.run ();
		m_syntax_error = true;
		return;
	}

	if (!Granule::check_markup (m_back_text->get_buffer ()->get_text ())) 
	{
		Gtk::MessageDialog err2 (
			_("BACK field contains illegal markup tag(s)."),
			false, Gtk::MESSAGE_ERROR);
		err2.run ();
		m_syntax_error = true;
		return;
	}

	if (!Granule::check_markup (m_example_text->get_buffer ()->get_text ())) 
	{
		Gtk::MessageDialog err3 (
			_("EXAMPLE field contains illegal markup tag(s)."),
			false, Gtk::MESSAGE_ERROR);
		err3.run ();
		m_syntax_error = true;
		return;
	}

	/** Transfer data from the widget to the Card object
	 */
	if (m_card != NULL && text_modified ()) 
	{
		m_card->set_question     (m_front_text   ->get_buffer ()->get_text ());
		m_card->set_answer       (m_back_text    ->get_buffer ()->get_text ());
		m_card->set_example      (m_example_text ->get_buffer ()->get_text ());

		m_card->set_alt_spelling (FRONT, m_asf_text->get_buffer()->get_text ());
		m_card->set_alt_spelling (BACK,  m_asb_text->get_buffer()->get_text ());
	}
	m_keep_open = false;
	m_front_text->grab_focus ();

	DL ((GRAPP,"Call Main::quit()\n"));
	Gtk::Main::quit ();
}

void
CardView::
take_snapshot ()
{
	trace_with_mask("CardView::take_snapshot",GUITRACE);

	m_snapshot_front    = m_front_text   ->get_buffer ()->get_text ();
	m_snapshot_back     = m_back_text    ->get_buffer ()->get_text ();
	m_snapshot_example  = m_example_text ->get_buffer ()->get_text ();
	m_snapshot_asf      = m_asf_text     ->get_buffer ()->get_text ();
	m_snapshot_asb      = m_asb_text     ->get_buffer ()->get_text ();
}

bool
CardView::
text_modified ()
{
	return (m_snapshot_front   != m_front_text   ->get_buffer ()->get_text () ||
			m_snapshot_back    != m_back_text    ->get_buffer ()->get_text () ||
			m_snapshot_example != m_example_text ->get_buffer ()->get_text () ||
			m_snapshot_asf     != m_asf_text     ->get_buffer ()->get_text () ||
			m_snapshot_asb     != m_asb_text     ->get_buffer ()->get_text ());
}

void
CardView::
on_add_clicked  ()
{
	trace_with_mask("CardView::on_add_clicked",GUITRACE);

	on_addnclose_clicked ();
	clear_dialog ();
	m_keep_open = true;
	m_response = Gtk::RESPONSE_APPLY;
}

void
CardView::
on_help_clicked  ()
{
	trace_with_mask("CardView::on_help_clicked",GUITRACE);

	if (m_help_dialog == NULL) {
		m_help_dialog = Gtk::manage (new CardViewHelpDialog (*this));
	}
	m_help_dialog->show ();
}

void
CardView::
clear_dialog ()
{
	m_front_text  ->get_buffer ()->set_text ("");
	m_back_text   ->get_buffer ()->set_text ("");
	m_example_text->get_buffer ()->set_text ("");
	m_asf_text    ->get_buffer ()->set_text ("");
	m_asb_text    ->get_buffer ()->set_text ("");

	m_snapshot_front    = "";
	m_snapshot_back     = "";
	m_snapshot_example  = "";
	m_snapshot_asf      = "";
	m_snapshot_asb      = "";
}

void
CardView::
size_allocate_cb (Gtk::Allocation& allocation_)
{
    trace_with_mask("CardView::size_allocate_cb",GEOM);

	CONFIG->set_card_view_geometry (allocation_.get_width (),
									allocation_.get_height ());
}

#ifdef IS_HILDON
void
CardView::
hildon_im_close_cb ()
{
    trace_with_mask("CardView(hildon)::hildon_im_close_cb",GUITRACE);

	set_size_request (672, 396);
}
#endif

void
CardView::
set_cards_count (DeckView& deckview_)
{
	static char buf [32];
	m_cards_count = deckview_.get_deck_size ();

#ifdef IS_PDA
	sprintf (buf, "%d", m_cards_count);
#else
	sprintf (buf, "%d cards", m_cards_count);
#endif
	m_count_button->set_label (buf);
}

void
CardView::
bump_cards_count ()
{
	static char buf [32];

	++m_cards_count;

#ifdef IS_PDA
	sprintf (buf, "%d", m_cards_count);
#else
	sprintf (buf, "%d cards", m_cards_count);
#endif

	m_count_button->set_label (buf);
}
