/* Geoclue status applet - control UI for Geoclue
 * Copyright (C) 2007 Jussi Kukkonen
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2 as published by the Free Software Foundation;
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, see
 * <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <glib.h>
#include <string.h> /* for strlen */
#include <pthread.h>

#include <gconf/gconf-client.h>
#include <gtk/gtk.h>

#include <hildon-status-bar-lib/hildon-status-bar-item.h>
#include <hildon-widgets/hildon-banner.h>
#include <geoclue/position.h>

#include "geoclue-status.h"


#define HILDON_STATUS_BAR_PRIORITY 1
#define HILDON_STATUS_BAR_MENU_Y_POS 60
#define STATUS_MENU_WIDTH 300
#define GEOCLUE_ICON "/usr/share/pixmaps/geoclue40x40.png"

/* gconf constants */
#define GEOCLUE_POS_DIR "/apps/geoclue/position"

#define GEOCLUE_POS_PATH_KEY GEOCLUE_POS_DIR"/defaultpath"
#define GEOCLUE_POS_PATH "/org/freedesktop/geoclue/position"

#define GEOCLUE_POS_SERVICE_KEY GEOCLUE_POS_DIR"/defaultservice"
#define GEOCLUE_POS_SERVICE "org.freedesktop.geoclue.position"

#define POS_MANUAL "/apps/geoclue/position/manual"
#define POS_MANUAL_DESCRIPTION POS_MANUAL"/description"
#define POS_MANUAL_STREET POS_MANUAL"/street"
#define POS_MANUAL_LOCALITY POS_MANUAL"/locality"
#define POS_MANUAL_COUNTRY POS_MANUAL"/country"
#define POS_MANUAL_REGION POS_MANUAL"/region"
#define POS_MANUAL_VALID POS_MANUAL"/valid-until"
#define POS_MANUAL_TIMESTAMP POS_MANUAL"/timestamp"

void* set_location_string (void* data);


typedef struct _GeoclueStatusPlugin {
	HildonStatusBarItem* item;
	GtkWidget* button;
	GtkWidget* menu;

	GtkWidget* location_label;

	GtkWidget *manual_menuitem, *hostip_menuitem, *plazes_menuitem, *gpsd_menuitem;

	GConfClient *gconf_client;
	gchar* str;
} GeoclueStatusPlugin;

/* FIXME: Temporary global vars because geoclue signals fuck up 
   userdata-pointer currently...*/
GeoclueStatusPlugin* tmp;
position_provider *provider;


typedef struct _GeoclueManualDialog {
	GtkWidget *dialog;
	GtkWidget *name_entry;
	GtkWidget *street_entry;
	GtkWidget *locality_entry;
	GtkWidget *region_entry;
	GtkWidget *country_entry;
	GtkWidget *valid_entry;
} GeoclueManualDialog;


static void cb (void* data, char* country,
                char* region,
                char* locality,
                char* area,
                char* postalcode,
                char* street,
                char* building,
                char* floor,
                char* room,
                char* description,
                char* text)
{
	/*temporarily use global tmp-variable instead of data
      -- geoclue signal seems to break userdata-pointer*/
	GeoclueStatusPlugin* plugin = tmp;
	g_return_if_fail (plugin);
	
	gchar *location = NULL;
	gchar *notification = NULL;
	gboolean use_description, use_street, use_locality, use_region, use_country;
	
	use_description = (description && strlen (description) > 0);
	use_street = (street && strlen (street) > 0);
	use_locality = (locality && strlen (locality) > 0);
	use_region = (region && strlen (region) > 0);
	use_country = (country && strlen (country) > 0);
	
	if (use_description && use_street) {
		location = g_strdup_printf ("%s (%s)", description, street);
	} else if (use_description && use_locality) {
		location = g_strdup_printf ("%s (%s)", description, locality);
	} else if (use_description) {
		location = g_strdup (description);
	} else if (use_street && use_locality) {
		location = g_strdup_printf ("%s, %s", street, locality);
	} else if (use_street) {
		location = g_strdup (street);
	} else if (use_locality && use_region) {
		location = g_strdup_printf ("%s, %s", locality, region);
	} else if (use_locality && use_country) {
		location = g_strdup_printf ("%s, %s", locality, country);
	} else if (use_region) {
		location = g_strdup (region);
	} else if (use_country) {
		location = g_strdup (country);
	}
	

	if ((location == NULL) || (strlen (location) == 0)) {
		gtk_label_set_text (GTK_LABEL (plugin->location_label), "(no location)");
	} else {
		gtk_label_set_text (GTK_LABEL (plugin->location_label), location);
		notification = g_strdup_printf ("New location :\n%s", location);
		hildon_banner_show_information  (NULL, NULL, notification);
		g_free (notification);
	}
	if (location) g_free (location);
}


static void
preferred_backend_changed_callback (GConfClient* client,
                                    guint cnxn_id,
                                    GConfEntry* entry,
                                    gpointer data)
{
	const gchar* service;
    const gchar* backend = NULL;
	GeoclueStatusPlugin* plugin = (GeoclueStatusPlugin*)data;
	
	g_return_if_fail (plugin);
	
	service = gconf_client_get_string (plugin->gconf_client,
	                                   GEOCLUE_POS_SERVICE_KEY, NULL);
	
	if (g_str_has_suffix (service, "manual")) {
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (plugin->manual_menuitem), TRUE);
		backend = g_strdup("manual");
	}else if (g_str_has_suffix (service, "hostip")) {
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (plugin->hostip_menuitem), TRUE);
		backend = g_strdup("hostip");
	} else if (g_str_has_suffix (service, "plazes")) {
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (plugin->plazes_menuitem), TRUE);
		backend = g_strdup("plazes");
	} else if (g_str_has_suffix (service, "gpsd")) {
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (plugin->gpsd_menuitem), TRUE);
		backend = g_strdup("gpsd");
	} else{
		return;
	}


	/* The rest is a temporary hack while geoclue doesn't do the right thing...
       We're going to use a specific backend because default just doesn't work.
	 *	- remove callback
	 *	- close geoclue until it's reallyreally closed
	 *	- init geoclue again
	 *	- add callback
	 */
	
	if(provider){
		geoclue_position_remove_civic_callback (provider, (civic_callback)cb, plugin);
    	geoclue_position_close (provider);
	}
	provider = g_malloc(sizeof(position_provider));
	provider->service = g_strdup_printf ("org.freedesktop.geoclue.position.%s", backend);
	provider->path = g_strdup_printf ("/org/freedesktop/geoclue/position/%s", backend);
	provider->connection = NULL;
	provider->proxy = NULL;
	provider->ref = 1;

	if (geoclue_position_init (provider) != GEOCLUE_POSITION_SUCCESS){
		return;
	}
	if (geoclue_position_add_civic_callback (provider, (civic_callback)cb, plugin) != GEOCLUE_POSITION_SUCCESS){
		return;
	}

	char *country = NULL;
	char *region = NULL;
	char *locality = NULL;
	char *area = NULL;
	char *postalcode = NULL;
	char *street = NULL;
	char *building = NULL;
	char *floor = NULL;
	char *room = NULL;
	char *description = NULL;
	char *text = NULL;

	/*update the position manually once, whether or not geoclue has info */
	geoclue_position_civic_location (NULL, 
	    &country, &region, &locality, &area, &postalcode, &street,
        &building, &floor, &description, &room, &text);
	
	cb (plugin,
	    country, region, locality, area, postalcode, street, 
	    building, floor, room, description, text);
}


static void set_preferred_backend (const gchar* backend, gpointer data)
{
	gchar* path;
	gchar* service;
	GConfChangeSet* changeset;
	
	GeoclueStatusPlugin* plugin = (GeoclueStatusPlugin*)data;
	g_return_if_fail (plugin);
	g_return_if_fail (plugin->gconf_client);
	
	path = g_strdup_printf ("%s/%s",GEOCLUE_POS_PATH, backend);
	service = g_strdup_printf ("%s.%s",GEOCLUE_POS_SERVICE, backend);
	
	/* update gconf keys in a single commit */
	changeset = gconf_change_set_new ();
	gconf_change_set_set_string (changeset, GEOCLUE_POS_PATH_KEY, path);
	gconf_change_set_set_string (changeset, GEOCLUE_POS_SERVICE_KEY, service);
	gconf_client_commit_change_set (plugin->gconf_client, changeset, FALSE, NULL);
	gconf_change_set_unref (changeset);
	
	g_free (path);
	g_free (service);
	return;
}

static void manual_menuitem_activated (GtkWidget *widget, gpointer data)
{
	set_preferred_backend ("manual", data);
	return;
}

static void gpsd_menuitem_activated (GtkWidget *widget, gpointer data)
{
	set_preferred_backend ("gpsd", data);
	return;
}

static void hostip_menuitem_activated (GtkWidget *widget, gpointer data)
{
	set_preferred_backend ("hostip", data);
	return;
}

static void plazes_menuitem_activated (GtkWidget *widget, gpointer data)
{
	set_preferred_backend ("plazes", data);
	return;
}


static void create_dialog (GeoclueManualDialog* dialog)
{
	GtkWidget *label, *table, *separator;
	gint row = 0;
	
	dialog->dialog = gtk_dialog_new_with_buttons ("Set location",
	                                              NULL,
	                                              GTK_DIALOG_DESTROY_WITH_PARENT,
	                                              GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
	                                              GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
	                                              NULL);
	table = gtk_table_new (6, 3, FALSE);
	
	label = gtk_label_new ("Place name");
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, row, row+1);
	dialog->name_entry = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), dialog->name_entry, 1, 3, row, row+1);
	
	row = row + 1;
	label = gtk_label_new ("Street address");
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, row, row+1);
	dialog->street_entry = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), dialog->street_entry, 1, 3, row, row+1);
	
	row = row + 1;
	label = gtk_label_new ("City");
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, row, row+1);
	dialog->locality_entry = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), dialog->locality_entry, 1, 3, row, row+1);
	
	row = row + 1;
	label = gtk_label_new ("State");
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, row, row+1);
	dialog->region_entry = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), dialog->region_entry, 1, 3, row, row+1);
	
	row = row + 1;
	label = gtk_label_new ("Country");
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, row, row+1);
	dialog->country_entry = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), dialog->country_entry, 1, 3, row, row+1);
	
	row = row + 1;
	separator = gtk_hseparator_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), separator, 0, 3, row, row+1);
	
	row = row + 1;
	label = gtk_label_new ("Staying here for");
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, row, row+1);
	
	dialog->valid_entry = gtk_entry_new ();
	gtk_entry_set_width_chars (GTK_ENTRY (dialog->valid_entry), 3);
	gtk_table_attach_defaults (GTK_TABLE (table), dialog->valid_entry, 1, 2, row, row+1);
	
	label = gtk_label_new ("hours");
	gtk_table_attach_defaults (GTK_TABLE (table), label, 2, 3, row, row+1);
	
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog->dialog)->vbox), table);
	
	/* table is built, show it */
	gtk_widget_show_all (table);
}

static void
set_entry_text_from_gconf (GConfClient* gconf, const gchar* gconf_key, GtkEntry* entry)
{
	gchar *tmp_str = gconf_client_get_string (gconf, gconf_key, NULL);
	gtk_entry_set_text (entry, tmp_str);
	g_free (tmp_str);
}

static void
add_entry_text_to_changeset (GConfChangeSet* changeset, const gchar* gconf_key, GtkEntry* entry)
{
	gchar* tmp_str = (gchar*)gtk_entry_get_text (entry);
	gconf_change_set_set_string (changeset, gconf_key, tmp_str);
}

static void
fill_dialog_from_gconf (GeoclueManualDialog* dialog, GConfClient* gconf)
{
	gchar* tmp_str;
	GTimeVal curr_time;
	gint valid_until, valid_hours;
	
	set_entry_text_from_gconf (gconf, POS_MANUAL_DESCRIPTION, GTK_ENTRY (dialog->name_entry));
	set_entry_text_from_gconf (gconf, POS_MANUAL_STREET, GTK_ENTRY (dialog->street_entry));
	set_entry_text_from_gconf (gconf, POS_MANUAL_LOCALITY, GTK_ENTRY (dialog->locality_entry));
	set_entry_text_from_gconf (gconf, POS_MANUAL_REGION, GTK_ENTRY (dialog->region_entry));
	set_entry_text_from_gconf (gconf, POS_MANUAL_COUNTRY, GTK_ENTRY (dialog->country_entry));
	
	g_get_current_time (&curr_time);
	valid_until = gconf_client_get_int (gconf, POS_MANUAL_VALID, NULL);
	valid_hours = (int)((valid_until - curr_time.tv_sec) / (60.0 * 60.0) + 0.5);
	if (valid_hours < 1) {
		valid_hours = 1;
	}
	tmp_str = g_strdup_printf ("%d", valid_hours);
	gtk_entry_set_text (GTK_ENTRY (dialog->valid_entry), tmp_str);
	g_free (tmp_str);
}

static void
update_gconf_from_dialog (GeoclueManualDialog* dialog, GConfClient* gconf)
{
	gchar* tmp_str;
	GTimeVal curr_time;
	gint valid_until;
	
	g_get_current_time (&curr_time);
	tmp_str = (gchar*)gtk_entry_get_text (GTK_ENTRY (dialog->valid_entry));
	valid_until = curr_time.tv_sec + 60 * 60 * atoi (tmp_str);
	
	GConfChangeSet* changeset = gconf_change_set_new ();
	add_entry_text_to_changeset (changeset, POS_MANUAL_DESCRIPTION, GTK_ENTRY (dialog->name_entry));
	add_entry_text_to_changeset (changeset, POS_MANUAL_STREET, GTK_ENTRY (dialog->street_entry));
	add_entry_text_to_changeset (changeset, POS_MANUAL_LOCALITY, GTK_ENTRY (dialog->locality_entry));
	add_entry_text_to_changeset (changeset, POS_MANUAL_REGION, GTK_ENTRY (dialog->region_entry));
	add_entry_text_to_changeset (changeset, POS_MANUAL_COUNTRY, GTK_ENTRY (dialog->country_entry));
	gconf_change_set_set_int (changeset, POS_MANUAL_VALID, valid_until);
	gconf_change_set_set_int (changeset, POS_MANUAL_TIMESTAMP, curr_time.tv_sec);
	gconf_client_commit_change_set (gconf, changeset, FALSE, NULL);
	gconf_change_set_unref (changeset);
}

/* user has selected "set manual position" menu item */
static void set_manually_activated (GtkWidget *widget, gpointer data)
{
	GeoclueStatusPlugin* plugin = (GeoclueStatusPlugin*)data;
	GeoclueManualDialog* dialog = g_new0 (GeoclueManualDialog, 1);
	
	create_dialog (dialog);
	fill_dialog_from_gconf (dialog, plugin->gconf_client);
	if (gtk_dialog_run (GTK_DIALOG (dialog->dialog)) == GTK_RESPONSE_ACCEPT) {
		update_gconf_from_dialog (dialog, plugin->gconf_client);
        if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (plugin->manual_menuitem))){
			/*if manual was already selected, make sure data gets updated */
		    preferred_backend_changed_callback (plugin->gconf_client, 0, NULL, (gpointer)plugin);			
		} else {
			set_preferred_backend ("manual", data);
		}
	}
	
	gtk_widget_destroy (dialog->dialog);
	g_free (dialog);
	
	return;
}

static void position_status_menu (GtkMenu *menu, gint *x, gint *y, gboolean *in, gpointer data)
{
	/* this is a bit of a hack -- see also comment about menu width in create_menu...*/
    /* FIXME: width is now based in clicking point, not menu button */
	*x = *x - STATUS_MENU_WIDTH + 26;
	*y = HILDON_STATUS_BAR_MENU_Y_POS;
}

static void status_icon_pressed (GtkWidget *widget, gpointer data)
{
	GeoclueStatusPlugin* plugin = (GeoclueStatusPlugin*)data;
	gtk_menu_popup (GTK_MENU (plugin->menu), NULL, NULL,
	                position_status_menu,
	                plugin, 1,
	                gtk_get_current_event_time ());
	return;
}


static void create_menu (GeoclueStatusPlugin* plugin)
{
	GtkWidget *separator = NULL;
	GtkWidget *location_menuitem = NULL;
	GtkWidget *set_manually_menuitem = NULL;
	
	plugin->menu = gtk_menu_new();

    /* FIXME:
       This is not correct, but if the menu width changes (because of 
       changing location_label) while it's open, there's an ugly flash when
       menu is first drawn in upper left corner of screen. */
    gtk_widget_set_size_request (plugin->menu, STATUS_MENU_WIDTH, -1);

	location_menuitem = gtk_menu_item_new_with_label ("(No location)");
	plugin->location_label = gtk_bin_get_child (GTK_BIN (location_menuitem));
	gtk_widget_set_sensitive (location_menuitem, FALSE);
	gtk_menu_shell_append (GTK_MENU_SHELL (plugin->menu), location_menuitem);
	
	set_manually_menuitem = gtk_menu_item_new_with_label ("Set manual position");
	gtk_menu_shell_append (GTK_MENU_SHELL (plugin->menu), set_manually_menuitem);
	g_signal_connect (G_OBJECT (set_manually_menuitem),
	                  "activate",
	                  G_CALLBACK (set_manually_activated),
	                  plugin);
	
	
	separator = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (plugin->menu), separator);
	
	/* FIXME: the backend list should be read and filled from gconf */
	GSList *group = NULL;
	
	plugin->manual_menuitem = gtk_radio_menu_item_new_with_label (group, "Manual");
	gtk_menu_shell_append (GTK_MENU_SHELL (plugin->menu), plugin->manual_menuitem);
	g_signal_connect (G_OBJECT (plugin->manual_menuitem),
	                  "activate",
	                  G_CALLBACK (manual_menuitem_activated),
	                  plugin);
	group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (plugin->manual_menuitem));

	plugin->gpsd_menuitem = gtk_radio_menu_item_new_with_label (group, "GPS");
	gtk_menu_shell_append (GTK_MENU_SHELL (plugin->menu), plugin->gpsd_menuitem);
	g_signal_connect (G_OBJECT (plugin->gpsd_menuitem),
	                  "activate",
	                  G_CALLBACK (gpsd_menuitem_activated),
	                  plugin);
	group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (plugin->gpsd_menuitem));

	plugin->plazes_menuitem = gtk_radio_menu_item_new_with_label (group, "Plazes");
	gtk_menu_shell_append (GTK_MENU_SHELL (plugin->menu), plugin->plazes_menuitem);
	g_signal_connect (G_OBJECT (plugin->plazes_menuitem),
	                  "activate",
	                  G_CALLBACK (plazes_menuitem_activated),
	                  plugin);
	group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (plugin->plazes_menuitem));
	
	plugin->hostip_menuitem = gtk_radio_menu_item_new_with_label (group, "Hostip");
	gtk_menu_shell_append (GTK_MENU_SHELL (plugin->menu), plugin->hostip_menuitem);
	g_signal_connect (G_OBJECT (plugin->hostip_menuitem),
	                  "activate",
	                  G_CALLBACK (hostip_menuitem_activated),
	                  plugin);
	group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (plugin->hostip_menuitem));
	
	gtk_widget_show_all (plugin->menu);
}

static void* initialize (HildonStatusBarItem* item, GtkWidget** button)
{
	GtkWidget *image = NULL;
	GeoclueStatusPlugin* plugin;
	
	g_return_val_if_fail (item, NULL);
	
	plugin = g_new0 (GeoclueStatusPlugin, 1);
	plugin->item = item;
	
	plugin->gconf_client = gconf_client_get_default();
	/* set the gconf event handler */
	/* TODO: should save the notify id, and remove the notify on exit...
	    also, gconf_client_remove_dir should be called*/
	gconf_client_add_dir (plugin->gconf_client,
	                      GEOCLUE_POS_DIR,
	                      GCONF_CLIENT_PRELOAD_NONE, NULL);
	gconf_client_notify_add (plugin->gconf_client,
	                         GEOCLUE_POS_SERVICE_KEY,
	                         (GConfClientNotifyFunc)preferred_backend_changed_callback,
	                         (gpointer)plugin,
	                         NULL, NULL);
	
	/* build statusbar button */
	plugin->button = gtk_button_new ();
	image = gtk_image_new_from_file (GEOCLUE_ICON);
	gtk_container_add (GTK_CONTAINER (plugin->button), image);
	g_signal_connect (G_OBJECT (plugin->button),
	                  "pressed",
	                  G_CALLBACK (status_icon_pressed),
	                  plugin);
	


	/* build statusbar menu */
	create_menu (plugin);

	/*FIXME*/
	tmp = plugin;

    /* run the callback function once, so current default backend gets 
       selected in the menu and location text is filled */
    preferred_backend_changed_callback (plugin->gconf_client, 0, NULL, (gpointer)plugin);

	gtk_widget_show_all (plugin->button);
	
	*button = plugin->button;
	return (void *)plugin;
}

static void destroy (void* data)
{
	GeoclueStatusPlugin* plugin = (GeoclueStatusPlugin*)data;
	g_return_if_fail (plugin);
	g_free (plugin);
}


static int get_priority (void* data)
{
	return HILDON_STATUS_BAR_PRIORITY;
}


/* exported function */
void geoclue_status_entry (HildonStatusBarPluginFn_st* fn)
{
	g_return_if_fail (fn);
	
	fn->initialize = initialize;
	fn->destroy = destroy;
	fn->get_priority = get_priority;
}

