#
# camera.py 
# Easy API - camera module.
#
# Copyright (C) 2005-2006 INdT - Instituto Nokia de Tecnologia
#
# Contact: Luciano Miguel Wolf <luciano.wolf@indt.org.br>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

import pygst
pygst.require("0.10")
import gst
import os
import time
import thread

import easy_exceptions

__all__ = ["is_playing", "is_paused", "is_recording", "is_started",
           "is_stopped", "play", "start", "pause", "stop_displaying",
           "stop_recording", "stop_playing", "record", "record_async", "click",
           "click_async", "get_length", "get_current_position", "seek",
           "configure", "set_window_id"]

class FileNotExistError(easy_exceptions.EasyError): pass
class DeviceNotStartedError(easy_exceptions.EasyError): pass
class OutOfRangeError(easy_exceptions.EasyError): pass
class FileNotSpecifiedError(easy_exceptions.EasyError): pass

class Camera(object):
    """Camera module is composed of manipulation functions for the camera
    device. These functions include recording, playing videos and taking
    snapshots. Some examples of features include seek, file time, current time
    position and changing video properties (brightness, contrast, hue and
    saturation).

    The Camera module has two pipelines. The first one, the display pipeline,
    has the capability of displaying captured video stream from the camera
    device on the screen, recording stream as a video file. The display
    pipeline also allows taking snapshots. The second one, the playback
    pipeline, has the capability of playing videos. All those operations can be
    done in a synchronous or asynchronous way.
    """

    def __init__(self):
        """Constructor for Camera class.
        """
        self._PLAYING = False
        self._PAUSED = False
        self._RECORDING = False
        self._STARTING = False

        self._current_pipeline = None
        self._window_id = None

        self._playback_pipeline = gst.parse_launch("playbinmaemo")
        self._make_source_pipeline()

        self._make_display_bin()
        self._make_recorder_bin()

        self.time_format = gst.Format(gst.FORMAT_TIME)

    def _set_playback_pipeline(self):
        """Build a GStreamer pipeline to play an video file.
        """
        if self._current_pipeline:
            self.stop_playing()
        self._current_pipeline = self._playback_pipeline

    def _set_source_pipeline(self):
        """Build a GStreamer pipeline to record an video file.
        """
        self._current_pipeline = self._source_pipeline

    def _make_source_pipeline(self):
        """Build a GStreamer pipeline.
        """
        self._source_pipeline = gst.Pipeline("source_pipeline")

        source = gst.element_factory_make("gconfv4l2src", "source")
        filter = gst.element_factory_make("capsfilter", "source_filter")
        fakesink = gst.element_factory_make("fakesink", "fake_sink")
        self._tee = gst.element_factory_make("tee", "tee")

        caps = gst.Caps("video/x-raw-yuv,width=176,height=144")

        self._source_pipeline.add(source)
        self._source_pipeline.add(filter)
        self._source_pipeline.add(self._tee)
        self._source_pipeline.add(fakesink)

        filter.set_property("caps", caps)

        source.link(filter)
        filter.link(self._tee)
        self._tee.link(fakesink)

    def _make_display_bin(self):
        """Build a GStreamer bin to display camera's image in the screen.
        """
        self._display_bin = gst.Bin("display_bin")

        self._sink = gst.element_factory_make("xvimagesink", "sink")
        self._display_bin.add(self._sink)

        pad = self._sink.get_pad("sink")
        ghostpad = gst.GhostPad("sink", pad)
        self._display_bin.add_pad(ghostpad)

        self._source_pipeline.add(self._display_bin)

    def _make_recorder_bin(self):
        """Build a GStreamer pipeline to record a video file.
        """
        self._recorder_bin = gst.Bin("recorder_bin")
        queue = gst.element_factory_make("queue", "queue")
        encoder = gst.element_factory_make("hantro4200enc", "encoder")
        muxer = gst.element_factory_make("avimux", "muxer")
        filter = gst.element_factory_make("capsfilter", "filter")
        self._filesink = gst.element_factory_make("gnomevfssink", "filesink")

        self._filesink.connect("allow-overwrite", lambda *x: True)

        caps = gst.Caps("video/x-raw-yuv")

        self._recorder_bin.add(queue)
        self._recorder_bin.add(filter)
        self._recorder_bin.add(encoder)
        self._recorder_bin.add(muxer)
        self._recorder_bin.add(self._filesink)

        queue.link(encoder)
        encoder.link(muxer)
        muxer.link(filter)
        filter.link(self._filesink)

        filter.set_property("caps", caps)

        pad = queue.get_pad("sink")
        ghostpad = gst.GhostPad("sink", pad)
        self._recorder_bin.add_pad(ghostpad)

    def _make_snapshot_bin(self):
        """Build a GStreamer pipeline to take photographs.
        """
        self._snapshot_bin = gst.Bin("snapshot_bin")
        self.caps = gst.element_factory_make("ffmpegcolorspace", "caps")
        jpegenc = gst.element_factory_make("jpegenc", "jpegenc")
        self._filesink_photo = gst.element_factory_make("filesink",
                                                       "filesink_photo")

        self._snapshot_bin.add(self.caps)
        self._snapshot_bin.add(jpegenc)
        self._snapshot_bin.add(self._filesink_photo)

        self.caps.link(jpegenc)
        jpegenc.link(self._filesink_photo)

        pad = self.caps.get_pad("sink")
        ghostpad = gst.GhostPad("sink", pad)
        self._snapshot_bin.add_pad(ghostpad)

        self._source_pipeline.add(self._snapshot_bin)

    def is_playing(self):
        """Check if device is playing.

        @rtype:  boolean
        @return: True, if device is playing
        """
        return self._PLAYING

    def is_paused(self):
        """Check if device is paused.

        @rtype:  boolean
        @return: True, if device is paused
        """
        return self._PAUSED

    def is_recording(self):
        """Check if device is recording.

        @rtype:  boolean
        @return: True, if device is recording
        """
        return self._RECORDING

    def is_started(self):
        """Check if device is recording.

        @rtype:  boolean
        @return: True, if device is started
        """
        return self._STARTING

    def is_stopped(self):
        """Check if device is stopped.

        @rtype:  boolean
        @return: True, if device is stopped
        """
        return not (self.is_playing() or self.is_paused() or \
                    self.is_recording() or self.is_started())

    def play(self, filename):
        """Start playing video file.

        @type   filename: string
        @param  filename: name of the recorded file
        """
        if os.path.exists(filename):
            self._set_playback_pipeline()
            self._set_playback_file(filename)

            if self._window_id:
                self._playback_pipeline.set_property("xid", self._window_id)

            self._current_pipeline.set_state(gst.STATE_PLAYING)
            self._PLAYING = True

            _bus = self._current_pipeline.get_bus()
            _bus.add_signal_watch()
            self._bus_watch_id = _bus.connect("message", self._catch_EOS)
        else:
            raise FileNotExistError("File does not exist")

    def start(self, start_time=None):
        """Start showing camera display.

        @type  start_time: number
        @param start_time: total time, in seconds, of the recorded file
        """
        self._set_source_pipeline()

        if not self._STARTING:
            self._tee.link(self._display_bin)

        if self._window_id:
            self._sink.set_xwindow_id(self._window_id)

        self._source_pipeline.set_state(gst.STATE_PLAYING)

        self._STARTING = True
        if start_time:
            time.sleep(start_time)
            self.stop()

    def pause(self):
        """Pause the current playing video file.
        """
        if self.is_playing():
            self._playback_pipeline.set_state(gst.STATE_PAUSED)
            self._PAUSED = True
            self._PLAYING = False
        elif self.is_paused():
            self._playback_pipeline.set_state(gst.STATE_PLAYING)
            self._PLAYING = True
            self._PAUSED = False

    def stop_displaying(self):
        """Stop the current camera display.
        """
        self._source_pipeline.set_state(gst.STATE_NULL)
        self._tee.unlink(self._display_bin)

        self._PLAYING = False
        self._PAUSED = False
        self._RECORDING = False
        self._STARTING = False

    def stop_recording(self, callback=None):
        """Stop recording video.

       @type  callback: function
       @param callback: function that will be called as a callback after record
       """
        if self.is_recording():
            self._tee.unlink(self._recorder_bin)
            self._source_pipeline.remove(self._recorder_bin)

            self._RECORDING = False
            if callable(callback):
                callback()

    def stop_playing(self):
        """Stop the current camera display.
        """
        if self.is_playing():
            self._playback_pipeline.set_state(gst.STATE_NULL)

            self._PLAYING = False
            self._PAUSED = False
            self._RECORDING = False
            self._STARTING = False

    def record(self, filename="file.avi", record_time=None, delay=0,
               callback=None):
        """Record image captured by the camera in a syncronous way. The
        program's main thread is blocked in this process.

        @type     filename: string
        @param    filename: name of the recorded file
        @type  record_time: number
        @param record_time: total time, in seconds, of the recorded file
        @type        delay: number
        @param       delay: delay time, in seconds, that will be waited until
                            start recording
        @type     callback: function
        @param    callback: function that will be called as a callback after
                            record
        """
        self._make_recorder_bin()
        self._source_pipeline.add(self._recorder_bin)

        self._set_recorder_file(filename)
        self._tee.link(self._recorder_bin)
        # Needed time to record correctly
        time.sleep(0.4)

        time.sleep(delay)
        if not self.is_started():
            self._source_pipeline.remove(self._display_bin)
        self._source_pipeline.set_state(gst.STATE_PLAYING)

        self._RECORDING = True

        if not self.is_started():
            self._source_pipeline.add(self._display_bin)
        if record_time:
            time.sleep(record_time)
            self.stop_recording(callback)

    def record_async(self, filename="file.avi", record_time=0, delay=0,
                     callback=None):
        """Record image captured by the camera in a asyncronous way. The
        program's main thread is not blocked in this process.

        @type     filename: string
        @param    filename: name of the recorded file
        @type  record_time: number
        @param record_time: total time, in seconds, of the recorded file
        @type        delay: number
        @param       delay: delay time, in seconds, that will be waited until
                            start recording
        @type     callback: function
        @param    callback: function that will be called as a callback after
                            record
        """
        thread.start_new_thread(self.record, (filename, record_time, delay,
                                              callback))

    def click(self, filename="pic.jpg", delay=0):
        """Capture an image via camera.

        @type  filename: string
        @param filename: name of the captured picture
        @type     delay: number
        @param    delay: delay time, in seconds, that will be waited until
                            start recording
        """
        self._make_snapshot_bin()
        self._set_photo_file(filename)

        time.sleep(delay)

        if not self.is_started():
            self._source_pipeline.remove(self._display_bin)

        self._source_pipeline.set_state(gst.STATE_PLAYING)

        time.sleep(0.4)

        self._tee.link(self._snapshot_bin)
        time.sleep(1)
        self._tee.unlink(self._snapshot_bin)

        self._source_pipeline.remove(self._snapshot_bin)
        if not self.is_started():
            self._source_pipeline.add(self._display_bin)

    def click_async(self, filename="pic.jpg", delay=0):
        """Capture an image via camera in an asynchronous way. The program's
        main thread is not blocked in this process.

        @type  filename: string
        @param filename: name of the captured picture
        @type     delay: number
        @param    delay: delay time, in seconds, that will be waited until
                         start recording
        """
        thread.start_new_thread(self.click, (filename))

    def get_length(self):
        """Returns the video file's length in nano-seconds.

        @rtype:  long integer
        @return: video file's length in nano-seconds
        """
        # TODO - the method is returning -1 in any video length
        if self.is_playing() or self.is_paused():
            query = self._playback_pipeline.query_duration(self.time_format,
                                                           None)
            if query:
                return query[0]

    def get_current_position(self):
        """Return file's currently position in seconds.

        @rtype:  long integer
        @return: video file's currently position in seconds
        """
        if self.is_playing() or self.is_paused():
            query = self._playback_pipeline.query_position(self.time_format,
                                                         None)
            if query:
                return query[0]

    def seek(self, value=None):
        """Seek for the 'value' position in the video file.

        @type  value: number
        @param value: position number that will be seeked in the video file
        """
        if self.is_playing() or self.is_paused():
            self._playback_pipeline.seek_simple(self.time_format,
                                        gst.SEEK_FLAG_FLUSH, value)

    def configure(self, brightness=None, contrast=None, hue=None,
                  saturation=None):
        """Set image brightness, constrast, hue, saturation.

        @type  brightness: number
        @param brightness: integer number ranged from -1000 to 1000 that set
                           image brightness. Default: 0
        @type    contrast: number
        @param   contrast: integer number ranged from -1000 to 1000 that set
                           image contrast. Default: 0
        @type         hue: number
        @param        hue: integer number ranged from -1000 to 1000 that set
                           image hue. Default: 0
        @type  saturation: number
        @param saturation: integer number ranged from -1000 to 1000 that set
                           image saturation. Default: 0
        """
        if self._playback_pipeline.get_state()[1] == gst.STATE_STOPPED:
            raise DeviceNotStartedError("Device not started")

        self._set_color_property("brightness", brightness)
        self._set_color_property("contrast", contrast)
        self._set_color_property("hue", hue)
        self._set_color_property("saturation", saturation)

    def set_window_id(self, window_id=None):
        """Set the window id to define the place where the video should be
        displayed.

        @type   window_id: integer
        @param  window_id: windows identifier that the image will be showed
        """
        self._window_id = window_id

    def _set_color_property(self, property, value):
        #TODO - maemo gstreamer doesn't allows this function
        """Set image color property.

        @type  property: string
        @param property: the property name (brightness, contrast, hue or sat-
                         uration)
        @type     value: number
        @param    value: integer number ranged from -1000 to 1000 that set ima-
                         ge color attribute. Default: 0
        """
        if value and (value < -1000 or value > 1000):
            raise OutOfRangeError("Value must is in range from -1000 to 1000")

        self._playback_pipeline.set_state(gst.STATE_PAUSED)
        self._playback_pipeline.set_property(property, value)
        self._playback_pipeline.set_state(gst.STATE_PLAYING)

    def _set_playback_file(self, filename):
        """Set the name of the playback file.

        @type  filename: string
        @param filename: name of the playback file
        """
        if filename:
            self._current_pipeline.set_property("uri", "file://" + filename)
        else:
            raise FileNotSpecifiedError("Filename not specified")

    def _set_recorder_file(self, filename):
        """Set the name of the recorded file.

        @type  filename: string
        @param filename: name of the recorded file
        """
        self._filesink.set_property("location", filename)

    def _set_photo_file(self, filename):
        """Set the name of the recorded photo.

        @type  filename: string
        @param filename: name of the recorded file
        """
        self._filesink_photo.set_property("location", filename)

    def _catch_EOS(self, bus, message):
        """Get bus EOS(end of streamer) message and and stop pipeline. This
        method only works when the application runs a GLib/Gtk+ main loop.

        @type      bus: GSreamer element
        @param     bus: file playing pipeline's bus
        @type  message: string
        @param message: bus message
        """
        if message.type == gst.MESSAGE_EOS:
            bus.disconnect(self._bus_watch_id)
            self.stop_playing()

_camera = Camera()

is_playing = _camera.is_playing
is_paused = _camera.is_paused
is_recording = _camera.is_recording
is_started = _camera.is_started
is_stopped = _camera.is_stopped
play = _camera.play
start = _camera.start
pause = _camera.pause
stop_displaying = _camera.stop_displaying
stop_recording = _camera.stop_recording
stop_playing = _camera.stop_playing
record = _camera.record
record_async = _camera.record_async
click = _camera.click
click_async = _camera.click_async
get_length = _camera.get_length
get_current_position = _camera.get_current_position
seek = _camera.seek
configure = _camera.configure
set_window_id = _camera.set_window_id