#!/usr/bin/env python

# Copyright (C) 2005-2007 INdT - Instituto Nokia de Tecnologia
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

__author__ = "Artur Duque de Souza"
__author_email__ = "artur.souza@openbossa.org"
__license__ = "GPL"
__version__ = "0.1.2"

import os
import stat
import cPickle

class PluginPrefs(dict):
    """Plugin preferences.

    This reads and writes a dictionary with all preferences for a
    specific plugin.
    """
    prefs_dir = os.path.join(os.path.expanduser("~"), ".canola", "prefs")

    def __init__(self, plugin_name):
        dict.__init__(self)
        self._prefs_filename = os.path.join(self.prefs_dir, plugin_name)
        self._load_from_file(self._prefs_filename)

    def _load_from_file(self, filename):
        if not os.path.exists(self.prefs_dir):
            os.makedirs(self.prefs_dir, 0700)

        if not os.path.exists(filename):
            return {}

        prefs = cPickle.load(open(filename, "rb"))
        if type(prefs) is not dict:
            raise TypeError("preference file %r should have a dict" %
                            filename)

        self.update(prefs)

    def save(self):
        prefs = {}
        prefs.update(self)
        cPickle.dump(prefs, open(self._prefs_filename, "wb"),
                     cPickle.HIGHEST_PROTOCOL)

    def reload(self):
        self._load_from_file(self._prefs_filename)

def create_initial_prefs(prefs):
    def which(app):
        """Function to implement which(1) unix command"""
        pl = os.environ["PATH"].split(os.pathsep)
        for p in pl:
            path = os.path.join(p, app)
            if os.path.isfile(path):
                st = os.stat(path)
                if st[stat.ST_MODE] & 0111:
                    return path
        return None

    # create initial canolad pref file
    prefs["audio_type"] = ["audio-dummy", "id3", "ogg",
                           "pls", "m3u", "rm", "asf", "flac"]
    prefs["photo_type"] = ["jpeg", "png"]
    prefs["video_type"] = ["video-dummy", "asf", "rm", "mp4"]

    prefs["audio_table"] = "audios"
    prefs["photo_table"] = "images"
    prefs["video_table"] = "videos"

    # lms configuration
    prefs["version"] = __version__
    prefs["charsets"] = ["ISO8859-15"]
    prefs["commit_interval"] = 100
    prefs["slave_timeout"] = 10000 # 10 seconds
    prefs["db_path"] = os.path.expanduser("~/.canola/canola.db")
    prefs.save()
