/*
 * This file is part of vncviewer
 *
 * Copyright (C) 2005-2007 Aaron Levinson.
 * Copyright (C) 2006-2007 Detlef Schmicker
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gtk/gtk.h>
#include <gtk/gtkmarshal.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <hildon-widgets/hildon-get-password-dialog.h>
#include <hildon-widgets/hildon-note.h>
#include "vnc-viewer.h"
#include "interface.h"

#include <string.h>
#include <unistd.h>

// Timer to test the performance, Macro perf(A) replaced by nothing 
// if performance measurement not wanted
//#define perf

#ifdef perf
GTimer * Tperf=NULL;
gdouble T1=0,T2=0,T3=0;
guint Tcount=0;
int frames=0;
#endif

extern guint32 ReadBytes;

//gdkvisual16 is the format of the display if it supports 16 bit depth
GdkVisual * gdkvisual16=NULL;

extern void create_menu(void *mainvw);
extern void disable_home_key_lock(void *mainvw);

extern gboolean arg_lockdown;
extern gchar *arg_lockdown_password;

enum {
  VNC_FRAMEBUFFER_UPDATE,
  VNC_SET_COLOURMAP_ENTRIES,
  VNC_BELL,
  VNC_SERVER_CUT_TEXT,
  CONNECTION_DIED,
  LAST_SIGNAL
};

static guint vnc_signals[LAST_SIGNAL] = { 0 };
static GtkWidgetClass *parent_class = NULL;

static void vnc_viewer_class_init(VncViewerClass *klass);
static void vnc_viewer_init(VncViewer *vnc);

GtkType vnc_viewer_get_type(void)
{
  static GtkType vnc_type = 0;
  if (!vnc_type)
  {
    GtkTypeInfo vnc_info =
    {
      "VncViewer",
      sizeof(VncViewer),
      sizeof(VncViewerClass),
      (GtkClassInitFunc) vnc_viewer_class_init,
      (GtkObjectInitFunc) vnc_viewer_init,
      NULL,
      NULL,
    };

    vnc_type = gtk_type_unique(gtk_widget_get_type(), &vnc_info);
  }

  return vnc_type;
}

static void vnc_viewer_framebuffer_update(VncViewer *vnc,
					  rfbServerToClientMsg *msg);
static void vnc_viewer_set_colourmap_entries(VncViewer *vnc,
					     rfbServerToClientMsg *msg,
					     guint16 *entries);
static void vnc_viewer_bell(VncViewer *vnc, rfbServerToClientMsg *msg);
static void vnc_viewer_server_cut_text(VncViewer *vnc,
				       rfbServerToClientMsg *msg, gchar *text);

static void vnc_viewer_realize(GtkWidget *widget);
static void vnc_viewer_unrealize(GtkWidget *widget);
static gint vnc_viewer_expose_event(GtkWidget *widget, GdkEventExpose *event);
static gint vnc_viewer_button_press_event(GtkWidget *widget,
					  GdkEventButton *event);
static gint vnc_viewer_button_release_event(GtkWidget *widget,
					    GdkEventButton *event);
static gint vnc_viewer_motion_notify_event(GtkWidget *widget,
					   GdkEventMotion *event);

static gboolean vnc_viewer_focus(GtkWidget *widget,
				 GdkEventFocus *event,
				 GtkIMContext *im_context);

static void vnc_viewer_finalize(GObject *object);

static void vnc_viewer_class_init(VncViewerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS(klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  vnc_signals[VNC_FRAMEBUFFER_UPDATE] =
    g_signal_new("vnc_framebuffer_update",
		 G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(VncViewerClass, vnc_framebuffer_update),
		 0, 0, gtk_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 GTK_TYPE_POINTER);
  vnc_signals[VNC_SET_COLOURMAP_ENTRIES] =
    g_signal_new("vnc_set_colourmap_entries",
		 G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(VncViewerClass,vnc_set_colourmap_entries),
		 0, 0, gtk_marshal_VOID__POINTER_POINTER,
		 G_TYPE_NONE, 2,
		 GTK_TYPE_POINTER, GTK_TYPE_POINTER);
  vnc_signals[VNC_BELL] =
    g_signal_new("vnc_bell",
		 G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(VncViewerClass, vnc_bell),
		 0, 0, gtk_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 GTK_TYPE_POINTER);
  vnc_signals[VNC_SERVER_CUT_TEXT] =
    g_signal_new("vnc_server_cut_text",
		 G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(VncViewerClass, vnc_server_cut_text),
		 0, 0, gtk_marshal_VOID__POINTER_POINTER,
		 G_TYPE_NONE, 2,
		 GTK_TYPE_POINTER, GTK_TYPE_STRING);
  vnc_signals[CONNECTION_DIED] =
    g_signal_new("connection_died",
		 G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(VncViewerClass, connection_died),
		 0, 0, gtk_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);
  //g_object_class_add_signals(object_class, vnc_signals, LAST_SIGNAL);

  klass->vnc_framebuffer_update    = vnc_viewer_framebuffer_update;
  klass->vnc_set_colourmap_entries = vnc_viewer_set_colourmap_entries;
  klass->vnc_bell                  = vnc_viewer_bell;
  klass->vnc_server_cut_text       = vnc_viewer_server_cut_text;

  // signals to do with redrawing and the like
  widget_class->realize = vnc_viewer_realize;
  widget_class->unrealize = vnc_viewer_unrealize;
  widget_class->expose_event = vnc_viewer_expose_event;

  // signals that create events that should be sent to the server
  widget_class->button_press_event = vnc_viewer_button_press_event;
  widget_class->button_release_event = vnc_viewer_button_release_event;
  widget_class->motion_notify_event = vnc_viewer_motion_notify_event;
  widget_class->key_press_event = vnc_viewer_key_press_event;
  widget_class->key_release_event = vnc_viewer_key_release_event;

  object_class->finalize = vnc_viewer_finalize;
}

static void vnc_viewer_init(VncViewer *self)
{
  const char dotSrcBits[] = { 0, 14,14,14, 0 };
  const char dotMskBits[] = { 31,31,31,31,31 };
  const char noneBits[] = { 0 };
  GdkBitmap *src, *mask;
  GdkColor white, black;

  GTK_WIDGET_UNSET_FLAGS(self, GTK_NO_WINDOW);
  GTK_WIDGET_SET_FLAGS(self, GTK_CAN_FOCUS);

  self->cursor_type = VNC_VIEWER_CURSOR_NORMAL;

  self->fd = -1;

  self->im_context = gtk_im_multicontext_new();

  // commented out--no longer works with IT2006
  //g_object_set(G_OBJECT(self->im_context), "use-show-hide", TRUE, NULL);
  
  g_signal_connect(self->im_context, "commit",
		   G_CALLBACK(vnc_viewer_im_commit), self);
  // it seems that the input managers available with maemo only use the
  // commit event
  //g_signal_connect(self->im_context, "preedit_changed",
  //   G_CALLBACK(vnc_viewer_im_preedit_changed), self);

  self->in_tag = 0;
  self->out_tag = 0;

  self->out_queue = 0;

  self->width = 0;
  self->height = 0;
  memset(&(self->pixel_format), '\0', sizeof(self->pixel_format));
  self->desk_name = NULL;

  self->button_mask = 0;
  self->last_pointer_x_loc = 0;
  self->last_pointer_y_loc = 0;

  self->offscreen = NULL;

  gdk_color_white(gtk_widget_get_colormap(GTK_WIDGET(self)), &white);
  gdk_color_black(gtk_widget_get_colormap(GTK_WIDGET(self)), &black);

  src  = gdk_bitmap_create_from_data(NULL, dotSrcBits, 5, 5);
  mask = gdk_bitmap_create_from_data(NULL, dotMskBits, 5, 5);
  self->dot = gdk_cursor_new_from_pixmap(src, mask, &black, &white, 2, 2);
  gdk_bitmap_unref(src);
  gdk_bitmap_unref(mask);

  src = gdk_bitmap_create_from_data(NULL, noneBits, 1, 1);
  self->none = gdk_cursor_new_from_pixmap(src, src, &black, &white, 0, 0);
  gdk_bitmap_unref(src);

  self->m_bGotFirstUpdate = FALSE;

  self->m_lockdownKeyPressKeyVal = 0;

  self->mainview = 0;
}

GtkWidget *vnc_viewer_new(gint fd, gboolean shared, VncViewerDepthType depth,
			  VncViewerEncodingType encodingType)
{
  VncViewer *vnc = gtk_type_new(vnc_viewer_get_type());

  if (!vnc)
    return NULL;
  vnc_viewer_configure(vnc, fd, shared, depth, encodingType);

  return GTK_WIDGET(vnc);
}

void vnc_viewer_close_connection(VncViewer *vnc)
{
  // if already closed, don't do it again
  if (vnc->fd < 0)
    return;

  close(vnc->fd);
  vnc->fd = -1;

  if (vnc->in_tag)
    gdk_input_remove(vnc->in_tag);
  if (vnc->out_tag)
    gdk_input_remove(vnc->out_tag);
  vnc->in_tag = vnc->out_tag = 0;

  g_list_foreach(vnc->out_queue, (GFunc)g_free, NULL);
  g_list_free(vnc->out_queue);
  vnc->out_queue = NULL;
}

void vnc_viewer_set_cursor(VncViewer *vnc, VncViewerCursorType cursor_type)
{
  GdkCursor *cur;

  vnc->cursor_type = cursor_type;

  if (GTK_WIDGET_REALIZED(vnc))
  {
    switch (vnc->cursor_type)
    {
    case VNC_VIEWER_CURSOR_NORMAL:
      cur = gdk_cursor_new(GDK_ARROW);
      gdk_window_set_cursor(GTK_WIDGET(vnc)->window, cur);
      gdk_cursor_destroy(cur);
      break;

    case VNC_VIEWER_CURSOR_DOT:
      gdk_window_set_cursor(GTK_WIDGET(vnc)->window, vnc->dot);
      break;

    case VNC_VIEWER_CURSOR_NONE:
      gdk_window_set_cursor(GTK_WIDGET(vnc)->window, vnc->none);
      break;
    }
  }
}

static void vnc_viewer_get_event(VncViewer *vnc, gint fd,
				 GdkInputCondition cond);
static void vnc_viewer_send_event(VncViewer *vnc, gint fd,
				  GdkInputCondition cond);

static void vnc_viewer_queue_event(VncViewer *vnc, rfbClientToServerMsg *msg)
{
  if (vnc->fd < 0)
  {
    g_free(msg);
    return;
  }

  vnc->out_queue = g_list_append(vnc->out_queue, msg);

  if (vnc->out_tag == 0)
    vnc->out_tag = gdk_input_add(vnc->fd, GDK_INPUT_WRITE,
				 (GdkInputFunction)vnc_viewer_send_event,
				 vnc);
}


GdkRgbCmap *cmap;
/* fd is a connection to the VNC server just after the connection has been
 * authenticated.  Shared says whether we allow other concurrent connections,
 * and bpp is the desired bpp
 */
void vnc_viewer_configure(VncViewer *vnc, gint fd, gboolean shared,
			  VncViewerDepthType depth,
			  VncViewerEncodingType encodingType)
{
  guint32 encodings[5];
  guint16 nEncodings;
  switch (encodingType)
  {
  case VNC_VIEWER_ENCODING_AUTO:
    encodings[0] = rfbEncodingHextile;
    encodings[1] = rfbEncodingCoRRE;
    encodings[2] = rfbEncodingRRE;
    encodings[3] = rfbEncodingCopyRect;
    encodings[4] = rfbEncodingRaw;
    nEncodings = 5;
    break;

  case VNC_VIEWER_ENCODING_RAW:
    encodings[0] = rfbEncodingCopyRect;
    encodings[1] = rfbEncodingRaw;
    nEncodings = 2;
    break;

  case VNC_VIEWER_ENCODING_HEXTILE:
    encodings[0] = rfbEncodingHextile;
    nEncodings = 1;
    break;

  default:
    // shouldn't get here
    nEncodings = 0;
    break;
  }

  //guint32 localEncodings[] = { rfbEncodingCopyRect, rfbEncodingRaw };
  //guint nLocalEncodings = sizeof(localEncodings) / sizeof(localEncodings[0]);

  vnc->fd = fd;

  if (sendClientInitMsg(fd, shared) < 0)
  {
    g_warning("client init message send failed");
    return;
  }

  if (getServerInitMsg(fd, &(vnc->width), &(vnc->height), &(vnc->pixel_format),
		       &(vnc->desk_name)) < 0)
  {
    g_warning("server init message recieve failed");
    return;
  }

  g_message("Desk name='%s'", vnc->desk_name);
  g_message("Desk size=%dx%d, Default depth=%d", vnc->width, vnc->height,
	    (gint) vnc->pixel_format.depth);
  // gdk_input_add is deprecated, should be replaced
  vnc->in_tag = gdk_input_add(vnc->fd, GDK_INPUT_READ,
			      (GdkInputFunction)vnc_viewer_get_event, vnc);


//This saves the client from changing the byte order of the pixels
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
  vnc->pixel_format.bigEndian = FALSE;
#else
  vnc->pixel_format.bigEndian = TRUE;
#endif



  //Checking the supported visuals
  GList *visuals=gdk_list_visuals();
  g_message("Supported visuals:");
  visuals=g_list_first(visuals);
  while (visuals)
  {
    g_message("-");
    g_message(
	      "visual depth %d, redshift %d, greenshift %d, blueshift %d, type %d TrueColor is %d",
	      ((GdkVisual*)visuals->data)->depth,
	      ((GdkVisual*)visuals->data)->red_shift,
	      ((GdkVisual*)visuals->data)->green_shift,
	      ((GdkVisual*)visuals->data)->blue_shift,
	      ((GdkVisual*)visuals->data)->type,
	      GDK_VISUAL_TRUE_COLOR
	      );
    if ((((GdkVisual*)visuals->data)->depth)==16)
      gdkvisual16=visuals->data;
    visuals=g_list_next(visuals);
  }
  //  if this is not fullfilled, the 16 bit format must be changed in the following switch
  g_assert(gdkvisual16!=NULL);
  g_assert(gdkvisual16->red_shift==11);
  g_assert(gdkvisual16->green_shift==5);
  g_assert(gdkvisual16->blue_shift==0);
  g_assert(gdkvisual16->type==GDK_VISUAL_TRUE_COLOR);
  
  switch (depth)
  {
  case VNC_VIEWER_DEPTH_8BIT:
    //    if (vnc->pixel_format.depth > 8)
    {
      // only change depth down
      vnc->pixel_format.bitsPerPixel = 8;
      vnc->pixel_format.depth = 8;
      vnc->pixel_format.trueColour = FALSE;
      vnc->pixel_format.redMax = 7;
      vnc->pixel_format.greenMax = 7;
      vnc->pixel_format.blueMax = 3;
      vnc->pixel_format.redShift = 0;
      vnc->pixel_format.greenShift = 3;
      vnc->pixel_format.blueShift = 6;

      guint32 colors[256];
      guint32 r,g,b,ii=0;
      for (b=0;b<=vnc->pixel_format.blueMax;b++)
      {
	for (g=0;g<=vnc->pixel_format.greenMax;g++)
	{
	  for (r=0;r<=vnc->pixel_format.redMax;r++)
	  {
	    guint32 tmp=
	      (((b<<8) |0xFF)/(vnc->pixel_format.blueMax+1)) |
	      (((g<<16) |0xFFFF)/(vnc->pixel_format.greenMax+1)&0xFF00) |
	      (((r<<24) |0xFFFFFF)/(vnc->pixel_format.redMax+1)&0xFF0000) ;
	    //g_message("c %d - %8X",ii,tmp);
	    colors[ii++]=tmp;
	  }
	  cmap=gdk_rgb_cmap_new(colors,256);
	}
      }
    }
    break;

    // The N770 and N800 display has only 16 bit depth. Therefore we force
    // this to be the highest depth
    // Even the pixel format must match the display for hextile encoding working
  case VNC_VIEWER_DEPTH_DEFAULT:
  case VNC_VIEWER_DEPTH_16BIT:
  case VNC_VIEWER_DEPTH_24BIT:
    //if (vnc->pixel_format.depth > 16)
    {
      // only change depth down
      vnc->pixel_format.bitsPerPixel = 16;
      vnc->pixel_format.depth = 16;
      vnc->pixel_format.trueColour = TRUE;
      vnc->pixel_format.redMax = 31;
      vnc->pixel_format.greenMax = 63;
      vnc->pixel_format.blueMax = 31;
      vnc->pixel_format.redShift = 11;
      vnc->pixel_format.greenShift = 5;
      vnc->pixel_format.blueShift = 0;
    }
    break;

    /*
  case VNC_VIEWER_DEPTH_24BIT:
    // we want to force this format
    // if (vnc->pixel_format.depth > 24)
    {
      vnc->pixel_format.bitsPerPixel = 32;
      vnc->pixel_format.depth = 24;
      vnc->pixel_format.trueColour = TRUE;
      vnc->pixel_format.redMax = 255;
      vnc->pixel_format.greenMax = 255;
      vnc->pixel_format.blueMax = 255;
      vnc->pixel_format.redShift = 16;
      vnc->pixel_format.greenShift = 8;
      vnc->pixel_format.blueShift = 0;
    }
    break;
    */
  }

  vnc_viewer_queue_event(vnc, newSetPixelFormatMsg(&(vnc->pixel_format)));

  /*
  if (sameMachine(vnc->fd))
    vnc_viewer_queue_event(vnc, newSetEncodingsMsg(nLocalEncodings,
						   localEncodings));
  else
  */
    vnc_viewer_queue_event(vnc, newSetEncodingsMsg(nEncodings, encodings));
}

static void vnc_viewer_get_event(VncViewer *vnc, gint fd,
				 GdkInputCondition cond)
{
  rfbServerToClientMsg *msg;

  if (cond == GDK_INPUT_EXCEPTION)
  {
    vnc_viewer_close_connection(vnc);
    g_signal_emit(G_OBJECT(vnc), vnc_signals[CONNECTION_DIED], 0);
    return;
  }

  msg = getServerMsg(fd);

  if (!msg)
  {
    vnc_viewer_close_connection(vnc);
    g_signal_emit(G_OBJECT(vnc), vnc_signals[CONNECTION_DIED], 0);
    return;
  }

  switch (msg->type)
  {
  case rfbFramebufferUpdate:
    g_signal_emit(G_OBJECT(vnc), vnc_signals[VNC_FRAMEBUFFER_UPDATE], 0, msg);
    break;

  case rfbSetColourMapEntries:
    {
      guint16 *entries = getColourMapEntries(fd, msg->scme.nColours);

      g_signal_emit(G_OBJECT(vnc), vnc_signals[VNC_SET_COLOURMAP_ENTRIES], 0, 
		      msg, entries);
      g_free(entries);
      break;
    }

  case rfbBell:
    g_signal_emit(G_OBJECT(vnc), vnc_signals[VNC_BELL], 0, msg);
    break;

  case rfbServerCutText:
    {
      gchar *text = getServerCutText(fd, msg->sct.length);

      g_signal_emit(G_OBJECT(vnc), vnc_signals[VNC_SERVER_CUT_TEXT], 0, 
		    msg, text);
      g_free(text);
      break;
    }

  default:
    break;
  }

  g_free(msg);
}

static void vnc_viewer_send_event(VncViewer *vnc, gint fd,
				  GdkInputCondition cond)
{
  rfbClientToServerMsg *msg;

  if (cond == GDK_INPUT_EXCEPTION)
  {
    vnc_viewer_close_connection(vnc);
    g_signal_emit(G_OBJECT(vnc), vnc_signals[CONNECTION_DIED], 0);
    return;
  }

  msg = vnc->out_queue->data;
  vnc->out_queue = g_list_remove(vnc->out_queue, msg);
  rfb_send_and_destroy_message(fd, msg);

  // if no more queued events exist, remove this handler
  if (vnc->out_queue == NULL)
  {
    gdk_input_remove(vnc->out_tag);
    vnc->out_tag = 0;
  }
}

static gboolean vnc_viewer_send_draw_request(VncViewer *vnc)
{
  if (vnc->fd < 0)
    return FALSE;

  //If the window can not be seen by the user, no update request is sent
  //But TRUE is returned for the function called again after timeout.
  if (!gtk_window_is_active(GTK_WINDOW(((MainView*)vnc->mainview)->data->main_view)))
    return TRUE;    

  vnc_viewer_queue_event(vnc,
			 newFramebufferUpdateRequestMsg(TRUE, 0, 0, vnc->width, vnc->height));
  return FALSE;
}

static void vnc_viewer_framebuffer_update(VncViewer *vnc,
					  rfbServerToClientMsg *msg)
{
  GdkImage * gdkimage;
  guint32 rect, srect, tile, i, j, k, bgRGB;
  guchar *buffer;
  guchar *buffer8;
  rfbPixelFormat *fmt = &(vnc->pixel_format);
  guint8 bpp = fmt->bitsPerPixel, tmpR, tmpG, tmpB;
  gdouble redScale = 255.0 / fmt->redMax, greenScale = 255.0 / fmt->greenMax,
    blueScale = 255.0 / fmt->blueMax;
//  g_message("framebuffer update");
  for (rect = 0; rect < msg->fu.nRects; rect++)
  {
#ifdef perf
    if (Tperf==NULL) Tperf=g_timer_new();
    T3+=g_timer_elapsed(Tperf,NULL);
    g_timer_start(Tperf);
#endif

    rfbRectangleDescription *desc =
      getNextRectangle(vnc->fd, vnc->pixel_format.bitsPerPixel / 8,
		       !vnc->m_bGotFirstUpdate);
#ifdef perf
    T1+=g_timer_elapsed(Tperf,NULL);
#endif
    gdkimage=NULL;
    buffer8=NULL;
    if (!vnc->m_bGotFirstUpdate)
      vnc->m_bGotFirstUpdate = TRUE;

    if (!desc)
    {
      g_signal_emit(G_OBJECT(vnc), vnc_signals[CONNECTION_DIED], 0);
      return;
    }

    guint16 rWidth = desc->header.r.w, rHeight = desc->header.r.h;
    guint16 xPos, yPos;
    guint32 totPixels = rWidth * rHeight;

    //g_message("Rectangle (%d,%d) %dx%d Encoding=%d", (gint)desc->header.r.x,
    //	      (gint)desc->header.r.y, (gint)desc->header.r.w,
    //	      (gint)desc->header.r.h, (gint)desc->header.encoding);
    switch (desc->header.encoding)
    {
    case rfbEncodingRaw:
      //g_message("raw encoding");
      if (bpp==8)
      	buffer8=g_malloc(totPixels);
      else
      {
	if (!vnc->m_bRotateDisplay)
	  gdkimage=gdk_image_new(GDK_IMAGE_FASTEST, gdkvisual16, rWidth,
				 rHeight);
	else
	  gdkimage=gdk_image_new(GDK_IMAGE_FASTEST, gdkvisual16, rHeight,
				 rWidth);
      }

      guint ii = 0;
      if (bpp == 8)
      {
	if (!vnc->m_bRotateDisplay)
	  memcpy(buffer8, &(desc->data.raw), totPixels);
	else
	{
	  guint tmp;
	  guint8 *tmpraw = (guint8 *) desc->data.raw;
	  for (j = 0; j < rHeight; j++)
	  {
	    tmp = rHeight - 1 - j;
	    for (i = 0; i < rWidth; i++)
	    {
	      buffer8[tmp] = *tmpraw;
	      tmp += rHeight;
	      tmpraw++;
	    }
	  }
	}
      }
      else
      {
	if (!vnc->m_bRotateDisplay)
	{
	  if (gdkimage->bpl == (rWidth * 2))
	  {
	    // then can do a straight memcpy of the entire buffer
	    //g_message("16-bit raw: able to do full memcpy");
	    memcpy((guint8 *) gdkimage->mem, &(desc->data.raw), totPixels * 2);
	  }
	  else
	  {
	    // then need to do each row separately
	    //g_message("16-bit raw: must do each row separately");
	    for (j = 0; j < rHeight; j++)
	    {
	      memcpy(&((guint8 *) gdkimage->mem)[j * gdkimage->bpl],
		     &(desc->data.raw)[ii], rWidth * 2);
	      ii += rWidth * 2;
	    }
	  }
	}
	else
	{
	  // must use gdkimage->bpl, instead of rHeight, because
	  // sometimes gdkimage->bpl is not equal to 2 * rHeight
	  // (for instance, if rHeight is an odd number)
	  guint16 *imgbegin = ((guint16 *) gdkimage->mem) + (rHeight - 1);
	  guint16 *tmpimg;
	  guint16 *tmpraw = (guint16 *) desc->data.raw;
	  
	  for (j = 0; j < rHeight; j++)
	  {
	    tmpimg = imgbegin - j;
	    for (i = 0; i < rWidth; i++)
	    {
	      *tmpimg = *tmpraw;
	      tmpimg = (guint16 *) (((guint8 *) tmpimg) + gdkimage->bpl);
	      tmpraw++;
	    }
	  }
	}
      }

      if (bpp == 8)
      {
	if (!vnc->m_bRotateDisplay)
	{
	  gdk_draw_indexed_image(vnc->offscreen, vnc->gc,
				 desc->header.r.x, desc->header.r.y,
				 rWidth, rHeight,
				 GDK_RGB_DITHER_NONE, buffer8, rWidth, cmap);
	}
	else
	{
	  gdk_draw_indexed_image(vnc->offscreen, vnc->gc,
				 vnc->height - desc->header.r.y - rHeight,
				 desc->header.r.x,
				 rHeight, rWidth,
				 GDK_RGB_DITHER_NONE, buffer8, rHeight, cmap);
	}
	g_free(buffer8);
	buffer8=NULL;
      }
      else
      {
	if (!vnc->m_bRotateDisplay)
	{
	  gdk_draw_image(vnc->offscreen, vnc->gc, gdkimage, 0, 0,
			 desc->header.r.x, desc->header.r.y,
			 rWidth, rHeight);
	}
	else
	{
	  gdk_draw_image(vnc->offscreen, vnc->gc, gdkimage, 0, 0,
			 vnc->height - desc->header.r.y - rHeight,
			 desc->header.r.x,
			 rHeight, rWidth);
	}
	g_object_unref(gdkimage);
      }

      break;

    case rfbEncodingCopyRect:
      if (!vnc->m_bRotateDisplay)
      {
	gdk_draw_pixmap(vnc->offscreen, vnc->gc, vnc->offscreen,
			desc->data.cr.srcX, desc->data.cr.srcY,
			desc->header.r.x, desc->header.r.y,
			rWidth, rHeight);
      }
      else
      {
	gdk_draw_pixmap(vnc->offscreen, vnc->gc, vnc->offscreen,
			vnc->height - desc->data.cr.srcY - rHeight,
			desc->data.cr.srcX,
			vnc->height - desc->header.r.y - rHeight,
			desc->header.r.x,
			rHeight, rWidth);
      }
      break;

    case rfbEncodingRRE:
      //g_message("rre encoding");
      buffer = g_malloc(totPixels * 3);
      tmpR = (desc->data.rre.bgPixel >> fmt->redShift & fmt->redMax) *
	redScale;
      tmpG = (desc->data.rre.bgPixel >> fmt->greenShift & fmt->greenMax) *
	greenScale;
      tmpB = (desc->data.rre.bgPixel >> fmt->blueShift & fmt->blueMax) *
	blueScale;

      for (i = 0; i < totPixels; i++)
      {
	if (!vnc->m_bRotateDisplay)
	{
	  buffer[3*i] = tmpR; buffer[3*i+1] = tmpG; buffer[3*i+2] = tmpB;
	}
	else
	{
	  j = rWidth * ((i % rHeight) + 1) - ((i / rHeight) + 1);
	  buffer[3*j] = tmpR; buffer[3*j+1] = tmpG; buffer[3*j+2] = tmpB;
	}
      }

      for (srect = 0; srect < desc->data.rre.nSubrects; srect++)
      {
	guint32 pixel = desc->data.rre.rects[srect].pixel;
	rfbRectangle *sr = &(desc->data.rre.rects[srect].rect);
	tmpR = (pixel >> fmt->redShift & fmt->redMax) * redScale;
	tmpG = (pixel >> fmt->greenShift & fmt->greenMax) * greenScale;
	tmpB = (pixel >> fmt->blueShift & fmt->blueMax) * blueScale;

	if (!vnc->m_bRotateDisplay)
	{
	  for (j = sr->y; j < sr->y + sr->h; j++)
	  {
	    for (i = sr->x; i < sr->x + sr->w; i++)
	    {
	      buffer[3*(rWidth*j+i)]   = tmpR;
	      buffer[3*(rWidth*j+i)+1] = tmpG;
	      buffer[3*(rWidth*j+i)+2] = tmpB;
	    }
	  }
	}
	else
	{
	  for (j = sr->y; j < sr->y + sr->h; j++)
	  {
	    for (i = sr->x; i < sr->x + sr->w; i++)
	    {
	      k = rHeight * (((rWidth * j + i) % rWidth) + 1) -
		(((rWidth * j + i) / rWidth) + 1);
	      buffer[3*k]   = tmpR;
	      buffer[3*k+1] = tmpG;
	      buffer[3*k+2] = tmpB;
	    }
	  }
	}
      }

      if (!vnc->m_bRotateDisplay)
      {
	gdk_draw_rgb_image(vnc->offscreen, vnc->gc,
			   desc->header.r.x, desc->header.r.y,
			   rWidth, rHeight,
			   GDK_RGB_DITHER_NORMAL, buffer, rWidth * 3);
      }
      else
      {
	gdk_draw_rgb_image(vnc->offscreen, vnc->gc,
			   vnc->height - desc->header.r.y - rHeight,
			   desc->header.r.x,
			   rHeight, rWidth,
			   GDK_RGB_DITHER_NORMAL, buffer, rHeight * 3);
      }

      g_free(buffer);
      break;

    case rfbEncodingCoRRE:
      //g_message("corre encoding");
      buffer = g_malloc(totPixels * 3);
      tmpR = (desc->data.corre.bgPixel >> fmt->redShift & fmt->redMax) *
	redScale;
      tmpG = (desc->data.corre.bgPixel >> fmt->greenShift & fmt->greenMax) *
	greenScale;
      tmpB = (desc->data.corre.bgPixel >> fmt->blueShift & fmt->blueMax) *
	blueScale;

      for (i = 0; i < totPixels; i++)
      {
	if (!vnc->m_bRotateDisplay)
	{
	  buffer[3*i] = tmpR; buffer[3*i+1] = tmpG; buffer[3*i+2] = tmpB;
	}
	else
	{
	  j = rWidth * ((i % rHeight) + 1) - ((i / rHeight) + 1);
	  buffer[3*j] = tmpR; buffer[3*j+1] = tmpG; buffer[3*j+2] = tmpB;
	}
      }

      for (srect = 0; srect < desc->data.corre.nSubrects; srect++)
      {
	guint32 pixel = desc->data.corre.rects[srect].pixel;
	rfbCoRRERectangle *sr = &(desc->data.corre.rects[srect].rect);
	tmpR = (pixel >> fmt->redShift & fmt->redMax) * redScale;
	tmpG = (pixel >> fmt->greenShift & fmt->greenMax) * greenScale;
	tmpB = (pixel >> fmt->blueShift & fmt->blueMax) * blueScale;

	if (!vnc->m_bRotateDisplay)
	{
	  for (j = sr->y; j < sr->y + sr->h; j++)
	  {
	    for (i = sr->x; i < sr->x + sr->w; i++)
	    {
	      buffer[3*(rWidth*j+i)]   = tmpR;
	      buffer[3*(rWidth*j+i)+1] = tmpG;
	      buffer[3*(rWidth*j+i)+2] = tmpB;
	    }
	  }
	}
	else
	{
	  for (j = sr->y; j < sr->y + sr->h; j++)
	  {
	    for (i = sr->x; i < sr->x + sr->w; i++)
	    {
	      k = rHeight * (((rWidth * j + i) % rWidth) + 1) -
		(((rWidth * j + i) / rWidth) + 1);
	      buffer[3*k]   = tmpR;
	      buffer[3*k+1] = tmpG;
	      buffer[3*k+2] = tmpB;
	    }
	  }
	}
      }

      if (!vnc->m_bRotateDisplay)
      {
	gdk_draw_rgb_image(vnc->offscreen, vnc->gc,
			   desc->header.r.x, desc->header.r.y,
			   rWidth, rHeight,
			   GDK_RGB_DITHER_NORMAL, buffer, rWidth * 3);
      }
      else
      {
	gdk_draw_rgb_image(vnc->offscreen, vnc->gc,
			   vnc->height - desc->header.r.y - rHeight,
			   desc->header.r.x,
			   rHeight, rWidth,
			   GDK_RGB_DITHER_NORMAL, buffer, rHeight * 3);
      }

      g_free(buffer);
      break;

    // optimized hextile encoding only supports 8 and 16 bit colordepth
    // Code depends on the documented GTK-structure GdkImage of the gpointer *mem
    // May break in case this is changed in GDK
    case rfbEncodingHextile:
      //g_message("hextile encoding");
      if (bpp==8)
      	buffer8=g_malloc(totPixels);
      else
      {
	if (!vnc->m_bRotateDisplay)
	  gdkimage=gdk_image_new(GDK_IMAGE_FASTEST, gdkvisual16, rWidth,
				 rHeight);
	else
	  gdkimage=gdk_image_new(GDK_IMAGE_FASTEST, gdkvisual16, rHeight,
				 rWidth);
      }

      xPos = yPos = 0;
      for (tile = 0; tile < desc->data.hextile.nTiles; tile++)
      {
	rfbHextile *ht = desc->data.hextile.tiles[tile];

	if (ht->type & rfbHextileRaw)
	{
	  //g_message("got raw hextile");
	  guint ii=0;
	  if (bpp==8)
	  {
	    if (!vnc->m_bRotateDisplay)
	    {
	      for (j = 0; j < ht->height; j++)
	      {
		//for (i = 0; i < ht->width; i++)
		// {
		//	buffer8[(yPos+j)*rWidth+xPos+i]=(ht->data.raw)[ii++];
		// }
		memcpy(&buffer8[(yPos+j)*rWidth+xPos],&(ht->data.raw)[ii],ht->width);
		ii+=ht->width;
	      }
	    }
	    else
	    {
	      guint beginpos = (xPos * rHeight) + (rHeight - yPos - 1);
	      guint tmp;
	      guint8 *tmpraw = (guint8 *) ht->data.raw;
	      for (j = 0; j < ht->height; j++)
	      {
		tmp = beginpos - j;
		for (i = 0; i < ht->width; i++)
		{
		  buffer8[tmp] = *tmpraw;
		  tmp += rHeight;
		  tmpraw++;
		  //buffer8[((xPos + i) * rHeight) + (rHeight - yPos - j - 1)] =
		  // (ht->data.raw)[ii++];
		}
	      }
	    }
	  }
	  else
	  {
	    if (!vnc->m_bRotateDisplay)
	    {
	      for (j = 0; j < ht->height; j++)
	      {
		memcpy(&((guint8 *) gdkimage->mem)[(yPos + j) * gdkimage->bpl + xPos * 2], //2 is gdkimage->bpp
		       &(ht->data.raw)[ii], ht->width * 2);

		/*
		for (i = 0; i < ht->width; i++)
		{
		  guint32 pixel = ((guint16 *)(ht->data.raw))[ii++];
		  gdk_image_put_pixel(gdkimage, xPos + i, yPos + j, pixel);
		}
		*/
		
		ii += 2 * ht->width;
	      }
	    }
	    else
	    {
	      // must use gdkimage->bpl, instead of rHeight, because
	      // sometimes gdkimage->bpl is not equal to 2 * rHeight
	      // (for instance, if rHeight is an odd number)
	      guint beginpos =
		(xPos * gdkimage->bpl) + ((rHeight - yPos - 1) * 2);
	      guint16 *imgbegin =
		(guint16 *) (((guint8 *) gdkimage->mem) + beginpos);
	      guint16 *tmpimg;
	      guint16 *tmpraw = (guint16 *) ht->data.raw;

	      for (j = 0; j < ht->height; j++)
	      {
		tmpimg = imgbegin - j;
		for (i = 0; i < ht->width; i++)
		{
		  *tmpimg = *tmpraw;
		  tmpimg = (guint16 *) (((guint8 *) tmpimg) + gdkimage->bpl);
		  tmpraw++;
		  //guint32 pixel = ((guint16 *)(ht->data.raw))[ii++];
		  //gdk_image_put_pixel(gdkimage, rHeight - yPos - j - 1,
		  //		      xPos + i, pixel);
		}
	      }
	    }
	  }
	}
	else
	{
	  if (ht->type & rfbHextileBackgroundSpecified)
	  {
	    bgRGB = ht->data.rects.bgcolour;
	  }

	  if (bpp==8)
	  {
	    if (!vnc->m_bRotateDisplay)
	    {
	      guint tmp=(yPos)*rWidth+xPos;
	      for (j = 0; j < ht->height; j++)
	      {
		memset(&buffer8[tmp],bgRGB,ht->width);
		tmp+=rWidth;
	      }
	    }
	    else
	    {
	      guint tmp = (xPos * rHeight) + (rHeight - yPos - ht->height);
	      for (j = 0; j < ht->width; j++)
	      {
		memset(&buffer8[tmp], bgRGB, ht->height);
		tmp+=rHeight;
	      }
	    }
	  }
	  else
	  {
	    if (!vnc->m_bRotateDisplay)
	    {
	      guint16 * tmpimg = (guint16 *)
		(&((guint8*)gdkimage->mem)[yPos * gdkimage->bpl + xPos*2]);
	      for (j = 0; j < ht->height; j++)
	      {
		guint8 * tmp=(guint8*) tmpimg;
		for (i = 0; i < ht->width; i++)
		{
		  //  gdk_image_put_pixel(gdkimage,xPos+i,yPos+j,bgRGB);
		  *tmpimg=bgRGB;
		  *tmpimg++;
		}
		tmpimg=(guint16*) (tmp+gdkimage->bpl);
	      }
	    }
	    else
	    {
	      guint16 * tmpimg = (guint16 *)
		(&((guint8*)gdkimage->mem)[xPos * gdkimage->bpl +
					   ((rHeight - yPos - ht->height) *
					    2)]);
	      for (j = 0; j < ht->width; j++)
	      {
		guint8 * tmp = (guint8 *) tmpimg;
		for (i = 0; i < ht->height; i++)
		{
		  //  gdk_image_put_pixel(gdkimage,xPos+i,yPos+j,bgRGB);
		  *tmpimg=bgRGB;
		  *tmpimg++;
		}
		tmpimg=(guint16*) (tmp+gdkimage->bpl);
	      }
	    }
	  }

	  if (ht->type & rfbHextileAnySubrects)
	  {
	    for (srect = 0; srect < ht->data.rects.nRects; srect++)
	    {
	      guint16 pixel;
	      guint8 wtype=0;
	      if (ht->type & rfbHextileSubrectsColoured)
		wtype=2;
	      if (bpp==16)
		wtype++;

	      if (!vnc->m_bRotateDisplay)
	      {
		switch (wtype)
		{
		case 0:
		  // no subrect color, 8-bit pixel
		  {
		    pixel = ht->data.rects.fgcolour;
		    guint tmp=
		      (yPos +
		       rfbHextileExtractY(ht->data.rects.rects0[srect].xy)) *
		      rWidth + xPos +
		      rfbHextileExtractX(ht->data.rects.rects0[srect].xy);
		    for (j = 0; j < rfbHextileExtractH(ht->data.rects.rects0[srect].wh); j++)
		    {
		      memset(&buffer8[tmp], pixel,
		           rfbHextileExtractW(ht->data.rects.rects0[srect].wh));
		      tmp+=rWidth;
		    }
		  }
		  break;
		case 1:
		  {
		    //g_message("case1");
		    // no subrect color, 16-bit pixel
		    pixel = ht->data.rects.fgcolour;
		    guint16 * tmpimg=(guint16*)
		      (&((guint8*)gdkimage->mem)
		       [(yPos+ rfbHextileExtractY(ht->data.rects.rects0[srect].xy))
			*gdkimage->bpl
			+(xPos+rfbHextileExtractX(ht->data.rects.rects0[srect].xy))*2]);
		    for (j = 0; j < rfbHextileExtractH(ht->data.rects.rects0[srect].wh); j++)
		    {
		      //	if (yPos>750)
		      //   g_message("j %d",j);
		      guint8 * tmp=(guint8*) tmpimg;
		      for (i = 0; i < rfbHextileExtractW(ht->data.rects.rects0[srect].wh); i++)
		      {
			//  gdk_image_put_pixel(gdkimage,xPos+i,yPos+j,pixel);
			*tmpimg=pixel;
			*tmpimg++;
		      }
		      tmpimg=(guint16*) (tmp+gdkimage->bpl);
		      // guint tmp=yPos + rfbHextileExtractY(ht->data.rects.rects0[srect].xy)+j;
		      // guint tmp2=xPos + rfbHextileExtractX(ht->data.rects.rects0[srect].xy);
		      // for (i = 0; i < rfbHextileExtractW(ht->data.rects.rects0[srect].wh); i++)
		      // {
		      //   gdk_image_put_pixel(gdkimage, tmp2++, tmp, pixel);
		      // }
		    }
		  }
		  break;
		case 2:
		  // subrect color, 8-bit pixel
		  {
		    pixel=ht->data.rects.rects1[srect].colour;
		    guint tmp=(yPos+ rfbHextileExtractY(ht->data.rects.rects1[srect].xy))*rWidth
		      +xPos+ rfbHextileExtractX(ht->data.rects.rects1[srect].xy);
		    for (j = 0; j < rfbHextileExtractH(ht->data.rects.rects1[srect].wh); j++)
		    {
		      memset(&buffer8[tmp], 
			     pixel,rfbHextileExtractW(ht->data.rects.rects1[srect].wh));
		      tmp+=rWidth;
		    }
		  }
		  break;
		case 3:
		  {
		    //g_message("case3");
		    // subrect color, 16-bit pixel
		    pixel=ht->data.rects.rects2[srect].colour;
		    guint16 * tmpimg=(guint16*)
		      (&((guint8*)gdkimage->mem)
		       [(yPos+ rfbHextileExtractY(ht->data.rects.rects2[srect].xy))
			*gdkimage->bpl
			+(xPos+rfbHextileExtractX(ht->data.rects.rects2[srect].xy))*2]);
		    for (j = 0; j < rfbHextileExtractH(ht->data.rects.rects2[srect].wh); j++)
		    {
		      // if (yPos>750)
		      //   g_message("yPos %d Y %d j %d",yPos,j);
		      guint8 * tmp=(guint8*) tmpimg;
		      for (i = 0; i < rfbHextileExtractW(ht->data.rects.rects2[srect].wh); i++)
		      {
			//  gdk_image_put_pixel(gdkimage,xPos+i,yPos+j,pixel);
			*tmpimg=pixel;
			*tmpimg++;
		      }
		      tmpimg=(guint16*) (tmp+gdkimage->bpl);
		      // guint tmp=yPos + rfbHextileExtractY(ht->data.rects.rects0[srect].xy)+j;
		      // guint tmp2=xPos + rfbHextileExtractX(ht->data.rects.rects0[srect].xy);
		      // for (i = 0; i < rfbHextileExtractW(ht->data.rects.rects0[srect].wh); i++)
		      // {
		      //   gdk_image_put_pixel(gdkimage, tmp2++, tmp, pixel);
		      // }
		    }
		  }
		  break;
		}
	      }
	      else
	      {
		// rotated display
		switch (wtype)
		{
		case 0:
		  // no subrect color, 8-bit pixel
		  {
		    pixel = ht->data.rects.fgcolour;

		    guint tmp =
		      ((xPos +
			rfbHextileExtractX(ht->data.rects.rects0[srect].xy)) *
		       rHeight) +
		      (rHeight - yPos -
		       rfbHextileExtractH(ht->data.rects.rects0[srect].wh) -
		       rfbHextileExtractY(ht->data.rects.rects0[srect].xy));

		    for (j = 0; j < rfbHextileExtractW(ht->data.rects.rects0[srect].wh); j++)
		    {
		      memset(&buffer8[tmp], pixel,
			     rfbHextileExtractH(ht->data.rects.rects0[srect].wh));
		      tmp+=rHeight;
		    }
		  }
		  break;
		case 1:
		  {
		    //g_message("case1");
		    // no subrect color, 16-bit pixel
		    pixel = ht->data.rects.fgcolour;

		    guint16 *tmpimg = (guint16 *)
		      (&((guint8 *) gdkimage->mem)
		      [((xPos +
			 rfbHextileExtractX(ht->data.rects.rects0[srect].xy)) *
			gdkimage->bpl) +
		       ((rHeight - yPos -
			 rfbHextileExtractH(ht->data.rects.rects0[srect].wh) -
			 rfbHextileExtractY(ht->data.rects.rects0[srect].xy)) *
			 2)]);

		    for (j = 0; j < rfbHextileExtractW(ht->data.rects.rects0[srect].wh); j++)
		    {
		      //	if (yPos>750)
		      //   g_message("j %d",j);
		      guint8 *tmp = (guint8 *) tmpimg;
		      for (i = 0; i < rfbHextileExtractH(ht->data.rects.rects0[srect].wh); i++)
		      {
			*tmpimg=pixel;
			*tmpimg++;
		      }

		      tmpimg = (guint16 *) (tmp + gdkimage->bpl);
		    }
		  }
		  break;
		case 2:
		  // subrect color, 8-bit pixel
		  {
		    pixel=ht->data.rects.rects1[srect].colour;

		    guint tmp =
		      ((xPos +
			rfbHextileExtractX(ht->data.rects.rects1[srect].xy)) *
		       rHeight) +
		      (rHeight - yPos -
		       rfbHextileExtractH(ht->data.rects.rects1[srect].wh) -
		       rfbHextileExtractY(ht->data.rects.rects1[srect].xy));

		    for (j = 0; j < rfbHextileExtractW(ht->data.rects.rects1[srect].wh); j++)
		    {
		      memset(&buffer8[tmp], 
			     pixel,rfbHextileExtractH(ht->data.rects.rects1[srect].wh));
		      tmp+=rHeight;
		    }
		  }
		  break;
		case 3:
		  {
		    //g_message("case3");
		    // subrect color, 16-bit pixel
		    pixel=ht->data.rects.rects2[srect].colour;

		    guint16 *tmpimg = (guint16 *)
		      (&((guint8 *) gdkimage->mem)
		      [((xPos +
			 rfbHextileExtractX(ht->data.rects.rects2[srect].xy)) *
			gdkimage->bpl) +
		       ((rHeight - yPos -
			 rfbHextileExtractH(ht->data.rects.rects2[srect].wh) -
			 rfbHextileExtractY(ht->data.rects.rects2[srect].xy)) *
			 2)]);

		    //g_message("initial tmpimg values are: xPos=%u, x2=%u, rHeight=%u, yPos=%u, h2=%u, y2=%u, w2=%u, rWidth=%u, w1=%u, h1=%u, srect=%u, ht=%p.\n", xPos, rfbHextileExtractX(ht->data.rects.rects2[srect].xy), rHeight, yPos, rfbHextileExtractH(ht->data.rects.rects2[srect].wh), rfbHextileExtractY(ht->data.rects.rects2[srect].xy), rfbHextileExtractW(ht->data.rects.rects2[srect].wh), rWidth, ht->width, ht->height, srect, ht);

		    for (j = 0; j < rfbHextileExtractW(ht->data.rects.rects2[srect].wh); j++)
		    {
		      // if (yPos>750)
		      //   g_message("yPos %d Y %d j %d",yPos,j);
		      guint8 *tmp = (guint8 *) tmpimg;
		      for (i = 0; i < rfbHextileExtractH(ht->data.rects.rects2[srect].wh); i++)
		      {
			*tmpimg = pixel;
			*tmpimg++;
		      }
		      tmpimg=(guint16*) (tmp+gdkimage->bpl);
		    }
		  }
		  break;
		}
	      }
	    }
	  }
	}

	xPos += ht->width;
	if (xPos >= desc->header.r.w)
	{
	  xPos = 0;
	  yPos += ht->height;
	}
	// g_message("xpos %d ypos %d",xPos,yPos);
      }

      if (bpp == 8)
      {
	if (!vnc->m_bRotateDisplay)
	{
	  gdk_draw_indexed_image(vnc->offscreen, vnc->gc,
				 desc->header.r.x, desc->header.r.y,
				 rWidth, rHeight,
				 GDK_RGB_DITHER_NONE, buffer8, rWidth, cmap);
	}
	else
	{
	  gdk_draw_indexed_image(vnc->offscreen, vnc->gc,
				 vnc->height - desc->header.r.y - rHeight,
				 desc->header.r.x,
				 rHeight, rWidth,
				 GDK_RGB_DITHER_NONE, buffer8, rHeight, cmap);
	}
	g_free(buffer8);
	buffer8=NULL;
      }
      else
      {
	if (!vnc->m_bRotateDisplay)
	{
	  gdk_draw_image(vnc->offscreen, vnc->gc, gdkimage, 0, 0,
			 desc->header.r.x, desc->header.r.y,
			 rWidth, rHeight);
	}
	else
	{
	  gdk_draw_image(vnc->offscreen, vnc->gc, gdkimage, 0, 0,
			 vnc->height - desc->header.r.y - rHeight,
			 desc->header.r.x,
			 rHeight, rWidth);
	}
	g_object_unref(gdkimage);
      }
      break;

    default:
      g_warning("Unknown rectangle encoding returned");
    }

    // g_message("drawpix");
    // draw the drawable to the window (I guess)
    if (!vnc->m_bRotateDisplay)
    {
      gdk_draw_pixmap(GTK_WIDGET(vnc)->window,
		      vnc->gc,
		      vnc->offscreen,
		      desc->header.r.x, desc->header.r.y,
		      desc->header.r.x,
		      desc->header.r.y,
		      rWidth, rHeight);
    }
    else
    {
      gdk_draw_pixmap(GTK_WIDGET(vnc)->window,
		      vnc->gc,
		      vnc->offscreen,
		      vnc->height - desc->header.r.y - rHeight,
		      desc->header.r.x,
		      vnc->height - desc->header.r.y - rHeight,
		      desc->header.r.x,
		      rHeight, rWidth);
    }
    //g_message("freeRect");
    freeRectangleDescription(desc);
    // g_free(desc); 
#ifdef perf
    T2+=g_timer_elapsed(Tperf,NULL);
#endif
    // g_timer_destroy(T);
  }

#ifdef perf
  frames++;
  if (ReadBytes>1000000)
  {
    Tcount=0;
    T3+=g_timer_elapsed(Tperf,NULL);
    g_message("%ld time T1 %f T2-T1 %f %f %f frames %d time measured %f Datarate %f",ReadBytes,
	      T1,T2-T1,T1*1000000/ReadBytes,(T2-T1)*1000000/ReadBytes,frames,T3,ReadBytes/T3);
    T1=0; T2=0; T3=0;ReadBytes=0; frames=0;
  }
#endif
  gtk_timeout_add(25, (GtkFunction) vnc_viewer_send_draw_request, vnc);
}

static void vnc_viewer_set_colourmap_entries(VncViewer *vnc,
					     rfbServerToClientMsg *msg,
					     guint16 *entries)
{
  g_message("colormap called %d %d",msg->scme.firstColour,msg->scme.nColours);
  int n=msg->scme.firstColour;
  int j;
  for (j=0;j<msg->scme.nColours;j++)
  {
    cmap->colors[n+j]=(((guint32)entries[3*j]&0xFF00)<<8) | 
      ((guint32)entries[3*j+1]&0xFF00) |
      ((guint32)entries[3*j+2]>>8);
    //g_message("c %d - %lX",n+j,cmap->colors[n+j]);
  }
}	

static void vnc_viewer_bell(VncViewer *vnc, rfbServerToClientMsg *msg)
{
  gdk_beep();
}

static void vnc_viewer_server_cut_text(VncViewer *vnc,
				       rfbServerToClientMsg *msg, gchar *text)
{
  // I am sure I should be doing something with a selection here...
}

static void vnc_viewer_realize(GtkWidget *widget)
{
  VncViewer *vnc = VNC_VIEWER(widget);
  GdkWindowAttr attributes;
  gint attributes_mask;

  //GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual(widget);
  attributes.colormap = gtk_widget_get_colormap(widget);
  attributes.event_mask = gtk_widget_get_events(widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_POINTER_MOTION_MASK |
			    GDK_BUTTON_MOTION_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK |
			    GDK_KEY_PRESS_MASK |
			    GDK_KEY_RELEASE_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  if (vnc->cursor_type == VNC_VIEWER_CURSOR_DOT)
  {
    attributes.cursor = vnc->dot;
    attributes_mask |= GDK_WA_CURSOR;
  } 
  else if (vnc->cursor_type == VNC_VIEWER_CURSOR_NONE)
  {
    attributes.cursor = vnc->none;
    attributes_mask |= GDK_WA_CURSOR;
  }

  widget->window = gdk_window_new(gtk_widget_get_parent_window(widget),
				  &attributes, attributes_mask);
  widget->style = gtk_style_attach(widget->style, widget->window);
  gtk_style_set_background(widget->style, widget->window, GTK_STATE_NORMAL);

  if (vnc->offscreen)
    gdk_pixmap_unref(vnc->offscreen);
  if (!vnc->m_bRotateDisplay)
  {
    vnc->offscreen = gdk_pixmap_new(widget->window,
				    vnc->width,
				    vnc->height,
				    -1);
  }
  else
  {
    vnc->offscreen = gdk_pixmap_new(widget->window,
				    vnc->height,
				    vnc->width,
				    -1);
  }
  g_message("drawable depth %d",gdk_drawable_get_depth(widget->window));

  if (vnc->gc)
    gdk_gc_unref(vnc->gc);
  vnc->gc = gdk_gc_new(widget->window);
  gdk_gc_copy(vnc->gc, GTK_WIDGET(vnc)->style->fg_gc[GTK_WIDGET_STATE(vnc)]);

  gdk_window_set_user_data(widget->window, widget);
  gdk_window_show(widget->window);
  GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);

  // associate the input method with this window
  gtk_im_context_set_client_window(vnc->im_context, widget->window);

  // need to connect to focus-in and focus-out events to setup input method
  // properly
  g_signal_connect(G_OBJECT(vnc), "focus-in-event",
		   G_CALLBACK(vnc_viewer_focus), vnc->im_context);

  g_signal_connect(G_OBJECT(vnc), "focus-out-event",
		   G_CALLBACK(vnc_viewer_focus), vnc->im_context);

  g_message("Sending request for full update");
  vnc_viewer_queue_event(vnc, newFramebufferUpdateRequestMsg(TRUE, 0, 0,
							     vnc->width,
							     vnc->height));
}

static void vnc_viewer_unrealize(GtkWidget *widget)
{
  VncViewer *vnc = VNC_VIEWER(widget);

  // disassociate input method from this window
  gtk_im_context_set_client_window(vnc->im_context, 0);

  if (GTK_WIDGET_CLASS(parent_class)->unrealize)
    (*GTK_WIDGET_CLASS(parent_class)->unrealize)(widget);
}

static gint vnc_viewer_expose_event(GtkWidget *widget, GdkEventExpose *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);

  if (GTK_WIDGET_DRAWABLE(widget))
  {
    // gdk_draw_pixmap is deprecated, should be replaced
    if (!vnc->m_bRotateDisplay)
    {
      gdk_draw_pixmap(widget->window,
		      vnc->gc,
		      vnc->offscreen,
		      event->area.x,
		      event->area.y,
		      event->area.x,     event->area.y,
		      event->area.width, event->area.height);
    }
    else
    {
      gdk_draw_pixmap(widget->window,
		      vnc->gc,
		      vnc->offscreen,
		      event->area.x,
		      event->area.y,
		      event->area.x,     event->area.y,
		      event->area.width, event->area.height);
    }
  }

  return FALSE;
}

static gint vnc_viewer_button_press_event(GtkWidget *widget,
					  GdkEventButton *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);
  guint16 x, y;

  if (vnc->fd < 0)
    return FALSE;

  switch (event->button)
  {
  case 1:
    vnc->button_mask |= rfbButton1Mask;
    break;

  case 2:
    vnc->button_mask |= rfbButton2Mask;
    break;

  case 3:
    vnc->button_mask |= rfbButton3Mask;
    break;

  default:
    break;
  }

  if (!vnc->m_bRotateDisplay)
  {
    x = (guint16) event->x;
    y = (guint16) event->y;
  }
  else
  {
    x = (guint16) event->y;
    y = (guint16) vnc->height - event->x;
  }
  vnc_viewer_queue_event(vnc, newPointerEventMsg(vnc->button_mask, x, y));
  return TRUE;
}

static gint vnc_viewer_button_release_event(GtkWidget *widget,
					    GdkEventButton *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);
  guint16 x, y;

  if (vnc->fd < 0)
    return FALSE;

  switch (event->button)
  {
  case 1:
    vnc->button_mask &= ~rfbButton1Mask;
    break;

  case 2:
    vnc->button_mask &= ~rfbButton2Mask;
    break;

  case 3:
    vnc->button_mask &= ~rfbButton3Mask;
    break;

  default:
    break;
  }

  if (!vnc->m_bRotateDisplay)
  {
    x = (guint16) event->x;
    y = (guint16) event->y;
  }
  else
  {
    x = (guint16) event->y;
    y = (guint16) vnc->height - event->x;
  }
  vnc_viewer_queue_event(vnc, newPointerEventMsg(vnc->button_mask, x, y));
  return TRUE;
}

static gint vnc_viewer_motion_notify_event(GtkWidget *widget,
					   GdkEventMotion *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);

  if (vnc->fd < 0)
    return FALSE;

  if (!vnc->m_bRotateDisplay)
  {
    vnc->last_pointer_x_loc = (guint16) event->x;
    vnc->last_pointer_y_loc = (guint16) event->y;
  }
  else
  {
    vnc->last_pointer_x_loc = (guint16) event->y;
    vnc->last_pointer_y_loc = (guint16) vnc->height - event->x;
  }
  vnc_viewer_queue_event(vnc, newPointerEventMsg(vnc->button_mask,
						 vnc->last_pointer_x_loc,
						 vnc->last_pointer_y_loc));

  return TRUE;
}

gint vnc_viewer_key_press_event(GtkWidget *widget,
				GdkEventKey *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);

  if (vnc->fd < 0)
    return FALSE;

  if (arg_lockdown)
  {
    if (vnc->m_lockdownKeyPressKeyVal != event->keyval)
    {
      vnc->m_lockdownKeyPressKeyVal = event->keyval;
      g_get_current_time(&vnc->m_lockdownKeyPressTime);
      g_get_current_time(&vnc->m_lockdownLastKeyPressTime);
    }
    else
    {
      // if the difference in time between this key press and the
      // last key press is a second or greater, then assume this is
      // a separate key press that corresponds to a legitimate key
      // release
      GTimeVal currentTime;
      g_get_current_time(&currentTime);
      glong nSecDiff =
	currentTime.tv_sec - (vnc->m_lockdownLastKeyPressTime).tv_sec;
      glong nMicroSecDiff =
	currentTime.tv_usec - (vnc->m_lockdownLastKeyPressTime).tv_usec;
      if ((nSecDiff > 1) || ((nSecDiff == 1) && (nMicroSecDiff >= 0)))
	vnc->m_lockdownKeyPressTime = currentTime;

      vnc->m_lockdownLastKeyPressTime = currentTime;
    }

    return TRUE;
  }

  if (GTK_WIDGET_REALIZED(widget) &&
      gtk_im_context_filter_keypress(vnc->im_context, event))
    return TRUE;

  if (event->state == 0)
  {
    guint8 button_mask = 0;
    if (event->keyval == GDK_F8)
      button_mask |= rfbButton2Mask;
    else if (event->keyval == GDK_F7)
      button_mask |= rfbButton3Mask;

    if (button_mask)
    {
      // simulate middle or right mouse button click
      vnc_viewer_queue_event(vnc, newPointerEventMsg(button_mask,
						     vnc->last_pointer_x_loc,
						     vnc->last_pointer_y_loc));
      vnc_viewer_queue_event(vnc, newPointerEventMsg(0,
						     vnc->last_pointer_x_loc,
						     vnc->last_pointer_y_loc));

      return TRUE;
    }
  }

  guint32 keyval = event->keyval;
  if (vnc->m_bRotateDisplay)
  {
    switch (keyval)
    {
    case GDK_Up:
      keyval = GDK_Left;
      break;

    case GDK_Down:
      keyval = GDK_Right;
      break;

    case GDK_Left:
      keyval = GDK_Down;
      break;

    case GDK_Right:
      keyval = GDK_Up;
      break;

    default:
      break;
    }
  }

  vnc_viewer_queue_event(vnc, newKeyEventMsg(TRUE, keyval));

  return TRUE;
}

gint vnc_viewer_key_release_event(GtkWidget *widget,
				  GdkEventKey *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);

  if (vnc->fd < 0)
    return FALSE;

  if (arg_lockdown)
  {
    // if the key was held down for at least 5 seconds and it was the last
    // key pressed, then display password dialog allowing user to exit lock
    // down mode
    if (vnc->m_lockdownKeyPressKeyVal == event->keyval)
    {
      GTimeVal currentTime;
      g_get_current_time(&currentTime);
      glong nSecDiff =
	currentTime.tv_sec - (vnc->m_lockdownKeyPressTime).tv_sec;
      glong nMicroSecDiff =
	currentTime.tv_usec - (vnc->m_lockdownKeyPressTime).tv_usec;
      if ((nSecDiff > 2) || ((nSecDiff == 2) && (nMicroSecDiff >= 0)))
      {
	if (arg_lockdown_password && strlen(arg_lockdown_password))
	{
	  GtkWidget *dialog = hildon_get_password_dialog_new(0, FALSE);
	  GtkWidget *reconnectButton =
	    gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CONNECT, 1);
	  gtk_button_set_label(GTK_BUTTON(reconnectButton), "Reconnect");
	  
	  gtk_widget_show(dialog);
	  gint nResponse = gtk_dialog_run(GTK_DIALOG(dialog));
	  if (nResponse == GTK_RESPONSE_OK)
	  {
	    HildonGetPasswordDialog *passDialog = 
	      HILDON_GET_PASSWORD_DIALOG(dialog);
	    if (strcmp(arg_lockdown_password,
		       hildon_get_password_dialog_get_password(passDialog)) == 0)
	    {
	      arg_lockdown = FALSE;
	      if (vnc->mainview)
	      {
		create_menu(vnc->mainview);
		disable_home_key_lock(vnc->mainview);
	      }
	    }
	    else
	    {
	      GtkWidget *msgDialog =
		hildon_note_new_information(0, "Password incorrect");
	      gtk_dialog_run(GTK_DIALOG(msgDialog));
	      gtk_widget_destroy(msgDialog);

	      vnc->m_lockdownKeyPressKeyVal = 0;
	    }
	  }
	  else if (nResponse == 1)
	  {
	    gtk_widget_destroy(dialog);
	    dialog = 0;

	    // cause a reconnect
	    vnc_viewer_close_connection(vnc);
	    g_signal_emit(G_OBJECT(vnc), vnc_signals[CONNECTION_DIED], 0);
	  }
	  else
	    vnc->m_lockdownKeyPressKeyVal = 0;

	  if (dialog)
	    gtk_widget_destroy(dialog);
	}
	else
	{
	  arg_lockdown = FALSE;
	  if (vnc->mainview)
	  {
	    create_menu(vnc->mainview);
	    disable_home_key_lock(vnc->mainview);
	  }
	}
      }
    }

    return TRUE;
  }

  if (GTK_WIDGET_REALIZED(widget) &&
      gtk_im_context_filter_keypress(vnc->im_context, event))
    return TRUE;

  guint32 keyval = event->keyval;
  if (vnc->m_bRotateDisplay)
  {
    switch (keyval)
    {
    case GDK_Up:
      keyval = GDK_Left;
      break;

    case GDK_Down:
      keyval = GDK_Right;
      break;

    case GDK_Left:
      keyval = GDK_Down;
      break;

    case GDK_Right:
      keyval = GDK_Up;
      break;

    default:
      break;
    }
  }

  vnc_viewer_queue_event(vnc, newKeyEventMsg(FALSE, keyval));

  //if (GTK_WIDGET_CLASS(parent_class)->key_press_event(widget, event))
  //  return TRUE;

  return FALSE;
}

void vnc_viewer_key_press(VncViewer *vnc, guint keyval)
{
  if (vnc->fd < 0)
    return;

  vnc_viewer_queue_event(vnc, newKeyEventMsg(TRUE, keyval));
}

void vnc_viewer_key_release(VncViewer *vnc, guint keyval)
{
  if (vnc->fd < 0)
    return;

  vnc_viewer_queue_event(vnc, newKeyEventMsg(FALSE, keyval));
}

static void vnc_viewer_finalize(GObject *object)
{
  VncViewer *vnc;

  g_return_if_fail(object != NULL);
  g_return_if_fail(VNC_IS_VIEWER(object));

  vnc = VNC_VIEWER(object);

  vnc_viewer_close_connection(vnc);

  if (vnc->desk_name) g_free(vnc->desk_name);

  if (vnc->offscreen)
  {
    gdk_pixmap_unref(vnc->offscreen);
    vnc->offscreen = 0;
  }

  gdk_cursor_destroy(vnc->dot);
  gdk_cursor_destroy(vnc->none);

  vnc->dot = 0;
  vnc->none = 0;
  
  (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}

void vnc_viewer_im_commit(GtkIMContext *im_context, gchar *text, gpointer data)
{
  VncViewer *vnc = VNC_VIEWER(data);
  gunichar uChar = 0;
  guint keyval = 0;
  gchar *text2 = text;

  // the string is in UTF-8 format--iterate through the characters
  while (*text2)
  {
    uChar = g_utf8_get_char(text2);
    keyval = gdk_unicode_to_keyval(uChar);
    vnc_viewer_queue_event(vnc, newKeyEventMsg(TRUE, keyval));
    vnc_viewer_queue_event(vnc, newKeyEventMsg(FALSE, keyval));

    text2 = g_utf8_next_char(text2);
  }  
}

/*
void vnc_viewer_im_preedit_changed(GtkIMContext *im_context, gpointer data)
{
  VncViewer *vnc = VNC_VIEWER(data);
  gchar *str;
  gint cursor;

  gtk_im_context_get_preedit_string(im_context, &str, 0, &cursor);

  // todo
  
  g_free(str);
}
*/

gboolean vnc_viewer_focus(GtkWidget *widget,
			  GdkEventFocus *event,
			  GtkIMContext *im_context)
{
  if (event->in)
    gtk_im_context_focus_in(im_context);
  else
    gtk_im_context_focus_out(im_context);

  return FALSE;
}
