import sys
import os
import fcntl
import threading
import traceback


__doc__ = """
Module providing threads dispatcher code.

Usually you'll create a L{CallbackPipe} or L{Dispatcher} object and embed it
in your objects (e.g. you can embed it in your graphical objects if you want
the UI in another thread). Then you can properly annotate the methods you
want to call from one thread but need to be executed on the other.
For that you have two functions that return decorators:

 * send_to_thread_async
 * send_to_thread_sync

After that all the calls to the decorated methods will be executed on the
expected thread.

@note: async handlers (C{ret_handler} and C{err_handler}) only work when you're
using a C{CallbackPipe} object (not a C{Dispatcher} object).
"""


__all__ = (
    "send_to_thread_async",
    "send_to_thread_sync",
    "Dispatcher",
    "CallbackPipe",
    )


_curr_thr = threading.currentThread


def _thr_name():
    return _curr_thr().getName()


def _do_send_to_thread(method, thr_attr, comm_attr, sync=False):
    def wrap(*args, **kargs):
        self = args[0]

        thr_name = getattr(self, thr_attr)
        if _thr_name() == thr_name:
            return method(*args, **kargs)

        comm = getattr(self, comm_attr)

        if sync:
            comp = Completion()
            kargs.update({"comp": comp,})

        comm.append_cb(method, *args, **kargs)

        if sync:
            comp.wait()
            if comp.exc_info is not None:
                raise comp.exc_info[0], comp.exc_info[1], comp.exc_info[2]
            return comp.retval

    return wrap


def send_to_thread_async(thr_attr, comm_attr):
    """Returns a decorator to send methods to be executed asynchronously.

    The caller of the method won't block. Async handlers for dealing with
    errors and return values can be provided with 'err_handler' and
    'ret_handler' keywords when calling the method. If the async handlers
    aren't supplied then return value and any exceptions raised will be
    discarded.

    @param thr_attr: the thread context attribute name in the object
    @param comm_attr: the L{CallbackPipe} or L{Dispatcher} attribute name
                      in the object
    @return: decorator
    """
    def wrap(method):
        return _do_send_to_thread(method, thr_attr, comm_attr)
    return wrap


def send_to_thread_sync(thr_attr, comm_attr):
    """Returns a decorator to send methods to be executed synchronously.

    The caller of the method will block and the return value and
    exceptions will work as usual.

    @param thr_attr: the thread context attribute name in the object
    @param comm_attr: the L{CallbackPipe} or L{Dispatcher} attribute name
                      in the object
    @return: decorator
    """
    def wrap(method):
        return _do_send_to_thread(method, thr_attr, comm_attr, sync=True)
    return wrap


class Completion(object):
    """Completion object.

    This class exists to hold the return value and possibly any
    exception thrown in a synchronous execution of a method (used by
    send_to_thread_sync).
    """
    def __init__(self):
        self.retval = None
        self.exc_info = None
        self._evt = threading.Event()

    def wait(self):
        self._evt.wait()

    def complete(self):
        self._evt.set()


class Dispatcher(object):
    """Dispatcher to send methods to other thread.

    The purpose of this class is to provide a thread-safe way of sending
    callbacks to another thread. You should add callbacks from one thread
    with append_cb method and run the callbacks in the other thread with
    run_cbs method. This code uses a pipe to notify the other thread
    there are callbacks to be executed. Thus, you can use use poll/select
    on the in_fd to know when to call the run_cbs method.
    """
    WAKE_UP_MSG = "1"

    def _lock_and_run(method):
        def wrap(*args, **kargs):
            args[0]._lock.acquire()
            try:
                ret = method(*args, **kargs)
            except:
                import traceback
                traceback.print_exc()
                ret = None
            args[0]._lock.release()

            return ret

        return wrap

    def __init__(self):
        self._cbs = []
        self._lock = threading.Lock()
        self.in_fd, self.out_fd = os.pipe()

        arg = fcntl.fcntl(self.in_fd, fcntl.F_GETFL)
        fcntl.fcntl(self.in_fd, fcntl.F_SETFL, arg | os.O_NONBLOCK)

        arg = fcntl.fcntl(self.out_fd, fcntl.F_GETFL)
        fcntl.fcntl(self.out_fd, fcntl.F_SETFL, arg | os.O_NONBLOCK)

    @_lock_and_run
    def _do_append_cb(self, comp, cb, *args, **kargs):
        self._cbs.append((comp, cb, args, kargs))
        return len(self._cbs)

    def append_cb(self, cb, *args, **kargs):
        if not callable(cb):
            raise TypeError("cb must be callable")

        if "err_handler" in kargs or \
           "ret_handler" in kargs:
            raise ValueError("dispatcher object don't support async handlers")

        comp = kargs.pop("comp", None)
        if comp is not None and not isinstance(comp, Completion):
            raise TypeError("comp must be a Completion object")

        length = self._do_append_cb(comp, cb, *args, **kargs)
        if length == 1:
            assert os.write(self.out_fd, self.WAKE_UP_MSG) == 1

    @_lock_and_run
    def _steal_cbs(self):
        assert os.read(self.in_fd, 1) == self.WAKE_UP_MSG
        cbs = self._cbs
        self._cbs = []

        return cbs

    def run_cbs(self):
        cbs = self._steal_cbs()
        for comp, func, args, kargs in cbs:
            try:
                ret = func(*args, **kargs)
            except:
                if comp is not None:
                    comp.exc_info = sys.exc_info()
                    ret = None
                else:
                    traceback.print_exc()
            if comp is not None:
                comp.retval = ret
                comp.complete()


class CallbackPipe(object):
    """Pipe between two threads to send/receive callbacks.

    @param thr1: thread identification as returned by getName method
                 of Thread object
    @param thr2: thread identification as returned by getName method
                 of Thread object
    @param disp1: dispatcher object for C{thr1} (optional)
    @param disp2: dispatcher object for C{thr2} (optional)

    This class provides a pipe between the two threads: thr1 and thr2.
    Each thread can use poll/select on the in_fd attribute to know when
    to call the run_cbs method to execute all the pending callbacks.
    Callbacks can be added by just calling the append_cb method.

    The in_fd attribute as well as the run_cbs and append_cb methods will
    do the right thing depending on which thread context they were called.
    Thus, one must be sure to set up the poll/select for in_fd in the
    two _different_ threads (thr1 and thr2).
    """
    def __init__(self, thr1, thr2, disp1=None, disp2=None):
        if disp1 is not None and \
           type(disp1) is not Dispatcher:
            raise TypeError("disp1 should be a Dispatcher")
        if disp2 is not None and \
           type(disp2) is not Dispatcher:
            raise TypeError("disp2 should be a Dispatcher")

        self._disps = { thr1: disp1 or Dispatcher(),
                        thr2: disp2 or Dispatcher() }

    @staticmethod
    def _raise_unexpected_thr():
        raise RuntimeError("unexpected thread: %s" % _thr_name())

    def _this_thr_disp(self):
        try:
            disp = self._disps[_thr_name()]
        except KeyError:
            self._raise_unexpected_thr()
            disp = None
        return disp

    def _other_thr_disp(self):
        name = _thr_name()
        keys = self._disps.keys()

        if not name in keys:
            self._raise_unexpected_thr()
            return None

        if name != keys[0]:
            return self._disps[keys[0]]
        return self._disps[keys[1]]

    def _get_in_fd(self):
        disp = self._this_thr_disp()
        return disp.in_fd

    in_fd = property(_get_in_fd)

    def run_cbs(self):
        disp = self._this_thr_disp()
        return disp.run_cbs()

    def append_cb(self, cb, *args, **kargs):
        if not callable(cb):
            raise TypeError("cb must be callable")

        err_handler = kargs.pop("err_handler", None)
        ret_handler = kargs.pop("ret_handler", None)

        if err_handler is not None and ret_handler is None:
            raise ValueError("must provide both err_handler and ret_handler")
        if ret_handler is not None and err_handler is None:
            raise ValueError("must provide both err_handler and ret_handler")

        if err_handler is not None and not callable(err_handler):
            raise TypeError("err_handler must be callable")
        if ret_handler is not None and not callable(ret_handler):
            raise TypeError("ret_handler must be callable")

        sync = kargs.get("comp", None) is not None
        async = err_handler is not None and ret_handler is not None

        if sync and async:
            raise ValueError("can't supply both comp and async handlers")

        odisp = self._other_thr_disp()

        if not async:
            return odisp.append_cb(cb, *args, **kargs)

        disp = self._this_thr_disp()

        def async_wrap():
            try:
                ret = cb(*args, **kargs)
            except:
                disp.append_cb(err_handler, sys.exc_info())
                return
            disp.append_cb(ret_handler, ret)

        return odisp.append_cb(async_wrap)
