/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=4 sts=2 et cindent: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the MICROB EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Oleg Romashin <oleg.romashin@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "gtkmozembed_common.h"
#include "common.h"
#include "gmozillapluginlist.h"
#include <unistd.h>
#include <string.h>
#include <glib/gstdio.h>
// The plugins array and the plugins counter
static GMozillaPluginList *gPluginList = NULL;

/* GMozillaPluginList */

static gboolean g_mozilla_create_plugin_folder(void);
static gboolean g_mozilla_plugin_list_add_disabled_plugins(GMozillaPluginList *self);
static gboolean g_mozilla_plugin_get_status (const gchar *name, gboolean *enabled);
static gboolean g_mozilla_plugin_set_status (const gchar *name, gboolean enabled, GtkMozPlugin* plugin);
static gboolean g_mozilla_extension_set_status (gchar *name, gboolean enabled, GtkMozPlugin* plugin);


static void
g_mozilla_plugin_list_destroy (GMozillaPluginList *self)
{
    TRACE_LOG();
    GList * ptr = (GList*)self->items;
    while(ptr)
    {
        GtkMozPlugin *pl = (GtkMozPlugin*) ptr->data;
        ptr = ptr->next;
        if (!pl)
            continue;
        EAL_IF_GFREE(pl->path);
        EAL_IF_GFREE(pl->creator);
        EAL_IF_GFREE(pl->type);
        EAL_IF_GFREE(pl->title);
        g_free(pl);
    }
    EAL_IF_GFREE_FUNC(self->items, g_list_free);
    self->num_items = 0;
    g_object_unref(self);
    gPluginList = NULL;
}

static GObject*
g_mozilla_plugin_list_get_items (GMozillaPluginList *self)
{
    TRACE_LOG();
    return G_OBJECT(G_MOZILLA_PLUGIN_LIST(self)->items);
}

static gint
g_mozilla_plugin_list_get_items_count (GMozillaPluginList *self)
{
    TRACE_LOG();
    return G_MOZILLA_PLUGIN_LIST(self)->num_items;
}

static gboolean
g_mozilla_plugin_list_position_ok (GMozillaPluginList *self, gint pos)
{
    TRACE_LOG();
    return (G_MOZILLA_PLUGIN_LIST(self)->items &&
            pos >= 0 &&
            pos < G_MOZILLA_PLUGIN_LIST(self)->num_items);
}

static const gchar*
g_mozilla_plugin_list_get_item_title (GMozillaPluginList *self, gint pos)
{
    TRACE_LOG();
    if (g_mozilla_plugin_list_position_ok(self, pos))
    {
        GtkMozPlugin *pl = g_list_nth_data((GList*)G_MOZILLA_PLUGIN_LIST(self)->items, pos);
        if (pl)
            return pl->title;
    }
    return NULL;
}

static const gchar*
g_mozilla_plugin_list_get_item_creator (GMozillaPluginList *self, gint pos)
{
    TRACE_LOG();
    if (g_mozilla_plugin_list_position_ok(self, pos))
    {
        GtkMozPlugin *pl = g_list_nth_data((GList*)G_MOZILLA_PLUGIN_LIST(self)->items, pos);
        if (pl)
            return pl->creator;
    }
    return NULL;
}

static const gchar*
g_mozilla_plugin_list_get_item_path (GMozillaPluginList *self, gint pos)
{
    TRACE_LOG();
    if (g_mozilla_plugin_list_position_ok(self, pos))
    {
        GtkMozPlugin *pl = g_list_nth_data((GList*)G_MOZILLA_PLUGIN_LIST(self)->items, pos);
        if (pl)
            return pl->path;
    }
    return NULL;
}

static const gchar*
g_mozilla_plugin_list_get_item_type (GMozillaPluginList *self, gint pos)
{
    TRACE_LOG();
    if (g_mozilla_plugin_list_position_ok(self, pos))
    {
        GtkMozPlugin *pl = g_list_nth_data((GList*)G_MOZILLA_PLUGIN_LIST(self)->items, pos);
        if (pl)
            return pl->type;
    }
    return NULL;
}

static void
g_mozilla_plugin_list_interface_init (GWebPluginListIface *iface)
{
    TRACE_LOG();
    iface->destroy         = (void (*) (GWebPluginList*))
                             g_mozilla_plugin_list_destroy;
    iface->get_items       = (GObject* (*) (GWebPluginList*))
                             g_mozilla_plugin_list_get_items;
    iface->get_items_count = (gint (*) (GWebPluginList*))
                             g_mozilla_plugin_list_get_items_count;
    iface->get_item_title  = (const gchar* (*) (GWebPluginList*, gint))
                             g_mozilla_plugin_list_get_item_title;
    iface->get_item_path  = (const gchar* (*) (GWebPluginList*, gint))
                            g_mozilla_plugin_list_get_item_path;
    iface->get_item_creator  = (const gchar* (*) (GWebPluginList*, gint))
                            g_mozilla_plugin_list_get_item_creator;
    iface->get_item_type   = (const gchar* (*) (GWebPluginList*, gint))
                             g_mozilla_plugin_list_get_item_type;
}

static void
g_mozilla_plugin_list_instance_init (GTypeInstance *instance, gpointer  g_class)
{
    TRACE_LOG();
    G_MOZILLA_PLUGIN_LIST(instance)->items     = NULL;
    G_MOZILLA_PLUGIN_LIST(instance)->num_items = 0;
}

GType
g_mozilla_plugin_list_get_type (void)
{
    static GType type = 0;
    if (type == 0)
    {
        static const GTypeInfo info =
            {
                sizeof (GMozillaPluginListClass),
                NULL,   /* base_init */
                NULL,   /* base_finalize */
                NULL,   /* class_init */
                NULL,   /* class_finalize */
                NULL,   /* class_data */
                sizeof (GMozillaPluginList),
                0,      /* n_preallocs */
                g_mozilla_plugin_list_instance_init    /* instance_init */
            };
        static const GInterfaceInfo iface_info =
            {
                (GInterfaceInitFunc) g_mozilla_plugin_list_interface_init,
                NULL,   /* interface_finalize */
                NULL    /* interface_data */
            };
        type = g_type_register_static (G_TYPE_OBJECT,
                                       "GMozillaPluginList",
                                       &info, 0);
        g_type_add_interface_static (type,
                                     G_TYPE_WEB_PLUGIN_LIST,
                                     &iface_info);
    }
    return type;
}

GObject*
g_mozilla_plugin_list_new (void)
{
    TRACE_LOG();

    g_mozilla_create_plugin_folder ();
    GObject* instance = G_OBJECT(g_object_new(g_mozilla_plugin_list_get_type(), NULL));
    if (gPluginList)
        g_mozilla_plugin_list_destroy(gPluginList);
    gPluginList = G_MOZILLA_PLUGIN_LIST(instance);
    gint status = gtk_moz_embed_common_get_plugins_list ((GList**)&G_MOZILLA_PLUGIN_LIST(instance)->items);
    if (status != -1)
        G_MOZILLA_PLUGIN_LIST(instance)->num_items = status;
    g_mozilla_plugin_list_add_disabled_plugins(G_MOZILLA_PLUGIN_LIST(instance));
    return instance;
}

gboolean
g_mozilla_extension_set_status (gchar *name, gboolean enabled, GtkMozPlugin* plugin)
{
   gboolean retval = FALSE;

   if(enabled && plugin->isDisabled)    
       retval = gtk_moz_embed_common_set_extension_status(name, TRUE);
   else if(!enabled && !plugin->isDisabled)    
       retval = gtk_moz_embed_common_set_extension_status(name, FALSE);

   return retval;
}

gboolean
g_mozilla_plugin_get_status (const gchar *name, gboolean *disabled)
{
    gboolean retval = FALSE;
    if (!name || !disabled)
        return retval;

    gchar *dirPluginPath = getenv ("MOZILLA_INTERNAL_PLUGINS_DIR");
    if (!dirPluginPath)
        return retval;
    gchar *disabledFilePath = g_strdup_printf("%s/disabled/%s", dirPluginPath, name);
    gchar *enabledFilePath = g_strdup_printf("%s/%s", dirPluginPath, name);

    if (-1 != g_access(enabledFilePath, R_OK))
    {
        *disabled = FALSE;
        retval = TRUE;
    }
    else if (-1 != g_access(disabledFilePath, R_OK))
    {
        *disabled = TRUE;
        retval = TRUE;
    }

    g_free(disabledFilePath);
    g_free(enabledFilePath);

    return retval;
}

gboolean
g_mozilla_plugin_set_status (const gchar *name, gboolean enabled, GtkMozPlugin* plugin)
{
    gboolean retval = FALSE;
    if (!name)
        return retval;

    gchar *dirPluginPath = getenv ("MOZILLA_INTERNAL_PLUGINS_DIR");
    if (!dirPluginPath)
        return retval;
    gchar *disabledFilePath = g_strdup_printf("%s/disabled/%s", dirPluginPath, name);
    gchar *enabledFilePath = g_strdup_printf("%s/%s", dirPluginPath, name);
    gchar *enabledStFilePath = g_strdup_printf("%s/%s.st", dirPluginPath, name);

    if (enabled && plugin->isDisabled)
    {
        if (g_rename(disabledFilePath, enabledFilePath) == 0)
        {
            g_unlink(enabledStFilePath);
            retval = TRUE;
        }
    }
    else if (!enabled && !plugin->isDisabled)
    {
        if (g_rename(enabledFilePath, disabledFilePath) == 0)
        {
            gchar * buffer = g_strdup_printf("%s|%s", plugin->title, plugin->type);
            if (!g_file_set_contents(enabledStFilePath, buffer, -1, NULL))
            {
                ULOG_ERROR_F("ERROR: Failed to save disabled plugin info\n");
            }
            g_free(buffer);
            retval = TRUE;
        }
    }

    g_free(disabledFilePath);
    g_free(enabledFilePath);
    g_free(enabledStFilePath);

    return retval;
}


// Return the plugin status (enabled/disabled) using the plugin file name
gboolean
g_mozilla_plugin_get_item_status (gchar *name)
{
    TRACE_LOG();
    if (!gPluginList || !gPluginList->items || gPluginList->num_items == 0)
    {
       GObject *pl_list = g_mozilla_plugin_list_new ();
       if (!pl_list) return FALSE;
    }

    gboolean status = FALSE;
    gint counter = 0;
    GList * ptr = (GList*)gPluginList->items;
    while(ptr)
    {
        counter++;
        GtkMozPlugin *pl = (GtkMozPlugin*) ptr->data;
        if (pl->path && 0 == g_ascii_strcasecmp(name, pl->path))
        {
            gboolean tmp_st = TRUE;
            if(pl->isPlugin) {
                if (g_mozilla_plugin_get_status(name, &tmp_st))
                {
                    pl->isDisabled = tmp_st;
                    status = !pl->isDisabled;
                }
            } 
            else { 
                status = !pl->isDisabled;
            }
            break;
        }
        ptr = ptr->next;
    }
    ULOG_DEBUG_F("Status for '%s'=%s\n", name, status ? "TRUE" : "FALSE");
    return status;
}

// Set the plugin status (enable/disable plugins)
gboolean
g_mozilla_plugin_set_item_status (gchar *name, gboolean enabled)
{
    TRACE_LOG();
    if (!gPluginList || !gPluginList->items || gPluginList->num_items == 0)
    {
       GObject *pl_list = g_mozilla_plugin_list_new ();
       if (!pl_list) return FALSE;
    }

    gboolean status = FALSE;
    gint counter = 0;
    GList * ptr = (GList*)gPluginList->items;
    while(ptr)
    {
        counter++;
        GtkMozPlugin *pl = (GtkMozPlugin*) ptr->data;
        if (pl->path && 0 == g_ascii_strcasecmp(name, pl->path))
        {
            if(pl->isPlugin) 
            {
                if (g_mozilla_plugin_set_status (name, enabled, pl))
                {
                    pl->isDisabled = enabled;
                    status = TRUE;
                }
            } 
            else
            {
                if(g_mozilla_extension_set_status(name, enabled, pl))
                    pl->isDisabled = !pl->isDisabled;
            }
            break;
        }
        ptr = ptr->next;
    }
    return status;
}

gboolean
g_mozilla_plugin_disable_plugins (GSList **plugin_list)
{
    if (!gPluginList || !gPluginList->items || gPluginList->num_items == 0)
    {
       GObject *pl_list = g_mozilla_plugin_list_new ();
       if (!pl_list) return FALSE;
    }
    
    gboolean status = FALSE;
    GList * ptr = (GList*)gPluginList->items;
    while(ptr)
    {
        GtkMozPlugin *pl = (GtkMozPlugin*) ptr->data;
        if(!pl->isPlugin)
            continue;
        gboolean tmp_st = TRUE;
        if (g_mozilla_plugin_get_status(pl->path, &tmp_st)) 
        {
            if(!tmp_st) 
            {
                if (g_mozilla_plugin_set_status (pl->path, FALSE, pl))
                {
                    pl->isDisabled = TRUE;
                    *plugin_list = g_slist_append(*plugin_list,  g_strdup(pl->path));
                    status = TRUE;
                }
            }
        }
        ptr = ptr->next;
    }

    return status;
}

gboolean
g_mozilla_plugin_enable_plugins (GSList **plugin_list)
{
    TRACE_LOG();
    if (!gPluginList || !gPluginList->items || gPluginList->num_items == 0)
    {
       GObject *pl_list = g_mozilla_plugin_list_new ();
       if (!pl_list) return FALSE;
    }

    gboolean status = FALSE;
    GList * ptr = (GList*)gPluginList->items;
    while(ptr)
    {
        GtkMozPlugin *pl = (GtkMozPlugin*) ptr->data;
        if(!pl->isPlugin)
            continue;
        GSList *temp = NULL;
        for (temp = *plugin_list; temp; temp = temp->next) 
        { 
            if (pl->path && temp->data && 0 == g_ascii_strcasecmp(temp->data, pl->path)) 
            {
                if (g_mozilla_plugin_set_status (pl->path, TRUE, pl)) 
                {
                    pl->isDisabled = FALSE;
                    status = TRUE;
                }
            }
        }
        ptr = ptr->next;
    }
    g_slist_foreach(*plugin_list, (GFunc) g_free, NULL);
    g_slist_free(*plugin_list);
    *plugin_list = NULL;

    return status;
}

gboolean
g_mozilla_plugin_list_add_disabled_plugins(GMozillaPluginList *self)
{
    const gchar *name = NULL;
    // the plugins path
    gchar *dirPluginPath = getenv ("MOZILLA_INTERNAL_PLUGINS_DIR");
    if (!dirPluginPath)
        return FALSE;
    // get the eal plugins directory path
    GDir *pluginDir = NULL;
    pluginDir = g_dir_open (dirPluginPath, 0, NULL);
    if (!pluginDir)
        return FALSE;
    while ((name = g_dir_read_name (pluginDir)))
    {
        if (g_str_has_suffix(name, ".st"))
        {
            gchar *enabledStFilePath = g_strdup_printf("%s/%s", dirPluginPath, name);
            gchar *plugindata = NULL;
            gsize size = 0;
            if (g_file_get_contents(enabledStFilePath, &plugindata, &size, NULL))
            {
                gchar** strarr = g_strsplit(plugindata, "|", 2);
                if (strarr)
                {
                    GtkMozPlugin * pl = g_new0(GtkMozPlugin, 1);
                    pl->path = g_strndup(name, strlen(name) - 3);
                    pl->title = g_strdup(strarr[0]);
                    pl->type = g_strdup(strarr[1]);
                    pl->isPlugin = TRUE;
                    pl->isDisabled = TRUE;
                    self->items = g_list_append(self->items, pl);
                    self->num_items++;
                    g_strfreev(strarr);
                }
                g_free(plugindata);
            }
            g_free(enabledStFilePath);
        }
    }
    g_dir_close (pluginDir);
    return TRUE;
}

gboolean
g_mozilla_create_plugin_folder(void)
{
    const gchar *name = NULL;
    // the plugins path
    gchar *dirPluginPath = getenv ("MOZILLA_INTERNAL_PLUGINS_DIR");
    // get the eal plugins directory path
    GDir *pluginDir;

    g_unlink(dirPluginPath);

    if (-1 == g_access(dirPluginPath, R_OK))
    {
        g_mkdir_with_parents(dirPluginPath, 0755);
    }
    gchar *dirPluginDisPath = g_strdup_printf("%s/disabled", dirPluginPath);
    g_mkdir(dirPluginDisPath, 0755);
    g_free(dirPluginDisPath);

    pluginDir = g_dir_open (OSSO_PLUGINS, 0, NULL);
    if (!pluginDir)
        return FALSE;
    // create a symbolic link of each plugin file stored in the eal plugin directory
    // inside the mozilla eal plugin directory
    while ((name = g_dir_read_name (pluginDir)))
    {
        gboolean enabled = FALSE;
        if (!g_mozilla_plugin_get_status(name, &enabled))
        {
            gchar *path = g_strdup_printf("%s/%s", OSSO_PLUGINS, name);
            gchar *destinyPath = g_strdup_printf("%s/%s", dirPluginPath, name);
            symlink(path, destinyPath);
            g_free(path);
            g_free(destinyPath);
        }
    }
    g_dir_close (pluginDir);
    return TRUE;
}
