#!/usr/bin/python
# Python Plazes handler (C) 2007 Henri Bergius and Eero af Heurlin
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Based on:
#   PyPlazes 0.4.0, (C) 2005 by Paul 'Enki' Boehm.
#   Improvements by Joe LaPenna
#   Nokia770 adaptation by Verena V. Hafner 06/2006
#
#   http://kybernet.org/Projects/PyPlazes/
#   http://www.plazes.com/

import commands
import getopt
import md5
import sys
import xmlrpclib

####
ARP="cat /proc/net/arp"
ROUTE="/sbin/route -n"
####
    
class PyPlazer:
    
    def __init__(self, configfile):
        #### Connection
        self.devKey = "3bebd1d7024bfc0cfe4b46cb5b7f1e8a"
        self.plazesURL = "http://beta.plazes.com/api/plazer/xmlrpc"
        self.plazesServer = xmlrpclib.Server(self.plazesURL)
        
        self.configfile = configfile
        self.initConfig()
        
    def initConfig(self):
        import ConfigParser
        self.config = ConfigParser.ConfigParser()
        self.config.read(self.configfile)
    
        if (self.config.has_section("account") == False):
            self.config.add_section("account")
        
        if (   self.config.has_option("account", "username") == False
            or self.config.has_option("account", "password") == False):
            stat = self.updateAccount()
            if (stat == False):
                return False

    """ These are the OS-dependent methods """
    def getMac(self):
        """Get a MAC Addr in XX:XX:XX:XX:XX:XX Format from wherever.
           OS DEPENDENT.
           XXX: Graceful Error Handling
        """
        status, allArp = commands.getstatusoutput(ARP)
        if status != 0:
            raise Exception
        allArp = allArp.split('\n')

        status, allRoutes = commands.getstatusoutput(ROUTE)
        if status != 0:
            raise Exception
        allRoutes = allRoutes.split('\n')

        defRoute = None
        for route in allRoutes:
          if route.startswith('0.0.0.0'):
            gwIP = route.split()[1]
            break
    
        for arp in allArp:
            if arp.startswith(gwIP):
                return arp.split()[3]

    def convertMac(self, mac):
        """ Convert a XX:XX:XX:XX:XX:XX MAC to Plazes's weird format. """
        if mac:
            mac = mac.lower()
            return '%s2%s3%s4%s5%s6%s' % (
                mac[0:2], mac[3:5], mac[6:8], mac[9:11], mac[12:14], mac[15:17]
            )
    
    def getLocationID(self):
        """Get and convert MAC into LocationID"""
        return self.convertMac(self.getMac())

    """ These are the UI dependent methods """
    
    def uiMessage(self, message):
        print message
    
    def updateAccount(self):
        self.config.set("account", "username", raw_input('Username: '))
        self.config.set("account", "password", raw_input('Password: '))
        
        file = open(self.configfile, 'w')
        self.config.write(file)
        
        return True
        
    def passwordDigest(self):
        password = self.config.get("account", "password")
        return md5.md5("PLAZES" + password).hexdigest()

    """ Generic XML-RPC Plazes communication handlers """
    def login(self, location):
        # Start a Plazes session
        try:
            session = self.plazesServer.launcher.login(self.config.get("account", "username"), self.passwordDigest(), location, self.devKey)
        except TypeError, e:
            self.uiMessage("Plazes error: Failed to start session. Try again later.")
            return False
        except xmlrpclib.Fault, fault:
            self.uiMessage("Plazes error: %s" % fault.faultString)
            if (fault.faultCode == 801):
                # We have wrong username / password, ask new password and try again
                stat = self.updateAccount()
                if (stat == False):
                    return False
                return self.login(location)
            return False
        except xmlrpclib.ProtocolError, e:
            self.uiMessage("Plazes protocol error: %s %s" % (e.errcode, e.errmsg))
            return False
        except xmlrpclib.Error, e:
            self.uiMessage("Plazes general error: %s" % e)
            
        return session

    def update(self, session, location):
        # Check the Plaze we're on
        try:
            status = self.plazesServer.launcher.update(session, location, self.devKey)
        except xmlrpclib.Fault, fault:
            self.uiMessage("Plazes error: %s" % fault.faultString)
            return False

        #print "Session: " + session           
        #for key in status:
        #    print "%s: %s" % (key, status[key])
        
        return status

    def plazeinfo(self, session, location):
        # Get info on the Plaze we're on
        try:
            plazeinfo = self.plazesServer.launcher.plazeinfo(session, location, self.devKey)
        except xmlrpclib.Fault, fault:
            self.uiMessage("Plazes error: %s" % fault.faultString)
            return False

        #print "Session: " + session           
        #for key in plazeinfo:
        #    print "%s: %s" % (key, plazeinfo[key])
        
        return plazeinfo

def run():    
    import os
    
    configfile = os.path.expanduser('~/.maemoplazes.cfg')
    
    # Instantiate the Plazer
    plazer = PyPlazer(configfile)
    location = plazer.getLocationID()
    
    # Start a new session
    session = plazer.login(location)
    if (session == False):
        return 1
        
    status = plazer.update(session, location)
    # Critical failure
    if (status == False):
        return 1

    # Sanity check our response data
    while (True):
        try:
            test = status["update"]
            test = status["editurl"]
            break
        except KeyError:
            # We're missing some critical keys, reget
            status = plazer.update(session, location)

    # Check if we're in a new Plaze
    if (status["update"] == 2):
        # This is a new plaze
        plazer.uiMessage("Plazes: You have discovered a new Plaze.")
        print "Please copy and paste the following url into your browser: ", str(status["editurl"])
        return 0

    # Get info on the Plaze we're in
    info = plazer.plazeinfo(session, location)
    if (status == False):
        return 1
        
    if (info["name"] == ""):
        # Plazes API burped, try again
        info = plazer.plazeinfo(session, location)
        if (info == False):
            return 1
            
        if (info["name"] == ""):
            plazer.uiMessage("Plazes: Failed to retrieve information about your location.")
            return 1

    # Send system note on the Plaze
    plazer.uiMessage("Plazes: You're in %s (%s, %s)." % (info["name"],  info["city"], info["country"]))

if __name__ == "__main__":
    # We're used as program, use runner return values as exit codes
    sys.exit(run())

