#!/usr/bin/env python2.5
# Maemo Plazes handler (C) 2007 Henri Bergius and Eero af Heurlin
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

"""maemoplazer: a Plazes geopositioning client"""

# PLEASE DO NOT CHANGE FORMAT OF __version__ LINE (setup.py reads this)

__author__    = "Henri Bergius <henri.bergius@iki.fi>"
__version__   = "1.0.0"
__date__      = "2007-02-11"
__copyright__ = "Copyright (c) 2007 %s. All rights reserved." % __author__
__licence__   = "GPL"

import osso
import dbus
import dbus.service
from pyplazer import PyPlazer
from service import MaemoPlazerService
import codecs


class MaemoPlazer(PyPlazer):
    def __init__(self, configfile, osso_context):
        self.osso_sysnote = osso.SystemNote(osso_context)

        # Register the GeoClue backend methods
        session_bus = dbus.SessionBus()
        service = dbus.service.BusName("com.plazes.maemoplazer", session_bus)
        self.plazer_geoclue_backend = MaemoPlazerService(self, service, '/com/plazes/maemoplazer')

        PyPlazer.__init__(self, configfile)

    #def initConfig(self):
    #    import gconf
    #    
    #    self.gconf_client = gconf.client_get_default()
    #    self.gconf_prefix = '/com/plazes/maemoplazer'
    #    #self.gconf_client.add_dir('/com/plazes/maemoplazer', gconf.CLIENT_PRELOAD_NONE)
        
    def uiMessage(self, message):
        self.osso_sysnote.system_note_infoprint(message)

    def saveUserPass(self, username, password):
        #self.gconf_client.set_text(self.gconf_prefix  + "/account/username", dialog.get_name())
        #self.gconf_client.set_text(self.gconf_prefix  + "/account/password", dialog.get_password())
        self.config.set("account", "username", username)
        self.config.set("account", "password", password)

        file = open(self.configfile, 'w')
        return self.config.write(file)

    def updateAccount(self):
        # FIXME won't work in OS2008
        import gtk
        import hildon

        window = hildon.Window()
        window.set_title("Maemo Plazer")
    
        dialog = hildon.NamePasswordDialog(window)
        
        window.show_all()
        
        response = dialog.run()
        dialog.hide()
        
        if response == gtk.RESPONSE_OK:
            self.saveUserPass(dialog.get_name(), dialog.username.get_text(), dialog.password.get_text())
            dialog.destroy()
            window.destroy()
            
            return True
        else:
            dialog.destroy()
            return False
            
    #def getUsername(self):
    #    return self.gconf_client.get_text(self.gconf_prefix  + "/account/username", dialog.get_name())
    
    #def getPassword(self):
    #    return self.gconf_client.get_text(self.gconf_prefix  + "/account/password", dialog.get_name())
            
    def updateLocation(self, plazeinfo):
        status = PyPlazer.updateLocation(self, plazeinfo)
        
        if (status == False):
            return False
            
        # TODO: Send the current_position_changed signal to GeoClue
        self.plazer_geoclue_backend.current_position_changed(self.lat, self.lon)
        return status

def run():
    import os
    configfile = os.path.expanduser('~/.maemoplazes.cfg')

    # Register OSSO DBUS context
    osso_c = osso.Context("maemoplazer", "1.0.0", False)
    
    # Instantiate the Plazer
    plazer = MaemoPlazer(configfile, osso_c)
    location = plazer.getLocationID()

    # Start a new session
    session = plazer.login(location)
    if (session == False):
        return 1
        
    status = plazer.update(session, location)
    # Critical failure
    if (status == False):
        return 1

    # Check if we're in a new Plaze
    if (status["update"] == 2):
        # This is a new plaze
        plazer.uiMessage("Plazes: You have discovered a new Plaze.")

        # Launch browser DBUS service
        bus = dbus.SessionBus()
        proxy_obj = bus.get_object('com.nokia.osso_browser', '/com/nokia/osso_browser')
        dbus_iface = dbus.Interface(proxy_obj, 'com.nokia.osso_browser')
        # Make browser use correct url
        dbus_iface.open_new_window(str(status["editurl"]))
        return 0

    # Get info on the Plaze we're in
    info = plazer.plazeinfo(session, location)
    if (info == False):
        return 1

    # Send system note on the Plaze
    plazer.uiMessage("Plazes: You're in %s (%s, %s)." % (plazer.convert(info["name"]), plazer.convert(info["city"]), plazer.convert(info["country"])))
    
if __name__ == "__main__":
    # If we're used as a program, just connect and open Maemo Browser once
    # we have a response from Plazes.
    import sys
    sys.exit(run())
