/*
 * Copyright (C) 2006, 2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * Default map data provided by http://www.openstreetmap.org/
 *
 * This file is part of Maemo Mapper.
 *
 * Maemo Mapper is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Maemo Mapper is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Maemo Mapper.  If not, see <http://www.gnu.org/licenses/>.
 */

#define _GNU_SOURCE

#include <string.h>
#include <math.h>
#include <gtk/gtk.h>
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/hildon-banner.h>


#include "types.h"
#include "data.h"
#include "defines.h"

#include "cmenu.h"
#include "display.h"
#include "gdk-pixbuf-rotate.h"
#include "menu.h"
#include "path.h"
#include "poi.h"
#include "util.h"


static void
cmenu_show_latlon(gint unitx, gint unity)
{
  gdouble lat, lon;
  gchar buffer[80], tmp1[LL_FMT_LEN], tmp2[LL_FMT_LEN];
  printf("%s()\n", __PRETTY_FUNCTION__);

  unit2latlon(unitx, unity, lat, lon);
  lat_format(lat, tmp1);
  lon_format(lon, tmp2);

  snprintf(buffer, sizeof(buffer),
            "%s: %s\n"
          "%s: %s",
          _("Latitude"), tmp1,
          _("Longitude"), tmp2);

  MACRO_BANNER_SHOW_INFO(_window, buffer);

  vprintf("%s(): return\n", __PRETTY_FUNCTION__);
}

static void
cmenu_clip_latlon(gint unitx, gint unity)
{
    gchar buffer[80];
    gdouble lat, lon;
    printf("%s()\n", __PRETTY_FUNCTION__);

    unit2latlon(unitx, unity, lat, lon);

    snprintf(buffer, sizeof(buffer), "%.06f,%.06f", lat, lon);

    gtk_clipboard_set_text(
            gtk_clipboard_get(GDK_SELECTION_CLIPBOARD), buffer, -1);

    vprintf("%s(): return\n", __PRETTY_FUNCTION__);
}

static void
cmenu_route_to(gint unitx, gint unity)
{
    gchar buffer[80];
    gchar strlat[32];
    gchar strlon[32];
    gdouble lat, lon;
    printf("%s()\n", __PRETTY_FUNCTION__);

    unit2latlon(unitx, unity, lat, lon);

    g_ascii_formatd(strlat, 32, "%.06f", lat);
    g_ascii_formatd(strlon, 32, "%.06f", lon);
    snprintf(buffer, sizeof(buffer), "%s, %s", strlat, strlon);

    route_download(buffer);

    vprintf("%s(): return\n", __PRETTY_FUNCTION__);
}

static void
cmenu_distance_to(gint unitx, gint unity)
{
    gchar buffer[80];
    gdouble lat, lon;
    printf("%s()\n", __PRETTY_FUNCTION__);

    unit2latlon(unitx, unity, lat, lon);

    snprintf(buffer, sizeof(buffer), "%s: %.02f %s", _("Distance"),
            calculate_distance(_gps.lat, _gps.lon, lat, lon)
              * UNITS_CONVERT[_units],
            UNITS_ENUM_TEXT[_units]);
    MACRO_BANNER_SHOW_INFO(_window, buffer);

    vprintf("%s(): return\n", __PRETTY_FUNCTION__);
}

static void
cmenu_add_route(gint unitx, gint unity)
{
    printf("%s()\n", __PRETTY_FUNCTION__);
    MACRO_PATH_INCREMENT_TAIL(_route);
    screen2unit(_cmenu_position_x, _cmenu_position_y,
            _route.tail->unitx, _route.tail->unity);
    route_find_nearest_point();
    map_force_redraw();
    vprintf("%s(): return\n", __PRETTY_FUNCTION__);
}

static gboolean
cmenu_cb_loc_show_latlon(GtkMenuItem *item)
{
    gint unitx, unity;
    gdouble lat, lon;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    unit2latlon(unitx, unity, lat, lon);

    latlon_dialog(lat, lon);

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_loc_route_to(GtkMenuItem *item)
{
    gint unitx, unity;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    cmenu_route_to(unitx, unity);

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_loc_download_poi(GtkMenuItem *item)
{
    gint unitx, unity;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    poi_download_dialog(unitx, unity);

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_loc_browse_poi(GtkMenuItem *item)
{
    gint unitx, unity;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    poi_browse_dialog(unitx, unity);

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_loc_distance_to(GtkMenuItem *item)
{
    gint unitx, unity;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    cmenu_distance_to(unitx, unity);

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_loc_add_route(GtkMenuItem *item)
{
    gint unitx, unity;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    cmenu_add_route(unitx, unity);

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_loc_add_way(GtkMenuItem *item)
{
    gint unitx, unity;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    route_add_way_dialog(unitx, unity);

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_loc_add_poi(GtkMenuItem *item)
{
    gint unitx, unity;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    poi_add_dialog(_window, unitx, unity);

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_loc_set_gps(GtkMenuItem *item)
{
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, _pos.unitx, _pos.unity);
    unit2latlon(_pos.unitx, _pos.unity, _gps.lat, _gps.lon);

    /* Move mark to new location. */
    map_refresh_mark(_center_mode > 0);

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_way_show_latlon(GtkMenuItem *item)
{
    gint unitx, unity;
    WayPoint *way;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if((way = find_nearest_waypoint(unitx, unity)))
        cmenu_show_latlon(way->point->unitx, way->point->unity);
    else
    {
        MACRO_BANNER_SHOW_INFO(_window, _("There are no waypoints."));
    }


    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_way_show_desc(GtkMenuItem *item)
{
    gint unitx, unity;
    WayPoint *way;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if((way = find_nearest_waypoint(unitx, unity)))
    {
        MACRO_BANNER_SHOW_INFO(_window, way->desc);
    }
    else
    {
        MACRO_BANNER_SHOW_INFO(_window, _("There are no waypoints."));
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_way_clip_latlon(GtkMenuItem *item)
{
    gint unitx, unity;
    WayPoint *way;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if((way = find_nearest_waypoint(unitx, unity)))
        cmenu_clip_latlon(way->point->unitx, way->point->unity);
    else
    {
        MACRO_BANNER_SHOW_INFO(_window, _("There are no waypoints."));
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_way_clip_desc(GtkMenuItem *item)
{
    gint unitx, unity;
    WayPoint *way;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if((way = find_nearest_waypoint(unitx, unity)))
        gtk_clipboard_set_text(
                gtk_clipboard_get(GDK_SELECTION_CLIPBOARD), way->desc, -1);
    else
    {
        MACRO_BANNER_SHOW_INFO(_window, _("There are no waypoints."));
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_way_route_to(GtkMenuItem *item)
{
    gint unitx, unity;
    WayPoint *way;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if((way = find_nearest_waypoint(unitx, unity)))
        cmenu_route_to(way->point->unitx, way->point->unity);
    else
    {
        MACRO_BANNER_SHOW_INFO(_window, _("There are no waypoints."));
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_way_distance_to(GtkMenuItem *item)
{
    gint unitx, unity;
    WayPoint *way;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if((way = find_nearest_waypoint(unitx, unity)))
        route_show_distance_to(way->point);
    else
    {
        MACRO_BANNER_SHOW_INFO(_window, _("There are no waypoints."));
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_way_delete(GtkMenuItem *item)
{
    gint unitx, unity;
    WayPoint *way;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if((way = find_nearest_waypoint(unitx, unity)))
    {
        gchar buffer[BUFFER_SIZE];
        GtkWidget *confirm;

        snprintf(buffer, sizeof(buffer), "%s:\n%s\n",
                _("Confirm delete of waypoint"), way->desc);
        confirm = hildon_note_new_confirmation(GTK_WINDOW(_window), buffer);

        if(GTK_RESPONSE_OK == gtk_dialog_run(GTK_DIALOG(confirm)))
        {
            Point *pdel_min, *pdel_max, *pdel_start, *pdel_end;
            gint num_del;

            /* Delete surrounding route data, too. */
            if(way == _route.whead)
                pdel_min = _route.head;
            else
                pdel_min = way[-1].point;

            if(way == _route.wtail)
                pdel_max = _route.tail;
            else
                pdel_max = way[1].point;

            /* Find largest continuous segment around the waypoint, EXCLUDING
             * pdel_min and pdel_max. */
            for(pdel_start = way->point - 1; pdel_start->unity
                    && pdel_start > pdel_min; pdel_start--) { }
            for(pdel_end = way->point + 1; pdel_end->unity
                    && pdel_end < pdel_max; pdel_end++) { }

            /* If pdel_end is set to _route.tail, and if _route.tail is a
             * non-zero point, then delete _route.tail. */
            if(pdel_end == _route.tail && pdel_end->unity)
                pdel_end++; /* delete _route.tail too */
            /* else, if *both* endpoints are zero points, delete one. */
            else if(!pdel_start->unity && !pdel_end->unity)
                pdel_start--;

            /* Delete BETWEEN pdel_start and pdel_end, exclusive. */
            num_del = pdel_end - pdel_start - 1;

            memmove(pdel_start + 1, pdel_end,
                    (_route.tail - pdel_end + 1) * sizeof(Point));
            _route.tail -= num_del;

            /* Remove waypoint and move/adjust subsequent waypoints. */
            g_free(way->desc);
            while(way++ != _route.wtail)
            {
                way[-1] = *way;
                way[-1].point -= num_del;
            }
            _route.wtail--;

            route_find_nearest_point();
            map_force_redraw();
        }
        gtk_widget_destroy(confirm);
    }
    else
    {
        MACRO_BANNER_SHOW_INFO(_window, _("There are no waypoints."));
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_way_add_poi(GtkMenuItem *item)
{
    gint unitx, unity;
    WayPoint *way;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if((way = find_nearest_waypoint(unitx, unity)))
        poi_add_dialog(_window, way->point->unitx, way->point->unity);
    else
    {
        MACRO_BANNER_SHOW_INFO(_window, _("There are no waypoints."));
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_poi_route_to(GtkMenuItem *item)
{
    gint unitx, unity;
    PoiInfo poi;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if(select_poi(unitx, unity, &poi, FALSE)) /* FALSE = not quick */
    {
        gint unitx, unity;
        latlon2unit(poi.lat, poi.lon, unitx, unity);
        cmenu_route_to(unitx, unity);
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_poi_distance_to(GtkMenuItem *item)
{
    gint unitx, unity;
    PoiInfo poi;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if(select_poi(unitx, unity, &poi, FALSE)) /* FALSE = not quick */
    {
        gint unitx, unity;
        latlon2unit(poi.lat, poi.lon, unitx, unity);
        cmenu_distance_to(unitx, unity);
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_poi_add_route(GtkMenuItem *item)
{
    gint unitx, unity;
    PoiInfo poi;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if(select_poi(unitx, unity, &poi, FALSE)) /* FALSE = not quick */
    {
        gint unitx, unity;
        latlon2unit(poi.lat, poi.lon, unitx, unity);
        cmenu_add_route(unitx, unity);
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_poi_add_way(GtkMenuItem *item)
{
    gint unitx, unity;
    PoiInfo poi;
    printf("%s()\n", __PRETTY_FUNCTION__);

    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    if(select_poi(unitx, unity, &poi, FALSE)) /* FALSE = not quick */
    {
        gint unitx, unity;
        latlon2unit(poi.lat, poi.lon, unitx, unity);
        route_add_way_dialog(unitx, unity);
    }

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_poi_edit_poi(GtkMenuItem *item)
{
    PoiInfo poi;
    gint unitx, unity;
    printf("%s()\n", __PRETTY_FUNCTION__);

    memset(&poi, 0, sizeof(poi));
    screen2unit(_cmenu_position_x, _cmenu_position_y, unitx, unity);
    select_poi(unitx, unity, &poi, FALSE); /* FALSE = not quick */
    poi_view_dialog(_window, &poi);
    if(poi.label)
        g_free(poi.label);
    if(poi.desc)
        g_free(poi.desc);
    if(poi.clabel)
        g_free(poi.clabel);

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

static gboolean
cmenu_cb_hide(GtkMenuItem *item)
{
    printf("%s()\n", __PRETTY_FUNCTION__);

    if(_mouse_is_down)
        g_mutex_unlock(_mouse_mutex);
    _mouse_is_down = _mouse_is_dragging = FALSE;

    vprintf("%s(): return TRUE\n", __PRETTY_FUNCTION__);
    return TRUE;
}

void cmenu_init()
{
    /* Create needed handles. */
    GtkMenu *menu;
    GtkWidget *submenu;
    GtkWidget *menu_item;
    printf("%s()\n", __PRETTY_FUNCTION__);

    /* Setup the context menu. */
    menu = GTK_MENU(gtk_menu_new());

    /* Setup the map context menu. */
    gtk_menu_append(menu, menu_item
            = gtk_menu_item_new_with_label(_("Tap Point")));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item),
            submenu = gtk_menu_new());

    /* Setup the map context menu. */
    gtk_menu_append(submenu, _cmenu_loc_show_latlon_item
            = gtk_menu_item_new_with_label(_("Show Lat/Lon")));
    gtk_menu_append(submenu, gtk_separator_menu_item_new());
    gtk_menu_append(submenu, _cmenu_loc_distance_to_item
            = gtk_menu_item_new_with_label(_("Show Distance to")));
    gtk_menu_append(submenu, _cmenu_loc_route_to_item
            = gtk_menu_item_new_with_label(_("Download Route to...")));
    gtk_menu_append(submenu, _cmenu_loc_download_poi_item
                = gtk_menu_item_new_with_label(_("Download POI...")));
    gtk_menu_append(submenu, _cmenu_loc_browse_poi_item
                = gtk_menu_item_new_with_label(_("Browse POI...")));
    gtk_menu_append(submenu, gtk_separator_menu_item_new());
    gtk_menu_append(submenu, _cmenu_loc_add_route_item
                = gtk_menu_item_new_with_label(_("Add Route Point")));
    gtk_menu_append(submenu, _cmenu_loc_add_way_item
                = gtk_menu_item_new_with_label(_("Add Waypoint...")));
    gtk_menu_append(submenu, _cmenu_loc_add_poi_item
                = gtk_menu_item_new_with_label(_("Add POI...")));
    gtk_menu_append(submenu, gtk_separator_menu_item_new());
    gtk_menu_append(submenu, _cmenu_loc_set_gps_item
                = gtk_menu_item_new_with_label(_("Set as GPS Location")));

    /* Setup the waypoint context menu. */
    gtk_menu_append(menu, menu_item
            = gtk_menu_item_new_with_label(_("Waypoint")));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item),
            submenu = gtk_menu_new());

    gtk_menu_append(submenu, _cmenu_way_show_latlon_item
            = gtk_menu_item_new_with_label(_("Show Lat/Lon")));
    gtk_menu_append(submenu, _cmenu_way_show_desc_item
            = gtk_menu_item_new_with_label(_("Show Description")));
    gtk_menu_append(submenu, _cmenu_way_clip_latlon_item
            = gtk_menu_item_new_with_label(_("Copy Lat/Lon")));
    gtk_menu_append(submenu, _cmenu_way_clip_desc_item
            = gtk_menu_item_new_with_label(_("Copy Description")));
    gtk_menu_append(submenu, gtk_separator_menu_item_new());
    gtk_menu_append(submenu, _cmenu_way_distance_to_item
            = gtk_menu_item_new_with_label(_("Show Distance to")));
    gtk_menu_append(submenu, _cmenu_way_route_to_item
            = gtk_menu_item_new_with_label(_("Download Route to...")));
    gtk_menu_append(submenu, _cmenu_way_delete_item
            = gtk_menu_item_new_with_label(_("Delete...")));
    gtk_menu_append(submenu, gtk_separator_menu_item_new());
    gtk_menu_append(submenu, _cmenu_way_add_poi_item
                = gtk_menu_item_new_with_label(_("Add POI...")));
    gtk_menu_append(submenu, gtk_separator_menu_item_new());
    gtk_menu_append(submenu, _cmenu_way_goto_nextway_item
                = gtk_menu_item_new_with_label(_("Go to Next")));

    /* Setup the POI context menu. */
    gtk_menu_append(menu, _cmenu_poi_submenu
            = gtk_menu_item_new_with_label(_("POI")));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(_cmenu_poi_submenu),
            submenu = gtk_menu_new());

    gtk_menu_append(submenu, _cmenu_poi_edit_poi_item
                = gtk_menu_item_new_with_label(_("View/Edit...")));
    gtk_menu_append(submenu, gtk_separator_menu_item_new());
    gtk_menu_append(submenu, _cmenu_poi_distance_to_item
            = gtk_menu_item_new_with_label(_("Show Distance to")));
    gtk_menu_append(submenu, _cmenu_poi_route_to_item
            = gtk_menu_item_new_with_label(_("Download Route to...")));
    gtk_menu_append(submenu, gtk_separator_menu_item_new());
    gtk_menu_append(submenu, _cmenu_poi_add_route_item
                = gtk_menu_item_new_with_label(_("Add Route Point")));
    gtk_menu_append(submenu, _cmenu_poi_add_way_item
                = gtk_menu_item_new_with_label(_("Add Waypoint...")));
    gtk_menu_append(submenu, gtk_separator_menu_item_new());
    gtk_menu_append(submenu, _cmenu_poi_goto_nearpoi_item
                = gtk_menu_item_new_with_label(_("Go to Nearest")));

    /* Connect signals for context menu. */
    g_signal_connect(G_OBJECT(_cmenu_loc_show_latlon_item), "activate",
                      G_CALLBACK(cmenu_cb_loc_show_latlon), NULL);
    g_signal_connect(G_OBJECT(_cmenu_loc_route_to_item), "activate",
                      G_CALLBACK(cmenu_cb_loc_route_to), NULL);
    g_signal_connect(G_OBJECT(_cmenu_loc_download_poi_item), "activate",
                      G_CALLBACK(cmenu_cb_loc_download_poi), NULL);
    g_signal_connect(G_OBJECT(_cmenu_loc_browse_poi_item), "activate",
                      G_CALLBACK(cmenu_cb_loc_browse_poi), NULL);
    g_signal_connect(G_OBJECT(_cmenu_loc_distance_to_item), "activate",
                      G_CALLBACK(cmenu_cb_loc_distance_to), NULL);
    g_signal_connect(G_OBJECT(_cmenu_loc_add_route_item), "activate",
                        G_CALLBACK(cmenu_cb_loc_add_route), NULL);
    g_signal_connect(G_OBJECT(_cmenu_loc_add_way_item), "activate",
                        G_CALLBACK(cmenu_cb_loc_add_way), NULL);
    g_signal_connect(G_OBJECT(_cmenu_loc_add_poi_item), "activate",
                        G_CALLBACK(cmenu_cb_loc_add_poi), NULL);
    g_signal_connect(G_OBJECT(_cmenu_loc_set_gps_item), "activate",
                        G_CALLBACK(cmenu_cb_loc_set_gps), NULL);

    g_signal_connect(G_OBJECT(_cmenu_way_show_latlon_item), "activate",
                      G_CALLBACK(cmenu_cb_way_show_latlon), NULL);
    g_signal_connect(G_OBJECT(_cmenu_way_show_desc_item), "activate",
                      G_CALLBACK(cmenu_cb_way_show_desc), NULL);
    g_signal_connect(G_OBJECT(_cmenu_way_clip_latlon_item), "activate",
                      G_CALLBACK(cmenu_cb_way_clip_latlon), NULL);
    g_signal_connect(G_OBJECT(_cmenu_way_clip_desc_item), "activate",
                      G_CALLBACK(cmenu_cb_way_clip_desc), NULL);
    g_signal_connect(G_OBJECT(_cmenu_way_route_to_item), "activate",
                      G_CALLBACK(cmenu_cb_way_route_to), NULL);
    g_signal_connect(G_OBJECT(_cmenu_way_distance_to_item), "activate",
                      G_CALLBACK(cmenu_cb_way_distance_to), NULL);
    g_signal_connect(G_OBJECT(_cmenu_way_delete_item), "activate",
                      G_CALLBACK(cmenu_cb_way_delete), NULL);
    g_signal_connect(G_OBJECT(_cmenu_way_add_poi_item), "activate",
                        G_CALLBACK(cmenu_cb_way_add_poi), NULL);
    g_signal_connect(G_OBJECT(_cmenu_way_goto_nextway_item), "activate",
                        G_CALLBACK(menu_cb_view_goto_nextway), NULL);

    g_signal_connect(G_OBJECT(_cmenu_poi_edit_poi_item), "activate",
                        G_CALLBACK(cmenu_cb_poi_edit_poi), NULL);
    g_signal_connect(G_OBJECT(_cmenu_poi_route_to_item), "activate",
                      G_CALLBACK(cmenu_cb_poi_route_to), NULL);
    g_signal_connect(G_OBJECT(_cmenu_poi_distance_to_item), "activate",
                      G_CALLBACK(cmenu_cb_poi_distance_to), NULL);
    g_signal_connect(G_OBJECT(_cmenu_poi_add_route_item), "activate",
                        G_CALLBACK(cmenu_cb_poi_add_route), NULL);
    g_signal_connect(G_OBJECT(_cmenu_poi_add_way_item), "activate",
                        G_CALLBACK(cmenu_cb_poi_add_way), NULL);
    g_signal_connect(G_OBJECT(_cmenu_poi_goto_nearpoi_item), "activate",
                        G_CALLBACK(menu_cb_view_goto_nearpoi), NULL);

    gtk_widget_show_all(GTK_WIDGET(menu));

    gtk_widget_tap_and_hold_setup(_map_widget, GTK_WIDGET(menu), NULL, 0);

    /* Add a "hide" signal event handler to handle dismissing the context
     * menu. */
    g_signal_connect(GTK_WIDGET(menu), "hide",
            G_CALLBACK(cmenu_cb_hide), NULL);

    vprintf("%s(): return\n", __PRETTY_FUNCTION__);
}
