/**
 * Copyright (C) 2006 by Koos Vriezen <koos ! vriezen ? xs4all ! nl>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License version 2 as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include <config.h>
#include <string.h>
#include <gtk/gtk.h>

#include "kmplayer_lists.h"
#include "kmplayer.h"
#include "kmplayercontrol.h"

using namespace KMPlayer;

KDE_NO_CDTOR_EXPORT FileDocument::FileDocument (short i, const String &s, KMPlayer::PlayListNotify * n)
 : KMPlayer::Document (s, n), load_tree_version ((unsigned int)-1) {
    id = i;
}

KDE_NO_EXPORT KMPlayer::NodePtr FileDocument::childFromTag(const String &tag) {
    if (tag == String (nodeName ()))
        return this;
    return 0L;
}

void FileDocument::readFromFile (const String & fn) {
    File file (fn);
    debugLog () << "readFromFile " << fn << endl;
    if (file.exists ()) {
        file.open (IO_ReadOnly);
        ByteArray data = file.readAll ();
        TextStream inxml (data);
        KMPlayer::readXML (this, inxml, String(), false);
        normalize ();
        load_tree_version = m_tree_version;
    }
}

void FileDocument::writeToFile (const String & fn) {
    File file (fn);
    debugLog () << "writeToFile " << fn << endl;
    file.open (IO_WriteOnly);
    String xml = outerXML ();
    const char * utf = (const char *) xml;
    file.writeBlock (utf, strlen (utf));
}

KDE_NO_CDTOR_EXPORT Recents::Recents (Application *a)
    : FileDocument (id_node_recent_document, "recents://"),
      app(a) {
    pretty_name = "Most Recent";
}

KDE_NO_EXPORT void Recents::activate () {
    if (!resolved)
        defer ();
}

KDE_NO_EXPORT void Recents::defer () {
    if (!resolved) {
        resolved = true;
        readFromFile (CONTROL(app)->m_config->readEntry
                (GCONF_KEY_RECENT_FILE, "/home/user/recent.xml"));
    }
}

KDE_NO_EXPORT KMPlayer::NodePtr Recents::childFromTag (const String & tag) {
    // kdDebug () << nodeName () << " childFromTag " << tag << endl;
    if (tag == String ("item"))
        return new Recent (m_doc, app);
    else if (tag == String ("group"))
        return new Group (m_doc, app);
    return FileDocument::childFromTag (tag);
}

KDE_NO_EXPORT void Recents::childDone (KMPlayer::NodePtr) {
    finish ();
}

KDE_NO_CDTOR_EXPORT
Recent::Recent (KMPlayer::NodePtr & doc, Application * a, const String &url)
  : KMPlayer::Mrl (doc, id_node_recent_node), app (a) {
    src = url;
    setAttribute (KMPlayer::StringPool::attr_url, url);
}

KDE_NO_EXPORT void Recent::closed () {
    if (src.isEmpty ())
        src = getAttribute (KMPlayer::StringPool::attr_url);
}

KDE_NO_EXPORT void Recent::activate () {
    app_open_location ((const char *) src, app);
}

KDE_NO_CDTOR_EXPORT
Group::Group (KMPlayer::NodePtr & doc, Application * a, const String & pn)
  : KMPlayer::Mrl (doc, KMPlayer::id_node_group_node), app (a) {
    pretty_name = pn;
    if (!pn.isEmpty ())
        setAttribute (KMPlayer::StringPool::attr_title, pn);
}

KDE_NO_EXPORT KMPlayer::NodePtr Group::childFromTag (const String & tag) {
    if (tag == String ("item"))
        return new Recent (m_doc, app);
    else if (tag == String ("group"))
        return new Group (m_doc, app);
    return 0L;
}

KDE_NO_EXPORT void Group::closed () {
    if (pretty_name.isEmpty ())
        pretty_name = getAttribute (KMPlayer::StringPool::attr_title);
}

// gconftool -s -t string /apps/maemo/kmplayer/playlist_file /home/koos/playlist.xml
KDE_NO_EXPORT void Playlist::defer () {
    if (playmode)
        KMPlayer::Document::defer ();
    else if (!resolved) {
        resolved = true;
        readFromFile (CONTROL(app)->m_config->readEntry
                (GCONF_KEY_PLAYLIST_FILE, "/home/user/playlist.xml"));
    }
}

KDE_NO_EXPORT void Playlist::activate () {
    if (playmode)
        KMPlayer::Document::activate ();
    else if (!resolved)
        defer ();
}

KDE_NO_CDTOR_EXPORT Playlist::Playlist (Application *a, KMPlayer::PlayListNotify *n, bool plmode)
    : FileDocument (KMPlayer::id_node_playlist_document, "Playlist://", n),
      app(a),
      playmode (plmode) {
    pretty_name = "Persistent Playlists";
}

KDE_NO_EXPORT KMPlayer::NodePtr Playlist::childFromTag (const String & tag) {
    // kdDebug () << nodeName () << " childFromTag " << tag << endl;
    const char * name = (const char *) tag;
    if (!strcmp (name, "item"))
        return new PlaylistItem (m_doc, app, playmode);
    else if (!strcmp (name, "group"))
        return new PlaylistGroup (m_doc, app, playmode);
    else if (!strcmp (name, "object"))
        return new HtmlObject (m_doc, app, playmode);
    return FileDocument::childFromTag (tag);
}

KDE_NO_EXPORT void Playlist::childDone (KMPlayer::NodePtr c) {
    if (!playmode)
        finish ();
    else
        FileDocument::childDone (c);
}

KDE_NO_CDTOR_EXPORT
PlaylistItemBase::PlaylistItemBase (KMPlayer::NodePtr &d, short i, Application *a, bool pm)
  : KMPlayer::Mrl (d, i), app (a), playmode (pm) {
}

KDE_NO_EXPORT void PlaylistItemBase::activate () {
    if (playmode) {
        Mrl::activate ();
    } else {
        NodePtr pl = new Playlist (app, 0L, true);
        String data;
        String pn;
        if (parentNode ()->id == KMPlayer::id_node_group_node &&
                (previousSibling() || nextSibling ())) {
            data = parentNode ()->innerXML ();
            pn = parentNode ()->mrl ()->pretty_name;
        } else {
            data = outerXML ();
            pn = pretty_name.isEmpty () ? src : pretty_name;
        }
        pl->mrl ()->pretty_name = pn;
        //debugLog () << "cloning to " << data << endl;
        TextStream inxml (data);
        KMPlayer::readXML (pl, inxml, String(), false);
        pl->normalize ();
        NodePtr cur = pl->firstChild ();
        pl->mrl ()->resolved = !!cur;
        if (parentNode ()->id == KMPlayer::id_node_group_node && cur) {
            KMPlayer::NodePtr sister = parentNode ()->firstChild ();
            while (sister && cur && sister.ptr () != this) {
                sister = sister->nextSibling ();
            cur = cur->nextSibling ();
            }
        }
        CONTROL(app)->openDocument (pl, cur);
    }
}

KDE_NO_EXPORT void PlaylistItemBase::closed () {
    if (pretty_name.isEmpty ())
        pretty_name = getAttribute (KMPlayer::StringPool::attr_title);
}

KDE_NO_CDTOR_EXPORT
PlaylistItem::PlaylistItem (KMPlayer::NodePtr & doc, Application * a, bool pm, const String &url)
  : PlaylistItemBase (doc, KMPlayer::id_node_playlist_item, a, pm) {
    src = url;
    setAttribute (KMPlayer::StringPool::attr_url, url);
}

KDE_NO_EXPORT void PlaylistItem::closed () {
    if (src.isEmpty ())
        src = getAttribute (KMPlayer::StringPool::attr_url);
    PlaylistItemBase::closed ();
}

KDE_NO_EXPORT void PlaylistItem::begin () {
    if (playmode && firstChild ())
        firstChild ()->activate ();
    else
        Mrl::begin ();
}

KDE_NO_EXPORT void PlaylistItem::setNodeName (const String & s) {
    src = s;
    setAttribute (KMPlayer::StringPool::attr_url, s);
}

KDE_NO_CDTOR_EXPORT
PlaylistGroup::PlaylistGroup (KMPlayer::NodePtr & doc, Application * a, const String & pn)
  : KMPlayer::Mrl (doc, KMPlayer::id_node_group_node), app (a), playmode (false) {
    pretty_name = pn;
    if (!pn.isEmpty ())
        setAttribute (KMPlayer::StringPool::attr_title, pn);
}

KDE_NO_CDTOR_EXPORT
PlaylistGroup::PlaylistGroup (KMPlayer::NodePtr & doc, Application * a, bool lm)
  : KMPlayer::Mrl (doc, KMPlayer::id_node_group_node), app (a), playmode (lm) {
}

KDE_NO_EXPORT KMPlayer::NodePtr PlaylistGroup::childFromTag (const String & tag) {
    const char * name = (const char *) tag;
    if (!strcmp (name, "item"))
        return new PlaylistItem (m_doc, app, playmode);
    else if (!strcmp (name, "group"))
        return new PlaylistGroup (m_doc, app, playmode);
    else if (!strcmp (name, "object"))
        return new HtmlObject (m_doc, app, playmode);
    return 0L;
}

KDE_NO_EXPORT void PlaylistGroup::closed () {
    if (pretty_name.isEmpty ())
        pretty_name = getAttribute (KMPlayer::StringPool::attr_title);
}

KDE_NO_EXPORT void PlaylistGroup::setNodeName (const String & t) {
    pretty_name = t;
    setAttribute (KMPlayer::StringPool::attr_title, t);
}

KDE_NO_CDTOR_EXPORT
HtmlObject::HtmlObject (KMPlayer::NodePtr &doc, Application *a, bool pm)
  : PlaylistItemBase (doc, KMPlayer::id_node_html_object, a, pm) {}

KDE_NO_EXPORT void HtmlObject::activate () {
    if (playmode)
        KMPlayer::Mrl::activate ();
    else
        PlaylistItemBase::activate ();
}

KDE_NO_EXPORT void HtmlObject::closed () {
    for (Node *n = firstChild ().ptr (); n; n = n->nextSibling ().ptr ()) {
        if (n->id == KMPlayer::id_node_param) {
            KMPlayer::Element *e = static_cast <KMPlayer::Element *> (n);
            String name = e->getAttribute (KMPlayer::StringPool::attr_name);
            if (name == "type")
                mimetype = e->getAttribute (KMPlayer::StringPool::attr_value);
            else if (name == "movie")
                src = e->getAttribute (KMPlayer::StringPool::attr_value);
        } else if (n->id == KMPlayer::id_node_html_embed) {
            KMPlayer::Element *e = static_cast <KMPlayer::Element *> (n);
            String type = e->getAttribute (KMPlayer::StringPool::attr_type);
            if (!type.isEmpty ())
                mimetype = type;
            String asrc = e->getAttribute (KMPlayer::StringPool::attr_src);
            if (!asrc.isEmpty ())
                src = asrc;
        }
    }
    PlaylistItemBase::closed ();
}

KDE_NO_EXPORT KMPlayer::NodePtr HtmlObject::childFromTag (const String & tag) {
    const char * name = (const char *) tag;
    if (!strcasecmp (name, "param"))
        return new DarkNode (m_doc, name, KMPlayer::id_node_param);
    else if (!strcasecmp (name, "embed"))
        return new DarkNode (m_doc, name, id_node_html_embed);
    return NULL;
}

