/*
* Copyright (C) 2007  Koos Vriezen <koos.vriezen@gmail.com>
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*
http://devedge-temp.mozilla.org/library/manuals/2002/plugin/1.0/
*/

#ifdef __ARMEL__

#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>

#include <glib/gprintf.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#define XP_UNIX
#define MOZ_X11
#include "moz-sdk/npupp.h"

#include "npplayer.h"

static GtkWidget *top_widget;
static int top_w, top_h;
static GtkWidget *xembed;
static Window socket_id;
static Window parent_id;
static int update_dimension_timer;

static void *browser_data;
static NSNotify *browser_notify;
static NPPluginFuncs np_funcs;       /* plugin functions              */
static NPP npp;                      /* single instance of the plugin */
static NPWindow np_window;
static NPObject *js_window;
static NPObject *scriptable_peer;
static NPSavedData *saved_data;
static NPClass js_class;
static GTree *stream_list;
static int stream_id_counter;
static GTree *identifiers;
static int js_obj_counter;
typedef struct _StreamInfo {
    NPStream np_stream;
    /*unsigned int stream_buf_pos;*/
    unsigned int stream_pos;
    unsigned int total;
    unsigned int reason;
    char *url;
    char *mimetype;
    char *target;
    bool notify;
    bool called_plugin;
    bool destroyed;
} StreamInfo;
struct JsObject;
typedef struct _JsObject {
    NPObject npobject;
    struct _JsObject * parent;
    char * name;
} JsObject;

/*----------------%<---------------------------------------------------------*/

static void print (const char * format, ...) {
    va_list vl;
    va_start (vl, format);
    vprintf (format, vl);
    va_end (vl);
    fflush (stdout);
}

/*----------------%<---------------------------------------------------------*/

static gint streamCompare (gconstpointer a, gconstpointer b) {
    return (long)a - (long)b;
}

static void freeStream (StreamInfo *si) {
    if (!g_tree_remove (stream_list, si->np_stream.ndata))
        print ("WARNING freeStream not in tree\n");
    g_free (si->url);
    if (si->mimetype)
        g_free (si->mimetype);
    if (si->target)
        g_free (si->target);
    free (si);
}

static gboolean requestStream (void * p) {
    StreamInfo *si = (StreamInfo *) g_tree_lookup (stream_list, p);
    if (si) {
        browser_notify->getUrl (npp, (int)(long)p, si->url, si->target, browser_data);
    } else {
        print ("requestStream %d not found", (long) p);
    }
    return 0; /* single shot */
}

static gboolean destroyStream (void * p) {
    StreamInfo *si = (StreamInfo *) g_tree_lookup (stream_list, p);
    print ("FIXME destroyStream\n");
    if (si)
        browser_notify->finishStream (npp, (int)(long)p, browser_data);
    return 0; /* single shot */
}

static void removeStream (uint32_t stream) {
    StreamInfo *si = (StreamInfo*)g_tree_lookup (stream_list, (gpointer)stream);

    if (si) {
        print ("removeStream %d rec:%d reason:%d\n", stream, si->stream_pos, si->reason);
        if (!si->destroyed) {
            if (si->called_plugin && !si->target)
                np_funcs.destroystream (npp, &si->np_stream, si->reason);
            if (si->notify)
                np_funcs.urlnotify (npp,
                        si->url, si->reason, si->np_stream.notifyData);
        }
        freeStream (si);
    }
}

static int32_t writeStream (uint32_t stream, const char *buf, uint32_t count) {
    int32_t sz = -1;
    StreamInfo *si = (StreamInfo*)g_tree_lookup (stream_list, (gpointer)stream);
    /*print ("writeStream found %d count %d\n", !!si, count);*/
    if (si) {
        if (si->reason > NPERR_NO_ERROR) {
            sz = count; /* stream closed, skip remainings */
        } else {
            if (!si->called_plugin) {
                NPError err;
                uint16 stype = NP_NORMAL;
                err = np_funcs.newstream (npp, si->mimetype ? si->mimetype : "text/plain", &si->np_stream, 0, &stype);
                if (err != NPERR_NO_ERROR) {
                    g_printerr ("newstream error %d\n", err);
                    destroyStream (stream);
                    return count;
                }
                print ("newStream %d type:%d %s mime:%s\n", (long) stream, stype, si->url, si->mimetype ? si->mimetype : "text/plain");
                si->called_plugin = true;
            }
            if (count) /* urls with a target returns zero bytes */
                sz = np_funcs.writeready (npp, &si->np_stream);
            if (sz > 0) {
                sz = np_funcs.write (npp, &si->np_stream, si->stream_pos,
                        (int32_t) count > sz ? sz : (int32_t) count, buf);
                if (sz < 0) /*FIXME plugin destroys stream here*/
                    g_timeout_add (0, destroyStream, (gpointer)stream);
            } else {
                sz = 0;
            }
            si->stream_pos += sz;
        }
    }
    return sz;
}

static StreamInfo *addStream (const char *url, const char *mime, const char *target, void *notify_data, bool notify) {
    StreamInfo *si = (StreamInfo *) malloc (sizeof (StreamInfo));

    memset (si, 0, sizeof (StreamInfo));
    si->url = g_strdup (url);
    si->np_stream.url = si->url;
    if (mime)
        si->mimetype = g_strdup (mime);
    if (target)
        si->target = g_strdup (target);
    si->np_stream.notifyData = notify_data;
    si->notify = notify;
    si->np_stream.ndata = (void *) (long) (stream_id_counter++);
    print ("add stream %d\n", (long) si->np_stream.ndata);
    g_tree_insert (stream_list, si->np_stream.ndata, si);

    g_timeout_add (0, requestStream, si->np_stream.ndata);

    return si;
}

/*----------------%<---------------------------------------------------------*/

static void createJsName (JsObject * obj, char **name, uint32_t * len) {
    int slen = strlen (obj->name);
    if (obj->parent) {
        *len += slen + 1;
        createJsName (obj->parent, name, len);
    } else {
        *name = (char *) malloc (*len + slen + 1);
        *(*name + *len + slen) = 0;
        *len = 0;
    }
    if (obj->parent) {
        *(*name + *len) = '.';
        *len += 1;
    }
    memcpy (*name + *len, obj->name, slen);
    *len += slen;
}

static char *nsVariant2Str (const NPVariant *value) {
    char *str;
    switch (value->type) {
        case NPVariantType_String:
            str = (char *) malloc (value->value.stringValue.utf8length + 3);
            sprintf (str, "'%s'", value->value.stringValue.utf8characters);
            break;
        case NPVariantType_Int32:
            str = (char *) malloc (16);
            snprintf (str, 15, "%d", value->value.intValue);
            break;
        case NPVariantType_Double:
            str = (char *) malloc (64);
            snprintf (str, 63, "%f", value->value.doubleValue);
            break;
        case NPVariantType_Bool:
            str = strdup (value->value.boolValue ? "true" : "false");
            break;
        case NPVariantType_Null:
            str = strdup ("null");
            break;
        case NPVariantType_Object:
            if (&js_class == value->value.objectValue->_class) {
                JsObject *jv = (JsObject *) value->value.objectValue;
                char *val;
                uint32_t vlen = 0;
                createJsName (jv, &val, &vlen);
                str = strdup (val);
                free (val);
                break;
            } /* else fall through */
        default:
            str = strdup ("");
            break;
    }
    return str;
}

/*----------------%<---------------------------------------------------------*/

static NPObject * nsCreateObject (NPP instance, NPClass *aClass) {
    NPObject *obj;
    if (aClass && aClass->allocate)
        obj = aClass->allocate (instance, aClass);
    else
        obj = js_class.allocate (instance, &js_class);/*add null class*/
    /*print ("NPN_CreateObject\n");*/
    obj->referenceCount = 1;
    return obj;
}

static NPObject *nsRetainObject (NPObject *npobj) {
    /*print( "nsRetainObject %p\n", npobj);*/
    npobj->referenceCount++;
    return npobj;
}

static void nsReleaseObject (NPObject *obj) {
    /*print ("NPN_ReleaseObject\n");*/
    if (! (--obj->referenceCount))
        obj->_class->deallocate (obj);
}

static NPError nsGetURL (NPP instance, const char* url, const char* target) {
    (void)instance;
    print ("nsGetURL %s %s\n", url, target);
    addStream (url, 0L, target, 0L, false);
    return NPERR_NO_ERROR;
}

static NPError nsPostURL (NPP instance, const char *url,
        const char *target, uint32 len, const char *buf, NPBool file) {
    (void)instance; (void)len; (void)buf; (void)file;
    print ("nsPostURL %s %s\n", url, target);
    addStream (url, 0L, target, 0L, false);
    return NPERR_NO_ERROR;
}

static NPError nsRequestRead (NPStream *stream, NPByteRange *rangeList) {
    (void)stream; (void)rangeList;
    print ("nsRequestRead\n");
    return NPERR_NO_ERROR;
}

static NPError nsNewStream (NPP instance, NPMIMEType type,
        const char *target, NPStream **stream) {
    (void)instance; (void)type; (void)stream; (void)target;
    print ("nsNewStream\n");
    return NPERR_NO_ERROR;
}

static int32 nsWrite (NPP instance, NPStream* stream, int32 len, void *buf) {
    (void)instance; (void)len; (void)buf; (void)stream;
    print ("nsWrite\n");
    return 0;
}

static NPError nsDestroyStream (NPP instance, NPStream *stream, NPError reason) {
    StreamInfo *si = (StreamInfo *) g_tree_lookup (stream_list, stream->ndata);
    (void)instance;
    print ("nsDestroyStream\n");
    if (si) {
        si->reason = reason;
        si->destroyed = true;
        g_timeout_add (0, destroyStream, stream->ndata);
        return NPERR_NO_ERROR;
    }
    return NPERR_NO_DATA;
}

static void nsStatus (NPP instance, const char* message) {
    (void)instance;
    print ("NPN_Status %s\n", message);
}

static const char* nsUserAgent (NPP instance) {
    (void)instance;
    print ("NPN_UserAgent\n");
    return "Mozilla/4.0 (compatible; MSIE 6.0; X11; Linux armv6l; U) Opera 8.5 [en_US] Tablet browser 0.0.14 RX-34_2007SE_4.2007.26-8";
    /*return "Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.8.1.3) Gecko/20070310 Iceweasel/2.0.0.3 (Debian-2.0.0.3-1)";
    return "";*/
}

static void *nsAlloc (uint32 size) {
    return malloc (size);
}

static void nsMemFree (void* ptr) {
    free (ptr);
}

static uint32 nsMemFlush (uint32 size) {
    (void)size;
    print ("NPN_MemFlush\n");
    return 0;
}

static void nsReloadPlugins (NPBool reloadPages) {
    (void)reloadPages;
    print ("NPN_ReloadPlugins\n");
}

static JRIEnv* nsGetJavaEnv () {
    print ("NPN_GetJavaEnv\n");
    return NULL;
}

static jref nsGetJavaPeer (NPP instance) {
    (void)instance;
    print ("NPN_GetJavaPeer\n");
    return NULL;
}

static NPError nsGetURLNotify (NPP instance, const char* url, const char* target, void *notify) {
    (void)instance;
    print ("NPN_GetURLNotify %s %s\n", url, target);
    addStream (url, 0L, target, notify, true);
    return NPERR_NO_ERROR;
}

static NPError nsPostURLNotify (NPP instance, const char* url, const char* target, uint32 len, const char* buf, NPBool file, void *notify) {
    (void)instance; (void)len; (void)buf; (void)file;
    print ("NPN_PostURLNotify\n");
    addStream (url, 0L, target, notify, true);
    return NPERR_NO_ERROR;
}

static NPError nsGetValue (NPP instance, NPNVariable variable, void *value) {
    print ("NPN_GetValue %d\n", variable & ~NP_ABI_MASK);
    switch (variable) {
        case NPNVxDisplay:
            *(void**)value = (void*)(long) gdk_x11_get_default_xdisplay ();
            break;
        case NPNVxtAppContext:
            *(void**)value = NULL;
            break;
        case NPNVnetscapeWindow:
            print ("NPNVnetscapeWindow\n");
            break;
        case NPNVjavascriptEnabledBool:
            *(int*)value = 1;
            break;
        case NPNVasdEnabledBool:
            *(int*)value = 0;
            break;
        case NPNVisOfflineBool:
            *(int*)value = 0;
            break;
        case NPNVserviceManager:
            *(int*)value = 0;
            break;
        case NPNVToolkit:
            *(int*)value = NPNVGtk2;
            break;
        case NPNVSupportsXEmbedBool:
            *(int*)value = 1;
            break;
        /*case NPNVWindowNPObject:
            if (!js_window) {
                JsObject *jo = (JsObject*) nsCreateObject (instance, &js_class);
                jo->name = g_strdup ("window");
                js_window = (NPObject *) jo;
            }
            *(NPObject**)value = nsRetainObject (js_window);
            break;
        case NPNVPluginElementNPObject: {
            JsObject * obj = (JsObject *) nsCreateObject (instance, &js_class);
            obj->name = g_strdup ("this");
            *(NPObject**)value = (NPObject *) obj;
            break;
        }*/
        default:
            *(int*)value = 0;
            print ("unknown value\n");
    }
    return NPERR_NO_ERROR;
}

static NPError nsSetValue (NPP instance, NPPVariable variable, void *value) {
    /* NPPVpluginWindowBool */
    (void)instance; (void)value;
    print ("NPN_SetValue %d\n", variable & ~NP_ABI_MASK);
    return NPERR_NO_ERROR;
}

static void nsInvalidateRect (NPP instance, NPRect *invalidRect) {
    (void)instance; (void)invalidRect;
    print ("NPN_InvalidateRect\n");
}

static void nsInvalidateRegion (NPP instance, NPRegion invalidRegion) {
    (void)instance; (void)invalidRegion;
    print ("NPN_InvalidateRegion\n");
}

static void nsForceRedraw (NPP instance) {
    (void)instance;
    print ("NPN_ForceRedraw\n");
}

static NPIdentifier nsGetStringIdentifier (const NPUTF8* name) {
    /*print ("NPN_GetStringIdentifier %s\n", name);*/
    gpointer id = g_tree_lookup (identifiers, name);
    if (!id) {
        id = strdup (name);
        g_tree_insert (identifiers, id, id);
    }
    return id;
}

static void nsGetStringIdentifiers (const NPUTF8** names, int32_t nameCount,
        NPIdentifier* ids) {
    (void)names; (void)nameCount; (void)ids;
    print ("NPN_GetStringIdentifiers\n");
}

static NPIdentifier nsGetIntIdentifier (int32_t intid) {
    print ("NPN_GetIntIdentifier %d\n", intid);
    return (NPIdentifier) (long) intid;
}

static bool nsIdentifierIsString (NPIdentifier name) {
    print ("NPN_IdentifierIsString\n");
    return !!g_tree_lookup (identifiers, name);
}

static NPUTF8 * nsUTF8FromIdentifier (NPIdentifier name) {
    char *str = g_tree_lookup (identifiers, name);
    print ("NPN_UTF8FromIdentifier\n");
    if (str)
        return (NPUTF8 *)strdup (str);
    return NULL;
}

static int32_t nsIntFromIdentifier (NPIdentifier identifier) {
    print ("NPN_IntFromIdentifier\n");
    return (int32_t) (long) identifier;
}

static bool nsInvoke (NPP instance, NPObject * npobj, NPIdentifier method,
        const NPVariant *args, uint32_t arg_count, NPVariant *result) {
    (void)instance;
    /*print ("NPN_Invoke %s\n", id);*/
    return npobj->_class->invoke (npobj, method, args, arg_count, result);
}

static bool nsInvokeDefault (NPP instance, NPObject * npobj,
        const NPVariant * args, uint32_t arg_count, NPVariant * result) {
    (void)instance;
    return npobj->_class->invokeDefault (npobj,args, arg_count, result);
}

static bool nsEvaluate (NPP instance, NPObject * npobj, NPString * script,
        NPVariant * result) {
    char * this_var;
    char * this_var_type;
    char * this_var_string;
    char * jsscript;
    (void) npobj; /*FIXME scope, search npobj window*/
    print ("NPN_Evaluate:");

    /* assign to a js variable */
    this_var = (char *) malloc (64);
    sprintf (this_var, "this.__kmplayer__obj_%d", js_obj_counter);

    jsscript = (char *) malloc (strlen (this_var) + script->utf8length + 3);
    sprintf (jsscript, "%s=%s;", this_var, script->utf8characters);
    this_var_string = browser_notify->evaluate (jsscript, browser_data);
    free (jsscript);

    if (this_var_string) {
        /* get type of js this_var */
        jsscript = (char *) malloc (strlen (this_var) + 9);
        sprintf (jsscript, "typeof %s;", this_var);
        this_var_type = browser_notify->evaluate (jsscript, browser_data);
        free (jsscript);

        if (this_var_type) {
            if (!strcasecmp (this_var_type, "undefined")) {
                result->type = NPVariantType_Null;
            } else if (!strcasecmp (this_var_type, "object")) {
                JsObject *jo = (JsObject *)nsCreateObject (instance, &js_class);
                js_obj_counter++;
                result->type = NPVariantType_Object;
                jo->name = g_strdup (this_var);
                result->value.objectValue = (NPObject *)jo;
            } else { /* FIXME numbers/void/undefined*/
                result->type = NPVariantType_String;
                result->value.stringValue.utf8characters =
                    g_strdup (this_var_string);
                result->value.stringValue.utf8length = strlen (this_var_string);
            }
            g_free (this_var_type);
        }
        g_free (this_var_string);
    } else {
        print ("   => error\n");
        return false;
    }
    free (this_var);

    return true;
}

static bool nsGetProperty (NPP instance, NPObject * npobj,
        NPIdentifier property, NPVariant * result) {
    (void)instance;
    return npobj->_class->getProperty (npobj, property, result);
}

static bool nsSetProperty (NPP instance, NPObject * npobj,
        NPIdentifier property, const NPVariant *value) {
    (void)instance;
    return npobj->_class->setProperty (npobj, property, value);
}

static bool nsRemoveProperty (NPP inst, NPObject * npobj, NPIdentifier prop) {
    (void)inst;
    return npobj->_class->removeProperty (npobj, prop);
}

static bool nsHasProperty (NPP instance, NPObject * npobj, NPIdentifier prop) {
    (void)instance;
    return npobj->_class->hasProperty (npobj, prop);
}

static bool nsHasMethod (NPP instance, NPObject * npobj, NPIdentifier method) {
    (void)instance;
    return npobj->_class->hasMethod (npobj, method);
}

static void nsReleaseVariantValue (NPVariant * variant) {
    /*print ("NPN_ReleaseVariantValue\n");*/
    switch (variant->type) {
        case NPVariantType_String:
            if (variant->value.stringValue.utf8characters)
                g_free ((char *) variant->value.stringValue.utf8characters);
            break;
        case NPVariantType_Object:
            if (variant->value.objectValue)
                nsReleaseObject (variant->value.objectValue);
            break;
        default:
            break;
    }
    variant->type = NPVariantType_Null;
}

static void nsSetException (NPObject *npobj, const NPUTF8 *message) {
    (void)npobj;
    print ("NPN_SetException %s\n", message);
}

static bool nsPushPopupsEnabledState (NPP instance, NPBool enabled) {
    (void)instance;
    print ("NPN_PushPopupsEnabledState %d\n", enabled);
    return false;
}

static bool nsPopPopupsEnabledState (NPP instance) {
    (void)instance;
    print ("NPN_PopPopupsEnabledState\n");
    return false;
}

/*----------------%<---------------------------------------------------------*/

static NPObject * windowClassAllocate (NPP instance, NPClass *aClass) {
    /*print ("windowClassAllocate\n");*/
    JsObject * jo = (JsObject *) malloc (sizeof (JsObject));
    (void)instance;
    memset (jo, 0, sizeof (JsObject));
    jo->npobject._class = aClass;
    return (NPObject *) jo;
}

static void windowClassDeallocate (NPObject *npobj) {
    JsObject *jo = (JsObject *) npobj;
    /*print ("windowClassDeallocate\n");*/
    if (jo->parent) {
        nsReleaseObject ((NPObject *) jo->parent);
    } else if (jo->name && !strncmp (jo->name, "this.__kmplayer__obj_", 21)) {
        char *script = (char *) malloc (strlen (jo->name) + 7);
        char *result;
        char *counter = strrchr (jo->name, '_');
        sprintf (script, "%s=null;", jo->name);
        result = browser_notify->evaluate (script, browser_data);
        free (script);
        g_free (result);
        if (counter) {
            int c = strtol (counter +1, NULL, 10);
            if (c == js_obj_counter -1)
                js_obj_counter--; /*poor man's variable name reuse */
        }
    }
    if (jo->name)
        g_free (jo->name);
    if (npobj == js_window) {
        print ("WARNING deleting window object\n");
        js_window = NULL;
    }
    free (npobj);
}

static void windowClassInvalidate (NPObject *npobj) {
    (void)npobj;
    print ("windowClassInvalidate\n");
}

static bool windowClassHasMethod (NPObject *npobj, NPIdentifier name) {
    (void)npobj; (void)name;
    print ("windowClassHasMehtod\n");
    return false;
}

static bool windowClassInvoke (NPObject *npobj, NPIdentifier method,
        const NPVariant *args, uint32_t arg_count, NPVariant *result) {
    JsObject * jo = (JsObject *) npobj;
    NPString str = { NULL, 0 };
    char buf[512];
    int pos, i;
    bool res;
    char * id = (char *) g_tree_lookup (identifiers, method);
    /*print ("windowClassInvoke\n");*/

    result->type = NPVariantType_Null;
    result->value.objectValue = NULL;

    if (!id) {
        print ("Invoke invalid id\n");
        return false;
    }
    print ("Invoke %s\n", id);
    createJsName (jo, (char **)&str.utf8characters, &str.utf8length);
    pos = snprintf (buf, sizeof (buf), "%s.%s(", str.utf8characters, id);
    free ((char *) str.utf8characters);
    for (i = 0; i < arg_count; i++) {
        char *arg = nsVariant2Str (args + i);
        pos += snprintf (buf + pos, sizeof (buf) - pos, i ? ",%s" : "%s", arg);
        free (arg);
    }
    pos += snprintf (buf + pos, sizeof (buf) - pos, ")");

    str.utf8characters = buf;
    str.utf8length = pos;
    res = nsEvaluate (npp, npobj, &str, result);

    return true;
}

static bool windowClassInvokeDefault (NPObject *npobj,
        const NPVariant *args, uint32_t arg_count, NPVariant *result) {
    (void)npobj; (void)args; (void)arg_count; (void)result;
    print ("windowClassInvokeDefault\n");
    return false;
}

static bool windowClassHasProperty (NPObject *npobj, NPIdentifier name) {
    (void)npobj; (void)name;
    print ("windowClassHasProperty\n");
    return false;
}

static bool windowClassGetProperty (NPObject *npobj, NPIdentifier property,
        NPVariant *result) {
    char * id = (char *) g_tree_lookup (identifiers, property);
    JsObject jo;
    NPString fullname = { NULL, 0 };
    bool res;

    print ("GetProperty %s\n", id);
    result->type = NPVariantType_Null;
    result->value.objectValue = NULL;

    if (!id)
        return false;

    if (!strcmp (((JsObject *) npobj)->name, "window") &&
                !strcmp (id, "top")) {
        result->type = NPVariantType_Object;
        result->value.objectValue = nsRetainObject (js_window);
        return true;
    }

    jo.name = id;
    jo.parent = (JsObject *) npobj;
    createJsName (&jo, (char **)&fullname.utf8characters, &fullname.utf8length);

    res = nsEvaluate (npp, npobj, &fullname, result);

    free ((char *) fullname.utf8characters);

    return res;
}

static bool windowClassSetProperty (NPObject *npobj, NPIdentifier property,
        const NPVariant *value) {
    char *id = (char *) g_tree_lookup (identifiers, property);
    char *script, *var_name, *var_val, *res;
    JsObject jo;
    uint32_t len = 0;

    if (!id)
        return false;

    jo.name = id;
    jo.parent = (JsObject *) npobj;
    createJsName (&jo, &var_name, &len);

    var_val = nsVariant2Str (value);
    script = (char *) malloc (len + strlen (var_val) + 3);
    sprintf (script, "%s=%s;", var_name, var_val);
    free (var_name);
    free (var_val);
    print ("SetProperty %s\n", script);

    res = browser_notify->evaluate (script, browser_data);
    if (res)
        g_free (res);
    free (script);


    return true;
}

static bool windowClassRemoveProperty (NPObject *npobj, NPIdentifier name) {
    (void)npobj; (void)name;
    print ("windowClassRemoveProperty\n");
    return false;
}


/*----------------%<---------------------------------------------------------*/

void nppPluginShutdown (NPPluginLib *lib) {
    print ("nppPluginShutdown\n");
    if (lib) {
        if (lib->npShutdown)
            lib->npShutdown();
        g_module_close (lib->module);
        free (lib);
    }
}

NPPluginLib *nppPluginInit (const char *lib) {
    NPNetscapeFuncs ns_funcs;
    NPError np_err;
    NPPluginLib *plugin_lib;
    GModule *library;

    print ("starting %s\n", lib);
    library = g_module_open (lib, G_MODULE_BIND_LAZY);
    if (!library) {
        print ("failed to load\n");
        return NULL;
    }

    plugin_lib = (NPPluginLib *)malloc (sizeof (NPPluginLib));
    plugin_lib->module = library;

    if (!g_module_symbol (library,
                "NP_GetMIMEDescription",
                (gpointer *)&plugin_lib->npGetMIMEDescription)) {
        print ("undefined reference to load NP_GetMIMEDescription\n");
        goto bail_out;
    }
    if (!g_module_symbol (library,
                "NP_Initialize", (gpointer *)&plugin_lib->npInitialize)) {
        print ("undefined reference to load NP_Initialize\n");
        goto bail_out;
    }
    if (!g_module_symbol (library,
                "NP_Shutdown", (gpointer *)&plugin_lib->npShutdown)) {
        print ("undefined reference to load NP_Shutdown\n");
        goto bail_out;
    }
    print ("startup succeeded %s\n", plugin_lib->npGetMIMEDescription ());

    memset (&ns_funcs, 0, sizeof (NPNetscapeFuncs));
    ns_funcs.size = sizeof (NPNetscapeFuncs);
    ns_funcs.version = (NP_VERSION_MAJOR << 8) + NP_VERSION_MINOR;
    ns_funcs.geturl = nsGetURL;
    ns_funcs.posturl = nsPostURL;
    ns_funcs.requestread = nsRequestRead;
    ns_funcs.newstream = nsNewStream;
    ns_funcs.write = nsWrite;
    ns_funcs.destroystream = nsDestroyStream;
    ns_funcs.status = nsStatus;
    ns_funcs.uagent = nsUserAgent;
    ns_funcs.memalloc = nsAlloc;
    ns_funcs.memfree = nsMemFree;
    ns_funcs.memflush = nsMemFlush;
    ns_funcs.reloadplugins = nsReloadPlugins;
    ns_funcs.getJavaEnv = nsGetJavaEnv;
    ns_funcs.getJavaPeer = nsGetJavaPeer;
    ns_funcs.geturlnotify = nsGetURLNotify;
    ns_funcs.posturlnotify = nsPostURLNotify;
    ns_funcs.getvalue = nsGetValue;
    ns_funcs.setvalue = nsSetValue;
    ns_funcs.invalidaterect = nsInvalidateRect;
    ns_funcs.invalidateregion = nsInvalidateRegion;
    ns_funcs.forceredraw = nsForceRedraw;
    ns_funcs.getstringidentifier = nsGetStringIdentifier;
    ns_funcs.getstringidentifiers = nsGetStringIdentifiers;
    ns_funcs.getintidentifier = nsGetIntIdentifier;
    ns_funcs.identifierisstring = nsIdentifierIsString;
    ns_funcs.utf8fromidentifier = nsUTF8FromIdentifier;
    ns_funcs.intfromidentifier = nsIntFromIdentifier;
    ns_funcs.createobject = nsCreateObject;
    ns_funcs.retainobject = nsRetainObject;
    ns_funcs.releaseobject = nsReleaseObject;
    ns_funcs.invoke = nsInvoke;
    ns_funcs.invokeDefault = nsInvokeDefault;
    ns_funcs.evaluate = nsEvaluate;
    ns_funcs.getproperty = nsGetProperty;
    ns_funcs.setproperty = nsSetProperty;
    ns_funcs.removeproperty = nsRemoveProperty;
    ns_funcs.hasproperty = nsHasProperty;
    ns_funcs.hasmethod = nsHasMethod;
    ns_funcs.releasevariantvalue = nsReleaseVariantValue;
    ns_funcs.setexception = nsSetException;
    ns_funcs.pushpopupsenabledstate = nsPushPopupsEnabledState;
    ns_funcs.poppopupsenabledstate = nsPopPopupsEnabledState;

    js_class.structVersion = NP_CLASS_STRUCT_VERSION;
    js_class.allocate = windowClassAllocate;
    js_class.deallocate = windowClassDeallocate;
    js_class.invalidate = windowClassInvalidate;
    js_class.hasMethod = windowClassHasMethod;
    js_class.invoke = windowClassInvoke;
    js_class.invokeDefault = windowClassInvokeDefault;
    js_class.hasProperty = windowClassHasProperty;
    js_class.getProperty = windowClassGetProperty;
    js_class.setProperty = windowClassSetProperty;
    js_class.removeProperty = windowClassRemoveProperty;

    np_funcs.size = sizeof (NPPluginFuncs);

    np_err = plugin_lib->npInitialize (&ns_funcs, &np_funcs);
    if (np_err != NPERR_NO_ERROR) {
        print ("NP_Initialize failure %d\n", np_err);
        goto bail_out;
    }

    identifiers = g_tree_new (strcmp);
    stream_list = g_tree_new (streamCompare);

    return plugin_lib;

bail_out:
    free (plugin_lib);
    return NULL;
}

void *nppInstanceOpen (const char *mime, uint16_t argc, char *argn[],
        char *argv[], void *ndata, NSNotify *notify) {
    NPSetWindowCallbackStruct ws_info;
    NPError np_err;
    Display *display;
    int screen;
    int i;
    int needs_xembed;
    unsigned int width = 0, height = 0;
    GtkAllocation allocation;

    browser_data = ndata;
    browser_notify = notify;

    npp = (NPP_t*)malloc (sizeof (NPP_t));
    memset (npp, 0, sizeof (NPP_t));
    for (i = 0; i < argc; i++) {
        print ("arg %s %s\n", argn[i], argv[i]);
        if (!strcasecmp (argn[i], "width"))
            width = strtol (argv[i], 0L, 10);
        else if (!strcasecmp (argn[i], "height"))
            height = strtol (argv[i], 0L, 10);
    }
    if (width > 0 && height > 0)
        notify->setDimension (ndata, width, height);

    np_err = np_funcs.newp (mime, npp, NP_EMBED, argc, argn, argv, saved_data);
    if (np_err != NPERR_NO_ERROR) {
        print ("NPP_New failure %d %p %p\n", np_err, np_funcs, np_funcs.newp);
        free (npp);
        return NULL;
    }
    if (np_funcs.getvalue) {
        np_err = np_funcs.getvalue ((void*)npp,
                NPPVpluginNeedsXEmbed, (void*)&needs_xembed);
        if (np_err != NPERR_NO_ERROR || !needs_xembed) {
            print ("NPP_GetValue NPPVpluginNeedsXEmbed failure %d\n", np_err);
            /*shutdownPlugin();*/
            free (npp);
            return NULL;
        }
        np_err = np_funcs.getvalue ((void*)npp,
                NPPVpluginScriptableNPObject, (void*)&scriptable_peer);
        if (np_err != NPERR_NO_ERROR || !scriptable_peer)
            print ("NPP_GetValue no NPPVpluginScriptableNPObject %d\n", np_err);
    }
    memset (&np_window, 0, sizeof (NPWindow));
    display = gdk_x11_get_default_xdisplay ();
    np_window.x = 0;
    np_window.y = 0;
    np_window.width = 800; /*width ? width : top_w;*/
    np_window.height = 480; /*height ? height : top_h;*/
    np_window.window = (void*)socket_id;
    np_window.type = NPWindowTypeWindow;
    ws_info.type = NP_SETWINDOW;
    screen = DefaultScreen (display);
    ws_info.display = (void*)(long)display;
    ws_info.visual = (void*)(long)DefaultVisual (display, screen);
    ws_info.colormap = DefaultColormap (display, screen);
    ws_info.depth = DefaultDepth (display, screen);
    print ("display %u %dx%d\n", socket_id, np_window.width, np_window.height);
    np_window.ws_info = (void*)&ws_info;

    allocation.x = 0;
    allocation.y = 0;
    allocation.width = np_window.width;
    allocation.height = np_window.height;
    gtk_widget_size_allocate (xembed, &allocation);

    np_err = np_funcs.setwindow (npp, &np_window);

    return npp;
}

void nppInstanceStart (const char *url, const char *mime, void *pdata) {
    (void) pdata;
    addStream (url, mime, 0L, 0L, false);
}

void nppStreamData (uint32_t stream, const char *buf, uint32_t sz, void *p) {
    (void)p;
    int written = writeStream (stream, buf, sz);
    if (written < sz)
        print ("FIXME: plugin didn't accept data %d written %d\n", sz, written);
}

void nppStreamFinished (uint32_t stream, uint32_t reason, void *pdata) {
    StreamInfo *si = (StreamInfo *) g_tree_lookup (stream_list, (void *)stream);
    print ("nppStreamFinished\n");
    if (si) {
        si->reason = reason;
        removeStream (stream);
    }
}

void nppStreamRedirected (uint32_t stream, const char *url) {
    StreamInfo *si = (StreamInfo *) g_tree_lookup (stream_list, stream);
    print ("nppStreamRedirected\n");
    if (si) {
        print ("nppStreamRedirected %s -> %s\n", si->url, url);
        g_free (si->url);
        si->url = g_strdup (url);
        si->np_stream.url = si->url;
    }
}

void nppStreamInfo (uint32_t sid, const char *mime, unsigned long length) {
    StreamInfo *si = (StreamInfo *) g_tree_lookup (stream_list, sid);
    if (si) {
        print ("nppStreamInfo %s %s\n", si->url, mime ? mime : "");
        if (si->mimetype)
            g_free (si->mimetype);
        si->mimetype = mime ? g_strdup (mime) : NULL;
        si->np_stream.end = length;
    }
}

void nppInstanceClose (void *pdata) {
    if (scriptable_peer) {
        nsReleaseObject (scriptable_peer);
        scriptable_peer = NULL;
    }
    if (npp) {
        np_funcs.destroy (npp, &saved_data);
        free (npp);
        npp = 0L;
    }
    top_w = 0;
    top_h = 0;
}

void nppInstanceWindowClose (void *p) {
    (void)p;
    if (top_widget)
        gtk_widget_destroy (top_widget);
    np_window.window = NULL;
    top_widget = NULL;
}

/*----------------%<---------------------------------------------------------*/

static void pluginAdded (GtkSocket *socket, gpointer d) {
    /*(void)socket;*/ (void)d;
    print ("pluginAdded\n");
    if (socket->plug_window) {
        gpointer user_data = NULL;
        gdk_window_get_user_data (socket->plug_window, &user_data);
        if (!user_data) {
            /**
             * GtkSocket resets plugins XSelectInput in
             * _gtk_socket_add_window
             *   _gtk_socket_windowing_select_plug_window_input
             **/
            XSelectInput (gdk_x11_get_default_xdisplay (),
                    gdk_x11_drawable_get_xid (socket->plug_window),
                    KeyPressMask | KeyReleaseMask |
                    ButtonPressMask | ButtonReleaseMask |
                    KeymapStateMask |
                    ButtonMotionMask |
                    PointerMotionMask |
                    EnterWindowMask | LeaveWindowMask |
                    FocusChangeMask |
                    ExposureMask |
                    StructureNotifyMask |
                    SubstructureRedirectMask |
                    PropertyChangeMask
                    );
        }
    }
    browser_notify->embedded (browser_data);
}

static void windowCreatedEvent (GtkWidget *w, gpointer d) {
    (void)d;
    socket_id = gtk_socket_get_id (GTK_SOCKET (xembed));
    print ("windowCreatedEvent socket:%d\n", socket_id);
    if (parent_id) {
        /*print ("windowCreatedEvent %p\n", GTK_PLUG (w)->socket_window);*/
        /*if (!GTK_PLUG (w)->socket_window)
            gtk_plug_construct (GTK_PLUG (w), parent_id);*/
        gdk_window_reparent( w->window,
                gdk_window_lookup (parent_id),
                0, 0);
        if (top_w > 0 && top_h > 0)
            nppInstanceWindowDimension (npp, 0, 0, top_w, top_h);
        gtk_widget_show_all (w);
        /*XReparentWindow (gdk_x11_drawable_get_xdisplay (w->window),
                gdk_x11_drawable_get_xid (w->window),
                parent_id,
                0, 0);*/
    }
}

/*static void embeddedEvent (GtkPlug *plug, gpointer d) {
    (void)plug; (void)d;
    print ("embeddedEvent\n");
}*/

static gboolean configureEvent(GtkWidget *w, GdkEventConfigure *e, gpointer d) {
    (void)w; (void)d;
    if (np_window.window &&
            (e->width != np_window.width || e->height != np_window.height)) {
        print ("Update size %dx%d\n", e->width, e->height);
        np_window.width = e->width;
        np_window.height = e->height;
        np_funcs.setwindow (npp, &np_window);
    }
    return FALSE;
}

static gboolean windowCloseEvent (GtkWidget *w, GdkEvent *e, gpointer d) {
    (void)w; (void)e; (void)d;
    print ("windowCloseEvent\n");
    /*shutDownPlugin();*/
    return FALSE;
}

static void windowDestroyEvent (GtkWidget *w, gpointer d) {
    (void)w; (void)d;
    print ("windowDestroyEvent\n");
    /*gtk_main_quit();*/
}

static gboolean updateDimension (void * p) {
    (void)p;
    if (np_window.window &&
            (np_window.width != top_w || np_window.height != top_h)) {
        np_window.width = top_w;
        np_window.height = top_h;
        np_funcs.setwindow (npp, &np_window);
    }
    update_dimension_timer = 0;
    return 0; /* single shot */
}

void nppInstanceWindowDimension (void *p, int x, int y, int w, int h) {
    (void)p; (void) x; (void) y;
    if (top_widget) {
        print ("nppInstanceWindowDimension %dx%d\n", w, h);
        gdk_window_move_resize (top_widget->window, 0, 0, w, h);
        if (!update_dimension_timer)
            g_timeout_add (100, updateDimension, NULL);
    }
    top_w = w;
    top_h = h;
}

void nppInstanceWindowParent (void *p, void * wid) {
    GdkColormap *color_map;
    GdkColor bg_color;
    (void)p;
    /*GdkWindow *parent;*/

    parent_id = (Window) wid;

    top_widget = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    g_signal_connect (G_OBJECT (top_widget), "delete_event",
            G_CALLBACK (windowCloseEvent), NULL);
    g_signal_connect (G_OBJECT (top_widget), "destroy",
            G_CALLBACK (windowDestroyEvent), NULL);
    g_signal_connect_after (G_OBJECT (top_widget), "realize",
            GTK_SIGNAL_FUNC (windowCreatedEvent), NULL);
    /*g_signal_connect (G_OBJECT (top_widget), "configure-event",
            GTK_SIGNAL_FUNC (configureEvent), NULL);*/

    xembed = gtk_socket_new();
    g_signal_connect (G_OBJECT (xembed), "plug-added",
            GTK_SIGNAL_FUNC (pluginAdded), NULL);
    /*g_signal_connect_after (G_OBJECT (xembed), "realize",
            GTK_SIGNAL_FUNC (windowCreatedEvent), NULL);*/

    color_map = gdk_colormap_get_system();
    gdk_colormap_query_color (color_map, 0, &bg_color);
    gtk_widget_modify_bg (xembed, GTK_STATE_NORMAL, &bg_color);

    gtk_container_add (GTK_CONTAINER (top_widget), xembed);

    gtk_widget_set_size_request (top_widget, 800, 480); /*top_w, top_h);*/

    /*g_signal_connect (G_OBJECT (top_widget), "embedded",
            GTK_SIGNAL_FUNC (embeddedEvent), NULL);*/

    /*parent = gdk_window_lookup (parent_id);
    print ("parent gdk window %p\n", parent);
    gtk_widget_set_parent_window (xembed, parent);

    gtk_widget_map (xembed);

    socket_id = gtk_socket_get_id (GTK_SOCKET (xembed));
    print ("socket %u\n", socket_id);
    */

    gtk_widget_realize (top_widget);
}


#endif // __ARMEL__
