/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2005 Koos Vriezen <koos.vriezen@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * until boost gets common, a more or less compatable one ..
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <gtk/gtkmain.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-async-ops.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>

#include "kmplayertypes.h"

using namespace KMPlayer;

KMPLAYER_NO_EXPORT void ByteArrayImpl::resize (unsigned ns) {
    if (ns > m_capacity) {
        char * tmp = (char *) malloc (ns);
        if (m_size > 0)
            memcpy (tmp, m_data, m_size);
        m_capacity = m_size = ns;
        free (m_data);
        m_data = tmp;
    } else
        m_size = ns;
}

//-----------------------------------------------------------------------------

void Shared::clear () {
    if (data)
        data->unref ();
    data = 0L;
}

void Shared::attach (const Shared & other) {
    if (data != other.data) {
        if (other.data)
            other.data->ref ();
        if (data)
            data->unref ();
        data = other.data;
    }
}

void Shared::attach (char * p) {
    if (!data || data->ptr != p) {
        SharedChars * tmp = p ? new SharedChars (p) : 0L;
        if (data)
            data->unref ();
        data = tmp;
    }
}

void Shared::detach () {
    if (data && data->ref_count > 1) {
        SharedChars * tmp = new SharedChars (g_strdup (data->ptr));
        if (data)
            data->unref ();
        data = tmp;
    }
}

//-----------------------------------------------------------------------------

GLibString::GLibString (const Char & ch) {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (ch.unicode (), buf);
    buf [nr] = 0;
    attach (g_strdup (buf));
}

int GLibString::compare (const GLibString & str) const {
    if (str.data == data)
        return 0;
    if (str.data && data)
        return strcmp (str.data->ptr, data->ptr);
    return data ? 1 : -1;
}

void GLibString::take (char * s) {
    const gchar *end;
    if (g_utf8_validate (s, -1, &end)) {
        attach (s);
    } else {
        gsize bytes_read, bytes_written;
        attach (g_locale_to_utf8 (s, -1, &bytes_read, &bytes_written, 0L));
    }
}

GLibString & GLibString::operator = (const char * s) {
    if (!s)
        clear ();
    else
        take (g_strdup (s));
    return *this;
}

GLibString & GLibString::operator = (const Char & ch) {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (ch.unicode (), buf);
    buf [nr] = 0;
    attach (g_strdup (buf));
    return *this;
}

GLibString & GLibString::operator += (const GLibString & str) {
    if (!data) {
        if (str.data)
            attach (g_strdup (str.data->ptr));
    } else if (str.data)
        attach (g_strconcat (data->ptr, str.data->ptr, 0L));
    return *this;
}

GLibString GLibString::fill (Char ch, int length) {
    GLibString str;
    str.take (g_strnfill (length, (const char) ch)); //FIME
    return str;
}

GLibString GLibString::stripWhiteSpace () const {
    String str;
    if (data && data->ptr [0]) {
        String ws;
        gchar * iter = data->ptr;
        Char ch (g_utf8_get_char (iter));
        iter = g_utf8_next_char (iter);
        while (ch.isSpace () && iter && *iter) { // skip leading ws
            ch = Char (g_utf8_get_char (iter));
            iter = g_utf8_next_char (iter);
        }
        if (ch.isSpace ())
            return str;
        str += ch;
        do {
            if (!iter || !*iter)
                break;
            ch = Char (g_utf8_get_char (iter));
            iter = g_utf8_next_char (iter);
            while (ch.isSpace () && iter && *iter) {
                ws += ch;
                ch = Char (g_utf8_get_char (iter));
                iter = g_utf8_next_char (iter);
            }
            if (ch.isSpace ())
                break;  // skip ending ws
            if (!ws.isEmpty ()) {
                str += ws;
                ws.truncate (0);
            }
            str += ch;
        } while (iter && *iter);
    }
    return str;
}

bool GLibString::hasPrefix (const GLibString & prefix) const {
    if (!prefix.data && !data)
        return true;
    if (prefix.data && data)
        return g_str_has_prefix (data->ptr, prefix.data->ptr);
    return !prefix.data;
}
    
bool GLibString::hasSuffix (const GLibString & suffix) const {
    if (!suffix.data && !data)
        return true;
    if (suffix.data && data)
        return g_str_has_suffix (data->ptr, suffix.data->ptr);
    return !suffix.data;
}

int GLibString::find (const GLibString & needle) const {
    if (!data)
        return needle.data ? -1 : 0;
    if (!needle.data)
        return 0;
    gchar * str = strstr (data->ptr, needle.data->ptr);
    if (!str)
        return -1;
    return int (str - data->ptr);
}

int GLibString::findRev (const GLibString & needle) const {
    if (!data)
        return needle.data ? -1 : 0;
    if (!needle.data)
        return length ();
    gchar * str = g_strrstr (data->ptr, needle.data->ptr);
    if (!str)
        return -1;
    return int (str - data->ptr);
}

int GLibString::toInt (bool * valid, int base) const {
    int i = 0;
    if (data) {
        char *endptr;
        i = (int) strtol (data->ptr, &endptr, base);
        if (valid)
            *valid = (data->ptr != endptr);
    } else if (valid)
        *valid = false;
    return i;
}

KMPLAYER_NO_EXPORT double GLibString::toDouble (bool * valid) const {
    if (data) {
        gchar * endptr;
        double d = g_ascii_strtod (data->ptr, &endptr);
        if (valid)
            *valid = (endptr != data->ptr);
        return d;
    } else {
        if (valid)
            *valid = false;
        return 0.0;
    }
}

GLibString GLibString::mid (const int index, unsigned len) const {
    if (!data)
        return GLibString ();
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    gchar * eos = off;
    if (!off)
        return GLibString ();
    while (eos && *eos && len--)
        eos = g_utf8_next_char (eos);
    GLibString str;
    if (!eos || !*eos)
        str.attach (g_strdup (off));
    else
        str.attach (g_strndup (off, eos-off));
    return str;
}

GLibString GLibString::left (int index) const {
    if (!data)
        return GLibString ();
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    if (!off)
        return GLibString ();
    GLibString str;
    str.attach (g_strndup (data->ptr, off - data->ptr));
    return str;
}

void GLibString::truncate (int pos) {
    if (data) {
        if (pos == 0) {  // speed up clearing string
            clear ();
        } else {
            gchar * epos = g_utf8_offset_to_pointer (data->ptr, pos);
            if (epos)
                attach (g_strndup (data->ptr, epos - data->ptr));
        }
    }
}

//-----------------------------------------------------------------------------

namespace KMPlayer {

class GnomeVFSFilePrivate {
public:
    GnomeVFSFilePrivate () : m_vfshandle (0L) {}
    ~GnomeVFSFilePrivate () {}
    GnomeVFSHandle * m_vfshandle;
    String m_path;
};

}

KMPLAYER_NO_CDTOR_EXPORT
GnomeVFSFile::GnomeVFSFile (const String & p) : d (new GnomeVFSFilePrivate) {
    d->m_path = p;
}

KMPLAYER_NO_CDTOR_EXPORT GnomeVFSFile::~GnomeVFSFile () {
    close ();
}

KMPLAYER_NO_EXPORT bool GnomeVFSFile::open (FileMode mode) {
    switch (mode) {
        case IO_ReadOnly:
            if (gnome_vfs_open (&d->m_vfshandle, (const char *) d->m_path, GNOME_VFS_OPEN_READ) == GNOME_VFS_OK)
                return true;
            d->m_vfshandle = 0L;
            break;
        case IO_WriteOnly: // fixme truncate
            if (gnome_vfs_open (&d->m_vfshandle, (const char *) d->m_path, GNOME_VFS_OPEN_WRITE) == GNOME_VFS_OK)
                return true;
            if (gnome_vfs_create (&d->m_vfshandle,(const char *) d->m_path, GNOME_VFS_OPEN_WRITE, TRUE, GNOME_VFS_PERM_USER_ALL) == GNOME_VFS_OK)
                return true;
            d->m_vfshandle = 0L;
            break;
        default:
            warningLog() << "open " << int (mode) << " not implemented" << endl;
    }
    return false;
}

KMPLAYER_NO_EXPORT void GnomeVFSFile::close () {
    if (d->m_vfshandle) {
        gnome_vfs_close (d->m_vfshandle);
        d->m_vfshandle = 0L;
    }
}

KMPLAYER_NO_EXPORT ByteArray GnomeVFSFile::readAll () {
    ByteArray str;
    char buf [2048];
    GnomeVFSFileSize nr;
    if (!d->m_vfshandle && !open (IO_ReadOnly))
        return str;
    gnome_vfs_seek (d->m_vfshandle, GNOME_VFS_SEEK_START, 0);
    do {
        if (gnome_vfs_read (d->m_vfshandle, buf, sizeof (buf) -1, &nr) != GNOME_VFS_OK)
            break;
        off_t size = str.size ();
        str.resize (size + nr);
        memcpy (str.data () + size, buf, nr);
    } while (nr > 0);
    return str;
}

KMPLAYER_NO_EXPORT off_t GnomeVFSFile::readBlock (char * buf, off_t max) {
    if (!d->m_vfshandle && !open (IO_ReadOnly))
        return 0;
    GnomeVFSFileSize nr;
    if (gnome_vfs_read (d->m_vfshandle, buf, max, &nr) != GNOME_VFS_OK)
        return 0;
    return nr;
}

KMPLAYER_NO_EXPORT bool GnomeVFSFile::exists () {
    GnomeVFSFileInfo info;
    return (gnome_vfs_get_file_info ((const char *) d->m_path, &info, GNOME_VFS_FILE_INFO_DEFAULT) == GNOME_VFS_OK);
}

KMPLAYER_NO_EXPORT off_t GnomeVFSFile::size () {
    GnomeVFSFileInfo info;
    if (gnome_vfs_get_file_info ((const char *) d->m_path, &info, GNOME_VFS_FILE_INFO_DEFAULT) != GNOME_VFS_OK)
        return 0;
    return info.size;
}

KMPLAYER_NO_EXPORT off_t GnomeVFSFile::writeBlock (const char * data, off_t length) {
    debugLog () << "writeBlock " << int (length) << endl;
    off_t len = length;
    while (len > 0) {
        GnomeVFSFileSize bytes_written;
        if (gnome_vfs_write (d->m_vfshandle, data + (length - len), len, &bytes_written) != GNOME_VFS_OK)
            break;
        len -= bytes_written;
    }
    return length - len;
}

//-----------------------------------------------------------------------------

GnomeVFSUrl::GnomeVFSUrl (const GnomeVFSUrl & base, const String & rel) {
    m_vfsurl= (base.m_vfsurl ?
            gnome_vfs_uri_resolve_relative (base.m_vfsurl, rel) :
            gnome_vfs_uri_new ((const char *) rel));
}

KMPLAYER_NO_CDTOR_EXPORT GnomeVFSUrl::GnomeVFSUrl (const String & url) {
    m_vfsurl = (url.isEmpty () ? 0L : gnome_vfs_uri_new ((const char *) url));
}

KMPLAYER_NO_CDTOR_EXPORT GnomeVFSUrl::GnomeVFSUrl (const GnomeVFSUrl & url) {
    m_vfsurl = (url.m_vfsurl ? gnome_vfs_uri_dup (url.m_vfsurl) : 0L);
}

KMPLAYER_NO_CDTOR_EXPORT GnomeVFSUrl::~GnomeVFSUrl () {
    if (m_vfsurl)
        gnome_vfs_uri_unref (m_vfsurl);
}

KMPLAYER_NO_EXPORT String GnomeVFSUrl::url () const {
    if (m_vfsurl) {
        GLibString s;
        s.take (gnome_vfs_uri_to_string (m_vfsurl, GNOME_VFS_URI_HIDE_NONE));
        return String (s);
    }
    return String ();
}

GnomeVFSUrl & GnomeVFSUrl::operator = (const String & url) {
    if (m_vfsurl)
        gnome_vfs_uri_unref (m_vfsurl);
    m_vfsurl = gnome_vfs_uri_new ((const char *) url);
    return *this;
}

GnomeVFSUrl & GnomeVFSUrl::operator = (const GnomeVFSUrl & url) {
    if (m_vfsurl)
        gnome_vfs_uri_unref (m_vfsurl);
    m_vfsurl = url.m_vfsurl ? gnome_vfs_uri_dup (url.m_vfsurl) : 0L;
    return *this;
}

KMPLAYER_NO_EXPORT String GnomeVFSUrl::path () const {
    if (m_vfsurl)
        return String (gnome_vfs_uri_get_path (m_vfsurl));
    return String ();
}

KMPLAYER_NO_EXPORT String GnomeVFSUrl::protocol () const {
    if (m_vfsurl)
        return String (gnome_vfs_uri_get_scheme (m_vfsurl));
    return String ();
}

KMPLAYER_NO_EXPORT String GnomeVFSUrl::decode_string (const String & s) {
    if (s.isEmpty ())
        return String ();
    return String (gnome_vfs_unescape_string ((const char *) s, ""));
}

KMPLAYER_NO_EXPORT bool GnomeVFSUrl::isLocalFile () const {
    if (m_vfsurl)
        return gnome_vfs_uri_is_local (m_vfsurl) ||
            protocol () == String ("file");
    return false;
}

//-----------------------------------------------------------------------------

String GMimeType::findByContent (const char * data, int size) {
    if (size <= 0)
        return String();
    const gchar *end;
    if (g_utf8_validate ((gchar *) data, size, &end))
        return "text/plain";
    if (g_utf8_get_char_validated (end, size - (end - data)) == (gunichar)-2)
        return "text/plain";
    return String (gnome_vfs_get_mime_type_for_data (data, size));
}

//-----------------------------------------------------------------------------

GLibTextStream & GLibTextStream::operator >> (Char & ch) {
    if (m_gstring) {
        m_utf8_str = g_string_free (m_gstring, false);
        m_gstring = 0L;
        m_iter = m_utf8_str;
    }
    if (m_iter) {
        ch = Char (g_utf8_get_char (m_iter));
        m_iter = g_utf8_next_char (m_iter);
    }
}

KMPLAYER_NO_EXPORT
GLibTextStream & GLibTextStream::operator = (const GLibTextStream & s) {
    if (m_gstring)
        g_string_free (m_gstring, true);
    else if (m_utf8_str)
        g_free (m_utf8_str);
    m_utf8_str = 0L;
    m_iter = 0L;
    m_gstring = g_string_new (s.m_gstring ? s.m_gstring->str :
            (s.m_utf8_str ? s.m_utf8_str : ""));
    return *this;
}

String GLibTextStream::readLine () {
    String str;
    Char ch;
    while (!atEnd ()) {
        *this >> ch;
        if (ch == '\r' || ch == '\n')
            break;
        str += ch;
    }
    gchar * iter = m_iter;
    while (!atEnd ()) { // FIXME: current is skipping emty lines as well
        *this >> ch;
        if (ch != '\r' && ch != '\n') {
            m_iter = iter;
            break;
        }
        iter = m_iter;
    }
    return str;
}

//-----------------------------------------------------------------------------

namespace KMPlayer {

    class GnomeVFSGetJob : public JobInterface {
    public:
        GnomeVFSGetJob (AsyncJobInterface * rec, const String & u);
        ~GnomeVFSGetJob ();
        void kill (bool quiet = true);
        void finished ();

        GnomeVFSAsyncHandle * network_handle;
        AsyncJobInterface * m_receiver;
        bool job_closed;
        bool keep_quiet;
        bool continuing;
        char read_buf [2048];
    };

    Job * asyncGet (AsyncJobInterface * receiver, const String & url) {
        return new GnomeVFSGetJob (receiver, url);
    }
}

static void cb_async_get_close (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer p) {
    //debugLog () << "cb_async_get_close " << endl;
    if (!p)
        return;
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    get->finished ();
}

static void cb_async_get_read (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer buffer, GnomeVFSFileSize bytes_requested, GnomeVFSFileSize bytes_read, gpointer p) {
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    if (get->job_closed)
        return;
    get->continuing = true;
    if (bytes_read > 0) {
        ByteArray data (bytes_read);
        memcpy (data.data (), (const char *) buffer, bytes_read);
        get->m_receiver->jobData (get, data);
    }
    //debugLog () << "cb_async_get_read " << int (bytes_read) << " canceled:" << get->job_closed << " not ok:" << (result != GNOME_VFS_OK) << " less:" << (bytes_read < bytes_requested) << " killed:" << !get->continuing << endl;
    if (result != GNOME_VFS_OK || bytes_read <= 0 || !get->continuing) {
        // || bytes_read < bytes_requested 
        get->job_closed = true;
        gnome_vfs_async_cancel (handle);
        gnome_vfs_async_close (handle, cb_async_get_close, p);
    } else {
        gnome_vfs_async_read (handle, get->read_buf, sizeof (get->read_buf) -1, cb_async_get_read, p);
    }
    get->continuing = false;
}

static void cb_async_get_open (GnomeVFSAsyncHandle *handle, GnomeVFSResult result, gpointer p) {
    //debugLog () << "cb_async_get_open " << endl;
    GnomeVFSGetJob * get = static_cast <GnomeVFSGetJob *> (p);
    if (get->job_closed)
        return;
    if (result != GNOME_VFS_OK)
        get->finished ();  // kills get
    else
        gnome_vfs_async_read (handle, get->read_buf, sizeof (get->read_buf) -1, cb_async_get_read, p);
}

GnomeVFSGetJob::GnomeVFSGetJob (AsyncJobInterface * rec, const String &url)
 : m_receiver (rec), network_handle (0L),
   job_closed (false), keep_quiet (false), continuing (false) {
    gnome_vfs_async_open (&network_handle, (const char *) url, GNOME_VFS_OPEN_READ, 0, cb_async_get_open , static_cast <void *> (this));
}

GnomeVFSGetJob::~GnomeVFSGetJob () {
    if (network_handle && !job_closed) {
        gnome_vfs_async_cancel (network_handle);
        gnome_vfs_async_close (network_handle, cb_async_get_close, 0L);
    }
}

void GnomeVFSGetJob::kill (bool quiet) {
    keep_quiet = quiet;
    if (network_handle && !job_closed) {
        if (!continuing) {
            job_closed = true;
            gnome_vfs_async_cancel (network_handle);
            gnome_vfs_async_close (network_handle, cb_async_get_close, static_cast <void *> (this));//
        } else
            continuing = false;  // killed from jobData receiver callback
    } else if (!quiet)
        m_receiver->jobResult (this);
}

void GnomeVFSGetJob::finished () {
    network_handle = 0L;
    if (!keep_quiet)
        m_receiver->jobResult (this);
    delete this;
}

//-----------------------------------------------------------------------------

static bool gconf_initialized = false;

KMPLAYER_NO_CDTOR_EXPORT GConf::GConf () {
    if (!gconf_initialized) {
        gconf_initialized = true;
        g_type_init ();
    }
    client = gconf_client_get_default ();
}

KMPLAYER_NO_CDTOR_EXPORT GConf::~GConf () {
    gconf_client_clear_cache (client);
    g_object_unref (client);
}

KMPLAYER_NO_EXPORT int GConf::readNumEntry (const String & key, int def) {
    GError * err;
    int val = gconf_client_get_int (client, (const char *) key, &err);
    bool has_err = !!err;
    g_error_free (err);
    return has_err ? def : val;
}

KMPLAYER_NO_EXPORT double GConf::readNumEntry (const String & key, double def) {
    GError * err;
    double val = gconf_client_get_float (client, (const char *) key, &err);
    bool has_err = !!err;
    g_error_free (err);
    return has_err ? def : val;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_EXPORT Log & Log::operator << (const char * s) {
    fprintf (stderr, "%s", s);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (int i) {
    fprintf (stderr, "%d", i);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (unsigned int i) {
    fprintf (stderr, "%u", i);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (double d) {
    fprintf (stderr, "%f", d);
    return *this;
}

KMPLAYER_NO_EXPORT Log & Log::operator << (void * d) {
    fprintf (stderr, "0x%x", d);
    return *this;
}

//-----------------------------------------------------------------------------

GRect GRect::unite (const GRect & r) const {
    GRect rect;
    gdk_rectangle_union (m_rect, (GdkRectangle *) r, (GdkRectangle *) rect);
    return rect;
}

GRect GRect::intersect (const GRect & r) const {
    GRect rect;
    gdk_rectangle_intersect (m_rect, (GdkRectangle *) r, (GdkRectangle *) rect);
    return rect;
}

//-----------------------------------------------------------------------------

#include <glib/gprintf.h>

static GdkColor * colorFromRGB (int rgb) {
    GdkColormap * cmap = gdk_colormap_get_system ();
    GdkColor * color = g_new (GdkColor, 1);
    char buf [16]; 
    g_sprintf (buf, "#%02X%02X%02X", (rgb & 0x00FF0000) >> 16, (rgb & 0x0000FF00) >> 8, (rgb & 0x000000FF));
    if (!(gdk_color_parse (buf, color) &&
                (gdk_colormap_alloc_color (cmap, color, 0, 0) ||
                 gdk_colormap_alloc_color (cmap, color, 0, 1)))) {
        g_free (color);
        color = 0L;
    }
    return color;
}

KMPLAYER_NO_CDTOR_EXPORT GColor::GColor (const String & desc) {
    GdkColormap * cmap = gdk_colormap_get_system ();
    m_color = g_new (GdkColor, 1); 
    if (!(gdk_color_parse ((const char *) desc, m_color) &&
                gdk_colormap_alloc_color (cmap, m_color, 0, 1))) {
        g_free (m_color);
        m_color = 0L;
        if (desc.lower () == "silver")
            m_color = colorFromRGB (0xbebebe);
        // TODO add more unsupported real live colors
        if (!m_color)
            errorLog () << "Could not parse " << desc << " color" << endl;
    }
    //debugLog () << desc << " -> " << int (m_color ? m_color->pixel : 0) << endl;
}

KMPLAYER_NO_CDTOR_EXPORT GColor::GColor (int rgb)
    : m_color (colorFromRGB(rgb)) {
    //debugLog () << "GColor::GColor (int " << rgb << ") -> " << int (m_color ? m_color->pixel : -1) << " str:" << buf << endl;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT GPixmap::GPixmap (int w, int h) {
    GdkVisual * visual = gdk_visual_get_system ();//gdk_rgb_get_visual
    m_pixmap = gdk_pixmap_new (0L, w, h, visual->depth);
}

KMPLAYER_NO_CDTOR_EXPORT
GPixmap::GPixmap (const ByteArray & data) : m_pixmap (0L) {
    GdkPixbufLoader * pix_loader = gdk_pixbuf_loader_new ();
    GError *gerror = 0L;
    if (gdk_pixbuf_loader_write (pix_loader, (const guchar *) data.data (), data.size (), &gerror) && gdk_pixbuf_loader_close (pix_loader, 0L)) {
        *this = gdk_pixbuf_loader_get_pixbuf (pix_loader);
    } else
        g_free (gerror);
    g_object_unref (pix_loader);
}

KMPLAYER_NO_CDTOR_EXPORT GPixmap::GPixmap (GdkPixbuf * pb) : m_pixmap (0L) {
    if (pb)
        *this = pb;
}

KMPLAYER_NO_CDTOR_EXPORT
GPixmap::GPixmap (const GPixmap & other) : m_pixmap (other.m_pixmap) {
    if (m_pixmap)
        g_object_ref (m_pixmap);
}

KMPLAYER_NO_CDTOR_EXPORT GPixmap::GPixmap (const Image & image) : m_pixmap(0L) {
    const GPixbuf & gpb = (const GPixbuf) image;
    GdkPixbuf * pb = gpb.pixbuf ();
    if (pb)
        *this = pb;
}

KMPLAYER_NO_CDTOR_EXPORT GPixmap::~GPixmap () {
    if (m_pixmap)
        g_object_unref (m_pixmap);
}

int GPixmap::width () const {
    int w = 0, h;
    if (m_pixmap)
        gdk_drawable_get_size (GDK_DRAWABLE (m_pixmap), &w, &h);
    return w;
}

int GPixmap::height () const {
    int w, h = 0;
    if (m_pixmap)
        gdk_drawable_get_size (GDK_DRAWABLE (m_pixmap), &w, &h);
    return h;
}

void GPixmap::fill (const Color & c) {
    if (m_pixmap) {
        int w, h;
        gdk_drawable_get_size (GDK_DRAWABLE (m_pixmap), &w, &h);
        GdkGC * gc = gdk_gc_new (GDK_DRAWABLE (m_pixmap));
        gdk_draw_rectangle (GDK_DRAWABLE (m_pixmap), gc, true, 0, 0, w, h);
        gdk_gc_unref (gc);
    }
}

GPixmap &  GPixmap::operator = (GdkPixbuf * pixbuf) {
    if (m_pixmap)
        g_object_unref (m_pixmap);
    int w = gdk_pixbuf_get_width (pixbuf);
    int h = gdk_pixbuf_get_height (pixbuf);
    GdkVisual * visual = gdk_visual_get_system ();
    m_pixmap = gdk_pixmap_new (0L, w, h, visual->depth);
    GdkColormap * cmap = gdk_colormap_get_system ();
    gdk_drawable_set_colormap (GDK_DRAWABLE (m_pixmap), cmap);
    GdkGC * gc = gdk_gc_new (GDK_DRAWABLE (m_pixmap));
    gdk_draw_pixbuf (GDK_DRAWABLE (m_pixmap), gc, pixbuf, 0, 0, 0, 0, w, h, GDK_RGB_DITHER_NORMAL, 0, 0);
    gdk_gc_unref (gc);
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (int w, int h) {
    GdkVisual * visual = gdk_visual_get_system ();//gdk_rgb_get_visual
    m_pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, visual->depth, w, h);
}

KMPLAYER_NO_CDTOR_EXPORT
GPixbuf::GPixbuf (const ByteArray & data) : m_pixbuf (0L) {
    GdkPixbufLoader * pix_loader = gdk_pixbuf_loader_new ();
    GError *gerror = 0L;
    if (gdk_pixbuf_loader_write (pix_loader, (const guchar *) data.data (), data.size (), &gerror) && gdk_pixbuf_loader_close (pix_loader, 0L)) {
        m_pixbuf = gdk_pixbuf_loader_get_pixbuf (pix_loader);
        g_object_ref (m_pixbuf);
    } else
        g_free (gerror);
    g_object_unref (pix_loader);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (GdkPixbuf * pb, bool skip_addref)
  : m_pixbuf (pb) {
    if (!skip_addref && m_pixbuf)
        g_object_ref (m_pixbuf);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::GPixbuf (const GPixbuf & other)
 : m_pixbuf (other.m_pixbuf) {
    if (m_pixbuf)
        g_object_ref (m_pixbuf);
}

KMPLAYER_NO_CDTOR_EXPORT GPixbuf::~GPixbuf () {
    reset ();
}

KMPLAYER_NO_EXPORT GPixbuf & GPixbuf::operator = (const GPixbuf & other) {
    if (&other != this) {
        reset ();
        m_pixbuf = other.m_pixbuf;
        if (m_pixbuf)
            g_object_ref (m_pixbuf);
    }
    return *this;
}

KMPLAYER_NO_EXPORT GPixbuf & GPixbuf::operator = (const Pixmap & pixmap) {
    const GPixmap & gpm = (const GPixmap) pixmap;
    GdkPixmap * pm = gpm.pixmap ();
    reset ();
    if (pm) {
        GdkColormap * cmap = gdk_colormap_get_system ();
        m_pixbuf = gdk_pixbuf_get_from_drawable (0L, GDK_DRAWABLE (pm), cmap, 0, 0, 0, 0, pixmap.width (), pixmap.height ());
    }
    return *this;
}

KMPLAYER_NO_EXPORT void GPixbuf::reset () {
    if (m_pixbuf) {
        g_object_unref (m_pixbuf);
        m_pixbuf = 0L;
    }
}

KMPLAYER_NO_EXPORT void GPixbuf::fill (const Color & c) {
    if (m_pixbuf)
        gdk_pixbuf_fill (m_pixbuf, c.rgb ());
}

KMPLAYER_NO_EXPORT GPixbuf GPixbuf::scale (int w, int h) {
    if (m_pixbuf)
        return GPixbuf (gdk_pixbuf_scale_simple (m_pixbuf, w, h, w > width () ? GDK_INTERP_NEAREST : GDK_INTERP_BILINEAR), true);
}


//-----------------------------------------------------------------------------

/*
namespace KMPlayer {
    
class GPainterPrivate {
public:
    GPainterPrivate ()
        : m_drawable (0L), m_gc (0L),
    m_pango_enabled (false), pango_bg ("white"), pango_fg ("black")  {}
    GdkDrawable * m_drawable;
    GdkGC * m_gc;
    PangoRenderer *renderer;
    PangoContext *context;
    PangoLayout * layout;
    Color pango_bg;
    Color pango_fg;
    bool m_pango_enabled;
    void enablePango ();
    void disablePango ();
};

}

KMPLAYER_NO_EXPORT
void GPainterPrivate::enablePango () {
    if (!m_pango_enabled) {
        m_pango_enabled = true;
        GdkScreen *screen = gdk_drawable_get_screen (m_drawable);
        renderer = gdk_pango_renderer_get_default (screen);
        context = gdk_pango_context_get_for_screen (screen);
        layout = pango_layout_new (context);
        //pango_context_set_base_dir (context, gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR ?  PANGO_DIRECTION_LTR : PANGO_DIRECTION_RTL);
        pango_context_set_base_dir (context, PANGO_DIRECTION_LTR);
        //pango_context_set_font_description (context, widget->style->font_desc);
        pango_context_set_language (context, gtk_get_default_language ());
    }
}

KMPLAYER_NO_EXPORT
void GPainterPrivate::disablePango () {
    if (m_pango_enabled) {
        m_pango_enabled = false;
        g_object_unref (layout);
        g_object_unref (context);
    }
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT 
GPainter::GPainter ()
    : d (new GPainterPrivate) {}

KMPLAYER_NO_CDTOR_EXPORT GPainter::~GPainter () {
    if (d->m_drawable)
        end ();
    delete d;
}

//KMPLAYER_NO_EXPORT GPainter::operator GdkGC * () const {
//    return d->m_gc;
//}

GPainter::operator GdkDrawable * () const {
    return d->m_drawable;
}

bool GPainter::begin (GdkDrawable * drawable, const Rect & rect) {
    if (d->m_drawable)
        return false;
    d->m_drawable = drawable;
    g_object_ref (G_OBJECT (d->m_drawable));
    if (rect.width () <= 0) {
        int w, h;
        gdk_drawable_get_size (d->m_drawable, &w, &h);
        m_drawrect = Rect (0, 0, w, h);
    } else
        m_drawrect = rect;
    d->m_gc = gdk_gc_new (d->m_drawable);
    d->m_pango_enabled = false;
    m_cliprect = m_drawrect;
    if (GDK_IS_WINDOW (d->m_drawable))
        gdk_window_begin_paint_rect (d->m_drawable, (GdkRectangle *) (const GRect &) rect); 
    return true;
}

bool GPainter::begin (const Pixmap * pm) {
    const GPixmap & gpm = (const GPixmap) *pm;
    return begin ((GdkDrawable *) gpm, Rect ());
}

bool GPainter::end () {
    if (GDK_IS_WINDOW (d->m_drawable))
        gdk_window_end_paint (d->m_drawable);
    //debugLog () << "copyRect " << m_drawrect.x() << "," << m_drawrect.y() << " " << m_drawrect.width () << "x" << m_drawrect.height () << endl;
    d->disablePango ();
    g_object_unref (G_OBJECT (d->m_drawable));
    d->m_drawable = 0L;
    gdk_gc_unref (d->m_gc);
    d->m_gc = 0L;
    return true;
}

void GPainter::setClipRect (const Rect & rect) {
    m_cliprect = rect;
    gdk_gc_set_clip_rectangle (d->m_gc, (GdkRectangle *) (const GRect &) rect);
}

void GPainter::setClipping (bool enable) {
    //setClipRect (enable ? m_cliprect : m_drawrect);
}

void GPainter::fillRect (int x, int y, int w, int h, const Color & c) {
    const GColor color = (const GColor) c;
  //debugLog () << " fillRect " << x << "," << y << " " << w << "x" << h<< endl;
    gdk_gc_set_foreground (d->m_gc, (GdkColor *) color);
    gdk_draw_rectangle (d->m_drawable, d->m_gc, true, x, y, w, h);
}

void GPainter::drawPixbuf (GdkPixbuf *p, int x, int y, int sx, int sy, int sw, int sh) {
    gdk_draw_pixbuf (d->m_drawable, d->m_gc, p, sx, sy, x, y, sw, sh, GDK_RGB_DITHER_NORMAL, 0, 0);
}

void GPainter::drawPixmap (int dx, int dy, const Pixmap & pm, int sx, int sy, int sw, int sh) {
    const GPixmap & gpm = (const GPixmap) pm;
    GdkDrawable * p = (GdkDrawable *) gpm;
    if (p && d->m_drawable) {
        if (sw < 0 || sh < 0) {
            int w, h;
            gdk_drawable_get_size (p, &w, &h);
            if (sw < 0)
                sw = w;
            if (sh < 0)
                sh = h;
        }
        gdk_draw_drawable (d->m_drawable, d->m_gc, p, sx, sy, dx, dy, sw, sh);
    }
}

void GPainter::drawImage (int dx, int dy, const Image & im, int sx, int sy, int sw, int sh) {
    const GPixbuf & gpb = (const GPixbuf) im;
    GdkPixbuf * pb = gpb.pixbuf ();
    if (pb)
        gdk_draw_pixbuf (d->m_drawable, d->m_gc, pb, sx, sy, dx, dy, sw, sh, GDK_RGB_DITHER_NORMAL, 0, 0);
}

void GPainter::drawText (const String & txt, int x, int y, int w, int h) {
    // debugLog () << " drawText " << txt << " " << x << "," << y << " clip:" << m_cliprect.x () << "," <<  m_cliprect.y () << endl;
    d->enablePango ();
    pango_layout_set_width(d->layout, PANGO_SCALE *(w>0 ? w : m_cliprect.width()));
    pango_layout_set_wrap (d->layout, PANGO_WRAP_WORD);
    pango_layout_set_text (d->layout, (const char *) txt, -1);
    //PangoMatrix matrix = PANGO_MATRIX_INIT;
    //pango_matrix_translate (&matrix, pw * (x - m_drawrect.x ()), ph * (y - m_drawrect.y ()));
    //pango_context_set_matrix (d->context, &matrix);
    //pango_layout_context_changed (layout);
    gdk_pango_renderer_set_drawable (GDK_PANGO_RENDERER (d->renderer), d->m_drawable);
    gdk_pango_renderer_set_gc (GDK_PANGO_RENDERER (d->renderer), d->m_gc);
    gdk_pango_renderer_set_override_color (GDK_PANGO_RENDERER (d->renderer),
            PANGO_RENDER_PART_FOREGROUND, (GdkColor*) ((const GColor&) d->pango_fg));
    gdk_pango_renderer_set_override_color (GDK_PANGO_RENDERER (d->renderer),
            PANGO_RENDER_PART_BACKGROUND, (GdkColor*) ((const GColor&) d->pango_bg));
    pango_renderer_set_matrix (d->renderer, NULL);
    pango_renderer_activate (d->renderer);
    pango_renderer_draw_layout (d->renderer, d->layout, PANGO_SCALE * x, PANGO_SCALE * y);
    pango_renderer_deactivate (d->renderer);
    gdk_pango_renderer_set_override_color (GDK_PANGO_RENDERER (d->renderer),
            PANGO_RENDER_PART_FOREGROUND, NULL);
    gdk_pango_renderer_set_override_color (GDK_PANGO_RENDERER (d->renderer),
            PANGO_RENDER_PART_BACKGROUND, NULL);
    gdk_pango_renderer_set_drawable (GDK_PANGO_RENDERER (d->renderer), NULL);
    gdk_pango_renderer_set_gc (GDK_PANGO_RENDERER (d->renderer), NULL);
}

void GPainter::setPen (const Color & c) {
    d->pango_fg = c;
}

void GPainter::setBackgroundColor (const Color & c) {
    d->pango_bg = c;
}

void GPainter::setFont (const String & str) {
    d->enablePango ();
    PangoFontDescription *desc = pango_font_description_from_string ((const char *) str);
    pango_layout_set_font_description (d->layout, desc);
    pango_font_description_free (desc);
}
*/
/*
void        gdk_pixbuf_xlib_init            (Display *display,
                                             int screen_num);


GdkPixbuf*  gdk_pixbuf_new_from_file        (const char *filename,
                                             GError **error);
// gdk_pixbuf_new_from_data gdk_pixbuf_new_from_xpm_data
void        gdk_pixbuf_copy_area            (const GdkPixbuf *src_pixbuf,
                                             int src_x,
                                             int src_y,
                                             int width,
                                             int height,
                                             GdkPixbuf *dest_pixbuf,
                                             int dest_x,
                                             int dest_y);

void        gdk_pixbuf_scale                (const GdkPixbuf *src,
                                             GdkPixbuf *dest,
                                             int dest_x,
                                             int dest_y,
                                             int dest_width,
                                             int dest_height,
                                             double offset_x,
                                             double offset_y,
                                             double scale_x,
                                             double scale_y,
                                             GdkInterpType interp_type);

g_object_unref() 

// http://developer.gnome.org/doc/API/2.0/gdk-pixbuf/gdk-pixbuf-animation.html

GdkPixbuf*  gdk_pixbuf_add_alpha            (const GdkPixbuf *pixbuf,
                                             gboolean substitute_color,
                                             guchar r,
                                             guchar g,
                                             guchar b);


typedef struct {
  gint x;
  gint y;
  gint width;
  gint height;
} GdkRectangle;

void        gdk_rectangle_union             (GdkRectangle *src1,
                                             GdkRectangle *src2,
                                             GdkRectangle *dest);

*/
