/* This file is part of the KMPlayer project
 *
 * Copyright (C) 2005 Koos Vriezen <koos.vriezen@xs4all.nl>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _KMPLAYERPROCESS_H_
#define _KMPLAYERPROCESS_H_

#include <libosso.h>
#include <list>

#include "kmplayerplaylist.h"

typedef struct DBusPendingCall DBusPendingCall;

namespace KMPlayer {
    
class Viewer;
class ProcessNotify;

/*
 * Base class for all backend processes
 */
class KMPLAYER_EXPORT Process {
public:
    enum State {
        NotRunning = 0, Ready, Buffering, Playing, Paused
    };
    Process (ProcessNotify * ctrl, const char * n, bool xv);
    virtual ~Process ();
    virtual void init ();
    virtual void initProcess (unsigned int);
    //virtual QString menuName () const;
    //virtual void setAudioLang (int, const QString &);
    //virtual void setSubtitle (int, const QString &);
    void setPosition (int p);
    bool playing () const;
    //KDE_NO_EXPORT KProcess * process () const { return m_process; }
    //KDE_NO_EXPORT Source * source () const { return m_source; }
    //virtual WId widget ();
    unsigned int viewer () const;
    //void setSource (Source * src) { m_source = src; }
    //virtual bool grabPicture (const KURL & url, int pos);
    //bool supports (const char * source) const;
    State state () const { return m_state; }
    NodePtr mrl () const { return m_mrl; }
//signals:
    //void grabReady (const QString & path);
public:// slots:
    virtual bool ready (unsigned int);
    virtual bool play (NodePtr mrl);
    virtual bool stop ();
    virtual bool quit ();
    virtual bool pause ();
    /* seek (pos, abs) seek positon in deci-seconds */
    virtual bool seek (int pos, bool absolute);
    /* volume from 0 to 100 */
    virtual bool volume (int pos, bool absolute);
    /* saturation/hue/contrast/brightness from -100 to 100 */
    virtual bool saturation (int pos, bool absolute);
    virtual bool hue (int pos, bool absolute);
    virtual bool contrast (int pos, bool absolute);
    virtual bool brightness (int pos, bool absolute);
    virtual void setVideoWindow (int x, int y, int w, int h);
    KMPLAYER_NO_EXPORT const char * name () { return m_name; }
    void scheduledStateChanged ();
    ProcessNotify * processNotify () { return m_notify; }
    void setState (State newstate);
    void setWidth (int w) { m_width = w; }
    void setHeight (int h) { m_height = h; }
    void updateAspects ();
    bool hasVideo () const { return playing () && !no_video; }
    bool hasXv () const { return has_xv; }
protected:
    //Source * m_source;
    //Settings * m_settings;
    NodePtrW m_mrl;
    State m_state;
    State m_old_state;
    //KProcess * m_process;
    String m_url;
    ProcessNotify * m_notify;
    unsigned int m_viewer;
    const char * m_name;
    int start_timer;
    int m_length;
    int m_position;
    int m_width;
    int m_height;
    bool no_video;
    bool has_xv;
    //int m_request_seek;
    //const char ** m_supported_sources;
};

/*
 * Listener for Process events
 */
class ProcessNotify {
public:
    virtual ~ProcessNotify () {}
    virtual void stateChanged (Process * proc, Process::State os, Process::State ns)=0;
    virtual void errorMsg (const String & msg) = 0;
    virtual void infoMsg (const String & msg) = 0;
    virtual void setLoading (int perc) = 0;
    virtual void setPosition (int pos) = 0;
    virtual void setLength (int len) = 0;
    virtual void setAspect (float aspect) = 0;
};

/*
 * MM backend using MPlayer
 */
class MPlayer : public Process {
public:
    MPlayer (ProcessNotify * ctrl, bool xv);
    ~MPlayer ();
    void initProcess (unsigned int);
    virtual bool play (NodePtr mrl);
    virtual bool pause ();
    virtual bool ready (unsigned int);
    virtual bool stop ();
    virtual bool quit ();
    virtual bool seek (int pos, bool absolute);
    virtual bool volume (int pos, bool absolute);
    virtual bool saturation (int pos, bool absolute);
    virtual bool hue (int pos, bool absolute);
    virtual bool contrast (int pos, bool absolute);
    virtual bool brightness (int pos, bool absolute);
    virtual void setVideoWindow (int x, int y, int w, int h);
    // for callback functions
    void processExited (GPid pid);
    void writeEvent ();
    void readEvent (GIOChannel * src);
private:
    bool sendCommand (const String &);
    typedef std::list <String> StringList;
    StringList commands;
    String command_chunk;
    Rect m_rect;
    GIOChannel *pin, *pout, *perr;
    int win, wout, werr;
    int request_seek;
    char outbuf[512];
    bool request_quit;
    GPid process_id;
};

/*
 * MM backend using dbus to control com.nokia.osso_media_server
 */
class OssoMediaServer : public Process {
public:
    OssoMediaServer (ProcessNotify * ctrl, osso_context_t * context, bool xv);
    ~OssoMediaServer ();
    void initProcess (unsigned int);
public: //slots:
    virtual bool play (NodePtr mrl);
    virtual bool pause ();
    virtual bool ready (unsigned int);
    virtual bool stop ();
    virtual bool quit ();
    virtual bool seek (int pos, bool absolute);
    virtual bool volume (int pos, bool absolute);
    virtual bool saturation (int pos, bool absolute);
    virtual bool hue (int pos, bool absolute);
    virtual bool contrast (int pos, bool absolute);
    virtual bool brightness (int pos, bool absolute);
    virtual void setVideoWindow (int x, int y, int w, int h);
    // called from dbus callbacks
    void processPlaying ();
    void processPaused ();
    void processStopped ();
    void processBuffering (int perc);
    void setLength (int len);
    const char * currentMediaServer () const { return current_service; }
    osso_context_t * ossoContext () const { return m_osso_context; }
    bool retry ();
    DBusPendingCall * play_pending;
    DBusPendingCall * pos_pending;
    DBusPendingCall * seek_pending;
    static const char *myname;
    void updateVideoSize ();
private:
    osso_context_t * m_osso_context;
    Rect m_rect;  // see setVideoWindow (..)
    Rect m_cur_rect; // last real setting
    const char * current_service;
    bool have_played;
    bool try_audio_only;
    int retry_timer;
    int pos_timer;
};

} // namespace

#endif //_KMPLAYERPROCESS_H_
