//#include <config.h>

#include <string.h>
#include <errno.h>
#include <glob.h>

#define DISABLE_PRINTING
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
extern "C" {
#include <gtkhtml/gtkhtml.h>
}
#include <hildon-widgets/hildon-program.h>
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-seekbar.h>
//#include <gdk/gdk.h>
//#include <pango/pango-font.h>

#include "kmplayerplaylist.h"
#include "kmplayer_lists.h"
#include "kmplayer_smil.h"
#include "kmplayercontrol.h"
#include "kmplayerprocess.h"
#include "kmplayer.h"
#include "viewarea.h"

using namespace KMPlayer;

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

namespace KMPlayer {
    class Control : public PlayListNotify, public ProcessNotify, public AsyncJobInterface {
        Application * m_app;
        Process * m_process;
        Rect video_rect;
        int m_doc_timer;
        struct ResolveInfo {
            ResolveInfo (NodePtr mrl, Job * j, SharedPtr <ResolveInfo> & n)
                : resolving_mrl (mrl), job (j), next (n) {}
            NodePtrW resolving_mrl;
            Job * job;
            //ByteArray data;
            TextStream data;
            SharedPtr <ResolveInfo> next;
        };
        SharedPtr <ResolveInfo> m_resolve_info;
        bool have_progress_banner;
    public:
        Control (Application * a);
        ~Control ();
        void play ();
        void stop ();
        void playCurrent ();
        String currentMrl ();
        void readPlaylist (NodePtr root, TextStream & in);
        void updatePlaylistView (bool force=false);
        /**
         * PlayListNotify implementation
         */
        bool requestPlayURL (NodePtr mrl);
        bool resolveURL (NodePtr mrl);
        bool setCurrent (NodePtr);
        bool applyBackRequest();
        void stateElementChanged (Node * n, Node::State os, Node::State ns);
        SurfacePtr getSurface (NodePtr node);
        void setInfoMessage (const String & msg);
        void updateViewAreaSizes ();
        void bitRates (int & prefered, int & maximal);
        void setTimeout (int ms);
        void setProcess (Process * p) { m_process = p; }
        Process * process () { return m_process; }
        bool docTimerEvent ();

        // ProcessNotify
        void stateChanged (Process * p, Process::State os, Process::State ns);
        void errorMsg (const String & msg);
        void infoMsg (const String & msg);
        void setLoading (int perc);
        void setPosition (int pos) ;
        void setLength (int len);
        void setAspect (float aspect);

        NodePtrW back_request;
        int paint_timer;
        int update_tree_timer;
        int blanking_timer;
        int width, height;
        float aspect;
        Rect paint_rect;
        bool in_progress_update;
    private:
        virtual void jobData (Job * job, ByteArray & data);
        virtual void jobResult (Job * job);
    };
}

static bool isPlayListMime (const String & mime) {
    const char * mimestr = (const char *) mime;
    //debugLog () << "isPlayListMime  mime:" << mimestr << endl;
    return mimestr && (!strcmp (mimestr, "audio/mpegurl") ||
            !strcmp (mimestr, "audio/x-mpegurl") ||
            !strncmp (mimestr, "video/x-ms", 10) ||
            !strncmp (mimestr, "audio/x-ms", 10) ||
            //!strcmp (mimestr, "video/x-ms-wmp") ||
            //!strcmp (mimestr, "video/x-ms-asf") ||
            //!strcmp (mimestr, "video/x-ms-wmv") ||
            //!strcmp (mimestr, "video/x-ms-wvx") ||
            //!strcmp (mimestr, "video/x-msvideo") ||
            !strcmp (mimestr, "video/x-real") ||
            !strcmp (mimestr, "audio/x-scpls") ||
            !strcmp (mimestr, "audio/x-pn-realaudio") ||
            !strcmp (mimestr, "audio/vnd.rn-realaudio") ||
            !strcmp (mimestr, "audio/m3u") ||
            !strcmp (mimestr, "audio/x-m3u") ||
            !strncmp (mimestr, "text/", 5) ||
            (!strncmp (mimestr, "application/", 12) &&
             strstr (mimestr + 12,"+xml")) ||
            !strncasecmp (mimestr, "application/smil", 16) ||
            !strncasecmp (mimestr, "application/xml", 15) ||
            !strcmp (mimestr, "application/news_reader") ||
            !strcmp (mimestr, "application/rss+xml") ||
            !strcmp (mimestr, "application/atom+xml") ||
            !strcmp (mimestr, "application/x-mplayer2"));
}

static bool isAudioOnlyMime (const String & mime) {
    const char * mimestr = (const char *) mime;
    return mimestr && (!strncmp ((const char *) mime, "audio/", 6));
}

static gboolean playSchedule (void * ctr) {
    static_cast <Control *> (ctr)->play ();
    return false;
}

static gboolean playCurrentSchedule (void * ctr) {
    static_cast <Control *> (ctr)->playCurrent ();
    return false;
}

Control::Control (Application * a)
    : m_app (a), m_process (0L), m_doc_timer (0), have_progress_banner (false),
      paint_timer (0), update_tree_timer (0), blanking_timer (0),
      in_progress_update (false) {}

Control::~Control () {
    delete m_process;
    m_process = 0L;
}

gboolean cb_blankingTimer (void * p) {
    Application * app = static_cast <Application *> (p);
    if (app->playlist_data->current && 
            (app->playlist_data->current->id == SMIL::id_node_smil ||
             (app->control->process ()->playing () &&
              app->control->process ()->hasVideo ()))) {
        osso_display_blanking_pause (app->osso_context);
        return 1;
    }
    app->control->blanking_timer = 0;
    return 0;
}

gboolean cb_docTimer (void * p) {
    return static_cast <Control *> (p)->docTimerEvent ();
}

gboolean cb_scheduledUpdateTree (void * vp) {
    Application * app = static_cast <Application *> (vp);
    Control * ctr = app->control;
    if (ctr->process ()) {
        if (!app->playlist_data->document)
            app->playlist_data->document = new Document(String(), app->control);
        updateTree (app, 0);
    }
    ctr->update_tree_timer = 0;
    return false;
}

void Control::play () {
    if (!m_app->playlist_data->document)                // nothing to play
        return;
    if (m_app->playlist_data->document->active ()) {    // is playing
        if (m_app->playlist_data->document->state == Node::state_deferred)
            m_app->playlist_data->document->undefer (); // unpause
        else
            m_app->playlist_data->document->defer ();   // pause
    } else {
        debugLog () << "play" << endl;
        gtk_paned_set_position (m_app->vsplit_view, VSPLIT);
        gtk_container_resize_children (GTK_CONTAINER (m_app->vsplit_view));
        if (!applyBackRequest () &&
                m_app->playlist_data->document) {
            playCurrent ();
        }
    }
}

void Control::stop () {
    debugLog () << "stop" << endl;
    for (SharedPtr<ResolveInfo> rinfo =m_resolve_info; rinfo; rinfo=rinfo->next)
        rinfo->job->kill ();
    m_resolve_info = 0L;
    setLoading (100); // no pogress bar
    if (blanking_timer) {
        g_source_remove (blanking_timer);
        blanking_timer = 0;
    }
    back_request = 0L;
    if (m_app->playlist_data->document &&
            m_app->playlist_data->document->state > Node::state_init) {
        m_app->playlist_data->document->reset ();
        gtk_paned_set_position (m_app->vsplit_view, 40);
        setCurrent (0L);
    }
    m_app->viewarea->setAspect (0.0);
    m_process->quit ();
    gdk_window_clear (m_app->viewarea->videoWindow ());
    gtk_widget_queue_draw (m_app->infopanel_view);
    m_app->normalScreen ();
    setLength (0);
    setPosition (0);
}

String Control::currentMrl () {
    Mrl * mrl = m_app->playlist_data->current ? m_app->playlist_data->current->mrl () : 0L;
    if (!mrl)
        return String ();
    return mrl->absolutePath ();
}

void Control::playCurrent () {
    String url = currentMrl ();
    // set title url;
    aspect = 0.0;
    width = height = 0;
    //debugLog () << "playCurrent" << !!m_app->playlist_data->document << !m_app->playlist_data->document->active () << endl;
    // view ())->videoStop (); //show buttonbar
    if (m_app->playlist_data->document &&
            !m_app->playlist_data->document->active ()) {
        if (!m_app->playlist_data->current)
            m_app->playlist_data->current = m_app->playlist_data->document;
        else { // ugly code duplicate w/ back_request
            for (NodePtr p = m_app->playlist_data->current->parentNode(); p; p = p->parentNode())
                p->setState (Element::state_activated);
        }
        m_app->playlist_data->current->activate ();
        updateViewAreaSizes ();
    } else if (!m_app->playlist_data->current) {
        //debugLog () << "playCurrent no current" << endl;
        stop ();
        ; //emit endOfPlayItems ();
    } else if (m_app->playlist_data->current->state ==Element::state_deferred) {
        //debugLog () << "playCurrent set undefer" << endl;
        ;//m_app->playlist_data->current->undefer ();
    } else if (m_process->state () == Process::NotRunning) {
        //debugLog () << "playCurrent set ready" << endl;
        m_process->ready (GDK_DRAWABLE_XID (GDK_DRAWABLE (
                        m_app->viewarea->videoWindow ())));
    } else if (!m_app->playlist_data->current->active ()) {
        m_app->playlist_data->current->activate ();
        updateViewAreaSizes ();
    } else {
        NodePtr mrl = back_request
            ? back_request->mrl ()->linkNode ()
            : m_app->playlist_data->current->mrl ()->linkNode ();
        //debugLog () << "playCurrent " << m_process->playing () << " " << mrl->mrl ()->resolved << endl;
        if (m_process->playing ()) {
            errorLog() << "playCurrent and backend still playing" << endl;
            if (m_process->mrl () != mrl)
                m_process->stop ();
            else
                return; // FIXME
        }
        back_request = 0L;
        if (!mrl->mrl ()->src.isEmpty () && mrl->mrl ()->resolved) {
            updateViewAreaSizes ();
            if (m_process->hasXv ())
                XSync (GDK_DISPLAY_XDISPLAY (gtk_widget_get_display (m_app->viewarea->viewWidget ())), false);
            m_process->play (mrl->mrl ()->linkNode ());
        } else
            errorLog() << "playCurrent and mrl not resolved" << endl;
    }
}

bool Control::applyBackRequest() {
    if (!back_request || back_request->mrl ()->view_mode == Mrl::WindowMode)
        return false;
    if (m_app->playlist_data->current)
        m_app->playlist_data->document->reset ();
    m_app->playlist_data->current = 0L;
    if (back_request->document() != m_app->playlist_data->document) {
        //debugLog () << "playlist/recent" << endl;
        NodePtr n = back_request;
        back_request = 0L;
        n->activate();
    } else {
        setCurrent (back_request);
        for (NodePtr p = back_request->parentNode(); p; p = p->parentNode())
            p->setState (Element::state_activated);
        back_request = 0L;
        g_timeout_add(0, playCurrentSchedule, static_cast<void*>(this));
    }
    return true;
}
    
bool Control::resolveURL (NodePtr m) {
  //debugLog() << "resolveURL " << !!back_request << " " << m->mrl()->src<<endl;
    if (back_request && back_request->mrl ()->view_mode == Mrl::SingleMode)
        return false; // block auto playlist traversal on jump
    Mrl * mrl = m->mrl ();
    if (!mrl || mrl->src.isEmpty ())
        return true;
    int depth = 0;
    for (NodePtr e = m->parentNode (); e; e = e->parentNode ())
        ++depth;
    if (depth > 40)
        return true;
    URL url (mrl->absolutePath ());
    String mimestr = mrl->mimetype;
    if (mimestr.isEmpty ()) {
        //mimestr = MimeType::findByURL (url);
        if (!mimestr.isEmpty ())
            mrl->mimetype = mimestr;
    }
    debugLog() << "resolveURL " << url << " mime:" << mimestr << endl;
    if (mimestr.isEmpty ()) {
        for (NodePtr p = mrl->parentNode (); p; p = p->parentNode ()) {
            Mrl * m = p->mrl ();
            if (m && m->linkNode ()->audio_only) {
                mrl->linkNode ()->audio_only = true;
                break;
            }
        }
    } else
        mrl->linkNode ()->audio_only = isAudioOnlyMime (mimestr);
    bool maybe_playlist = mimestr.isEmpty() || isPlayListMime (mimestr);
    if (url.isLocalFile ()) {
        File file (url.path ());
        struct stat stats;
        if (!file.exists ()) {
            errorLog () << url.path () << " not found" << endl;
            return true;
        }
        //debugLog () << "opening " << url << " mime:" << mimestr << endl;
        if (maybe_playlist && file.size () < 2000000) {
            char buf[256];
            off_t nr = file.readBlock (buf, sizeof (buf) - 1);
            if (nr > 0) {
                buf[nr] = 0;
                String mimestr = MimeType::findByContent (buf, nr);
                if (mimestr.hasPrefix (String ("text/")) &&
                        strncmp (buf, "RIFF", 4)) {
                    ByteArray ba = file.readAll ();
                    String data (ba.data (), ba.size ());
                    if (!data.isEmpty ()) {
                        TextStream in (data);
                        readPlaylist (m, in);
                    }
                }
            }
        }
    } else if ((maybe_playlist && // exclude mms/rtsp/rtp 
                url.protocol ().compare (String ("mms")) &&
                url.protocol ().compare (String ("rtsp")) &&
                url.protocol ().compare (String ("rtp"))) ||
            (mimestr.isEmpty () && // try http(s) ones too
             url.protocol ().hasPrefix (String ("http")))) {
        debugLog () << "maybe playlist, network " << endl;
        Job * job = asyncGet (this, url.url ());
        m_resolve_info = new ResolveInfo (m, job, m_resolve_info);
        return false; // wait for network operation ..
    }
    return true;
}

void Control::jobData (Job * job, ByteArray & data) {
    SharedPtr <ResolveInfo> rinfo = m_resolve_info;
    while (rinfo && rinfo->job != job)
        rinfo = rinfo->next;
    if (!rinfo) {
        warningLog () << "Spurious kioData" << endl;
        return;
    }
    //debugLog () << "jobData of " << data.size () << endl;
    if (rinfo->data.isEmpty ()) {
        String mimestr = MimeType::findByContent (data.data (), data.size ());
        if (mimestr.isEmpty() || !mimestr.hasPrefix (String ("text/")) ||
                (data.size () > 4 && !strncmp (data.data (), "RIFF", 4)) ||
                !rinfo->resolving_mrl) {
            job->kill (false); // not quiet, wants jobResult
            return;
        } else if (rinfo->resolving_mrl->mrl ()->mimetype.isEmpty ())
            rinfo->resolving_mrl->mrl ()->mimetype = mimestr;
    }
    rinfo->data.readRawBytes (data.data (), data.size ());
}

void Control::jobResult (Job * job) {
    debugLog () << "jobResult" << endl;
    SharedPtr <ResolveInfo> previnfo, rinfo = m_resolve_info;
    while (rinfo && rinfo->job != job) {
        previnfo = rinfo;
        rinfo = rinfo->next;
    }
    if (!rinfo) {
        warningLog () << "Spurious kioData" << endl;
        return;
    }
    if (previnfo)
        previnfo->next = rinfo->next;
    else
        m_resolve_info = rinfo->next;
    if (rinfo->resolving_mrl) {
        if (!rinfo->data.isEmpty ())
            readPlaylist (rinfo->resolving_mrl, rinfo->data);
    //debugLog () << "set resolved " << rinfo->resolving_mrl->mrl ()->src << endl;
        rinfo->resolving_mrl->mrl ()->resolved = true;
        rinfo->resolving_mrl->undefer ();
    }
}

void Control::readPlaylist (NodePtr root, TextStream & in) {
    String line;
    Mrl * mrl = root->mrl ();
    do {
        line = in.readLine ().stripWhiteSpace ();
    } while (!in.atEnd () && line.isEmpty ());
    if (!mrl || line.isEmpty ())
        return;
    NodePtr cur_elm = root;
    if (mrl->isPlayable ())
        cur_elm = mrl->linkNode ();
    if (mrl->mimetype.hasPrefix ("text/") && line.find ("[playlist]") > -1)
        mrl->mimetype = "audio/x-scpls";
    if (mrl->mimetype == String ("audio/x-scpls")) {
        bool groupfound = false;
        int nr = -1;
        struct Entry {
            String url, title;
        } * entries = 0L;
        do {
            if (line.hasPrefix (String ("[")) && line.hasSuffix (String ("]"))) {
                if (!groupfound && line.mid (1, line.length () - 2).stripWhiteSpace () == String ("playlist"))
                    groupfound = true;
                else
                    break;
                //debugLog () << "Group found: " << line << endl;
            } else if (groupfound) {
                int eq_pos = line.find (Char ('='));
                if (eq_pos > 0) {
                    if (line.lower ().hasPrefix (String ("numberofentries"))) {
                        nr = line.mid (eq_pos + 1).stripWhiteSpace ().toInt ();
                        debugLog () << "numberofentries : "<< nr<< endl;
                        if (nr > 0 && nr < 1024)
                            entries = new Entry[nr];
                        else
                            nr = 0;
                    } else if (nr > 0) {
                        String ll = line.lower ();
                        if (ll.hasPrefix (String ("file"))) {
                            int i = line.mid (4, eq_pos-4).toInt ();
                            if (i > 0 && i <= nr)
                                entries[i-1].url = line.mid (eq_pos + 1).stripWhiteSpace ();
                        } else if (ll.hasPrefix (String ("title"))) {
                            int i = line.mid (5, eq_pos-5).toInt ();
                            if (i > 0 && i <= nr)
                                entries[i-1].title = line.mid (eq_pos + 1).stripWhiteSpace ();
                        }
                    }
                }
            }
            line.truncate (0);
            while (!in.atEnd () && line.isEmpty ())
                line = in.readLine ().stripWhiteSpace ();
        } while (!line.isEmpty ());
        for (int i = 0; i < nr; i++)
            if (!entries[i].url.isEmpty ()) {
                Mrl * m = new GenericURL (m_app->playlist_data->document, URL::decode_string (entries[i].url), entries[i].title);
                cur_elm->appendChild (m);
                m->linkNode ()->audio_only = true;
            }
        delete [] entries;
    } else if (line.stripWhiteSpace ().hasPrefix (Char ('<'))) {
        readXML (cur_elm, in, line);
        //cur_elm->normalize ();
        if (m_app->playlist_data->document &&
                m_app->playlist_data->document->firstChild ()) {
            // SMIL documents have set its size of root-layout
            Mrl * mrl = m_app->playlist_data->document->firstChild ()->mrl ();
            if (mrl) {
                width = mrl->width;
                height = mrl->height;
                aspect = height > 0 ? width/height : 0.0;
            }
        }
    } else if (line.lower () != String ("[reference]")) do {
        if (line.lower ().hasPrefix (String ("asf ")))
            line = line.mid (4).stripWhiteSpace ();
        if (!line.isEmpty () && !line.hasPrefix (Char ('#')))
            cur_elm->appendChild (new GenericURL (m_app->playlist_data->document, line));
        line.truncate (0);
        while (!in.atEnd () && line.isEmpty ())
            line = in.readLine ().stripWhiteSpace ();
    } while (!line.isEmpty () && line != String ("--stop--")); /* TODO && m_document.size () < 1024 / * support 1k entries * /);*/
}

bool Control::requestPlayURL (NodePtr mrl) {
    debugLog () << "requestPlayURL " << mrl->mrl()->src << (mrl->mrl ()->view_mode == Mrl::SingleMode) << endl;
    if (process ()->state () > Process::Ready) {
        if (process ()->mrl () == mrl->mrl ()->linkNode ())
            return true;
        back_request = mrl; // still playing, schedule it
        process ()->stop ();
    } else {
        if (mrl->mrl ()->view_mode == Mrl::SingleMode)
            setCurrent (mrl);
        else
            back_request = mrl;
        g_timeout_add (0, playCurrentSchedule, static_cast <void*>(m_app->control));
    }
    return true;
}

void Control::setTimeout (int ms) {
    //debugLog () << "Control::setTimeout " << ms << endl;
    if (m_doc_timer)
        if (!g_source_remove (m_doc_timer))
            errorLog () << "failed to kill doc timer " << m_doc_timer << endl;
    m_doc_timer = ms > -1 ? g_timeout_add (ms, cb_docTimer, static_cast <void *> (this)) : 0;
}

bool Control::docTimerEvent () {
    int id = m_doc_timer;
    if (m_app->playlist_data->document)
        m_app->playlist_data->document->document ()->timer (); // will call setTimeout()
    return id == m_doc_timer;
}

bool Control::setCurrent (NodePtr n) {
    debugLog () << "setCurrent " << (n ? n->nodeName () : "-") << endl;
    if (m_app->playlist_data->current == n)
        return true;
    m_app->playlist_data->current = n;
    updatePlaylistView ();
    Mrl * mrl = n ? n->mrl () : 0L;
    if (mrl && (m_app->playlist_data->current->id == SMIL::id_node_smil ||
                !mrl->linkNode ()->audio_only)) {
        if (!blanking_timer)
            blanking_timer = g_timeout_add (25000, cb_blankingTimer, m_app);
        osso_display_blanking_pause (m_app->osso_context); // 60s delay
    } else if (blanking_timer) {
        g_source_remove (blanking_timer);
        blanking_timer = 0;
    }
    if (mrl) {
        String url;
        while (mrl) {
            if (mrl->src.hasPrefix ("Playlist://"))
                break;
            if (mrl->view_mode == Mrl::SingleMode)
                url = mrl->absolutePath ();
            NodePtr p = mrl->parentNode ();
            for (mrl = 0L; p; p = p->parentNode ())
                if (p->mrl ()) {
                    mrl = p->mrl ();
                    break;
                }
        }
        insertLocation (m_app, url);
    }
    return true;
}

static char * stateDesc [] = {
    "init", "deferred", "activated", "began", "finished", "deactivated"
};

void Control::stateElementChanged (Node* n, Node::State os, Node::State ns) {
    //debugLog () << "state " << n->nodeName () << " changed to " << stateDesc [int (ns)] << endl;
    if (!m_process) // destructing
        return;
    if (ns == Node::state_deactivated &&
            n == m_app->playlist_data->document && !back_request) {
        stop ();
        m_app->setPlaying (false);
    } else if ((ns == Node::state_deactivated || ns == Node::state_finished) &&
            n->mrl () && process ()->mrl ().ptr () == n->mrl()->linkNode ()) {
        process ()->stop ();
        updateViewAreaSizes (); // move away video widget
    } else if ((ns == Node::state_deferred ||
                (os == Node::state_deferred && ns > Node::state_deferred)) &&
            n == m_app->playlist_data->document) {
    debugLog () << "doc state " << stateDesc [int (os)] << " changed to " << stateDesc [int (ns)] << endl;
        if ((ns == Node::state_deferred &&
                    m_process->state() != Process::Paused) ||
                (ns > Node::state_deferred &&
                 m_process->state() == Process::Paused))
            m_process->pause ();
        m_app->setPlaying (ns > Node::state_deferred);
    } else if (ns == Node::state_activated) {
        if (n == m_app->playlist_data->document)
            m_app->setPlaying (true);
        if (n->id == SMIL::id_node_body)
            updateViewAreaSizes (); // move away video widget
        if ((!m_app->playlist_data->current ||
                 !m_app->playlist_data->current->active() ||
                 !m_app->playlist_data->current->isPlayable ()) &&
                n->isPlayable () &&
                n->mrl ()->view_mode == Mrl::SingleMode)
            setCurrent (n);
    }
    if (n->expose () &&
            (ns == Node::state_activated || ns == Node::state_deactivated))
        updatePlaylistView ();
}

void Control::updateViewAreaSizes () {
    if (!m_app->viewarea->videoWindow () || !m_process)
        return;
    m_app->viewarea->resizeEvent();
    Rect video_rect = m_app->viewarea->videoGeometry ();
    int x = video_rect.x ();
    int y = video_rect.y ();
    for (GdkWindow * win = gdk_window_get_parent (
                m_app->viewarea->videoWindow ());
            win;
            win = gdk_window_get_parent (win)) {
        int x1, y1;
        gdk_window_get_position (win, &x1, &y1);
        x += x1;
        y += y1;
    }
    debugLog () << "updateViewAreaSizes " << x << ", " << y << " " << video_rect.width () << "x" << video_rect.height () << endl;
    m_process->setVideoWindow (x, y,
            video_rect.width (), video_rect.height ());
}

void Control::setInfoMessage (const String & msg) {
    m_app->setInfoContent ((const char *) msg, false);
}

SurfacePtr Control::getSurface (NodePtr node) {
    return m_app->viewarea->getSurface (node);
}

void Control::updatePlaylistView (bool force) {
    if (force) {
        if (update_tree_timer) {
            g_source_remove (update_tree_timer);
            update_tree_timer = 0;
        }
        updateTree (m_app, 0);
    } else if (!update_tree_timer)
        update_tree_timer = g_timeout_add (200, cb_scheduledUpdateTree, m_app);
}

static const char * statemap [] = {
        "Not Running", "Ready", "Buffering", "Playing", "Paused"
};

void Control::stateChanged (Process * p, Process::State os, Process::State ns) {
    debugLog () << "Process stateChanged " << statemap [(int)os] << " -> " << statemap [(int)ns] << endl;
    if (!m_process) // destructing
        return;
    if (!p->mrl () && ns > Process::Ready) {
        p->stop (); // reschedule to the NotRunning state
    } else if (ns == Process::NotRunning) {
        if (m_app->playlist_data->document &&
                m_app->playlist_data->document->state > Node::state_init) {
            stop (); // this should not happen
        } else if (blanking_timer) {
            g_source_remove (blanking_timer);
            blanking_timer = 0;
        }
        if (!m_app->playlist_data->edit_mode)
            m_app->setInfoContent ("", false);
    } else if (ns == Process::Ready) {
        setLoading (100); // move possible buffer info dialog
        if (os > Process::Ready) {
            setLength (0);
            setPosition (0);
            m_app->viewarea->setAspect (0.0);
            NodePtr node = p->mrl (); // p->mrl is weak, needs check
            Mrl * mrl = node ? node->mrl () : 0L;
            if (mrl && !applyBackRequest () &&  // if cause is no jump
                    mrl->active ()) {        // if cause is eof
                mrl->endOfFile ();           // set node to finished
                if (back_request)            // overlapping audio/video in SMIL
                    g_timeout_add(0, playCurrentSchedule, (void *) this);
            }
        } else
            g_timeout_add (0, playCurrentSchedule, static_cast <void *> (this));
    } else if (ns == Process::Buffering) {
        if (p->mrl ()->mrl ()->view_mode != Mrl::SingleMode)
            p->mrl ()->defer ();
    } else if (ns == Process::Paused &&
            m_app->playlist_data->document->state !=Node::state_deferred) {
        if (gtk_window_has_toplevel_focus (GTK_WINDOW (m_app->window_fullscreen)) ||
                gtk_window_has_toplevel_focus (GTK_WINDOW (m_app->window)))
            m_process->pause ();
        else
            m_app->playlist_data->document->defer ();
    } else if (ns == Process::Playing) {
        if (p->mrl ()->state == Element::state_deferred)
            p->mrl ()->undefer ();
        setLoading (100); // move possible buffer info dialog
        NodePtr recents = m_app->recents;
        m_app->recents->defer ();
        NodePtr c = recents->firstChild ();
        String url = m_app->playlist_data->document->mrl()->src;
        if (url != String ("Playlist://") && (!c || c->mrl()->src != url)) {
            recents->insertBefore (new Recent (recents, m_app, url), c);
            c = recents->firstChild ()->nextSibling ();
            int count = 1;
            KMPlayer::NodePtr more;
            while (c) {
                if (c->id == id_node_recent_node &&
                        c->mrl ()->src == url) {
                    KMPlayer::NodePtr tmp = c->nextSibling ();
                    recents->removeChild (c);
                    c = tmp;
                } else {
                    if (c->id == id_node_group_node)
                        more = c;
                    c = c->nextSibling ();
                    count++;
                }
            }
            if (!more && count > 10) {
                more = new Group (recents, m_app, "More ...");
                recents->appendChild (more);
            }
            if (more) {
                KMPlayer::NodePtr item;
                if (count > 10) {
                    KMPlayer::NodePtr item = more->previousSibling ();
                    recents->removeChild (item);
                    more->insertBefore (item, more->firstChild ());
                }
                if (more->firstChild ())
                    c = more->firstChild ()->nextSibling ();
                count = 0;
                while (c) {
                    if (c->id == id_node_recent_node &&
                            c->mrl ()->src == url) {
                        KMPlayer::NodePtr tmp = c->nextSibling ();
                        more->removeChild (c);
                        c = tmp;
                    } else {
                        c = c->nextSibling ();
                        count++;
                    }
                }
                if (count > 50)
                    more->removeChild (more->lastChild ());
            }
            updateTree (m_app, m_app->recents_tree_id);
        }
    }
}

void Control::errorMsg (const String & msg) {
    gtk_infoprint (GTK_WINDOW (m_app->window), (const char *) msg);
}

void Control::infoMsg (const String & msg) {
    m_app->setInfoContent ((const char *) msg, false);
}

void Control::setLoading (int perc) {
    if (perc >= 100) {
        if (have_progress_banner)
            gtk_banner_close (GTK_WINDOW (m_app->window));
        have_progress_banner = false;
    } else {
        if (!have_progress_banner) {
            gtk_banner_show_bar (GTK_WINDOW (m_app->window), "Buffering");
            have_progress_banner = true;
        }
        gtk_banner_set_fraction (GTK_WINDOW (m_app->window), 1.0 * perc / 100);
    }
}

void Control::setPosition (int p) {
    if (p < hildon_seekbar_get_total_time(HILDON_SEEKBAR(m_app->progress_bar))){
        in_progress_update = true;
        if (have_progress_banner)
            setLoading (100);
        hildon_seekbar_set_position (HILDON_SEEKBAR (m_app->progress_bar), p);
        in_progress_update = false;
    }
}

void Control::setLength (int len) {
    hildon_seekbar_set_total_time (HILDON_SEEKBAR (m_app->progress_bar), len);
    hildon_seekbar_set_fraction (HILDON_SEEKBAR (m_app->progress_bar), len);
    if (len > 50) {
        gtk_widget_hide (GTK_WIDGET (m_app->tb_location));
        gtk_widget_show (GTK_WIDGET (m_app->tb_progress));
    } else {
        gtk_widget_show (GTK_WIDGET (m_app->tb_location));
        gtk_widget_hide (GTK_WIDGET (m_app->tb_progress));
    }
}

void Control::setAspect (float aspect) {
    debugLog () << "Control::setAspect " << aspect << endl;
    m_app->viewarea->setAspect (aspect);
    updateViewAreaSizes ();
}

void cb_view_area_configure (GtkWidget *widget, GdkEventConfigure *event, Application * app) {
    //debugLog () << "cb_view_area_configure " << event->width<< "x" << event->height << endl;
    app->control->updateViewAreaSizes ();
}

void Control::bitRates (int & prefered, int & maximal) {
}

PlayListData::PlayListData(short i, PlayListData * n)
    : id (i), show_all (false), edit_mode (false), next (n) {}

Application::Application ()
    : playlist_view (0L), viewarea (0), html_infopanel_view (0L),
      m_config (0L), playlist_data (0L), control (new Control (this)),
      in_tree_update (false), fullscreen (false), playing (false)
{
}

Application::~Application () {
    delete control;
    delete viewarea;
    if (playlist_data->document)
        playlist_data->document->document()->dispose ();
    delete playlist_data;
    delete m_config;
}

void Application::initControl () {
    //control->setProcess (new OssoMediaServer (control, osso_context));
    unsigned int ver, rel, req, evb, err;
    glob_t gt;
    bool has_xv = !glob ("/usr/lib/libXv.so*", GLOB_ERR|GLOB_NOSORT, 0, &gt);
    globfree (&gt);
    viewarea = new ViewArea;
    media_server = new OssoMediaServer (control, osso_context, has_xv);
    mplayer = new MPlayer (control, has_xv);
    if (!strcmp (OssoMediaServer::myname, (const char *)
            config()->readEntry (GCONF_KEY_PLAYER, OssoMediaServer::myname))) {
        control->setProcess (media_server);
        gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu_item_media_server), true);
    } else {
        control->setProcess (mplayer);
        gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu_item_mplayer), true);
    }
}

void Application::jump (Node * n) {
    Mrl * mrl = n ? n->mrl () : 0L;
    if (mrl && mrl->view_mode == Mrl::SingleMode) {
        if (!mrl->isPlayable ())
            return;
        //debugLog() << "jump " << mrl->src << endl;
        control->back_request = n;
        if (control->process ()->playing ())
            control->process ()->stop ();
        else
            control->applyBackRequest();
    } else
        control->updatePlaylistView ();
}

void Application::fullScreen () {
    if (!playlist_data->document ||
            !playlist_data->document->active ()) {
        return;
    }
    if (!fullscreen) {
        gtk_widget_show_all (GTK_WIDGET (window_fullscreen));
        fullscreen = true;
    }
}

void Application::normalScreen () {
    if (fullscreen) {
        gtk_widget_hide (GTK_WIDGET (window_fullscreen));
        fullscreen = false;
    }
}

Config * Application::config () {
    if (!m_config)
        m_config = new Config;
    return m_config;
}

void Application::setInfoContent (const char * utf8, bool plaintext) {
    if (!plaintext && utf8[0]) {
        if (!m_config)
            m_config = new Config; // force GConf init
        if (isplit_view == info_viewport->parent)
            gtk_container_remove (GTK_CONTAINER (isplit_view), info_viewport);
        if (isplit_view != html_viewport->parent)
            gtk_container_add (GTK_CONTAINER (isplit_view), html_viewport);
        gtk_html_load_from_string (GTK_HTML (html_infopanel_view), utf8, -1);
    } else {
        if (isplit_view != info_viewport->parent) {
            gtk_container_remove (GTK_CONTAINER (isplit_view), html_viewport);
            gtk_container_add (GTK_CONTAINER (isplit_view), info_viewport);
        }
        gtk_text_buffer_set_text (gtk_text_view_get_buffer
                (GTK_TEXT_VIEW (infopanel_view)), utf8, -1);
    }
}

void cb_playlist_prev (GtkButton * button, Application * app) {
    if (app->playlist_data->current &&
            app->playlist_data->current->previousSibling ())
        app->jump (app->playlist_data->current->previousSibling ()); //FIXME
}

void cb_playlist_play (GtkButton * button, Application * app) {
    g_timeout_add (0, playSchedule, static_cast <void *> (app->control));
}

void cb_playlist_next (GtkButton * button, Application * app) {
    if (app->control->process ()->playing ())
        app->control->process ()->stop ();
}

void cb_playlist_stop (GtkButton * button, Application * app) {
    app->control->stop ();
}

void openFile (Application * app, const gchar * path) {
    app->control->stop ();
    if (app->playlist_data->document)
        app->playlist_data->document->document()->dispose ();
    app->playlist_data->document = new Document (String (path), app->control);
    app->url = path;
    g_timeout_add (0, playSchedule, static_cast <void *> (app->control));
    app->control->updatePlaylistView (true);
}

void openDocument (Application * app, NodePtr doc, NodePtr cur) {
    app->control->stop ();
    if (app->playlist_data->document)
        app->playlist_data->document->document()->dispose ();
    doc->document()->notify_listener = app->control;
    app->playlist_data->document = doc;
    app->url = doc->mrl()->pretty_name;
    setTitle ((const char *) app->url, app);
    app->control->back_request = cur;
    g_timeout_add (0, playSchedule, static_cast <void *> (app->control));
    app->control->updatePlaylistView (true);
}

void cb_openUrls (void *app, int argc, gchar **argv) {
}

bool saveFile (Application * app, const gchar * path) {
    int fd = open (path, O_WRONLY | O_CREAT | O_TRUNC, 0664);
    if (fd < 0)
        return false;
    String str = app->playlist_data->document ? app->playlist_data->document->childNodes ()->length () == 1 ? app->playlist_data->document->firstChild ()->outerXML () : app->playlist_data->document->outerXML () : "";
    ::write (fd, (const char *) str, str.length ());
    g_print ("saved %s %d bytes\n", path, str.length ());
    ::close (fd);
    app->url = path;
    return true;
}

void newFile (Application * app) {
    if (app->playlist_data->document) {
        app->playlist_data->document->reset ();
        app->playlist_data->document->document ()->dispose ();
    }
    app->playlist_data->document = new Document (String (), 0L);
    app->control->updatePlaylistView (true);
    app->url = String ();
}

void cb_mainview_deleted (GtkObject *, Application * app) {
    app->control->stop ();
    app->control->setProcess (0L);
    Recents *recents = static_cast <Recents *> (app->recents.ptr());
    if (recents->resolved &&
            recents->load_tree_version != recents->m_tree_version)
        recents->writeToFile (app->config ()->readEntry
                (GCONF_KEY_RECENT_FILE, "/home/user/recent.xml"));
    Playlist *playlist = static_cast <Playlist *> (app->playlists.ptr());
    if (playlist->resolved &&
            playlist->load_tree_version != playlist->m_tree_version)
        playlist->writeToFile (app->config ()->readEntry
                (GCONF_KEY_PLAYLIST_FILE, "/home/user/playlist.xml"));
    gtk_main_quit ();
    debugLog () << "destoyed" << endl;
}

gboolean cb_keyPressed (GtkWidget *, GdkEventKey * e, Application *app) {
    switch (e->keyval) {
        case GDK_F6: //HILDON_HARDKEY_FULLSCREEN:
            //if (app->control->process ()->playing ()) {
             //   int w, h;
                if (!app->fullscreen)
                    app->fullScreen ();
                else
                    app->normalScreen ();
            //}
            break;
        case GDK_F7: //HILDON_HARDKEY_INCREASE
            if (app->control->process ()->playing ())
                app->control->process ()->volume (2, false);
            break;
        case GDK_F8: //HILDON_HARDKEY_DECREASE
            if (app->control->process ()->playing ())
                app->control->process ()->volume (-2, false);
            break;
        case GDK_Left:
            if (app->control->process ()->playing ())
                app->control->process ()->seek (-200, false); // 20s
            break;
        case GDK_Right:
            if (app->control->process ()->playing ())
                app->control->process ()->seek (200, false); // 20s
            break;
        case GDK_Return: // HILDON_HARDKEY_SELECT
            rowSelected (app);
            break;
        default:
            KMPlayer::debugLog () << "Keypress " << e->keyval << KMPlayer::endl;
    }
    return false;
}

void progressValueChanged(GtkWidget *w, Application * app) {
    if (!app->control->in_progress_update) {
        int p = hildon_seekbar_get_position(HILDON_SEEKBAR (app->progress_bar));
        app->control->process ()->seek (p, true);
        debugLog () << "progressBarClicked " << p << endl;
    }
}

void toggleMediaServer (GtkCheckMenuItem *radio, Application * app) {
    Process * p =  gtk_check_menu_item_get_active (radio) 
        ? app->media_server : app->mplayer;
    app->control->setProcess (p);
    app->config()->writeEntry (GCONF_KEY_PLAYER, String (p->name ()));
}

static void setSuspendState (Application * a, bool suspend) {
    if (a->playlist_data->document &&
            a->playlist_data->document->active () &&
            (!a->control->process ()->playing () ||
             a->control->process()->mrl()->mrl()->view_mode ==Mrl::WindowMode ||
             a->control->process ()->hasVideo ())) {
        if (a->playlist_data->document->state == Node::state_deferred) {
            if (!suspend)
                ; // do nothing  a->playlist_data->document->undefer ();
        } else if (suspend)
            a->playlist_data->document->defer ();
    }
}

void mainMenuShow (GtkWidget *, Application * app) {
    setSuspendState (app, true);
}

void mainMenuHide (GtkWidget *, Application * app) {
    setSuspendState (app, false);
}

void programTopMost (GObject * self, GParamSpec *, Application * app) {
    HildonProgram *program = HILDON_PROGRAM (self);
    setSuspendState (app, !hildon_program_get_is_topmost (program));
}

/*
 * TODO
static void cb_iap (struct iap_event_t * event, void * arg) {
    if (event->type == OSSO_IAP_CONNECTED) {
    }
}

osso_iap_cb (cb_iap);
osso_iap_connect (OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, 0L);

'dbus-send --print-reply --dest=com.nokia.osso_browser /usr/bin/browser com.nokia.osso_browser.load_url string:%s' % url
*/
