// -*- c++ -*-
//------------------------------------------------------------------------------
// $Id: TextAlignment.cpp,v 1.6 2006/12/18 03:33:15 vlg Exp $
//------------------------------------------------------------------------------
//                            TextAlignment.cpp
//------------------------------------------------------------------------------
//  Copyright (c) 2006 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Wed Oct 25 20:49:21 EDT 2006
//
//------------------------------------------------------------------------------

#include "Granule-main.h"
#include "TextAlignment.h"

#include "Intern.h"             // i18n macros

using sigc::mem_fun;

/**-----------------------------------------------------------------------------
 *	TextAlignment
 **-----------------------------------------------------------------------------
 */
TextAlignment::
TextAlignment (PrefWindow& grandpa_) 
	: 
	Gtk::VBox (false, 0),
	m_pref_window (grandpa_)
{
	trace_with_mask("TextAlignment::TextAlignment",GUITRACE);

	m_vbox = this;

	MyTextAlignmentsWidget::value_list_t align_x_list;
	align_x_list.push_back ("center");
	align_x_list.push_back ("left");
	align_x_list.push_back ("right");

	MyTextAlignmentsWidget::value_list_t align_y_list;
	align_y_list.push_back ("center");
	align_y_list.push_back ("top");
	align_y_list.push_back ("bottom");

	MyTextAlignmentsWidget::value_list_t paragraph_list;
	paragraph_list.push_back ("center");
	paragraph_list.push_back ("left");
	paragraph_list.push_back ("right");
	paragraph_list.push_back ("fill");

	m_front_widget = new MyTextAlignmentsWidget ("<b>Front Text Alignment</b>",
												 align_x_list,
												 align_y_list,
												 paragraph_list);

	m_back_widget = new MyTextAlignmentsWidget ("<b>Back Text Alignment</b>",
												align_x_list,
												align_y_list,
												paragraph_list);

	m_example_widget = new MyTextAlignmentsWidget ("<b>Example Text Alignment</b>",
												align_x_list,
												align_y_list,
												paragraph_list);
	/** Pack all together
	 */
	m_vbox->pack_start (*(m_front_widget->frame   ()), Gtk::PACK_SHRINK, 4);
	m_vbox->pack_start (*(m_back_widget->frame    ()), Gtk::PACK_SHRINK, 4);
	m_vbox->pack_start (*(m_example_widget->frame ()), Gtk::PACK_SHRINK, 4);

	/** Setup callbacks
	 */
	/// Front
	m_front_widget->x_align_combo ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_front_widget->y_align_combo ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_front_widget->x_padding_entry ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_front_widget->y_padding_entry ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_front_widget->paragraph_combo ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	/// Back
	m_back_widget->x_align_combo ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_back_widget->y_align_combo ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_back_widget->x_padding_entry ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_back_widget->y_padding_entry ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_back_widget->paragraph_combo ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	/// Example
	m_example_widget->x_align_combo ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_example_widget->y_align_combo ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_example_widget->x_padding_entry ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_example_widget->y_padding_entry ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	m_example_widget->paragraph_combo ()->signal_changed ().connect (
		mem_fun (m_pref_window, &PrefWindow::changed_cb));

	show_all_children ();
}

/** Set property value
 */
void
TextAlignment::
set_x_alignment (SideSelection side_, const std::string& v_)
{
	int value;

	if      (v_ == "center") { value = TA_X_ALIGN_CENTER; }
	else if (v_ == "left")   { value = TA_X_ALIGN_LEFT;   }
	else if (v_ == "right")  { value = TA_X_ALIGN_RIGHT;  }
	
	if (side_ == FRONT) {
		m_front_widget->x_align_combo ()->set_active (value);
	}
	else if (side_ == BACK) {
		m_back_widget->x_align_combo ()->set_active (value);
	}
	else { /* side_ == EXAMPLE */
		m_example_widget->x_align_combo ()->set_active (value);
	}
}

void
TextAlignment::
set_y_alignment (SideSelection side_, const std::string& v_)
{
	int value;

	if      (v_ == "center") { value = TA_Y_ALIGN_CENTER; }
	else if (v_ == "top")    { value = TA_Y_ALIGN_TOP;    }
	else if (v_ == "bottom") { value = TA_Y_ALIGN_BOTTOM; }

	if (side_ == FRONT) {
		m_front_widget->y_align_combo ()->set_active (value);
	}
	else if (side_ == BACK) {
		m_back_widget->y_align_combo ()->set_active (value);
	}
	else { /* side_ == EXAMPLE */
		m_example_widget->y_align_combo ()->set_active (value);
	}
}

void
TextAlignment::
set_x_padding (SideSelection side_, const std::string& v_)
{
	if (side_ == FRONT)     { m_front_widget  ->x_padding_entry ()->set_text (v_); }
	else if (side_ == BACK) { m_back_widget   ->x_padding_entry ()->set_text (v_); }
	else /* == EXAMPLE */	{ m_example_widget->x_padding_entry ()->set_text (v_); }
}

void
TextAlignment::
set_y_padding (SideSelection side_, const std::string& v_)
{
	if (side_ == FRONT)     { m_front_widget  ->y_padding_entry ()->set_text (v_); }
	else if (side_ == BACK) { m_back_widget   ->y_padding_entry ()->set_text (v_); }
	else /* == EXAMPLE */	{ m_example_widget->y_padding_entry ()->set_text (v_); }
}

void
TextAlignment::
set_justification (SideSelection side_, const std::string& v_)
{
	int value;

	if      (v_ == "center") { value = TA_JUSTIFY_CENTER; }
	else if (v_ == "left")   { value = TA_JUSTIFY_LEFT;   }
	else if (v_ == "right")  { value = TA_JUSTIFY_RIGHT;  }
	else if (v_ == "fill")   { value = TA_JUSTIFY_FILL;   }

	if (side_ == FRONT) {
		m_front_widget->paragraph_combo ()->set_active (value);
	}
	else if (side_ == BACK) {
		m_back_widget->paragraph_combo ()->set_active (value);
	}
	else { /* EXAMPLE */
		m_example_widget->paragraph_combo ()->set_active (value);
	}
}

/** Get property value
 */
std::string
TextAlignment::
get_x_alignment (SideSelection side_) const
{
	std::string result;
	int value;

	if (side_ == FRONT) {
		value = m_front_widget->x_align_combo ()->get_active_row_number ();
	}
	else if (side_ == BACK) {
		value = m_back_widget->x_align_combo ()->get_active_row_number ();
	}
	else { /* side_ == EXAMPLE */
		value = m_example_widget->x_align_combo ()->get_active_row_number ();
	}

	switch (value) {
	case TA_X_ALIGN_CENTER:	result = "center"; break;
	case TA_X_ALIGN_LEFT:	result = "left";   break;
	case TA_X_ALIGN_RIGHT:	result = "right";  break;
	default:                result = "center";
	}

	return (result);
}


std::string
TextAlignment::
get_y_alignment (SideSelection side_) const
{
	std::string result;
	int value;

	if (side_ == FRONT) {
		value = m_front_widget->y_align_combo ()->get_active_row_number ();
	}
	else if (side_ == BACK) {
		value = m_back_widget->y_align_combo ()->get_active_row_number ();
	}
	else { /* EXAMPLE */
		value = m_example_widget->y_align_combo ()->get_active_row_number ();
	}

	switch (value) {
	case TA_Y_ALIGN_CENTER:	result = "center";	break;
	case TA_Y_ALIGN_TOP:    result = "top";    	break;
	case TA_Y_ALIGN_BOTTOM: result = "bottom"; 	break;
	default:                result = "center";
	}

	return (result);
}

std::string
TextAlignment::
get_x_padding (SideSelection side_) const
{
	if (side_ == FRONT)   { return m_front_widget  ->x_padding_entry()->get_text();}
	else if(side_ == BACK){ return m_back_widget   ->x_padding_entry()->get_text();}
	else /* == EXAMPLE */ {	return m_example_widget->x_padding_entry()->get_text();}
}

std::string
TextAlignment::
get_y_padding (SideSelection side_) const
{
	if (side_ == FRONT)   { return m_front_widget  ->y_padding_entry()->get_text();}
	else if(side_ == BACK){ return m_back_widget   ->y_padding_entry()->get_text();}
	else /* == EXAMPLE */ {	return m_example_widget->y_padding_entry()->get_text();}
}


/** Get property value
 */
std::string
TextAlignment::
get_justification (SideSelection side_) const
{
	std::string result;
	int value;

	if (side_ == FRONT) {
		value = m_front_widget->paragraph_combo ()->get_active_row_number ();
	}
	else if (side_ == BACK) {
		value = m_back_widget->paragraph_combo ()->get_active_row_number ();
	}
	else { /* side_ == EXAMPLE */
		value = m_example_widget->paragraph_combo ()->get_active_row_number ();
	}

	switch (value) {
	case TA_JUSTIFY_CENTER:	result = "center";	break;
	case TA_JUSTIFY_LEFT:	result = "left";	break;
	case TA_JUSTIFY_RIGHT:	result = "right";	break;
	case TA_JUSTIFY_FILL:	result = "fill";	break;
	default:                result = "center";
	}

	return (result);
}

