#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "ui-impl.h"
#include <string.h>

#if USE_HILDON
#include <libosso.h>
#endif

struct tag_desc
{
    const char *str;
    const char *color;
};

static const struct tag_desc tags_descs[] =
{
    {"ERROR", "Red"},
    {"INFO", "Green"},
    {"WARN", "OrangeRed"},
    {"IMPORTANT", "Blue"},
    {"COMMAND", "DarkMagenta"},
    {"COMMAND_STATUS", "Magenta"}
};
#define TAGS_DESCS_SIZE (sizeof(tags_descs)/sizeof(struct tag_desc))

struct app
{
    char *executable;
    GtkTextView *textview;
    GtkTextBuffer *textbuffer;
    struct app_ui
    {
        GtkWidget *window;
        GtkWidget *run;
        GtkWidget *close;
    } ui;
    GtkTextTag *tags[TAGS_DESCS_SIZE];
    char *argv[3];
    GPid child_pid;
    struct app_fds
    {
        int out;
        int err;
    } fds;
    struct apps_iochannels
    {
        GIOChannel *out;
        GIOChannel *err;
    } iochannels;
    struct app_event_sources
    {
        guint out;
        guint err;
        guint child;
    } event_sources;
};

#define LINE_START_MARKER "### "
#define LINE_START_MARKER_SIZE (sizeof(LINE_START_MARKER) - 1)

static GtkTextTag *
get_tag_for_line(GtkTextTag **tags, const char *line, int *offset)
{
    const char *sep;
    int i;

    *offset = 0;
    if (!line || strncmp(line, LINE_START_MARKER, LINE_START_MARKER_SIZE) != 0)
        return NULL;
    line += LINE_START_MARKER_SIZE;

    sep = strchr(line, ':');
    if (!sep)
        return NULL;

    *offset = sep - line;
    if (*offset < 0) {
        *offset = 0;
        return NULL;
    }

    for (i = 0; i < TAGS_DESCS_SIZE; i++)
        if (strncmp(line, tags_descs[i].str, *offset) == 0) {
            *offset += LINE_START_MARKER_SIZE + 1; /* include ':' */
            return tags[i];
        }

    *offset = 0;
    return NULL;
}

static void
add_line(struct app *app, const char *line)
{
    GtkTextIter end_itr;
    GtkTextTag *tag;
    int offset;

    gtk_text_buffer_get_end_iter(app->textbuffer, &end_itr);
    tag = get_tag_for_line(app->tags, line, &offset);

    if (!tag)
        gtk_text_buffer_insert(app->textbuffer, &end_itr, line, -1);
    else {
        gtk_text_buffer_insert_with_tags(app->textbuffer, &end_itr,
                                         line, offset, tag, NULL);

        gtk_text_buffer_insert(app->textbuffer, &end_itr, line + offset,
                               -1);
    }
}

static void
close_iochannel(GIOChannel **chan)
{
    g_io_channel_shutdown(*chan, 0, NULL);
    g_io_channel_unref(*chan);
    *chan = NULL;
}

static void
remove_source(guint *id)
{
    g_source_remove(*id);
    *id = 0;
}

static void
cb_stopped(struct app *app)
{
    gtk_widget_set_sensitive(app->ui.run, 1);
    gtk_widget_set_sensitive(app->ui.close, 1);
}

static void
cb_child(GPid pid, gint status, gpointer data)
{
    struct app *app = data;

    app->child_pid = -1;

    remove_source(&app->event_sources.out);
    remove_source(&app->event_sources.err);
    remove_source(&app->event_sources.child);

    close_iochannel(&app->iochannels.out);
    close_iochannel(&app->iochannels.err);

    cb_stopped(app);
}

static gboolean
cb_input(GIOChannel *chan, GIOCondition cond, struct app *app)
{
    GIOStatus status;
    GError *error;
    char *str;

    str = NULL;
    error = NULL;
    status = g_io_channel_read_line(chan, &str, NULL, NULL, &error);
    add_line(app, str);
    g_free(str);

    return status == G_IO_STATUS_NORMAL;
}

static gboolean
cb_input_out(GIOChannel *chan, GIOCondition cond, gpointer data)
{
    struct app *app = data;
    gboolean r;

    r = cb_input(chan, cond, app);
    if (!r)
        app->event_sources.out = 0;

    return r;
}

static gboolean
cb_input_err(GIOChannel *chan, GIOCondition cond, gpointer data)
{
    struct app *app = data;
    gboolean r;

    r = cb_input(chan, cond, app);
    if (!r)
        app->event_sources.err = 0;

    return r;
}

static int
run(struct app *app)
{
    GSpawnFlags flags;
    GError *error;
    gboolean r;
    GIOCondition cond;

    app->argv[0] = "canola-cleanup-launcher";
    app->argv[1] = app->executable;
    app->argv[2] = NULL;

    flags = G_SPAWN_SEARCH_PATH | G_SPAWN_DO_NOT_REAP_CHILD;
    error = NULL;
    r = g_spawn_async_with_pipes("/tmp", app->argv, NULL, flags, NULL, NULL,
                                 &app->child_pid,
                                 NULL, &app->fds.out, &app->fds.err,
                                 &error);

    if (!r && error) {
        char buf[1024];

        snprintf(buf, sizeof(buf),
                 LINE_START_MARKER "ERROR: Could not start executable: %s\n",
                 error->message);
        add_line(app, buf);
        g_error_free(error);
        return 0;
    }

    app->event_sources.child = g_child_watch_add(app->child_pid, cb_child, app);

    app->iochannels.out = g_io_channel_unix_new(app->fds.out);
    app->iochannels.err = g_io_channel_unix_new(app->fds.err);

    cond = G_IO_IN | G_IO_ERR;
    app->event_sources.out =
        g_io_add_watch(app->iochannels.out, cond, cb_input_out, app);
    app->event_sources.err =
    g_io_add_watch(app->iochannels.err, cond, cb_input_err, app);

    gtk_widget_set_sensitive(app->ui.run, 0);
    gtk_widget_set_sensitive(app->ui.close, 0);

    return 1;
}

static void
cb_run(GtkWidget *btn, struct app *app)
{
    if (app->child_pid >= 0)
        return;

    gtk_text_buffer_set_text(app->textbuffer, "", -1);
    run(app);
}

static void
warn_dialog(GtkWidget *parent, const char *msg)
{
    GtkWidget *dialog;

    dialog = gtk_message_dialog_new_with_markup(GTK_WINDOW(parent),
                                                GTK_DIALOG_DESTROY_WITH_PARENT,
                                                GTK_MESSAGE_WARNING,
                                                GTK_BUTTONS_CLOSE,
                                                NULL);
    gtk_message_dialog_set_markup(GTK_MESSAGE_DIALOG(dialog), msg);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
}

static gboolean
do_quit(struct app *app)
{
    if (app->child_pid < 0) {
        gtk_main_quit();
        return 0;
    }

    warn_dialog(app->ui.window,
                "<big><b>Still processing.</b></big>\n"
                "You cannot close application yet.\n"
                "<b>It is dangerous, may corrupt your system.</b>");
    return 1;
}

static gboolean
cb_close(GtkWidget *wid, struct app *app)
{
    return do_quit(app);
}

static GtkWidget *
create_buttons(struct app *app)
{
    GtkWidget *hbox;

    hbox = gtk_hbutton_box_new();

    app->ui.close = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    GTK_WIDGET_SET_FLAGS(app->ui.close, GTK_CAN_DEFAULT);
    g_signal_connect(app->ui.close, "clicked", G_CALLBACK(cb_close), app);

    app->ui.run = gtk_button_new_from_stock(GTK_STOCK_EXECUTE);
    g_signal_connect(app->ui.run, "clicked", G_CALLBACK(cb_run), app);

    gtk_box_pack_start(GTK_BOX(hbox), app->ui.close, 0, 1, 0);
    gtk_box_pack_start(GTK_BOX(hbox), app->ui.run, 0, 1, 0);

    return hbox;
}

static GtkTextView *
create_textview(struct app *app)
{
    GtkTextView *tv;
    PangoFontDescription *font_desc;
    int i;

    tv = GTK_TEXT_VIEW(gtk_text_view_new());
    gtk_text_view_set_wrap_mode(tv, GTK_WRAP_NONE);
    gtk_text_view_set_editable(tv, 0);

    font_desc = pango_font_description_from_string("Mono");
    gtk_widget_modify_font(GTK_WIDGET(tv), font_desc);
    pango_font_description_free(font_desc);

    app->textbuffer = gtk_text_view_get_buffer(tv);

    for (i = 0; i < TAGS_DESCS_SIZE; i++) {
        const struct tag_desc *desc = tags_descs + i;
        app->tags[i] = gtk_text_buffer_create_tag(app->textbuffer,
                                                  desc->str,
                                                  "foreground", desc->color,
                                                  "weight", PANGO_WEIGHT_BOLD,
                                                  NULL);
    }

    return tv;
}

static GtkWidget *
create_contents(struct app *app)
{
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *label;
    GtkWidget *entry;
    GtkWidget *scrolledwindow;
    GtkWidget *buttons;

    vbox = gtk_vbox_new(0, 5);

    hbox = gtk_hbox_new(0, 2);
    label = gtk_label_new("Executable");
    entry = gtk_entry_new();
    GTK_WIDGET_UNSET_FLAGS(entry, GTK_CAN_FOCUS);
    gtk_entry_set_text(GTK_ENTRY(entry), app->executable);
    gtk_entry_set_editable(GTK_ENTRY(entry), 0);
    gtk_entry_set_alignment(GTK_ENTRY(entry), 1.0);
    gtk_box_pack_start(GTK_BOX(hbox), label, 0, 1, 0);
    gtk_box_pack_start(GTK_BOX(hbox), entry, 1, 1, 0);

    app->textview = create_textview(app);
    buttons = create_buttons(app);

    scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolledwindow),
                                        GTK_SHADOW_ETCHED_IN);
    gtk_container_add(GTK_CONTAINER(scrolledwindow), GTK_WIDGET(app->textview));

    gtk_box_pack_start(GTK_BOX(vbox), hbox, 0, 1, 0);
    gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow, 1, 1, 0);
    gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(), 0, 1, 0);
    gtk_box_pack_start(GTK_BOX(vbox), buttons, 0, 0, 0);

    return vbox;
}

static gboolean
cb_quit(GtkWidget *wid, GdkEvent *event, struct app *app)
{
    return do_quit(app);
}

static int
usage(int argc, char *argv[])
{
    g_printerr("Usage:\n"
               "\t%s <executable>\n"
               "\n",
               argv[0]);
    return 0;
}

int
main(int argc, char *argv[])
{
    UiProgram *program;
    GtkWidget *contents;
    struct app app;

#if USE_HILDON
    osso_context_t *osso_context;
#endif

    if (argc < 2)
        return usage(argc, argv);

    app.executable = argv[1];
    app.child_pid = -1;

    gtk_init(&argc, &argv);

#if USE_HILDON
    osso_context = osso_initialize("br.org.indt.canola_cleanup", "0.1.0", TRUE, NULL);
#endif

    program = program_new();
    app.ui.window = create_window();
    g_set_application_name("Canola2 Utility");
    gtk_window_set_title(GTK_WINDOW(app.ui.window), "Canola2 Utility");
    program_add_window(program, app.ui.window);

    contents = create_contents(&app);
    gtk_container_add(GTK_CONTAINER(app.ui.window), contents);
    gtk_container_set_border_width(GTK_CONTAINER(app.ui.window), 10);
    gtk_widget_grab_focus(app.ui.close);
    gtk_widget_grab_default(app.ui.close);

    gtk_widget_show_all(app.ui.window);
    g_signal_connect(app.ui.window, "delete_event", G_CALLBACK(cb_quit), &app);
    gtk_main();

#if USE_HILDON
    osso_deinitialize(osso_context);
#endif

    return 0;
}
