/*
 * This file is part of vncviewer
 *
 * Copyright (C) 2005, 2006 Aaron Levinson.
 * Copyright (C) 2006, Detlef Schmicker
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gtk/gtk.h>
#include <gtk/gtkmarshal.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <hildon-widgets/hildon-get-password-dialog.h>
#include <hildon-widgets/hildon-note.h>
#include "vnc-viewer.h"

#include <string.h>
#include <unistd.h>

extern void create_menu(void *mainvw);
extern void disable_home_key_lock(void *mainvw);

extern gboolean arg_lockdown;
extern gchar *arg_lockdown_password;

enum {
  VNC_FRAMEBUFFER_UPDATE,
  VNC_SET_COLOURMAP_ENTRIES,
  VNC_BELL,
  VNC_SERVER_CUT_TEXT,
  CONNECTION_DIED,
  LAST_SIGNAL
};

static guint vnc_signals[LAST_SIGNAL] = { 0 };
static GtkWidgetClass *parent_class = NULL;

static void vnc_viewer_class_init(VncViewerClass *klass);
static void vnc_viewer_init(VncViewer *vnc);

GtkType vnc_viewer_get_type(void)
{
  static GtkType vnc_type = 0;
  if (!vnc_type)
  {
    GtkTypeInfo vnc_info =
    {
      "VncViewer",
      sizeof(VncViewer),
      sizeof(VncViewerClass),
      (GtkClassInitFunc) vnc_viewer_class_init,
      (GtkObjectInitFunc) vnc_viewer_init,
      NULL,
      NULL,
    };

    vnc_type = gtk_type_unique(gtk_widget_get_type(), &vnc_info);
  }

  return vnc_type;
}

static void vnc_viewer_framebuffer_update(VncViewer *vnc,
					  rfbServerToClientMsg *msg);
static void vnc_viewer_set_colourmap_entries(VncViewer *vnc,
					     rfbServerToClientMsg *msg,
					     guint16 *entries);
static void vnc_viewer_bell(VncViewer *vnc, rfbServerToClientMsg *msg);
static void vnc_viewer_server_cut_text(VncViewer *vnc,
				       rfbServerToClientMsg *msg, gchar *text);

static void vnc_viewer_realize(GtkWidget *widget);
static void vnc_viewer_unrealize(GtkWidget *widget);
static gint vnc_viewer_expose_event(GtkWidget *widget, GdkEventExpose *event);
static gint vnc_viewer_button_press_event(GtkWidget *widget,
					  GdkEventButton *event);
static gint vnc_viewer_button_release_event(GtkWidget *widget,
					    GdkEventButton *event);
static gint vnc_viewer_motion_notify_event(GtkWidget *widget,
					   GdkEventMotion *event);

static gboolean vnc_viewer_focus(GtkWidget *widget,
				 GdkEventFocus *event,
				 GtkIMContext *im_context);

static void vnc_viewer_finalize(GObject *object);

static void vnc_viewer_class_init(VncViewerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS(klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  vnc_signals[VNC_FRAMEBUFFER_UPDATE] =
    g_signal_new("vnc_framebuffer_update",
		 G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(VncViewerClass, vnc_framebuffer_update),
		 0, 0, gtk_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 GTK_TYPE_POINTER);
  vnc_signals[VNC_SET_COLOURMAP_ENTRIES] =
    g_signal_new("vnc_set_colourmap_entries",
		 G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(VncViewerClass,vnc_set_colourmap_entries),
		 0, 0, gtk_marshal_VOID__POINTER_POINTER,
		 G_TYPE_NONE, 2,
		 GTK_TYPE_POINTER, GTK_TYPE_POINTER);
  vnc_signals[VNC_BELL] =
    g_signal_new("vnc_bell",
		 G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(VncViewerClass, vnc_bell),
		 0, 0, gtk_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 GTK_TYPE_POINTER);
  vnc_signals[VNC_SERVER_CUT_TEXT] =
    g_signal_new("vnc_server_cut_text",
		 G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(VncViewerClass, vnc_server_cut_text),
		 0, 0, gtk_marshal_VOID__POINTER_POINTER,
		 G_TYPE_NONE, 2,
		 GTK_TYPE_POINTER, GTK_TYPE_STRING);
  vnc_signals[CONNECTION_DIED] =
    g_signal_new("connection_died",
		 G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(VncViewerClass, connection_died),
		 0, 0, gtk_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);
  //g_object_class_add_signals(object_class, vnc_signals, LAST_SIGNAL);

  klass->vnc_framebuffer_update    = vnc_viewer_framebuffer_update;
  klass->vnc_set_colourmap_entries = vnc_viewer_set_colourmap_entries;
  klass->vnc_bell                  = vnc_viewer_bell;
  klass->vnc_server_cut_text       = vnc_viewer_server_cut_text;

  // signals to do with redrawing and the like
  widget_class->realize = vnc_viewer_realize;
  widget_class->unrealize = vnc_viewer_unrealize;
  widget_class->expose_event = vnc_viewer_expose_event;

  // signals that create events that should be sent to the server
  widget_class->button_press_event = vnc_viewer_button_press_event;
  widget_class->button_release_event = vnc_viewer_button_release_event;
  widget_class->motion_notify_event = vnc_viewer_motion_notify_event;
  widget_class->key_press_event = vnc_viewer_key_press_event;
  widget_class->key_release_event = vnc_viewer_key_release_event;

  object_class->finalize = vnc_viewer_finalize;
}

static void vnc_viewer_init(VncViewer *self)
{
  const char dotSrcBits[] = { 0, 14,14,14, 0 };
  const char dotMskBits[] = { 31,31,31,31,31 };
  const char noneBits[] = { 0 };
  GdkBitmap *src, *mask;
  GdkColor white, black;

  GTK_WIDGET_UNSET_FLAGS(self, GTK_NO_WINDOW);
  GTK_WIDGET_SET_FLAGS(self, GTK_CAN_FOCUS);

  self->cursor_type = VNC_VIEWER_CURSOR_NORMAL;

  self->fd = -1;

  self->im_context = gtk_im_multicontext_new();

  // commented out--no longer works with IT2006
  //g_object_set(G_OBJECT(self->im_context), "use-show-hide", TRUE, NULL);
  
  g_signal_connect(self->im_context, "commit",
		   G_CALLBACK(vnc_viewer_im_commit), self);
  // it seems that the input managers available with maemo only use the
  // commit event
  //g_signal_connect(self->im_context, "preedit_changed",
  //   G_CALLBACK(vnc_viewer_im_preedit_changed), self);

  self->in_tag = 0;
  self->out_tag = 0;

  self->out_queue = 0;

  self->width = 0;
  self->height = 0;
  memset(&(self->pixel_format), '\0', sizeof(self->pixel_format));
  self->desk_name = NULL;

  self->button_mask = 0;
  self->last_pointer_x_loc = 0;
  self->last_pointer_y_loc = 0;

  self->offscreen = NULL;

  gdk_color_white(gtk_widget_get_colormap(GTK_WIDGET(self)), &white);
  gdk_color_black(gtk_widget_get_colormap(GTK_WIDGET(self)), &black);

  src  = gdk_bitmap_create_from_data(NULL, dotSrcBits, 5, 5);
  mask = gdk_bitmap_create_from_data(NULL, dotMskBits, 5, 5);
  self->dot = gdk_cursor_new_from_pixmap(src, mask, &black, &white, 2, 2);
  gdk_bitmap_unref(src);
  gdk_bitmap_unref(mask);

  src = gdk_bitmap_create_from_data(NULL, noneBits, 1, 1);
  self->none = gdk_cursor_new_from_pixmap(src, src, &black, &white, 0, 0);
  gdk_bitmap_unref(src);

  self->m_bGotFirstUpdate = FALSE;

  self->m_lockdownKeyPressKeyVal = 0;

  self->mainview = 0;
}

GtkWidget *vnc_viewer_new(gint fd, gboolean shared, VncViewerDepthType depth)
{
  VncViewer *vnc = gtk_type_new(vnc_viewer_get_type());

  if (!vnc)
    return NULL;
  vnc_viewer_configure(vnc, fd, shared, depth);

  return GTK_WIDGET(vnc);
}

void vnc_viewer_close_connection(VncViewer *vnc)
{
  // if already closed, don't do it again
  if (vnc->fd < 0)
    return;

  close(vnc->fd);
  vnc->fd = -1;

  if (vnc->in_tag)
    gdk_input_remove(vnc->in_tag);
  if (vnc->out_tag)
    gdk_input_remove(vnc->out_tag);
  vnc->in_tag = vnc->out_tag = 0;

  g_list_foreach(vnc->out_queue, (GFunc)g_free, NULL);
  g_list_free(vnc->out_queue);
  vnc->out_queue = NULL;
}

void vnc_viewer_set_cursor(VncViewer *vnc, VncViewerCursorType cursor_type)
{
  GdkCursor *cur;

  vnc->cursor_type = cursor_type;

  if (GTK_WIDGET_REALIZED(vnc))
  {
    switch (vnc->cursor_type)
    {
    case VNC_VIEWER_CURSOR_NORMAL:
      cur = gdk_cursor_new(GDK_ARROW);
      gdk_window_set_cursor(GTK_WIDGET(vnc)->window, cur);
      gdk_cursor_destroy(cur);
      break;

    case VNC_VIEWER_CURSOR_DOT:
      gdk_window_set_cursor(GTK_WIDGET(vnc)->window, vnc->dot);
      break;

    case VNC_VIEWER_CURSOR_NONE:
      gdk_window_set_cursor(GTK_WIDGET(vnc)->window, vnc->none);
      break;
    }
  }
}

static void vnc_viewer_get_event(VncViewer *vnc, gint fd,
				 GdkInputCondition cond);
static void vnc_viewer_send_event(VncViewer *vnc, gint fd,
				  GdkInputCondition cond);

static void vnc_viewer_queue_event(VncViewer *vnc, rfbClientToServerMsg *msg)
{
  if (vnc->fd < 0)
  {
    g_free(msg);
    return;
  }

  vnc->out_queue = g_list_append(vnc->out_queue, msg);

  if (vnc->out_tag == 0)
    vnc->out_tag = gdk_input_add(vnc->fd, GDK_INPUT_WRITE,
				 (GdkInputFunction)vnc_viewer_send_event,
				 vnc);
}

/* fd is a connection to the VNC server just after the connection has been
 * authenticated.  Shared says whether we allow other concurrent connections,
 * and bpp is the desired bpp
 */
void vnc_viewer_configure(VncViewer *vnc, gint fd, gboolean shared,
			  VncViewerDepthType depth)
{
  guint32 encodings[] =
  { 
    //    rfbEncodingHextile,
    rfbEncodingCoRRE,
    rfbEncodingRRE,
    rfbEncodingCopyRect, rfbEncodingRaw
  };

  guint16 nEncodings = sizeof(encodings) / sizeof(encodings[0]);
  //guint32 localEncodings[] = { rfbEncodingCopyRect, rfbEncodingRaw };
  //guint nLocalEncodings = sizeof(localEncodings) / sizeof(localEncodings[0]);

  vnc->fd = fd;

  if (sendClientInitMsg(fd, shared) < 0)
  {
    g_warning("client init message send failed");
    return;
  }

  if (getServerInitMsg(fd, &(vnc->width), &(vnc->height), &(vnc->pixel_format),
		       &(vnc->desk_name)) < 0)
  {
    g_warning("server init message recieve failed");
    return;
  }

  g_message("Desk name='%s'", vnc->desk_name);
  g_message("Desk size=%dx%d, Default depth=%d", vnc->width, vnc->height,
	    (gint) vnc->pixel_format.depth);
  // gdk_input_add is deprecated, should be replaced
  vnc->in_tag = gdk_input_add(vnc->fd, GDK_INPUT_READ,
			      (GdkInputFunction)vnc_viewer_get_event, vnc);


//This saves the client from changing the byte order of the pixels
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
  vnc->pixel_format.bigEndian = FALSE;
#else
  vnc->pixel_format.bigEndian = TRUE;
#endif

  switch (depth)
  {
  case VNC_VIEWER_DEPTH_DEFAULT:
    break;

  case VNC_VIEWER_DEPTH_8BIT:
    if (vnc->pixel_format.depth > 8)
    {
      // only change depth down
      vnc->pixel_format.bitsPerPixel = 8;
      vnc->pixel_format.depth = 8;
      vnc->pixel_format.trueColour = TRUE;
      vnc->pixel_format.redMax = 7;
      vnc->pixel_format.greenMax = 7;
      vnc->pixel_format.blueMax = 3;
      vnc->pixel_format.redShift = 0;
      vnc->pixel_format.greenShift = 3;
      vnc->pixel_format.blueShift = 6;
    }
    break;

  case VNC_VIEWER_DEPTH_16BIT:
    if (vnc->pixel_format.depth > 16)
    {
      // only change depth down
      vnc->pixel_format.bitsPerPixel = 16;
      vnc->pixel_format.depth = 16;
      vnc->pixel_format.trueColour = TRUE;
      vnc->pixel_format.redMax = 31;
      vnc->pixel_format.greenMax = 63;
      vnc->pixel_format.blueMax = 31;
      vnc->pixel_format.redShift = 11;
      vnc->pixel_format.greenShift = 5;
      vnc->pixel_format.blueShift = 0;
    }
    break;

  case VNC_VIEWER_DEPTH_24BIT:
    // we want to force this format
    // if (vnc->pixel_format.depth > 24)
    {
      vnc->pixel_format.bitsPerPixel = 32;
      vnc->pixel_format.depth = 24;
      vnc->pixel_format.trueColour = TRUE;
      vnc->pixel_format.redMax = 255;
      vnc->pixel_format.greenMax = 255;
      vnc->pixel_format.blueMax = 255;
      vnc->pixel_format.redShift = 16;
      vnc->pixel_format.greenShift = 8;
      vnc->pixel_format.blueShift = 0;
    }
    break;
  }

  vnc_viewer_queue_event(vnc, newSetPixelFormatMsg(&(vnc->pixel_format)));

  /*
  if (sameMachine(vnc->fd))
    vnc_viewer_queue_event(vnc, newSetEncodingsMsg(nLocalEncodings,
						   localEncodings));
  else
  */
    vnc_viewer_queue_event(vnc, newSetEncodingsMsg(nEncodings, encodings));
}

static void vnc_viewer_get_event(VncViewer *vnc, gint fd,
				 GdkInputCondition cond)
{
  rfbServerToClientMsg *msg;

  if (cond == GDK_INPUT_EXCEPTION)
  {
    vnc_viewer_close_connection(vnc);
    g_signal_emit(G_OBJECT(vnc), vnc_signals[CONNECTION_DIED], 0);
    return;
  }

  msg = getServerMsg(fd);

  if (!msg)
  {
    vnc_viewer_close_connection(vnc);
    g_signal_emit(G_OBJECT(vnc), vnc_signals[CONNECTION_DIED], 0);
    return;
  }

  switch (msg->type)
  {
  case rfbFramebufferUpdate:
    g_signal_emit(G_OBJECT(vnc), vnc_signals[VNC_FRAMEBUFFER_UPDATE], 0, msg);
    break;

  case rfbSetColourMapEntries:
    {
      guint16 *entries = getColourMapEntries(fd, msg->scme.nColours);

      g_signal_emit(G_OBJECT(vnc), vnc_signals[VNC_SET_COLOURMAP_ENTRIES], 0, 
		      msg, entries);
      g_free(entries);
      break;
    }

  case rfbBell:
    g_signal_emit(G_OBJECT(vnc), vnc_signals[VNC_BELL], 0, msg);
    break;

  case rfbServerCutText:
    {
      gchar *text = getServerCutText(fd, msg->sct.length);

      g_signal_emit(G_OBJECT(vnc), vnc_signals[VNC_SERVER_CUT_TEXT], 0, 
		    msg, text);
      g_free(text);
      break;
    }

  default:
    break;
  }

  g_free(msg);
}

static void vnc_viewer_send_event(VncViewer *vnc, gint fd,
				  GdkInputCondition cond)
{
  rfbClientToServerMsg *msg;

  if (cond == GDK_INPUT_EXCEPTION)
  {
    vnc_viewer_close_connection(vnc);
    g_signal_emit(G_OBJECT(vnc), vnc_signals[CONNECTION_DIED], 0);
    return;
  }

  msg = vnc->out_queue->data;
  vnc->out_queue = g_list_remove(vnc->out_queue, msg);
  rfb_send_and_destroy_message(fd, msg);

  // if no more queued events exist, remove this handler
  if (vnc->out_queue == NULL)
  {
    gdk_input_remove(vnc->out_tag);
    vnc->out_tag = 0;
  }
}

static gboolean vnc_viewer_send_draw_request(VncViewer *vnc)
{
  if (vnc->fd < 0)
    return FALSE;

  vnc_viewer_queue_event(vnc, newFramebufferUpdateRequestMsg(TRUE, 0, 0,
							     vnc->width,
							     vnc->height));
  return FALSE;
}

static void vnc_viewer_framebuffer_update(VncViewer *vnc,
					  rfbServerToClientMsg *msg)
{
  guint32 rect, srect, tile, i, j;
  guchar *buffer;
  rfbPixelFormat *fmt = &(vnc->pixel_format);
  guint8 bpp = fmt->bitsPerPixel, tmpR, tmpG, tmpB, bgR, bgG, bgB;
  gdouble redScale = 255.0 / fmt->redMax, greenScale = 255.0 / fmt->greenMax,
    blueScale = 255.0 / fmt->blueMax;

  for (rect = 0; rect < msg->fu.nRects; rect++)
  {
    rfbRectangleDescription *desc =
      getNextRectangle(vnc->fd, vnc->pixel_format.bitsPerPixel / 8,
		       !vnc->m_bGotFirstUpdate);
    if (!vnc->m_bGotFirstUpdate)
      vnc->m_bGotFirstUpdate = TRUE;

    if (!desc)
    {
      g_signal_emit(G_OBJECT(vnc), vnc_signals[CONNECTION_DIED], 0);
      return;
    }

    guint16 rWidth = desc->header.r.w, rHeight = desc->header.r.h;
    guint16 xPos, yPos;
    guint32 totPixels = rWidth * rHeight;

    //g_message("Rectangle (%d,%d) %dx%d Encoding=%d", (gint)desc->header.r.x,
    //	      (gint)desc->header.r.y, (gint)desc->header.r.w,
    //	      (gint)desc->header.r.h, (gint)desc->header.encoding);
    switch (desc->header.encoding)
    {
    case rfbEncodingRaw:
      buffer = g_malloc(totPixels * 3);
      for (i = 0; i < totPixels; i++)
      {
	guint32 pixel = 0;
	switch (bpp)
	{
	case 8:
	  pixel = desc->data.raw[i];
	  break;

	case 16:
	  pixel = ((guint16 *)(desc->data.raw))[i];
	  break;

	case 32:
	  pixel = ((guint32 *)(desc->data.raw))[i];
	  break;

	default:
	  g_assert_not_reached();
	}

	buffer[3*i]   = (pixel >> fmt->redShift & fmt->redMax) * redScale;
	buffer[3*i+1] = (pixel >> fmt->greenShift & fmt->greenMax) *greenScale;
	buffer[3*i+2] = (pixel >> fmt->blueShift & fmt->blueMax) * blueScale;
      }

      gdk_draw_rgb_image(vnc->offscreen, vnc->gc,
			 desc->header.r.x, desc->header.r.y,
			 rWidth, rHeight,
			 GDK_RGB_DITHER_NORMAL, buffer, rWidth * 3);
      g_free(buffer);
      break;

    case rfbEncodingCopyRect:
      gdk_draw_pixmap(vnc->offscreen, vnc->gc, vnc->offscreen,
		      desc->data.cr.srcX, desc->data.cr.srcY,
		      desc->header.r.x, desc->header.r.y,
		      rWidth, rHeight);
      break;

    case rfbEncodingRRE:
      buffer = g_malloc(totPixels * 3);
      tmpR = (desc->data.rre.bgPixel >> fmt->redShift & fmt->redMax) *
	redScale;
      tmpG = (desc->data.rre.bgPixel >> fmt->greenShift & fmt->greenMax) *
	greenScale;
      tmpB = (desc->data.rre.bgPixel >> fmt->blueShift & fmt->blueMax) *
	blueScale;

      for (i = 0; i < totPixels; i++)
      {
	buffer[3*i] = tmpR; buffer[3*i+1] = tmpG; buffer[3*i+2] = tmpB;
      }

      for (srect = 0; srect < desc->data.rre.nSubrects; srect++)
      {
	guint32 pixel = desc->data.rre.rects[srect].pixel;
	rfbRectangle *sr = &(desc->data.rre.rects[srect].rect);
	tmpR = (pixel >> fmt->redShift & fmt->redMax) * redScale;
	tmpG = (pixel >> fmt->greenShift & fmt->greenMax) * greenScale;
	tmpB = (pixel >> fmt->blueShift & fmt->blueMax) * blueScale;
	for (j = sr->y; j < sr->y + sr->h; j++)
	{
	  for (i = sr->x; i < sr->x + sr->w; i++)
	  {
	    buffer[3*(rWidth*j+i)]   = tmpR;
	    buffer[3*(rWidth*j+i)+1] = tmpG;
	    buffer[3*(rWidth*j+i)+2] = tmpB;
	  }
	}
      }

      gdk_draw_rgb_image(vnc->offscreen, vnc->gc,
			 desc->header.r.x, desc->header.r.y,
			 rWidth, rHeight,
			 GDK_RGB_DITHER_NORMAL, buffer, rWidth * 3);
      g_free(buffer);
      break;

    case rfbEncodingCoRRE:
      buffer = g_malloc(totPixels * 3);
      tmpR = (desc->data.corre.bgPixel >> fmt->redShift & fmt->redMax) *
	redScale;
      tmpG = (desc->data.corre.bgPixel >> fmt->greenShift & fmt->greenMax) *
	greenScale;
      tmpB = (desc->data.corre.bgPixel >> fmt->blueShift & fmt->blueMax) *
	blueScale;

      for (i = 0; i < totPixels; i++)
      {
	buffer[3*i] = tmpR; buffer[3*i+1] = tmpG; buffer[3*i+2] = tmpB;
      }

      for (srect = 0; srect < desc->data.corre.nSubrects; srect++)
      {
	guint32 pixel = desc->data.corre.rects[srect].pixel;
	rfbCoRRERectangle *sr = &(desc->data.corre.rects[srect].rect);
	tmpR = (pixel >> fmt->redShift & fmt->redMax) * redScale;
	tmpG = (pixel >> fmt->greenShift & fmt->greenMax) * greenScale;
	tmpB = (pixel >> fmt->blueShift & fmt->blueMax) * blueScale;
	for (j = sr->y; j < sr->y + sr->h; j++)
	{
	  for (i = sr->x; i < sr->x + sr->w; i++)
	  {
	    buffer[3*(rWidth*j+i)]   = tmpR;
	    buffer[3*(rWidth*j+i)+1] = tmpG;
	    buffer[3*(rWidth*j+i)+2] = tmpB;
	  }
	}
      }

      gdk_draw_rgb_image(vnc->offscreen, vnc->gc,
			 desc->header.r.x, desc->header.r.y,
			 rWidth, rHeight,
			 GDK_RGB_DITHER_NORMAL, buffer, rWidth * 3);
      g_free(buffer);
      break;

    case rfbEncodingHextile:
      buffer = g_malloc(totPixels * 3);
      bgR = bgG = bgB = 0;
      xPos = yPos = 0;
      for (tile = 0; tile < desc->data.hextile.nTiles; tile++)
      {
	rfbHextile *ht = desc->data.hextile.tiles[tile];
	
	if (ht->type & rfbHextileRaw)
	{
	  for (j = 0; j < ht->height; j++)
	  {
	    for (i = 0; i < ht->width; i++)
	    {
	      guint offs = (yPos + j) * rWidth + xPos + i;
	      guint32 pixel = 0;

	      switch (bpp)
	      {
	      case 8:
		pixel = desc->data.raw[i];
		break;

	      case 16:
		pixel = ((guint16 *)(desc->data.raw))[i];
		break;

	      case 32:
		pixel = ((guint32 *)(desc->data.raw))[i];
		break;

	      default:
		g_assert_not_reached();
	      }

	      buffer[offs] = (pixel >> fmt->redShift & fmt->redMax) * redScale;
	      buffer[offs+1] = (pixel >> fmt->greenShift & fmt->greenMax) * greenScale;
	      buffer[offs+2] = (pixel >> fmt->blueShift & fmt->blueMax) * blueScale;
	      if (offs == 0)
		g_print("%06x ", pixel);
	    }
	  }
	}
	else
	{
	  if (ht->type & rfbHextileBackgroundSpecified)
	  {
	    bgR = (ht->data.rects.bgcolour >> fmt->redShift & fmt->redMax) * redScale;
	    bgG = (ht->data.rects.bgcolour >> fmt->greenShift & fmt->greenMax) * greenScale;
	    bgB = (ht->data.rects.bgcolour >> fmt->blueShift & fmt->blueMax) * blueScale;
	    g_print("%06x ", ht->data.rects.bgcolour);
	  }

	  for (j = 0; j < ht->height; j++)
	  {
	    for (i = 0; i < ht->width; i++)
	    {
	      guint offs = (yPos + j) * rWidth + xPos + i;
	      buffer[offs] = bgR;
	      buffer[offs+1] = bgG;
	      buffer[offs+2] = bgB;
	    }
	  }

	  if (ht->type & rfbHextileAnySubrects)
	  {
	    for (srect = 0; srect < ht->data.rects.nRects; srect++)
	    {
	      for (j = 0; j < ht->data.rects.rects[srect].h; j++)
	      {
		for (i = 0; i < ht->data.rects.rects[srect].w; i++)
		{
		  guint offs = (yPos + ht->data.rects.rects[srect].y + j) *
		    rWidth + xPos + ht->data.rects.rects[srect].x + i;
		  guint32 pixel = ht->data.rects.rects[srect].colour;

		  buffer[offs] = (pixel >> fmt->redShift & fmt->redMax) * redScale;
		  buffer[offs+1] = (pixel >> fmt->greenShift & fmt->greenMax) * greenScale;
		  buffer[offs+2] = (pixel >> fmt->blueShift & fmt->blueMax) * blueScale;
		}
	      }
	    }
	  }
	}

	xPos += ht->width;
	if (xPos >= desc->header.r.w)
	{
	  xPos = 0;
	  yPos += ht->height;
	}
      }

      gdk_draw_rgb_image(vnc->offscreen, vnc->gc,
			 desc->header.r.x, desc->header.r.y,
			 rWidth, rHeight,
			 GDK_RGB_DITHER_NORMAL, buffer, rWidth * 3);
      g_free(buffer);
      break;

    default:
      g_warning("Unknown rectangle encoding returned");
    }

    gdk_draw_pixmap(GTK_WIDGET(vnc)->window,
		    vnc->gc,
		    vnc->offscreen,
		    desc->header.r.x, desc->header.r.y,
		    desc->header.r.x,
		    desc->header.r.y,
		    rWidth, rHeight);
    freeRectangleDescription(desc);
  }

  gtk_timeout_add(25, (GtkFunction) vnc_viewer_send_draw_request, vnc);
}

static void vnc_viewer_set_colourmap_entries(VncViewer *vnc,
					     rfbServerToClientMsg *msg,
					     guint16 *entries)
{
}

static void vnc_viewer_bell(VncViewer *vnc, rfbServerToClientMsg *msg)
{
  gdk_beep();
}

static void vnc_viewer_server_cut_text(VncViewer *vnc,
				       rfbServerToClientMsg *msg, gchar *text)
{
  // I am sure I should be doing something with a selection here...
}

static void vnc_viewer_realize(GtkWidget *widget)
{
  VncViewer *vnc = VNC_VIEWER(widget);
  GdkWindowAttr attributes;
  gint attributes_mask;

  //GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual(widget);
  attributes.colormap = gtk_widget_get_colormap(widget);
  attributes.event_mask = gtk_widget_get_events(widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_POINTER_MOTION_MASK |
			    GDK_BUTTON_MOTION_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK |
			    GDK_KEY_PRESS_MASK |
			    GDK_KEY_RELEASE_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  if (vnc->cursor_type == VNC_VIEWER_CURSOR_DOT)
  {
    attributes.cursor = vnc->dot;
    attributes_mask |= GDK_WA_CURSOR;
  } 
  else if (vnc->cursor_type == VNC_VIEWER_CURSOR_NONE)
  {
    attributes.cursor = vnc->none;
    attributes_mask |= GDK_WA_CURSOR;
  }

  widget->window = gdk_window_new(gtk_widget_get_parent_window(widget),
				  &attributes, attributes_mask);
  widget->style = gtk_style_attach(widget->style, widget->window);
  gtk_style_set_background(widget->style, widget->window, GTK_STATE_NORMAL);

  if (vnc->offscreen)
    gdk_pixmap_unref(vnc->offscreen);
  vnc->offscreen = gdk_pixmap_new(widget->window,
				  vnc->width,
				  vnc->height,
				  -1);
  if (vnc->gc)
    gdk_gc_unref(vnc->gc);
  vnc->gc = gdk_gc_new(widget->window);
  gdk_gc_copy(vnc->gc, GTK_WIDGET(vnc)->style->fg_gc[GTK_WIDGET_STATE(vnc)]);

  gdk_window_set_user_data(widget->window, widget);
  gdk_window_show(widget->window);
  GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);

  // associate the input method with this window
  gtk_im_context_set_client_window(vnc->im_context, widget->window);

  // need to connect to focus-in and focus-out events to setup input method
  // properly
  g_signal_connect(G_OBJECT(vnc), "focus-in-event",
		   G_CALLBACK(vnc_viewer_focus), vnc->im_context);

  g_signal_connect(G_OBJECT(vnc), "focus-out-event",
		   G_CALLBACK(vnc_viewer_focus), vnc->im_context);

  g_message("Sending request for full update");
  vnc_viewer_queue_event(vnc, newFramebufferUpdateRequestMsg(TRUE, 0, 0,
							     vnc->width,
							     vnc->height));
}

static void vnc_viewer_unrealize(GtkWidget *widget)
{
  VncViewer *vnc = VNC_VIEWER(widget);

  // disassociate input method from this window
  gtk_im_context_set_client_window(vnc->im_context, 0);

  if (GTK_WIDGET_CLASS(parent_class)->unrealize)
    (*GTK_WIDGET_CLASS(parent_class)->unrealize)(widget);
}

static gint vnc_viewer_expose_event(GtkWidget *widget, GdkEventExpose *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);

  if (GTK_WIDGET_DRAWABLE(widget))
  {
    // gdk_draw_pixmap is deprecated, should be replaced
    gdk_draw_pixmap(widget->window,
		    vnc->gc,
		    vnc->offscreen,
		    event->area.x,
		    event->area.y,
		    event->area.x,     event->area.y,
		    event->area.width, event->area.height);
  }

  return FALSE;
}

static gint vnc_viewer_button_press_event(GtkWidget *widget,
					  GdkEventButton *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);
  guint16 x, y;

  if (vnc->fd < 0)
    return FALSE;

  switch (event->button)
  {
  case 1:
    vnc->button_mask |= rfbButton1Mask;
    break;

  case 2:
    vnc->button_mask |= rfbButton2Mask;
    break;

  case 3:
    vnc->button_mask |= rfbButton3Mask;
    break;

  default:
    break;
  }

  x = (guint16) event->x;
  y = (guint16) event->y;
  vnc_viewer_queue_event(vnc, newPointerEventMsg(vnc->button_mask, x, y));
					    
  return TRUE;
}

static gint vnc_viewer_button_release_event(GtkWidget *widget,
					    GdkEventButton *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);
  guint16 x, y;

  if (vnc->fd < 0)
    return FALSE;

  switch (event->button)
  {
  case 1:
    vnc->button_mask &= ~rfbButton1Mask;
    break;

  case 2:
    vnc->button_mask &= ~rfbButton2Mask;
    break;

  case 3:
    vnc->button_mask &= ~rfbButton3Mask;
    break;

  default:
    break;
  }

  x = (guint16) event->x;
  y = (guint16) event->y;
  vnc_viewer_queue_event(vnc, newPointerEventMsg(vnc->button_mask, x, y));

  return TRUE;
}

static gint vnc_viewer_motion_notify_event(GtkWidget *widget,
					   GdkEventMotion *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);

  if (vnc->fd < 0)
    return FALSE;

  vnc->last_pointer_x_loc = (guint16) event->x;
  vnc->last_pointer_y_loc = (guint16) event->y;
  vnc_viewer_queue_event(vnc, newPointerEventMsg(vnc->button_mask,
						 vnc->last_pointer_x_loc,
						 vnc->last_pointer_y_loc));

  return TRUE;
}

gint vnc_viewer_key_press_event(GtkWidget *widget,
				GdkEventKey *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);

  if (vnc->fd < 0)
    return FALSE;

  if (arg_lockdown)
  {
    if (vnc->m_lockdownKeyPressKeyVal != event->keyval)
    {
      vnc->m_lockdownKeyPressKeyVal = event->keyval;
      g_get_current_time(&vnc->m_lockdownKeyPressTime);
      g_get_current_time(&vnc->m_lockdownLastKeyPressTime);
    }
    else
    {
      // if the difference in time between this key press and the
      // last key press is a second or greater, then assume this is
      // a separate key press that corresponds to a legitimate key
      // release
      GTimeVal currentTime;
      g_get_current_time(&currentTime);
      glong nSecDiff =
	currentTime.tv_sec - (vnc->m_lockdownLastKeyPressTime).tv_sec;
      glong nMicroSecDiff =
	currentTime.tv_usec - (vnc->m_lockdownLastKeyPressTime).tv_usec;
      if ((nSecDiff > 1) || ((nSecDiff == 1) && (nMicroSecDiff >= 0)))
	vnc->m_lockdownKeyPressTime = currentTime;

      vnc->m_lockdownLastKeyPressTime = currentTime;
    }

    return TRUE;
  }

  if (GTK_WIDGET_REALIZED(widget) &&
      gtk_im_context_filter_keypress(vnc->im_context, event))
    return TRUE;

  if (event->state == 0)
  {
    guint8 button_mask = 0;
    if (event->keyval == GDK_F8)
      button_mask |= rfbButton2Mask;
    else if (event->keyval == GDK_F7)
      button_mask |= rfbButton3Mask;

    if (button_mask)
    {
      // simulate middle or right mouse button click
      vnc_viewer_queue_event(vnc, newPointerEventMsg(button_mask,
						     vnc->last_pointer_x_loc,
						     vnc->last_pointer_y_loc));
      vnc_viewer_queue_event(vnc, newPointerEventMsg(0,
						     vnc->last_pointer_x_loc,
						     vnc->last_pointer_y_loc));

      return TRUE;
    }
  }

  vnc_viewer_queue_event(vnc, newKeyEventMsg(TRUE, event->keyval));

  return TRUE;
}

gint vnc_viewer_key_release_event(GtkWidget *widget,
				  GdkEventKey *event)
{
  VncViewer *vnc = VNC_VIEWER(widget);

  if (vnc->fd < 0)
    return FALSE;

  if (arg_lockdown)
  {
    // if the key was held down for at least 5 seconds and it was the last
    // key pressed, then display password dialog allowing user to exit lock
    // down mode
    if (vnc->m_lockdownKeyPressKeyVal == event->keyval)
    {
      GTimeVal currentTime;
      g_get_current_time(&currentTime);
      glong nSecDiff =
	currentTime.tv_sec - (vnc->m_lockdownKeyPressTime).tv_sec;
      glong nMicroSecDiff =
	currentTime.tv_usec - (vnc->m_lockdownKeyPressTime).tv_usec;
      if ((nSecDiff > 2) || ((nSecDiff == 2) && (nMicroSecDiff >= 0)))
      {
	if (arg_lockdown_password && strlen(arg_lockdown_password))
	{
	  GtkWidget *dialog = hildon_get_password_dialog_new(0, FALSE);
	  GtkWidget *reconnectButton =
	    gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CONNECT, 1);
	  gtk_button_set_label(GTK_BUTTON(reconnectButton), "Reconnect");
	  
	  gtk_widget_show(dialog);
	  gint nResponse = gtk_dialog_run(GTK_DIALOG(dialog));
	  if (nResponse == GTK_RESPONSE_OK)
	  {
	    HildonGetPasswordDialog *passDialog = 
	      HILDON_GET_PASSWORD_DIALOG(dialog);
	    if (strcmp(arg_lockdown_password,
		       hildon_get_password_dialog_get_password(passDialog)) == 0)
	    {
	      arg_lockdown = FALSE;
	      if (vnc->mainview)
	      {
		create_menu(vnc->mainview);
		disable_home_key_lock(vnc->mainview);
	      }
	    }
	    else
	    {
	      GtkWidget *msgDialog =
		hildon_note_new_information(0, "Password incorrect");
	      gtk_dialog_run(GTK_DIALOG(msgDialog));
	      gtk_widget_destroy(msgDialog);

	      vnc->m_lockdownKeyPressKeyVal = 0;
	    }
	  }
	  else if (nResponse == 1)
	  {
	    gtk_widget_destroy(dialog);
	    dialog = 0;

	    // cause a reconnect
	    vnc_viewer_close_connection(vnc);
	    g_signal_emit(G_OBJECT(vnc), vnc_signals[CONNECTION_DIED], 0);
	  }
	  else
	    vnc->m_lockdownKeyPressKeyVal = 0;

	  if (dialog)
	    gtk_widget_destroy(dialog);
	}
	else
	{
	  arg_lockdown = FALSE;
	  if (vnc->mainview)
	  {
	    create_menu(vnc->mainview);
	    disable_home_key_lock(vnc->mainview);
	  }
	}
      }
    }

    return TRUE;
  }

  if (GTK_WIDGET_REALIZED(widget) &&
      gtk_im_context_filter_keypress(vnc->im_context, event))
    return TRUE;

  vnc_viewer_queue_event(vnc, newKeyEventMsg(FALSE, event->keyval));

  //if (GTK_WIDGET_CLASS(parent_class)->key_press_event(widget, event))
  //  return TRUE;

  return FALSE;
}

void vnc_viewer_key_press(VncViewer *vnc, guint keyval)
{
  if (vnc->fd < 0)
    return;

  vnc_viewer_queue_event(vnc, newKeyEventMsg(TRUE, keyval));
}

void vnc_viewer_key_release(VncViewer *vnc, guint keyval)
{
  if (vnc->fd < 0)
    return;

  vnc_viewer_queue_event(vnc, newKeyEventMsg(FALSE, keyval));
}

static void vnc_viewer_finalize(GObject *object)
{
  VncViewer *vnc;

  g_return_if_fail(object != NULL);
  g_return_if_fail(VNC_IS_VIEWER(object));

  vnc = VNC_VIEWER(object);

  vnc_viewer_close_connection(vnc);

  if (vnc->desk_name) g_free(vnc->desk_name);

  if (vnc->offscreen)
  {
    gdk_pixmap_unref(vnc->offscreen);
    vnc->offscreen = 0;
  }

  gdk_cursor_destroy(vnc->dot);
  gdk_cursor_destroy(vnc->none);

  vnc->dot = 0;
  vnc->none = 0;
  
  (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}

void vnc_viewer_im_commit(GtkIMContext *im_context, gchar *text, gpointer data)
{
  VncViewer *vnc = VNC_VIEWER(data);
  gunichar uChar = 0;
  guint keyval = 0;
  gchar *text2 = text;

  // the string is in UTF-8 format--iterate through the characters
  while (*text2)
  {
    uChar = g_utf8_get_char(text2);
    keyval = gdk_unicode_to_keyval(uChar);
    vnc_viewer_queue_event(vnc, newKeyEventMsg(TRUE, keyval));
    vnc_viewer_queue_event(vnc, newKeyEventMsg(FALSE, keyval));

    text2 = g_utf8_next_char(text2);
  }  
}

/*
void vnc_viewer_im_preedit_changed(GtkIMContext *im_context, gpointer data)
{
  VncViewer *vnc = VNC_VIEWER(data);
  gchar *str;
  gint cursor;

  gtk_im_context_get_preedit_string(im_context, &str, 0, &cursor);

  // todo
  
  g_free(str);
}
*/

gboolean vnc_viewer_focus(GtkWidget *widget,
			  GdkEventFocus *event,
			  GtkIMContext *im_context)
{
  if (event->in)
    gtk_im_context_focus_in(im_context);
  else
    gtk_im_context_focus_out(im_context);

  return FALSE;
}
