/*
 * mweather-window.c: Maemo weather main window
 *
 * Copyright (C) 2006 Philip Langdale
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors:
 *     Philip Langdale <philipl@mail.utexas.edu>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n.h>
#include <gdk/gdkkeysyms.h>

#include <gtk/gtktreemodel.h>
#include <gtk/gtktreeviewcolumn.h>

#include <gconf/gconf-client.h>

#include "mweather-window.h"
#include "mweather-details.h"
#include "mweather-location.h"
#include "mweather-pref.h"


struct _MWeatherWindowPrivate {
   osso_context_t *ctx;

   GtkListStore *store;
   GtkWidget *combo;
   GtkWidget *details;

   GtkWidget *updateItem;
   GtkWidget *removeItem;
   GtkWidget *prefsItem;

   gboolean fullscreen;
};

typedef struct {
   GtkListStore *store;
   GtkTreeRowReference *reference;
} RowState;

enum
{
   PROP_0,
   PROP_OSSO_CONTEXT,
};

enum
{
   COLUMN_ICON,
   COLUMN_MARKUP,
   COLUMN_LOCATION,
   COLUMN_STATE,
   N_COLUMNS,
};

G_DEFINE_TYPE(MWeatherWindow, mweather_window, HILDON_TYPE_WINDOW)
#define MWEATHER_WINDOW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), MWEATHER_TYPE_WINDOW, MWeatherWindowPrivate)) 


static void
update_locations(MWeatherWindow *window)
{
   gboolean valid;
   GtkTreeIter iter;
   GtkTreeModel *model = GTK_TREE_MODEL(window->priv->store);

   for(valid = gtk_tree_model_get_iter_first(model, &iter) ;
       valid ;
       valid = gtk_tree_model_iter_next(model, &iter)) {
      MWeatherLocation *location;

      gtk_tree_model_get(model, &iter,
                         COLUMN_LOCATION, &location,
                         -1);

      mweather_location_update(location);

      g_object_unref(G_OBJECT(location));
   }
}


static void
location_updated_cb(MWeatherLocation *location,
                    RowState *state)
{
   gboolean valid;
   GtkTreeIter iter;

   valid = gtk_tree_model_get_iter(GTK_TREE_MODEL(state->store), &iter,
                                   gtk_tree_row_reference_get_path(state->reference));
   g_return_if_fail(valid);

   const char *name = mweather_location_get_name(location);
   const char *temp = mweather_location_get_temp(location);
   const char *summary = mweather_location_get_summary(location);
   GdkPixbuf *pixbuf = mweather_location_get_pixbuf(location, FALSE);

   char *markup = g_strdup_printf("<b>%s</b>: %s, %s", name, summary, temp);

   gtk_list_store_set(state->store, &iter,
                      COLUMN_ICON, pixbuf,
                      COLUMN_MARKUP, markup,
                      -1);

   g_free(markup);
}


static void
list_location(MWeatherWindow *window,
              const char *dir)
{
   MWeatherLocation *location;
   RowState *state;
   GtkTreeIter iter;
   GtkTreePath *path;
   GtkListStore *store = window->priv->store;

   gtk_list_store_append(store, &iter);
   path = gtk_tree_model_get_path(GTK_TREE_MODEL(store), &iter);

   state = g_new0(RowState, 1);
   state->store = store;
   state->reference = gtk_tree_row_reference_new(GTK_TREE_MODEL(store), path);
   gtk_tree_path_free(path);

   location = mweather_location_new(dir);
   g_signal_connect(location, "updated",
                    G_CALLBACK(location_updated_cb), state);

   gtk_list_store_set(store, &iter,
                      COLUMN_LOCATION, location,
                      COLUMN_STATE, state,
                      -1);

   gtk_combo_box_set_active_iter(GTK_COMBO_BOX(window->priv->combo),
                                 &iter);

   g_object_unref(G_OBJECT(location));

   mweather_location_update(location);
}

static void
new_location(MWeatherWindow *window)
{
   unsigned int i = 0;
   gboolean dirExists;
   GConfClient *gconf = gconf_client_get_default();
   char *dir;

   do {
      dir = g_strdup_printf("/apps/gweather/%u", i);
      if (!dir) {
         break;
      }

      dirExists = gconf_client_dir_exists(gconf, dir, NULL);
      if (dirExists) {
         g_free(dir);
      }
      i++;
   } while (dirExists);

   list_location(window, dir);

   g_object_unref(gconf);
}


static void
remove_location(MWeatherWindow *window)
{
   gboolean valid;
   char *path;
   GConfClient *gconf;
   GtkTreeIter iter;
   MWeatherLocation *location;
   GtkTreeModel *model = GTK_TREE_MODEL(window->priv->store);

   valid = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(window->priv->combo),
                                         &iter);
   g_return_if_fail(valid);

   gtk_tree_model_get(model, &iter,
                      COLUMN_LOCATION, &location,
                      -1);
   g_object_get(location,
                "gconf-path", &path,
                NULL);
   g_object_unref(location);
   gtk_list_store_remove(window->priv->store, &iter);

   gconf = gconf_client_get_default();
   gconf_client_recursive_unset(gconf, path, 0, NULL);
   g_object_unref(gconf);   

   valid = gtk_tree_model_get_iter_first(model, &iter);
   if (valid) {
      gtk_combo_box_set_active_iter(GTK_COMBO_BOX(window->priv->combo),
                                    &iter);
   } else {
      gtk_combo_box_set_active(GTK_COMBO_BOX(window->priv->combo), -1);
   }
}


static void
show_prefs(MWeatherWindow *window)
{
   gboolean valid;
   GtkTreeIter iter;
   MWeatherLocation *location;
   GtkWidget *prefs;
   GtkTreeModel *model = GTK_TREE_MODEL(window->priv->store);

   valid = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(window->priv->combo),
                                         &iter);
   if (!valid) {
      valid = gtk_tree_model_get_iter_first(model, &iter);
      g_return_if_fail(valid);
   }

   gtk_tree_model_get(model, &iter,
                      COLUMN_LOCATION, &location,
                      -1);

   prefs = mweather_pref_new(location);
   gtk_window_set_transient_for(GTK_WINDOW(prefs),
                                GTK_WINDOW(gtk_widget_get_parent(GTK_WIDGET(window))));
   gtk_widget_show_all(prefs);
   gtk_window_set_destroy_with_parent(GTK_WINDOW(prefs), TRUE);
}


static void
show_about(MWeatherWindow *window)
{
   gtk_show_about_dialog(GTK_WINDOW(window),
                         "comments", _("A weather reporting program"),
                         "copyright", _("Copyright 2005,2006 Philip Langdale\n"
                                        "and the GWeather developers"),
                         "logo-icon-name", "stock_weather-few-clouds",
                         "name", _("Maemo Weather"),
                         "version", PACKAGE_VERSION,
                         "website", "https://garage.maemo.org/projects/mweather",
                         NULL);
}


static void
create_menu(MWeatherWindow *window)
{
   GtkMenu *top;
   GtkWidget *item;

   top = GTK_MENU(gtk_menu_new());
   hildon_window_set_menu(HILDON_WINDOW(window), top);

   item = gtk_menu_item_new_with_mnemonic(_("_Update"));
   gtk_widget_show(item);
   gtk_menu_append(top, item);
   g_signal_connect_swapped(G_OBJECT(item), "activate",
                            G_CALLBACK(update_locations),
                            window);
   window->priv->updateItem = item;

   item = gtk_separator_menu_item_new();
   gtk_widget_show(item);
   gtk_menu_append(top, item);

   item = gtk_menu_item_new_with_mnemonic(_("_New Location"));
   gtk_widget_show(item);
   gtk_menu_append(top, item);
   g_signal_connect_swapped(G_OBJECT(item), "activate",
                            G_CALLBACK(new_location),
                            window);

   item = gtk_menu_item_new_with_mnemonic(_("_Remove Location"));
   gtk_widget_show(item);
   gtk_menu_append(top, item);
   g_signal_connect_swapped(G_OBJECT(item), "activate",
                            G_CALLBACK(remove_location),
                            window);
   window->priv->removeItem = item;

   item = gtk_menu_item_new_with_mnemonic(_("_Preferences"));
   gtk_widget_show(item);
   gtk_menu_append(top, item);
   g_signal_connect_swapped(G_OBJECT(item), "activate",
                            G_CALLBACK(show_prefs),
                            window);
   window->priv->prefsItem = item;

   item = gtk_separator_menu_item_new();
   gtk_widget_show(item);
   gtk_menu_append(top, item);

   item = gtk_menu_item_new_with_mnemonic(_("_About"));
   gtk_widget_show(item);
   gtk_menu_append(top, item);
   g_signal_connect_swapped(G_OBJECT(item), "activate",
			    G_CALLBACK(show_about),
                            window);

   item = gtk_separator_menu_item_new();
   gtk_widget_show(item);
   gtk_menu_append(top, item);

   item = gtk_menu_item_new_with_mnemonic(_("_Quit"));
   gtk_widget_show(item);
   gtk_menu_append(top, item);
   g_signal_connect(G_OBJECT (item), "activate",
                    G_CALLBACK(gtk_main_quit), NULL);
}


static void
load_locations(MWeatherWindow *window)
{
   unsigned int i = 0;
   gboolean dirExists;
   GConfClient *gconf = gconf_client_get_default();

   do {
      char *dir = g_strdup_printf("/apps/gweather/%u", i);
      if (!dir) {
         break;
      }

      dirExists = gconf_client_dir_exists(gconf, dir, NULL);
      if (dirExists) {
         list_location(window, dir);
      }

      i++;
      g_free(dir);
   } while (dirExists);

   g_object_unref(gconf);
}


static void
combo_changed_cb(MWeatherWindow *window)
{
   gboolean valid;
   GtkTreeIter iter;
   MWeatherLocation *location;
   GtkTreeModel *model = GTK_TREE_MODEL(window->priv->store);

   valid = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(window->priv->combo),
                                         &iter);
   if (valid) {
      gtk_tree_model_get(model, &iter,
                         COLUMN_LOCATION, &location,
                         -1);

      mweather_details_set_location(MWEATHER_DETAILS(window->priv->details),
                                    location);

      g_object_unref(location);
   } else {
      mweather_details_set_location(MWEATHER_DETAILS(window->priv->details),
                                    NULL);
   }

   gtk_widget_set_sensitive(window->priv->updateItem, valid);
   gtk_widget_set_sensitive(window->priv->removeItem, valid);
   gtk_widget_set_sensitive(window->priv->prefsItem, valid);
}


static void
populate_window(MWeatherWindow *window)
{
   GtkWidget *vbox;
   GtkCellRenderer *renderer;
   GtkWidget *combo;

   vbox = gtk_vbox_new(FALSE, 0);
   gtk_widget_show(vbox);
   gtk_container_add(GTK_CONTAINER(window), vbox);

   combo = gtk_combo_box_new_with_model(GTK_TREE_MODEL(window->priv->store));
   gtk_widget_show(combo);
   gtk_box_pack_start(GTK_BOX(vbox), combo, FALSE, TRUE, 0);

   renderer = gtk_cell_renderer_pixbuf_new();
   gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo), renderer, FALSE);
   gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(combo), renderer, "pixbuf", COLUMN_ICON);

   renderer = gtk_cell_renderer_text_new();
   gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo), renderer, TRUE);
   gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(combo), renderer, "markup", COLUMN_MARKUP);

   g_signal_connect_swapped(GTK_COMBO_BOX(combo), "changed",
                            G_CALLBACK(combo_changed_cb), window);

   window->priv->combo = combo;

   window->priv->details = mweather_details_new();
   gtk_widget_show(window->priv->details);
   gtk_box_pack_start(GTK_BOX(vbox), window->priv->details, TRUE, TRUE, 0);
}


static gboolean
key_press_cb(MWeatherWindow *window,
             GdkEventKey *event)
{
   switch (event->keyval) {
   case HILDON_HARDKEY_FULLSCREEN:
      if (window->priv->fullscreen) {
         gtk_window_unfullscreen(GTK_WINDOW(window));
      } else {
         gtk_window_fullscreen(GTK_WINDOW(window));
      }
      window->priv->fullscreen = !window->priv->fullscreen;
      return TRUE;
   default:   
      return FALSE;
   }
}


static void
mweather_window_set_property(GObject *object,
                             guint prop_id,
                             const GValue *value,
                             GParamSpec *pspec)
{
   MWeatherWindow *window = MWEATHER_WINDOW(object);

   switch (prop_id) {
      case PROP_OSSO_CONTEXT:
         window->priv->ctx = g_value_get_pointer(value);
         break;
   }
}


static void
mweather_window_get_property(GObject *object,
                             guint prop_id,
                             GValue *value,
                             GParamSpec *pspec)
{
   MWeatherWindow *window = MWEATHER_WINDOW(object);

   switch (prop_id) {
      case PROP_OSSO_CONTEXT:
         g_value_set_pointer(value, window->priv->ctx);
         break;
   }
}


static void
mweather_window_init (MWeatherWindow *self)
{
   self->priv = MWEATHER_WINDOW_GET_PRIVATE(self);

   self->priv->store = gtk_list_store_new(N_COLUMNS,
                                          GDK_TYPE_PIXBUF,
                                          G_TYPE_STRING,
                                          MWEATHER_TYPE_LOCATION,
                                          G_TYPE_POINTER);

   populate_window(self);
   create_menu(self);
   load_locations(self);

   gtk_window_set_icon_name(GTK_WINDOW(self), "stock_window-few-clouds");

   g_signal_connect(self, "delete_event", gtk_main_quit, NULL);

   g_signal_connect(self, "key_press_event", G_CALLBACK(key_press_cb), NULL);
}


static void
mweather_window_finalize (GObject *object)
{
   MWeatherWindow *self = MWEATHER_WINDOW(object);

   g_object_unref(self->priv->store);

   G_OBJECT_CLASS(mweather_window_parent_class)->finalize(object);
}


GtkWidget *
mweather_window_new (osso_context_t *ctx)
{
    return g_object_new(MWEATHER_TYPE_WINDOW,
                        "osso-context", ctx,
                        NULL);
}


static void
mweather_window_class_init (MWeatherWindowClass *klass)
{
   GObjectClass *object_class = G_OBJECT_CLASS(klass);

   mweather_window_parent_class = g_type_class_peek_parent(klass);

   object_class->set_property = mweather_window_set_property;
   object_class->get_property = mweather_window_get_property;
   object_class->finalize = mweather_window_finalize;

   g_object_class_install_property(object_class,
                                   PROP_OSSO_CONTEXT,
                                   g_param_spec_pointer("osso-context",
                                                        "OSSO Context",
                                                        "The OSSO Context",
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_CONSTRUCT_ONLY));

   g_type_class_add_private(klass, sizeof(MWeatherWindowPrivate));
}
