/*
 * mweather-location.c: Maemo weather location
 *
 * Copyright (C) 2006 Philip Langdale
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors:
 *     Philip Langdale <philipl@mail.utexas.edu>
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <glib/gi18n.h>

#include <libgweather/gweather-gconf.h>
#include <libgweather/gweather-prefs.h>
#include <libgweather/weather.h>

#include "mweather-location.h"


struct _MWeatherLocationPrivate {
   GWeatherPrefs prefs;

   char *gconf_path;
   GWeatherGConf *gconf;

   WeatherInfo *info;
};

enum
{
   PROP_0,
   PROP_GWEATHER_PREFS,
   PROP_GCONF_PATH,
   PROP_GCONF_CONTEXT,
};

enum
{
   SIGNAL_UPDATED,
   NUM_SIGNALS,
};

static guint location_signals[NUM_SIGNALS] = { 0 };

G_DEFINE_TYPE(MWeatherLocation, mweather_location, G_TYPE_OBJECT)
#define MWEATHER_LOCATION_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), MWEATHER_TYPE_LOCATION, MWeatherLocationPrivate)) 


static void
mweather_location_set_property(GObject *object,
                               guint prop_id,
                               const GValue *value,
                               GParamSpec *pspec)
{
   MWeatherLocation *location = MWEATHER_LOCATION(object);

   switch (prop_id) {
      case PROP_GCONF_PATH:
         location->priv->gconf_path = g_value_dup_string(value);
         break;
   }
}


static void
mweather_location_get_property(GObject *object,
                               guint prop_id,
                               GValue *value,
                               GParamSpec *pspec)
{
   MWeatherLocation *location = MWEATHER_LOCATION(object);

   switch (prop_id) {
      case PROP_GCONF_PATH:
      {
         char *out_path = g_strdup(location->priv->gconf_path);
         g_value_take_string(value, out_path);
         break;
      }
      case PROP_GCONF_CONTEXT:
         g_value_set_pointer(value, location->priv->gconf);
         break;
      case PROP_GWEATHER_PREFS:
         g_value_set_pointer(value, &location->priv->prefs);
         break;
   }
}


static void
mweather_location_init (MWeatherLocation *self)
{
   self->priv = MWEATHER_LOCATION_GET_PRIVATE(self);
}


static GObject *
mweather_location_constructor (GType type,
                               guint n_construct_params,
                               GObjectConstructParam *construct_params)
{
   GObject *object;
   MWeatherLocation *self;

   object = G_OBJECT_CLASS(mweather_location_parent_class)->
      constructor(type, n_construct_params, construct_params);
   self = MWEATHER_LOCATION(object);

   self->priv->gconf = gweather_gconf_new(self->priv->gconf_path);
   gweather_prefs_load(&self->priv->prefs, self->priv->gconf);

   return object;
}


static void
mweather_location_finalize (GObject *object)
{
   MWeatherLocation *self = MWEATHER_LOCATION(object);

   gweather_gconf_free(self->priv->gconf);
   g_free(self->priv->gconf_path);

   G_OBJECT_CLASS(mweather_location_parent_class)->finalize(object);
}


MWeatherLocation *
mweather_location_new (const char *gconf_path)
{
    return g_object_new(MWEATHER_TYPE_LOCATION,
                        "gconf-path", gconf_path,
                        NULL);
}


static void
mweather_location_class_init (MWeatherLocationClass *klass)
{
   GObjectClass *object_class = G_OBJECT_CLASS(klass);

   mweather_location_parent_class = g_type_class_peek_parent(klass);

   object_class->set_property = mweather_location_set_property;
   object_class->get_property = mweather_location_get_property;
   object_class->constructor = mweather_location_constructor;
   object_class->finalize = mweather_location_finalize;

   g_object_class_install_property(object_class,
                                   PROP_GCONF_PATH,
                                   g_param_spec_string("gconf-path",
                                                       "GConf Path",
                                                       "The GConf path",
                                                       "/",
                                                       G_PARAM_READWRITE |
                                                       G_PARAM_CONSTRUCT_ONLY));
   g_object_class_install_property(object_class,
                                   PROP_GCONF_CONTEXT,
                                   g_param_spec_pointer("gconf-context",
                                                        "GConf Context",
                                                        "The GConf Context",
                                                        G_PARAM_READABLE));
   g_object_class_install_property(object_class,
                                   PROP_GWEATHER_PREFS,
                                   g_param_spec_pointer("gweather-prefs",
                                                        "GWeather Prefs",
                                                        "The GWeather Prefs",
                                                        G_PARAM_READABLE));

   location_signals[SIGNAL_UPDATED] =
      g_signal_new("updated",
                   G_OBJECT_CLASS_TYPE(object_class),
                   G_SIGNAL_RUN_FIRST,
                   G_STRUCT_OFFSET(MWeatherLocationClass, updated),
                   NULL, NULL,
                   g_cclosure_marshal_VOID__VOID,
                   G_TYPE_NONE,
                   0);

   g_type_class_add_private(klass, sizeof(MWeatherLocationPrivate));
}


static void
weather_info_update_cb(WeatherInfo *info,
                       gpointer data)
{
   MWeatherLocation *location = MWEATHER_LOCATION(data);

   g_signal_emit(G_OBJECT(location), location_signals[SIGNAL_UPDATED], 0);
}


void
mweather_location_update(MWeatherLocation *location)
{
   GWeatherPrefs *gwprefs = &location->priv->prefs;
   WeatherPrefs prefs;

   if (location->priv->info) {
      weather_info_free(location->priv->info);
   }

   /* Set preferred forecast type */
   prefs.type = gwprefs->detailed ? FORECAST_ZONE : FORECAST_STATE;
        
   /* Set radar map retrieval option */
//   prefs.radar = gwprefs->radar_enabled;
   prefs.radar = TRUE;
   prefs.radar_custom_url = gwprefs->use_custom_radar_url && gwprefs->radar ?
                            gwprefs->radar : NULL;
                                                                                   
   /* Set the units */
   prefs.temperature_unit = gwprefs->temperature_unit;
   prefs.speed_unit = gwprefs->speed_unit;
   prefs.pressure_unit = gwprefs->pressure_unit;
   prefs.distance_unit = gwprefs->distance_unit;
                                                                                                        
   location->priv->info = weather_info_new(gwprefs->location, &prefs,
                                           weather_info_update_cb, location);
}


gboolean
mweather_location_is_valid(MWeatherLocation *location)
{
   return location->priv->info ? weather_info_is_valid(location->priv->info) : FALSE;
}


const char *
mweather_location_get_name(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_location_name(location->priv->info);
}


const char *
mweather_location_get_update(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_update(location->priv->info);
}


const char *
mweather_location_get_cond(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_conditions(location->priv->info);
}


const char *
mweather_location_get_sky(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_sky(location->priv->info);
}


const char *
mweather_location_get_temp(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_temp_summary(location->priv->info);
}


const char *
mweather_location_get_feel(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_apparent(location->priv->info);
}


const char *
mweather_location_get_dewpoint(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_dew(location->priv->info);
}


const char *
mweather_location_get_humidity(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_humidity(location->priv->info);
}


const char *
mweather_location_get_wind(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_wind(location->priv->info);
}


const char *
mweather_location_get_pressure(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_pressure(location->priv->info);
}


const char *
mweather_location_get_visi(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_visibility(location->priv->info);
}


const char *
mweather_location_get_sunrise(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_sunrise(location->priv->info);
}


const char *
mweather_location_get_sunset(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_sunset(location->priv->info);
}


const char*
mweather_location_get_summary(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   const char *value = mweather_location_get_cond(location);
   if (strcmp(value, "-") == 0) {
      value = mweather_location_get_sky(location);
   }

   return value;
}


GdkPixbuf *
mweather_location_get_pixbuf(MWeatherLocation *location,
			     gboolean small)
{
   GdkPixbuf *pixbuf;

   g_return_val_if_fail(location->priv->info, NULL);

   _weather_info_get_pixbuf(location->priv->info, small, &pixbuf);
   return pixbuf;
}


GdkPixbufAnimation *
mweather_location_get_radar(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_radar(location->priv->info);
}


const char *
mweather_location_get_forecast(MWeatherLocation *location)
{
   g_return_val_if_fail(location->priv->info, NULL);

   return weather_info_get_forecast(location->priv->info);
}
